/*
 * Copyright (C) 1997-2009, R3vis Corporation.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA, or visit http://www.gnu.org/copyleft/lgpl.html.
 *
 * Original Contributor:
 *   Wes Bethel, R3vis Corporation, Marin County, California
 *   http://www.r3vis.com/
 * Additional Contributor(s):
 *
 * The OpenRM project is located at http://openrm.sourceforge.net/.
 */
/*
 * $Id: rmerror.c,v 1.8 2007/07/22 17:35:28 wes Exp $
 * Version: $Name: v180-alpha-02 $
 * $Revision: 1.8 $
 * $Log: rmerror.c,v $
 * Revision 1.8  2007/07/22 17:35:28  wes
 * Replaced errno with my_errno to avoid symbol conflict under mingw
 *
 * Revision 1.7  2005/07/23 16:16:06  wes
 * Minor tweak to have RM_NOTIFY_SILENCE actually mute output on rmInit.
 *
 * Revision 1.6  2005/03/16 16:43:46  wes
 * Fixed problem with Win32 code for rmSplash where stderr was
 * not defined. Workaround is to use printf rather than fprintf.
 *
 * Revision 1.5  2005/02/19 16:24:44  wes
 * Distro sync and consolidation.
 * Added rmSplash().
 *
 * Revision 1.4  2005/01/23 17:00:22  wes
 * Copyright updated to 2005.
 *
 * Revision 1.3  2004/01/16 16:44:05  wes
 * Updated copyright line for 2004.
 *
 * Revision 1.2  2003/02/02 02:07:15  wes
 * Updated copyright to 2003.
 *
 * Revision 1.1.1.1  2003/01/28 02:15:23  wes
 * Manual rebuild of rm150 repository.
 *
 * Revision 1.7  2003/01/16 22:21:17  wes
 * Updated all source files to reflect new organization of header files:
 * all header files formerly located in include/rmaux, include/rmi, include/rmv
 * are now located in include/rm.
 *
 * Revision 1.6  2002/04/30 19:31:39  wes
 * Updated copyright dates.
 *
 * Revision 1.5  2001/06/03 20:45:21  wes
 * Removed last vestiges of the notion of a "current pipe."
 *
 * Revision 1.4  2001/03/31 17:12:38  wes
 * v1.4.0-alpha-2 checkin.
 *
 * Revision 1.3  2000/12/03 22:34:52  wes
 * Win32-specific modifications.
 *
 * Revision 1.2  2000/04/20 16:29:47  wes
 * Documentation additions/enhancements, some code rearragement.
 *
 * Revision 1.1.1.1  2000/02/28 21:29:40  wes
 * OpenRM 1.2 Checkin
 *
 * Revision 1.1.1.1  2000/02/28 17:18:47  wes
 * Initial entry - pre-RM120 release, source base for OpenRM 1.2.
 *
 */

#include <rm/rm.h>
#include "rmprivat.h"

/* not thread safe! */
static RMenum notifyLevel = RM_NOTIFY_FULL;

/*
 * This file contains 3 routines used to communicate with the user.
 * They all do basically the same thing: tell the user something. They
 * each prefix the message with one of "rmError", "rmWarning" or
 * "rmNotice".
 */

/*
 * ----------------------------------------------------
 * @Name rmError
 @pstart
 void rmError (const char *msg)
 @pend

 @astart 
 const char *msg - a character string containing an
    application-supplied error message. 
 @aend

 @dstart

 This routine posts an error message in an implementation-specific
 way.

 X11: The error message supplied by the application is printed to
 stderr, and is prefaced with the string "rmError:".

 Win32: A Win32 MessageBox containing the application string will pop
 up on the screen.

 In both Win32 and X11, this routine will return control to the
 application (does not call exit()).
 
 @dend
 * ----------------------------------------------------
 */
void
rmError (const char *msg)
{
    if (notifyLevel == RM_NOTIFY_SILENCE)
	return;
    /* print msg to stderr/message box and return (future sophistication?) */
#ifdef RM_WIN
    {
	MessageBox(NULL, msg, "rmError", MB_OK);
    }
#else
    fprintf(stderr, "rmError: %s \n", msg);
#endif
}


/*
 * ----------------------------------------------------
 * @Name rmWarning
 @pstart
 void rmWarning (const char *msg)
 @pend

 @astart
 const char *msg - a character string containing an
    application-supplied warning message. 
 @aend

 @dstart

 This routine posts an warning message in an implementation-specific
 way.

 X11: The warning message supplied by the application is printed to
 stderr, and is prefaced with the string "rmWarning:".

 Win32: A Win32 MessageBox containing the application string will pop
 up on the screen.

 In both Win32 and X11, this routine will return control to the
 application (does not call exit()).
 
 @dend
 * ----------------------------------------------------
 */
void
rmWarning (const char *msg)
{
    if (notifyLevel == RM_NOTIFY_SILENCE)
	return;
    /* print msg to stderr/message box and return (future sophistication?) */
#ifdef RM_WIN
    {
	MessageBox(NULL, msg, "rmWarning", MB_OK);
    }
#else
    fprintf(stderr, "rmWarning: %s \n", msg);
#endif
}


/*
 * ----------------------------------------------------
 * @Name rmNotice
 @pstart
 void rmNotice (const char *msg)
 @pend

 @astart
 const char *msg - a character string containing an
    application-supplied notice message.
 @aend

 @dstart

 This routine posts an notice message in an implementation-specific
 way.

 X11: The notice message supplied by the application is printed to
 stderr, and is prefaced with the string "rmNotice:".

 Win32: A Win32 MessageBox containing the application string will pop
 up on the screen.

 In both Win32 and X11, this routine will return control to the
 application (does not call exit()).
 
 @dend
 * ----------------------------------------------------
 */
void
rmNotice (const char *msg)
{
    if (notifyLevel == RM_NOTIFY_SILENCE)
	return;
    /* print msg to stderr/message box and return (future sophistication?) */
#ifdef RM_WIN
    {
	MessageBox(NULL, msg, "rmNotice", MB_OK);
    }
#else
    fprintf(stderr, "rmNotice: %s \n", msg);
#endif
}

#ifdef RM_WIN
private_rmW32Error(void)
{
    char errbuf[256];
    DWORD my_errno;
	
    my_errno = GetLastError();
    FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,NULL,my_errno,0,errbuf,256,NULL);
    rmError(errbuf);
}
#endif

/*
 * ----------------------------------------------------
 * @Name rmNotifyLevel
 @pstart
 RMenum rmNotifyLevel(RMenum level)
 @pend

 @astart
 RMenum level - an RMenum value. Must be either RM_NOTIFY_FULL or
    RM_NOTIFY_SILENCE.
 @aend

 @dstart
 Use this routine to set the volume on the notice/warning/error
 messages. When set to RM_NOTIFY_SILENCE, no notice/warning/error
 messages whatsoever will be generated by OpenRM. The only other
 option at this time is RM_NOTIFY_FULL. The default value is
 RM_NOTIFY_FULL.

 This routine is not thread safe.
 @dend
 * ----------------------------------------------------
 */
RMenum
rmNotifyLevel(RMenum level)
{
    if ((level != RM_NOTIFY_FULL) &&
        (level != RM_NOTIFY_SILENCE))
    {
        rmError("rmNotifyLevel() error: the input RMenum value must be either RM_NOTIFY_FULL or RM_NOTIFY_SILENCE");
        return(RM_WHACKED);
    }
    notifyLevel = level;
    return(RM_CHILL);
}

void
rmSplash (const char *msg)
{
    if (notifyLevel == RM_NOTIFY_SILENCE)
	return;
#ifdef RM_WIN
#if 0
    {
	MessageBox(NULL, msg, "rmSplash", MB_OK);
    }
#endif
    /* the message box on Windows is really annoying.
       RMSG commercial licensees should implement a blurb on their
       splash screen that says their product uses RM Scene Graph [tm]
       (C) Copyright 1997-2005 R3vis Corporation, All Rights Reserved.wes-9/11/04 */
    printf("%s \n", msg);
#else
    fprintf(stderr, "%s \n", msg);
#endif
}
/* EOF */
