// $Id: app.js 15853 2012-06-20 14:01:58Z chris $

// JavaScript architecture is as follows:
//
// 1. "app.js" defines the App object and various top-level items which we don't want
// to be part of Doc.
// 2. "dos.js" runs inside the Doc object (ie with "this" as a Doc). It should
// initialize the Doc.
//
// The following rules apply:
// * app.__engine is "rhino" or "mustang", for engine specific code.
// * app.__bfo is the JSCoreMethods implementation, doc.__pdf is the PDF
// * "event" object is a JavaScript object, event.__javaevent is the original Java.
// * Do not use Rhino syntax "get author() { ... }", as it's an error for Mustang.
// * Any objects defined in "app.js" will NOT be accessible to the Doc, due to Mustang
//   inability to chain contexts. "doc" has an "app" property, hang stuff off that.
// * When using a JSAdapter inside a __put__(k,v),  you can get magic properties but
//   cannot set them - ie "this.type" is ok but to set it's this.__put__("type", v);
// * Remember when creating an object (Field, App) to wrap in JSAdapter if it's Mustang.
//   This is why it's best not to call the constructor directly but have a __create method.
//   (actually can return value from Constructor if necessary)
// * Doc is not implemented with a JSAdapter in Mustang, due to JSAdapter not being a
//   ScriptContext. Magic properties must be managed in JSMustangParser.DocumentBindings.
// * "event" is a property of Doc, as that's what Acrobat does. So do not multi-thread.
// * If you're going to hang non-standard stuff off a standard object, use a __prefix.
// * For methods with more than one parameter, make sure methods have the same
//   parameter names as in the JS Reference, then call __fixcall as the first line.
//

function App(bfo) {
    this.__bfo = bfo;
    this.__engine = this.__bfo.getImplementation();
    this.app = this;

    this.console = new Console(this);
    this.util = new Util(this);
    this.color = new Color(this);


    this.alert = function(cMsg, nIcon, nType, cTitle, oDoc, oCheckbox) {
        if (arguments.length==1 && typeof(arguments[0])=="object") return __fixcall(this, arguments.callee, arguments[0], "cMsg", "nIcon", "nType", "cTitle", "oDoc", "oCheckbox");
        if (nIcon==undefined) nIcon=0;
        if (nType==undefined) nType=0;
        if (cTitle==undefined) cTitle="JavaScript";
        if (oDoc==undefined) oDoc=null;
        if (oCheckbox==undefined) oCheckbox=null;
        return this.__bfo.appAlert(cMsg, nIcon, nType, cTitle, oDoc, oCheckbox);
    };

    this.response = function(cQuestion, cTitle, cDefault, bPassword, cLabel) {
        if (arguments.length==1 && typeof(arguments[0])=="object") return __fixcall(this, arguments.callee, arguments[0], "cQuestion", "cTitle", "cDefault", "bPassword", "cLabel");
        if (cTitle==undefined) cTitle="JavaScript";
        if (bPassword==undefined) bPassword=false;
        if (cLabel==undefined) cLabel="";
        return __fixstring(this.__bfo.appResponse(cQuestion, cTitle, cDefault, bPassword, cLabel));
    };

    this.openDoc = function(cPath, oDoc, cFS, bHidden, bUseConv, cDest) { 
        if (arguments.length==1 && typeof(arguments[0])=="object") return __fixcall(this, arguments.callee, arguments[0], "cPath", "oDoc", "cFS", "bHidden", "bUseConv", "cDest");
        if (oDoc != undefined && (cPath[0]!='/' && cPath[0]!='\\')) {
            try {
                var origfile = this.__bfo.getDocumentPanel(oDoc.__pdf).getClientProperty("file");
                var base = new Packages.java.io.File(origfile);
                cPath = new Packages.java.io.File(base.getParentFile(), cPath);
            } catch (e) { 
            }
        } else {
            cPath = new Packages.java.io.File(cPath);
        }
        this.__bfo.getViewer().loadPDF(cPath);
        // return ref to file
    }

    this.beep = function() {
        this.__bfo.appBeep();
    };

    this.execMenuItem = function(s) {
        Packages.org.faceless.pdf2.viewer2.feature.GenericNamedActionHandler.run(this.__bfo.getViewer().getActiveDocumentPanel(), s);
    };

    this.activeDocs = [];
    this.toString = function() {
        return "[object App]";
    }

    if (this.__engine=="rhino") {
        this.__defineGetter__("focusRect", function() {
            return bfo.getAppFocusRect();
        });
        this.__defineSetter__("focusRect", function(x) {
            bfo.setAppFocusRect(x);
        });
        this.__defineGetter__("runtimeHighlight", function() {
            return bfo.getAppRuntimeHighlight();
        });
        this.__defineSetter__("runtimeHighlight", function(x) {
            bfo.setAppRuntimeHighlight(x);
        });
        this.__defineGetter__("runtimeHighlightColor", function() {
            return bfo.colorFromAWT(bfo.getAppRuntimeHighlightColor());
        });
        this.__defineSetter__("runtimeHighlightColor", function(x) {
            bfo.setAppRuntimeHighlightColor(bfo.colorToAWT(x));
        });
        this.__defineGetter__("viewerType", function() {
            return __fixstring(bfo.appViewerType());
        });
        this.__defineGetter__("viewerVariation", function() {
            return __fixstring(bfo.appViewerVariation());
        });
        this.__defineGetter__("viewerVersion", function() {
            return bfo.appViewerVersion();
        });
    } else if (this.__engine=="mustang") {
       this.__get__ = function (name) {
            if (name=="focusRect") {
                return bfo.getAppFocusRect();
            } else if (name=="runtimeHighlight") {
                return bfo.getAppRuntimeHighlight();
            } else if (name=="runtimeHighlightColor") {
                return bfo.colorFromAWT(bfo.getAppRuntimeHighlightColor());
            } else if (name=="viewerType") {
                return __fixstring(bfo.appViewerType());
            } else if (name=="viewerVersion") {
                return bfo.appViewerVersion();
            } else if (name=="viewerVariation") {
                return __fixstring(bfo.appViewerVariation());
            } else {
                return this[name];
            }
        };
        this.__put__ = function (name, value) {
            if (name=="focusRect") {
                bfo.setAppFocusRect(value);
            } else if (name=="runtimeHighlight") {
                bfo.setAppRuntimeHighlight(value);
            } else if (name=="runtimeHighlightColor") {
                bfo.setAppRuntimeHighlightColor(bfo.colorToAWT(value));
            } else {
                this[name] = value;
            }
        };
        this.__has__ = function (name) {
            return name in this || typeof(this['get'+name]) == 'function';
        };
        this.__delete__ = function (name) {
            delete this[name];
        };
    }

    this.__createField = function(field, fieldname, annotnum, doc) {
        var obj = new Field(field, fieldname, annotnum, doc);
        if (this.__engine=="mustang") obj = new JSAdapter(obj);
        return obj;
    }

    //------------------------------------------------------------------------------

    // Called to add Document to App.activeDocs. Called from end of "doc.js" (ie on Doc init)
    //
    this.__addDoc = function(doc) {
       this.activeDocs.push(doc);
    }

    // Called to remove Document to App.activeDocs. Called by JSEngine.destroyDoc
    //
    // For some weird reason, activeDocs[i]==doc will fail here under Mustang, but not
    // in Rhino. I think this is due to the way Rhino returns a new Scriptable on each
    // eval (returned from RhinoScriptEngine.getRuntimeScope). However we can identtify
    // if it's the doc we want by comparing the __pdf item, which will be constant and
    // works in both engines.
    //
    this.__delDoc = function(doc) {
        for (var i=0;i<this.activeDocs.length;i++) {
            if (this.activeDocs[i].__pdf==doc.__pdf) {
                this.activeDocs.splice(i, 1);
            }
        }
    }
}

//-----------------------------------------------------------------------------------
// Methods used directly by the JSEngines - do not be messin'
//-----------------------------------------------------------------------------------

function __createEvent(javaevent, app) {
    var event = new Event(javaevent, app);
    if (app.__engine=="mustang") event = new JSAdapter(event);
    return event;
}

function __createApp(bfo) {
    var app = new App(bfo);
    if (app.__engine=="mustang") app = new JSAdapter(app);
    return app;
}

//--------------------------------------------------------------------------------------
// Static objects
//--------------------------------------------------------------------------------------

function __fixstring(x) {
    return x==null ? "" : ""+x;
}

// Convert from deletePages({nPage: 1, nAfter:2}) to regular arguments
// Usage: if (arguments.length==1 && typeof(arguments[0])=="object") return __fixcall(this, arguments.callee, arguments[0], "nPage", "nAfter");
function __fixcall(/*...*/) {
    var that = arguments[0];
    var func = arguments[1];
    var struct = arguments[2];
    var newargs = []
    for (var i=3;i<arguments.length;i++) {
        newargs[newargs.length] = struct[arguments[i]];
    }
    func.apply(that, newargs);
}

function Util(app) {
    this.app = app;
}
Util.prototype = {
    longmonths: [ "January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December" ],
    abbrmonths: [ "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" ],
    longdays:   [ "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday" ],
    abbrdays:   [ "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" ],
    printf:     function() {
        var sb = "";
        var objnum = 0;
        var format = arguments[objnum++];
        for (var i=0;i<format.length;) {
            var c = format[i++];
            if (c=='%') {
                var comma = '1', cflags = 0, nwidth = 0, precision = 0;
                if (format[i]==',') {
                    comma = format[i+1];
                    i+=2;
                }
                c = format[i];
                if (c=='+' || c==' ' || c=='0' || c=='#') {
                    cflags = c;
                    c = format[++i];
                }
                while (c>='0' && c<='9') {
                    nwidth = (nwidth*10) + c-'0';
                    c = format[++i];
                }
                if (c=='.') {
                    c = format[++i];
                    while (c>='0' && c<='9') {
                        precision = (precision*10) + c-'0';
                        c = format[++i];
                    }
                }
                if (c=='d' || c=='f') {
                    var val;
                    try {
                        val = parseFloat(arguments[objnum]);
                    } catch (e) {
                        val = 0;
                    }
                    if (c=='d') val = Math.round(val);
                    var ival = c=='f' && precision==0 ? Math.round(val) : Math.floor(val);

                    var prefix = "", suffix = "", body = "";
                    if (cflags==' ') {
                        prefix = val<0 ? "-" : " ";
                        nwidth--;
                    } else if (cflags=='+') {
                        prefix = val<0 ? "-" : "+";
                        nwidth--;
                    } else if (val<0) {
                        prefix = "-";
                        nwidth--;
                    }
                    var pos = 0;
                    while (ival>0) {
                        if (comma=='0' && (pos++%3)==0) body = ','+body;
                        if (comma=='2' && (pos++%3)==0) body = '.'+body;
                        body = (ival%10) + body;
                        ival = Math.floor(ival/ 10);
                    }
                    if (body.length==0) body = "0";
                    if (c=='f') {
                        var f = precision==0 ? 0 : Math.round((val-Math.floor(val))*Math.pow(10, precision));
                        print(val+" "+ival+" "+f);
                        if (f==0) {
                            if (cflags=='#') suffix = (comma=='0' || comma=='1' ? "." : ",");
                        } else {
                            suffix = String(f);
                            while (suffix.length<precision) suffix = "0" + suffix;
                            suffix = (comma=='0' || comma=='1' ? '.' : ',') + suffix;
                        }
                    }
                    var diff = nwidth - (prefix.length+body.length+suffix.length);
                    if (cflags=='0') {
                        while (diff-->0) prefix += '0';
                    } else {
                        while (diff-->0) prefix = ' '+prefix;
                    }
                    sb += prefix;
                    sb += body;
                    sb += suffix;
                } else if (c=='s') {
                    var value;
                    try {
                        value = String(arguments[objnum]);
                    } catch (e) {
                        value = "";
                    }
                    while (nwidth-- > value.length) {
                        value = ' '+value;
                    }
                    sb += value;
                } else if (c=='x') {
                    var value;
                    try {
                        value = parseInt(arguments[objnum]).toString(16);
                    } catch (e) {
                        value = "0";
                    }
                    while (nwidth-- > value.length) {
                        value = ' '+value;
                    }
                    sb += value;
                }
                objnum++;
                i++;
            } else {
                sb += c;
            }
        }
        return sb;
    },
    printx :    function(format, v) {
        var sb = "";
        var i=0, j=0;
        var lc = false, uc = false;
        while (i<format.length && j<v.length) {
            var a, c = format[i++];
            switch(c) {
              case '\\':
                if (i<format.length) sb += format[i++];
                break;
              case '?':
                sb += v[j++];
                break;
              case 'X':
                while ((a=v[j++]).matches(/[A-Za-z0-9]/));
                sb += (uc ? a.toUpperCase() : lc ? a.toLowerCase() : a);
                break;
              case 'A':
                while ((a=v[j++]).matches(/[A-Za-z]/));
                sb += (uc ? a.toUpperCase() : lc ? a.toLowerCase() : a);
                break;
              case '9':
                while ((a=v[j++]).matches(/[0-9]/));
                sb += a;
                break;
              case '*':
                while (j<v.length) {
                    a = v[j++];
                    sb += (uc ? a.toUpperCase() : lc ? a.toLowerCase() : a);
                }
                break;
              case '>':
                lc = false;
                uc = true;
                break;
              case '<':
                lc = true;
                uc = false;
                break;
              case '=':
                lc = false;
                uc = false;
                break;
            }
        }
        return sb;
    },
    printd :    function(f, c) {
        var sb = "";
        var zone;
        if (f=="0" || f=="1") {
            var z = -c.getTimezoneOffset();
            if (z==0) {
                zone = "Z";
            } else {
                if (z<0) {
                    zone = "-";
                    z = -z;
                } else {
                    zone = "+";
                }
                zone += (z<600?"0":"")+Math.floor(z/60)+"'"+(z%60<10?"0":"")+(z%60)+"'";
            }
        } else {
            zone = "";
        }
        if (f=="0") {
            sb = "D:";
            f = "yyyymmddHHMMss";
        } else if (f==1) {
            f = "yyyy.mm.dd HH:MM:ss";
        } else if (f==2) {
            f = "yyyy.mm.dd HH:MM:ss";
        }
        while (f.length>0) {
            if (f.match(/^mmmmm/)) {
                sb += (this.longmonths[c.getMonth()]);
                f = f.substring(4);
            } else if (f.match(/^mmm/)) {
                sb += (this.abbrmonths[c.getMonth()]);
                f = f.substring(3);
            } else if (f.match(/^mm/)) {
                if (c.getMonth()<9) sb += ('0');
                sb += (c.getMonth()+1);
                f = f.substring(2);
            } else if (f.match(/^m/)) {
                sb += (c.getMonth()+1);
                f = f.substring(1);
            } else if (f.match(/^dddd/)) {
                sb += (this.longdays[c.getDay()]);
                f = f.substring(4);
            } else if (f.match(/^ddd/)) {
                sb += (this.abbrdays[c.getDay()]);
                f = f.substring(3);
            } else if (f.match(/^dd/)) {
                if (c.getDate()<10) sb += ('0');
                sb += (c.getDate());
                f = f.substring(2);
            } else if (f.match(/^d/)) {
                sb += (c.getDate());
                f = f.substring(1);
            } else if (f.match(/^yyyy/)) {
                sb += (c.getFullYear());
                f = f.substring(4);
            } else if (f.match(/^yy/)) {
                var y = c.getFullYear()-1900;
                while (y>100) y-=100;
                if (y<10) sb += ('0');
                sb += (y);
                f = f.substring(2);
            } else if (f.match(/^HH/)) {
                if (c.getHours()<10) sb += ('0');
                sb += (c.getHours());
                f = f.substring(2);
            } else if (f.match(/^H/)) {
                sb += (c.getHours());
                f = f.substring(1);
            } else if (f.match(/^hh/)) {
                if (c.getHours()<10) sb += ('0');
                sb += (c.getHours());
                f = f.substring(2);
            } else if (f.match(/^h/)) {
                sb += (c.getHours());
                f = f.substring(1);
            } else if (f.match(/^MM/)) {
                if (c.getMinutes()<10) sb += ('0');
                sb += (c.getMinutes());
                f = f.substring(2);
            } else if (f.match(/^M/)) {
                sb += (c.getMinutes());
                f = f.substring(1);
            } else if (f.match(/^ss/)) {
                if (c.getSeconds()<10) sb += ('0');
                sb += (c.getSeconds());
                f = f.substring(2);
            } else if (f.match(/^s/)) {
                sb += (c.getSeconds());
                f = f.substring(1);
            } else if (f.match(/^tt/)) {
                sb += (c.getHours()<12 ? "am" : "pm");
                f = f.substring(2);
            } else if (f.match(/^t/)) {
                sb += (c.getHours()<12 ? "a" : "p");
                f = f.substring(1);
            } else if (f.match(/^\\/)) {
                sb += (f[1]);
                f = f.substring(2);
            } else {
                sb += (f[0]);
                f = f.substring(1);
            }
        }
        return sb+zone;
    },
    scand :     function(f, val) {
        var c = new Date(0);
        if (f=="0" || f=="1") {
            if (val.substring(-1)=="Z") {
                c.setTimezoneOffset(0);
            } else if (val.match(/\([+-]\)\([0-9][0-9]\)'\([0-9][0-9]\)'$/)) {
                c.setTimezoneOffset((($2*60)+$3) * ($1=="+" ? 1 : -1));
            }
        }
        if (f=="0") {
            if (!val.match(/^D:/)) return null;
            val = val.substring(2);
            f = "yyyymmddHHMMss";
        } else if (f.equals("1")) {
            f = "yyyy.mm.dd HH:MM:ss";
        } else if (f.equals("2")) {
            f = "yyyy.mm.dd HH:MM:ss";
        }
        val = String(val);
        while (f.length>0) {
            var olen = val.length;
            if (olen==0) throw "OutOfBounds";
            try {
                var i;
                if (f.match(/^mmmm/)) {
                    for (i=0;i<this.longmonths.length;i++) {
                        if (val.substring(0, this.longmonths[i].length).equalsIgnoreCase(this.longmonths[i])) break;
                    }
                    if (i==12) throw "Exception";
                    c.setMonth(i);
                    val = val.substring(this.longmonths[i].length);
                    f = f.substring(4);
                } else if (f.match(/^mmm/)) {
                    for (i=0;i<this.abbrmonths.length;i++) {
                        if (val.substring(0, this.abbrmonths[i].length).equalsIgnoreCase(this.abbrmonths[i])) break;
                    }
                    if (i==12) throw "Exception";
                    c.setMonth(i);
                    val = val.substring(this.abbrmonths[i].length);
                    f = f.substring(3);
                } else if (f.match(/^mm/)) {
                    i = parseInt(val.substring(0, 2));
                    if (isNaN(i)) throw "NaN";
                    if (i<1 || i>12) throw "Exception";
                    c.setMonth(i-1);
                    val = val.substring(2);
                    f = f.substring(2);
                } else if (f.match(/^m/)) {
                    i = parseInt(val.substring(0, val[1].match(/[0-9]/) ? 2 : 1));
                    if (isNaN(i)) throw "NaN";
                    if (i<1 || i>12) throw "Exception";
                    c.setMonth(i-1);
                    val = val.substring(val[1].match(/[0-9]/) ? 2 : 1);
                    f = f.substring(1);
                } else if (f.match(/^dddd/)) {
                    for (i=0;i<this.longdays.length;i++) {
                        if (val.substring(0, this.longdays[i].length).equalsIgnoreCase(this.longdays[i])) break;
                    }
                    if (i==7) throw "Exception";
                    c.setDay(i);
                    val = val.substring(this.longdays[i].length);
                    f = f.substring(4);
                } else if (f.match(/^ddd/)) {
                    for (i=0;i<this.abbrmonths.length;i++) {
                        if (val.substring(0, this.abbrdays[i].length).equalsIgnoreCase(this.abbrdays[i])) break;
                    }
                    if (i==7) throw "Exception";
                    c.setDay(i);
                    val = val.substring(this.abbrdays[i].length);
                    f = f.substring(3);
                } else if (f.match(/^dd/)) {
                    i = parseInt(val.substring(0, 2));
                    if (isNaN(i)) throw "NaN";
                    if (i<1 || i>31) throw "Exception";
                    c.setDate(i);
                    val = val.substring(2);
                    f = f.substring(2);
                } else if (f.match(/^d/)) {
                    i = parseInt(val.substring(0, val[1].match(/[0-9]/) ? 2 : 1));
                    if (isNaN(i)) throw "NaN";
                    if (i<1 || i>31) throw "Exception";
                    c.setDate(i);
                    val = val.substring(val[1].match(/[0-9]/) ? 2 : 1);
                    f = f.substring(1);
                } else if (f.match(/^yyyy/)) {
                    i = parseInt(val.substring(0, 4));
                    if (isNaN(i)) throw "NaN";
                    c.setFullYear(i);
                    val = val.substring(4);
                    f = f.substring(4);
                } else if (f.match(/^yy/)) {
                    i = parseInt(val.substring(0, 2));
                    if (isNaN(i)) throw "NaN";
                    i += i<50 ? 2000 : 1900;
                    c.setFullYear(i);
                    val = val.substring(2);
                    f = f.substring(2);
                } else if (f.match(/^HH/)) {
                    i = parseInt(val.substring(0, 2));
                    if (isNaN(i)) throw "NaN";
                    if (i<0 || i>23) throw "Exception";
                    c.setHours(i);
                    val = val.substring(2);
                    f = f.substring(2);
                } else if (f.match(/^H/)) {
                    i = parseInt(val.substring(0, val[1].match(/[0-9]/) ? 2 : 1));
                    if (isNaN(i)) throw "NaN";
                    if (i<0 || i>23) throw "Exception";
                    c.setHours(i);
                    val = val.substring(val[1].match(/[0-9]/) ? 2 : 1);
                    f = f.substring(1);
                } else if (f.match(/^hh/)) {
                    i = parseInt(val.substring(0, 2));
                    if (isNaN(i)) throw "NaN";
                    if (i<1 || i>12) throw "Exception";
                    c.setHours(i);
                    val = val.substring(2);
                    f = f.substring(2);
                } else if (f.match(/^h/)) {
                    i = parseInt(val.substring(0, val[1].match(/[0-9]/) ? 2 : 1));
                    if (isNaN(i)) throw "NaN";
                    if (i<1 || i>12) throw "Exception";
                    c.setHours(i);
                    val = val.substring(val[1].match(/[0-9]/) ? 2 : 1);
                    f = f.substring(1);
                } else if (f.match(/^MM/)) {
                    i = parseInt(val.substring(0, 2));
                    if (isNaN(i)) throw "NaN";
                    if (i<0 || i>59) throw "Exception";
                    c.setMinutes(i);
                    val = val.substring(2);
                    f = f.substring(2);
                } else if (f.match(/^M/)) {
                    i = parseInt(val.substring(0, val[1].match(/[0-9]/) ? 2 : 1));
                    if (isNaN(i)) throw "NaN";
                    if (i<0 || i>59) throw "Exception";
                    c.setMinutes(i);
                    val = val.substring(val[1].match(/[0-9]/) ? 2 : 1);
                    f = f.substring(1);
                } else if (f.match(/^ss/)) {
                    i = parseInt(val.substring(0, 2));
                    if (isNaN(i)) throw "NaN";
                    if (i<0 || i>59) throw "Exception";
                    c.setSeconds(i);
                    val = val.substring(2);
                    f = f.substring(2);
                } else if (f.match(/^s/)) {
                    i = parseInt(val.substring(0, val[1].match(/[0-9]/) ? 2 : 1));
                    if (isNaN(i)) throw "NaN";
                    if (i<0 || i>59) throw "Exception";
                    c.setSeconds(i);
                    val = val.substring(val[1].match(/[0-9]/) ? 2 : 1);
                    f = f.substring(1);
                } else if (f.match(/^tt/)) {
                    if (val.substring(0,2).equalsIgnoreCase("am")) {
                       c.setHour(c.getHour()%12);
                    } else if (val.substring(0,2).equalsIgnoreCase("pm")) {
                       c.setHour((c.getHour()%12)+12);
                    } else {
                        throw "Exception";
                    }
                    f = f.substring(2);
                } else if (f.match(/^t/)) {
                    if (val.substring(0,1).equalsIgnoreCase("a")) {
                       c.setHour(c.getHour()%12);
                    } else if (val.substring(0,1).equalsIgnoreCase("p")) {
                       c.setHour((c.getHour()%12)+12);
                    } else {
                        throw "Exception";
                    }
                    f = f.substring(1);
                } else {
                    if (f[0]=='\\') f = f.substring(1);
                    if (f[0]==val[0]) {
                        val = val.substring(1);
                        f = f.substring(1);
                    } else {
                        return null;
                    }
                }
            } catch (e) {
                return null;
            }
        }
        return c;
    },
    toString : function() {
        return "[object Util]";
    }
};

function Console(app) {
    this.app = app;
}

Console.prototype = {
    show        : function() {
        this.app.__bfo.consoleShow();
    },
    hide        : function() {
        this.app.__bfo.consoleHide();
    },
    clear        : function() {
        this.app.__bfo.consoleClear();
    },
    println        : function(x) {
        this.app.__bfo.consolePrintln(x);
    },
    toString : function() {
        return "[object Console]";
    }
};

function Color(app) {
    this.app = app;
}
Color.prototype = {
    transparent         : [ "T" ],
    black               : [ "G", 0 ],
    white               : [ "G", 1 ],
    red                 : [ "RGB", 1, 0, 0 ],
    green               : [ "RGB", 0, 1, 0 ],
    blue                : [ "RGB", 0, 0, 1 ],
    cyan                : [ "CMYK", 1, 0, 0, 0 ],
    magenta             : [ "CMYK", 0, 1, 0, 0 ],
    yellow              : [ "CMYK", 0, 0, 1, 0 ],
    dkGray              : [ "G", 0.25 ],
    gray                : [ "G", 0.5 ],
    ltGray              : [ "G", 0.75 ],
    convert             : function(colorArray, cColorspace) {
        if (arguments.length==1 && typeof(arguments[0])=="object") return __fixcall(this, arguments.callee, arguments[0], "colorArray", "cColorspace");
        var out;
        if (cColorspace=="G") {
            if (colorArray[0]=="RGB") {
                out = new Array("G", 0.3*colorArray[1]+0.59*colorArray[2]+0.11*colorArray[3]);
            } else if (colorArray[0]=="CMYK") {
                out = new Array("G", 1.0-Math.min(1.0, 0.3*colorArray[1]+0.59*colorArray[2]+0.11*colorArray[3]+colorArray[4]));
            }
        } else if (cColorspace=="RGB") {
            if (colorArray[0] == "G") {
                out = new Array("RGB", colorArray[1], colorArray[1], colorArray[1]);
            } else if (color[0] == "CMYK") {
                out = new Array("RGB", 1.0-Math.min(1.0, colorArray[1]+colorArray[4]), 1.0-Math.min(1.0, colorArray[2]+colorArray[4]), 1.0-Math.min(1.0, colorArray[3]+colorArray[4]));
            }
        } else if (cColorspace=="CMYK") {
            if (colorArray[0] == "G") {
                out = new Array("CMYK", 0, 0, 0, 1.0-colorArray[1]);
            } else if (colorArray[0] == "RGB") {
                out = new Array("CMYK", 1.0-colorArray[1], 1.0-colorArray[2], 1.0-colorArray[3], 0); 
            }
        } else {
            out = colorArray;
        }
        return out;
    },
    equal              : function(c1, c2) {
    if (c1[0] == "G") {
            c1 = color.convert(c1, c2[0]);
        } else {
            c2 = color.convert(c2, c1[0]);
        }
        for (var i=0;i<c1.length;i++) {
            if (c1[i]!=c2[i]) return false;
        }
        return true;
    },
    toString : function() {
        return "[object Color]";
    }
};

//--------------------------------------------------------------------------------------
// Interesting objects
//--------------------------------------------------------------------------------------

function Event(je, app) {
    var app = app;
    if (app.__engine=="rhino") {
        this.__defineGetter__("change", function() { return __fixstring(je.getChange()); });
        this.__defineGetter__("changeEx", function() { return __fixstring(je.getChangeEx()); });
        this.__defineGetter__("commitKey", function() { return je.getCommitKey(); });
        this.__defineGetter__("fieldFull", function() { return je.getFieldFull(); });
        this.__defineGetter__("keyDown", function() { return je.getKeyDown(); });
        this.__defineGetter__("modifier", function() { return je.getModifier(); });
        this.__defineGetter__("name", function() { return __fixstring(je.getName()); });
        this.__defineGetter__("rc", function() { return je.getRc(); });
        this.__defineGetter__("richChange", function() { return __fixstring(je.getRichChange()); });
        this.__defineGetter__("richChangeEx", function() { return __fixstring(je.getRichChangeEx()); });
        this.__defineGetter__("richValue", function() { return __fixstring(je.getRichValue()); });
        this.__defineGetter__("selStart", function() { return je.getSelStart(); });
        this.__defineGetter__("selEnd", function() { return je.getSelEnd(); });
        this.__defineGetter__("shift", function() { return je.getShift(); });
        this.__defineGetter__("source", function() { return this.javaToJS(je.getSource()); });
        this.__defineGetter__("target", function() { return this.javaToJS(je.getTarget()); });
        this.__defineGetter__("targetName", function() { return __fixstring(je.getTargetName()); });
        this.__defineGetter__("type", function() { return __fixstring(je.getType()); });
        this.__defineGetter__("value", function() { return __fixstring(je.getValue()); });
        this.__defineGetter__("willCommit", function() { return je.getWillCommit(); });
        this.__defineGetter__("__javaevent", function() { return je; });

        this.__defineSetter__("change", function(x) { je.setChange(x); });
        this.__defineSetter__("rc", function(x) { je.setRc(x); });
        this.__defineSetter__("richChange", function(x) { return __fixstring(je.setRichChange(x)); });
        this.__defineSetter__("richChangeEx", function(x) { return __fixstring(je.setRichChangeEx(x)); });
        this.__defineSetter__("richValue", function(x) { return __fixstring(je.setRichValue(x)); });
        this.__defineSetter__("selStart", function(x) { return je.setSelStart(x); });
        this.__defineSetter__("selEnd", function(x) { return je.setSelEnd(x); });
        this.__defineSetter__("value", function(x) { return __fixstring(je.setValue(x)); });
    } else if (app.__engine=="mustang") {
        this.__get__ = function(key) {
            if (key=="change")  return __fixstring(je.getChange());
            if (key=="changeEx")  return __fixstring(je.getChangeEx());
            if (key=="commitKey")  return je.getCommitKey();
            if (key=="fieldFull")  return je.getFieldFull();
            if (key=="keyDown")  return je.getKeyDown();
            if (key=="modifier")  return je.getModifier();
            if (key=="name")  return __fixstring(je.getName());
            if (key=="rc")  return je.getRc();
            if (key=="richChange")  return __fixstring(je.getRichChange());
            if (key=="richChangeEx")  return __fixstring(je.getRichChangeEx());
            if (key=="richValue")  return __fixstring(je.getRichValue());
            if (key=="selStart")  return je.getSelStart();
            if (key=="selEnd")  return je.getSelEnd();
            if (key=="shift")  return je.getShift();
            if (key=="source")  return this.javaToJS(je.getSource());
            if (key=="target")  return this.javaToJS(je.getTarget());
            if (key=="targetName")  return __fixstring(je.getTargetName());
            if (key=="type")  return __fixstring(je.getType());
            if (key=="value")  return __fixstring(je.getValue());
            if (key=="willCommit")  return je.getWillCommit();
            if (key=="__javaevent") return je;
            return this[key];
        };
        this.__put__ = function(key, x) {
            if (key=="change") je.setChange(x);
            else if (key=="rc") je.setRc(x); 
            else if (key=="richChange") je.setRichChange(x); 
            else if (key=="richChangeEx") je.setRichChangeEx(x); 
            else if (key=="richValue") je.setRichValue(x); 
            else if (key=="selStart") je.setSelStart(x); 
            else if (key=="selEnd") je.setSelEnd(x); 
            else if (key=="value") je.setValue(x); 
            else this[key] = x;
        };
        this.__has__ = function(key) {
            return key in this || key=="change" || key=="changeEx" || key=="commitKey" || key=="fieldFull" || key=="keyDown" || key=="modifier" || key=="name" || key=="rc" || key=="richChange" || key=="richChangeEx" || key=="richValue" || key=="selStart" || key=="selEnd" || key=="shift" || key=="source" || key=="target" || key=="targetName" || key=="type" || key=="value" || key=="willCommit";
        };
        this.__getIds__ = function() {
            return [ "change", "changeEx", "commitKey", "fieldFull", "keyDown", "modifier", "name", "rc", "richChange", "richChangeEx", "richValue", "selStart", "selEnd", "shift", "source", "target", "targetName", "type", "value", "willCommit" ];
        }
    }

    this.javaToJS = function(obj) {
        if (obj instanceof Packages.org.faceless.pdf2.PDF) {
            for (var i=0;i<app.activeDocs.length;i++) {
                if (app.activeDocs[i].__pdf==obj) return app.activeDocs[i];
            }
            return undefined;
        } else if (obj instanceof Packages.org.faceless.pdf2.WidgetAnnotation) {
            var pdf = obj.getPage().getPDF();
            var annots = obj.getField().getAnnotations();
            var name = pdf.getForm().getName(obj.getField());
            if (annots.size()>1) {
                for (var i=0;i<annots.size();i++) {
                    if (obj==annots.get(i)) {
                        name += "."+i;
                        break;
                    }
                }
            }
            return this.javaToJS(pdf).getField(name);
        }
    };
    this.toString = function() {
        return "[object Event]";
    }
}

// field = FormElement object
// fieldname = Field name, excluding ".[\d+]" suffix for annotation
// annotnum = which annotation we refer to, or -1 for all
// doc = the Doc object
function Field(field, fieldname, annotnum, doc) {
    this.__element = field;
    if (doc.app.__engine=="rhino") {
        this.__defineGetter__("field", function() {
            return field;
        });

        this.__defineGetter__("alignment", function() {
            var a = this.__getAnnot().getTextStyle().getTextAlign();
            if (a==Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_RIGHT) {
                return "right";
            } else if (a==Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_CENTER) {
                return "center";
            } else {
                return "left";
            }
        });

        this.__defineSetter__("alignment", function(value) {
            this.__setAnnot(function() {
                var newstyle = this.getTextStyle().clone();
                if (value=="right") {
                    newstyle.setTextAlign(Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_RIGHT);
                    this.setTextStyle(newstyle);
                } else if (value=="center") {
                    newstyle.setTextAlign(Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_CENTER);
                    this.setTextStyle(newstyle);
                } else if (value=="left") {
                    newstyle.setTextAlign(Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_LEFT);
                    this.setTextStyle(newstyle);
                }
            });
        });
        this.__defineGetter__("strokeColor", function() {
            return doc.app.__bfo.colorFromAWT(this.__getAnnot().getBackgroundStyle().getLineColor());
        });
        this.__defineGetter__("borderColor", function() {
            return doc.app.__bfo.colorFromAWT(this.__getAnnot().getBackgroundStyle().getLineColor());
        });
        this.__defineSetter__("strokeColor", function(x) {
            this.__setAnnot(function() {
                var newstyle = this.getBackgroundStyle().clone();
                newstyle.setLineColor(doc.app.__bfo.colorToAWT(x));
                this.setBackgroundStyle(newstyle);
            });
        });
        this.__defineSetter__("borderColor", function(x) {
            this.__setAnnot(function() {
                var newstyle = this.getBackgroundStyle().clone();
                newstyle.setLineColor(doc.app.__bfo.colorToAWT(x));
                this.setBackgroundStyle(newstyle);
            });
        });
        this.__defineGetter__("borderStyle", function() {
            var a = this.__getAnnot().getBackgroundStyle().getFormStyle();
            if (a==Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_BEVEL) {
                return "beveled";
            } else if (a==Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_INSET) {
                return "inset";
            } else if (a==Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_UNDERLINE) {
                return "underline";
            } else if (this.__getAnnot().getBackgroundStyle().getLineDashPattern() != null) {
                return "dashed";
            } else {
                return "solid";
            }
        });
        this.__defineSetter__("borderStyle", function(x) {
            this.__setAnnot(function() {
                var newstyle = this.getBackgroundStyle().clone();
                if (value=="beveled") {
                    newstyle.setFormStyle(Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_BEVEL);
                } else if (value=="dashed") {
                    newstyle.setLineDash([3], 0);
                } else if (value=="inset") {
                    newstyle.setFormStyle(Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_INSET);
                } else if (value=="underline") {
                    newstyle.setFormStyle(Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_UNDERLINE);
                } else {
                    newstyle.setFormStyle(Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_SOLID);
                }
                this.setBackgroundStyle(newstyle);
            });
        });
        this.__defineGetter__("charLimit", function() {
            return this.type=="text" ?  field.getMaxLength() + field.getNumberOfCombs() : undefined;
        });
        this.__defineSetter__("charLimit", function(x) {
            if (this.type=="text") {
                if (field.getNumberOfCombs() > 0) {
                    field.setNumberOfCombs(x);
                } else {
                    field.setMaxLength(x);
                }
            }
        });
        this.__defineGetter__("comb", function() {
            return this.type=="text" && field.getNumberOfCombs() > 0;
        });
        this.__defineSetter__("comb", function(x) {
            if (this.type=="text") {
                if (x) {
                    field.setNumberOfCombs(field.getNumberOfCombs() + field.getMaxLength());
                } else {
                    field.setMaxLength(field.getNumberOfCombs() + field.getMaxLength());
                }
            }
        });
        this.__defineGetter__("defaultValue", function() {
            var ltype = this.type;
            var value = undefined;
            if (ltype=="text" || ltype=="combobox" || ltype=="radiobutton" || ltype=="listbox" || ltype=="checkbox") {
                var value = __fixstring(field.getDefaultValue());
                if (value=="" && (ltype=="checkbox" || ltype=="radiobutton")) {
                    value = "Off";
                }
            }
            return value;
        });
        this.__defineSetter__("defaultValue", function(x) {
            var ltype = this.type;
            if (ltype=="text" || ltype=="combobox" || ltype=="radiobutton" || ltype=="listbox" || ltype=="checkbox") {
               field.setDefaultValue(x);
            }
        });
        this.__defineGetter__("display", function() {
            return this.print ? (this.hidden ? doc.display.noView : doc.display.visible) : (this.hidden ? doc.display.hidden : doc.display.noPrint); 
        });
        this.__defineSetter__("display", function(x) {
            if (x==doc.display.visible) {
                this.hidden = false;
                this.print = true;
            } else if (x==doc.display.hidden) {
                this.hidden = true;
                this.print = false;
            } else if (x==doc.display.noPrint) {
                this.hidden = false;
                this.print = false;
            } else if (x==doc.display.noView) {
                this.hidden = true;
                this.print = true;
            }
        });
        this.__defineGetter__("doc", function() {
            return doc;
        });
        this.__defineGetter__("doNotScroll", function() {
            return this.type=="text" ? field.isScrollable() : undefined;
        });
        this.__defineSetter__("doNotScroll", function(x) {
            if (this.type=="text") field.setScrollable(x); 
        });
        this.__defineGetter__("editable", function() {
            return this.type=="combobox";
        });
        this.__defineGetter__("fillColor", function() {
            return doc.app.__bfo.colorFromAWT(this.__getAnnot().getBackgroundStyle().getFillColor());
        });
        this.__defineSetter__("fillColor", function(x) {
            this.__setAnnot(function() {
                var newstyle = this.getBackgroundStyle().clone();
                newstyle.setFillColor(doc.app.__bfo.colorToAWT(x));
                this.setBackgroundStyle(newstyle);
            });
        });
        this.__defineGetter__("hidden", function() {
            return !this.__getAnnot().isVisible();
        });
        this.__defineSetter__("hidden", function(x) {
            this.__setAnnot(function() { this.setVisible(!x); });
        });
        this.__defineGetter__("name", function() { return fieldname; });
        this.__defineGetter__("page", function() { return this.__getAnnot().getPage().getPageNumber(); });
        this.__defineGetter__("print", function() { return this.__getAnnot().isPrintable(); });
        this.__defineSetter__("print", function(x) { this.__setAnnot(function() { this.setPrintable(x); }); });
        this.__defineGetter__("readonly", function() { return field.isReadOnly(); });
        this.__defineSetter__("readonly", function(x) { field.setReadOnly(x); });
        this.__defineGetter__("rect", function() { var r = this.__getAnnot().getRectangle(); return [r[0], r[1], r[2], r[3]] });
        this.__defineSetter__("rect", function(x) { this.__getAnnot().setRectangle(x[0], x[1], x[2], x[3]); });
        this.__defineGetter__("required", function() { return field.isRequired(); });
        this.__defineSetter__("required", function(x) { field.setRequired(x); });
        this.__defineGetter__("textColor", function() {
            return doc.app.__bfo.colorFromAWT(this.__getAnnot().getTextStyle().getFillColor());
        });
        this.__defineSetter__("textColor", function(x) {
            this.__setAnnot(function() {
                var newstyle = this.getTextStyle().clone();
                newstyle.setFillColor(doc.app.__bfo.colorToAWT(x));
                this.setTextStyle(newstyle);
            });
        });
        this.__defineGetter__("textFont", function() { return this.__getAnnot().getTextStyle().getFont().getBaseName(); });
        this.__defineGetter__("type", function() {
            if (field instanceof Packages.org.faceless.pdf2.FormChoice) {
                if (field.getType()==Packages.org.faceless.pdf2.FormChoice.TYPE_COMBO) {
                    return "combobox";
                } else {
                    return "listbox";
                }
            } else if (field instanceof Packages.org.faceless.pdf2.FormText) {
                return "text";
            } else if (field instanceof Packages.org.faceless.pdf2.FormButton) {
                return "button";
            } else if (field instanceof Packages.org.faceless.pdf2.FormRadioButton) {
                return "radiobutton";
            } else if (field instanceof Packages.org.faceless.pdf2.FormCheckbox) {
                return "checkbox";
            } else if (field instanceof Packages.org.faceless.pdf2.FormSignature) {
                return "signature";
            } else {
                return undefined;
            }
        });
        this.__defineGetter__("userName", function() {
            return __fixstring(field.getDescription());
        });
        this.__defineSetter__("userName", function(x) {
            field.setDescription(x);
        });
        this.__defineGetter__("value", function() {
            // TODO should be String, Date or Number, but when?
            return this.valueAsString;
        });
        this.__defineSetter__("value", function(x) {
            field.setValue(x);
        });
        this.__defineGetter__("valueAsString", function() {
            var val = __fixstring(field.getValue());
            var ltype = this.type;
            if (val=="" && (ltype=="checkbox" || ltype=="radiobutton")) {
                val = "Off";
            }
            return val;
        });
    } else if (doc.app.__engine=="mustang") {
        this.__get__ = function(key) {
            if (key=="alignment") {
                var a = this.__getAnnot().getTextStyle().getTextAlign();
                if (a==Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_RIGHT) {
                    return "right";
                } else if (a==Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_CENTER) {
                    return "center";
                } else {
                    return "left";
                }
            } else if (key=="borderColor" || key=="strokeColor") {
                return doc.app.__bfo.colorFromAWT(this.__getAnnot().getBackgroundStyle().getLineColor());
            } else if (key=="borderStyle") {
                var a = this.__getAnnot().getBackgroundStyle().getFormStyle();
                if (a==Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_BEVEL) {
                    return "beveled";
                } else if (a==Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_INSET) {
                    return "inset";
                } else if (a==Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_UNDERLINE) {
                    return "underline";
                } else if (this.__getAnnot().getBackgroundStyle().getLineDashPattern() != null) {
                    return "dashed";
                } else {
                    return "solid";
                }
            } else if (key=="charLimit") {
                return this.__get__("type")=="text" ?  field.getMaxLength() + field.getNumberOfCombs() : undefined;
            } else if (key=="comb") {
                return this.__get__("type")=="text" && field.getNumberOfCombs() > 0;
            } else if (key=="defaultValue") {
                var ltype = this.__get__("type");
                var value = undefined;
                if (ltype=="text" || ltype=="combobox" || ltype=="radiobutton" || ltype=="listbox" || ltype=="checkbox") {
                    var value = __fixstring(field.getDefaultValue());
                    if (value=="" && (ltype=="checkbox" || ltype=="radiobutton")) {
                        value = "Off";
                    }
                }
                return value;
            } else if (key=="display") {
                var print = this.__get__("print");
                var hidden = this.__get__("hidden");
                return print ? (hidden ? doc.display.noView : doc.display.visible) : (hidden ? doc.display.hidden : doc.display.noPrint);
            } else if (key=="doc") {
                return doc;
            } else if (key=="doNotScroll") {
                return this.__get__("type")=="text" ? field.isScrollable() : undefined;
            } else if (key=="editable") {
                return this.__get__("type")=="combobox";
            } else if (key=="fillColor") {
                return doc.app.__bfo.colorFromAWT(this.__getAnnot().getBackgroundStyle().getFillColor());
            } else if (key=="hidden") {
                return !this.__getAnnot().isVisible();
            } else if (key=="name") {
                return fieldname;
            } else if (key=="page") {
                return this.__getAnnot().getPage().getPageNumber();
            } else if (key=="print") {
                return this.__getAnnot().isPrintable();
            } else if (key=="readonly") {
                return field.isReadOnly();
            } else if (key=="rect") {
                var r = this.__getAnnot().getRectangle();
                return [r[0], r[1], r[2], r[3]];
            } else if (key=="required") {
                return field.isRequired();
            } else if (key=="textColor") {
                return doc.app.__bfo.colorFromAWT(this.__getAnnot().getTextStyle().getFillColor());
            } else if (key=="textFont") {
                return this.__getAnnot().getTextStyle().getFont().getBaseName();
            } else if (key=="type") {
                if (field instanceof Packages.org.faceless.pdf2.FormChoice) {
                    if (field.getType()==Packages.org.faceless.pdf2.FormChoice.TYPE_COMBO) {
                        return "combobox";
                    } else {
                        return "listbox";
                    }
                } else if (field instanceof Packages.org.faceless.pdf2.FormText) {
                    return "text";
                } else if (field instanceof Packages.org.faceless.pdf2.FormButton) {
                    return "button";
                } else if (field instanceof Packages.org.faceless.pdf2.FormRadioButton) {
                    return "radiobutton";
                } else if (field instanceof Packages.org.faceless.pdf2.FormCheckbox) {
                    return "checkbox";
                } else if (field instanceof Packages.org.faceless.pdf2.FormSignature) {
                    return "signature";
                } else {
                    return undefined;
                }
            } else if (key=="userName") {
                return __fixstring(field.getDescription());
            } else if (key=="value") {
                // TODO should be String, Date or Number, but when?
                return this.__get__("valueAsString");
            } else if (key=="valueAsString") {
                var val = __fixstring(field.getValue());
                var ltype = this.__get__("type");
                if (val=="" && (ltype=="checkbox" || ltype=="radiobutton")) {
                    val = "Off";
                }
                return val;
            } else {
                return this[key];
            }
        };
        this.__put__ = function(key, value) {
            if (key=="alignment") {
                this.__setAnnot(function() {
                    var newstyle = this.getTextStyle().clone();
                    if (value=="right") {
                        newstyle.setTextAlign(Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_RIGHT);
                        this.setTextStyle(newstyle);
                    } else if (value=="center") {
                        newstyle.setTextAlign(Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_CENTER);
                        this.setTextStyle(newstyle);
                    } else if (value=="left") {
                        newstyle.setTextAlign(Packages.org.faceless.pdf2.PDFStyle.TEXTALIGN_LEFT);
                        this.setTextStyle(newstyle);
                    }
                });
            } else if (key=="borderColor" || key=="strokeColor") {
                this.__setAnnot(function() {
                    var newstyle = this.getBackgroundStyle().clone();
                    newstyle.setLineColor(doc.app.__bfo.colorToAWT(value));
                    this.setBackgroundStyle(newstyle);
                });
            } else if (key=="borderStyle") {
                this.__setAnnot(function() {
                    var newstyle = this.getBackgroundStyle().clone();
                    if (value=="beveled") {
                        newstyle.setFormStyle(Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_BEVEL);
                    } else if (value=="dashed") {
                        newstyle.setLineDash([3], 0);
                    } else if (value=="inset") {
                        newstyle.setFormStyle(Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_INSET);
                    } else if (value=="underline") {
                        newstyle.setFormStyle(Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_UNDERLINE);
                    } else {
                        newstyle.setFormStyle(Packages.org.faceless.pdf2.PDFStyle.FORMSTYLE_SOLID);
                    }
                    this.setBackgroundStyle(newstyle);
                });
            } else if (key=="hidden") {
                this.__setAnnot(function() { this.setVisible(!value); });
            } else if (key=="charLimit") {
                if (this.__get__("type")=="text") {
                    if (field.getNumberOfCombs() > 0) {
                        field.setNumberOfCombs(value);
                    } else {
                        field.setMaxLength(value);
                    }
                }
            } else if (key=="comb") {
                if (this.__get__("type")=="text") {
                    if (value) {
                        field.setNumberOfCombs(field.getMaxLength() + field.getNumberOfCombs());
                    } else {
                        field.setMaxLength(field.getMaxLength() + field.getNumberOfCombs());
                    }
                }
            } else if (key=="defaultValue") {
                var ltype = this.__get__("type");
                if (ltype=="text" || ltype=="combobox" || ltype=="radiobutton" || ltype=="listbox" || ltype=="checkbox") {
                    field.setDefaultValue(value);
                }
            } else if (key=="display") {
                if (value==doc.display.visible) {
                    this.__put__("hidden", false);
                    this.__put__("printable", true);
                } else if (value==doc.display.hidden) {
                    this.__put__("hidden", true);
                    this.__put__("printable", false);
                } else if (value==doc.display.noPrint) {
                    this.__put__("hidden", false);
                    this.__put__("printable", false);
                } else if (value==doc.display.noView) {
                    this.__put__("hidden", true);
                    this.__put__("printable", true);
                }
            } else if (key=="doNotScroll") {
                if (this.__get__("type")=="text") field.setScrollable(value);
            } else if (key=="fillColor") {
                this.__setAnnot(function() {
                    var newstyle = this.getBackgroundStyle().clone();
                    newstyle.setFillColor(doc.app.__bfo.colorToAWT(value));
                    this.setBackgroundStyle(newstyle);
                });
            } else if (key=="hidden") {
                this.__setAnnot(function() { this.setVisible(!value); });
            } else if (key=="print") {
                this.__setAnnot(function() { this.setPrintable(value); });
            } else if (key=="readonly") {
                field.setReadOnly(value);
            } else if (key=="rect") {
                this.__getAnnot().setRectangle(value[0], value[1], value[2], value[3]);
            } else if (key=="required") {
                field.setRequired(value);
            } else if (key=="textColor") {
                this.__setAnnot(function() {
                    var newstyle = this.getTextStyle().clone();
                    newstyle.setFillColor(doc.app.__bfo.colorToAWT(value));
                    this.setTextStyle(newstyle);
                });
            } else if (key=="hidden") {
            } else if (key=="userName") {
                field.setDescription(value);
            } else if (key=="value") {
                field.setValue(value);
            } else {
                this[key] = value;
            }
        };
        this.__has__ = function(key) {
            return key in this || typeof(this['get'+key]) == 'function';
        };
        this.__delete__ = function(key) {
            delete this[key];
        };
    }

    this.getArray = function() {
        var a = new Array(), i = 0;
        var form = field.getForm();
        var j = form.getElements().keySet().iterator();
        while (j.hasNext()) {
            var name = j.next();
            var f = form.getElement(name);
            if (name != null && (name == fieldname || name.startsWith(fieldname + "."))) {
                var annotnum = -1;
                var result;
                if (f==null && (result=name.match(/\.([0-9]+)$/))) {
                    annotnum = result[1];
                    name = name.replace(/\.[0-9]+$/, "");
                    f = form.getElement(name);
                }
                if (f != null) {
                    var newfield = new Field(f, name, annotnum, doc);
                    if (doc.app.__engine=="mustang") {
                        newfield = new JSAdapter(newfield);
                    }
                    a[i++] = newfield;
                }
            }
        }
        return a;
    };

    this.checkThisBox = function(nWidget, bCheckIt) {
        if (bCheckIt==undefined) bCheckIt=true;
        var ltype = this.type;
        if (ltype=="checkbox" || (bCheckIt && ltype=="radiobutton")) {
            var annots = field.getAnnotations();
            if (nWidget < annots.size()) {
                annots.get(nWidget).setSelected(bCheckIt);
            }
        }
    };

    this.defaultIsChecked = function(nWidth, bCheckIt) {
        if (bCheckIt==undefined) bCheckIt=true;
        var ltype = this.type;
        if (ltype=="checkbox" || (bCheckIt && ltype=="radiobutton")) {
            var annots = field.getAnnotations();
            if (nWidget < annots.size()) {
                // TODO should properly handle multiple identical export values here
                field.setDefaultValue(annots.get(nWidget).getValue());
            }
        }
    };

    this.isBoxChecked = function(nWidget) {
        var ltype = this.type;
        if (ltype=="checkbox" || ltype=="radiobutton") {
            var annots = field.getAnnotations();
            if (nWidget < annots.size()) {
                return annots.get(nWidget).isSelected();
            }
        }
        return false;
    };

    this.isDefaultChecked = function() {
        var ltype = this.type;
        if (ltype=="checkbox" || ltype=="radiobutton") {
            var annots = field.getAnnotations();
            if (nWidget < annots.size()) {
                // TODO should properly handle multiple identical export values here
                return annots.getValue()==field.getDefaultValue();
            }
        }
        return false;
    };

    this.setAction = function(cTrigger, cScript) {
        var target, event;
        if (cTrigger=="MouseUp") {
            target = this.__getAnnot();
            event = Packages.org.faceless.pdf2.Event.MOUSEUP;
        } else if (cTrigger=="MouseDown") {
            target = this.__getAnnot();
            event = Packages.org.faceless.pdf2.Event.MOUSEDOWN;
        } else if (cTrigger=="MouseEnter") {
            target = this.__getAnnot();
            event = Packages.org.faceless.pdf2.Event.MOUSEOVER;
        } else if (cTrigger=="MouseExit") {
            target = this.__getAnnot();
            event = Packages.org.faceless.pdf2.Event.MOUSEOUT;
        } else if (cTrigger=="OnFocus") {
            target = this.__getAnnot();
            event = Packages.org.faceless.pdf2.Event.ONFOCUS;
        } else if (cTrigger=="OnBlur") {
            target = this.__getAnnot();
            event = Packages.org.faceless.pdf2.Event.ONBLUR;
        } else if (cTrigger=="Keystroke") {
            target = field;
            event = Packages.org.faceless.pdf2.Event.KEYSTROKE;
        } else if (cTrigger=="Validate") {
            target = field;
            event = Packages.org.faceless.pdf2.Event.CHANGE;
        } else if (cTrigger=="Calculate") {
            target = field;
            event = Packages.org.faceless.pdf2.Event.OTHERCHANGE;
        } else if (cTrigger=="Format") {
            target = field;
            event = Packages.org.faceless.pdf2.Event.FORMAT;
        }
        if (target) {
            var action = !cScript ? null : Packages.org.faceless.pdf2.PDFAction.formJavaScript(cScript);
            target.setAction(event, action);
        }
    };

    this.setFocus = function() {
        var component = doc.app.__bfo.getComponent(this.__getAnnot());
        if (component!=null) {
            component.requestFocusInWindow();
        }
    };

    // This method can be called on a "Field" object to get the specific
    // annotation we refer to.
    this.__getAnnot = function() {
        return field.getAnnotation(annotnum<0 ? 0 : annotnum);
    };

    // This method can be called on a "Field" object to set the property
    // for the annotation(s) we refer to. x = function to run on the annot(s).
    this.__setAnnot = function(x) {
        var all = field.getAnnotations();
        for (var i=0;i<all.size();i++) {
            if (annotnum<0 || i==annotnum) {
                x.apply(all.get(i));
            }
        }
    };

    this.toString = function() {
        return "[object Field]";
    }

    this.buttonGetIcon = function(nFace) {
        if (nFace==undefined) nFace=0;
        if (nFace==0) {
            return this.__getAnnot().getButtonImage();
        } else {
            return undefined;
        }
    };

    this.buttonSetIcon = function(oIcon, nFace) {
        if (nFace==undefined) nFace=0;
        if (nFace==0) {
            this.__getAnnot().setButtonImage(oIcon, 'A', false, 0, 0);
        }
    };
}
