/*
  The contents of this file are subject to the Sun Public License
	Version 1.0 (the "License"); you may not use this file except in
	compliance with the License. A copy of the License is available at
	http://www.sun.com/

	The Original Code is winlaf. The Initial Developer of the
	Original Code is Brian Duff. Portions created by Brian Duff are Copyright
	(C)Brian Duff. All Rights Reserved.

	Contributor(s): ______________________________________.
*/
package net.java.plaf;

import java.awt.Toolkit;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.StringTokenizer;

import javax.swing.UIManager;

import com.sun.java.swing.plaf.windows.WindowsLookAndFeel;

/**
 * Utility class representing the current environment. This is used by
 * LookAndFeelPatch implementations to decide whether the look and feel 
 * needs to be patched.
 * 
 * @author Brian.Duff@oracle.com
 */
public final class Environment 
{
  private final boolean m_isWindowsLookAndFeel;
  private final boolean m_isWindowsXPThemed;
  private final JavaVersion m_javaVersion;
  private final boolean m_isWindowsXP;
  
  // Bug we should change this if desktop properties change...
  private final boolean m_is32bitColor;

  public static final JavaVersion JAVA_1_4_0 = new JavaVersion( "1.4.0" ); //$NON-NLS-1$
  public static final JavaVersion JAVA_1_4_2 = new JavaVersion( "1.4.2" ); //$NON-NLS-1$


  public Environment()
  {
    m_isWindowsLookAndFeel = 
      ( UIManager.getLookAndFeel() instanceof WindowsLookAndFeel );
    m_isWindowsXPThemed = 
      isWindowsXPThemedImpl();
    m_javaVersion = new JavaVersion( System.getProperty( "java.version" ) ); //$NON-NLS-1$
    
    // Hmm returns 24 on my 32bit color system. Need to check if this is a 
    // java bug.
    m_is32bitColor = 
      Toolkit.getDefaultToolkit().getColorModel().getPixelSize() == 24;
      
    m_isWindowsXP = System.getProperty( "os.name" ).indexOf( "XP" ) > 0; //$NON-NLS-1$ //$NON-NLS-2$
  }
  

  // Test code... should move into a JUnit test.
  public static void main( String[] args )
  {
    Environment env = new Environment();
    
    boolean atLeast13 = env.isJavaVersionAtLeast( "1.3.0" ); //$NON-NLS-1$
    System.out.println( "true? "+atLeast13 ); //$NON-NLS-1$

    boolean atLeast14 = env.isJavaVersionAtLeast( "1.4.0" ); //$NON-NLS-1$
    System.out.println( "true? "+atLeast14 ); //$NON-NLS-1$

    boolean atLeast142 = env.isJavaVersionAtLeast( "1.4.2" ); //$NON-NLS-1$
    System.out.println( "true? "+atLeast142 ); //$NON-NLS-1$

    boolean atLeast143 = env.isJavaVersionAtLeast( "1.4.3" ); //$NON-NLS-1$
    System.out.println( "false? "+atLeast143 ); //$NON-NLS-1$

    boolean atLeast15 = env.isJavaVersionAtLeast( "1.5.0" ); //$NON-NLS-1$
    System.out.println( "false? "+atLeast15 ); //$NON-NLS-1$
  }
  
  /**
   * Is the current look and feel Windows?
   */
  public boolean isWindowsLookAndFeel()
  {
    return m_isWindowsLookAndFeel;
  }
  
  /**
   * Is the current operating system Windows XP?
   */
  public boolean isWindowsXP()
  {
    return m_isWindowsXP;
  }
  
  public boolean isWindowsXPThemed()
  {
    return m_isWindowsXPThemed;
  }
  
  /**
   * Is the current look and feel capable of doing alpha effects? Returns true
   * iff isWindowsXPThemed() and the color depth is 32 bit.
   * 
   * 
   */
  public boolean isAlphaIconCapable()
  {
    return isWindowsXPThemed() && m_is32bitColor;
  }
  
  /**
   * Returns true if the current version of Java is the specified version or
   * a later version. You can pass an arbitrary version string in, e.g.
   * "1.3.1_02", or use one of the contstants in this class to identify a 
   * public release of Java.
   * 
   * @param version
   * @return 
   */
  public boolean isJavaVersionAtLeast( String version )
  {
    return isJavaVersionAtLeast( new JavaVersion( version ) );

  }

  public boolean isJavaVersionAtLeast( JavaVersion version )
  {
    return ( m_javaVersion.compareTo( version ) >= 0 );    
  }

  /**
   * Is the java version less than the specified version?
   * @param version
   * @return 
   */
  public boolean isJavaVersionLessThan( String version )
  {
    return isJavaVersionLessThan( new JavaVersion( version ) );
  }
  
  public boolean isJavaVersionLessThan( JavaVersion version )
  {
    return ( m_javaVersion.compareTo( version ) < 0 );
  }
  
  /**
   * Are we running in XP-themed mode?
   */
  private boolean isWindowsXPThemedImpl()
  {
    // Use desktop properties to find out...
    Toolkit toolkit = Toolkit.getDefaultToolkit();
	  Boolean themeActive = 
      (Boolean)toolkit.getDesktopProperty("win.xpstyle.themeActive"); //$NON-NLS-1$
	  if (themeActive == null) 
    {
      return false;
	  }
    
    if ( themeActive.booleanValue() && 
      System.getProperty( "swing.noxp" ) == null ) //$NON-NLS-1$
    {
      return true;
    }

    return false;
  }
  
  /**
   * Returns true if the specified bug number is known to be present in this
   * version of Java.
   * 
   * @param sunBugNumber
   * @return true if the specified bug number is known to be present in this
   *    version of Java.
   */
  public boolean hasBug( int sunBugNumber )
  {
    return false;
  }
  
  
  public final static class JavaVersion implements Comparable
  {
    ArrayList parts = new ArrayList( 4 );
    
    JavaVersion( String versionString )
    {
      StringTokenizer tok = new StringTokenizer( versionString, "._-" ); //$NON-NLS-1$
    
      while ( tok.hasMoreTokens() )
      {
        String token = tok.nextToken();
        
        try
        {
          int number = Integer.parseInt( token );
          parts.add( new Integer( number ) );
        }
        catch ( Exception e )
        {
          parts.add( token );
        }
      }
    }
    
    public int compareTo(Object o)
    {
      JavaVersion that = (JavaVersion) o;
      
      int partIndex = 0;
      for ( Iterator i = this.parts.iterator(); i.hasNext(); )
      {
        Object thisPart = i.next();
        Object thatPart = that.parts.get( partIndex );
        
        partIndex++;
        
        if ( thisPart instanceof Integer )
        {
          if ( thatPart instanceof Integer )
          {
            int difference = ((Integer)thisPart).intValue() - 
                             ((Integer)thatPart).intValue();
            if ( difference != 0 )
            {
              return difference;
            }
            // Else, we let the loop go round again to see if a later part 
            // is different
          }
        }
      }
      
      // If we get here, the two versions are maybe identical. could
      // have 1.4.1-beta and 1.4.1 though, so check for the existance of a 
      // final String part. If present, the presence of this part makes the
      // version earlier than the other.
      Object thatLastPart = that.parts.get( that.parts.size() -1 );
      Object thisLastPart = this.parts.get( this.parts.size() -1 );
      
      if ( thatLastPart instanceof String && thisLastPart instanceof String )
      {
        if ( thatLastPart.equals( thisLastPart ) )
        {
          return 0;
        }
        // Only know they're different, can't tell if one is older or later
        // than the other without some extra logic here.
        return -1;
      }
      else if ( thatLastPart instanceof String )
      {
        // OK this < that
        return -1;
      }
      else
      {
        return 1;
      }
    }
    
    
  }
}