/*
  The contents of this file are subject to the Sun Public License
	Version 1.0 (the "License"); you may not use this file except in
	compliance with the License. A copy of the License is available at
	http://www.sun.com/

	The Original Code is winlaf. The Initial Developer of the
	Original Code is Gerhard Leonhartsberger. Portions created by 
	Gerhard Leonhartsberger are Copyright(C) Gerhard Leonhartsberger. 
	All Rights Reserved.

	Contributor(s): ______________________.
*/
package net.java.plaf.windows.xp;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.KeyboardFocusManager;
import java.awt.Rectangle;
import java.awt.Window;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JTree;
import javax.swing.UIManager;
import javax.swing.plaf.ColorUIResource;
import javax.swing.plaf.FontUIResource;
import javax.swing.tree.TreeCellRenderer;

/**
 * Class <code>DefaultTreeCellREnderer</code> fixes problem when a 
 * <code>JTree</code> looses focus then the selected node contained by the
 * <code>JTree</code> is not drawn in "focus lost selection color".
 * 
 * It was necessary to re-implemenent 
 * <code>javax.swing.tree.DefaultTreeCellRenderer</code> because there was no
 * way to change rendering in order to consider focus lost issue.
 *
 * The following issues is fixed:
 * <ul>
 *  <li><b>Issue 27:</b> Selected nodes contained by a <code>JTree</code> 
 *                       remains selected highlighted when <code>JTree</code> 
 *                       looses focus.
 *  <li><b>Issue 30:</b> A selected tree node does not change selection color 
 *                       when the <code>JFrame</code> where the tree is added 
 *                       looses focus.</li>
 * </ul>
 * 
 * @author Gerhard Leonhartsberger
 */
public class DefaultTreeCellRenderer
	extends JLabel
	implements TreeCellRenderer {

	/** Last tree the renderer was painted in. */
	protected JTree tree;

	/** Is the value currently selected. */
	protected boolean selected;

	/** True if has focus. */
	protected boolean hasFocus;

	/** True if draws focus border around icon as well. */
	protected boolean drawsFocusBorderAroundIcon;

	// Icons
	/** Icon used to show non-leaf nodes that aren't expanded. */
	transient protected Icon closedIcon;

	/** Icon used to show leaf nodes. */
	transient protected Icon leafIcon;

	/** Icon used to show non-leaf nodes that are expanded. */
	transient protected Icon openIcon;

	// Colors
	/** Color to use for the foreground for selected nodes. */
	protected Color textSelectionColor;

	/** Color to use for the foreground for non-selected nodes. */
	protected Color textNonSelectionColor;

	/** Color to use for the background when a node is selected. */
	protected Color backgroundSelectionColor;

	/** Color to use for the background when the node isn't selected. */
	protected Color backgroundNonSelectionColor;

	/** Color to use for the background when the node is seleced but has not 
	 *  focus. */
	protected Color backgroundNoFocusSelectionColor;

	/** Color to use for the focus indicator when the node has focus. */
	protected Color borderSelectionColor;

	/**
	  * Returns a new instance of DefaultTreeCellRenderer.  Alignment is
	  * set to left aligned. Icons and text color are determined from the
	  * UIManager.
	  */
	public DefaultTreeCellRenderer() {

		setHorizontalAlignment(JLabel.LEFT);
		setLeafIcon(UIManager.getIcon("Tree.leafIcon")); //$NON-NLS-1$
		setClosedIcon(UIManager.getIcon("Tree.closedIcon")); //$NON-NLS-1$
		setOpenIcon(UIManager.getIcon("Tree.openIcon")); //$NON-NLS-1$
		setTextSelectionColor(UIManager.getColor("Tree.selectionForeground")); //$NON-NLS-1$
		setTextNonSelectionColor(UIManager.getColor("Tree.textForeground")); //$NON-NLS-1$
		setBackgroundSelectionColor(UIManager.getColor("Tree.selectionBackground")); //$NON-NLS-1$
		setBackgroundNonSelectionColor(UIManager.getColor("Tree.textBackground")); //$NON-NLS-1$
		setBorderSelectionColor(UIManager.getColor("Tree.selectionBorderColor")); //$NON-NLS-1$
		setBackgroundNoFocusSelectionColor(UIManager.getColor("Label.background")); //$NON-NLS-1$
		Object value = UIManager.get("Tree.drawsFocusBorderAroundIcon"); //$NON-NLS-1$
		drawsFocusBorderAroundIcon =
			(value != null && ((Boolean) value).booleanValue());
	}

	/**
	  * Returns the default icon, for the current laf, that is used to
	  * represent non-leaf nodes that are expanded.
	  */
	public Icon getDefaultOpenIcon() {
		return UIManager.getIcon("Tree.openIcon"); //$NON-NLS-1$
	}

	/**
	  * Returns the default icon, for the current laf, that is used to
	  * represent non-leaf nodes that are not expanded.
	  */
	public Icon getDefaultClosedIcon() {
		return UIManager.getIcon("Tree.closedIcon"); //$NON-NLS-1$
	}

	/**
	  * Returns the default icon, for the current laf, that is used to
	  * represent leaf nodes.
	  */
	public Icon getDefaultLeafIcon() {
		return UIManager.getIcon("Tree.leafIcon"); //$NON-NLS-1$
	}

	/**
	  * Sets the icon used to represent non-leaf nodes that are expanded.
	  */
	public void setOpenIcon(Icon newIcon) {
		openIcon = newIcon;
	}

	/**
	  * Returns the icon used to represent non-leaf nodes that are expanded.
	  */
	public Icon getOpenIcon() {
		return openIcon;
	}

	/**
	  * Sets the icon used to represent non-leaf nodes that are not expanded.
	  */
	public void setClosedIcon(Icon newIcon) {
		closedIcon = newIcon;
	}

	/**
	  * Returns the icon used to represent non-leaf nodes that are not
	  * expanded.
	  */
	public Icon getClosedIcon() {
		return closedIcon;
	}

	/**
	  * Sets the icon used to represent leaf nodes.
	  */
	public void setLeafIcon(Icon newIcon) {
		leafIcon = newIcon;
	}

	/**
	  * Returns the icon used to represent leaf nodes.
	  */
	public Icon getLeafIcon() {
		return leafIcon;
	}

	/**
	  * Sets the color the text is drawn with when the node is selected.
	  */
	public void setTextSelectionColor(Color newColor) {
		textSelectionColor = newColor;
	}

	/**
	  * Returns the color the text is drawn with when the node is selected.
	  */
	public Color getTextSelectionColor() {
		return textSelectionColor;
	}

	/**
	  * Sets the color the text is drawn with when the node isn't selected.
	  */
	public void setTextNonSelectionColor(Color newColor) {
		textNonSelectionColor = newColor;
	}

	/**
	  * Returns the color the text is drawn with when the node isn't selected.
	  */
	public Color getTextNonSelectionColor() {
		return textNonSelectionColor;
	}

	/**
	  * Sets the color to use for the background if node is selected.
	  */
	public void setBackgroundSelectionColor(Color newColor) {
		backgroundSelectionColor = newColor;
	}

	/**
	  * Returns the color to use for the background if node is selected.
	  */
	public Color getBackgroundSelectionColor() {
		return backgroundSelectionColor;
	}

	/**
	  * Sets the background color to be used for non selected nodes.
	  */
	public void setBackgroundNonSelectionColor(Color newColor) {
		backgroundNonSelectionColor = newColor;
	}

	/**
	  * Returns the background color to be used for non selected nodes.
	  */
	public Color getBackgroundNonSelectionColor() {
		return backgroundNonSelectionColor;
	}

	/**
	  * Sets the color to use for the border.
	  */
	public void setBorderSelectionColor(Color newColor) {
		borderSelectionColor = newColor;
	}

	/**
	  * Returns the color the border is drawn.
	  */
	public Color getBorderSelectionColor() {
		return borderSelectionColor;
	}

	/**
	 * Sets the color to use for selected nodes that have no focus.
	 *  
	 * @param newColor the new color
	 */
	public void setBackgroundNoFocusSelectionColor(Color newColor) {
		backgroundNoFocusSelectionColor = newColor;
	}

	/**
	 * Returns the color to use for selected nodes that have no focus.
	 * 
	 * @return the color to use for selected nodes that have no focus.
	 */
	public Color getBackgroundNoFocusSelectionColor() {
		return backgroundNoFocusSelectionColor;
	}

	/**
	 * Subclassed to map <code>FontUIResource</code>s to null. If 
	 * <code>font</code> is null, or a <code>FontUIResource</code>, this
	 * has the effect of letting the font of the JTree show
	 * through. On the other hand, if <code>font</code> is non-null, and not
	 * a <code>FontUIResource</code>, the font becomes <code>font</code>.
	 */
	public void setFont(Font font) {
		if (font instanceof FontUIResource)
			font = null;
		super.setFont(font);
	}

	/**
	 * Gets the font of this component.
	 * @return this component's font; if a font has not been set
	 * for this component, the font of its parent is returned
	 */
	public Font getFont() {
		Font font = super.getFont();
		if (font == null && tree != null) {
			// Strive to return a non-null value, otherwise the html support
			// will typically pick up the wrong font in certain situations.
			font = tree.getFont();
		}
		return font;
	}

	/**
	 * Subclassed to map <code>ColorUIResource</code>s to null. If 
	 * <code>color</code> is null, or a <code>ColorUIResource</code>, this
	 * has the effect of letting the background color of the JTree show
	 * through. On the other hand, if <code>color</code> is non-null, and not
	 * a <code>ColorUIResource</code>, the background becomes
	 * <code>color</code>.
	 */
	public void setBackground(Color color) {
		if (color instanceof ColorUIResource)
			color = null;
		super.setBackground(color);
	}

	/**
	  * Configures the renderer based on the passed in components.
	  * The value is set from messaging the tree with
	  * <code>convertValueToText</code>, which ultimately invokes
	  * <code>toString</code> on <code>value</code>.
	  * The foreground color is set based on the selection and the icon
	  * is set based on on leaf and expanded.
	  */
	public Component getTreeCellRendererComponent(
		JTree tree,
		Object value,
		boolean selected,
		boolean expanded,
		boolean leaf,
		int row,
		boolean hasFocus) {

		this.tree = tree;
		this.hasFocus = hasFocus;
		this.selected = selected;
		// set the text for the renderer
		String textValue =
			convertValueToText(value, selected, expanded, leaf, row, hasFocus);
		setText(textValue);
		// set enablement of the renderer
		setEnabled(tree.isEnabled());
		// set icon for the renderer.
		Icon icon =
			convertValueToIcon(value, selected, expanded, leaf, row, hasFocus);
		if (!tree.isEnabled()) {
			setDisabledIcon(icon);
		}
		else {
			setIcon(icon);
		}
		// set the text color of the node depending on the selection.
		boolean focusPermanentlyLost = isFocusPermanentlyLost();
		if ((selected && hasFocus)
			|| (selected && !hasFocus && !focusPermanentlyLost)) {
			setForeground(getTextSelectionColor());
		}
		else {
			setForeground(getTextNonSelectionColor());
		}
		setComponentOrientation(tree.getComponentOrientation());
		return this;
	}

	/**
	 * @param value
	 * @param sel
	 * @param expanded
	 * @param leaf
	 * @param row
	 * @param hasFocus
	 * @return
	 */
	protected String convertValueToText(
		Object value,
		boolean sel,
		boolean expanded,
		boolean leaf,
		int row,
		boolean hasFocus) {
		return tree.convertValueToText(
			value,
			sel,
			expanded,
			leaf,
			row,
			hasFocus);
	}

	/**
	 * @param value
	 * @param sel
	 * @param expanded
	 * @param leaf
	 * @param row
	 * @param hasFocus
	 * @return
	 */
	protected Icon convertValueToIcon(
		Object value,
		boolean sel,
		boolean expanded,
		boolean leaf,
		int row,
		boolean hasFocus) {
		if (value == null) {
			if (leaf) {
				return getDefaultLeafIcon();
			}
			else {
				return getDefaultOpenIcon();
			}
		}
		// if we do not have a known Resource object we deliver the default
		// icons.
		if (leaf) {
			return getLeafIcon();
		}
		else {
			if (expanded) {
				return getOpenIcon();
			}
			else {
				return getClosedIcon();
			}
		}
	}

	/**
	  * Paints the value.  The background is filled based on selected.
	  */
	public void paint(Graphics g) {

		Color bColor;

		// Now we determine if we lost focus permanently or just temporarily.
		// This is done by asking KeyboardFocusManager if the permanent focus
		// owner is the same component as the jtree. if that is the case
		// we did not loose the focus permanently but temporarily.
		// We need this in order to draw the selection of the selected node
		// in different colors depending on permanently or temporarily loosing
		// the focus. 
		boolean weLostFocusPermanently = isFocusPermanentlyLost();
		if ((selected && hasFocus)
			|| (selected && !hasFocus && !weLostFocusPermanently)) {
			bColor = getBackgroundSelectionColor();
		}
		else if (selected && !hasFocus && weLostFocusPermanently) {
			bColor = getBackgroundNoFocusSelectionColor();
		}
		else {
			bColor = getBackgroundNonSelectionColor();
			if (bColor == null)
				bColor = getBackground();
		}
		int imageOffset = -1;
		if (bColor != null) {
			imageOffset = getLabelStart();
			g.setColor(bColor);
			if (getComponentOrientation().isLeftToRight()) {
				g.fillRect(
					imageOffset,
					0,
					getWidth() - 1 - imageOffset,
					getHeight());
			}
			else {
				g.fillRect(0, 0, getWidth() - 1 - imageOffset, getHeight());
			}
		}
		if (hasFocus) {
			if (drawsFocusBorderAroundIcon) {
				imageOffset = 0;
			}
			else if (imageOffset == -1) {
				imageOffset = getLabelStart();
			}
			Color bsColor = getBorderSelectionColor();
			if (bsColor != null) {
				g.setColor(bsColor);
				if (getComponentOrientation().isLeftToRight()) {
					g.drawRect(
						imageOffset,
						0,
						getWidth() - 1 - imageOffset,
						getHeight() - 1);
				}
				else {
					g.drawRect(
						0,
						0,
						getWidth() - 1 - imageOffset,
						getHeight() - 1);
				}
			}
		}
		super.paint(g);
	}

    /**
     * Returns if the focus for this component is permanently lost or 
     * temporarily lost.
     * 
     * @return <code>true</code> if the focus for this component is permanently
     *         lost, else <code>false</code>. 
     */
    protected boolean isFocusPermanentlyLost() {

        KeyboardFocusManager kfm =
            KeyboardFocusManager.getCurrentKeyboardFocusManager();

        Component component = kfm.getPermanentFocusOwner();
        if (component == null) {
            return hasFocus;
        }

        // Issue 30:
        Window window = windowForComponent(tree);
        Window activeWindow = kfm.getActiveWindow();
        if (activeWindow == null) {
            // The active window is not a window or dialog of the calling 
            // thread's context. 
            return true;
        }
        return !component.equals(tree)
            || !activeWindow.equals(window);
    }

    protected Window windowForComponent(Component aComponent) {

        if (aComponent == null) {
            return null;
        }
        Component parent = aComponent.getParent();
        Window window = null;
        boolean found = false;
        while ((parent != null) && (!found)) {
            if (parent instanceof Window) {
                window = (Window) parent;
                found = true;
            }
            else {
                parent = parent.getParent();
            }
        }

        return window;
    }

	private int getLabelStart() {
		Icon currentI = getIcon();
		if (currentI != null && getText() != null) {
			return currentI.getIconWidth() + Math.max(0, getIconTextGap() - 1);
		}
		return 0;
	}

	/**
	 * Overrides <code>JComponent.getPreferredSize</code> to
	 * return slightly wider preferred size value.
	 */
	public Dimension getPreferredSize() {
		Dimension retDimension = super.getPreferredSize();
		if (retDimension != null)
			retDimension =
				new Dimension(retDimension.width + 3, retDimension.height);
		return retDimension;
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void validate() {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void revalidate() {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void repaint(long tm, int x, int y, int width, int height) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void repaint(Rectangle r) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	protected void firePropertyChange(
		String propertyName,
		Object oldValue,
		Object newValue) {
		// Strings get interned...
		if (propertyName == "text") //$NON-NLS-1$
			super.firePropertyChange(propertyName, oldValue, newValue);
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void firePropertyChange(
		String propertyName,
		byte oldValue,
		byte newValue) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void firePropertyChange(
		String propertyName,
		char oldValue,
		char newValue) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void firePropertyChange(
		String propertyName,
		short oldValue,
		short newValue) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void firePropertyChange(
		String propertyName,
		int oldValue,
		int newValue) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void firePropertyChange(
		String propertyName,
		long oldValue,
		long newValue) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void firePropertyChange(
		String propertyName,
		float oldValue,
		float newValue) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void firePropertyChange(
		String propertyName,
		double oldValue,
		double newValue) {
	}

	/**
	 * Overridden for performance reasons.
	 * See the <a href="#override">Implementation Note</a>
	 * for more information.
	 */
	public void firePropertyChange(
		String propertyName,
		boolean oldValue,
		boolean newValue) {
	}
}
