﻿#!/bin/bash
# Copyright (c) Verax Systems. All rights reserved.
#
# This is a bash script responsible for installing Verax SNMP Agent Simulator on Linux.
# INSTALLATION INSTRUCTIONS:
#  1. Create a new directory (e.g. /opt/verax/vxsnmpsimulator).
#  2. Unzip the simulator archive into this directory.
#  3. Change this directory (e.g. cd /opt/verax/vxsnmpsimulator). 
#  4. Make sure this script has execution right or issue: chmod a+x install.sh
#  5. Run installation script: ./install.sh
#  6. Follow instructions appearing on the screen. 



if [ -z "$CHMOD" ]; then CHMOD=/bin/chmod; fi
if [ ! -f "$CHMOD" ]; then CHMOD=chmod; fi
if [ -z "$GREP" ]; then GREP=/usr/bin/grep; fi
if [ ! -f "$GREP" ]; then GREP=/bin/grep; fi
if [ -z "$SED" ]; then SED=/bin/sed; fi
if [ ! -f "$SED" ]; then SED=sed; fi
if [ -z "$AWK" ]; then AWK=/bin/awk; fi

if [ $(id -u) != "0" ]
then
    echo "You must be root user to run this script."
    echo "Login as root and then run this script."
    exit 1
fi

echo "Welcome to the Verax SNMP Agent Simulator installer for Linux."
echo "=============================================================="
echo


# Change to local dir
LOCALDIR=`pwd`
cd $LOCALDIR


# Check if the package is complete
PACKAGE_PRESENT=1
if [ ! -f ./simulator.conf ]; then
    PACKAGE_PRESENT=0
fi
if [ ! -f ./simulatord ]; then 
    PACKAGE_PRESENT=0
fi
if [ ! -f jar/snmp-simulator-server.jar ]; then
    PACKAGE_PRESENT=0
fi
if [ ! -f conf/devices.conf.xml ]; then
    PACKAGE_PRESENT=0
fi
if [ ! -f conf/vlan_up ]; then
    PACKAGE_PRESENT=0
fi
if [ ! -f conf/vlan_down ]; then
    PACKAGE_PRESENT=0
fi
if [ ! -f conf/stop ]; then
    PACKAGE_PRESENT=0
fi


if [ "$PACKAGE_PRESENT" == "0" ]; then
    echo "Installation package appears incomplete."
    echo "Check if the installation package is complete or if you are in the directory containing unzipped package."
    exit 1
fi

cat <<EOF
EOF

read -p "Do you want to continue with installation? (y/n) [y]:" RESP
if [ "$RESP" == "n" ]; then
    echo "Installation canceled by user."
    exit 1
fi

status=`ps -ef | $GREP "snmp-simulator-server.jar" | $GREP -v grep`
if [ "$status" != "" ]
then
    read -p "Another Verax SNMP Agent Simulator instance is running and must be stopped now. Do you want to stop it? (y/n) [y]:" RESP
    if [ "$RESP" != "n" ]; then
        service simulatord stop
    fi
fi


# Copy files
echo "Copying files..."

$CHMOD a+x conf/vlan_up
$CHMOD a+x conf/vlan_down
$CHMOD a+x conf/stop
$CHMOD a+x ./simulatord

if [ ! -d /etc/verax.d ] 
then
    mkdir /etc/verax.d
fi

# Handle configuration file
if [ -f /etc/verax.d/simulator.conf ]
then
    echo "The configuration file /etc/verax.d/simulator.conf already exists."
    read -p "Do you want to overwrite it? (y/n) [y]:" CONF_FILE_OVERWRITTEN
    if [ "$CONF_FILE_OVERWRITTEN" == "n" ]; then
        echo "Configuration skipped by user. Please review /etc/verax.d/simulator.conf file."
    else
        cp ./simulator.conf /etc/verax.d/
    fi
else
    CONF_FILE_OVERWRITTEN=y
    cp ./simulator.conf /etc/verax.d/
fi


# Handle startup script
if [ -d /etc/init.d ]
then
    cp ./simulatord /etc/init.d
elif [ -d /etc/rc.d ]
then
    cp ./simulatord /etc/rc.d
else
    echo "This Linux distribution is not supported by the installer."
    echo "Please install the startup script manually."
fi


# Verify Java directory
if [ -z "$JAVA_HOME" -a -z "$JRE_HOME" ]; then
    JAVA_PATH=`which java 2>/dev/null`   
    if [ "$JAVA_PATH" == "" ]; then
        if [ -x /usr/bin/java ]; then
           JAVA_PATH=/usr/bin
        fi
    else
        JAVA_PATH=`dirname $JAVA_PATH`
    fi
else
    if [ -z "$JAVA_HOME" ]; then
        JAVA_PATH=$JRE_HOME/bin
    else
        JAVA_PATH=$JAVA_HOME/bin
    fi
fi


if [ -z "$JAVA_PATH" ]
then
    echo "Looking for Java executable in PATH."
    if [ ! -x java ]; then
        echo "No Java executable found in PATH. Please install the Java and point to Java localization in JRE_HOME variable in /etc/verax.d/simulator.conf file."
    fi
else
    echo "Looking for Java executable in $JAVA_PATH."

    if [ ! -x "$JAVA_PATH"/java  ]; then
        echo "No Java executable found in $JAVA_PATH. Please install the Java and point to Java localization in JRE_HOME variable in /etc/verax.d/simulator.conf file."
        JAVA_PATH=
    fi
fi


# Verify configuration file
if [ "$CONF_FILE_OVERWRITTEN" != "n" ]; then
    echo "Making changes in /etc/verax.d/simulator.conf file..."
    $SED -i "s|^SIMULATOR_HOME=.*|SIMULATOR_HOME=$LOCALDIR|g" /etc/verax.d/simulator.conf
    $SED -i "s|^JRE_HOME=.*|JRE_HOME='$JAVA_PATH/'|g" /etc/verax.d/simulator.conf
fi


# Check id port 161 is in use
PORT=`netstat -n -l | $GREP ":161 "`
if [ ! -z "$PORT" ]
then
    echo
    echo "Port 161 appears to be in use by another application (probably snmpd)."
    echo "Please stop this application or specify a different port in conf/devices.xml.conf."
    echo "To stop snmpd, issue: service snmpd stop."
fi


# Check id port 43500 is in use
PORT=`netstat -n -l | $GREP ":43500 "`
if [ ! -z "$PORT" ]
then
    echo
    echo "Port 43500 appears to be in use by another application."
    echo "Please stop this application or specify a different port in SERVER_PORT variable in /etc/verax.d/simulator.conf file."
fi

echo
echo "Installation completed."

status=`ps -ef | $GREP "snmp-simulator-server.jar" | $GREP -v grep`
if [ "$status" == "" ]
then
    read -p "Do you want Verax SNMP Agent Simulator to be started on now? (y/n) [y]:" RESP
    if [ "$RESP" != "n" ]; then
        service simulatord start
    fi
else
    read -p "Do you want Verax SNMP Agent Simulator to be restarted now? (y/n) [y]:" RESP
    if [ "$RESP" != "n" ]; then
        service simulatord stop
        service simulatord start
    fi
fi


