#region Licensing Header
// ---------------------------------------------------------------------------------
//  Copyright (C) 2007-2011 Chillisoft Solutions
//  
//  This file is part of the Habanero framework.
//  
//      Habanero is a free framework: you can redistribute it and/or modify
//      it under the terms of the GNU Lesser General Public License as published by
//      the Free Software Foundation, either version 3 of the License, or
//      (at your option) any later version.
//  
//      The Habanero framework is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU Lesser General Public License for more details.
//  
//      You should have received a copy of the GNU Lesser General Public License
//      along with the Habanero framework.  If not, see <http://www.gnu.org/licenses/>.
// ---------------------------------------------------------------------------------
#endregion
using System;
using Habanero.Base;
using Habanero.BO;
using Habanero.Faces.Test.Base;
using Habanero.Faces.Base;
using Habanero.Faces.Test.Base.Grid;
using Habanero.Faces.Win;
using Habanero.Test;
using NUnit.Framework;
using Rhino.Mocks;
// ReSharper disable InconsistentNaming

namespace Habanero.Faces.Test.Win.Grid
{
    public class GridBaseWinStub : GridBaseWin
    {
        public override bool CheckUserWantsToDelete()
        {
            return true;
        }

        /// <summary>
        /// Creates a dataset provider that is applicable to this grid. For example, a readonly grid would
        /// return a read only datasetprovider, while an editable grid would return an editable one.
        /// </summary>
        /// <param name="col">The collection to create the datasetprovider for</param>
        /// <returns></returns>
        public override IDataSetProvider CreateDataSetProvider(IBusinessObjectCollection col)
        {
            ReadOnlyDataSetProvider dataSetProvider = new ReadOnlyDataSetProvider(col);
            dataSetProvider.RegisterForBusinessObjectPropertyUpdatedEvents = true;
            return dataSetProvider;
        }
    }

    [TestFixture]
    public class TestGridBaseWin : TestGridBase
    {
        protected override IControlFactory GetControlFactory()
        {
            return new ControlFactoryWin();
        }

        protected override IGridBase CreateGridBaseStub()
        {
            GridBaseWinStub gridBase = new GridBaseWinStub();
            System.Windows.Forms.Form frm = new System.Windows.Forms.Form();
            frm.Controls.Add(gridBase);
            return gridBase;
        }

        [Test]
        public void TestWin_RowShowingBusinessObjectsValues()
        {
            //---------------Set up test pack-------------------
            MyBO.LoadDefaultClassDef();
            BusinessObjectCollection<MyBO> col = CreateCollectionWith_4_Objects();
            IGridBase gridBase = CreateGridBaseStub();
            SetupGridColumnsForMyBo(gridBase);
            const string propName = "TestProp";
            const int rowIndex = 1;
            //---------------Execute Test ----------------------
#pragma warning disable 618,612 //Maintained for backward compatibility testing
            gridBase.SetBusinessObjectCollection(col);
#pragma warning restore 618,612

            //---------------Test Result -----------------------
            MyBO selectedBo = (MyBO)gridBase.GetBusinessObjectAtRow(rowIndex);
            IDataGridViewRow row = gridBase.Rows[rowIndex];
            IDataGridViewCell cell = row.Cells[propName];
            Assert.AreEqual(selectedBo.TestProp, cell.Value);
        }
        [Test]
        public void TestWin_Set_BusinessObjectCollection()
        {
            //---------------Set up test pack-------------------
            MyBO.LoadDefaultClassDef();
            BusinessObjectCollection<MyBO> col = CreateCollectionWith_4_Objects();
            IGridBase gridBase = CreateGridBaseStub();
            SetupGridColumnsForMyBo(gridBase);
            const string propName = "TestProp";
            const int rowIndex = 1;
            //---------------Execute Test ----------------------
            gridBase.BusinessObjectCollection = col;

            //---------------Test Result -----------------------
            MyBO selectedBo = (MyBO)gridBase.GetBusinessObjectAtRow(rowIndex);
            IDataGridViewRow row = gridBase.Rows[rowIndex];
            IDataGridViewCell cell = row.Cells[propName];
            Assert.AreEqual(selectedBo.TestProp, cell.Value);
        }

        [Test]
        public void TestWinRowIsRefreshed()
        {
            //---------------Set up test pack-------------------
            BusinessObjectCollection<MyBO> col;
            IGridBase gridBase = GetGridBaseWith_4_Rows(out col);
            MyBO bo = col[0];

            //---------------Execute Test ----------------------
            const string propName = "TestProp";
            bo.SetPropertyValue(propName, "UpdatedValue");

            //---------------Test Result -----------------------
            gridBase.SelectedBusinessObject = bo;
            //System.Windows.Forms.DataGridViewRow row = (System.Windows.Forms.DataGridViewRow) gridBase.Rows[0];
            //System.Windows.Forms.DataGridViewCell cell = row.Cells[propName];
            System.Windows.Forms.DataGridViewCell cell = GetCell(0, propName, gridBase);
            Assert.AreEqual("UpdatedValue", cell.Value);
        }

        [Test]
        public virtual void TestWinApplyFilterFiresFilterUpdatedEvent()
        {
            //---------------Set up test pack-------------------
            BusinessObjectCollection<MyBO> col;
            GridBaseWin gridBase = (GridBaseWin)GetGridBaseWith_4_Rows(out col).GetControl();
            string filterString = col[2].ID.ToString().Substring(5, 30);
            IFilterClauseFactory factory = new DataViewFilterClauseFactory();
            IFilterClause filterClause =
                factory.CreateStringFilterClause(_gridIdColumnName, FilterClauseOperator.OpLike, filterString);
            bool filterUpdatedFired = false;
            gridBase.FilterUpdated += delegate { filterUpdatedFired = true; };
            //---------------Execute Test ----------------------
            gridBase.ApplyFilter(filterClause);
            //---------------Test Result -----------------------
            Assert.IsTrue(filterUpdatedFired);
        }
        [Test]
        public virtual void TestWinApplySearch_ShouldFireFilterUpdatedEvent()
        {
            //---------------Set up test pack-------------------
            BusinessObjectCollection<MyBO> col;
            GridBaseWin gridBase = (GridBaseWin)GetGridBaseWith_4_Rows(out col).GetControl();
            bool filterUpdatedFired = false;
            gridBase.FilterUpdated += delegate { filterUpdatedFired = true; };
            //---------------Execute Test ----------------------
            gridBase.ApplySearch("", "");
            //---------------Test Result -----------------------
            Assert.IsTrue(filterUpdatedFired);
        }
        [Test]
        public virtual void TestWinApplySearch_WithFilterClause_ShouldFireFilterUpdatedEvent()
        {
            //---------------Set up test pack-------------------
            BusinessObjectCollection<MyBO> col;
            GridBaseWin gridBase = (GridBaseWin)GetGridBaseWith_4_Rows(out col).GetControl();
            string filterString = col[2].ID.ToString().Substring(5, 30);
            IFilterClauseFactory factory = new DataViewFilterClauseFactory();
            IFilterClause filterClause =
                factory.CreateStringFilterClause("TestProp", FilterClauseOperator.OpLike, filterString);

            bool filterUpdatedFired = false;
            gridBase.FilterUpdated += delegate { filterUpdatedFired = true; };
            //---------------Execute Test ----------------------
            gridBase.ApplySearch(filterClause, "");
            //---------------Test Result -----------------------
            Assert.IsTrue(filterUpdatedFired);
        }

        [Test]
        public void TestSelectedBusinessObject_SetsCurrentRow()
        {
            //---------------Set up test pack-------------------
            BusinessObjectCollection<MyBO> col;
            IGridBase gridBase = GetGridBaseWith_4_Rows(out col);
            SetupGridColumnsForMyBo(gridBase);
            MyBO firstBO = col[0];
            MyBO secondBO = col[1];
            //---------------Assert Precondition----------------
            Assert.AreEqual(firstBO, gridBase.SelectedBusinessObject);
            Assert.IsNull(gridBase.CurrentRow);
            //Assert.AreEqual(0, gridBase.Rows.IndexOf(gridBase.CurrentRow));   //surely the currentrow should be active on setCol?
            //---------------Execute Test ----------------------
            gridBase.SelectedBusinessObject = secondBO;
            //---------------Test Result -----------------------
            Assert.AreEqual(secondBO, gridBase.SelectedBusinessObject);
            Assert.AreEqual(1, gridBase.Rows.IndexOf(gridBase.CurrentRow));
        }

        private static System.Windows.Forms.DataGridViewCell GetCell(int rowIndex, string propName,
                                                                     IGridBase gridBase)
        {
            System.Windows.Forms.DataGridView dgv = (System.Windows.Forms.DataGridView)gridBase.GetControl();
            System.Windows.Forms.DataGridViewRow row = dgv.Rows[rowIndex];
            return row.Cells[propName];
        }

        protected override void AddControlToForm(IGridBase gridBase)
        {
            System.Windows.Forms.Form frm = new System.Windows.Forms.Form();
            frm.Controls.Add((System.Windows.Forms.Control)gridBase);
        }


    }

}