#region Licensing Header
// ---------------------------------------------------------------------------------
//  Copyright (C) 2007-2011 Chillisoft Solutions
//  
//  This file is part of the Habanero framework.
//  
//      Habanero is a free framework: you can redistribute it and/or modify
//      it under the terms of the GNU Lesser General Public License as published by
//      the Free Software Foundation, either version 3 of the License, or
//      (at your option) any later version.
//  
//      The Habanero framework is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU Lesser General Public License for more details.
//  
//      You should have received a copy of the GNU Lesser General Public License
//      along with the Habanero framework.  If not, see <http://www.gnu.org/licenses/>.
// ---------------------------------------------------------------------------------
#endregion
using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Globalization;
using Habanero.Base;
using Habanero.Base.Exceptions;
using Habanero.Base.Logging;
using Habanero.Util;
using System.Linq;

// Limiting the number of records for a Select
// -------------------------------------------
// SQL Server: SELECT TOP 10 * FROM [TABLE]
// DB2: SELECT * FROM [TABLE] FETCH FIRST 10 ROWS ONLY
// PostgreSQL: SELECT * FROM [TABLE] LIMIT 10
// Oracle: SELECT * FROM [TABLE] WHERE ROWNUM <= 10
// Sybase: SET ROWCOUNT 10 SELECT * FROM [TABLE]
// Firebird: SELECT FIRST 10 * FROM [TABLE]
// MySQL: SELECT * FROM [TABLE] LIMIT 10

//pagination: LIMIT in MySQL vs. FIRST / SKIP in FireBird
//
//    I got stuck writing a query for Firebird today. And here I was thinking that SQL was a standard - if only the S was for Standard
//
//I was busy writing scripts at work to paginate results in our BI portals. In MySQL I would usually write something like this to return the first 20 records:
//
//SELECT * FROM table LIMIT 1, 20
//
//Not so in Firebird:
//
//SELECT FIRST 20 SKIP 0 * FROM table
//
//In the MySQL, the 1 denotes from which record to start, and the 20 how many records to return. In the Firebird syntax, the FIRST denotes the number of records to return, and the SKIP specifies from which record to start.
//
//Naturally, to effect the pagination, you would use variables passed between the pages to control moving between sets of records.
//
//The support for Firebird is nothing like that for MySQL - and it took me quite a while to find the answer. Sometimes the geeks disappoint

namespace Habanero.DB
{
    /// <summary>
    /// A super-class to manage a database connection and execute sql commands
    /// </summary>
    /// "See registry (480) think typesafe as well."
    public abstract class DatabaseConnection : MarshalByRefObject, IDatabaseConnection
    {
        private static readonly object LockObject = new object();
        private readonly string _assemblyName;
        private readonly string _className;
        private string _connectString;
        private List<IDbConnection> _connections;
        private static IDatabaseConnection _currentDatabaseConnection;
        private static readonly IHabaneroLogger Log = GlobalRegistry.LoggerFactory.GetLogger("Habanero.DB.DatabaseConnection");
        private int _timeoutPeriod = -1;

        /// <summary>
        /// The <see cref="SqlFormatter"/> that is used to format the Swl for the database type represented by this database connection
        /// </summary>
        protected SqlFormatter _sqlFormatter;
        
        /// <summary>
        /// Constructor that initialises a new set of null connections
        /// </summary>
        protected DatabaseConnection()
        {
            _connections = new List<IDbConnection>(5);
            _sqlFormatter = new SqlFormatter("[", "]", "TOP", "");
        }

        /// <summary>
        /// Constructor that allows an assembly name and class name to
        /// be specified
        /// </summary>
        /// <param name="assemblyName">The assembly name</param>
        /// <param name="className">The database class name</param>
        protected DatabaseConnection(string assemblyName, string className) : this()
        {
            _assemblyName = assemblyName;
            _className = className;
        }

        /// <summary>
        /// Constructor to initialise the connection with the assembly name,
        /// class name and connection string provided
        /// </summary>
        /// <param name="assemblyName">The assembly name</param>
        /// <param name="className">The class name</param>
        /// <param name="connectString">The connection string. This can be
        /// generated by the various GetConnectionString() methods, tailored
        /// for the appropriate database vendors.</param>
        protected DatabaseConnection(string assemblyName, string className, string connectString)
            : this(assemblyName, className)
        {
            this.ConnectionString = connectString;
        }

        /// <summary>
        /// Creates a database connection using the assembly name and class
        /// name provided.
        /// </summary>
        /// <returns>Returns an IDbConnection object</returns>
        private IDbConnection CreateDatabaseConnection()
        {
            try
            {
                Type connectionType = TypeLoader.LoadType(_assemblyName, _className);
                return (IDbConnection) Activator.CreateInstance(connectionType);
            }
            catch (Exception ex)
            {
                throw new DatabaseConnectionException
                    (String.Format
                         ("An error occurred while attempting to connect to the "
                          + "database.  The assembly, '{0}', required for the "
                          + "database vendor specified in the configuration (eg. "
                          + "app.config) could not be loaded.  Please check that it "
                          + "has been included in the dependencies or references, "
                          + "or that it has been copied to the output/execution " + "folder for this application.",
                          _assemblyName), ex);
            }
        }

        /// <summary>
        /// Creates a new connection object and assigns to it the
        /// connection string that is stored in this instance
        /// </summary>
        /// <returns>Returns the new IDbConnection object</returns>
        protected IDbConnection NewConnection
        {
            get
            {
                IDbConnection con = this.CreateDatabaseConnection();
                try
                {
                    con.ConnectionString = this.ConnectionString;
                }
                catch (Exception ex)
                {
                    throw new DatabaseConnectionException
                        ("There was an error " + "connecting to the database. The connection information was "
                         + "rejected by the database - connection information is either "
                         + "missing or incorrect.  In your configuration (eg. in app."
                         + "config), you require settings for vendor, server and "
                         + "database.  Depending on your setup, you may also need "
                         + "username, password and port. Consult the documentation for "
                         + "more detail on available options for these settings.", ex);
                }
                return con;
            }
        }

        /// <summary>
        /// Gets or sets the current database connection
        /// </summary>
        public static IDatabaseConnection CurrentConnection
        {
            get { return _currentDatabaseConnection; }
            set { _currentDatabaseConnection = value; }
        }

        /// <summary>
        /// Gets and sets the connection string used to connect with the
        /// database.  Closes and disposes existing connections before
        /// assigning a new connection string.
        /// </summary>
        public string ConnectionString
        {
            get { return _connectString; }
            set
            {
                foreach (IDbConnection dbConnection in _connections)
                {
                    try
                    {
                        dbConnection.Close();
                        dbConnection.Dispose();
                    }
                    catch (Exception ex)
                    {
                        Log.Log("Error closing and disposing connection", ex, LogCategory.Warn);
                    }
                }
                _connections = new List<IDbConnection>(5);
                _connectString = value;
            }
        }

        /// <summary>
        /// Creates a connection and assigns this instance's connection string
        /// </summary>
        /// <returns>Returns a new IDbConnection object</returns>
        internal IDbConnection TestConnection
        {
            get
            {
                IDbConnection con = this.CreateDatabaseConnection();
                con.ConnectionString = this.ConnectionString;
                return con;
            }
        }

        /// <summary>
        /// Returns a connection string with the password removed.  This method
        /// serves as a secure way of displaying an error message in the case 
        /// of a connection error, without compromising confidentiality.
        /// </summary>
        /// <returns>Returns a connect string with no password</returns>
        public string ErrorSafeConnectString()
        {
            string connectString = ConnectionString;
            int pwdStartPos = connectString.ToUpper(CultureInfo.InvariantCulture).IndexOf
                ("pwd=".ToUpper(CultureInfo.InvariantCulture));
            if (pwdStartPos <= 0)
            {
                pwdStartPos = connectString.ToUpper(CultureInfo.InvariantCulture).IndexOf
                    ("password=".ToUpper(CultureInfo.InvariantCulture));
            }
            if (pwdStartPos >= 0)
            {
                int pwdEndPos = connectString.IndexOf(";", pwdStartPos);
                if (pwdEndPos < pwdStartPos)
                {
                    pwdEndPos = connectString.Length;
                }
                int numChars = pwdEndPos - pwdStartPos;
                connectString = connectString.Remove(pwdStartPos, numChars);
            }
            return connectString;
        }

        /// <summary>
        /// Either finds a closed connection and opens and returns it,
        /// or creates a new connection and returns that.  Throws an 
        /// exception and adds a message to the log if there is an 
        /// error opening a connection.
        /// </summary>
        /// <returns>Returns a new IDbConnection object</returns>
        internal IDbConnection GetOpenConnectionForReading()
        {
            try
            {
                lock (LockObject)
                {
                    //log.Debug(string.Format("GetOpenConnectionForReading: Open connection count: {0}/{1}", _connections.FindAll(connection => connection.State == ConnectionState.Open).Count, _connections.Count));

                    // looks for closed connections for reading because open 
                    // connections could have readers still associated with them.
                    foreach (IDbConnection dbConnection in _connections)
                    {
                        if (dbConnection.State != ConnectionState.Closed) continue;
                        dbConnection.Open();
                        return dbConnection;
                    }
                    IDbConnection newDbConnection = this.NewConnection;
                    newDbConnection.Open();
                    _connections.Add(newDbConnection);
                    return newDbConnection;
                }
            }
            catch (Exception ex)
            {
                Log.Log
                    ("Error opening connection to db : " + ex.GetType().Name + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 8, true), LogCategory.Exception);
                throw;
            }
        }

        /// <summary>
        /// Returns the first closed connection available or returns a
        /// new connection object.  Throws an exception and adds a message 
        /// to the log if there is an error opening a connection.
        /// </summary>
        /// <returns>Returns a new IDbConnection object</returns>
        public IDbConnection GetConnection()
        {
            try
            {
                //log.Debug(string.Format("GetConnection: Open connection count: {0}/{1}", _connections.FindAll(connection => connection.State == ConnectionState.Open).Count, _connections.Count));
                lock (LockObject)
                {
                    foreach (var dbConnection in _connections)
                    {
                        if (dbConnection.State == ConnectionState.Closed)
                        {
                            return dbConnection;
                        }
                    }
                    var newDbConnection = this.NewConnection;
                    _connections.Add(newDbConnection);
                    return newDbConnection;
                }
            }
            catch (Exception ex)
            {
                Log.Log
                    ("Error opening connection to db : " + ex.GetType().Name + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 8, true), LogCategory.Exception);
                throw new DatabaseConnectionException
                    ("An error occurred while attempting " + "to connect to the database.", ex);
            }
        }

        /// <summary>
        /// Creates and returns an open connection
        /// </summary>
        /// <returns>Returns an IDbConnection object</returns>
        protected IDbConnection OpenConnection
        {
            get
            {
                lock (LockObject)
                {
                    var connection = this.GetConnection();
                    connection.Open();
                    return connection;
                }
            }
        }

        /// <summary>
        /// Loads a data reader and specifies an order-by clause
        /// </summary>
        /// <param name="selectSql">The sql statement object</param>
        /// <param name="strOrderByCriteria">A sql order-by clause</param>
        /// <returns>Returns an IDataReader object</returns>
        public IDataReader LoadDataReader(ISqlStatement selectSql, string strOrderByCriteria)
        {
            if (selectSql == null) throw new ArgumentNullException("selectSql");

            AppendOrderBy(selectSql, strOrderByCriteria);
            return LoadDataReader(selectSql);
        }

        /// <summary>
        /// Appends an order-by clause to the sql statement. " ORDER BY " is
        /// automatically prefixed by this method.
        /// </summary>
        /// <param name="statement"></param>
        /// <param name="orderByCriteria">The order-by clause</param>
        private static void AppendOrderBy(ISqlStatement statement, string orderByCriteria)
        {
            if (!string.IsNullOrEmpty(orderByCriteria))
            {
                statement.Statement.Append(" ORDER BY " + orderByCriteria);
            }
        }

        /// <summary>
        /// Loads a data reader with the given raw sql select statement
        /// </summary>
        /// <param name="selectSql">The sql statement as a string</param>
        /// <returns>Returns an IDataReader object with the results of the query</returns>
        /// <exception cref="DatabaseReadException">Thrown when an error
        /// occurred while setting up the data reader.  Also sends error
        /// output to the log.</exception>        
        public IDataReader LoadDataReader(string selectSql)
        {
            if (selectSql == null) throw new ArgumentNullException("selectSql");
            IDbConnection con = null;
            try
            {
                con = GetOpenConnectionForReading();
                IDbCommand cmd = CreateCommand(con);
                cmd.CommandType = CommandType.Text;
                cmd.CommandText = selectSql;
                cmd.Transaction = BeginTransaction(con);
                return cmd.ExecuteReader(CommandBehavior.CloseConnection);
            }
            catch (Exception ex)
            {
                Log.Log
                    ("Error reading from database : " + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 10, true), LogCategory.Exception);
                Log.Log("Sql: " + selectSql, LogCategory.Exception);
                throw new DatabaseReadException
                    ("There was an error reading the database. Please contact your system administrator.",
                     "The DataReader could not be filled with", ex, selectSql, ErrorSafeConnectString());
            }
            finally
            {
                //if (con != null) log.Debug(string.Format("LoadDataReader(string): Final Connection state: {0}", con.State));
            }
        }

        /// <summary>
        /// Loads a data reader with the given raw sql select statement for the specified transaction
        /// </summary>
        /// <param name="selectSql">The sql statement as a string</param>
        /// <param name="transaction">Thransaction that gives the context within which the sql statement should be executed</param>
        /// <returns>Returns an IDataReader object with the results of the query</returns>
        /// <exception cref="DatabaseReadException">Thrown when an error
        /// occurred while setting up the data reader.  Also sends error
        /// output to the log.</exception>        
        public IDataReader LoadDataReader(string selectSql, IDbTransaction transaction)
        {
            if (selectSql == null) throw new ArgumentNullException("selectSql");
            IDbConnection con = null;
            try
            {
                con = transaction.Connection;
                IDbCommand cmd = CreateCommand(con);
                cmd.Transaction = transaction;
                cmd.CommandType = CommandType.Text;
                cmd.CommandText = selectSql;
                return cmd.ExecuteReader();
            }
            catch (Exception ex)
            {
                Log.Log
                    ("Error reading from database : " + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 10, true), LogCategory.Exception);
                Log.Log("Sql: " + selectSql, LogCategory.Exception);
                throw new DatabaseReadException
                    ("There was an error reading the database. Please contact your system administrator.",
                     "The DataReader could not be filled with", ex, selectSql, ErrorSafeConnectString());
            }
            finally
            {
                //if (con != null) log.Debug(string.Format("LoadDataReader(string, IDbTransaction): Final Connection state: {0}", con.State));
            }
        }

        /// <summary>
        /// Loads a data reader
        /// </summary>
        /// <param name="selectSql">The sql statement object</param>
        /// <returns>Returns an IDataReader object</returns>
        /// <exception cref="DatabaseReadException">Thrown when an error
        /// occurred while setting up the data reader.  Also sends error
        /// output to the log.</exception>
        public IDataReader LoadDataReader(ISqlStatement selectSql)
        {
            if (selectSql == null)
            {
                throw new DatabaseConnectionException
                    ("The sql statement object " + "that has been passed to LoadDataReader() is null.");
            }
            IDbConnection con = null;
            try
            {
                con = GetOpenConnectionForReading();
                IDbCommand cmd = CreateCommand(con);
                selectSql.SetupCommand(cmd);
                cmd.Transaction = BeginTransaction(con);
                return cmd.ExecuteReader(CommandBehavior.CloseConnection);
            }
            catch (Exception ex)
            {
                Log.Log
                    ("Error reading from database : " + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 10, true), LogCategory.Exception);
                Log.Log("Sql: " + selectSql, LogCategory.Exception);

                Console.Out.WriteLine
                    ("Error reading from database : " + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 10, true));
                Console.Out.WriteLine("Sql: " + selectSql);
                throw new DatabaseReadException
                    ("There was an error reading the database. Please contact your system administrator." + Environment.NewLine +
                     selectSql.ToString() + Environment.NewLine,
                     "The DataReader could not be filled with", ex, selectSql.ToString(), ErrorSafeConnectString());
            }
            finally
            {
                //if (con != null) log.Debug(string.Format("LoadDataReader(ISqlStatement): Final Connection state: {0}", con.State));
            }
        }

        /// <summary>
        /// Creates a transaction using the <see cref="IsolationLevel"/> set as the IsolationLevel. If this doesn't work, creates
        /// a transaction using the default IsolationLevel. Uses the connection passed in to create the transaction on - this should be
        /// open (so use <see cref="GetOpenConnectionForReading"/> to get the connection first).  Override this method to 
        /// do special transaction creation logic.
        /// </summary>
        /// <param name="openConnection"></param>
        /// <returns></returns>
        public virtual IDbTransaction BeginTransaction(IDbConnection openConnection)
        {
            return openConnection.BeginTransaction(IsolationLevel);
        }

        /// <summary>
        /// Executes a sql command that returns no result set and takes no 
        /// parameters, using the provided connection
        /// </summary>
        /// <param name="statements">A valid sql statement (typically "insert",
        /// "update" or "delete"). Note_ that this assumes that the
        /// sqlCommand is not a stored procedure.</param>
        /// <returns>Returns the number of rows affected</returns>
        /// <future>
        /// In future override this method with others that allow you to 
        /// pass in stored procedures and parameters.
        /// </future>
        public int ExecuteSql(IEnumerable<ISqlStatement> statements)
        {
            return ExecuteSql(statements, null);
        }

        /// <summary>
        /// Executes a sql command as before, but with the full sql string
        /// provided, rather than with a sql statement object
        /// </summary>
        /// <param name="sql">The sql statement as a string</param>
        /// <returns>Returns the number of rows affected</returns>
        public int ExecuteRawSql(string sql)
        {
            return ExecuteRawSql(sql, null);
        }


        /// <summary>
        /// Executes a sql command that returns no result set and takes no 
        /// parameters, using the provided connection.
        /// This method can be used effectively where the database vendor
        /// supports the execution of several sql statements in one
        /// ExecuteNonQuery.  However, for database vendors like Microsoft
        /// Access and MySql, the sql statements will need to be split up
        /// and executed as separate transactions.
        /// </summary>
        /// <param name="sql">A valid sql statement (typically "insert",
        /// "update" or "delete") as a string. Note_ that this assumes that the
        /// sqlCommand is not a stored procedure.</param>
        /// <param name="transaction">A valid transaction object in which the 
        /// sql must be executed, or null</param>
        /// <returns>Returns the number of rows affected</returns>
        /// <exception cref="DatabaseWriteException">Thrown if there is an
        /// error writing to the database.  Also outputs error messages to the log.
        /// </exception>
        /// <future>
        /// In future override this method with methods that allow you to 
        /// pass in stored procedures and parameters.
        /// </future>
        public virtual int ExecuteRawSql(string sql, IDbTransaction transaction)
        {
            ArgumentValidationHelper.CheckArgumentNotNull(sql, "sql");
            IDbConnection con = null;
            try
            {
                IDbCommand cmd;
                if (transaction != null)
                {
                    con = transaction.Connection;
                    if (con.State != ConnectionState.Open)
                    {
                        con.Open();
                    }

                    cmd = CreateCommand(con);
                    cmd.Transaction = transaction;
                }
                else
                {
                    con = OpenConnection;
                    cmd = CreateCommand(con);
                }
                cmd.CommandText = sql;
                return cmd.ExecuteNonQuery();
            }
            catch (Exception ex)
            {
                Log.Log
                    ("Error writing to database : " + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 10, true), LogCategory.Exception);
                Log.Log("Sql: " + sql, LogCategory.Exception);
                Console.WriteLine
                    ("Error writing to database : " + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 10, true));
                Console.WriteLine("Connect string: " + this.ErrorSafeConnectString());
                throw new DatabaseWriteException
                    ("There was an error writing to the database. Please contact your system administrator.",
                     "The command executeNonQuery could not be completed.", ex, sql, ErrorSafeConnectString());
            }
            finally
            {
                if (transaction == null)
                {
                    if (con != null && con.State != ConnectionState.Closed)
                    {
                        con.Close();
                    }
                }
            }
        }

        private IDbCommand CreateCommand(IDbConnection dbConnection)
        {
            IDbCommand dbCommand = dbConnection.CreateCommand();
            try
            {
                if (_timeoutPeriod > 0) dbCommand.CommandTimeout = _timeoutPeriod;
            }
            catch (NotSupportedException ex)
            {
                Log.Log("Error setting command timeout period", ex, LogCategory.Warn);
            }
            return dbCommand;
        }


        /// <summary>
        /// Executes a collection of sql commands that returns no result set 
        /// and takes no parameters, using the provided connection.
        /// This method can be used effectively where the database vendor
        /// supports the execution of several sql statements in one
        /// ExecuteNonQuery.  However, for database vendors like Microsoft
        /// Access and MySql, the sql statements will need to be split up
        /// and executed as separate transactions.
        /// </summary>
        /// <param name="statements">A valid sql statement object (typically "insert",
        /// "update" or "delete"). Note_ that this assumes that the
        /// sqlCommand is not a stored procedure.</param>
        /// <param name="transaction">A valid transaction object in which the 
        /// sql must be executed, or null</param>
        /// <returns>Returns the number of rows affected</returns>
        /// <exception cref="DatabaseWriteException">Thrown if there is an
        /// error writing to the database.  Also outputs error messages to the log.
        /// </exception>
        /// <future>
        /// In future override this method with methods that allow you to 
        /// pass in stored procedures and parameters.
        /// </future>
        public virtual int ExecuteSql(IEnumerable<ISqlStatement> statements, IDbTransaction transaction)
        {
            bool inTransaction = false;
            ArgumentValidationHelper.CheckArgumentNotNull(statements, "statements");
            IDbConnection con = null;
            try
            {
                IDbCommand cmd;
                if (transaction != null)
                {
                    inTransaction = true;
                    con = transaction.Connection;
                    if (con.State != ConnectionState.Open)
                    {
                        con.Open();
                    }

                    cmd = CreateCommand(con);
                    cmd.Transaction = transaction;
                }
                else
                {
                    con = OpenConnection;
                    cmd = CreateCommand(con);
                    transaction = con.BeginTransaction();
                    cmd.Transaction = transaction;
                }
                int totalRowsAffected = 0;
                foreach (SqlStatement statement in statements)
                {
                    statement.SetupCommand(cmd);
                    //cmd.CommandText = sql;
                    try
                    {
                    totalRowsAffected += cmd.ExecuteNonQuery();
                    }
                    catch (Exception ex)
                    {
                        throw new DatabaseWriteException
                            ("There was an error executing the statement : " + Environment.NewLine + cmd.CommandText, ex);
                    }
                    statement.DoAfterExecute(this, transaction, cmd);
                }
                if (!inTransaction)
                {
                    transaction.Commit();
                }
                return totalRowsAffected;
            }
            catch (Exception ex)
            {
                Log.Log
                    ("Error writing to database : " + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 10, true), LogCategory.Exception);
                Log.Log("Sql: " + statements, LogCategory.Exception);
                if (transaction != null)
                {
                    transaction.Rollback();
                }
                throw new DatabaseWriteException
                    ("There was an error writing to the database. Please contact your system administrator."
                     + Environment.NewLine + "The command executeNonQuery could not be completed. :" + statements.ToString(),
                     "The command executeNonQuery could not be completed.", ex, statements.ToString(), ErrorSafeConnectString());
            }
            finally
            {
                if (!inTransaction)
                {
                    if (con != null && con.State != ConnectionState.Closed)
                    {
                        con.Close();
                    }
                }
            }
        }

        /// <summary>
        /// Executes a single sql statement object
        /// </summary>
        /// <param name="sql">The sql statement object</param>
        /// <returns>Returns the number of rows affected</returns>
        public int ExecuteSql(ISqlStatement sql)
        {
            if (sql == null) throw new ArgumentNullException("sql");
            return ExecuteSql(new[] { sql });
        }

        /// <summary>
        /// Returns a left field delimiter appropriate for the database (based on the <see cref="SqlFormatter"/> setup for
        ///   connection
        /// </summary>
        public virtual string LeftFieldDelimiter
        {
            get { return _sqlFormatter.LeftFieldDelimiter; }
        }

        /// <summary>
        /// Returns a Right field delimiter appropriate for the database (based on the <see cref="SqlFormatter"/> setup for
        ///   connection
        /// </summary>
        public virtual string RightFieldDelimiter
        {
            get { return _sqlFormatter.RightFieldDelimiter; }
        }

        ///<summary>
        /// Creates a SQL formatter for the specified database.
        ///</summary>
        public ISqlFormatter SqlFormatter
        {
            get { return _sqlFormatter; }
        }

        /// <summary>
        /// Gets the IsolationLevel to use for this connection
        /// </summary>
        public virtual IsolationLevel IsolationLevel
        {
            get { return IsolationLevel.RepeatableRead; }
        }


        /// <summary>
        /// Returns a limit clause with the limit specified, with the format
        /// as " TOP [limit] " (eg. " TOP 4 ")
        /// </summary>
        /// <param name="limit">The limit</param>
        /// <returns>Returns a string</returns>
        [Obsolete("please use the SqlFormatter directly")]
        public virtual string GetLimitClauseForBeginning(int limit)
        {
            return _sqlFormatter.GetLimitClauseCriteriaForBegin(limit);
        }

        /// <summary>
        /// Returns an empty string in this implementation
        /// </summary>
        /// <param name="limit">The limit - has no relevance in this 
        /// implementation</param>
        /// <returns>Returns an empty string in this implementation</returns>
        [Obsolete("please use the SqlFormatter directly")]
        public virtual string GetLimitClauseForEnd(int limit)
        {
            return _sqlFormatter.GetLimitClauseCriteriaForEnd(limit);
        }

        /// <summary>
        /// Set the time-out period in seconds, after which the connection
        /// attempt will fail
        /// </summary>
        /// <param name="timeoutSeconds">The time-out period in seconds</param>
        public void SetTimeoutPeriod(int timeoutSeconds)
        {
            _timeoutPeriod = timeoutSeconds;
        }

        /// <summary>
        /// Creates an <see cref="IParameterNameGenerator"/> for this database connection.  This is used to create names for parameters
        /// added to an <see cref="ISqlStatement"/> because each database uses a different naming convention for their parameters.
        /// </summary>
        /// <returns>The <see cref="IParameterNameGenerator"/> valid for this <see cref="IDatabaseConnection"/></returns>
        public abstract IParameterNameGenerator CreateParameterNameGenerator();

        /// <summary>
        /// Creates a <see cref="ISqlStatement"/> initialised with this <see cref="IDatabaseConnection"/>
        /// </summary>
        /// <returns></returns>
        public ISqlStatement CreateSqlStatement()
        {
            return new SqlStatement(this);
        }

        /// <summary>
        /// Loads data from the database into a DataTable object, using the
        /// sql statement object provided
        /// </summary>
        /// <param name="selectSql">The sql statement object</param>
        /// <param name="strSearchCriteria">The search criteria as a string
        /// to append</param>
        /// <param name="strOrderByCriteria">The order by criteria as a string
        /// to append</param>
        /// <returns>Returns a DataTable object</returns>
        /// <exception cref="DatabaseReadException">Thrown if there is an
        /// error reading the database.  Also outputs error messages to the log.
        /// </exception>
        public DataTable LoadDataTable(ISqlStatement selectSql, string strSearchCriteria, string strOrderByCriteria)
        {
            //It was chosen to use a datareader to fill the DataTable instead of using an
            //  adapter because the data adapter approach requires creating a different adapter depending on the 
            //  database type. This is understandable but for simple loading of a datatable without all the additional
            //  schema data it is unneccessary.
            //  It could however be easily achieved since there is a physical instance of a database connection object
            //   per database type that inherit from this class e.g. DatabaseConnectionMySQL.
            if (selectSql == null) throw new ArgumentNullException("selectSql");
            IDbConnection con = null;
            try
            {
                con = GetOpenConnectionForReading();
                IDbCommand cmd = CreateCommand(con);
                selectSql.SetupCommand(cmd);
                DataTable dt;
                using (IDataReader reader = cmd.ExecuteReader(CommandBehavior.CloseConnection))
                {
                    dt = GetDataTable(reader);
                    reader.Close();
                }
                return dt;
            }
            catch (Exception ex)
            {
                Log.Log
                    ("Error in LoadDataTable:" + Environment.NewLine
                     + ExceptionUtilities.GetExceptionString(ex, 8, true), LogCategory.Exception);
                Log.Log("Sql string: " + selectSql, LogCategory.Exception);
                throw new DatabaseReadException
                    ("There was an error reading the database. Please contact your system administrator.",
                     "The DataReader could not be filled with", ex, selectSql.ToString(), ErrorSafeConnectString());
            }
            finally
            {
                //if (con != null) log.Debug(string.Format("LoadDataTable(ISqlStatement, string, string): Final Connection state: {0}", con.State));
            }
        }
        /// <summary>
        /// Returns the DataTable for the DataReader.
        /// </summary>
        /// <param name="reader"></param>
        /// <returns></returns>
        public DataTable GetDataTable(IDataReader reader)
        {
            DataTable dt = new DataTable {TableName = "TableName"};
            if (reader.Read())
            {
                for (int i = 0; i < reader.FieldCount; i++)
                {
                    DataColumn add = dt.Columns.Add();
                    string columnName = reader.GetName(i);
                    if (!String.IsNullOrEmpty(columnName)) add.ColumnName = columnName;
                    add.DataType = reader.GetFieldType(i);
                }
                do
                {
                    DataRow row = dt.NewRow();
                    for (int i = 0; i < reader.FieldCount; i++)
                    {
                        row[i] = reader.GetValue(i);
                    }
                    dt.Rows.Add(row);
                } while (reader.Read());
            }
            return dt;
        }

        /// <summary>
        /// Returns a dataTable with the data from the reader and the columns names and field types set up.
        /// </summary>
        /// <param name="reader">the Reader that the dataTable will be made from</param>
        /// <param name="dataTableName">the name of the DataTable</param>
        /// <returns></returns>
        public static DataTable GetDataTable(IDataReader reader, string dataTableName)
        {
            DataTable dt = new DataTable {TableName = dataTableName};
            if (reader.Read())
            {
                CreateDataColumns(reader, dt);
                do
                {
                    DataRow row = dt.NewRow();
                    for (int i = 0; i < reader.FieldCount; i++)
                    {
                        row[i] = reader.GetValue(i);
                    }
                    dt.Rows.Add(row);
                } while (reader.Read());
            }
            reader.Close();
            return dt;
        }

        private static void CreateDataColumns(IDataRecord reader, DataTable dt)
        {
            for (int i = 0; i < reader.FieldCount; i++)
            {
                DataColumn add = dt.Columns.Add();
                string columnName = reader.GetName(i);
                if (!String.IsNullOrEmpty(columnName)) add.ColumnName = columnName;
                add.DataType = reader.GetFieldType(i);
            }
        }

        /// <summary>
        /// Gets the value of the last auto-incrementing number.  This called after doing an insert statement so that
        /// the inserted auto-number can be retrieved.  The table name, current IDbTransaction and IDbCommand are passed
        /// in so that they can be used if necessary.  Note_, this must be overridden in subclasses to include support
        /// for this feature in different databases - otherwise a NotImplementedException will be thrown.
        /// </summary>
        /// <param name="tableName">The name of the table inserted into</param>
        /// <param name="tran">The current transaction, the one the insert was done in</param>
        /// <param name="command">The Command the did the insert statement</param>
        /// <returns></returns>
        public virtual long GetLastAutoIncrementingID(string tableName, IDbTransaction tran, IDbCommand command)
        {
            throw new NotImplementedException
                ("GetLastAutoIncrementingID is not implemented on DatabaseConnection of type " + _className
                 + " in assembly " + _assemblyName);
        }
    }
}
