#region Licensing Header
// ---------------------------------------------------------------------------------
//  Copyright (C) 2007-2011 Chillisoft Solutions
//  
//  This file is part of the Habanero framework.
//  
//      Habanero is a free framework: you can redistribute it and/or modify
//      it under the terms of the GNU Lesser General Public License as published by
//      the Free Software Foundation, either version 3 of the License, or
//      (at your option) any later version.
//  
//      The Habanero framework is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU Lesser General Public License for more details.
//  
//      You should have received a copy of the GNU Lesser General Public License
//      along with the Habanero framework.  If not, see <http://www.gnu.org/licenses/>.
// ---------------------------------------------------------------------------------
#endregion
using Habanero.Base;
using Habanero.BO;
using Habanero.BO.ClassDefinition;
using Habanero.Test.BO.RelatedBusinessObjectCollection;
using Habanero.Util;
using NUnit.Framework;

namespace Habanero.Test.BO.BusinessObjectCollection
{
    /// <summary>
    /// 	A typical example of an associative relationship is a Car and its Drivers (assuming a car can have many drivers but a driver may only drive one car). A Driver can exist independently of any Car and a Car can exist independently of a driver. The Driver may however be associated with one car and later associated with a different car. 
    ///	The rules for whether a car that is associated with one or more drivers can be deleted or not is dependent upon the rules configured for the Drivers relationship (i.e. a cars drivers relationship could be marked prevent delete, dereference or do nothing). 
    ///	An already persisted driver can be added to a car (In habanero a new driver can be added to a car).
    ///	A driver can be removed from its related car. 
    ///	A car can create a new driver via its Drivers Relationship (this is not a strict implementation of domain design but is allowed due to the convenience of this).
    ///	A car is considered to be dirty only if it has added, created or removed dirty drivers. 
    ///	If a car is persisted then it will only persist its drivers relationship and will not persist a related driver that is dirty.
    /// </summary>
    [TestFixture]
    public class TestRelatedBOCol_Association
    {
        private readonly TestUtilsRelated util = new TestUtilsRelated();

        [TestFixtureSetUp]
        public virtual void TestFixtureSetup()
        {
        }

        [SetUp]
        public virtual void SetupTest()
        {
            ClassDef.ClassDefs.Clear();
            BORegistry.DataAccessor = new DataAccessorInMemory();
            ContactPersonTestBO.LoadClassDefOrganisationTestBORelationship_MultipleReverse();
            OrganisationTestBO.LoadDefaultClassDef_PreventAddChild();
        }

        [TearDown]
        public void TearDownTest()
        {
            TestUtil.WaitForGC();
        }

        [Test]
        public void Test_AddMethod_AddPersistedChild()
        {
            //An already persisted driver can be added to a car
            //---------------Set up test pack-------------------
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            MultipleRelationship<ContactPersonTestBO> associationRelationship = GetAssociationRelationship
                (organisationTestBO, out cpCol);
            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateSavedContactPerson();
            util.RegisterForAddedAndRemovedEvents(cpCol);
            //---------------Assert Precondition----------------
            util.AssertAllCollectionsHaveNoItems(cpCol);
            //---------------Execute Test ----------------------
            cpCol.Add(contactPerson);
            //---------------Test Result -----------------------
            util.AssertAddedEventFired();
            util.AssertOneObjectInCurrentAndAddedCollection(cpCol);
            Assert.AreSame(contactPerson.Organisation, associationRelationship.OwningBO);
        }

        [Test]
        public void Test_AddMethod_AddPersistedChildAndSaveChild()
        {
            //An already persisted driver can be added to a car
            //---------------Set up test pack-------------------
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            MultipleRelationship<ContactPersonTestBO> associationRelationship = GetAssociationRelationship
                (organisationTestBO, out cpCol);
            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateSavedContactPerson();
            util.RegisterForAddedAndRemovedEvents(cpCol);
            //---------------Assert Precondition----------------
            util.AssertAllCollectionsHaveNoItems(cpCol);
            //---------------Execute Test ----------------------
            cpCol.Add(contactPerson);
            contactPerson.Save();
            //---------------Test Result -----------------------
            util.AssertAddedEventFired();
            util.AssertOneObjectInCurrentPersistedCollection(cpCol);
            Assert.AreSame(contactPerson.Organisation, associationRelationship.OwningBO);
        }

        [Test]
        public void Test_AddMethod_AddPersistedChildAndSaveParent()
        {
            //An already persisted driver can be added to a car
            //---------------Set up test pack-------------------
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            MultipleRelationship<ContactPersonTestBO> associationRelationship = GetAssociationRelationship
                (organisationTestBO, out cpCol);
            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateSavedContactPerson();
            util.RegisterForAddedAndRemovedEvents(cpCol);
            //---------------Assert Precondition----------------
            util.AssertAllCollectionsHaveNoItems(cpCol);
            //---------------Execute Test ----------------------
            cpCol.Add(contactPerson);
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            util.AssertAddedEventFired();
            util.AssertOneObjectInCurrentPersistedCollection(cpCol);
            Assert.AreSame(contactPerson.Organisation, associationRelationship.OwningBO);
        }

        [Test]
        public void Test_AddMethod_AddPersistedChildAndSaveParent_WithNoReverseRelationship()
        {
            //An already persisted driver can be added to a car
            //---------------Set up test pack-------------------
            ClassDef.ClassDefs.Clear();
            ContactPersonTestBO.LoadClassDef_NoOrganisationRelationship();
            OrganisationTestBO.LoadDefaultClassDef_NoReverseRelationship();
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            GetAssociationRelationship(organisationTestBO, out cpCol);

            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateSavedContactPerson();
            util.RegisterForAddedAndRemovedEvents(cpCol);
            //---------------Assert Precondition----------------
            util.AssertAllCollectionsHaveNoItems(cpCol);
            //---------------Execute Test ----------------------
            cpCol.Add(contactPerson);
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            util.AssertAddedEventFired();
            util.AssertOneObjectInCurrentPersistedCollection(cpCol);
            Assert.AreEqual(contactPerson.OrganisationID, organisationTestBO.OrganisationID);
        }

        private static MultipleRelationship<ContactPersonTestBO> GetAssociationRelationship
            (OrganisationTestBO organisationTestBO, out BusinessObjectCollection<ContactPersonTestBO> cpCol)
        {
            MultipleRelationship<ContactPersonTestBO> associationRelationship =
                organisationTestBO.Relationships.GetMultiple<ContactPersonTestBO>("ContactPeople");
            RelationshipDef relationshipDef = (RelationshipDef) associationRelationship.RelationshipDef;
            relationshipDef.RelationshipType = RelationshipType.Association;
            cpCol = associationRelationship.BusinessObjectCollection;
            return associationRelationship;
        }


        [Test]
        public void Test_AddMethod_AddNewChild()
        {
            // (In habanero a new driver can be added to a car).
            //---------------Set up test pack-------------------
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            MultipleRelationship<ContactPersonTestBO> associationRelationship = GetAssociationRelationship
                (organisationTestBO, out cpCol);
            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateUnsavedContactPerson();
            util.RegisterForAddedAndRemovedEvents(cpCol);

            //---------------Assert Precondition----------------
            util.AssertAllCollectionsHaveNoItems(cpCol);

            //---------------Execute Test ----------------------
            cpCol.Add(contactPerson);

            //---------------Test Result -----------------------
            util.AssertAddedEventFired();
            util.AssertOneObjectInCurrentAndCreatedCollection(cpCol);
            Assert.AreSame(contactPerson.Organisation, associationRelationship.OwningBO);
        }


        [Test]
        public void Test_ResetParent_PersistedChild()
        {
            //A driver can be removed from its related car and transferred to another. 
            //---------------Set up test pack-------------------
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            GetAssociationRelationship(organisationTestBO, out cpCol);
            //util.AssertOneObjectInCurrentPersistedCollection(cpCol);
            util.AssertAllCollectionsHaveNoItems(cpCol);
            ContactPersonTestBO contactPerson = cpCol.CreateBusinessObject();
            util.AssertOneObjectInCurrentAndCreatedCollection(cpCol);
            contactPerson.Surname = TestUtil.GetRandomString();
            contactPerson.FirstName = TestUtil.GetRandomString();
            util.AssertOneObjectInCurrentAndCreatedCollection(cpCol);
            contactPerson.Save();
            util.AssertOneObjectInCurrentPersistedCollection(cpCol);
            OrganisationTestBO alternateOrganisationTestBO = OrganisationTestBO.CreateSavedOrganisation();

            //---------------Assert Precondition----------------
            util.AssertOneObjectInCurrentPersistedCollection(cpCol);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.AreSame(contactPerson.Organisation, organisationTestBO);
            // Assert.AreEqual(0, cpAltCol.Count);

            //---------------Execute Test ----------------------
            contactPerson.Organisation = alternateOrganisationTestBO;

            //---------------Test Result -----------------------
            Assert.AreEqual(0, cpCol.Count);
            Assert.IsFalse(cpCol.Contains(contactPerson));
            util.AssertOneObjectInRemovedAndPersisted(cpCol);
            MultipleRelationship<ContactPersonTestBO> relationship =
                alternateOrganisationTestBO.Relationships.GetMultiple<ContactPersonTestBO>("ContactPeople");
            BusinessObjectCollection<ContactPersonTestBO> cpAltCol = relationship.BusinessObjectCollection;
            Assert.AreSame(contactPerson.Organisation, relationship.OwningBO);
            Assert.AreSame(alternateOrganisationTestBO, contactPerson.Organisation);
            util.AssertOneObjectInCurrentAndAddedCollection(cpAltCol);
        }

        [Test]
        public void Test_ResetParent_NewChild_ReverseRelationship_Loaded()
        {
            //A driver can be removed from its related car and transferred to another. 
            //---------------Set up test pack-------------------
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            GetAssociationRelationship(organisationTestBO, out cpCol);
            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateUnsavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            util.RegisterForAddedAndRemovedEvents(cpCol);

            //---------------Assert Precondition----------------
            util.AssertAllCollectionsHaveNoItems(cpCol);

            //---------------Execute Test ----------------------
            contactPerson.Organisation = organisationTestBO;

            //---------------Test Result -----------------------
            Assert.AreEqual(contactPerson.OrganisationID, organisationTestBO.OrganisationID);
            util.AssertOneObjectInCurrentAndCreatedCollection(cpCol);
            Assert.IsTrue(cpCol.Contains(contactPerson));
            util.AssertAddedEventFired();
        }

        [Test]
        public void Test_SetParentNull()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateUnsavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());





            //---------------Assert Precondition----------------
            Assert.IsNull(contactPerson.Organisation);

            //---------------Execute Test ----------------------
            contactPerson.Organisation = null;

            //---------------Test Result -----------------------
            Assert.IsNull(contactPerson.Organisation);
        }

        [Test]
        public virtual void Test_RemoveMethod()
        {
            //A driver can be removed from its related car
            //---------------Set up test pack-------------------
            BORegistry.BusinessObjectManager.ClearLoadedObjects();
            TestUtil.WaitForGC();
            DeleteAllContactPersonAndOrganisations();

            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            GetAssociationRelationship(organisationTestBO, out cpCol);
            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateUnsavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            contactPerson.OrganisationID = organisationTestBO.OrganisationID;
            contactPerson.Save();
            cpCol.LoadAll();
            util.RegisterForAddedAndRemovedEvents(cpCol);

            //---------------Assert Precondition----------------
            util.AssertOneObjectInCurrentPersistedCollection(cpCol);

            //---------------Execute Test ----------------------
            cpCol.Remove(contactPerson);

            //---------------Test Result -----------------------
            util.AssertOneObjectInRemovedAndPersisted(cpCol);
            util.AssertRemovedEventFired();
        }

        protected virtual void DeleteAllContactPersonAndOrganisations()
        {
        }


        [Test]
        public virtual void Test_ResetParent_NewChild_SetToNull()
        {
            //A driver can be removed from its related car.   This test is removing via the reverse relationship
            //---------------Set up test pack-------------------
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            GetAssociationRelationship(organisationTestBO, out cpCol);
            ContactPersonTestBO contactPerson = ContactPersonTestBO.CreateUnsavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            contactPerson.OrganisationID = organisationTestBO.OrganisationID;
            contactPerson.Save();
            cpCol.LoadAll();
            util.RegisterForAddedAndRemovedEvents(cpCol);

            //---------------Assert Precondition----------------
            util.AssertOneObjectInCurrentPersistedCollection(cpCol);

            //---------------Execute Test ----------------------

            contactPerson.Organisation = null;

            //---------------Test Result -----------------------
            Assert.IsNull(contactPerson.Organisation);
            util.AssertOneObjectInRemovedAndPersisted(cpCol);
            util.AssertRemovedEventFired();
        }

        #region Delete Parent

//-------------------------------------------DELETE Parent with added, removed and created children ----
        [Test]
        public void Test_DeleteParent_WithCreatedChild_DeleteRule_DeleteRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidCreatedContactPerson
                (out contactPerson, out relationship);
            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.CreatedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.AreEqual(DeleteParentAction.DeleteRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        private static OrganisationTestBO CreateSavedOrganisation_WithOneValidCreatedContactPerson
            (out ContactPersonTestBO contactPerson, out MultipleRelationship<ContactPersonTestBO> relationship)
        {
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            relationship = GetAssociationRelationship(organisationTestBO, out cpCol);
            contactPerson = ContactPersonTestBO.CreateUnsavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            cpCol.Add(contactPerson);
            return organisationTestBO;
        }

        [Test]
        public void Test_DeleteParent_WithTwoCreatedChild_DeleteRule_DeleteRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidCreatedContactPerson
                (out contactPerson, out relationship);
            ContactPersonTestBO secondContactPerson = ContactPersonTestBO.CreateUnsavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            organisationTestBO.ContactPeople.Add(secondContactPerson);

            //---------------Assert Precondition----------------
            Assert.AreEqual(2, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.CreatedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.IsTrue(secondContactPerson.Status.IsNew);
            Assert.AreEqual(DeleteParentAction.DeleteRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should be deleted");
            Assert.IsTrue(secondContactPerson.Status.IsNew);
            Assert.IsTrue(secondContactPerson.Status.IsDeleted, "Should be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithCreatedChild_DeleteRule_DeleteRelated_DoesNotValidate()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidCreatedContactPerson
                (out contactPerson, out relationship);
            contactPerson.Surname = null;
            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.CreatedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsValid());
            Assert.AreEqual(DeleteParentAction.DeleteRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithCreatedChild_DeleteRule_DoNothing()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidCreatedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction = DeleteParentAction.DoNothing;
            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.CreatedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.AreEqual(DeleteParentAction.DoNothing, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithCreatedChild_DeleteRule_DoNothing_DoesNotValidate()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidCreatedContactPerson
                (out contactPerson, out relationship);

            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction = DeleteParentAction.DoNothing;
            contactPerson.Surname = null;
            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.CreatedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.AreEqual(DeleteParentAction.DoNothing, relationship.DeleteParentAction);
            Assert.IsFalse(contactPerson.Status.IsValid());
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithCreatedChild_DeleteRule_PreventDelete()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidCreatedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction = DeleteParentAction.Prevent;
            contactPerson.Surname = null;
            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.CreatedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsValid());
            Assert.AreEqual(DeleteParentAction.Prevent, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            ReflectionUtilities.SetPropertyValue(organisationTestBO.Status, "IsDeleted", true);
            try
            {
                organisationTestBO.Save();
                Assert.Fail("expected Err");
            }
                //---------------Test Result -----------------------
            catch (BusObjPersistException ex)
            {
                StringAssert.Contains
                    ("1 Business Objects via the ContactPeople relationship", ex.Message);
            }
        }

        [Test]
        public void Test_DeleteParent_WithTwoCreatedChild_DeleteRule_DerefRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidCreatedContactPerson
                (out contactPerson, out relationship);
            ContactPersonTestBO secondContactPerson = ContactPersonTestBO.CreateUnsavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            organisationTestBO.ContactPeople.Add(secondContactPerson);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction =
                DeleteParentAction.DereferenceRelated;
            //---------------Assert Precondition----------------
            Assert.AreEqual(2, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.CreatedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsNew);
            Assert.IsNotNull(contactPerson.OrganisationID);
            Assert.IsTrue(secondContactPerson.Status.IsNew);
            Assert.AreEqual(DeleteParentAction.DereferenceRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsTrue(contactPerson.Status.IsNew, "Should be new");
            Assert.IsFalse(contactPerson.Status.IsDeleted, "Should not be deleted");
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.IsTrue(secondContactPerson.Status.IsNew, "Should be new");
            Assert.IsFalse(secondContactPerson.Status.IsDeleted, "Should not be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithAddedChild_DeleteRule_DeleteRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidAddedContactPerson
                (out contactPerson, out relationship);


            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.AddedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.AreEqual(DeleteParentAction.DeleteRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should be permanently deleted");
            Assert.IsTrue(contactPerson.Status.IsNew, "Should be permanently deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithAddedDirtyChild_DeleteRule_DeRefRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidAddedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction =
                DeleteParentAction.DereferenceRelated;
            contactPerson.Surname = TestUtil.GetRandomString();
            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.AddedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsDirty, "Should be dirty");
            Assert.IsNotNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.DereferenceRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.IsFalse(contactPerson.Status.IsNew, "Should not be new");
            Assert.IsTrue(contactPerson.Status.IsDirty, "Should be dirty");
            Assert.IsFalse(contactPerson.Status.IsDeleted, "Should not be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithAddedChild_DeleteRule_DeRefRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidAddedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction =
                DeleteParentAction.DereferenceRelated;
            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.AddedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsDirty, "Should be dirty");
            Assert.IsNotNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.DereferenceRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.IsFalse(contactPerson.Status.IsNew, "Should not be new");
            Assert.IsTrue(contactPerson.Status.IsDirty, "Should be dirty");
            Assert.IsFalse(contactPerson.Status.IsDeleted, "Should not be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithAddedChild_DeleteRule_PreventDelete()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidAddedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction = DeleteParentAction.Prevent;
            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(1, organisationTestBO.ContactPeople.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.AddedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsNotNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.Prevent, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            ReflectionUtilities.SetPropertyValue(organisationTestBO.Status, "IsDeleted", true);
            try
            {
                organisationTestBO.Save();
                Assert.Fail("expected Err");
            }
                //---------------Test Result -----------------------
            catch (BusObjPersistException ex)
            {
                StringAssert.Contains
                    ("since it is related to 1 Business Objects via the ContactPeople relationship", ex.Message);
            }
        }

        [Test]
        public void Test_DeleteParent_WithRemovedChild_DeleteRule_PreventDelete()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidRemovedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction = DeleteParentAction.Prevent;

            //---------------Assert Precondition----------------
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(1, organisationTestBO.ContactPeople.RemovedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.RemovedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.Prevent, relationship.DeleteParentAction);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.IsFalse(contactPerson.Status.IsNew, "Should not be new");
            Assert.IsFalse(contactPerson.Status.IsDirty, "Should not be dirty");
            Assert.IsFalse(contactPerson.Status.IsDeleted, "Should not be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.RemovedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithRemovedChild_DeleteRule_Deref()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidRemovedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction =
                DeleteParentAction.DereferenceRelated;

            //---------------Assert Precondition----------------
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(1, organisationTestBO.ContactPeople.RemovedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.RemovedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.DereferenceRelated, relationship.DeleteParentAction);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.IsFalse(contactPerson.Status.IsNew, "Should not be new");
            Assert.IsFalse(contactPerson.Status.IsDirty, "Should not be dirty");
            Assert.IsFalse(contactPerson.Status.IsDeleted, "Should not be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.RemovedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithRemovedChild_DeleteRule_Delete()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneValidRemovedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction =
                DeleteParentAction.DeleteRelated;

            //---------------Assert Precondition----------------
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(1, organisationTestBO.ContactPeople.RemovedBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.RemovedBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.DeleteRelated, relationship.DeleteParentAction);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsNull(contactPerson.OrganisationID);
            Assert.IsFalse(contactPerson.Status.IsNew, "Should not be new");
            Assert.IsFalse(contactPerson.Status.IsDirty, "Should not be dirty");
            Assert.IsFalse(contactPerson.Status.IsDeleted, "Should not be deleted");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.AddedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.RemovedBusinessObjects.Count);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithMark4Child_DeleteRule_DeRefRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneMarkForDeleteContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction =
                DeleteParentAction.DereferenceRelated;


            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.MarkedForDeleteBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.MarkedForDeleteBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsDirty, "Should be dirty");
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should b deleted");
            Assert.IsNotNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.DereferenceRelated, relationship.DeleteParentAction);
//                        contactPerson.Updated += delegate(object sender, BOEventArgs e)
//                                               {
//                                                   string aa = "";
//                                               };
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsTrue(contactPerson.Status.IsNew, "Should not be permanetly deleted");
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should be permanetly deleted");
//            Assert.IsNull(contactPerson.OrganisationID);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.MarkedForDeleteBusinessObjects.Count);
            //Assert.IsFalse(contactPerson.Status.IsDirty, "Should not be dirty (permanetly deleted)");           
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithMark4Child_DeleteRule_DeleteRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithOneMarkForDeleteContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction =
                DeleteParentAction.DeleteRelated;



            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.MarkedForDeleteBusinessObjects.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople.MarkedForDeleteBusinessObjects[0]);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsTrue(contactPerson.Status.IsDirty, "Should be dirty");
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should b deleted");
            Assert.IsNotNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.DeleteRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
//            Assert.IsNull(contactPerson.OrganisationID);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.MarkedForDeleteBusinessObjects.Count);
            Assert.IsTrue(contactPerson.Status.IsNew, "Should not be permanetly deleted");
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should be permanetly deleted");
            //Assert.IsFalse(contactPerson.Status.IsDirty, "Should not be dirty (permanetly deleted)");           
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        [Test]
        public void Test_DeleteParent_WithPersistedChild_DeleteRule_DeleteRelated()
        {
            //---------------Set up test pack-------------------
            ContactPersonTestBO contactPerson;
            MultipleRelationship<ContactPersonTestBO> relationship;
            OrganisationTestBO organisationTestBO = CreateSavedOrganisation_WithPersistedContactPerson
                (out contactPerson, out relationship);
            ((MultipleRelationshipDef) relationship.RelationshipDef).DeleteParentAction =
                DeleteParentAction.DeleteRelated;








            //---------------Assert Precondition----------------
            Assert.AreEqual(1, organisationTestBO.ContactPeople.Count);
            Assert.AreEqual(contactPerson, organisationTestBO.ContactPeople[0]);
            Assert.AreEqual(0, organisationTestBO.ContactPeople.CreatedBusinessObjects.Count);
            Assert.IsFalse(organisationTestBO.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsNew);
            Assert.IsFalse(contactPerson.Status.IsDirty, "Should be dirty");
            Assert.IsFalse(contactPerson.Status.IsDeleted, "Should b deleted");
            Assert.IsNotNull(contactPerson.OrganisationID);
            Assert.AreEqual(DeleteParentAction.DeleteRelated, relationship.DeleteParentAction);
            //---------------Execute Test ----------------------
            organisationTestBO.MarkForDelete();
            organisationTestBO.Save();
            //---------------Test Result -----------------------
            Assert.IsNotNull
                (contactPerson.OrganisationID, "Contact Person should be deleted does not set OrgID to null");
            Assert.AreEqual(0, organisationTestBO.ContactPeople.MarkedForDeleteBusinessObjects.Count);
            Assert.IsTrue(contactPerson.Status.IsNew, "Should not be permanetly deleted");
            Assert.IsTrue(contactPerson.Status.IsDeleted, "Should be permanetly deleted");
            //Assert.IsFalse(contactPerson.Status.IsDirty, "Should not be dirty (permanetly deleted)");           
            Assert.AreEqual(0, organisationTestBO.ContactPeople.Count);
        }

        private static OrganisationTestBO CreateSavedOrganisation_WithOneValidAddedContactPerson
            (out ContactPersonTestBO contactPerson, out MultipleRelationship<ContactPersonTestBO> relationship)
        {
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            relationship = GetAssociationRelationship(organisationTestBO, out cpCol);
            contactPerson = ContactPersonTestBO.CreateSavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            cpCol.Add(contactPerson);
            return organisationTestBO;











        }

        private static OrganisationTestBO CreateSavedOrganisation_WithOneMarkForDeleteContactPerson
            (out ContactPersonTestBO contactPerson, out MultipleRelationship<ContactPersonTestBO> relationship)
        {
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            relationship = GetAssociationRelationship(organisationTestBO, out cpCol);
            contactPerson = ContactPersonTestBO.CreateSavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            cpCol.Add(contactPerson);
            cpCol.SaveAll();
            contactPerson.MarkForDelete();
            return organisationTestBO;












        }

        private static OrganisationTestBO CreateSavedOrganisation_WithPersistedContactPerson
            (out ContactPersonTestBO contactPerson, out MultipleRelationship<ContactPersonTestBO> relationship)
        {
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            relationship = GetAssociationRelationship(organisationTestBO, out cpCol);
            contactPerson = ContactPersonTestBO.CreateSavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            cpCol.Add(contactPerson);
            cpCol.SaveAll();
            return organisationTestBO;





















        }

        private static OrganisationTestBO CreateSavedOrganisation_WithOneValidRemovedContactPerson
            (out ContactPersonTestBO contactPerson, out MultipleRelationship<ContactPersonTestBO> relationship)
        {
            OrganisationTestBO organisationTestBO = OrganisationTestBO.CreateSavedOrganisation();
            BusinessObjectCollection<ContactPersonTestBO> cpCol;
            relationship = GetAssociationRelationship(organisationTestBO, out cpCol);
            contactPerson = ContactPersonTestBO.CreateSavedContactPerson
                (TestUtil.GetRandomString(), TestUtil.GetRandomString());
            cpCol.Add(contactPerson);
            cpCol.SaveAll();
            organisationTestBO.Save();
            cpCol.Remove(contactPerson);
            return organisationTestBO;
        }

        #endregion
    }




}
