/*---------------------------------------------------------

   ISFileDims.h : part of the ImgSource library.

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..
   No portion of this code may be copied or distributed without permission of
   Smaller Animals Software, Inc..
   http://www.smalleranimals.com

   Note : all is6_Get*Dims and is6_Get*Palette functions move the "file pointer" 
   for the source manager used. So you can't do this : 

   HISSRC hSrc = is6_OpenFileSource(...);
   is6_GetJPGDims(hSrc...); // moves file pointer
   is6_ReadJPGToRGB(hSrc...); // not at start of file!! read will fail
   is6_CloseSource(hSrc);

   Instead, do this : 

   HISSRC hSrc = is6_OpenFileSource(...);
   is6_GetJPGDims(hSrc...); // moves file pointer
   is6_Seek(hSrc, 0,0); // moves file pointer back to the start of the file
   is6_ReadJPGToRGB(hSrc...); // succeeds
   is6_CloseSource(hSrc);

---------------------------------------------------------*/

#ifndef is6_FILEDIMSH
#define is6_FILEDIMSH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_GetJPGDims / _is6_GetJPGDims

   Find dimensions of the JPG file.

   Param             Use
   ------------------------------------
   hSource           open source manager
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   puBitDepth        receives image bit depth

   puColorSpace      receives image color space flag.
                     this is the color space that the image is encoded
                     with. it doesn't necessarily mean the image was captured
                     or should be processed in that color space.

                        0 - unknown color space
                        1 - monochrome / grayscale
                        2 - RGB
                        3 - Y/Cb/Cr (also known as YUV) (this is the most common)
                        4 - CMYK
                        5 - Y/Cb/Cr/K

   puFlags           receives additional file information

                     bit      purpose
                     ---      -------
                      0       set if image is progressively encoded
                      1       set if image uses arithmetic encoding

   pDPIStruct        receives resolution info

                     pDPIStruct->uDPIUnits is one of:
                        0 - no units specified in file
                        1 - dots / inch
                        2 - dots / cm

   uFlags            unused

---------------------------------------------------------*/
_ISDeclSpec  BOOL      _ISCConv _ISFn( is6_GetJPGDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, UINT32 *puColorSpace, UINT32 * puFlags, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPNGDims / _is6_GetPNGDims

   Purpose : 
   Find dimensions of the PNG file, along with color depth info.

   See is6_GetPNGInputTextCount, is6_GetPNGInputText,
   is6_ClearPNGInputText.

   Note: the values for puTransRed, puTransGreen, puTransBlue and 
   puTransGray may be 16-bit values (if the input image had 16-bit 
   pixel components). In this case, discard the lower 8 bits.

   Param                Use
   ------------------------------------
   hSource              open source manager
   puWidth              receives image width (in pixels)
   puHeight             receives image height (in pixels)

   puBitDepth           receives image bits per component.
                        this is not the same as bits per pixel!!

   puColorType          receives PNG color type. 
                        0 = gray, 2 = RGB, 3 = colormapped,
                        4 = gray-alpha, 6 = RGBA

   puInterlaceType      receives interlace type 
                        0 = not interlaced
                        1 = Adam7 interlaced

   puTransIndex         unused. pass NULL.
   puTransRed           receives red component, for RGB images
   puTransGreen         green component 
   puTransBlue          blue component 
   puTransGray          receives gray value, for grayscale images

   pDPIStruct           receives resolution info.
                        pDPIStruct->uDPIUnits is one of:
                           0 - no units specified in file
                           1 - dots / meter

   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec   BOOL        _ISCConv _ISFn( is6_GetPNGDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, UINT32 *puColorType, UINT32 *puInterlaceType, UINT32 * puTransIndex, UINT32 * puTransRed, UINT32 * puTransGreen, UINT32 * puTransBlue, UINT32 * puTransGray, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPNGPalette / _is6_GetPNGPalette

   Purpose : 
   Get the PNG palette. Only valid for PNGs with 8bpp or less.

   Param             Use
   ------------------------------------
   hSource           open source manager
   pPalColors        receives number of colors in the palette
   pPal              256 RGBQUADs . receives palette
   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetPNGPalette )(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetTIFFDims / _is6_GetTIFFDims

   Purpose : 
   Find dimensions of the TIFF file.

   Note:
   This will not reflect the effects of the "orientation" of the image.
   If the image has a rotated orientation, the width and height will
   be reversed, compared to how the image is intended to be displayed.
   See is6_GetTIFFTag(.., TIFFTAG_ORIENTATION,..) to test the image orientation.

   Param             Use
   -----------------------------------        
   hSource           open source manager
   uPageIndex        sub-image index for multi-page images (0 for single page)
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   puBitDepth        receives image bit depth

   pDPIStruct        receives resolution info.
                     pDPIStruct->uDPIUnits is one of:
                        1 - no units specified in file
                        2 - dots / inch
                        3 - dots / cm

   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec   BOOL        _ISCConv _ISFn( is6_GetTIFFDims )(HISSRC hSource, UINT32 uPageIndex, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
    is6_GetBMPDims / _is6_GetBMPDims

    Purpose : 
    Find dimensions of the BMP file.

    Param               Use
    ------------------------------------
    hSource             open source manager
    puWidth             receives image width (in pixels)
    puHeight            receives image height (in pixels)
    puBitDepth          receives image bit depth

    pDPIStruct          receives resolution info
                        pDPIStruct->uDPIUnits is one of:
                            2 - dots / m

    uFlags              unused

    Return
    ------
    TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec  BOOL        _ISCConv _ISFn( is6_GetBMPDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetBMPPalette / _is6_GetBMPPalette

   Purpose : 
   Get the BMP palette. Only valid for BMPs with 8bpp or less.

   Param             Use
   ------------------------------------
   hSource           open source manager
   pPalColors        receives number of colors in the palette.
                     if this is zero, you can assume the palette contains the maximum
                     number of colors allowed for that bit depth

   pPal              256 RGBQUADs . receives palette

   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetBMPPalette )(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPCXDims / _is6_GetPCXDims

   Purpose : 
   Find dimensions of the PCX file.

   Param             Use
   ------------------------------------
   hSource           open source manager
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   puBitDepth        receives image bit depth 
   pDPIStruct        receives resolution info
                     
                     pDPIStruct->uDPIUnits is:
                         0 - dots / inch

   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec   BOOL        _ISCConv _ISFn( is6_GetPCXDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, IS4DPIStruct* pDPI, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPCXPalette / _is6_GetPCXPalette

   Purpose : 
   Get the PCX palette. Only valid for PCXs with 8bpp or less.

   Param             Use
   ------------------------------------
   hSource           open source manager
   pPalColors        receives number of colors in the palette
   pPal              256 RGBQUADs . receives palette
   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetPCXPalette )(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetTGADims / _is6_GetTGADims

   Purpose : 
   Find dimensions of the TGA file.

   Param             Use
   ------------------------------------
   hSource           open source manager
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   puBitDepth        receives image bit depth
   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec   BOOL        _ISCConv _ISFn( is6_GetTGADims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetTGAPalette / _is6_GetTGAPalette

   Purpose : 
   Get the TGA palette. Only valid for TGAs with 8bpp or less.

   Param             Use
   ------------------------------------
   hSource           open source manager
   pPalColors        receives number of colors in the palette
   pPal              256 RGBQUADs . receives palette
   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetTGAPalette )(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPAXDims / _is6_GetPAXDims

   Purpose : 
   Get dimension and bit depth information about a PAX file

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   puBitDepth        receives image bits per pixel (8, 24)

   pPassword         password for this file. if the correct
                     password is not supplied, this operation
                     will fail.

   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetPAXDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, const char *pPassword, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPAXPalette / _is6_GetPAXPalette

   Purpose : 
   Get the PAX palette. Only valid for PAXs with 8bpp or less.

   Param             Use
   ------------------------------------
   hSource           open source manager

   pPassword         password for this file. if the correct
                     password is not supplied, this operation
                     will fail.

   pPalColors        receives number of colors in the palette
   pPal              256 RGBQUADs . receives palette
   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetPAXPalette )(HISSRC hSource, const char *pPassword, UINT32 *pPalColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPSDDims / _is6_GetPSDDims

   Purpose : 
   Get dimension, DPI, and color mode information about a PSD 
   (Adobe Photoshop) file. 

    Note:
    In mode list, modes marked with '*' are not supported by ImgSource 
    PSD Read functions.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   puBitDepth        receives image bit depth per pixel

   puMode            Bitmap = 0
                     Grayscale = 1
                     Indexed = 2
                     RGB = 3
                     CMYK = 4
                     * MultiChannel = 7
                     DuoTone = 8
                     * Lab = 9

   pDPIStruct        receives resolution info
                     pDPIStruct->uDPIUnits is one of:
                        1 - no units specified in file
                        2 - dots / inch
                        3 - dots / cm
   uFlags            unused

    

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetPSDDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, UINT32 *puMode, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPSDPalette / _is6_GetPSDPalette

   Purpose : 
   Get the PSD palette. Only valid for PSDs with 8bpp or less.

   Param             Use
   ------------------------------------
   hSource           open source manager
   pPalColors        receives number of colors in the palette
   pPal              256 RGBQUADs . receives palette
   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetPSDPalette )(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetTLADims / _is6_GetTLADims

   Purpose : 
   Get dimension and bit depth information about a TLA file

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   puBitDepth        receives image bits per pixel (8, 24)

   pPassword         password for this file. if the file is 
                     encrypted and the correct password is 
                     not supplied, this operation will fail.

   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetTLADims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, const char *pPassword, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetTLAPalette / _is6_GetTLAPalette

   Purpose : 
   Get the TLA palette. Only valid for TLAs with 8bpp or less.

   Param             Use
   ------------------------------------
   hSource           valid source object

   pPassword         password for this file. if the file is 
                     encrypted and the correct password is 
                     not supplied, this operation will fail.

   pPalColors        receives number of colors in the palette
   pPal              256 RGBQUADs . receives palette
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetTLAPalette )(HISSRC hSource, const char *pKey, UINT32 *pPalColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetWBMPDims / _is6_GetWBMPDims

   Purpose : 
   Find dimensions of the WBMP file.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec   BOOL        _ISCConv _ISFn( is6_GetWBMPDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPCDIndexImageCount / _is6_GetPCDIndexImageCount

   Purpose : 
   Determine the number of thumbnail images in a Kodak PhotoCD
   index ("overview") file.

   Param             Use
   ------------------------------------
   hSource           valid source object
   uFlags            unused

   Return
   ------
   The number of thumbnails in the image
---------------------------------------------------------*/
_ISDeclSpec UINT32       _ISCConv _ISFn( is6_GetPCDIndexImageCount )(HISSRC hSource, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetGIFDims / _is6_GetGIFDims

   Purpose : 
   Get size and palette information about a GIF image. If
   the input is a multi-framed (animated) GIF, this returns
   the dimensions of the first frame in the image. 

   See uFlags, to get the overall animation "screen" size.
   
   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   puBitDepth        receives image bit depth (1 to 8)
   piBackgdColor     receives index of background color
   piTransColor      receives index of transparent color. -1 if none.

   pPal              array of 256 RGBQUAD entries. 
                     receives image palette

   pbInterlaced      receives interlaced setting (TRUE if interlaced)

   uFlags            bit      purpose
                     ---      -------
                      0       return GIF "screen" width and height
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_GetGIFDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, __int32 *piBackgdIdx, __int32 *piTransIdx, RGBQUAD *pal, BOOL *pbInterlaced, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPNMDims / _is6_GetPNMDims

   Purpose : 
   Get dimensions of a PPM, PGM or PBM file.
   
   Param                Use
   ------------------------------------
   hSource              valid source object
   puWidth              receives image width (in pixels)
   puHeight             receives image height (in pixels)
   puMaxVal             maximum pixel value, as specified in the file

   pFormat              20529 : Plain (ASCII) PBM
                        20530 : Plain PGM
                        20531 : Plain PPM
                        20532 : Binary PBM
                        20533 : Binary PGM
                        20534 : Binary PPM
   
   uFlags               unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_GetPNMDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puMaxVal, UINT32 *pFormat, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetCALSDims / _is6_GetCALSDims

   Purpose : 
   Find the dimensions of a CALS file.

   Param             Use
   -----------------------------------        
   hSource           open source manager
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   pDPIStruct        receives resolution info.
                     pDPIStruct->uDPIUnits is 2 (dots / inch)

   uFlags            unused

   Return
   ------
   TRUE if successful
---------------------------------------------------------*/
_ISDeclSpec   BOOL        _ISCConv _ISFn( is6_GetCALSDims )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, IS4DPIStruct *pDPIStruct, UINT32 uFlags);


#ifdef __cplusplus
}
#endif

#endif