/*---------------------------------------------------------
   ISGradientFill.h : part of the ImgSource library.
   
   Image processing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Do not include this file directly. Include ISource.h
#endif

#ifndef ISGRADIENTFILLH
#define ISGRADIENTFILLH

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_GradientFillImageRadial

   Purpose : 
   Fill an image with a radial color gradient.

   Param             Use
   ------------------------------------
   pOut              output image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel in the image (1 to 4)
   fGradientSize     the gradient is spread across this distance.
   iCenterX          center of the gradient                  
   iCenterY          center of the gradient
   
   clrs              array of RGBQUADs. these are the gradient colors.
                     if uBytesPerPixel = 1, fill uses the rgbRed member
                     if uBytesPerPixel = 2, fill uses the rgbRed and rgbGreen members
                     if uBytesPerPixel = 3, fill uses rgbRed, rgbGreen & rgbBlue
                     if uBytesPerPixel = 4, fill uses all members

   fPercentages      array of doubles. these specify how much of the gradient
                     the corresponding color (from clrs) occupies.

   uNumColors        number of entries in clrs and fPercentages. 
                     this must be at least 2

   iMode             unused
      
   fPower            controls blending power. this can be any non-zero positive value.
                     1 is the typical value and gives the same effect as is5_GradientFillImageRadial.

   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR color order

   Uses callback : Yes
      
   Ex.:

      // fade from yellow to green
      RGBQUAD clrs[2];
      double fPercentages[2];

      clrs[0].rgbRed = 255;
      clrs[0].rgbGreen = 255;
      clrs[0].rgbBlue = 0;

      clrs[1].rgbRed = 0;
      clrs[1].rgbGreen = 255;
      clrs[1].rgbBlue = 0;

      // everything below 50% of fGradientSize from the center
      // is solid yellow. at 50%, the transition to green starts.
      fPercentages[0] = 50;  
   
      // every point above 90% of fGradientSize from the center
      // is solid green
      fPercentages[1] = 90;

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_GradientFillImageRadial )(BYTE *pOut, UINT32 uWidth, UINT32 uHeight, UINT32 uBpp, UINT32 uRowStride, double fMaxDistance, int iCenterX, int iCenterY, RGBQUAD *clrs, double *fPercentages, UINT32 uNumColors, int iMode, double fPower, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GradientFillImageLinear

   Purpose : 
   Fill an image with a linear color gradient.

   You need to supply two points that define a line. The gradient
   will be constructed about that line. (see iMode for options)

   Color is determined by the distance from the pixel to the start of the
   gradient line.
   
   See is5_GradientFillImageRadial for an example of specifying the
   colors.

   Param             Use
   ------------------------------------
   pOut              output image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel in the image (1 to 4)
   uRowStride        bytes per pixel row
   fLinePoint1X      start of the gradient line
   fLinePoint1Y      start of of the gradient line

   fLinePoint2X      end of the gradient line
   fLinePoint2Y      end of the gradient line

   clrs              array of RGBQUADs. these are the gradient colors.
                     if uBytesPerPixel = 1, fill uses the rgbRed member
                     if uBytesPerPixel = 2, fill uses the rgbRed and rgbGreen members
                     if uBytesPerPixel = 3, fill uses rgbRed, rgbGreen & rgbBlue
                     if uBytesPerPixel = 4, fill uses all members

   fPercentages      array of doubles. these specify where the center of each color lies.
                     ex. with three colors, perc = [25,50,75] means that color 0 
                     is solid until 25% of the gradient length, then starts to blend 
                     with color[1], until 50% of the gradient length, when it starts to blend
                     with color[2]. the final 25% will be solid color[2].

                     also note that the percentage value determines the order of the color,
                     not the position in the clrs array.

   uNumColors        number of entries in clrs and fPercentages. 
                     this must be at least 2

   iMode             Mode 0:
                     The gradient runs parallel to the line. The start point defines 
                     the start of the gradient and the end point determines the direction 
                     and the length of the gradient. The gradient is symmetric about
                     the normal of the line.

                     Mode 1:
                     The gradient extends only on the side of the gradient line
                     that contains the the end point. Pixels on the other side
                     of the line will be colored with the lowest color in the color array.
                     This is similar to the way you draw a gradient in PhotoShop,
                     where you click, drag and release to define the start, direction 
                     and distance of the gradient.

   fPower            controls blending power. this can be any non-zero positive value.
                     1 is the typical value.

   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR color order

   Uses callback : Yes
     

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_GradientFillImageLinear )(BYTE *pOut, UINT32 uWidth, UINT32 uHeight, UINT32 uBpp, UINT32 uRowStride, double fLinePoint1X, double fLinePoint1Y, double fLinePoint2X, double fLinePoint2Y, RGBQUAD *clrs, double *fPercentages, UINT32 uNumColors, int iMode, double fPower, UINT32 uFlags);

#ifdef __cplusplus
}
#endif

#endif