/*---------------------------------------------------------
   ISIPTC.h : part of the ImgSource library.

   IPTC data reading code

   This code copyright (c) 2000-2015, Smaller Animals Software, Inc..
   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   
   http://www.smalleranimals.com
---------------------------------------------------------*/
#ifndef is6_IPTCH
#define is6_IPTCH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_IPTCInitialize / _is6_IPTCInitialize

   Purpose : 
   Create an ImgSource IPTC data reading object and fill it 
   from a file. This currently only supports IPTC data from JPG, 
   PSD (Photoshop) and TIFF files.

   IPTC/IIM data is image metadata stored in JPG files
   by various software packages, including Adobe PhotoShop.

   To fetch the individual strings from the data, use is6_IPTCGetTag.

   Also see is6_IPTCInitializeFromByteArray and 
   is6_IPTCCopyTagsFromInputToOutput.

   Note:
   This object must be destroyed with is6_IPTCRelease. Failure 
   to do so will result in memory leaks.

   IPTC = International Press Telecommunications Council 

   Param             Use
   ------------------------------------
   hSrc              open source manager object
   uFrameIdx         TIFF frame idx (if input is a TIFF file)
   uFlags            unused

   Return
   ------
   HISIPTC IPTC reader object. NULL on failure.
---------------------------------------------------------*/
_ISDeclSpec HISIPTC         _ISCConv _ISFn( is6_IPTCInitialize )(HISSRC hSrc, UINT32 uFrameIdx, UINT32 uFlags);

/*---------------------------------------------------------
   is6_IPTCInitializeFromByteArray / _is6_IPTCInitializeFromByteArray

   Purpose : 
   Create an ImgSource IPTC data reading object and fill it 
   from a byte array. 
   
   IPTC/IIM data is image metadata stored in JPG files
   by various software packages, including Adobe PhotoShop.

   To fetch the individual strings from the data, use is6_IPTCGetTag.
   
   Also see is6_IPTCCopyTagsFromInputToOutput.

   Note:
   This object must be destroyed with is6_IPTCRelease. Failure 
   to do so will result in memory leaks.

   IPTC = International Press Telecommunications Council 

   Param             Use
   ------------------------------------
   pData             pointer to data
   uDataLength       length of data
   uFlags            unused

   Return
   ------
   HISIPTC IPTC reader object. NULL on failure.
---------------------------------------------------------*/
_ISDeclSpec HISIPTC         _ISCConv _ISFn(is6_IPTCInitializeFromByteArray)(const BYTE *pData, UINT32 uDataLength, UINT32 uFlags);

/*---------------------------------------------------------
   is6_IPTCGetTag / _is6_IPTCGetTag

   Purpose : 
   Extract a data string for a single tag from an ImgSource
   IPTC data reader object.

   The data is generally ASCII data - but not always. Beware.

   These are some common tag IDs:
   id     description
   -------------------
   5      Origin Object Name
   15     Category 
   20     Supplemental categories (expect multiples)
   25     Keyword (expect multiples)
   40     Special Instructions
   55     Date string (yyyymmdd)
   80     Byline
   85     Byline title
   90     City
   95     State/Province
   101    Country
   103    Original transmission Reference
   105    Caption headline
   110    Credit
   115    Source
   116    Copyright
   120    Caption
   122    Caption writer
   
   Other IDs are possible.

   Param             Use
   ------------------------------------
   hIPTC             IPTC data reader object created by
                     is6_IPTCInit.

   uSection          data section. generally "2"
   uTagID            tag ID
   uIndex            index, in case the tag has multiple
                     data blocks. zero-based.

   phData            receives a handle to the tag data. 
                     you must use GlobalFree to release this
                     memory when you are finished with it.

   puLen             receives the length of the data
                     
   Return
   ------
   result code.
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_IPTCGetTag )(HISIPTC hIPTC, UINT32 uSection, UINT32 uTagID, UINT32 uIndex, HGLOBAL *phData, UINT32 *puLen);

/*---------------------------------------------------------
   is6_IPTCGetTagList / _is6_IPTCGetTagList

   Purpose : 
   Get a list of all data tags available from this IPTC 
   parser. Tags are listed as "section,id,index" seperated 
   by a 0x13 (new line).

   For example:

      2,110,0
      2,112,0
      2,115,0
      2,115,1
      ...

   Or, if translated:

      Object name, GEORGE BURLEY
      Category, SPT
      Supplemental categories, FOOTBALL
      Keywords, FOOTBALL
      Keywords, LEAGUE
      ...

   For a list of tags and the values associated with them, 
   see is6_IPTCGetTag
   

   Param             Use
   ------------------------------------
   hIPTC             IPTC data reader object created by
                     is6_IPTCInit.

   phData            receives a handle to the tag data. 
                     you must use GlobalFree to release this
                     memory when you are finished with it.

   puLen             receives the length of the data
                     
   uFlags            bit         purpose
                     ---         -------
                      8          return translated tags

   Return
   ------
   result code.
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_IPTCGetTagList )(HISIPTC hIPTC, HGLOBAL *phString, UINT32 *puLen, UINT32 uFlags);

/*---------------------------------------------------------
   is6_IPTCRelease / _is6_IPTCRelease

   Purpose : 
   Destroy an ImgSource IPTC data reader object. This
   will free all memory associated with the IPTC reader object.

   Param             Use
   ------------------------------------
   hIPTC             IPTC data reader object created by
                     is6_IPTCInit.

    Return
   ------
   result code.
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_IPTCRelease )(HISIPTC hIPTC);

/*---------------------------------------------------------
   is6_IPTCWriteInit / _is6_IPTCWriteInit

   Purpose : 
   Create an ImgSource IPTC export object. This object will
   create a block of memory suitable for use as an IPTC data
   block in a Photoshop resource block, which can then be used
   in a JPG, TIFF or PSD file.

   Use is6_IPTCWriteAddString to add strings, and is6_IPTCExport
   to create the IPTC data block. 
   
   This object must be destroyed with is6_IPTCWriteDestroy.     
       
   Param               Use
   ------------------------------------
   none

   Return
   ------
   HISIPTCW object

---------------------------------------------------------*/
_ISDeclSpec HISIPTCW         _ISCConv _ISFn( is6_IPTCWriteInit )();

/*---------------------------------------------------------
   is6_IPTCWriteAddTagData / _is6_IPTCWriteAddTagData

   Purpose : 
   Add a data tag to an ImgSource IPTC writer object. 

   While most commonly-used IPTC data is ASCII text, some fields
   are non-textual (numeric data types, structures, etc.).
   
   See is6_IPTCGetTag for a list of common tag IDs. 
   
   Note:
   Writing tags larger than 65535 bytes is not supported in 
   this implementation.

   Note:
   Some IDs allow multiples (Keywords and Categorys, for example). 
   
   Also see is6_IPTCCopyTagsFromInputToOutput

   Param             Use
   ------------------------------------
   hIPTC             valid IPTC writer object, created with is6_IPTCWriteInit
   uSection          generally "2"
   uID               tag ID (see is6_IPTCGetTag)
   pData             data to write
   uDataLen          length of data to write

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_IPTCWriteAddTagData)(HISIPTCW hIPTC, UINT32 uSection, UINT32 uID, const BYTE *pData, UINT32 uDataLen);

/*--------------------------------------------------------
   is6_IPTCWriteRemoveTag / _is6_IPTCWriteRemoveTag

   Purpose : 
   Remove a data tag from an ImgSource IPTC writer object. 

   See is6_IPTCGetTag for a list of common tag IDs. 
   
   Note:
   Some IDs allow multiples (Keywords and Categorys, for example). 
   This will remove the first tag which matches the section, and tag ID.
   To remove all tags with a given section/ID call this in a loop
   until it returns FALSE.
   
   Also see is6_IPTCCopyTagsFromInputToOutput, is6_IPTCWriteAddTagData.

   Param             Use
   ------------------------------------
   hIPTC             valid IPTC writer object, created with is6_IPTCWriteInit
   uSection          generally "2"
   uID               tag ID (see is6_IPTCGetTag)

   Return
   ------
   FALSE if tag was not found. TRUE if tag was removed sucessfully.

---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_IPTCWriteRemoveTag )(HISIPTCW hIPTC, UINT32 uSection, UINT32 uID);

/*---------------------------------------------------------
   is6_IPTCExport / _is6_IPTCExport

   Purpose : 
   Generate an IPTC data block based on the strings added via
   is6_IPTCWriteAddTagData. This can be used as input to
   is6_AddPSDResource to create a Photoshop/IPTC resource 
   to a TIFF, JPG or PSD file.

   Note:
   When creating a resource block for is6_AddPSDResource, 
   all headers should be supressed (see uFlags).

   Note:
   When creating a resource block for use in a TIFF file,
   you should suppress the "Photoshop 3.0" header (uFlags bit 0)

   Param             Use
   ------------------------------------
   hIPTC             valid IPTC writer object, created with is6_IPTCWriteInit
   phOut             receives a handle to the allocated memory. caller must free
                     this with GlobalFree.
   puSize            receives the size of the allocated memory block

   uFlags            bit      purpose
                     ---      -------
                      0       suppress "Photoshop 3.0" header   
                      1       suppress "8BIM" header
                      2       suppress length header
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_IPTCExport )(HISIPTCW hIPTC, HGLOBAL *hOut, UINT32 *uSize, UINT32 uFlags);

/*---------------------------------------------------------
   is6_IPTCWriteDestroy / _is6_IPTCWriteDestroy

   Purpose : 
   Destroy the IPTC writer object.

   Param             Use
   ------------------------------------   
   valid IPTC writer object, created with is6_IPTCWriteInit.
   

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_IPTCWriteDestroy )(HISIPTCW hIPTC);

/*---------------------------------------------------------
   is6_IPTCCopyTagsFromInputToOutput / _is6_IPTCCopyTagsFromInputToOutput

   Purpose : 
   Copy all the IPTC strings from your IPTC parsing object 
   to an IPTC writer object. This removes any strings already 
   in the writer object.

   This is a quick way to copy the IPTC from one file to another.
   
   Param             Use
   ------------------------------------
   hIPTCW            valid IPTC writer object, created 
                     with is6_IPTCWriteInit

   hIPTCR            valid IPTC data reader object created 
                     with is6_IPTCInitialize

   uFlags            unused

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_IPTCCopyTagsFromInputToOutput )(HISIPTCW hIPTCW, HISIPTC hIPTCR, UINT32 uFlags);

#ifdef __cplusplus
}
#endif
#endif