/*---------------------------------------------------------
   ISImageDraw.h : part of the ImgSource library.
   
   image drawing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is6_IMAGEDRAWH
#define is6_IMAGEDRAWH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_DrawHBITMAP / _is6_DrawHBITMAP
                             
   Purpose :
   Draw an HBITMAP to a device context

   The device referred to by hDC must support 
   the following RASTERCAPS values : 
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  
  
   Param             Use
   ------------------------------------
   hDC               output device context
   hBmp              HBITMAP to draw
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   uOutWidth         X pixels to draw (may be less than 
                     image width)

   uOutHeight        Y pixels to draw (may be less than 
                     image width)

   hPal              optional palette

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DrawHBITMAP )(HDC hDC, HBITMAP hBmp, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight, HPALETTE hPal);

/*---------------------------------------------------------
   is6_DrawHBITMAPCropped / _is6_DrawHBITMAPCropped

   Purpose :
   Draw an HBITMAP to a device context with cropping 

   The device referred to by hDC must support 
   the following RASTERCAPS values : 
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

  
   Param             Use
   ------------------------------------
   hDC               output device context
   hBmp              HBITMAP to draw
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)

   uSrcXStart        start drawing from this X position 
                     in the source

   uSrcYStart        start drawing from this Y position 
                     in the source

   uOutWidth         X pixels to draw (may be less than 
                     image width)

   uOutHeight        Y pixels to draw (may be less than 
                     image width)

   hPal              optional palette

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DrawHBITMAPCropped )(HDC hDC, HBITMAP hBmp, __int32 iXPos, __int32 iYPos, UINT32 uSrcXStart, UINT32 uSrcYStart, UINT32 uOutWidth,   UINT32 uOutHeight, HPALETTE hPal);

/*---------------------------------------------------------
   is6_DrawRGB / _is6_DrawRGB

   Purpose :
   Draw an RGB image to a device context

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

   When drawing on a 256 color display, the palette is 
   used in the creation of a temporary HBITMAP. This 
   will give better color accuracy than is possible with 
   the default Windows palette.

   Param             Use
   ------------------------------------
   hDC               output device context
   pRGB              RGB buffer to draw
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)

   hPal              optional palette

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DrawRGB )(HDC hDC, const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, __int32 iXPos, __int32 iYPos, HPALETTE hPal);

/*---------------------------------------------------------
   is6_DrawRGBA / _is6_DrawRGBA

   Purpose :
   Draw an RGBA image to a device context. The image is 
   blended with whatever happens to be in the DC.

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

   When drawing on a 256 color display, the palette is 
   used in the creation of a temporary HBITMAP. This 
   will give better color accuracy than is possible with 
   the default Windows palette.

   Param             Use
   ------------------------------------
   hDC               output device context
   pRGBA             RGBA buffer to draw
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   hPal              optional palette

    uFlags                bit        use
                            ---        ---
                             0            image is in BGRA order
                             1            flip image vertically

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DrawRGBA)(HDC outDC, const BYTE *pRGBA, UINT32 inWidth, UINT32 inHeight, UINT32 uRowStride, __int32 xPos, __int32 yPos, HPALETTE hPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DrawRGBAScaled / _is6_DrawRGBAScaled

   Purpose :
   Draw a scaled RGBA image to a device context. The image is 
   blended with whatever happens to be in the DC.

    The RGBA image is scaled before merging with the background.
    The background is not scaled.

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

   When drawing on a 256 color display, the palette is 
   used in the creation of a temporary HBITMAP. This 
   will give better color accuracy than is possible with 
   the default Windows palette.

   Param             Use
   ------------------------------------
   hDC               output device context
   pRGBA             RGBA buffer to draw
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   fScale             amount to scale the RGBA image
   hPal              optional palette

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_DrawRGBAScaled)(HDC outDC, const BYTE *pRGBA, UINT32 inWidth, UINT32 inHeight, UINT32 uRowStride, __int32 xPos, __int32 yPos, double fScale, HPALETTE hPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Draw8Bit / _is6_Draw8Bit

   Purpose :
   Draw an 8-bit image to a device context

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  


   When drawing on a 256 color display, the palette is 
   used in the creation of a temporary HBITMAP. This 
   will give better color accuracy than is possible with 
   the default Windows palette.
  
   Param             Use
   ------------------------------------
   hDC               output device context
   p8Bit             8-bit buffer to draw
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   pPal              image palette
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   hPal              optional screen palette

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_Draw8Bit)(HDC hDC, const BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, HPALETTE hPal);

/*---------------------------------------------------------
   is6_Draw1Bit / _is6_Draw1Bit

   Purpose :
   Draw an 1-bit image to a device context

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

  
   Param             Use
   ------------------------------------
   hDC               output device context
   p1Bit             1-bit buffer to draw
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   pPal              image palette
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   hPal              optional screen palette

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_Draw1Bit)(HDC hDC, const BYTE *p1Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, HPALETTE hPal);

/*---------------------------------------------------------
   is6_StretchDrawRGB / _is6_StretchDrawRGB

   Purpose :
   Draw an RGB image to a device context using
   StretchDIBits. This will stretch the image if the
   output dimensions do not match the image dimensions.
   
   To improve the results, try calling 
      SetStretchBltMode(hDC, COLORONCOLOR);
   before this function.
   
   This function is 10-20% faster than is6_DrawRGB, when not
   stretching.

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_STRETCHDIB

   Param             Use
   ------------------------------------
   hDC               output device context
   pRGB              RGB buffer to draw
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   uOutWidth         output width
   uOutHeight        output height

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_StretchDrawRGB)(HDC hDC, const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight);

/*---------------------------------------------------------
   is6_StretchDraw8Bit / _is6_StretchDraw8Bit

   Purpose :
   Draw an 8 bit image to a device context using
   StretchDIBits. This will stretch the image if the
   output dimensions do not match the image dimensions.

   To improve the results, try calling 
      SetStretchBltMode(hDC, COLORONCOLOR);
   before this function.
   
   This function is 10-20% faster than is6_Draw8Bit.
   
   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_STRETCHDIB
  
   Param             Use
   ------------------------------------
   hDC               output device context
   p8Bit             8-bit buffer to draw
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   pPal              image palette
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   uOutWidth         output width
   uOutHeight        output height

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_StretchDraw8Bit)(HDC hDC, const BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight);

/*---------------------------------------------------------
   is6_StretchDraw1Bit / _is6_StretchDraw1Bit

   Purpose :
   Draw an 1 bit image to a device context using
   StretchDIBits. This will stretch the image if the
   output dimensions do not match the image dimensions.

   Note that the stretching results are usually less than
   satisfactory. This is because the StretchDIBits function
   is really not very good at stretching. It is recommended
   that you use one of the ImgSource resizing functions to 
   do any image resizing.
   
   This function is 10-20% faster than is6_Draw1Bit.
   

   To improve the results, try calling 
      SetStretchBltMode(hDC, COLORONCOLOR);
   or 
      SetStretchBltMode(hDC, HALFTONE);
   before this function.

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_STRETCHDIB
  
   Param             Use
   ------------------------------------
   hDC               output device context
   p1Bit             1-bit buffer to draw
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   pPal              image palette
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   uOutWidth         output width
   uOutHeight        output height

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_StretchDraw1Bit)(HDC hDC, const BYTE *p1Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight);

/*---------------------------------------------------------
   is6_DrawTransparentBitmap / _is6_DrawTransparentBitmap

   Purpose :
   Draw an HBITMAP to a device context with one color
   transparent

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  
  
   Param             Use
   ------------------------------------
   hDC               output device context
   hBmp              HBITMAP to draw
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   clrTransparent    transparent color

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DrawTransparentBitmap )(HDC hDC, HBITMAP hBmp, __int32 iXPos, __int32 iYPos, COLORREF clrTransparent);

/*---------------------------------------------------------
   is6_DrawTransparentRGB / _is6_DrawTransparentRGB

   Purpose :
   Draw an RGB image to a device context with one color
   transparent

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
      RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  
  
   Param             Use
   ------------------------------------
   hDC               output device context
   pRGB              RGB image
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   clrTransparent    transparent color

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DrawTransparentRGB)(HDC hDC, const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, __int32 iXPos, __int32 iYPos, COLORREF clrTransparent);

/*---------------------------------------------------------
   is6_StretchDrawDIB / _is6_StretchDrawDIB

   Purpose :
   Draw a DIB image to a device context. This will stretch the
   image if the uOutWidth or uOutHeight params do not match
   the width and height of the image in the DIB.


   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_STRETCHDIB  

   Param             Use
   ------------------------------------
   hDC               output device context
   pDIB              DIB image
   iXPos             X position (may be negative)
   iYPos             Y position (may be negative)
   uOutWidth         output width
   uOutHeight        output height

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_StretchDrawDIB)(HDC hDC, const BITMAPINFOHEADER *pDIB, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight);

/*---------------------------------------------------------
   is6_StretchDrawDIBCropped / _is6_StretchDrawDIBCropped

   Purpose :
   Draw a DIB image to a device context. This will stretch the
   image if the uOutWidht or uOutHeight params do not match
   the width and height of the image in the DIB. This allows
   drawing sub-sections of the source image.


   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_STRETCHDIB  

   Param             Use
   ------------------------------------
   hDC               output device context
   pDIB              DIB image
   pSrcRect          rectangle from source image
   pDestRect         rectangle on output image

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_StretchDrawDIBCropped)(HDC hDC, const BITMAPINFOHEADER *pDIB, const RECT *pSrcRect, const RECT *pDestRect);

/*---------------------------------------------------------
   is6_LoadResourceBitmap / _is6_LoadResourceBitmap

   Purpose :
   Load a bitmap from a resource, using a caller-supplied
   palette. This generally gives better results than
   CBitmap::LoadBitmap.
   
    See is6_LoadResourceBitmapW is you're using 
    Unicode resource names.

   Param             Use
   ------------------------------------
   hInstance         calling app's hInstance
   pName             resource name
   hInPalette        palette to use, NULL if none

   Return
      HBITMAP object.
      Caller must call DeleteObject to release this memory
---------------------------------------------------------*/
_ISDeclSpec HBITMAP         _ISCConv _ISFn( is6_LoadResourceBitmap )(HINSTANCE hInstance, const char *pName, HPALETTE hInPalette);

/*---------------------------------------------------------
   is6_LoadResourceBitmapW / _is6_LoadResourceBitmapW

   Purpose :
   Load a bitmap from a resource, using a caller-supplied
   palette. This generally gives better results than
   CBitmap::LoadBitmap.

   This is the wide-character version. 
    See is6_LoadResourceBitmap for non-Unicode resource names.
   
   Param             Use
   ------------------------------------
   hInstance         calling app's hInstance
   pName             resource name (wide-character string)
   hInPalette        palette to use, NULL if none

   Return
      HBITMAP object.
      Caller must call DeleteObject to release this memory
---------------------------------------------------------*/
_ISDeclSpec HBITMAP         _ISCConv _ISFn( is6_LoadResourceBitmapW )(HINSTANCE hInstance, const wchar_t *pName, HPALETTE hInPalette);

/*---------------------------------------------------------
   is6_HBITMAPToRGB / _is6_HBITMAPToRGB

   Purpose :
   Convert an HBITMAP to an RGB buffer.

   Caller must allocate the output buffer:
   
     HBITMAP_Width * HBITMAP_Height * 3 BYTEs

   (see is6_GetHBITMAPDimensions)

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
  

   Param             Use
   ------------------------------------
   hBmp              input HBITMAP
   hDC               device context to use
   hPal              optional palette
   pOutBuf           output image

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_HBITMAPToRGB )(HBITMAP hBmp, HDC hDC, HPALETTE hPal, BYTE *pOutBuf);

/*---------------------------------------------------------
   is6_HBITMAPToRGBA / _is6_HBITMAPToRGBA

   Purpose :
   Convert an HBITMAP to an RGBA buffer

   Caller must allocate the output buffer:
   
     HBITMAP_Width * HBITMAP_Height * 4 BYTEs

   (see is6_GetHBITMAPDimensions)

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
  
   Param             Use
   ------------------------------------
   hBmp              input HBITMAP
   hDC               device context to use
   hPal              optional palette
   uDefAlpha         value to use if the source bitmap is not 32 bits

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_HBITMAPToRGBA )(HBITMAP hBmp, HDC hDC, HPALETTE hPal, UINT32 uDefAlpha, BYTE *pOutBuf);

/*---------------------------------------------------------
   is6_GetHBITMAPDimensions / _is6_GetHBITMAPDimensions

   Purpose :
   Get the width and height of an HBITMAP

   Param             Use
   ------------------------------------
   hBmp              input HBITMAP
   puWidth           receives width
   puHeight          receives height

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetHBITMAPDimensions )(HBITMAP hBmp, UINT32 *puWidth, UINT32 *puHeight);

/*---------------------------------------------------------
   is6_DCToRGB / _is6_DCToRGB

   Purpose :
   Capture a device context to an RGB buffer. 
   
   Caller must allocate output image.

   The device referred to by hDC must support 
   the following RASTERCAPS values : 
      RC_BITBLT
      RC_DI_BITMAP

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  
  
   Param             Use
   ------------------------------------
   hDC               device context to use
   uXStart           image rectangle left
   uYStart           image rectangle top
   uWidth            image rectangle width
   uHeight           image rectangle height
   pOutBuf           output image. 
   
   Return
      FALSE on failure
      
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DCToRGB )(HDC hDC, UINT32 uXStart, UINT32 uYStart, UINT32 uWidth, UINT32 uHeight, BYTE *pOutBuf);

/*---------------------------------------------------------
   is6_RGBToHBITMAP / _is6_RGBToHBITMAP

   Purpose :
   Create an HBITMAP based on an RGB buffer.

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
  
   Note:
   If this fails with large bitmaps, see is6_SetHBITMAPCreateMethod.

   Param             Use
   ------------------------------------
   pRGB              input RGB image
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
   hPal              optional palette
   hDC               device context to use

   Return
      HBITMAP object.
      Caller must call DeleteObject to release this memory
---------------------------------------------------------*/
_ISDeclSpec HBITMAP         _ISCConv _ISFn(is6_RGBToHBITMAP)(const BYTE * pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, HPALETTE hPal, HDC hDC);

/*---------------------------------------------------------
   is6_DIBToHBITMAP / _is6_DIBToHBITMAP

   Purpose :
   Create an HBITMAP based on a DIB.

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
  
   Param             Use
   ------------------------------------
   pDIB              input DIB
   hPal              optional palette
   hDC               device context to use

   Return
      HBITMAP object.
      Caller must call DeleteObject to release this memory
---------------------------------------------------------*/
_ISDeclSpec HBITMAP         _ISCConv _ISFn(is6_DIBToHBITMAP)(const BITMAPINFOHEADER * pDIB, HPALETTE hPal, HDC hDC);

/*---------------------------------------------------------
   is6_ImageToHBITMAP / _is6_ImageToHBITMAP

   Purpose :
   Create an HBITMAP from an image.

    The input image can be 1, 8, 16, 24 or 32 bits per pixel.

    You can also specify a crop rectangle to select
    a sub-rect of the input image to use as the source for
    the HBITMAP.

    Note:
    If the image is 1 bpp, pPal must point to a palette with two
    RGBQUADS.

    If the image is 8 bpp, pPal must point to a palette with 
    uPalColors RGBQUADs.

    Note:
    pClipRect can be used to select a sub-rect of the source image.
    Instead of creating an HBITMAP of the full image, the function
    will use the selected sub-rect. 

   The device referred to by hDC must support the 
   following RASTERCAPS values : 
      RC_DI_BITMAP
  
   Param             Use
   ------------------------------------
   pImage            input image
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per pixel row
    uBpp                    bits per pixel of source image
    pPal                    array of RGBQUADs used as the palette for 1 and 8 bpp images
    uPalColors            number of colors in pPal, for 8 bpp images

    pClipRect            optional RECT - can be used to select a 
                            sub-rect, instead of using the whole image.
                            pass NULL, to use the full image.

   hPal              optional palette to select into the DC before
                            creating the image.

   hDC               device context to use
    uFlags                unused


   Return
      HBITMAP object.
      Caller must call DeleteObject to release this memory
---------------------------------------------------------*/
_ISDeclSpec HBITMAP         _ISCConv _ISFn(is6_ImageToHBITMAP)(const BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uBpp, const RGBQUAD *pPal, UINT32 uPalColors, const RECT *pClipRect, HPALETTE hPal, HDC hDC, UINT32 uFlags);

/*---------------------------------------------------------
   is6_HBITMAPToDIB / _is6_HBITMAPToDIB

   Purpose :
   Create a 24-bit DIB from an HBITMAP
  
   Param             Use
   ------------------------------------
   HBITMAP              input HBITMAP
   hPal              optional palette

   Return
      DIB
      Caller must call GlobalFree to release this memory
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_HBITMAPToDIB )(HBITMAP hBmp, HPALETTE hPal);

/*---------------------------------------------------------
   is6_DIBWidth / _is6_DIBWidth

   Purpose :
   Get the width of a DIB image
  
   Param             Use
   ------------------------------------
   pDIB              pointer to the DIB
   puWidth           receives image width

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DIBWidth)(const BITMAPINFOHEADER *pDIB, UINT32 *puWidth);

/*---------------------------------------------------------
   is6_DIBHeight / _is6_DIBHeight

   Purpose :
   Get the height of a DIB image
  
   Param             Use
   ------------------------------------
   pDIB              pointer to the DIB
   piHeight          receives image height

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DIBHeight)(const BITMAPINFOHEADER *pDIB, __int32 *piWidth);

/*---------------------------------------------------------
    is6_DIBBitCount / _is6_DIBBitCount

    Purpose :
    Get the number of bits used by a pixel in a DIB image.
    See MSDN for BITMAPINFOHEADER for an explanation of
    the values.
  
    Param              Use
    ------------------------------------
    pDIB               pointer to the DIB
    puBitCount         receives image bit count

    Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DIBBitCount)(const BITMAPINFOHEADER *pDIB, UINT32 *puBitCount);

/*---------------------------------------------------------
    is6_DIBPixelStart / _is6_DIBPixelStart

    Purpose :
    Return a pointer to the start of the DIB pixels.
  
    Param              Use
    ------------------------------------
    pDIB               pointer to the DIB

    Return
      NULl on failure
---------------------------------------------------------*/
_ISDeclSpec BYTE*           _ISCConv _ISFn(is6_DIBPixelStart)(const BITMAPINFOHEADER *pDIB);

/*---------------------------------------------------------
   is6_GetDIBPalette / _is6_GetDIBPalette

   Purpose :
   Given a pointer to a DIB, return the pointer to 
   the palette it contains (if there is one)
  
   Param             Use
   ------------------------------------
   pDIB              pointer to the DIB
   puColorsUsed      returns the number of colors used

   Return
      NULL on failure
---------------------------------------------------------*/
_ISDeclSpec RGBQUAD*      _ISCConv _ISFn(is6_GetDIBPalette)(const BITMAPINFOHEADER *pDIB, UINT32 *puColorsUsed);

/*---------------------------------------------------------
   is6_GetDIBRowStride / _is6_GetDIBRowStride

   Purpose :
   Given a width and a bit depth, return the number of
   bytes in a DIB pixel row.
  
   Param             Use
   ------------------------------------
   uWidth            number of pixels in a row
   uBitDepth         bits per pixel

   Return
      NULL on failure
---------------------------------------------------------*/
_ISDeclSpec UINT32      _ISCConv _ISFn( is6_GetDIBRowStride )(UINT32 uWidth, UINT32 uBitDepth);

/*---------------------------------------------------------
   is6_RGBQUADToHPALETTE / _is6_RGBQUADToHPALETTE

   Purpose :
   Create an HPALETTE based on an array of RGBQUADs
   
   Param                Use
   ------------------------------------
   pPal                 array of RGBQUADs
   uColors              number of entries in pPal
   
   Return
      DIB
      Caller must call DeleteObject to release this memory
---------------------------------------------------------*/
_ISDeclSpec HPALETTE         _ISCConv _ISFn(is6_RGBQUADToHPALETTE)(const RGBQUAD *pPal, UINT32 uColors);

/*---------------------------------------------------------
   is6_DIBToRGB / _is6_DIBToRGB

   Purpose :
   Convert a DIB to an RGB buffer. Caller must allocate
   the output buffer:
   
     is6_DIBWidth * is6_DIBHeight * 3
   
   Param             Use
   ------------------------------------
   pDIB              pointer to a DIB
   pOutBuf           receives RGB image

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DIBToRGB)(const BITMAPINFOHEADER * pDIB, BYTE *pOutBuf);

/*---------------------------------------------------------
   is6_DIBToRGBA / _is6_DIBToRGBA

   Purpose :
   Convert a DIB to an RGBA buffer. Caller must allocate
   the output buffer:
   
     is6_DIBWidth * is6_DIBHeight * 4
   
   Param             Use
   ------------------------------------
   pDIB              pointer to a DIB
   uDefAlpha         default alpha value, if the input < 32 bits
   pOutBuf           receives RGBA image

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DIBToRGBA)(const BITMAPINFOHEADER * pDIB, UINT32 uDefAlpha, BYTE *pOutBuf);

/*---------------------------------------------------------
   is6_DIBPartsToRGB / _is6_DIBPartsToRGB

   Purpose :
   Convert a DIB to an RGB buffer. Caller must allocate
   the output buffer:
   
     is6_DIBWidth * is6_DIBHeight * 3

   
   Param             Use
   ------------------------------------
   pDIBBits          pointer to the DIB image bits
   pBMIH             pointer to the BITMAPINFOHEADER
   pPalette          pointer to the palette data
   pOutBuf           receives RGB image. (null for 24, 32 bit images)

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DIBPartsToRGB)(const BYTE *pDIBBits, const BITMAPINFOHEADER * pBMIH, RGBQUAD *pPalette, BYTE *pOutBuf);

/*---------------------------------------------------------
   is6_DIBToColormapped / _is6_DIBToColormapped

   Purpose :
   Convert a DIB to a colormapped buffer. Caller must allocate
   the output buffer:
   
     is6_DIBWidth * is6_DIBHeight BYTEs

   
   Param             Use
   ------------------------------------
   pDIB              pointer to a DIB
   uDesiredBitDepth  bit depth to return 
                     (currently only 1 and 8 are supported)
   puColorsInPal     receives number of colors in palette
   pPal              receives palette (should be 256 RGBQUADs)
   pOutBuf           receives 8 bit image

   Return
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_DIBToColormapped)(const BITMAPINFOHEADER * pDIB, UINT32 uDesiredBitDepth, UINT32 *uColorsInPal, RGBQUAD *pPal, BYTE *pOutBuf);

/*---------------------------------------------------------
   is6_RGBToDIB / _is6_RGBToDIB

   Purpose :
   Create a 24-bit DIB from an RGB buffer
   
   Note:
   This function allows you to specify the output buffer 
   that the DIB is written to. Do this by setting the 
   pOutBuf param to a non-NULL value. To calculate the amount 
   of memory required for the pOutBuf, see is6_GetISDIBSize. 
   If you don't provide an output buffer, ImgSource will 
   allocate the memory required and will return an HGLOBAL 
   handle for the memory.
   
   Param             Use
   ------------------------------------
   pRGB              input RGB image
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per input pixel row

   pOutBuf           output buffer. 
                     NULL if you want ImgSource to allocate
                     the memory.

   uFlags            unused

   Return
      If ImgSource allocates the DIB, this will return a 
      handle to the DIB, and the caller must call 
      GlobalFree to release the DIB memory. 

      If the caller allocates the memory (and specifies it with
      pOutBuf) the return value is pOutBuf. The caller should
      deallocate it in whatever manner is appropriate.

      See Notes above.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn(is6_RGBToDIB)(const BYTE * pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, BYTE *pOutBuf, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBAToDIB / _is6_RGBAToDIB

   Purpose :
   Create a 32-bit DIB from an RGBA buffer
   
   Note:
   This function allows you to specify the output buffer 
   that the DIB is written to. Do this by setting the 
   pOutBuf param to a non-NULL value. To calculate the amount 
   of memory required for the pOutBuf, see is6_GetISDIBSize. 
   If you don't provide an output buffer, ImgSource will 
   allocate the memory required and will return an HGLOBAL 
   handle for the memory.
   
   Param             Use
   ------------------------------------
   pRGB              input RGBA image
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per input pixel row
   pOutBuf           output buffer. 

                     NULL if you want ImgSource to allocate
                     the memory.

   uFlags            unused

   Return
      If ImgSource allocates the DIB, this will return a 
      handle to the DIB, and the caller must call 
      GlobalFree to release the DIB memory. 

      If the caller allocates the memory (and specifies it with
      pOutBuf) the return value is pOutBuf. The caller should
      deallocate it in whatever manner is appropriate.

      See Notes above.

---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn(is6_RGBAToDIB)(const BYTE * pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, BYTE *pOutBuf, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGB16ToDIB / _is6_RGB16ToDIB

   Purpose :
   Create a 16-bit DIB from an RGB-16 buffer
   
   Note:
   This function allows you to specify the output buffer 
   that the DIB is written to. Do this by setting the 
   pOutBuf param to a non-NULL value. To calculate the amount 
   of memory required for the pOutBuf, see is6_GetISDIBSize. 
   If you don't provide an output buffer, ImgSource will 
   allocate the memory required and will return an HGLOBAL 
   handle for the memory.
   
   Param             Use
   ------------------------------------
   pRGB              input RGB-16 image
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per input pixel row

   pOutBuf           output buffer. 
                     NULL if you want ImgSource to allocate
                     the memory.

   uFlags            unused

   Return
      If ImgSource allocates the DIB, this will return a 
      handle to the DIB, and the caller must call 
      GlobalFree to release the DIB memory. 

      If the caller allocates the memory (and specifies it with
      pOutBuf) the return value is pOutBuf. The caller should
      deallocate it in whatever manner is appropriate.

      See Notes above.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn(is6_RGB16ToDIB)(const BYTE * pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, BYTE *pOutBuf, UINT32 uFlags);

/*---------------------------------------------------------
   is6_8BitToDIB / _is6_8BitToDIB

   Purpose :
   Create a DIB from an 8-bit buffer and palette
   
   If you specify 1 or 4 for uBpP, the DIB will be a
   1 or 4 (respectively) bit-per-pixel DIB.
   
   Note:
   This function allows you to specify the output buffer 
   that the DIB is written to. Do this by setting the 
   pOutBuf param to a non-NULL value. To calculate the amount 
   of memory required for the pOutBuf, see is6_GetISDIBSize. 
   If you don't provide an output buffer, ImgSource will 
   allocate the memory required and will return an HGLOBAL 
   handle for the memory.

   Param             Use
   ------------------------------------
   p8Bit             input 8-bit image
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per input pixel row
   uColors           colors in pPal
   uBPP              1,4 or 8 - bits per pixel for output DIB
   pPal              image palette (256 colors max)

   pOutBuf           output buffer. 
                     NULL if you want ImgSource to allocate
                     the memory.

   uFlags            unused

   Return
      If ImgSource allocates the DIB, this will return a 
      handle to the DIB, and the caller must call 
      GlobalFree to release the DIB memory. 

      If the caller allocates the memory (and specifies it with
      pOutBuf) the return value is pOutBuf. The caller should
      deallocate it in whatever manner is appropriate.

      See Notes above.

---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn(is6_8BitToDIB)(const BYTE * p8Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uColors, UINT32 uBPP, const RGBQUAD *pPal, BYTE *pOutBuf, UINT32 uFlags);

/*---------------------------------------------------------
   is6_1BitToDIB / _is6_1BitToDIB

   Purpose :
   Create a DIB from a 1-bit buffer and palette. Input
   rows must start on whole BYTE boundaries
   
   Note:
   This function allows you to specify the output buffer 
   that the DIB is written to. Do this by setting the 
   pOutBuf param to a non-NULL value. To calculate the amount 
   of memory required for the pOutBuf, see is6_GetISDIBSize. 
   If you don't provide an output buffer, ImgSource will 
   allocate the memory required and will return an HGLOBAL 
   handle for the memory.
   
   Param             Use
   ------------------------------------
   p1Bit             input 1-bit image
   uWidth            image width
   uHeight           image height
   uRowStride        bytes per input pixel row
   pPal              image palette (2 colors)

   pOutBuf           output buffer. 
                     NULL if you want ImgSource to allocate
                     the memory.

   uFlags            unused

   Return
      If ImgSource allocates the DIB, this will return a 
      handle to the DIB, and the caller must call 
      GlobalFree to release the DIB memory. 

      If the caller allocates the memory (and specifies it with
      pOutBuf) the return value is pOutBuf. The caller should
      deallocate it in whatever manner is appropriate.

      See Notes above.

---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn(is6_1BitToDIB)(const BYTE * p1Bit, UINT32 uWidthPix, UINT32 uHeight, UINT32 uRowStride, const RGBQUAD *pPal, BYTE *pOutBuf, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetISDIBSize / _is6_GetISDIBSize

   Purpose :
   Calculate the number of BYTEs required to create a DIB
   with the given specifications. 

   Note:
   This value is only correct for DIBs created by the following
   functions:
      is6_RGBToDIB
      is6_RGBAToDIB
      is6_RGB16ToDIB
      is6_8BitToDIB
      is6_1BitToDIB

   The value returned may not be applicable for DIBs created outside
   of ImgSource.

   Param                Use
   ------------------------------------
   uWidth               image width
   uHeight              image height
   uBitsPerPixel        1, 8, 24 or 32
   uPalColors           number of colors in the palette (1, 8 bpp only)
   
   Return
      DIB
      Caller must call GlobalFree to release this memory
---------------------------------------------------------*/
_ISDeclSpec UINT32      _ISCConv _ISFn( is6_GetISDIBSize )(UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uPalColors);

/*---------------------------------------------------------
   is6_SetHBITMAPCreateMethod / _is6_SetHBITMAPCreateMethod

   Purpose :
   Set the method used to create HBITMAPs.

   This affects all internal and external HBITMAP creation 
   operations.
   
   CreateDIBitmap is the standard method of creating an
   HBITMAP. 

   CreateDIBSection is much faster for large images, but
   the HBITMAPs created are subject to the following
   restriction and/or bugs by Windows:

   1) "Dithered brushes do not work in device contexts that 
   have DIB sections selected into them. If you create an 
   HBITMAP using CreateDIBSection(), select it into a memory 
   DC, and then attempt to use a dithered brush returned by
   CreateSolidBrush() to draw into the DIB section, the 
   brush will not be drawn with a dithered color. ... This
   behavior is by design." Microsoft Win32 Software 
   Development Kit (SDK) versions 3.5, 3.51, 4.0

   2) "When using BitBlt()with DIBSections to create a 
   monochrome mask, the target pixel color is chosen without 
   regard for the background color in the target device 
   context. The AND mask that results is not what the 
   programmer intended, and subsequent transparent blts using 
   that mask do not maintain proper transparency...This
   behavior is by design." Microsoft Windows NT versions 3.5, 
   3.51 Microsoft Windows 95

   3) "The program displays the wrong colors or causes a 
   general protection (GP) fault when using a DIB section and 
   a custom palette in a memory DC... There is a problem in 
   Windows 95 where an internal palette structure is not set up 
   until the palette is selected into a screen DC. "
  
   These are quotes from the Microsoft Developers Network 
   (MSDN).

   Param             Use
   ------------------------------------
   uMethod              0 = use CreateDIBitmap
                        1 = use CreateDIBSection

   Return
      none
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_SetHBITMAPCreateMethod )(UINT32 uMethod);

#ifdef __cplusplus
}
#endif

#endif