/*---------------------------------------------------------
   ISImageText.h : part of the ImgSource library.
   
   Image processing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is6_TEXTPROCH
#define is6_TEXTPROCH

// shut up, stupid compiler!
#pragma warning(disable:4786)

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_DrawTextOnRGB / _is6_DrawTextOnRGB

   Purpose : 
   Draw text on an RGB image

   Text transparency is controlled with is6_SetTextTransparency.

   Note:
   Even though Unicode text is stored as a sequence of 
   "unsigned short" values, you still need to pass a
   pointer to a char to the text output functions. To do this,
   simply cast your Unicode string to (const char *):

      is6_DrawTextOnRGB(pImg, w, h, (const char *)pUnicodeText...);

   Unicode output mode is only supported by Windows NT, 
   Windows 2000 or higher. Text drawing will fail if you 
   attempt to draw Unicode text on Windows 95 or 98.
   ImgSource will not test to verify the proper OS.

   Note:
    The clipping, wrapping and justification behavior of this function
    should be identical to DrawTextEx. The only difference is that this
    function does not incorrectly clip the under/overhang of italic
    text, like DrawTextEx may. 

    If you use the DT_NOCLIP flag in the uDrawText param, the rendered 
    text may extend past your bounding rectangle. If you don't use 
    DT_NOCLIP, the text will be bound by your rectangle.

   Callback is called once per font height line

   Param             Use
   ------------------------------------
   pRGB              RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per image pixel row
   pText             text to write
   pFont             name of font
   uFontPixels       height of font
   textRect          bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      0          assume BGR order

                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

                      6          take advantage of Windows' built-in text anti-aliasing, if possible.
                                 this generates smoother text than the standard method. 

   clrText           color of text

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_DrawTextOnRGB )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pText, const char *pFontName, UINT32 uFontPixels, const RECT * pTextRect, UINT32 uDrawFlags, UINT32 uFlags, COLORREF clrText);

/*---------------------------------------------------------
   is6_DrawTextOnRGB2 / _is6_DrawTextOnRGB2

   Purpose : 
   Draw text on an RGB image

   Text transparency is controlled with is6_SetTextTransparency.

   Callback is called once per font height line

   See Notes for is6_DrawTextOnRGB.

   Param             Use
   ------------------------------------
   pRGB              RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per image pixel row
   pText             text to write
   pLogFont          valid LOGFONT structure
   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      0          assume BGR order

                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

                      6          take advantage of Windows' built-in text anti-aliasing, if possible.
                                 this generates smoother text than the standard method. 

   clrText           color of text

    
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_DrawTextOnRGB2 )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pText, const LOGFONT *pLogFont, const RECT *textRect, UINT32 uDrawFlags, UINT32 uFlags, COLORREF clrText);

/*---------------------------------------------------------
   is6_DrawTextOnRGBA / _is6_DrawTextOnRGBA

   Purpose : 
   Draw text on an RGBA image

   Text transparency is controlled with is6_SetTextTransparency.

    See Notes for is6_DrawTextOnRGB.

    Note:
    Standard blending modes (bits 3 & 4 of uFlags, when bit 6 set to 0):

        If neither bit is set, text is blended into the RGB channels using the global 
        text transparency value (is6_SetTextTransparency). The Alpha channel is set 
        to the global text transparency for text foreground pixels, and to 0 for 
        text background pixels.

        If only bit 3 is set, text goes to the Alpha channel only, using the high
        byte of clrText for text pixels - background pixels are not modified. 
        No transparency is used. The R, G, and B channels are not modified.

        If only bit 4 is set, text goes to all channels, using the colors in clrText.
        Alpha is set to 0 for background pixels. No transparency is used.

        If both bits are set, text is blended into the RGB channels using the global 
        text transparency value. The Alpha channel is not modified.

    Anti-aliased blending modes (bits 3 & 4 of uFlags, when bit 6 set to 1):

        The global text transparency value affects all modes.

        If neither bit is set, the foreground of the text is blended into all channels.
 
        If only bit 3 is set, the foreground of the text is set into the Alpha 
        channel only. RGB channels are not modified.

        If only bit 4 is set, all pixels in the bounding rect are set to the clrText
        color, and the alpha channel is set to the text image.

        If both bits are set, the foreground of the text is blended into the RGB channels. 
        The Alpha channel is not modified.


   Callback is called once per font height line


   Param             Use
   ------------------------------------
   pRGBA             RGBA image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per image pixel row
   pText             text to write
   pFont             name of font
   uFontPixels       height of font
   textRect          bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      0          assume BGR order

                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

                      3-4        blending mode options. see Notes

                      6          take advantage of Windows' built-in text anti-aliasing, if possible.
                                 this generates smoother text than the standard method. 
                                 see Notes.

   clrText           color of text

   
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_DrawTextOnRGBA )(BYTE *pRGBA, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pText, const char *pFontName, UINT32 uFontPixels, const RECT * pTextRect, UINT32 uDrawFlags, UINT32 uFlags, COLORREF clrText);

/*---------------------------------------------------------
   is6_DrawTextOnRGBA2 / _is6_DrawTextOnRGBA2

   Purpose : 
   Draw text on an RGBA image

   Text transparency is controlled with is6_SetTextTransparency.

   See Notes for is6_DrawTextOnRGB.

    See blending Notes for is6_DrawTextOnRGBA.

    Callback is called once per font height line

   Param             Use
   ------------------------------------
   pRGBA             RGBA image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per image pixel row
   pText             text to write
   pLogFont          valid LOGFONT structure
   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      0          assume BGRA order

                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

                      3-4        see notes for is6_DrawTextOnRGBA

                      6          see notes for is6_DrawTextOnRGBA


   clrText           color of text


   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_DrawTextOnRGBA2 )(BYTE *pRGBA, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pText, const LOGFONT *pLogFont, const RECT *textRect, UINT32 uDrawFlags, UINT32 uFlags, COLORREF clrText);

/*---------------------------------------------------------
   is6_DrawTextOn8Bit / _is6_DrawTextOn8Bit

   Purpose : 
   Draw text on an 8-bit image

   Text transparency is controlled with is6_SetTextTransparency.

   Callback is called once per font height line

   See Notes for is6_DrawTextOnRGB.

   Param             Use
   ------------------------------------
   p8Bit             8-bit image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per image pixel row
   pText             text to write
   pFont             name of font
   uFontPixels       height of font
   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

   uTextColorIndex   palette index for text color
   pPal              palette 
   uPalColors        number of colors in the palette

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_DrawTextOn8Bit )(BYTE *p8Bit, UINT32 width, UINT32 height, UINT32 uRowStride, const char *drawText, const char *fontName, UINT32 uFontPixels, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, UINT32 uTextColorIndex, RGBQUAD *pPal, UINT32 uPalColors);

/*---------------------------------------------------------
   is6_DrawTextOn8Bit2 / _is6_DrawTextOn8Bit2

   Purpose : 
   Draw text on an 8-bit image

   Text transparency is controlled with is6_SetTextTransparency.

   Callback is called once per font height line

   See Notes for is6_DrawTextOnRGB.

   Param             Use
   ------------------------------------
   p8Bit             8-bit image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per image pixel row
   pText             text to write
   pLogFont          valid LOGFONT structure
   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      1          vertically invert the text and the 
                                 bounding rect on the image. 
                      2          if set, input text is Unicode

   uTextColorIndex   palette index for text color
   pPal              palette 
   uPalColors        number of colors in the palette

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_DrawTextOn8Bit2 )(BYTE *p8Bit, UINT32 width, UINT32 height, UINT32 uRowStride, const char *drawText, const LOGFONT *pLogFont, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, UINT32 uTextColorIndex, RGBQUAD *pPal, UINT32 uPalColors);

/*---------------------------------------------------------
   is6_DrawTextOn1Bit / _is6_DrawTextOn1Bit

   Purpose : 
   Draw text on a 1-bit image

   Callback is called once per font height line

   See Notes for is6_DrawTextOnRGB.

   Param             Use
   ------------------------------------
   p1Bit             ont bit image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per image pixel row
   pText             text to write
   pFont             name of font
   uFontPixels       height of font
   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

   uTextBit          palette index for text color (1 or 0)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_DrawTextOn1Bit )(BYTE *p1Bit, UINT32 width, UINT32 height, UINT32 uRowStride, const char *drawText, const char *fontName, UINT32 uFontPixels, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, UINT32 uTextBit);

/*---------------------------------------------------------
   is6_DrawTextOn1Bit2 / _is6_DrawTextOn1Bit2

   Purpose : 
   Draw text on a 1-bit image

   Callback is called once per font height line

   See Notes for is6_DrawTextOnRGB.

   Param             Use
   ------------------------------------
   p1Bit             ont bit image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per image pixel row
   pText             text to write
   pLogFont          valid LOGFONT structure
   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

   uTextBit          palette index for text color (1 or 0)


   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_DrawTextOn1Bit2 )(BYTE *p1Bit, UINT32 width, UINT32 height, UINT32 uRowStride, const char *drawText, const LOGFONT *pLogFont, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, UINT32 uTextBit);

/*---------------------------------------------------------
   is6_GetTextLineSize / _is6_GetTextLineSize

   Purpose : 
   Determine pixels required to draw a single line of text.
   This returns size of text as would be drawn by any of the
   ImgSource text drawing functions.

   See Notes for is6_DrawTextOnRGB.

   Note:
   Unfortunately, it is not possible to get a perfectly accurate 
   bounding rectangle if you are using an italic font. The first 
   and last characters on a line may extend horizontaly past the 
   edges of the rectangle that Windows reports for the bounding 
   rectangle (and which ImgSource then passes back, here). ImgSource
    will correct for these when drawing, but it cannot correct for 
    them here. If this is a problem, add the width of one blank 
    character the the front and end of the string, when using italic 
    text.

   Note:
   To get an accurate size for the "SmoothText" functions, first 
   multiply the font size by the smooth text "smoothing factor"
   then divide the result from this function by the smoothing factor:

      1. new_font_size = font_size x smoothing_factor
      2. is6_GetTextLineSize(using new_font_size)
      3a. width = (pTextRect.right - pTextRect.left) / smoothing_factor 
      3b. height = (pTextRect.bottom - pTextRect.top) / smoothing_factor 

   Callback is not called

   Param             Use
   ------------------------------------
   drawText          text to measure
   fontName          name of font
   fontPixels        height of font
   pTextRect         bounding rect for the text
   uDrawFlags        flags for DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      2          if set, input text is Unicode
                            
   pWidth            receives text width
   pHeight           receives text height


   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_GetTextLineSize )(const char *drawText, const char *fontName, UINT32 fontPixels, const RECT* pTextRect, UINT32 uDrawFlags, UINT32 uFlags, UINT32 *pWidth, UINT32 *pHeight);

/*---------------------------------------------------------
   is6_GetTextLineSize2 / _is6_GetTextLineSize2

   Purpose : 
   Determine pixels required to draw a single line of text.
   This returns size of text as would be drawn by the
   ImgSource text drawing functions.

   Callback is not called

   See Notes for is6_DrawTextOnRGB, is6_GetTextLineSize

   Param             Use
   ------------------------------------
   drawText          text to measure
   pLogFont          valid LOGFONT structure
   pTextRect         bounding rect for the text
   uDrawFlags        flags for DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      2          if set, input text is Unicode

   pWidth            receives text width
   pHeight           receives text height

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_GetTextLineSize2 )(const char *drawText, const LOGFONT *pLogFont, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, UINT32 *pWidth, UINT32 *pHeight);

/*---------------------------------------------------------
   is6_DrawSmoothTextOnRGB / _is6_DrawSmoothTextOnRGB

   Purpose : 
   Draw text on an RGB image. The text is smoothed with
   resizing and masking to simulate anti-aliasing.

   Text transparency is controlled with is6_SetTextTransparency.

   The higher the smoothing factor, the more memory
   this function requires and the slower it will be.

    By default, this function clips text to the bounding 
    rectangle. This is somewhat different from other text 
    rendering functions, which can go outside the bounding rect, if
    the DT_NOCLIP flag is passed as part of the DrawTextEx flags.
    
    You can make this function follow the clipping rules of the
    other functions by setting bit 7.

   See notes for is6_DrawTextOnRGB.

   Callback is called once per font height line (bit 7 set), or 
    once per bounding rect line (bit 7 clear).

   Param             Use
   ------------------------------------
   pRGB              RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per pixel row
   pText             text to write
   pFont             name of font
   uFontPixels       height of font

   uSmoothingFactor  amount of smoothing 
                        1 equals no smoothing
                        2 is good
                        4 is better (recommended)

                        higher values are possible, 
                        but not recommended

   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      0          assume BGR order

                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

                      7          use clipping DrawTextEx clipping rules

   clrText           color of text

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DrawSmoothTextOnRGB )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pText, const char *pFontName, UINT32 uFontHeight, UINT32 uSmoothingFactor, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, COLORREF textColor);

/*---------------------------------------------------------
   is6_DrawSmoothTextOnRGB2 / _is6_DrawSmoothTextOnRGB2

   Purpose : 
   Draw text on an RGB image using a LOGFONT. The text 
   is smoothed with resizing and masking to simulate 
   anti-aliasing.

   Text transparency is controlled with is6_SetTextTransparency.

   The higher the smoothing factor, the more memory
   this funciton requires and the slower it will be.

    By default, this function clips text to the bounding 
    rectangle. This is somewhat different from other text 
    rendering functions, which can go outside the bounding rect, if
    the DT_NOCLIP flag is passed as part of the DrawTextEx flags.
    
    You can make this function follow the clipping rules of the
    other functions by setting bit 7.

   Callback is called once per font height line (bit 7 set), or 
    once per bounding rect line (bit 7 clear).
    
   See notes for is6_DrawTextOnRGB.

   Param             Use
   ------------------------------------
   pRGB              RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per pixel row
   pText             text to write
   pLogFont          valid LOGFONT structure

   uSmoothingFactor  amount of smoothing 
                        1 equals no smoothing
                        2 is good
                        4 is better (recommended)

                        higher values are possible, 
                        but not recommended

   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      0          assume BGR order

                      1          vertically invert the text and the 
                                 bounding rect on the image. 
                 
                      2          if set, input text is Unicode

                      7          use clipping DrawTextEx clipping rules

   clrText           color of text


   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DrawSmoothTextOnRGB2 )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pText, const LOGFONT *pLogFont, UINT32 uSmoothingFactor, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, COLORREF textColor);

/*---------------------------------------------------------
   is6_DrawSmoothTextOnRGBA / _is6_DrawSmoothTextOnRGBA

   Purpose : 
   Draw text on an RGBA image. The text is smoothed with
   resizing and masking to simulate anti-aliasing.

   Text transparency is controlled with is6_SetTextTransparency.

   The higher the smoothing factor, the more memory
   this function requires and the slower it will be.

    By default, this function clips text to the bounding 
    rectangle. This is somewhat different from other text 
    rendering functions, which can go outside the bounding rect, if
    the DT_NOCLIP flag is passed as part of the DrawTextEx flags.
    
    You can make this function follow the clipping rules of the
    other functions by setting bit 7.

   Callback is called once per font height line (bit 7 set), or 
    once per bounding rect line (bit 7 clear).

   See notes for is6_DrawTextOnRGB.

   Param             Use
   ------------------------------------
   pRGBA             RGBA image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per pixel row
   pText             text to write
   pFont             name of font
   uFontPixels       height of font

   uSmoothingFactor  amount of smoothing 
                        1 equals no smoothing
                        2 is good
                        4 is better (recommended)

                        higher values are possible, 
                        but not recommended

   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      0          assume BGR order

                      1          vertically invert the text and the 
                                 bounding rect on the image. 

                      2          if set, input text is Unicode

                      5          if set, do not modify the alpha channel
                                                                                 
                      7          use clipping DrawTextEx clipping rules

   clrText           color of text . high byte is Alpha value. (see RGBA(,,,) macro)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DrawSmoothTextOnRGBA )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pText, const char *pFontName, UINT32 uFontHeight, UINT32 uSmoothingFactor, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, COLORREF textColor);

/*---------------------------------------------------------
   is6_DrawSmoothTextOnRGBA2 / _is6_DrawSmoothTextOnRGBA2

   Purpose : 
   Draw text on an RGBA image using a LOGFONT. The text 
   is smoothed with resizing and masking to simulate 
   anti-aliasing.

    Text transparency is controlled with is6_SetTextTransparency.

   The higher the smoothing factor, the more memory
   this funciton requires and the slower it will be.

    By default, this function clips text to the bounding 
    rectangle. This is somewhat different from other text 
    rendering functions, which can go outside the bounding rect, if
    the DT_NOCLIP flag is passed as part of the DrawTextEx flags.
    
    You can make this function follow the clipping rules of the
    other functions by setting bit 7.

   Callback is called once per font height line (bit 7 set), or 
    once per bounding rect line (bit 7 clear).
    
   See notes for is6_DrawTextOnRGB.

   Param             Use
   ------------------------------------
   pRGBA             RGBA image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per pixel row
   pLogFont          valid LOGFONT structure

   uSmoothingFactor  amount of smoothing 
                        1 equals no smoothing
                        2 is good
                        4 is better (recommended)

                        higher values are possible, 
                        but not recommended

   pTextRect         bounding rectangle for the text
   uDrawFlags        flags to be passed to DrawTextEx

   uFlags            bit         purpose
                     ---         -------
                      0          assume BGR order

                      1          vertically invert the text and the 
                                 bounding rect on the image. 
                 
                      2          if set, input text is Unicode

                      5          if set, do not modify the alpha channel

                      7          use clipping DrawTextEx clipping rules

   clrText           color of text . high byte is Alpha value. (see RGBA(,,,) macro)


   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DrawSmoothTextOnRGBA2 )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pText, const LOGFONT *pLogFont, UINT32 uSmoothingFactor, const RECT *pTextRect, UINT32 uDrawFlags, UINT32 uFlags, COLORREF textColor);

/*---------------------------------------------------------
   is6_SetTextTransparency / _is6_SetTextTransparency

   Purpose : 
   Set the transparency that is6_DrawText* and 
   is6_DrawSmoothText* use.

   Param                Use
   ------------------------------------
   fTransparency        text transparency (0 <-> 1)
                        0 = completely transparent (invisible)
                        1.0 = completely opaque (default)

---------------------------------------------------------*/ 
_ISDeclSpec void         _ISCConv _ISFn( is6_SetTextTransparency )(double fTransparency);

#ifdef __cplusplus
}
#endif

#endif