/*---------------------------------------------------------
   ISIOManagers.h : part of the ImgSource library.
   
   Source and destination manager functions.

   See ImgSource.rtf for usage notes.

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is5_SOURCEFILEIOH
#define is5_SOURCEFILEIOH

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif


////////////////////////////////////////////////////////
// Source open

/*---------------------------------------------------------
   is5_OpenFileSource / _is5_OpenFileSource

   Purpose : 
   Create a file source object.

	See is5_OpenFileSourceW, for wide-character filenames.

   Param                Use
   ------------------------------------
   pFileName            input file path

   Return
   ------
   NULL on failure, else a valid ImgSource source object.
   You must call is5_CloseSource to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISSRC     	_ISCConv _ISFn( is5_OpenFileSource )(const char * pFileName);
                       						 
/*---------------------------------------------------------
   is5_OpenFileSourceW / _is5_OpenFileSourceW

   Purpose : 
   Create a file source object, using a wide character filename.

   Param                Use
   ------------------------------------
   pFileNameW           input file path

   Return
   ------
   NULL on failure, else a valid ImgSource source object.
   You must call is5_CloseSource to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISSRC     	_ISCConv _ISFn( is5_OpenFileSourceW )(const wchar_t * pFileNameW);

/*---------------------------------------------------------
   is5_OpenHandleSource / _is5_OpenHandleSource

   Purpose : 
   Create a source object using a Windows HANDLE.

   The HANDLE must already be opened and ready for reading.
   
   If you are using a HANDLE to a non-seeking device, such as a pipe 
   or a communications channel, some ImgSource file format reading 
   and writing operations may fail. Use ::GetFileType to determine 
   the HANDLE type, if unsure.

   Calling is5_CloseSource will not close the handle, that is left
   to the caller. But, you still need to call is5_CloseSource to
   release the ImgSource source object.

   Param                Use
   ------------------------------------
   handle               Windows HANDLE

   Return
   ------
   NULL on failure, else a valid ImgSource source object.
   
---------------------------------------------------------*/
_ISDeclSpec HISSRC     	_ISCConv _ISFn( is5_OpenHandleSource )(HANDLE handle);

/*---------------------------------------------------------
   is5_OpenMemorySource / _is5_OpenMemorySource

   Purpose : 
   Create a memory-based source object.

   Param                Use
   ------------------------------------
   pInBuf               input buffer
   uBufLen              length of input buffer

   Return
   ------
   NULL on failure, else a valid ImgSource source object.
   You must call is5_CloseSource to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISSRC     	_ISCConv _ISFn( is5_OpenMemorySource )(const BYTE *pInBuf, const UINT32 uBufLen);

/*---------------------------------------------------------
   is5_OpenUserSource / _is5_OpenUserSource

   Purpose : 
   Create a user-defined source object. Call this with
   pointers to caller-defined functions. 

   Param                Use
   ------------------------------------
   pOpenFn              source open function
                        called during is5_OpenUserSource
                           BOOL bOk = Open(uUserData);

   pReadFn              file read   
                           UINT32 uReadLen = Read(pBuffer, uNumBytes, uUserData);

   pSeekFn              file seek. not all formats use this (PCX, TIFF and TGA do)
                           BOOL bOK = Seek(uOffset, uOrigin, uUserData);

                        uOrigin is one of :
                           0: Set absolute position
                           1: Offset from current position
                           2: From EOF. Offset is negative
                        

   pPosFn               get file position
                           BOOL ok = GetPos(uUserData, &uPos);

   pSizeFn              get file size
                           BOOL ok = GetSize(uUserData, &uSize);

   pEOFFn               is end of file
                           BOOL bEOF = EOF(uUserData);

   pCloseFn             source close
                           BOOL ok = Close(uUserData);
               
   uUserInfo            user-defined value. This will be
                        passed to all of the functions in this
                        ImgSource source object. The value can
                        be used to identify which source object is
                        being used, if you have multiple source
                        objects open.

   Return
   ------
   NULL on failure, else a valid ImgSource source object.
   You must call is5_CloseSource to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISSRC     	_ISCConv _ISFn( is5_OpenUserSource )(const is5_SourceOpenFn pOpenFn,
                                  const is5_SourceReadFn pReadFn,
                                  const is5_SourceSeekFn pSeekFn,
                                  const is5_SourceGetPosFn pPosFn,
                                  const is5_SourceGetSizeFn pSizeFn,
                                  const is5_SourceEOFFn pEOFFn,
                                  const is5_SourceCloseFn pCloseFn,
                                  const IS_CALLBACK_DATA uUserInfo);

/*---------------------------------------------------------
   is5_CloseSource / _is5_CloseSource

   Purpose : 
   Close the source object.

   Param                Use
   ------------------------------------
   hSource              valid source manager

   Return
   ------
   Final position reached by the source data pointer. 
   Usually equal to the number of BYTEs read.
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_CloseSource )(HISSRC hSource);

/*---------------------------------------------------------
   is5_OpenFileDest / _is5_OpenFileDest

   Purpose : 
   Create a file destination object.

   See is5_OpenFileDestW, for wide-character filenames.

   Param             Use
   ------------------------------------
   pFileName         output file path

   bAppend           allow the destination object to be used in
                     operations that require appending to an existing
                     data stream.
   Return
   ------
   NULL on failure, else a valid ImgSource destination object.
   You must call is5_CloseDest to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISDEST     	_ISCConv _ISFn( is5_OpenFileDest )(const char *pFileName, BOOL bAppend);

/*---------------------------------------------------------
   is5_OpenFileDestW / _is5_OpenFileDestW

   Purpose : 
   Create a file destination object, using a wide character filename.

   Param             Use
   ------------------------------------
   pFileNameW        output file path

   bAppend           allow the destination object to be used in
                     operations that require appending to an existing
                     data stream.
   Return
   ------
   NULL on failure, else a valid ImgSource destination object.
   You must call is5_CloseDest to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISDEST     	_ISCConv _ISFn( is5_OpenFileDestW )(const wchar_t *pFileName, BOOL bAppend);

/*---------------------------------------------------------
   is5_OpenHandleDest / _is5_OpenHandleDest

   Purpose : 
   Create a file destination object using a Windows HANDLE.

   The HANDLE must be opened, positioned at the start of the
   file/stream and ready for writing.

   If you are using a HANDLE to a non-seeking device, such as a pipe 
   or a communications channel, some ImgSource file format reading 
   and writing operations may fail. Use ::GetFileType to determine 
   the HANDLE type, if unsure.

   Calling is5_CloseDest will not close the handle, that is left
   to the caller. But, you still need to call is5_CloseDest to
   release the ImgSource destination object.

   Param             Use
   ------------------------------------
   handle            opened Windows HANDLE

   bAppend           allow the destination object to be used in
                     operations that require appending to an existing
                     data stream.
   Return
   ------
   NULL on failure, else a valid ImgSource destination object.

---------------------------------------------------------*/
_ISDeclSpec HISDEST     	_ISCConv _ISFn( is5_OpenHandleDest )(HANDLE handle, BOOL bAppend);

/*---------------------------------------------------------
   is5_OpenMemoryDest / _is5_OpenMemoryDest

   Purpose : 
   Create a memory destination object.

   Param             Use
   ------------------------------------
   pOutBuf           output buffer. allocated by called
   uBufLen           length of output buffer

   bAppend           allow the destination object to be used in
                     operations that require appending to an existing
                     data stream.

   Return
   ------
   NULL on failure, else a valid ImgSource destination object.
   You must call is5_CloseDest to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISDEST     	_ISCConv _ISFn( is5_OpenMemoryDest )(BYTE *pOutBuf, const UINT32 uBufLen, BOOL bAppend);

/*---------------------------------------------------------
   is5_OpenDynamicMemoryDest / _is5_OpenDynamicMemoryDest

   Purpose : 
   Create a dynamic memory destination object. This object will
	increase its internal buffer size as you write to it.

	You can use this as an easy way to create in-memory image files:

		// 1. create the dest object
		HISDEST hDynMem = is5_OpenDynamicMemoryDest(0);

		// 2. use it in a file write
		is5_WriteJPG(hDynMem, ...);

		// 3. get the number of bytes written
		UINT32 size;
		is5_GetSize(hDynMem, &size);

		// 4. rewind the file to the beginning
		is5_Seek(hDynMem, 0, 0);

		// 5. read the contents to a memory buffer
		BYTE *pOutput = new BYTE[size];
		is5_Read(hDynMem, pOutput, size);

		// 6. done
		is5_CloseDest(hDynMem);

   Param             Use
   ------------------------------------
   bAppend           allow the destination object to be used in
                     operations that require appending to an existing
                     data stream.

   Return
   ------
   NULL on failure, else a valid ImgSource destination object.
   You must call is5_CloseDest to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISDEST     	_ISCConv _ISFn( is5_OpenDynamicMemoryDest )(BOOL bAppend);

/*---------------------------------------------------------
   is5_OpenUserDest / _is5_OpenUserDest

   Purpose : 
   Create a user-defined destination object. Call this with
   pointers to caller-defined functions. 

   Param             Use
   ------------------------------------
   pOpenFn           source open function
                     called during is5_OpenUserDest!!
                        BOOL bOk = Open(uUserData);

   pWriteFn          file write
                        BOOL bOK = Write(pBuffer, uNumBytes, uUserData);

   pReadFn           file read. this is optional unless you are using
                        this dest object in a situation where both reading 
                        and writing are required (appending to a multi-page 
                        TIFF file, for example).

                        UINT32 uReadLen = Read(pBuffer, uNumBytes, uUserData);

   pSeekFn           file seek. not all formats use this (TIFF does)
                        BOOL bOK = Seek(uOffset, uOrigin, uUserData);
                        uOrigin is one of :
                           0: Set absolute position
                           1: Offset from current position
                           2: From EOF. Offset is negative

   pPosFn            get file position
                        BOOL ok = GetPos(uUserData, &uPos);

   pSizeFn           get file size
                        BOOL ok = GetSize(uUserData, &uSize);

   pEOFFn            is end of file
                        BOOL bEOF = EOF(uUserData);

   pCloseFn          source close
                        BOOL ok = Close(uUserData);
         
   bAppend           allow the destination object to be used in
                     operations that require appending to an existing
                     data stream.

   uUserInfo         user-defined value. This will be
                     passed to all of the functions in this
                     ImgSource destination object. The value can
                     be used to identify which destination object is
                     being used, if you have multiple destination
                     objects open.

   Return
   ------
   NULL on failure, else a valid ImgSource destination object.
   You must call is5_CloseDest to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISDEST     	_ISCConv _ISFn( is5_OpenUserDest )(const is5_SourceOpenFn pOpenFn,
                                  const is5_SourceWriteFn pWriteFn,
                                  const is5_SourceReadFn pReadFn,
                                  const is5_SourceSeekFn pSeekFn,
                                  const is5_SourceGetPosFn pPosFn,
                                  const is5_SourceGetSizeFn pSizeFn,
                                  const is5_SourceEOFFn pEOFFn,
                                  const is5_SourceCloseFn pCloseFn,
                                  BOOL bAppend,
                                  const IS_CALLBACK_DATA uUserInfo);

/*---------------------------------------------------------
   is5_OpenNullDest / _is5_OpenNullDest

   Purpose : 
   Create a Null destination object.

   A Null destination object discards all input but maintains 
   a file pointer, so you can see how many bytes would have 
   been written, without actually writing the data anywhere.

   This can be used to do file size calculations with most 
   formats, as it's much faster than actually writing the 
   file data.

   It can be used with any format, except for TIFF. 

   Example:

      HISDEST hDest = is5_OpenNullDest(0);
      is5_WriteJPG(hDest, pImage, w, h, ...);
      UINT32 uFileSize = is5_CloseDest(hDest);

      Now, without actually having created a file, you'll
      know how large that file would have been.

   Param             Use
   ------------------------------------

   uFlags            bit      purpose
                     ---      -------
                      0       if set, this will allow reads
                              to succeed - though all data 
                              returned will be zero. otherwise,
                              reads will simply fail.

   Return
   ------
   NULL on failure, else a valid ImgSource destination object.
   You must call is5_CloseDest to delete this object!
---------------------------------------------------------*/
_ISDeclSpec HISDEST     	_ISCConv _ISFn( is5_OpenNullDest )(UINT32 uFlags);

/*---------------------------------------------------------
   is5_CloseDest / _is5_CloseDest

   Purpose : 
   Close the destination object. 

   Param               Use
   ------------------------------------
   hDest               valid dest manager

   Return
   ------
   Highest position reached by the destination data pointer.
   Usually equal to the number of BYTEs written.
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_CloseDest )(HISDEST hDest);

/*---------------------------------------------------------
   is5_Read / _is5_Read

   Purpose : 
   Read from the source manager object.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

   Param             Use
   ------------------------------------
   hSource           valid source manager
   pData             buffer to hold the data
   uDataLen          length of data to read

   Return
   ------
   The number of bytes read.
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_Read )(HANDLE hSource, BYTE * pData, const UINT32 uDataLen);

/*---------------------------------------------------------
   is5_Write / _is5_Write

   Purpose : 
   Write from a destination manager object.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

   Param             Use
   ------------------------------------
   hDest             valid destination manager
   pData             data to write
   uDataLen          length of data to write

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_Write )(HANDLE hSource, const BYTE * pData, const UINT32 uDataLen);

/*---------------------------------------------------------
   is5_Seek / _is5_Seek

   Purpose : 
   Set the file pointer.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

   This will fail, if you are using a HANDLE manager and 
   the HANDLE is to a non-seeking device, such as a pipe or
   a communications channel. Use ::GetFileType to determine 
   the HANDLE type, if unsure.

   Param               Use
   ------------------------------------
   hMgr              valid source or destination manager
   uOffset           offset bytes

   uOrigin           starting place for repositioning :
                     0: Set absolute position (uOffset >= 0)
                     1: Offset from current position 
                     2: From EOF. Offset is negative (uOffset <= 0)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_Seek )(HANDLE hMgr, __int32 uOffset, UINT32 uOrigin);

/*---------------------------------------------------------
   is5_GetPos / _is5_GetPos

   Purpose : 
   Return the file pointer.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

   This will fail, if you are using a HANDLE manager and 
   the HANDLE is to a non-seeking device, such as a pipe or
   a communications channel. Use ::GetFileType to determine 
   the HANDLE type, if unsure.

   Param             Use
   ------------------------------------
   hMgr              valid source or destination manager
   puPos             receives file position

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    	_ISCConv _ISFn( is5_GetPos )(HANDLE hMgr, UINT32 *puPos);

/*---------------------------------------------------------
   is5_GetSize / _is5_GetSize

   Purpose : 
   Return the size of the current file. 
   
   If this is used on a memory destination object, it 
   returns the highest position reached by a write 
   operation.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

   This will fail, if you are using a HANDLE manager and 
   the HANDLE is to a non-seeking device, such as a pipe or
   a communications channel. Use ::GetFileType to determine 
   the HANDLE type, if unsure.

   Param             Use
   ------------------------------------
   hMgr              valid source or destination manager
   puSize            The size of the current file.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    	_ISCConv _ISFn( is5_GetSize )(HANDLE hMgr, UINT32 *puSize);

/*---------------------------------------------------------
   is5_EOF / _is5_EOF

   Purpose : 
   Test end of file condition

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

   Param             Use
   ------------------------------------
   hMgr              valid source or destination manager

   Return
   ------
   TRUE if EOF
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_EOF )(HANDLE hMgr);

/*---------------------------------------------------------
   is5_IsValidManager / _is5_IsValidManager

   Purpose : 
   Test that the source / dest manager object is valid.

   Param             Use
   ------------------------------------
   hMgr              source or destination manager

   Return
   ------
   TRUE if valid
---------------------------------------------------------*/
_ISDeclSpec BOOL      	_ISCConv _ISFn( is5_IsValidManager )(HANDLE hMgr);

/*---------------------------------------------------------
   is5_IsValidManagerNoErr / _is5_IsValidManagerNoErr

   Purpose : 
   Test that the source / dest manager object is valid,
   but do no set the global error value.

   Param             Use
   ------------------------------------
   hMgr              source or destination manager

   Return
   ------
   TRUE if valid
---------------------------------------------------------*/
_ISDeclSpec BOOL      	_ISCConv _ISFn( is5_IsValidManagerNoErr )(HANDLE hMgr);


#ifdef __cplusplus
}
#endif

#endif