/*---------------------------------------------------------
   ISource.h : part of the ImgSource library.
   
   Main ImgSource header file

   This is the only file an application needs to #include

   C/C++ users, see note below about Exported Symbols.

   For ImgSource build configuration options, see ISConfig.h and ISTypes.h.

   Credits:
      Color quantization code modified from DL1Quant, by 
      Dennis Lee (denlee@ecf.utoronto.ca)

      JPEG code from IJG v6b.
      PNG code from LibPng
      TIFF code from LibTiff
      Compression from ZLib
      TGA code from Timothy Bish
      PCX code from Bob Johnson's PCXLib
      Some crypto from Wei Dai's CryptLib 3.1
      PAX code from Smaller Animals Software's PALib
      TLA code from Smaller Animals Software's TLALib
      RGBA Color quantization code modified from PNGQuant, by Greg Roelofs.

      All other code, Smaller Animals Software. 

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is6_SOURCEH
#define is6_SOURCEH

///////////////////////////////////////////////////////
// Exported Symbols
//
// C/C++ users of the DLL must #define IS60_USING_DLL (see below).
//
// Also, when using the DLL, all function names will start with "_" : 
//    Example: _is6_Initialize
//
// Users of the static libs should #undef IS60_USING_DLL, or simply comment-out the line below.
// When using the static libraries, function names will NOT start with "_"
//    Example: is6_Initialize
//
// If you don't define IS60_USING_DLL correctly, you will get undefined symbol errors
// when you try to link.
//
///////////////////////////////////////////////////////

//#define IS60_USING_DLL

#ifdef __cplusplus
extern "C" {
#endif

///////////////////////////////////////////////////////
// source / destination manager function defintions
#include "ISTypes.h"

///////////////////////////////////////////////////////
// source / destination manager function defintions
#include "ISErrorCodes.h"

///////////////////////////////////////////////////////
// source / destination manager function defintions
#include "ISIOManagers.h"

////////////////////////////////////////////////////////
// file write function defintions
#include "ISFilesW.h"

////////////////////////////////////////////////////////
// single line file write function defintions
#include "ISFilesWL.h"

////////////////////////////////////////////////////////
// file read function defintions
#include "ISFilesR.h"

////////////////////////////////////////////////////////
// extended TIFF function defintions
#include "ISFilesTIFFExt.h"

////////////////////////////////////////////////////////
// single line file read function defintions
#include "ISFilesRL.h"

////////////////////////////////////////////////////////
// file dimension / bit depth function defintions
#include "ISFileDims.h"

////////////////////////////////////////////////////////
// file utilities (DPI, text, etc) function defintions
#include "ISFileUtil.h"

////////////////////////////////////////////////////////
// file utilities (DPI, text, etc) function defintions for JPEG
#include "ISFileUtilJPG.h"

////////////////////////////////////////////////////////
// image drawing function defintions
#include "ISImageDraw.h"

////////////////////////////////////////////////////////
// image processing function defintions
#include "ISImageProcess.h"

////////////////////////////////////////////////////////
// image histogram processing function defintions
#include "ISImageEnhance.h"

////////////////////////////////////////////////////////
// image resizing function defintions
#include "ISImageResize.h"

////////////////////////////////////////////////////////
// image rotation function defintions
#include "ISImageRotate.h"

////////////////////////////////////////////////////////
// image overlay function defintions
#include "ISImageOverlay.h"

////////////////////////////////////////////////////////
// image text draw function defintions
#include "ISImageText.h"

////////////////////////////////////////////////////////
// image effects function defintions
#include "ISImageEffects.h"
    
////////////////////////////////////////////////////////
// image morphology function defintions
#include "ISImageMorpho.h"

////////////////////////////////////////////////////////
// noise generation 
#include "ISImageNoise.h"

////////////////////////////////////////////////////////
// image filter function defintions
#include "ISImageFilter.h"

////////////////////////////////////////////////////////
// image transformation function defintions
#include "ISImageTransformEffects.h"

////////////////////////////////////////////////////////
// EXIF data parser
#include "ISEXIF.h"

////////////////////////////////////////////////////////
// IPTC data parser
#include "ISIPTC.h"

////////////////////////////////////////////////////////
// PSD resource manager
#include "ISPSDResources.h"

////////////////////////////////////////////////////////
// Fast Fourier Transform functions
#include "ISImageFFT.h"
                                        
////////////////////////////////////////////////////////
// Generic zLib compression
#include "ISZLibComp.h"

////////////////////////////////////////////////////////
// general library function defintions

/*---------------------------------------------------------
   is6_GetImgSourceVersion / _is6_GetImgSourceVersion

   Purpose : 
   Get ImgSource version string

   Param             Use
   ------------------------------------
   pVers             receives version string   
                     must be at least 20 chars.

---------------------------------------------------------*/
_ISDeclSpec void      _ISCConv _ISFn( is6_GetImgSourceVersion )(char *buf);

/*---------------------------------------------------------
   is6_Initialize / _is6_Initialize

   Purpose : 
   Initialize the library.
   
   Among other things, this initializes the ImgSource thread pool. 
   To initialize the library without using the thread pool, use
   is6_InitializeEx.

   Note:
   You must call is6_Exit to free any resources allocated
   by this library.

   Note:
   Do not call this function from DllMain - doing so will 
   likely cause your application to crash.

   Param             Use
   ------------------------------------
   pKey              library unlock key, 
                     or any other string, if you don't have a key.
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_Initialize )(const char *pKey);

/*---------------------------------------------------------
   is6_InitializeEx / _is6_InitializeEx

   Purpose : 
   Initialize the library.
   
   Among other things, this initializes the ImgSource thread pool. 
   If the thread pool is never initialized, multi-threaded 
   processing will be disabled. 
   
   See also: is6_Initialize

   Note:
   You must call is6_Exit to free any resources allocated
   by this library.

   Note:
   If you are using the thread pool, do not call this 
   function from DllMain - doing so will likely cause 
   your application to crash.

   Param             Use
   ------------------------------------
   pKey              library unlock key, 
                     or any other string, if you don't have a key.

   uFlags            bit      purpose
                     ---      -------
                      0       if set, the ImgSoruce thread pool will not
                              be started. all functions will run in their
                              single-threaded version.

---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_InitializeEx )(const char *pKey, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Exit / _is6_Exit

   Purpose : 
   Free any resources allocated by this library. This is
   a process-level function. For thread resource release,
   see is6_ThreadExit.
   
   Do not call this unless you've called is6_Initialize first!

   Note:
   This function MUST be called when your application exits.
   If you do not, you will certainly leak memory and resources,
   and you may experience an application crash.

   Note:
   Do not call this function from DllMain. Doing so will 
   cause your application to crash on exit.

   Note:
   If this call seems to be taking a long time to return,
   odds are good that you are calling it from DllMain, or 
   from a function which is called by DllMain. You must call 
   this before DllMain is called.

---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_Exit )();

/*---------------------------------------------------------
   is6_GetLastError / _is6_GetLastError

   Purpose : 
   Get error code set by last function.

   All functions in ImgSource set the global error value.
   It is recommended that you test the error value often.

   Return
   ------
   Error value
---------------------------------------------------------*/
_ISDeclSpec UINT32        _ISCConv _ISFn( is6_GetLastError )();

/*---------------------------------------------------------
   is6_SetLastError / _is6_SetLastError

   Purpose : 
   Set the global ImgSource error code.

   Param                Use
   ------------------------------------
   error                error value to set
   pCallerFunctionName  name of calling function
---------------------------------------------------------*/
_ISDeclSpec void          _ISCConv _ISFn( is6_SetLastError )(int error, const char *pCallerFunctionName);

/*---------------------------------------------------------
   is6_GetErrorList / _is6_GetErrorList

   Purpose : 
   ImgSource maintains a list of the last 50 errors it encountered.
   This function retrieves that list as a single string with
   errors separated by new lines.

    The string is of this form:

        Error 2 in is6_OpenFileSource (12:32:02)
        Error 5 in is6_DrawRGB (12:32:02)

    Also, there may be an optional bit of text on the end of the string.
    Ex.:
        Error #5 in is6_ReadJPG (04:22:04) [unsupported bit depth]

    The information in the parenthesis is the current system
    time in HH:MM:SS format.

    Param               Use
    ------------------------------------
    hOutData            receives HGLOBAL for the error string
    puLen               receives number of bytes allocated to the error string

    uFlags              unused
---------------------------------------------------------*/
_ISDeclSpec void          _ISCConv _ISFn( is6_GetErrorList )(HGLOBAL *hOutData, UINT32 *puLen, UINT32 uFlags);

/*---------------------------------------------------------
   is6_SetCallback / _is6_SetCallback

   Purpose : 
   Set global callback function. This function is called
   by all file read / write operations, and most image
   processing calls. It is not called by image display
   functions.

   The callback function must be a function of this type :
   
      BOOL CALLBACK MyCallbackFunction(const UINT32 uCurRow, 
                              const UINT32 uTotalRows,
                              const IS_CALLBACK_DATA uUserData)
      {
         if (time to quit)
         {
            return FALSE;
         }

         return TRUE;
      }

   Set it with:

      is6_SetCallback(MyCallbackFunction, uUserData);

   The library will also call the callback function once 
   per row for most image processing calls. For some complex
   operations, it may be called a fixed number of times, 
   regardless of image size.
   
   Operations which do not operate on image buffers are not 
   "image processing calls" and so, these won't use the callback 
   function. This includes calls such as JPG and PNG text calls, 
   PNG gamma and background operations, DPI queries, error 
   queries, etc.. 
   
   Basically, if there is no loop over pixels, there is no 
   callback call. However, this does not mean that *all* 
   operations which do have a pixel loop use the callback 
   function; some don't. Check the function comments in this 
   file, to be sure.

   If the callback function returns FALSE the operation aborts and 
   is6_GetLastError will return IS_ERR_CALLBACKCANCEL

   You can use this to monitor the progress of ImgSource operations.

   It is not neccessary for you to use this function. If you 
   do not set the callback function or pass a NULL to this call, 
   no callback will be performed.

   The callback is thread safe. Each thread can have its own 
   callback function.

   Parameter            Use
   ------------------------------------
   pCallback            pointer to callback function
   uUserData            user data. this data is thread safe.

   Return
   ------
   Error value
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_SetCallback )(ISCALLBACKPTR pCallback, IS_CALLBACK_DATA uUserData);

/*---------------------------------------------------------
   is6_GetCallback / _is6_GetCallback

   Purpose : 
   Retrieve the callback state.

   Parameter            Use
   ------------------------------------
   puUserData           receives user data. 

   Return
   ------
   pointer to callback function
---------------------------------------------------------*/
_ISDeclSpec ISCALLBACKPTR         _ISCConv _ISFn( is6_GetCallback )(IS_CALLBACK_DATA *puUserData);

/*---------------------------------------------------------
   is6_SetErrorCallback / _is6_SetErrorCallback

   Purpose : 
   Set a callback function to be called when an ImgSource function 
   encounters an error during processing. This can be useful
   for diagnostics or application logging.

   The callback must be a function of this type:

       BOOL myErrorCallback(UINT32 uErrorCode, 
                            const char *pErrorText, 
                            const IS_CALLBACK_DATA uUserData)
       {
            // uErrorCode is the error value. it's the 
            // same value you'll get when you call is6_GetLastError.

            // pErrorText is typically the name of the function which 
            // is setting the error

            // uUserData is the value you passed to is6_SetErrorCallback

            printf("function %s set an error value of %d\n", pErrorText, uErrorCode);

            // you should return TRUE, always.
            return TRUE;
       }

   Parameter            Use
   ------------------------------------
   pCallback            callback function pointer

   userData             user-defined data which will 
                        be passed to your callback function.

   Return
   ------
   none
---------------------------------------------------------*/
_ISDeclSpec void  _ISCConv  _ISFn( is6_SetErrorCallback )(ISERRORCALLBACK pCallback, IS_CALLBACK_DATA userData);

/*---------------------------------------------------------
   is6_ThreadExit / _is6_ThreadExit

   Purpose : 
   Free any ImgSource resources associated with this
   thread.
   
   These resources include JPG and PNG metadata, error
   state info and thread-local storage areas (TLS).
   
   This function is useful in applications that are heavily 
   multithreaded. If you do not call this when your thread
   exits, any resources ImgSource has associated it may 
   remain in memory until your process exits.

   So, it it not a fatal mistake to not call this when a thread
   exits. Not calling this simply means any resources allocated 
   for the thread will remain allocated until the parent process exits.

   It is safe to call this even if the thread has not yet called
   any other ImgSource functions.

   This should be the last ImgSource call that your thread makes.

   This function is not available in the DLL. The DLL
   will handle thread cleanup by itself.
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_ThreadExit )();

/*---------------------------------------------------------
   is6_GetExtHook / _is6_GetExtHook

   Purpose : 
   Get a hook object to initialize an add-on library. 
   
   The hook object allows the add-on library to access ImgSource's
   I/O functions (to allow files to be read and written), 
   provides access to the progress callbacks (is6_SetCallback),
   provides access to the file reading memory allocation callbacks
   (is6_SetFileReadAllocCallbacks) and provides access to the
   ImgSource global error value. Add-on libraries will use 
   these services in their own operation.

   Usage:

   HISEXTHK hHook = is6_GetExtHook(0);
   AddOnLibrary_SetImgSourceHook(hHook);
   GlobalFree(hHook);

   Param             Use
   ------------------------------------
   uFlags            unused

   Return
   ------
   A HISEXTHK object to be passed to an add-on library.
   After passing this to the add-on library, you must free this
   memory by using GlobalFree.

---------------------------------------------------------*/
_ISDeclSpec HISEXTHK      _ISCConv _ISFn( is6_GetExtHook )(UINT32 uFlags);

/*---------------------------------------------------------
   is6_SetDebugMessageLevel / _is6_SetDebugMessageLevel

   Purpose : 
   Control the level of debug messages shown by the library.
   These messages are written to the debug console (they are
   not popup windows or logfile messages).

   This only affects debug builds, since no debug messages
   are shown in release builds.

   Param             Use
   ------------------------------------
   uLevel            0: show all messages - verbose
                     1: show informational, warning and critical messages
                     2: show warning and critical messages (default)
                     3: show critical messages only
   
---------------------------------------------------------*/
_ISDeclSpec void      _ISCConv _ISFn( is6_SetDebugMessageLevel )(UINT32 uLevel);

/*---------------------------------------------------------
   is6_SwapLong / _is6_SwapLong

   Purpose : 
   Convert a 32-bit integer from Intel byte order to Motorola
   byte order, and vice versa.
   
   Param             Use
   ------------------------------------
   pLong             32-bit integer to swap

   Return
   ------
   none
---------------------------------------------------------*/
_ISDeclSpec void      _ISCConv _ISFn( is6_SwapLong )(UINT32 *pLong);

/*---------------------------------------------------------
   is6_SwapShort / _is6_SwapShort

   Purpose : 
   Convert a 16-bit integer from Intel byte order to Motorola
   byte order, and vice versa.
   
   Param             Use
   ------------------------------------
   pShort            16-bit integer to swap

   Return
   ------
   none
---------------------------------------------------------*/
_ISDeclSpec void      _ISCConv _ISFn( is6_SwapShort )(ISUINT16 *pShort);

/*---------------------------------------------------------
   is6_GetSIMDSupport / _is6_GetSIMDSupport

   Purpose : 
   Test to see if the CPU supports the specific SIMD technology.

   Note:
   ImgSource currently uses MMX, SSE* and AVX support.

   See is6_EnableSIMDFunctions.

   Param             Use
   ------------------------------------
   uLevel            one of:
                        MMX : 0
                        SSE : 1
                        SSE2 : 2
                        SSE3 : 3
                        SSSE3 : 4
                        SSE4.1 : 5
                        SSE4.2 : 6
                        AES : 7
                        AVX : 8
                        AMD MMXplus : 9
                        AMD 3Dnow : 10
                        AMD 3Dnow Ext : 11
                        AMD SSE4a : 12
                        AMD SSE5 : 13

   Return
   ------
   TRUE if the specified technology is supported
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_GetSIMDSupport )(UINT32 uLevel);

/*---------------------------------------------------------
   is6_DisableSIMDSupport / _is6_DisableSIMDSupport

   Purpose : 
   Disable a specific SIMD technology.

   Note:
   ImgSource currently uses MMX, SSE* and AVX support.

   See is6_EnableSIMDFunctions.

   Param             Use
   ------------------------------------
   uLevel            one of:
                        MMX : 0
                        SSE : 1
                        SSE2 : 2
                        SSE3 : 3
                        SSSE3 : 4
                        SSE4.1 : 5
                        SSE4.2 : 6
                        AES : 7
                        AVX : 8
                        AMD MMXplus : 9
                        AMD 3Dnow : 10
                        AMD 3Dnow Ext : 11
                        AMD SSE4a : 12
                        AMD SSE5 : 13

---------------------------------------------------------*/
_ISDeclSpec void      _ISCConv _ISFn( is6_DisableSIMDSupport )(UINT32 uLevel);

/*---------------------------------------------------------
   is6_EnableSIMDFunctions / _is6_EnableSIMDFunctions

   Purpose : 
   By default, ImgSource will use SIMD (MMX,SSE,AVX) technology
   to optimize certain functions. This function can enable or disable 
   those optimizations.
   
   Param             Use
   ------------------------------------
   bUseSIMD           TRUE (1) to enable all SIMD optimizations, 0 to disable

   Return
   ------
   none
---------------------------------------------------------*/
_ISDeclSpec void      _ISCConv _ISFn( is6_EnableSIMDFunctions )(BOOL bUseSIMD);


/*---------------------------------------------------------
   is6_SetWorkerThreadCount / _is6_SetWorkerThreadCount

   Purpose : 
   Sets the number of worker threads for functions with 
   multithreaded processing options. 
   
   The minimum is 1. The maximum is 255. 

   The default worker thread count is 4. 
   
   Setting this to 1 disables all multi-threading.

   These threads are pulled from threads in the global ImgSource 
   thread pool. If you set a number that is greater than the
   number of threads in the thread pool, tasks will be queued
   and then handled as pool threads become available.
   
   See is6_SetWorkerThreadCallbackFrequency, is6_SetWorkerThreadTimeout,
   is6_SetWorkerThreadPoolSize, is6_Initialize.

   Param             Use
   ------------------------------------
   n                 number of worker threads

---------------------------------------------------------*/
_ISDeclSpec void  _ISCConv  _ISFn(is6_SetWorkerThreadCount)(UINT32 n);

/*---------------------------------------------------------
   is6_SetWorkerThreadCallbackFrequency / _is6_SetWorkerThreadCallbackFrequency

   Purpose : 
   Functions with multithreaded options do not call the 
   global progress callback (is6_SetCallback) once for every
   line processed. Rather they are, in effect, called on a timer. 
   This function controls how frequently the callbacks happen.

   Note:
   The ImgSource multi-threaded callback model is:

        1. Threads are created in suspended mode and thread parameters are set.
        2. All threads are launched.
        3. The thread controller Waits on all threads for the time given here.
        4. If all threads finish before the Wait is over, the function exits with
        no callbacks.
        5. If the Wait expires but one or more threads are still working,
        the callback is called, and a new Wait begins.

   For fast operations, it's entirely likely that a mutli-threaded function 
    will finish before any callbacks are called. 

   The default is 200ms. Longer times reduce the number of times the
    callback is called. Shorter times increase the nubmer of callbacks.

   See is6_SetWorkerThreadCount, is6_SetWorkerThreadTimeout,
   is6_SetWorkerThreadPoolSize.

   Param             Use
   ------------------------------------
   ms                number of milliseconds to Wait between callbacks

   Return
   ------
   pDest
---------------------------------------------------------*/
_ISDeclSpec void  _ISCConv  _ISFn(is6_SetWorkerThreadCallbackFrequency)(UINT32 ms);

/*---------------------------------------------------------
   is6_SetWorkerThreadTimeout / _is6_SetWorkerThreadTimeout

   Purpose : 
   This sets the thread timeout length for functions with
   multithreaded options. If a worker thread fails to finish
    when the timeout happens, the thread will be terminated.
   
   Note:
   This is only used when there is no global progress 
   callback set (is6_SetCallback).
      
   The default is infinity - never timeout.
   
   See is6_SetWorkerThreadCount, is6_SetWorkerThreadCallbackFrequency,
   is6_SetWorkerThreadPoolSize.

   Param             Use
   ------------------------------------
   ms                number of milliseconds for the timeout

   Return
   ------
   pDest
---------------------------------------------------------*/
_ISDeclSpec void  _ISCConv  _ISFn(is6_SetWorkerThreadTimeout)(UINT32 ms);

/*---------------------------------------------------------
   is6_SetWorkerThreadPoolSize / _is6_SetWorkerThreadPoolSize

   Purpose : 
   This sets the size of the ImgSource global thread pool. 
   
   This function gives all pool threads a 2 second timeout period 
   to let any pending jobs finish. After that, all existing 
   pool threads are terminated and the specified number of new threads
   is created. Be careful to call this only when there are no
   pending ImgSource calls.
   
   The default worker thread pool size is eight, and these threads
   are started when the library is initialized (when the ImgSource
   DLL is loaded, or when a statically-linked application is launched).
   They are terminated when the DLL unloads (or when the 
   statically-linked application terminates).

   The minimum thread pool size is 2. The maximum is 255.

   These threads will remain idle until one of the ImgSource
   multithreaded functions is called. Then, up to is6_SetWorkerThreadCount
   of these threads will be activated. 
   
   If the value set with is6_SetWorkerThreadCount is greater than the 
   pool size, or if there are more jobs waiting than the number of 
   available threads, waiting jobs will be queued.   
   
   The thread pool is global to an instance of ImgSource. All application 
   threads which call into ImgSource share the thread pool.

   See is6_SetWorkerThreadCallbackFrequency, is6_SetWorkerThreadTimeout,
   is6_SetWorkerThreadCount, is6_Initialize.

   Param             Use
   ------------------------------------
   n                 number of worker threads in the pool

---------------------------------------------------------*/
_ISDeclSpec void  _ISCConv  _ISFn(is6_SetWorkerThreadPoolSize)(UINT32 n);


#ifdef __cplusplus
}
#endif


#endif