/*---------------------------------------------------------
   ISEXIF.h : part of the ImgSource library.
 
   EXIF data reading code

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..
   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   
   http://www.smalleranimals.com
---------------------------------------------------------*/
#ifndef is5_EXIFREADH
#define is5_EXIFREADH

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_EXIFCreateReader / _is5_EXIFCreateReader

   Purpose : 
   Create an ImgSource EXIF data reading object. This
   call creates the EXIF reader object and parses the 
   EXIF data stream. To fetch the individual strings from 
   the data, use is5_EXIFGetString.

   Use is5_EXIFDestroyReader to release the memory associated with
   this object.

   EXIF data is image metadata stored in JPG files
   generated by many digital cameras and scanners. It can
   contain information about the camera, software, image,
   environmental conditions, etc.. 

   Note:
   The UserComment field (0x9286) can contain data encoded in one of any 
   number of character sets. An 8-byte header at the start of the text 
   describes the character set used. 
   
   The default behavior for ImgSource is to only return the text of the 
   UserComment tag (stripping away the header), and then only if the 
   data is in ASCII format - all other comments are discarded.

   If you wish to retrieve the full comment tag, including the header, you 
   can set bit 2 of the uFlags parameter. This will cause ImgSource to 
   return all of the tag contents (including the 8 byte header which 
   specifies the character set). To retrieve this raw data, you 
   should use is5_EXIFGetUserComment.

   Note:
   EXIF data may contain an optional MakerNote section. This is
   data put there by the manufacturer to hold information that
   isn't supported by the EXIF specifications. There is no 
   standardized data format for the MakerNote section - every 
   camera manufacturer is free to include any data, in any format.
   Some manufacturers stick with one format for all of their cameras;
   some use different formats for each; nearly all of them are 
   undocumented.
   
   ImgSource supports MakerNotes from Canon, Casio, Fuji, Nikon and 
   Olympus cameras. But, as some of these manufacturers have multiple 
   MakerNote data formats, and can create new formats at any time, 
   it is possible that some camera models may create MakerNote 
   data that ImgSource can't parse. This is *not* a bug in ImgSource, 
   it's a consequence of the fact that there is no standard for EXIF 
   MakerNote data.

   See http://www.ba.wakwak.com/%7etsuruzoh/Computer/Digicams/exif-e.html#APP2
   for a description of some of the MakerNote data fields supported by 
   ImgSource. Many of the fields are simply unknown, for some cameras, 
   *all* fields are unknown. Note that the tags are different for each
   camera manufacturer and possibly from camera to camera.


   Param             Use
   ------------------------------------
   hSource           open source manager object

   uFrameIdx         TIFF frame index. usually 0

   uFlags            bit         purpose
                     ---         -------
                      0          read JPG EXIF thumbnail. this
                                 can be retrieved with is5_EXIFGetThumbnail.
                                 TIFF thumbnails are usually just the second 
                                 frame in the image and can be read with ISReadTIFF

                      1          read and attempt to parse the MakerNote data. 

                      2          read and store the entire UserComment tag, including the
                                 character set header. see Note above.

                      3          hSource is reading an EXIF data stream, and not a JPG 
                                 of TIFF. if hSource is a memory source, this is similar 
                                 to is5_EXIFCreateReaderFromByteArray.
                                 

   Return
   ------
   HISEXIFREADER EXIF reader object. NULL on failure.
---------------------------------------------------------*/
_ISDeclSpec HISEXIFREADER     	_ISCConv _ISFn( is5_EXIFCreateReader )(HISSRC hSource, UINT32 uFrameIdx, UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFCreateReaderFromByteArray / _is5_EXIFCreateReaderFromByteArray

   Purpose : 
   Create an ImgSource EXIF data reading object from a 
   block of memory (ex. a JPEG_APP1 marker block).

   See is5_EXIFCreateReader for notes.

   Use is5_EXIFDestroyReader to release the memory associated with
   this object.


   Param             Use
   ------------------------------------
   pData             pointer to data
   uDataLength       length of data

   uFlags            bit         purpose
                     ---         -------
                      0          read JPG EXIF thumbnail. this
                                 can be retrieved with is5_EXIFGetThumbnail.
                                 TIFF thumbnails are usually just the second 
                                 frame in the image and can be read with ISReadTIFF

                      1          read and attempt to parse the MakerNote data. 

                      2          read and store the entire UserComment tag, including the
                                 character set header. see Note above.

   Return
   ------
   HISEXIFREADER EXIF reader object. NULL on failure.
---------------------------------------------------------*/
_ISDeclSpec HISEXIFREADER     	_ISCConv _ISFn( is5_EXIFCreateReaderFromByteArray )(BYTE * pData, UINT32 uDataLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFGetTagData / _is5_EXIFGetTagData

   Purpose : 
   Get the tag data for a single tag from an ImgSource
   EXIF data reader object.

   All data is returned as an array of values.

   For a list of tags and the values associated with them, see:
   http://www.exif.org/Exif2-2.PDF

   Note: 
   phData will point to an array of values of the data type 
   used by the tag:

	   is5_EXIFTYPE_BYTE	    data is an array of BYTEs 
	   is5_EXIFTYPE_ASCII	    data is an array of ASCII-encoded, zero-terminated string
	   is5_EXIFTYPE_SHORT	    data in an array of 16-bit unsigned integers 
	   is5_EXIFTYPE_LONG	    data in an array of 32-bit unsigned integers
	   is5_EXIFTYPE_SBYTE	    data in an array of BYTEs
	   is5_EXIFTYPE_UNDEFINED	 data in an array of BYTEs
	   is5_EXIFTYPE_SSHORT	    data in an array of 16-bit signed integers 
	   is5_EXIFTYPE_SLONG	    data in an array of 32-bit signed integers
	   is5_EXIFTYPE_FLOAT	    data in an array of floats (32-bit IEEE floating point)
	   is5_EXIFTYPE_DOUBLE	    data in an array of doubles (64-bit IEEE floating point)

	   is5_EXIFTYPE_RATIONAL	 data in an array of 64-bit unsigned fractions,
                               where each fraction is composed of two 32-bit values;
                               the first 32-bit integer is the numerator, and the second
                               is the denominator. so, a single tag value is two 32-bit values.

      is5_EXIFTYPE_SRATIONAL	 like is5_EXIFTYPE_RATIONAL, but the values may be signed

   Param             Use
   ------------------------------------
   hEXIFReader       EXIF data reader object created by
                     is5_EXIFCreateReader.

   uTag              Tag ID

   uFlags            the first five bits determine which
                     section of the EXIF data to search.

                     bit     section
                     ---------------
                     0       Main IFD
                     1       EXIF IFD
                     2       Interoperability IFD
                     3       IFD 1 (thumbnail)
                     4       GPS IFD (GPS data)
                     5       MakerNote (see is5_EXIFCreateReader for notes)

                     These values may be logically OR'd together.
                     Ex. use "63" to read IDs from all sections.
                     If a tag occurs in multiple sections (which 
                     shouldn't happen), the function returns the 
                     tag from the first section that contains the ID.

   puTagType         receives tag data type. this determines the format
                     of the data returned in phData

   phData            receives a handle to an array of data. see Notes.
                     you must use GlobalFree to release this
                     memory when you are finished with it.

   puLen             receives the number of bytes in the data array.
                     
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_EXIFGetTagData )(HISEXIFREADER hEXIFReader, UINT32 uTag, UINT32 uFlags, UINT32 *puTagType, HGLOBAL *phData, UINT32 *puLen);

/*---------------------------------------------------------
   is5_EXIFGetString / _is5_EXIFGetString

   Purpose : 
   Extract a data string for a single tag from an ImgSource
   EXIF data reader object.

   All data is returned as a text string. If the tag contains
   multiple values, all values will be returned in the same string,
   separated by commas.

   For a list of tags and the values associated with them, see:
   http://www.exif.org/Exif2-2.PDF

   Param             Use
   ------------------------------------
   hEXIFReader       EXIF data reader object created by
                     is5_EXIFCreateReader.

   uTag              Tag ID

   uFlags            the first five bits determine which
                     section of the EXIF data to search.

                     bit     section
                     ---------------
                     0       Main IFD
                     1       EXIF IFD
                     2       Interoperability IFD
                     3       IFD 1 (thumbnail)
                     4       GPS IFD (GPS data)
                     5       MakerNote (see is5_EXIFCreateReader for notes)

                     These values may be logically OR'd together.
                     Ex. use "63" to read IDs from all sections.
                     If a tag occurs in multiple sections (which 
                     shouldn't happen), the function returns the 
                     tag from the first section that contains the ID.

   phString          receives a handle to a zero-terminated text 
                     string. you must use GlobalFree to release this
                     memory when you are finished with it.

   puLen             receives the length of the string
                     
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_EXIFGetString )(HISEXIFREADER hEXIFReader, UINT32 uTag, UINT32 uFlags, HGLOBAL *phString, UINT32 *puLen);

/*---------------------------------------------------------
   is5_EXIFGetTagList / _is5_EXIFGetTagList

   Purpose : 
   Get a list of all data tags available from this EXIF
   parser. Tags are listed as "id, count, label, value" seperated 
   by a new line.


   For example:

      0x10f, 9, Make, FUJIFILM
      0x110, 16, Model, FinePix1400Zoom
      0x112, 1, Orientation, 1
      0x11a, 1, XResolution, 72/1
      0x11b, 1, YResolution, 72/1
      0x128, 1, ResolutionUnit, 2
      ...

   With bit #9 (include tag data type):

      0x10f, 2, 9, Make, FUJIFILM
      0x110, 2, 16, Model, FinePix1400Zoom
      0x112, 3, 1, Orientation, 1
      0x11a, 5, 1, XResolution, 72/1
      0x11b, 5, 1, YResolution, 72/1
      0x128, 3, 1, ResolutionUnit, 2
      ...

   The "brief" output style (see uFlags) looks like:
      Make, FUJIFILM
      Model, FinePix1400Zoom
      Orientation, 1
      XResolution, 72/1
      YResolution, 72/1
      ResolutionUnit, 2
      ....

   Note:
   Tags with more than 100 characters worth of data will be truncated.
   This is new behavior, as of v4.0.9.5

   Param             Use
   ------------------------------------
   hEXIFReader       EXIF data reader object created by
                     is5_EXIFCreateReader.

   phString          receives a handle to the string data.
                     you must use GlobalFree to release this
                     memory when you are finished with it.

   puLen             receives the length of the string

   uFlags            the first five bits determine which
                     section of the EXIF data to search.

                     bit      section
                     ---------------
                     0        Main IFD
                     1        EXIF IFD
                     2        Interoperability IFD
                     3        IFD 1 (thumbnail)
                     4        GPS IFD (GPS data)
                     5        MakerNote (see is5_EXIFCreateReader for notes)

                     8        brief output style
                     9        include tag data type
   Return
   ------
   result code.
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_EXIFGetTagList )(HISEXIFREADER hEXIFReader, HGLOBAL *phString, UINT32 *puLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFGetTagListData / _is5_EXIFGetTagListData

   Purpose : 
   Get a list of all data tags available from this EXIF
   parser. Tags are returned in an array of 
   ISEXIFTagInfo structs:

	ISEXIFTagInfo
	{
		UINT16	uTagID		: receives tag EXIF ID
		BYTE	uIFDID		: receives ID of the IFD in which the tag resides
		BYTE	uTagType	: receives tag type (one of is5_EXIFTYPE_*)
		UINT32	uTagCount	: number of items the tag contains
	} 
   
   Possible IFD ID values are:

		1: Main IFD
		2: EXIF IFD
		4: Interoperatbility IFD
		8: IFD 1 (thumbnail)
		16: GPS IFD
		32: MakerNote

   Param             Use
   ------------------------------------
   hEXIFReader       EXIF data reader object created by
                     is5_EXIFCreateReader.

   phData            receives a handle to the tag data.
                     you must use GlobalFree to release this
                     memory when you are finished with it.

   puLen             receives the number of ISEXIFTagInfo structures
					 which are stored in phData. 

   uFlags            the first five bits determine which
                     section of the EXIF data to search.

                     bit      section
                     ---------------
                     0        Main IFD
                     1        EXIF IFD
                     2        Interoperability IFD
                     3        IFD 1 (thumbnail)
                     4        GPS IFD (GPS data)
                     5        MakerNote (see is5_EXIFCreateReader for notes)
   Return
   ------
   result code.
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn(is5_EXIFGetTagListData)(HISEXIFREADER hEXIFData, HGLOBAL *phData, UINT32 *puLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFGetThumbnail / _is5_EXIFGetThumbnail

   Purpose : 
   Extract the thumbnail data block from an ImgSource
   EXIF data reader object. 

   Most cameras use JPG streams for the thumbnails. You
   can get the thumbnail image by creating an ImgSource memory 
   source object with the thumbnail data stream and reading
   it with is5_ReadJPG. 

   Some cameras store their thumbnails as TIFF files. Use is5_GuessFileType
   to determine the type.

   Note:
   This only applies if you told is5_EXIFCreateReader to
   read the EXIF thumbnail.

   Note:
   You should not call GlobalFree on the memory returned by this 
   function (via phData). This memory will be released when you
   call is5_EXIFDestroyReader.

   Param                Use
   ------------------------------------
   hEXIFReader          EXIF data reader object created by
                        is5_EXIFCreateReader.

   phData               receives a handle to the thumbnail data.
                        this memory is maintained by ImgSource internally.
                        you should not call GlobalFree on this. this 
                        memory will be released when you call is5_EXIFDestroyReader.

   puDataLen            receives the length of the thumbnail
                        data stream.
                     
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_EXIFGetThumbnail )(HISEXIFREADER hEXIFReader, HGLOBAL *phData, UINT32 *puDataLen);

/*---------------------------------------------------------
   is5_EXIFGetUserComment / _is5_EXIFGetUserComment

   Purpose : 
   Fetch the raw data for the UserComment tag from an ImgSource
   EXIF data reader object.

   Note:
   This only applies if you told is5_EXIFCreateReader to read
   the raw UserComment tag. See notes in is5_EXIFCreateReader.

   Note:
   You should not call GlobalFree on the memory returned by this 
   function (via phData). This memory will be released when you
   call is5_EXIFDestroyReader.

   Note:
   The raw UserComment tag data will start off with an 8-byte header
   that specifies the character set used for the text. Some common
   headers are shown below:

   Character Set  8 Byte Header                           Reference
   -------------  -------------                           ---------
   ASCII          x41, x53, x43, x49, x49, x00, x00, x00  ITU-T T.50 IA5 x
   JIS            x4A, x49, x53, x00, x00, x00, x00, x00  JIS X0208-1990x i
   Unicode        x55, x4E, x49, x43, x4F, x44, x45, x00  Unicode Standard xii
   Undefined      x00, x00, x00, x00, x00, x00, x00, x00  Undefined

   The text will immediately follow this 8-byte header, regardless of 
   character set. 

   Param                Use
   ------------------------------------
   hEXIFReader          EXIF data reader object created by
                        is5_EXIFCreateReader.

   phData               receives a handle to the user comment data.
                        this memory is maintained by ImgSource internally.
                        you should not call GlobalFree on this. this 
                        memory will be released when you call is5_EXIFDestroyReader.

   puDataLen            receives the length of the user comment
                        data stream.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    	_ISCConv _ISFn( is5_EXIFGetUserComment )(HISEXIFREADER hEXIFReader, HGLOBAL *phData, UINT32 *puDataLen);

/*---------------------------------------------------------
   is5_EXIFGetTagPos / _is5_EXIFGetTagPos

   Purpose : 
   Fetch the offset of the tag, and the offset of its IFD.
   This is most useful when you want to change the EXIF data.

   Note:
   This is only currently useful with EXIF tags read from 
   memory blocks with is5_EXIFCreateReaderFromByteArray (primarily
   JPEG_APP1 markers from JPEG files). This is not intended to 
   be used with EXIF data that was read with is5_EXIFCreateReader.

   Param                Use
   ------------------------------------
   hEXIFReader          EXIF data reader object created by
                        is5_EXIFCreateReader .

   uTag                 tag ID
   uFlags               see uFlags in is5_EXIFGetString
   puPos                receives tag position. -1 on error.
   puIFDPos             receives position of the IFD that contains the tag.

   puOutFlags           receives a set of flags

                        bit      purpose
                        ---      -------
                         0       tag was read from TIFF file.

                         1       tag data is in Motorola byte order

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_EXIFGetTagPos )(HISEXIFREADER hEXIFReader, UINT32 uTag, UINT32 uFlags, __int32 *puPos, __int32 *puIFDPos, UINT32 *puOutFlags);

/*---------------------------------------------------------
   is5_EXIFDestroyReader / _is5_EXIFDestroyReader

   Purpose : 
   Destroy an ImgSource EXIF data reader object. This
   will free all memory associated with the EXIF reader object.

   Param                Use
   ------------------------------------
   hEXIFReader          EXIF data reader object created by
                        is5_EXIFCreateReader .

    Return
   ------
    FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_EXIFDestroyReader )(HISEXIFREADER hEXIFReader);

/*---------------------------------------------------------
   is5_EXIFCreateWriter / _is5_EXIFCreateWriter

   Purpose : 
   Create an ImgSource EXIF writing object.

   Param                Use
   ------------------------------------
   uFlags               bit      purpose
                        ---      -------
                         0       use Motorola byte order
   Return
   ------
   HISEXIFWRITER EXIF writer object. NULL on failure.
---------------------------------------------------------*/
_ISDeclSpec HISEXIFWRITER _ISCConv _ISFn( is5_EXIFCreateWriter )(UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFWriterAddTag / _is5_EXIFWriterAddTag

   Purpose : 
   Add a data tag to the EXIF Writer object. This can also
   be used to remove tags from the object.

   Note:
   For a list of EXIF tag IDs, data types and acceptable values,
   see http://www.exif.org/Exif2-2.PDF . ImgSource does not
   enforce any rules about tag types, values or IFD locations. 

   Note:
   This function requires that you have at least one tag 
   in each of IFD0, and the EXIF IFD. As far as this function is 
   concerned, all other IFDs may be empty. This is not a reflection
   of the EXIF specification, merely an implementation requirement.

   Note: 
   You may only have one tag per IFD with the same tag ID.
   Adding a new tag with the same ID an an existing tag (in that
   IFD) will cause the old tag to be deleted.

   Note: 
   pData must point to an array of values organized correctly
   for the data type specified.

	   is5_EXIFTYPE_BYTE	    data is an array of BYTEs 
	   is5_EXIFTYPE_ASCII	    data is an ASCII-encoded, zero-terminated string
	   is5_EXIFTYPE_SHORT	    data in an array of 16-bit unsigned integers 
	   is5_EXIFTYPE_LONG	    data in an array of 32-bit unsigned integers
	   is5_EXIFTYPE_SBYTE	    data in an array of BYTEs
	   is5_EXIFTYPE_UNDEFINED	 data in an array of BYTEs
	   is5_EXIFTYPE_SSHORT	    data in an array of 16-bit signed integers 
	   is5_EXIFTYPE_SLONG	    data in an array of 32-bit signed integers
	   is5_EXIFTYPE_FLOAT	    data in an array of floats (32-bit IEEE floating point)
	   is5_EXIFTYPE_DOUBLE	    data in an array of doubles (64-bit IEEE floating point)

	   is5_EXIFTYPE_RATIONAL	 data in an array of 64-bit unsigned fractions,
                               where each fraction is composed of two 32-bit values;
                               the first 32-bit integer is the numerator, and the second
                               is the denominator. so, a single tag value is two 32-bit values.

      is5_EXIFTYPE_SRATIONAL	 like is5_EXIFTYPE_RATIONAL, but the values may be signed
   
   Param                Use
   ------------------------------------
   hEXIFWriter          valid EXIF data writer object
   uTagID               ID to assign this tag

   uTagType             See ISTypes.h for the list of 
                        EXIF data types.

   uDataCount           number of data items to assign

   pData                pointer to tag data. passing NULL here
                        will remove the tag with ID uTagID from 
                        IFD uIFD.

   uIFD                 IFD in which to place this tag:
                           0: IFD0
                           1: IFD1
                           2: EXIF IFD
                           3: interoperability IFD
                           4: GPS IFD

   uFlags               unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_EXIFWriterAddTag )(HISEXIFWRITER hEXIFWriter, UINT32 uTagID, UINT32 uTagType, UINT32 uDataCount, BYTE *pData, UINT32 uIFD, UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFWriteTags / _is5_EXIFWriteTags

   Purpose : 
   Write the EXIF tags added by is5_EXIFWriterAddTag to
   a memory buffer. This buffer can then be used as an APP0 marker
   in a JPG file, or parsed with is5_EXIFCreateReaderFromByteArray.

   Note:
   If using this in a JPG file, you must skip the first four bytes.
   Ex.   
      HGLOBAL hData = NULL;
      UINT32 uDataLen = 0;
      is5_EXIFWriteTags(hEXIFWriter, &hData, &uDataLen, (1 << 5));
      is5_AddJPGOutputMarker((BYTE *)hData + 4, uDataLen - 4, 0xE1);

   Param                Use
   ------------------------------------
   hEXIFWriter          valid EXIF data writer object

   phData               receives HGLOBAL handle to the memory allocated
                        to hold the EXIF block

   puDataLen            receives length of EXIF data block 

   uFlags               unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_EXIFWriteTags )(HISEXIFWRITER hEXIFWriter, HGLOBAL *phData, UINT32 *puDataLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFCopyTagsToWriter / _is5_EXIFCopyTagsToWriter

   Purpose : 
   Copy all tag data from the EXIF reader object to the
   EXIF Writer object. 

   Note:
   This does not copy maker note or thumbnail data.

   Param                Use
   ------------------------------------
   hEXIFWriter          EXIF data reader object created by
                        is5_EXIFCreateWriter.

   hEXIFReader          EXIF data reader object created by
                        is5_EXIFCreateReader.

   uFlags               unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_EXIFCopyTagsToWriter )(HISEXIFWRITER hEXIFWriter, HISEXIFREADER hEXIFReader, UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFSetThumbnailInWriter / _is5_EXIFSetThumbnailInWriter

   Purpose : 
   Set a thumbnail image to the EXIF writer. This is 
   typically a small JPG-encoded version of the main 
   image. 

   Param                Use
   ------------------------------------
   hEXIFWriter          EXIF data reader object created by
                        is5_EXIFCreateWriter.

   pThumbnailData       pointer to JPG-encoded thumbnail data
   uDataLen             number of bytes in pThumbnailData
   uFlags               unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv  _ISFn( is5_EXIFSetThumbnailInWriter )(HISEXIFWRITER hEXIFWriter, BYTE *pThumbnailData, UINT32 uDataLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_EXIFDestroyWriter / _is5_EXIFDestroyWriter

   Purpose : 
   Destroy an ImgSource EXIF data reader object. This
   will free all memory associated with the EXIF reader object.

   Param                Use
   ------------------------------------
   hEXIFWriter          EXIF data reader object created by
                        is5_EXIFCreateWriter.

   uFlags               unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_EXIFDestroyWriter )(HISEXIFWRITER hEXIFWriter, UINT32 uFlags);

#ifdef __cplusplus
}
#endif

#endif