/*---------------------------------------------------------
    ISFileUtil.h : Part of the ImgSource library
    
    Image file utilities

    This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

    No portion of this code may be copied or distributed without 
    permission of Smaller Animals Software, Inc..
    http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is5_FILEUTILSH
#define is5_FILEUTILSH

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_GuessFileType / _is5_GuessFileType

   Purpose : 
   Try to guess at the format of the input image 
   
   Param             Use
   ------------------------------------
   hSource           valid source object

   Return
   ------
   file type :
      0 - unable to identify or file read error
      1 - BMP
      2 - GIF
      3 - JPG
      4 - PNG
      5 - PCX
      6 - TIFF
      7 - WMF
      8 - EMF
      9 - PAX
      10 - PSD (Photoshop)
      11 - TLA (encrypted)
      12 - TLA (unencrypted)
      13 - WBMP
      14 - TGA
      15 - PCD Index / Overview
      16 - PCD
      17 - JPEG 2000
      18 - JPEG 2000 data stream
      19 - FlashPix
      20 - PBM
      21 - PGM
      22 - PPM

   Note : 
   This function can confuse certain varieties of TGA
   files for WBMP files. The TGA specification does not provide
   for a identifying "signature" in the TGA file, so there is
   no sure way to tell a TGA file from a random stream of BYTEs.
   WBMP does provide such a signature, unfortunately, 
   the signature is simply two zero BYTEs at the start of 
   the file and it is quite common for TGA files to start with two
   zero bytes. 

   To get around the problem, this function will examine the 
   file extension of the source file (if the source manager was
   created with is5_OpenFileSource). If the signature says WBMP
   but the file extension says "TGA", this will return 14 (TGA).

   Note : This function moves the file pointer for the
   source manager used. So you can't do this : 

   HISSRC hSrc = is5_OpenFileSource(...);
   is5_GuessFileType(hSrc);   // moves file pointer
   is5_ReadJPG(hSrc...); // not at start of file!! read will fail
   is5_CloseSource(hSrc);

   Instead, do this : 

   HISSRC hSrc = is5_OpenFileSource(...);
   is5_GuessFileType(hSrc);   // moves file pointer
   is5_Seek(hSrc, 0,0);       // moves file pointer back to the start of the file
   is5_ReadJPG(hSrc...); // succeeds
   is5_CloseSource(hSrc);

---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_GuessFileType )(HISSRC hSource);

/*---------------------------------------------------------
   is5_SetFileReadAllocCallbacks / _is5_SetFileReadAllocCallbacks

   Purpose :
   Set the memory management callbacks for file reading.
   This will alloc the caller to handle allocation/deallocation
   of the memory used to hold images for file reads. Set all 
   three functions to NULL if you want ImgSource to handle
   allocation (default).

   Note:
   Not currently used with metafile reading.

   Note: 
   This only applied to the output buffers for file reading. This
   does not affect other functions, such as image processing 
   or rendering .

   Note:
   If you use this, you must provide all three functions. 

   Note:
   When you have allocated the output image memory, the read
   function will return the address of your output buffer as
   its return value cast to an HGLOBAL. Only you know for certain
   the proper way of deallocating this memory. For example, if
   you allocated the output buffer with "malloc", the return value
   is really a void* pointer, not an HGLOBAL, and you must use 
   "free" to deallocate it, not GlobalFree. 

   Note:
   * The allocation function is called in every file reading function.

   * The reallocation function is called infrequently (only with TIFFs 
   and certain PSD files).

   * The deallocation function is generally only called during serious 
   error conditions. 

   Example, using the standard C allocation functions:

   void * CALLBACK MyAlloc(const UINT32 uSize, const IS_CALLBACK_DATA uUserData)
   {
      TRACE("allocating %d (%d)\n", uSize, uUserData);
      return malloc(uSize);
   }

   void * CALLBACK MyReAlloc(void * pData, const UINT32 uSize, const IS_CALLBACK_DATA uUserData)
   {
      TRACE("rallocating %x %d (%d)\n", pData, uSize, uUserData);
      return realloc(pData, uSize);
   }

   BOOL CALLBACK MyFree(void * pData, const IS_CALLBACK_DATA uUserData)
   {
      TRACE("freeing %x (%d)\n", pData, uUserData);
      free(pData);
      return TRUE;
   }

   // tell ImgSource to use my functions for allocation of the output image
   is5_SetFileReadAllocCallbacks(MyAlloc, MyReAlloc, MyFree, 42);

   // read a file
   UINT32 w, h;
   HISSRC hSrc = is5_OpenFileSource(filename);

   // this will allocate with the MyAlloc
   BYTE * pImg = (BYTE *)is5_ReadBMP(hSrc, &w, &h, 24, NULL);
   is5_CloseSource(hSrc);

   // clean up with "free" because we allocated with "malloc"
   free(pImg);

   // reset to the default allocation method
   is5_SetFileReadAllocCallbacks(NULL, NULL, NULL, 0);

   Param             Use
   ------------------------------------
   pAlloc            pointer to your allocation callback
   pReAlloc          pointer to your reallocation callback
   pFree             pointer to your deallocation callback
   uUserData         user data. this data is thread safe.

   Return
      none

---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_SetFileReadAllocCallbacks )(IS3IMGMEM_ALLOC_CBPTR pAlloc, IS3IMGMEM_REALLOC_CBPTR pReAlloc, IS3IMGMEM_FREE_CBPTR pFree, IS_CALLBACK_DATA uUserData);

/*---------------------------------------------------------
	is5_PNGSetCompressionOptions / _is5_PNGSetCompressionOptions

	Purpose :

	Adjust one or more PNG compression parameters. These 
	settings can improve (or degrade) the way PNGs are 
	compressed, but it's impossible to predict exactly what 
	the effects will be, since the biggest factor in PNG 
	compression is the input image data iteslf.

	Note:
	The values for compression level, compression memory limit, 
	compression strategy and compression window bits control zLib's
	operation for PNG writing (PNG only, this doesn't affect any 
	other zLib usage: TIFF, TLA, PAX, etc). You should consult the 
	zLib documentation for precise explanations of what these 
	parameters do and how they interact with each other.

	Some of the descriptions here are from the zLib 
	documentation.

	Note:
	Each call of this overrides the settings from previous calls, and
	the uMask parameter from the last call determines which options are
	passed to the PNG writing code. 

	Example:
		is5_PNGSetCompressionOptions(0, 0, 0, 0, 0x80, 1, 0); // set filters to Paeth

		...

		is5_PNGSetCompressionOptions(-1, 0, 0, 0, 0, 2, 0); // set compression level to -1

		// only the compression level change is used, the filter change is ignored, 
		// because the second call to is5_PNGSetCompressionOptions has overriden the first.

		is5_WritePNG(...)

		
   Param								Use
   ------------------------------------
	uCompressionLevel				zLib compression level:
											0 - no compression
											1 - best speed
											9 - best compression
 										  -1 - default

	uCompressionMemoryLimit		specifies how much memory is allocated 
										for the internal compression state. 
										1 : use minimum memory. this is slow and reduces compression ratio
										9 : use maximum memory for optimal speed
										The default value is 8. 

	uCompressionStrategy			Tune the compression algorithm.
											1 : data is pre-filtered
											2 : Huffman-only
											3 : RLE
											4 : Fixed
											0 : default

   uCompressionWindowBits		Valid values are 9-15.
										Larger values of this parameter result in better
										compression at the expense of memory usage.

   uFilterTypes					Selects which PNG pre-compression filters will be used
										on the input image data. Choose one or more filters
										from the list below and combine them with logical OR.

											0x08 : None
											0x10 : Subtraction
											0x20 : Up
											0x40 : Average
											0x80 : Paeth

	uMask								Selects which of the compression fields
										you wish to set. Combine these flags with 
										logical OR to set multiple values:

											0x01 : Set filters
											0x02 : Set compression level
											0x04 : Set compression memory limit
											0x08 : Set compression strategy
											0x10 : Set Compression window bits
										
	uFlags							unused

---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_PNGSetCompressionOptions )(__int32 iCompressionLevel, UINT32 uCompressionMemoryLimit, UINT32 uCompressionStrategy, UINT32 uCompressionWindowBits, UINT32 uFilterTypes, UINT32 uMask, UINT32 uFlags);

/*---------------------------------------------------------
   is5_PNGSetScreenGamma / _is5_PNGSetScreenGamma

   Purpose :
   Set the screen gamma value for PNG reads and writes.

   Param                   Use
   ------------------------------------
   fScreenGamma        value to set as gamma. 

---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_PNGSetScreenGamma )(double fScreenGamma);

/*---------------------------------------------------------
    is5_PNGSetDefBackground / _is5_PNGSetDefBackground

    Purpose :
        Set the background colors to be used in blending the
    alpha channel from PNG images that have one. This is used
    for both reading and writing of PNG images - if the input image
    uses an alpha channel but has no background specified, this
    value is used. When writing a file, this value is written
    as the default background.


    Param                   Use
    ------------------------------------
    uIndex              palette index to use for background
                            (if you know the incoming image has a 
                            certain palette)

    uRed                    red component of RGB background value
    uGreen              green component of RGB background value
    uBlue                   blue component of RGB background value
    uGray                   gray value to be used for grayscale images

    Return
        none
---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_PNGSetDefBackground )(UINT32 uIndex, UINT32 uRed, UINT32 uGreen, UINT32 uBlue, UINT32 uGray);

/*---------------------------------------------------------
    is5_PNGGetLastBackground / _is5_PNGGetLastBackground

    Purpose :
        Get the default background color for the last PNG
    image read. This is the color that is used when 
    you allow ImgSource to alpha-blend images read as 24-bit RGB.

   Note: the values may returned may be 16-bit values (if the
   input image had 16-bit pixel components). In this case,
   discard the lower 8 bits.

    Param               Use
    ------------------------------------
    puIndex             receives palette index, if 8-bit paletted image
    puRed               receives red component, for RGB images
    puGreen             green component 
    puBlue              blue component 
    puGray              receives gray value, for grayscale images

    Return
        none
---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_PNGGetLastBackground )(UINT32 * puIndex, UINT32 * puRed, UINT32 * puGreen, UINT32 * puBlue, UINT32 * puGray);

/*---------------------------------------------------------
   is5_PNGSetTransparencyIndex / _is5_PNGSetTransparencyIndex

   Purpose :
      Set the transparent colors. When writing a file, 
   these value are written as the transparency chunk.

   This tells the PNG reader to treat the RGB value or 
   grayscale value as transparent. This is in addition to
   any per-pixel transparency values your 
   image might have. Not all readers support PNG transparency.

   Param               Use
   ------------------------------------
   uIndex              unused
   uRed                red component of RGB transparency value
   uGreen              green component of RGB transparency value
   uBlue               blue component of RGB transparency value
   uGray               gray value transparency to be used for grayscale images
   bSet                 TRUE if setting/enabling. 
                        FALSE to disable.


    Return
        none
---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_PNGSetTransparencyIndex )(BOOL bSet, UINT32 uIndex, UINT32 uRed, UINT32 uGreen, UINT32 uBlue, UINT32 uGray);

/*---------------------------------------------------------
   is5_PNGReadPNGText  / _is5_PNGReadPNGText 

   Purpose :
   A PNG file may contain any number of key/text pairs.
   This function reads and stores these strings in the 
   ImgSource PNG text buffer. These strings remain 
   valid until :

      1) the next time you call this function
      2) is5_ClearPNGInputText is called

   See is5_GetPNGInputTextCount, is5_GetPNGInputText,
   is5_ClearPNGInputText.

   Return
     number of strings 
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_PNGReadPNGText )(HISSRC hSource, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GetPNGInputTextCount / _is5_GetPNGInputTextCount

   Purpose :
   Get count of text strings read by last is5_PNGReadPNGText call.

   Return
     number of strings 
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_GetPNGInputTextCount )();

/*---------------------------------------------------------
   is5_GetPNGInputText / _is5_GetPNGInputText

   Purpose :
   Fetch a comment string read in the last is5_PNGReadPNGText call.

   Param             Use
   ------------------------------------
   uIndex            index of string to fetch
   hKey              receives handle to key text
   hText             receives handle to string text
   puCompressed      receives compression flag

   Caller must use GlobalFree to delete key and string!

   Return
     FALSE on error
        
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetPNGInputText )(UINT32 uIndex, HGLOBAL *pKey, HGLOBAL *pText, UINT32 *puCompression);

/*---------------------------------------------------------
   is5_ClearPNGInputText / _is5_ClearPNGInputText

   Purpose :
   Clear all PNG input strings

   Return
     TRUE if ok
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_ClearPNGInputText )();

/*---------------------------------------------------------
   is5_AddPNGOutputText / _is5_AddPNGOutputText

   Purpose :
   Add a text field to the ImgSource global PNG text buffers.
   PNG text fields consist of a mandatory 1-79 character key and 
   an optional text buffer of any size. The text can be written 
   compressed or uncompressed. It is not recommended that you 
   compress text fields of length < 1000 bytes.

   Keys should be plain ASCII, no control or non-printable chars.

   You may add as many fields as you wish. 

   The keywords that are given in the PNG Specification are:

      Title            Short (one line) title or caption for image
      Author           Name of image's creator
      Description      Description of image (possibly long)
      Copyright        Copyright notice
      Creation Time    Time of original image creation
      Software         Software used to create the image
      Disclaimer       Legal disclaimer
      Warning          Warning of nature of content
      Source           Device used to create the image
      Comment          Miscellaneous comment; conversion from other
                       image format

   Param                Use
   ------------------------------------
   pText                zero-terminated string to write
   pKey                 zero-terminated string to use as the key
   bCompress            1 to compress (for strings > 1K)

   Return
     FALSE on error
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_AddPNGOutputText )(const char *pText, const char *pKey, UINT32 bCompress);

/*---------------------------------------------------------
   is5_ClearPNGOutputText / _is5_ClearPNGOutputText

   Purpose :
   Clear all PNG output strings

   Return
     TRUE if ok
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_ClearPNGOutputText )();

/*---------------------------------------------------------
   is5_DeletePNGText / _is5_DeletePNGText

   Purpose :
   Remove PNG text strings, from either the input or
	the output array, based on key/text. All strings matching the 
	search values are removed.

	Note:
   Compression value is ignored for all comparisons.

   Param                Use
   ------------------------------------
   pText                text to search for
   pKey                 key to search for
	bInput					search input strings, else search output strings

	uFlags					0 : match based on key and string
								1 : match on key only
								2 : match on text only

  Return
     TRUE if ok (even if no strings match)
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv  _ISFn( is5_DeletePNGText )(const char *pText, const char *pKey, BOOL bInput, UINT32 uFlags);

/*---------------------------------------------------------
   is5_CopyPNGInputMetadataToOutputMetadata / _is5_CopyPNGInputMetadataToOutputMetadata

   Purpose :
   Copy all comment strings read by the last is5_PNGReadPNGText
   call to the PNG output text, for writing to the 
   next output file. See is5_AddPNGOutputText.

   Return
     TRUE if ok
---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_CopyPNGInputMetadataToOutputMetadata )();

/*---------------------------------------------------------
   is5_GetTIFFTag / _is5_GetTIFFTag

   Get a single TIFF tag data from a file.

   Also see is5_GetTIFFBinaryTag.

   Tag name                   ID      Variant type
   -----------------------------------------------------
   (tags and note from tiff.h, part of LibTiff)

   TIFFTAG_ORIENTATION        274    VT_UI2
      Values:
         image orientation 
         ORIENTATION_TOPLEFT     1    row 0 top, col 0 lhs 
         ORIENTATION_TOPRIGHT    2    row 0 top, col 0 rhs 
         ORIENTATION_BOTRIGHT    3    row 0 bottom, col 0 rhs 
         ORIENTATION_BOTLEFT     4    row 0 bottom, col 0 lhs 
         ORIENTATION_LEFTTOP     5    row 0 lhs, col 0 top 
         ORIENTATION_RIGHTTOP    6    row 0 rhs, col 0 top 
         ORIENTATION_RIGHTBOT    7    row 0 rhs, col 0 bottom 
         ORIENTATION_LEFTBOT     8    row 0 lhs, col 0 bottom 

   TIFFTAG_IMAGEWIDTH         256         VT_UI4
   TIFFTAG_IMAGELENGTH        257         VT_UI4
   TIFFTAG_BITSPERSAMPLE      258         VT_UI2

   TIFFTAG_COMPRESSION        259         VT_UI2
      Values:
         COMPRESSION_NONE        1       dump mode 
         COMPRESSION_CCITTRLE    2       CCITT modified Huffman RLE 
         COMPRESSION_CCITTFAX3   3       CCITT Group 3 fax encoding 
         COMPRESSION_CCITTFAX4   4       CCITT Group 4 fax encoding 
         COMPRESSION_LZW         5       Lempel-Ziv  & Welch 
         COMPRESSION_OJPEG       6       6.0 JPEG 
         COMPRESSION_JPEG        7       JPEG DCT compression 
         COMPRESSION_NEXT        32766   NeXT 2-bit RLE 
         COMPRESSION_CCITTRLEW   32771   #1 w/ word alignment 
         COMPRESSION_PACKBITS    32773   Macintosh RLE 
         COMPRESSION_THUNDERSCAN 32809   ThunderScan RLE 
      
         codes 32895-32898 are reserved for 
         ANSI IT8 TIFF/IT <dkelly@etsinc.com) 
         COMPRESSION_IT8CTPAD    32895   IT8 CT w/padding 
         COMPRESSION_IT8LW       32896   IT8 Linework RLE 
         COMPRESSION_IT8MP       32897   IT8 Monochrome picture 
         COMPRESSION_IT8BL       32898   IT8 Binary line art 
      
         compression codes 32908-32911 are reserved for Pixar 
         COMPRESSION_PIXARFILM   32908   Pixar companded 10bit LZW 
         COMPRESSION_PIXARLOG    32909   Pixar companded 11bit ZIP 
         COMPRESSION_DEFLATE     32946   Deflate compression 
         COMPRESSION_ADOBE_DEFLATE   8   Deflate compression, 
                                         as recognized by Adobe 
      
         compression code 32947 is reserved for 
         Oceana Matrix <dev@oceana.com> 
         COMPRESSION_DCS         32947   Kodak DCS encoding 
      
         COMPRESSION_JBIG        34661   is5_O JBIG 
         COMPRESSION_SGILOG      34676   SGI Log Luminance RLE 
         COMPRESSION_SGILOG24    34677   SGI Log 24-bit packed 

   TIFFTAG_PHOTOMETRIC     262         VT_UI2
      Values:
         PHOTOMETRIC_MINISWHITE  0    min value is white 
         PHOTOMETRIC_MINISBLACK  1    min value is black 
         PHOTOMETRIC_RGB         2    RGB color model 
         PHOTOMETRIC_PALETTE     3    color map indexed 
         PHOTOMETRIC_MASK        4    holdout mask 
         PHOTOMETRIC_SEPARATED   5    color separations 
         PHOTOMETRIC_YCBCR       6    CCIR 601 
         PHOTOMETRIC_CIELAB      8    1976 CIE L*a*b* 
         PHOTOMETRIC_LOGL        32844    CIE Log2(L) 
         PHOTOMETRIC_LOGLUV      32845    CIE Log2(L) (u',v') 

   TIFFTAG_RESOLUTIONUNIT  296         VT_UI2
      Values:
         RESUNIT_NONE            1   no meaningful units
         RESUNIT_INCH            2   english
         RESUNIT_CENTIMETER      3   metric

   TIFFTAG_XRESOLUTION        282         VT_R4
   TIFFTAG_YRESOLUTION        283         VT_R4
   TIFFTAG_NUMBEROFINKS       334         VT_UI2
   TIFFTAG_DOCUMENTNAME       269         VT_BSTR
   TIFFTAG_IMAGEDESCRIPTION   270         VT_BSTR
   TIFFTAG_MAKE               271         VT_BSTR
   TIFFTAG_MODEL              272         VT_BSTR
   TIFFTAG_ROWSPERSTRIP       278         VT_UI4
   TIFFTAG_PAGENAME           285         VT_BSTR
   TIFFTAG_SOFTWARE           305         VT_BSTR
   TIFFTAG_DATETIME           306         VT_BSTR
   TIFFTAG_ARTIST             315         VT_BSTR
   TIFFTAG_HOSTCOMPUTER       316         VT_BSTR
   TIFFTAG_INKNAMES           333         VT_BSTR
   TIFFTAG_TARGETPRINTER      337         VT_BSTR
   TIFFTAG_TILEWIDTH          322         VT_UI4
   TIFFTAG_TILELENGTH         323         VT_UI4
   
   TIFFTAG_INKSET             332         VT_UI4
      Values:
      INKSET_CMYK                1  CMYK image

   Note:
   It is the caller's responsibility to call SysFreeString
   to release any string data that was allocated by this
   function.

   ex.
   is5_GetTIFFTag(hSrc, 256, &varData);
   if (varData.vt == VT_BSTR)
   {
     if (varData.bstrVal!=NULL)
     {
         ::SysFreeString(varData.bstrVal);
     }
   }

   Note:
   The caller must call is5_Seek(hSrc, 0, 0) after
   each call to reset the file pointer, if you are
   getting multiple tags, or if you plan to read the
   file using any functions that assume the file pointer
   is at the start of the file.

   Param                   Use
   ------------------------------------
   hSource                 source manager
   uFrameIdx               frame index of the TIFF (0 for first)
   uTiffTagID              ID value for the tag you want (see table above)
   pVarData                receives data. 

   returns FALSE on failure
---------------------------------------------------------*/

/* !!!!!!!!!!!!!!!!!!
    
    if you get compiler errors here, you probably need 
    to #include "windows.h". 
    
    you might also need to #undef WIN32_LEAN_AND_MEAN in 
    your stdafx.h file.

    VARIANT is a standard Win32 type.

    !!!!!!!!!!!!!!!!!!
*/
#ifndef IS_EXCLUDE_VARIANT
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetTIFFTag )(HISSRC hSource, UINT32 uFrameIdx, UINT32 uTiffTagID, VARIANT *pVarData);
#endif

/*---------------------------------------------------------
   is5_GetTIFFBinaryTag / _is5_GetTIFFBinaryTag

   Purpose :
   Retrieve the TIFF tag data for the specified tag.

   This currently supports the following tags:

      Name                 ID       Description

      TIFFTAG_DOTRANGE     336      --

      TIFFTAG_EXTRASAMPLES 338      Purpose of any extra channels in the TIFF image.
                                    Note that ImgSource does not return any of these
                                    extra channels.

      TIFFTAG_XMP_DATA     700      Adobe XMS data
      TIFFTAG_ANNOTATIONS  32932    Wang annotations data
      TIFFTAG_RICHTIFFIPTC 33723    IPTC Data from the RichTIFF specification
      TIFFTAG_PHOTOSHOP    34377    Private tag registered to Adobe for PhotoShop
      TIFFTAG_ICCPROFILE   34675    ICC profile block
   

   Param                Use
   ------------------------------------
   hSource              opened source manager object
   uTIFFFrameIdx        TIFF trame index
   phData               receives HGLOBAL to the data
   puDataLen            receives length of data. 

   uFlags               unused 

   Return
   ------
   FALSE on failure

   Caller must use GlobalFree to free the memory

---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_GetTIFFBinaryTag )(HISSRC hSource, UINT32 uFrameIdx, UINT32 uTiffTagID, HGLOBAL* phData, UINT32 *puDataLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_SetTIFFBinaryTag / _is5_SetTIFFBinaryTag

   Purpose :
   Set a chunk of data to be written to a specific TIFF tag.
   These chunks will be written on the next file write

   This currently supports the following tags:

      Name                 ID       Description

      TIFFTAG_XMP_DATA     700      Adobe XMS data

      TIFFTAG_ANNOTATIONS  32932    Wang annotations data

      TIFFTAG_RICHTIFFIPTC 33723    IPTC Data from the RichTIFF specification

      TIFFTAG_PHOTOSHOP    34377    Private tag registered to Adobe for PhotoShop.

   Param                Use
   ------------------------------------
   uTagID               TIFF tag ID to set
   pData                pointer to the data block
   uDataLen             size of the data block
   uFlags               unused 

   Return
   ------
   None
---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SetTIFFBinaryTag )(UINT32 uTagID, BYTE *pData, UINT32 uDataLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_SetTIFFOutputStripSize / _is5_SetTIFFOutputStripSize

   Set the number of rows per output TIFF strip. This
   affects all subsequent TIFF writes.

   Param                Use
   ------------------------------------
   iSize                Strip size. 

                        Use -1, to allow ImgSource to 
                        automatically choose a strip size. (default). 

                        This number must be smaller than the
                        height of the output image.

---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SetTIFFOutputStripSize )(int iSize);

/*---------------------------------------------------------
   is5_SetTIFFJPGQuality / _is5_SetTIFFJPGQuality

   Set the JPEG-in-TIFF quality/compression ratio. Only
   applied to JPEG-in-TIFF compression.

   Param                Use
   ------------------------------------
   iQuality             JPEG compression quality.
                        1..100 , lower values give higher 
                        compression, but lower quality.

---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SetTIFFJPGQuality )(int iQuality);

/*---------------------------------------------------------
   is5_SetTIFFOutputTileSize / _is5_SetTIFFOutputTileSize

   Set the dimensions for the tiles in tiled TIFFs. The
   default size is 256x256, though this may vary depending
   on compression type.
   
   Note:
   Dimensions must be multiples of 16.

   Note:
   Actual sizes used may be subject to modification based 
   on compression type.

   Param                Use
   ------------------------------------
   uTileX               width of tiles
   uTileT               height of tiles

---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SetTIFFOutputTileSize )(UINT32 uTileX, UINT32 uTileY);

/*---------------------------------------------------------
   is5_SetTIFF4thChannelAlphaHandlingMode / _is5_SetTIFF4thChannelAlphaHandlingMode

   TIFF allows for any number of "extra" image channels to
   be stored in an image. These channels can be marked as 
   alpha channels or they can be marked as "unspecified". 
   Some TIFF writers (Adobe's in particular) store their 
   alpha data in "unspecified" channels - even though 
   there is a perfectly valid way of labelling channel 
   data as alpha data. 

   So, this means, in order to read the alpha channel in a
   TIFF created by Photoshop, you have to assume that the 
   "unspecified" extra channel data is actually alpha data. 
   This function allows you to do that - at least for the 
   4th channel.

   But, beware, other writers might actually be storing 
   something other than alpha data in an unspecified 4th channel.
   It's up to you to decide if you want to read it as alpha or 
   not, ImgSource can't help.

   Param                Use
   ------------------------------------
   iAlphaHandlingMode   mode value:    
                           -1 : take LibTiff's default handling
                           0  : ignore 4th channel (fill with 0s)
                           1  : read alpha as "associated" 
                           2  : read alpha as "unassociated" 

---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SetTIFF4thChannelAlphaHandlingMode )(int iAlphaHandlingMode);

/*---------------------------------------------------------
   is5_SetTIFFOrientation  / _is5_SetTIFFOrientation 

   Set the TIFF orientation flag. This is a note to TIFF
   readers that they should rotate and/or flip your image
   upon reading it.

   Param                Use
   ------------------------------------
   iOrientation         orientation flag. one of:
                        1 top-left      row 0 top, col 0 left side 
                        2 top-right     row 0 top, col 0 right side
                        3 bottom-right  row 0 bottom, col 0 right side
                        4 bottom-left   row 0 bottom, col 0 left side
                        5 left-top      row 0 left side, col 0 top 
                        6 right-top     row 0 right side, col 0 top 
                        7 right-bottom  row 0 right side, col 0 bottom 
                        8 left-bottom   row 0 left side, col 0 bottom 

---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SetTIFFOrientation )(int iOrientation);

/*---------------------------------------------------------
   is5_GetPAXID / _is5_GetPAXID

   Purpose :
   Get the ID for a PAX file.

   Param                Use
   ------------------------------------
   hSource              source manager
   pID                  PAX file ID (128 BYTEs)

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetPAXID )(HISSRC hSource, BYTE *pID);

/*---------------------------------------------------------
   is5_GetPAXInputTagCount / _is5_GetPAXInputTagCount

   Purpose :
   Get count of PAX data tags read by last PAX file 
   (or dimension) read.

   Return
   ------
   tag count

---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_GetPAXInputTagCount )();

/*---------------------------------------------------------
   is5_GetPAXInputTag / _is5_GetPAXInputTag

   Purpose :
   Retrieve a PAX data tag as read by the last PAX file
   read or dimension read.

   Param                Use
   ------------------------------------
   uIndex               index of tag to retrieve

   pName                receives tag name. this will always
                        be 4 characters plus a trailing NULL BYTE.

   phData               receives HGLOBAL to tag data
   puDataLen            receives length of tag data

   Return
   ------
   FALSE on failure

   caller must use GlobalFree to free the tag data memory

---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetPAXInputTag )(UINT32 uIndex, BYTE *pName, HGLOBAL *phData, UINT32 *puDataLen);
                                              
/*---------------------------------------------------------
   is5_AddPAXOutputTag / _is5_AddPAXOutputTag

   Purpose :
   Add a PAX data tag to be written to the next PAX file.

   Param                Use
   ------------------------------------
   pName                tag name. 4 characters, plus a zero byte
   pData                tag data
   DataLen              length of tag data

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_AddPAXOutputTag )(const char *pName, const BYTE *pData, UINT32 uDataLen);

/*---------------------------------------------------------
   is5_ClearPAXOutputTags / _is5_ClearPAXOutputTags

   Purpose :
   Clear all PAX output tags. 

---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_ClearPAXOutputTags )();
                                          // 32
/*---------------------------------------------------------
   is5_SetPAXOutputInfo / _is5_SetPAXOutputInfo

   Purpose :
   Set output application, author and PAX file ID
   for the next PAX file.

   Param                Use
   ------------------------------------
   pAppName             name of application (max 32 chars)
   pAuthor              author name (max 32 chars)
   pID                  PAX file ID (128 BYTEs)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_SetPAXOutputInfo )(const char *pAppName, const char *pAuthor, BYTE *pID);

/*---------------------------------------------------------
   is5_GetPAXInputInfo / _is5_GetPAXInputInfo

   Purpose :
   Get application, author and PAX file ID
   from the last read PAX file. 

   This data is set on PAX file read or dimension read.

   Param                Use
   ------------------------------------
   pAppName             name of application (max 32 chars)
   pAuthor              author name (max 32 chars)
   pID                  PAX file ID (128 BYTEs)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetPAXInputInfo )(char *pAppName, char *pAuthor, BYTE *pID);

/*---------------------------------------------------------
   is5_MakePAXID / _is5_MakePAXID

   Purpose :
   Generate a new PAX ID.

   Each PAX file should have a unique PAX ID. You can
   use this function, with two seed strings of your 
   choice, to generate PAX IDs.

   It is recommended that you change at least one of 
   these seed strings for _every_ PAX file your application
   creates.

   Some good choices for seed strings are:
     ** Windows user name:
     HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\RegisteredOwner

     ** output file name

     ** current time as a string

   Param                   Use
   ------------------------------------
   pSeedString1            seed strings
   pSeedString2            
   pOutID                  receives PAX file ID 
                           (128 BYTEs, allocated by the caller)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_MakePAXID )(const char *pSeedString1, const char *pSeedString2, BYTE *pOutID);

/*---------------------------------------------------------
   is5_GetTLAInputTagCount / _is5_GetTLAInputTagCount

   Purpose :
   Get count of TLA data tags read by last PAX file 
   (or dimension) read.

   Return
   ------
   tag count

---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_GetTLAInputTagCount )();

/*---------------------------------------------------------
   is5_GetTLAInputTag / _is5_GetTLAInputTag

   Purpose :
   Retrieve a TLA data tag as read by the last TLA file
   read or dimension read.

   Param                Use
   ------------------------------------
   uIndex               index of tag to retrieve
   phData               receives HGLOBAL to tag data
   puDataLen            receives length of tag data

   Return
   ------
   FALSE on failure

   caller must use GlobalFree to free the tag data memory

---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetTLAInputTag )(UINT32 uIndex, HGLOBAL *phData, UINT32 *puDataLen);
                                        
/*---------------------------------------------------------
   is5_GetPSDThumbnail / _is5_GetPSDThumbnail

   Purpose :
   Retrieve the thumbnail image from a Photoshop file. 
   
   The memory returned from this function is generally
   a JPG-compressed image. Check the 'format' member of
   the ISPSDThumbnailInfo struct, to verfiy. 

   Note:
   These are the members of the ISPSDThumbnailInfo struct:

      Parameter      Purpose
      ---------      ---
      colorOrder     if 1, pixels are in RGB order. if 0, BGR

      format         if 1, image is JPG compressed. 
                     if 0, image is "kRawRGB". (but, this 
                     format is not defined anywhere in the 
                     Photoshop 6.0 spec)

      width          width of uncompressed thumbnail image
      height         height of uncompressed thumbnail image
      widthbytes     padded row bytes. = (width * bitspixel + 31) / 32 * 4
      size           total size. = widthbytes * height * planes
      csize          size of compressed data
      bitspixel      should be 24
      planes         should be 1

   It's not clear from the Photoshop spec what all these members
   do - ie. why store widthBytes or 'size', if they can be trivially
   calculated from other members ? ImgSource provides them all
   simply for completeness.

   Param                Use
   ------------------------------------
   hSource              opened source manager object

   pThumbnailInfo       receives information about the 
                        thumbnail image. see Note.

   uFlags               unused

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.

---------------------------------------------------------*/
_ISDeclSpec HGLOBAL  	_ISCConv _ISFn( is5_GetPSDThumbnail )(HISSRC hSource, ISPSDThumbnailInfo* pThumbnailInfo, UINT32 uFlags);

/*---------------------------------------------------------
   is5_SetPSDOutputResourceBlock / _is5_SetPSDOutputResourceBlock

   Set a data block to be written as the resource section of the 
   next PSD (Photoshop) file. This will be written to all subsequent PSD
   files. To clear the resource block, pass a NULL pointer.

   This has been tested with PhotoShop v6.0 and Photoshop CS.

   Param                Use
   ------------------------------------
   pData                IPTC data block. 

   uDataCount           Size of the IPTC data block.

---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SetPSDOutputResourceBlock )(BYTE *pData, UINT32 uDataCount);

/*---------------------------------------------------------
   is5_GetICCProfile / _is5_GetICCProfile

   Purpose :
   Retrieve the ICC profile from the file. 

   Currently works with JPG, TIFF and PNG. 

   Param                Use
   ------------------------------------
   hSource              opened source manager object
   uTIFFFrameIdx        TIFF trame index, if TIFF file
   phICC                receives HGLOBAL to the ICC profile data

   puICCLen             receives max length of ICC data. 
                        actual data length may be smaller.
                        according to the specs, the first 4 bytes of this
                        block should contain the actual data length.

   uFlags               unused 

   Return
   ------
   FALSE on failure

   caller must use GlobalFree to free the ICC profile memory

---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_GetICCProfile )(HISSRC hSource, UINT32 uTIFFFrameIdx, HGLOBAL *phICC, UINT32 *puICCLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_SetOutputICCProfile / _is5_SetOutputICCProfile

   Purpose :
   Set the global ImgSource output ICC profile. This profile
   will be written to all JPG, TIFF or PNG files created with
   ImgSource. 

   To clear the profile, and disable profile writing, call this
   with a NULL pointer and a size of zero.

   Param                Use
   ------------------------------------
   pProfile             ICC profile data
   uICCLen              length of the data
   uFlags               unused 

   Return
   ------
   FALSE on failure

   caller must use GlobalFree to free the ICC profile memory

---------------------------------------------------------*/
_ISDeclSpec BOOL   	_ISCConv _ISFn( is5_SetOutputICCProfile )(BYTE *pProfile, UINT32 uICCLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ClearIcons / _is5_ClearIcons

   Purpose :
   Clear one of ImgSource's internal icon frame buffers (either
   input or output).

   Param                Use
   ------------------------------------
   bClearOutputIcons    TRUE = clear output icons
                        FALSE = clear input icons
   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_ClearIcons )(BOOL bClearOutputIcons);

#ifdef __cplusplus
}
#endif

#endif