/*---------------------------------------------------------
   ISFilesW.h : part of the ImgSource library.
   
   Whole image file output

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is5_FILESWH
#define is5_FILESWH

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_WriteBMP / _is5_WriteBMP

   Purpose :                                               
   Write 1, 4, 8, 16, 24  or 32 bits per pixel BMP file.

   Note:
   The format of your input image data depends on the 
   value you use for uBitsPerPixel. 

   * If you set uBitsPerPixel to 32, pImg must be 24-bit RGBx. 
     The BMP specification says that the 4th channel is "unused".

   * If you set uBitsPerPixel to 24, pImg must be 24-bit RGB. 

   * If 16, pImg must be in the "x555" format (see is5_RGB24ToRGB16)

   * If 8, pImg must be 8-bit colormapped and pPal must contain 
     a valid palette. 
   
   * If 4, pImg must be four bits per pixel with input 
     rows ((w + 1) / 2) bytes wide. 
   
   * If 1, the image data must be 1 bit per pixel, a row must 
     be ((w + 7) / 8) bytes wide.

   
   In all cases, the pixel rows must start on whole byte boundaries.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image. 

   uWidth            image width (in pixels)
   iHeight           image height (in pixels). may be 
                     negative for top-down BMPs

   uBitsPerPixel     1, 4, 8, 16, 24 or 32
   uInRowStride      number of bytes per pixel row
   uColors           colors in pPal. 0, if uBitsPerPixel
                     is 16, 24 or 32

   pPal              array of uColors RGBQUAD entries. the 
                     image palette. NULL, if uBitsPerPixel
                     is 16, 24 or 32.

   pDPIStruct        optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density (dots / meter)
                     fDPIY      Y pixel density (dots / meter)
                     uDPIUnits: ignored (only dots / meter allowed)

   uFlags            bit         purpose
                     ---         -------
                      0          if set, compress with RLE (8-bit only)
                      1          input image is DIB
                      2          input image is in BGR order (24 and 32 bpp only)
                      3          input image is flipped vertically

   Return
   ------
   size of the file after the writing finished

---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WriteBMP )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, __int32 iHeight, UINT32 uBitsPerPixel, UINT32 uInRowStride, UINT32 uColors, RGBQUAD * pPal, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteJPG / _is5_WriteJPG

   Purpose : 
   Write a JPG file.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image data
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uBitsPerPixel     8 - input image is 8 bit grayscale
                     24 - input image is 24-bit RGB
                     32 - input is 32-bit CMYK or YCCK  

   uInRowStride      number of bytes per pixel row

   uQuality          1 to 100 . quality is the inverse of compression.

   pDPIStruct        optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density
                     fDPIY      Y pixel density

                     uDPIUnits   0 - no units specified in file
                                 1 - dots / inch
                                 2 - dots / cm

   uFlags            bit         purpose
                     ---         -------
                      0          write a "progressive" JPG

                      1          if set, write 32-bit as YCCK, else 
                                 write 32-bit as CMYK.

                      2          input image is flipped vertically
                      3          input pixels are in BGR order

                      4          if set, input image is 24-bit YUV/YCbCr

                      5          if set, do not write JFIF (APP0) header



   Note: 
   DPI info is not saved for CMYK JPGs.

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WriteJPG )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uRowStride, UINT32 uQuality, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteTIFF / _is5_WriteTIFF

   Purpose : 
   Write a TIFF file.

   If uBitsPerPixel = 64, input image must be 64-bit RGBA or CMYK (4x16-bit components).
   If uBitsPerPixel = 48, input image must be 48-bit RGB (3x16-bit components).
   If uBitsPerPixel = 32, input image must be 32-bit RGBA or CMYK.
   If uBitsPerPixel = 24, input image must be 24-bit RGB.
   If uBitsPerPixel = 16, input image must be 16-bit grayscale (one 16-bit value per pixel)
   If uBitsPerPixel = 8, input image must be 8-bit.
   If uBitsPerPixel = 4, input image must be 4-bit, see notes below.
   If uBitsPerPixel = 1, input image must be 1-bit, see notes below.

   Note:
   For 1 bit images:
      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

      These rows must be (int)((uWidth + 7) / 8) bytes wide.

   For 4 bit images:
      Pixels must be packed 2 to a BYTE.

      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

      These rows must be (int)((uWidth + 1) / 2) bytes wide.

   Note:
      1-bit images do not use the palette. They must be arranged
      so that 0-pixels are white and 1-pixels are black (use
      uFlags to reverse this, see below).
      
	Note:
		For G2 (Huffman)/G3/G4 fax-compressed images, zero pixels
		are _always_ assumed to be white (1 = black). This is 
		regardless of what you set for uFlags bit #5.

   See is5_SetTIFFOutputStripSize to control strip size.
   
   See is5_SetTIFFOrientation to control orientation.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uBitsPerPixel     1 = write 1-bit image. see notes.
                     4 = write 4-bit image, pPal must be valid
                     8 = write 8-bit image, pPal must be valid
                     16 = write 16-bit grayscale image. 
                     24 = RGB image
                     32 = RGBA image
							48 = RGB with 16 bits per component
							64 = RGBA with 16 bits per component
   
   uInRowStride      number of bytes per pixel row

   pPal              array of 256 RGBQUAD entries. the 
                     image palette.
   
   uCompMode         1 : no compression

                     2 : CCITT modified Huffman RLE (1-bit only. see notes)
                     3 : CCITT Group 3 fax encoding (1-bit only. see notes)
                     4 : CCITT Group 4 fax encoding (1-bit only. see notes)
                     5 : LZW
                     32771 : CCITT RLE / word aligned (1-bit only)

                     7 : JPEG. 24 or 8-bit only. if 8-bit, the image
                     must be 8-bit grayscale. See is5_SetTIFFJPGQuality.

                     8 : Deflate / ZIP (as used by Adobe)
                     
                     32773 : Macintosh PackBits

                     32946 :  Deflate / ZIP (same algorithm as 8, 
                     but with different compression ID)

                     32909 : Pixar companded 11-bit ZIP (8,24,32 bpp only)

   pDPIStruct        optional resolution info: (pass NULL if you don't care)

                     fDPIX      X pixel density
                     fDPIY      Y pixel density

                     uDPIUnits   1 - no units specified in file
                                 2 - dots / inch
                                 3 - dots / cm

   uFlags            bit         purpose
                     ---         -------
                      0          write image in append mode.
                                 hDest must have been created with 
                                 the "append" flag, and the file must 
                                 already exist.

                      1          write 4 and 8-bit image as grayscale (no palette).

                      2          input image is flipped vertically
                      3          input image is in BGR order (24, 32, 48, 64 bpp only)

                      4          write 32/64-bit images as CMYK instead of RGBA

                      5          write 1-bit images where 0 pixels are black.
                                 default is 0=white. not applicable with G2/G3/G4
											compression.

                      6          enable TIFF predictor (only for LZW and Deflate 
                                 compression schemes, > 4 bits per pixel )

                      7          write tiled TIFF. see is5_SetTIFFOutputTileSize.

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WriteTIFF )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uInRowStride, RGBQUAD *pPal, UINT32 uCompMode, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteTiffMultiViaCB / _is5_WriteTiffMultiViaCB

   Purpose : 
   Write a multi-paged TIFF file. The input images 
   can be 1, 4, 8, 16, 24, 32, 48 or 64 bits per pixel. 

   This function can be used when writing multi-page TIFFs
   where the pages are too large to store all in-memory at
   once (as is required with is5_WriteMultiTIFF). Instead of
   requiring you to provide all the pages at once, this function
   will call a function that you provide to get each page in
   turn. This way, you only need to store a single page at a time.

   When this function needs a page, it will call your callback 
   function. Return TRUE if you have more pages left, or FALSE 
   to end. 

   The callback function must follow this form:

   BOOL CALLBACK TiffMultiCB(BYTE **pImage, 
                              UINT32 *nWidth, 
                              UINT32 *nHeight, 
                              RGBQUAD **pPal, 
                              UINT32 *uBitsPerPixel, 
                              UINT32 *uCompression, 
                              IS4DPIStruct *pDPI, 
                              IS_CALLBACK_DATA uData)
   {
      ...prepare the image...

      ... now give the image information to the caller
      *pImage = myImageData;
      *nWidth = image Width;
      *nHeight = image Height;
      *pPal = image palette;
      *uBitDepth = image bit depth;
      *uCompression = compression type;
      pDPI = pointer to a DPI struct;
      
      return bMorePagesLeft;
   }

   See is5_WriteMultiTIFF for bit-depth and compression rules.


   Param             Use
   ------------------------------------
   hDest             valid destination object
   pageCallback      pointer to your callback function

   uUserData         caller-defined data value

   uFlags            (See is5_WriteTIFF for values)

   Return
   ------
      size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32  	_ISCConv _ISFn( is5_WriteTiffMultiViaCB )(HISDEST hDest, is5_TIFFMULTIPAGE_CBPTR pageCallback, IS_CALLBACK_DATA uUserData, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WritePCX / _is5_WritePCX

   Purpose : 
   Write a 1, 8, or 24-bit PCX file.

   Note:
   For 1 bit images:
      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uBitsPerPixel     1 = write 1 bit PCX, pPal must be valid
                     8 = write 8-bit image, pPal must be valid
                     24 = RGB image

   uRowStride        number of bytes per pixel row

   pPal              array of 256 RGBQUAD entries. the 
                     image palette.

   pDPIStruct        optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      dots per inch, horizontal
                     fDPIY      dots per inch, vertical

                     uDPIUnits   unused

   uFlags            bit         purpose
                     ---         -------
                      0          if set, set the 'grayscale' flag in the file header
                                 this applies to 1 and 8 bit images only

                      2          input image is flipped vertically
                      3          input image is in BGR order (24 bpp only)

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WritePCX )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uRowStride, RGBQUAD *pPal, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteTGA / _is5_WriteTGA

   Purpose : 
   Write an 8-bit, 24-bit or 32-bit TGA file.

   If uBitsPerPixel = 32, input image must be 32-bit RGBA.
   If uBitsPerPixel = 24, input image must be 24-bit RGB.
   If uBitsPerPixel = 8, input image must be 8-bit.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uBitsPerPixel     8 = write 8-bit image, pPal must be valid
                     24 = RGB image
                     32 = RGBA image

   uRowStride        number of bytes per pixel row

   pPal              array of 256 RGBQUAD entries. the 
                     image palette - for 8-bit only

   pDPIStruct        unused

   uFlags            bit         purpose
                     ---         -------
                      3           input image is in BGR order (24 bpp only)

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WriteTGA )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uInRowStride, RGBQUAD *pPal, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WritePNG / _is5_WritePNG

   Purpose : 
   Write a PNG file.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image data
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uRowStride        number of bytes per pixel row

   uComponentBitDepth      
                     bits per component:
                     1, for 1 bpp
                     8, for 8-bit, RGB, RGBA
                     16, for 48 or 64 bits per pixel


   uColors           colors in image. if between 1 and 256,
                     you must supply a valid palette at pPal.
                     if 16 bits or higher, 0 is acceptable here.

   pPal              array of up to 256 RGBQUAD entries. 
                     the image palette.

   uColorType        0 = gray, 2 = RGB, 3 = colormapped,
                     4 = gray-alpha, 6 = RGBA

   fGamma            file gamma value. 0 if don't care.
                     note that this value is usually 1/screen_gamma.
                     ie. it's generally less than 1.
   
   pDPIStruct        optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density
                     fDPIY      Y pixel density

                     uDPIUnits   one of:
                                 0 - no units specified in file
                                 1 - dots / meter

   uFlags            bit         purpose
                     ---         -------
                      0          if set, write image interlaced

                      1          set, if palette contains alpha info.
                                 the PNG writer will use the rgbReserved 
                                 entries of the input palette as alpha values 
                                 (0 = transparent, 255 = opaque).

                      2          input image is flipped vertically
                      3          input image is in BGR order (24 bpp only)

                      4          suppress automatic time-stamping

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WritePNG )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uWidthBytes, UINT32 uComponentBitDepth, UINT32 uColors, RGBQUAD * pPal, UINT32 uColorType, double fGammaVal, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WritePAX / _is5_WritePAX

   Purpose: 
   Write a PAX file.

    Note:
    Use is5_SetPAXOutputInfo to set the PAX ID
    for each output file - before calling this 
    function!

   If uBitsPerPixel = 24, input image must be 24-bit RGB.
   If uBitsPerPixel = 8, input image must be 8-bit.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image data
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uRowStride        number of bytes per pixel row
   pPassword         password for this file
   uEncType          PAX encryption type (1 - 6)

   uBitsPerPixel     8 = write 8-bit image, pPal must be valid
                     24 = RGB image

   pPal              array of 256 RGBQUAD entries. the 
                     image palette.

   uFlags            bit         purpose
                     ---         -------
                      2          input image is flipped vertically
                      3          input image is in BGR order (24 bpp only)

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WritePAX )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const char *pPassword, UINT32 uEncType, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WritePSD / _is5_WritePSD

   Purpose : 
   Write a PSD (Adobe Photoshop) file.

    Note:
   This does not write any layer, mask or effect data.
   When Photoshop reads this file, it will place all 
   image data into the background layer.

   If uBitsPerPixel = 32, input image must be 32-bit CMYK, and
   encoded as 0 = max, 255=min.

   If uBitsPerPixel = 24, input image must be 24-bit RGB.
   If uBitsPerPixel = 8, input image must be 8-bit.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image data
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uBitsPerPixel     8 = write 8-bit image, pPal must be valid
                     24 = RGB image
                     32 = CMYK image. input pixels must be CMYK-encoded (not RGB)

   uRowStride        number of bytes per pixel row

   pPal              array of 256 RGBQUAD entries. the 
                     image palette.

   uFlags            bit         purpose
                     ---         -------
                      2          input image is flipped vertically
                      3          input image is in BGR order (24 bpp only)

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WritePSD )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uRowStride, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteTLA / _is5_WriteTLA

   Purpose : 
   Write a TLA image.

    Notes:
   TLA supports encrypted and un-encrypted varieties. These
   have different header signatures, so it should be possible
   for readers to tell if a file is encrypted or not before
   attempting to decode it. Other than the signature, the 
   encrypted and un-encrypted formats are identical. Encryption
   will have no affect on compression.

   The format of the image data you provide must
   agree with what you use for uBitsPerPixel. ImgSource
   does not verify that you are providing data in the 
   correct format.

   It is legal to write an 8-bit TLA image with 0 colors 
   in the palette. In this situation, the reading application
   should know what the palette is by means outside the scope
   of this library.

   uStripSize and uFilterType will directly affect the
   compression obtained on your pixel data. Just what kind of
   effect they will have is difficult to determine. The standard
   values of uStripSize = 20 and uFilterType = 5 should give
   good results for most applications.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImage            pointer to image data
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uBitsPerPixel:    8:   pPal must contain uPalColors RGBQUADs
                          image data is 1 byte per pixel.

                     16:   pPal is ignored
                           image data is 16 bits per pixel 
                           A-RRRRR-GGGGG-BBBBB

                     24:   pPal is ignored
                           image data is 24-bit RGB

                     32:   pPal is ignored
                           image data is 32-bit RGBA

   uRowStride        number of bytes per pixel row

   uStripSize        pixel strip size. this may affect
                     the compression ratio. 20 is a good choice.

   uFilterType       set compression pre-filter.
                     0 = no filter
                     1 = horizontal subtractive filter
                     2 = vertical subtractive filter
                     4 = Paeth filter

                     these may be OR'd together. 5 is a good choice.

   uPalColors        number of colors in palette.

   pPal              palette of uPalColors RGBQUADs
   
   pPassword         encryption password. if this is an 
                     empty string (not NULL!), the file 
                     will be written as an un-encrypted TLA.

   pDataArray        array of pointers to tag data

   dataSizeArray     array of UNIT32s that provide the 
                     lengths of each element in pDataArray

   uTags             number of elements in pDataArray and
                     dataSizeArray.

   uFlags            bit         purpose
                     ---         -------
                      2          input image is flipped vertically
                      3          input image is in BGR order (24 bpp only)

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WriteTLA )(HISDEST hDest, BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uRowStride, UINT32 uStripSize, UINT32 uFilterType, UINT32 uPalColors, RGBQUAD * pPal, const char *pPassword, BYTE **pDataArray, UINT32 *dataSizeArray, UINT32 uTags, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteRGBToMetafile / _is5_WriteRGBToMetafile

   Purpose : 
   Write a Metafile from an RGB image.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pRGB              pointer to RGB image
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uRowStride        number of bytes per pixel row

   uType             0 = Windows Metafile
                     1 = Enhanced Windows Metafile
                     2 = Aldus Placeable Metafile

   uFlags            bit         purpose
                     ---         -------
                      2          input image is in BGR order (24 bpp only)
                      3          input image is flipped vertically

   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WriteRGBToMetafile )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uType, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteWBMP / _is5_WriteWBMP

   Purpose : 
   Write a Wireless Bitmap. These are images used on
   cell phones, PDAs, etc.. 

   The input image must be a two-color, 8-bit image.
   All non-zero pixels are written as white. All zero
   pixels are written as black. 

   Because of their intended use, these images are 
   generally very small. Exact dimension limits are 
   impossible to give, since every cell phone or PDA has 
   a different amount of display area available. 
   
   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to 8-bit image
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uRowStride        number of bytes per pixel row

   uFlags            bit         purpose
                     ---         -------
                      2          input image is flipped vertically


   Return
   ------
   size of the file after the writing finished
---------------------------------------------------------*/
_ISDeclSpec UINT32  	_ISCConv _ISFn( is5_WriteWBMP )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteGIF / _is5_WriteGIF

   Purpose : 
   Write a GIF file from an 8 bit image.

   Param                Use
   ------------------------------------
   hDest                valid destination object
   p8Bit                pointer to 8 bit image
   uWidth               image width (in pixels)
   uHeight              image height (in pixels)
   uRowStride           number of bytes per pixel row
   uBackground          background color index
   iTransparentIndex    index of transparent color. -1 if none.
   uBitsPerPixel        1..8. bits per output pixel.

   pPal                 array of 2^uBitsPerPixel RGBQUAD 
                        entries. the image palette.

   uFlags            bit         purpose
                     ---         -------
                      2          input image is flipped vertically
                      4          write interlaced
   Return
   ------
   BYTEs written
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_WriteGIF )(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 iBackground, __int32 iTransparentIdx, UINT32 iBitsPerPixel, RGBQUAD *pPal, UINT32 uFlags);
                                                 
/*---------------------------------------------------------
   is5_InitWriteMultiframedGIF / _is5_InitWriteMultiframedGIF

   Purpose : 
   Initialize a multi-framed GIF object. This object is
   to be used with is5_AddGIFFrame and is5_DestroyMultiGIF.

   A multi-framed GIF can contain any number of frames.
   Each frame can be any bit depth from 1 to 8 bits per pixel.

   The "screen" size control the overall size of the 
   image. It is the size of the screen on which each 
   frame is rendered. All frames must fit inside the screen.
   
   A GIF file contains a global palette which affects 
   all frames which do not have their own "local" palette. 
   Local palettes are not widely supported; we recommend that
   you do not use them.

   When rendering GIF images, the GIF image "screen" is 
   typically filled with the background color (uBackground), and
   the frames are positioned, drawn and removed from the screen 
   individually. 

   Param                Use
   ------------------------------------
   hDest                valid destination object
   uScreenWidth         overall width of the image
   uScreenHeight        overall height of the image
   uBitsPerPixel        bits per pixel (also controls size of global palette)
   pPal                 256-color global palette (affects all frames without local palettes)
   uBackground          background color index
   iTransparent         transparent color index. -1, if none
   uLoopCount           number of times to show the frame sequence. 0 = infinite
   uFlags               unused. must be zero

   Return
   ------
   NULL on failure
---------------------------------------------------------*/
_ISDeclSpec HCGIF  	_ISCConv _ISFn( is5_InitWriteMultiframedGIF )(HISDEST hDest, UINT32 uScreenWidth, UINT32 uScreenHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uBackground, __int32 iTransparent,  UINT32 uLoopCount, UINT32 uFlags);

/*---------------------------------------------------------
   is5_AddGIFFrame / _is5_AddGIFFrame

   Purpose : 
   Add a frame to the output GIF file.

   Each frame is positioned on the output screen individually.

   The previous frame may be erased, replaced, or filled 
   over with the background color before drawing this frame. 
   See uFlags.

   Notes:
   Each frame can be 1 to 8 bits, and you can mix bit depths 
   within a multi-frame GIF. All input images to this function 
   must be 8-bits per pixel; however, if you wish to write 
   frames with less than 8-bits per pixel, you can specify a 
   bit depth of less than 8, and only the low bits in each 
   8-bit pixel will be used.

   A GIF file contains a global palette which affects 
   all frames which do not have their own "local" palette, 
   including a local transparent index. However, local palettes 
   are not widely supported by GIF readers; we strongly recommend 
   that you do not use them.

   Each frame can have a "user input" flag. This signals to the
   rendering program to stop display and wait for user input of 
   some kind before continuing. This is not widely supported.

   Param                Use
   ------------------------------------
   hGIS                 valid HCGIF object. see is5_InitWriteMultiframedGIF
   p8                   pointer to an 8-bit image
   uWidth               frame width
   uHeight              frame height
   uRowStride           number of bytes per pixel row
   uLeftOfs             offset from screen left
   uTopOfs              offset from screen top
   delayTime            delay in 1/100 sec to show this frame
   transIndex           transparent color index, if using local palette
   uBitsPerPixel        bits per pixel of image
   pLocalPal            local frame palette. see Notes.

   uDisposal            controls how the frame is handled before drawing 
                        the next frame. value:
                           0 - perform no action 

                           1 - leave the image on screen, 
                               should be same as '0', basically.

                           2 - replace this image with what was there before

                           3 - replace with the GIF screen background color


   uFlags               bit         use
                        ---         ---------
                         2          input image is flipped vertically
                         4          if 1, write frame interlaced
                         5          if 1, write with the "user input" flag
                         6          if 1, write frame with local palette

  Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_AddGIFFrame )(HCGIF hGIS, BYTE *p8, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 LeftOfs, UINT32 TopOfs, UINT32 delayTime, __int32 transIndex, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uDisposal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_DestroyMultiGIF / _is5_DestroyMultiGIF

   Purpose : 
   Destroy the mutli-frame GIF writing object.

   Param                Use
   ------------------------------------
   hGIS                 valid HCGIF object. see is5_InitWriteMultiframedGIF

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_DestroyMultiGIF )(HCGIF hGIS);

/*---------------------------------------------------------
   is5_AddIcon / _is5_AddIcon

   Purpose : 
   Add an icon to ImgSource's icon output frame buffer.
   All frame are written the output ICO file when you call
   is5_FinishIconWrite.
   
   See is5_ClearIcons
   ex.
      // add three icon frames
      is5_AddIcon(pIcon1);
      is5_AddIcon(pIcon2);
      is5_AddIcon(pIcon3);

      // write those three frames to an icon (.ICO) file
      is5_FinishIconWrite(hDest, 0);

   Note about icon frames:
   Icons are composed of two parts: 
      1. The "XOR" mask. This is a standard DIB that is combined
      with the destination using an XOR operation. 

      2. The "AND" mask. This is a monochrome image (returned in ImgSource
      as a 1-bpp DIB) that is used to controls transparency. Only draw pixels
      from the XOR image where the corresponding AND image pixel is a 1.

   The pICS ISIconStruct must be properly filled-out, including
   a DIB for the icon XOR mask (the colors) and a monochrome DIB 
   for the AND mask (to control transparency). The bitDepth member
   refers to the XOR image. the width and height members refer to both (since
   both images have the same dimensions). The "index" member is not used
   when writing.

	Note:
	The DIBs for the icons _must_ have been allocated with GlobalAlloc.

   Param                Use
   ------------------------------------
   pICS                 pointer to a ISIconStruct structure

	uFlags					bit		use
								---		---
								0			if set, the image is to be encoded 
											in the .ICO as 32-bit PNG data. the XOR mask
											image must be a 32-bit DIB with a valid Alpha
											channel. the AND mask is ignored.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/

_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_AddIcon )(ISIconStruct* pICS, UINT32 uFlags);

/*---------------------------------------------------------
   is5_FinishIconWrite / _is5_FinishIconWrite

   Purpose : 
   Write all frames added via is5_AddIcon to the output file.
   
   Param                Use
   ------------------------------------
   hDest                valid destination object
   uFlags               unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_FinishIconWrite )(HISDEST hDest, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WritePNM / _is5_WritePNM

   Purpose : 
   Write a PPM, PGM or PBM file

   PPM : Portable Pixel Map . can contain 24 or 48 bit image data
   PGM : Portable Gray Map . can contain 8 or 16 bit image data
   PBM : Portable Bit Map . can contain 1 bit per pixel image data

   Note:
   uBitsPerPixel determines which type of file is written
   * If 1, this writes a PBM file. the image data must 
     be 1 bit per pixel.

   * If 8 or 16, this writes a PGM file. the image data must 
     be 8 or 16 bits per pixel, grayscale.

   * If 24 or 48, this writes a PPM file. the image data must 
     be 8 or 16 bits per pixel, R,G,B.

   If uBitsPerPixel = 8 or 16 and uMaxVal > 255, image data must be 16 bits per pixel.
   If uBitsPerPixel = 24 or 48 and uMaxVal > 255, image data must be 48 bits per pixel.

   Param                Use
   ------------------------------------
   hDest                valid destination object
   pImg                 pointer to image data
   uWidth               image width (in pixels)
   uHeight              image height (in pixels)
   uBitsPerPixel        number of bits in image (1, 8, 24, 16 or 48)
   
   uRowStride           for 1, 8 or 24 bpp, number of bytes per pixel row
                        for 16, 48 bpp, the number of pixel components per row

   uMaxVal              maximum pixel value in image (1..65535)

   uFlags            bit         purpose
                     ---         -------
                      2          input image is flipped vertically
                      3          input image is in BGR order (24 bpp only)
   Return
   ------
   BYTEs written
---------------------------------------------------------*/
_ISDeclSpec UINT32  	_ISCConv _ISFn( is5_WritePNM )(HISDEST hDest, void *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uRowStride, UINT32 uMaxVal, UINT32 uFlags);
 
#ifdef __cplusplus
}
#endif

#endif
