/*---------------------------------------------------------
   ISFilesWL.h : part of the ImgSource library.
   
   Single lines file output

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is5_FILESWLH
#define is5_FILESWLH

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_InitWriteBMPLines / _is5_InitWriteBMPLines

   Purpose : 
   Set up for single line BMP file writing.
   Use with is5_WriteNextBMPLine and is5_DestroyWriteBMPLines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   * If you set uBitsPerPixel to 24, pImg must be 24-bit RGB. 

   * If 16, pImg must be in the "x555" format (see is5_RGB24ToRGB16)

   * If 8, pImg must be 8-bit colormapped and pPal must contain 
     a valid palette. 
   
   * If 4, pImg must be four bits per pixel with input 
     rows ((w + 1) / 2) bytes wide. 
   
   * If 1, the image data must be 1 bit per pixel, a row must 
     be ((w + 7) / 8) bytes wide.

   Param             Use
   ------------------------------------
   hDest             valid dest object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uBitsPerPixel     1, 4, 8, 16, 24, 32
   uColors           colors in pPal. 0, if uBitsPerPixel
                     is 16 or 24.

   pPal              palette

   pDPIStruct        optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density (dots / meter)
                     fDPIY      Y pixel density (dots / meter)
                     uDPIUnits: ignored (only dots / meter allowed)

   uFlags            bit         purpose
                     ---         -------
                      3          input image is in BGR order (24 and 32 bpp only)

   Return
   ------
   HCBMP - BMP single line write object
---------------------------------------------------------*/
_ISDeclSpec HCBMP     	_ISCConv _ISFn( is5_InitWriteBMPLines )(HISDEST hDest, UINT32 uWidth, __int32 iHeight, UINT32 uBitsPerPixel, UINT32 uColors, RGBQUAD * pPal, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteNextBMPLine / _is5_WriteNextBMPLine

   Purpose : 
   Write a single BMP line

   Param             Use
   ------------------------------------
   hSingle           single line HCBMP object

   pRow              image data. See is5_InitWriteBMPLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_WriteNextBMPLine )(HCBMP hSingle, BYTE *pRow);

/*---------------------------------------------------------
   is5_DestroyWriteBMPLines / _is5_DestroyWriteBMPLines

   Purpose : 
   Destroy the HCBMP object

   Param             Use
   ------------------------------------
   hSingle           single line BMP object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_DestroyWriteBMPLines )(HCBMP hSingle);

/*---------------------------------------------------------
   is5_InitWritePCXLines / _is5_InitWritePCXLines

   Purpose : 
   Set up for single line PCX file writing.
   Use with is5_WriteNextPCXLine and is5_DestroyWritePCXLines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid dest object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uBitsPerPixel     8 or 24
   pPal              256-color palette, if uBitsPerPixel = 8

   uFlags            bit         purpose
                     ---         -------
                      0          if set, set the 'grayscale' flag in the file header
                                 this applies to 1 and 8 bit images only

                      3          input image is in BGR order (24 bpp only)


   Return
   ------
   HCPCX - PCX single line write object
---------------------------------------------------------*/
_ISDeclSpec HCPCX     	_ISCConv _ISFn( is5_InitWritePCXLines )(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD * pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteNextPCXLine / _is5_WriteNextPCXLine

   Purpose : 
   Write a single PCX line

   Param             Use
   ------------------------------------
   hSingle           single line HCPCX object

   pRow              image data. See is5_InitWritePCXLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_WriteNextPCXLine )(HCPCX hSingle, BYTE *pRow);

/*---------------------------------------------------------
   is5_DestroyWritePCXLines / _is5_DestroyWritePCXLines

   Purpose : 
   Destroy the HCPCX object

   Param             Use
   ------------------------------------
   hSingle           single line PCX object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_DestroyWritePCXLines )(HCPCX hSingle);

/*---------------------------------------------------------
   is5_InitWriteTGALines / _is5_InitWriteTGALines

   Purpose : 
   Set up for single line TGA file writing.
   Use with is5_WriteNextTGALine and is5_DestroyWriteTGALines.

   If pPal==NULL, is5_WriteNextTGALine expects RGB lines
   else 8 bit lines

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid dest object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   pPal              palette

   uFlags            bit         purpose
                     ---         -------
                      3          input image is in BGR order (24 bpp only)

   Return
   ------
   HCTGA - TGA single line write object
---------------------------------------------------------*/
_ISDeclSpec HCTGA     	_ISCConv _ISFn( is5_InitWriteTGALines )(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD * pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteNextTGALine / _is5_WriteNextTGALine

   Purpose : 
   Write a single TGA line

   Param             Use
   ------------------------------------
   hSingle           single line HCTGA object

   pRow              image data. See is5_InitWriteTGALines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_WriteNextTGALine )(HCTGA hSingle, BYTE *pRow);

/*---------------------------------------------------------
   is5_DestroyWriteTGALines / _is5_DestroyWriteTGALines

   Purpose : 
   Destroy the HCTGA object

   Param             Use
   ------------------------------------
   hSingle           single line HCTGA object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_DestroyWriteTGALines )(HCTGA hSingle);

/*---------------------------------------------------------
   is5_InitWriteTIFFLines / _is5_InitWriteTIFFLines

   Purpose : 
   Set up for single line TIFF file writing.
   Use with is5_WriteNextTIFFLine and is5_DestroyWriteTIFFLines.

   If uBitsPerPixel = 32, is5_WriteNextTIFFLine expects RGBA lines
   If uBitsPerPixel = 24, is5_WriteNextTIFFLine expects RGB lines
   If uBitsPerPixel = 16, input image must be 16-bit grayscale (one 16-bit value per pixel)
   If uBitsPerPixel = 8, is5_WriteNextTIFFLine expects 8-bit lines
   If uBitsPerPixel = 4, is5_WriteNextTIFFLine expects 4-bit lines, see notes below.
   If uBitsPerPixel = 1, input image must be 1-bit, see notes below.

   Notes:

   For 4 bit images:
      Pixels must be packed 2 to a BYTE.

      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

      These rows must be (int)((uWidth + 1) / 2) bytes wide.

   For 1 bit images:
      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

      These rows must be (int)((uWidth + 7) / 8) bytes wide.

      Zero is white, one is black. 

   Note: Single line input/output functions are not 
   available for unregistered users.

   See is5_SetTIFFOutputStripSize to control strip size.

   See is5_SetTIFFOrientation to control orientation.

   Parameters
   ----------
   See is5_WriteTIFF for parameter descriptions.
 
   Return
   ------
   HISTIFFWRITER - TIFF single line write object

---------------------------------------------------------*/
_ISDeclSpec HISTIFFWRITER     	_ISCConv _ISFn( is5_InitWriteTIFFLines )(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD * pPal, UINT32 uCompMode, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteNextTIFFLine / _is5_WriteNextTIFFLine

   Purpose : 
   Write a single TIFF line. 
   
   Note:
   If you are using the TIFF predictor, this function will 
   modify the image data for this row. You should make a copy of
   the row and pass that to this function.

   Param             Use
   ------------------------------------
   hSingle           single line TIFF object

   pRow              image data. See is5_InitWriteTIFFLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_WriteNextTIFFLine )(HISTIFFWRITER hSingle, BYTE *pRow);

/*---------------------------------------------------------
   is5_DestroyWriteTIFFLines / _is5_DestroyWriteTIFFLines

   Purpose : 
   Destroy the HISTIFFWRITER object

   Param               Use
   ------------------------------------
   hSingle            single line HISTIFFWRITER object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_DestroyWriteTIFFLines )(HISTIFFWRITER hSingle);

/*---------------------------------------------------------
   is5_InitWritePNGLines / _is5_InitWritePNGLines

   Purpose : 
   Set up for single line PNG file writing.
   Use with is5_WriteNextPNGLine and is5_DestroyWritePNGLines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   p8Bit             pointer to image
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uWidthBytes       image row width, in bytes
   uBitDepth         bits per component

   uColors           colors in image. if between 1 and 256,
                     you must supply a valid palette at pPal.
                     if 16 bits or higher, 0 is acceptable here.

   pPal              array of up to 256 RGBQUAD entries. 
                     the image palette.

   uColorType        0 = gray, 2 = RGB, 3 = palettized,
                     4 = gray-alpha, 6 = RGBA

   fGamma            file gamma value. 0 if don't care.
                     note that this value is usually 1/screen_gamma.
                     ie. it's generally less than 1.

    pDPIStruct       optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density
                     fDPIY      Y pixel density

                     uDPIUnits   0 - no units specified in file
                                 1 - dots / meter

   uFlags            bit         purpose
                     ---         -------
                      1          set, if palette contains alpha info.
                                 the PNG writer will use the rgbReserved 
                                 entries of the input palette as alpha values 
                                 (0 = transparent, 255 = opaque).

                      3          input pixels are in BGR order

                      4          suppress automatic time-stamping

   Return
   ------
   BYTEs written
---------------------------------------------------------*/
_ISDeclSpec HCPNG     	_ISCConv _ISFn( is5_InitWritePNGLines )(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uWidthBytes, UINT32 uBitDepth, UINT32 uColors, RGBQUAD * pPal, UINT32 uColorType, double fGammaVal, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteNextPNGLine / _is5_WriteNextPNGLine

   Purpose : 
   Write a single PNG line

   Param               Use
   ------------------------------------
   hSingle            single line PNG object
   pRow               image data. See is5_InitWritePNGLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_WriteNextPNGLine )(HCPNG hSingle, BYTE *pRow);

/*---------------------------------------------------------
   is5_DestroyWritePNGLines / _is5_DestroyWritePNGLines

   Purpose : 
   Destroy the HCPNG object

   Param               Use
   ------------------------------------
   hSingle            single line HCPNG object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_DestroyWritePNGLines )(HCPNG hSingle);

/*---------------------------------------------------------
   is5_InitWriteJPGLines / _is5_InitWriteJPGLines

   Purpose : 
   Set up for single line JPG file writing.
   Use with is5_WriteNextJPGLine and is5_DestroyWriteJPGLines.

   If bGrayscale = 0, is5_WriteNextJPGLine expects RGB lines
   else 8 bit lines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image data
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uQuality          1<->100 . quality is the inverse of
                     compression.

   uBitsPerPixel     8 = input image is 8 bit grayscale
                     24 = input image is 24-bit RGB
                     32 - input is 32-bit CMYK or YCCK

   pDPIStruct        optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density
                     fDPIY      Y pixel density
                     uDPIUnits   0 - no units specified in file
                                 1 - dots / inch
                                 2 - dots / cm

   uFlags            bit         purpose
                     ---         -------
                      0          write "progressive"

                      1          if set, write 32-bit as YCCK, else 
                                 write 32-bit as CMYK.

                      3          input image is in BGR order (24 bpp only)

                      4          if set, input image is 24-bit YUV/YCbCr

                      5          if set, do not write JFIF (APP0) header


   Note: 
   DPI info is not saved for CMYK JPGs.
   
   Return
   ------
   HCJPG - JPG single line write object
---------------------------------------------------------*/
_ISDeclSpec HCJPG     	_ISCConv _ISFn( is5_InitWriteJPGLines )(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uQuality, UINT32 uBitsPerPixel, IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteNextJPGLine / _is5_WriteNextJPGLine

   Purpose : 
   Write a single JPG line

   Param             Use
   ------------------------------------
   hSingle           single line JPG object

   pRow              image data. See is5_InitWriteJPGLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_WriteNextJPGLine )(HCJPG hSingle, BYTE *pRow);

/*---------------------------------------------------------
   is5_DestroyWriteJPGLines / _is5_DestroyWriteJPGLines

   Purpose : 
   Destroy the HCJPG object

   Param             Use
   ------------------------------------
   hSingle           single line HCJPG object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_DestroyWriteJPGLines )(HCJPG hSingle);

/*---------------------------------------------------------
   is5_InitWritePAXLines / _is5_InitWritePAXLines

   Purpose : 
   Set up for single line PAX file writing.
   Use with is5_WriteNextPAXLine and is5_DestroyWritePAXLines.

    Note:
    Use is5_SetPAXOutputInfo to set the PAX ID
    for each output file - before calling this 
    function!

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid dest object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   pPal              palette of 256 RGBQUADs 

   uBitsPerPixel     BYTEs per pixel (1 or 3)
                        8 : format is 8-bit PAX. pPal
                        must point to a palette. lines must 
                        be written with is5_WriteNextPAXLine8Bit.

                        24 : format is 24-bit PAX. pPal
                        may be NULL. lines must be written 
                        with is5_WriteNextPAXLineRGB.

   pPassword         password for this file
   uEncType          PAX encryption type (1 - 6)

   uFlags            bit         purpose
                     ---         -------
                      3          input image is in BGR order (24 bpp only)


   Return
   ------
   HCPAX - PAX single line write object
---------------------------------------------------------*/
_ISDeclSpec HCPAX     	_ISCConv _ISFn( is5_InitWritePAXLines )(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, RGBQUAD * pPal, UINT32 uBitsPerPixel, const char *pPassword, UINT32 uEncType, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteNextPAXLine / _is5_WriteNextPAXLine

   Purpose : 
   Write a single PAX line

   Param             Use
   ------------------------------------
   hSingle           single line HCPAX object

   pRow              image data. See is5_InitWritePAXLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_WriteNextPAXLine )(HCPAX hSingle, BYTE *pRow);

/*---------------------------------------------------------
   is5_DestroyWritePAXLines / _is5_DestroyWritePAXLines

   Purpose : 
   Destroy the HCPAX object

   Param             Use
   ------------------------------------
   hSingle           single line PAX object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_DestroyWritePAXLines )(HCPAX hSingle);

/*---------------------------------------------------------
   is5_InitWriteTLALines / _is5_InitWriteTLALines

   Purpose : 
   Set up for single line TLA file writing.
   Use with is5_WriteNextTLALine and is5_DestroyWriteTLALines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   See comments for is5_WriteTLA for an explanation of
   the parameters.

   Return
   ------
   HCTLA - TLA single line write object
---------------------------------------------------------*/
_ISDeclSpec HCTLA     	_ISCConv _ISFn( is5_InitWriteTLALines )(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, const char *pPassword, UINT32 uBitsPerPixel, UINT32 uStripSize, UINT32 uFilterType, UINT32 uPalColors, RGBQUAD * pPal, BYTE **pDataArray, UINT32 *dataSizeArray, UINT32 uTags, UINT32 uFlags);

/*---------------------------------------------------------
   is5_WriteNextTLALine / _is5_WriteNextTLALine

   Purpose : 
   Write a single TLA line

   Param               Use
   ------------------------------------
   hSingle             single line HCTLA object

   pRow                image data. See comments for 
                       is5_WriteTLA for an explanation 
                       of the image data format.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_WriteNextTLALine )(HCTLA hSingle, BYTE *pRow);

/*---------------------------------------------------------
   is5_DestroyWriteTLALines / _is5_DestroyWriteTLALines

   Purpose : 
   Destroy the HCTLA object

   Param               Use
   ------------------------------------
   hSingle                single line TLA object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_DestroyWriteTLALines )(HCTLA hSingle);

#ifdef __cplusplus
}
#endif

#endif
