/*---------------------------------------------------------
   ISIPTC.h : part of the ImgSource library.

   IPTC data reading code

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..
   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   
   http://www.smalleranimals.com
---------------------------------------------------------*/
#ifndef is5_IPTCH
#define is5_IPTCH

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_IPTCInitialize / _is5_IPTCInitialize

   Purpose : 
   Create an ImgSource IPTC data reading object and fill it 
   from a file. This currently only supports IPTC data from JPG, 
   PSD (Photoshop) and TIFF files.

   IPTC/IIM data is image metadata stored in JPG files
   by various software packages, including Adobe PhotoShop.

   To fetch the individual strings from the data, use is5_IPTCGetTag.

   Also see is5_IPTCInitializeFromByteArray and 
   is5_IPTCCopyTagsFromInputToOutput.

   Note:
   This object must be destroyed with is5_IPTCRelease. Failure 
   to do so will result in memory leaks.

   IPTC = International Press Telecommunications Council 

   Param             Use
   ------------------------------------
   hSrc              open source manager object
   uFrameIdx         TIFF frame idx (if input is a TIFF file)
   uFlags            unused

   Return
   ------
   HISIPTC IPTC reader object. NULL on failure.
---------------------------------------------------------*/
_ISDeclSpec HISIPTC     	_ISCConv _ISFn( is5_IPTCInitialize )(HISSRC hSrc, UINT32 uFrameIdx, UINT32 uFlags);

/*---------------------------------------------------------
   is5_IPTCInitializeFromByteArray / _is5_IPTCInitializeFromByteArray

   Purpose : 
   Create an ImgSource IPTC data reading object and fill it 
   from a byte array. 
   
   IPTC/IIM data is image metadata stored in JPG files
   by various software packages, including Adobe PhotoShop.

   To fetch the individual strings from the data, use is5_IPTCGetTag.
   
   Also see is5_IPTCCopyTagsFromInputToOutput.

   Note:
   This object must be destroyed with is5_IPTCRelease. Failure 
   to do so will result in memory leaks.

   IPTC = International Press Telecommunications Council 

   Param             Use
   ------------------------------------
   pData             pointer to data
   uDataLength       length of data
   uFlags            unused

   Return
   ------
   HISIPTC IPTC reader object. NULL on failure.
---------------------------------------------------------*/
_ISDeclSpec HISIPTC     	_ISCConv _ISFn( is5_IPTCInitializeFromByteArray )(BYTE *pData, UINT32 uDataLength, UINT32 uFlags);

/*---------------------------------------------------------
   is5_IPTCGetTag / _is5_IPTCGetTag

   Purpose : 
   Extract a data string for a single tag from an ImgSource
   IPTC data reader object.

   The data is generally ASCII data - but not always. Beware.

   These are some common tag IDs:
   id     description
   -------------------
   5      Origin Object Name
   15     Category 
   20     Supplemental categories (expect multiples)
   25     Keyword (expect multiples)
   40     Special Instructions
   55     Date string (yyyymmdd)
   80     Byline
   85     Byline title
   90     City
   95     State/Province
   101    Country
   103    Original transmission Reference
   105    Caption headline
   110    Credit
   115    Source
   116    Copyright
   120    Caption
   122    Caption writer
   
   Other IDs are possible.

   Param             Use
   ------------------------------------
   hIPTC             IPTC data reader object created by
                     is5_IPTCInit.

   uSection          data section. generally "2"
   uTagID            tag ID
   uIndex            index, in case the tag has multiple
                     data blocks. zero-based.

   phData            receives a handle to the tag data. 
                     you must use GlobalFree to release this
                     memory when you are finished with it.

   puLen             receives the length of the data
                     
   Return
   ------
   result code.
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_IPTCGetTag )(HISIPTC hIPTC, UINT32 uSection, UINT32 uTagID, UINT32 uIndex, HGLOBAL *phData, UINT32 *puLen);

/*---------------------------------------------------------
   is5_IPTCGetTagList / _is5_IPTCGetTagList

   Purpose : 
   Get a list of all data tags available from this IPTC 
   parser. Tags are listed as "section,id,index" seperated 
   by a 0x13 (new line).

   For example:

      2,110,0
      2,112,0
      2,115,0
      2,115,1
      ...

   Or, if translated:

      Object name, GEORGE BURLEY
      Category, SPT
      Supplemental categories, FOOTBALL
      Keywords, FOOTBALL
      Keywords, LEAGUE
      ...

   For a list of tags and the values associated with them, 
   see is5_IPTCGetTag
   

   Param             Use
   ------------------------------------
   hIPTC             IPTC data reader object created by
                     is5_IPTCInit.

   phData            receives a handle to the tag data. 
                     you must use GlobalFree to release this
                     memory when you are finished with it.

   puLen             receives the length of the data
                     
   uFlags            bit         purpose
                     ---         -------
                      8          return translated tags

   Return
   ------
   result code.
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_IPTCGetTagList )(HISIPTC hIPTC, HGLOBAL *phString, UINT32 *puLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_IPTCRelease / _is5_IPTCRelease

   Purpose : 
   Destory an ImgSource IPTC data reader object. This
   will free all memory associated with the IPTC reader object.

   Param             Use
   ------------------------------------
   hIPTC             IPTC data reader object created by
                     is5_IPTCInit.

    Return
   ------
   result code.
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_IPTCRelease )(HISIPTC hIPTC);

/*---------------------------------------------------------
   is5_IPTCWriteInit / _is5_IPTCWriteInit

   Purpose : 
   Create an ImgSource IPTC export object. This object will
   create a block of memory suitable for use as an IPTC data
   block in a Photoshop resource block, which can then be used
   in a JPG, TIFF or PSD file.

   Use is5_IPTCWriteAddString to add strings, and is5_IPTCExport
   to create the IPTC data block. 
   
   This object must be destroyed with is5_IPTCWriteDestroy.     
       
   Param               Use
   ------------------------------------
   none

   Return
   ------
   HISIPTCW object

---------------------------------------------------------*/
_ISDeclSpec HISIPTCW     	_ISCConv _ISFn( is5_IPTCWriteInit )();

/*---------------------------------------------------------
   is5_IPTCWriteAddTagData / _is5_IPTCWriteAddTagData

   Purpose : 
   Add a data tag to an ImgSource IPTC writer object. 

   While most commonly-used IPTC data is ASCII text, some fields
   are non-textual (numeric data types, structures, etc.).
   
   See is5_IPTCGetTag for a list of common tag IDs. 
   
   Note:
   Writing tags larger than 65535 bytes is not supported in 
   this implementation.

   Note:
   Some IDs allow multiples (Keywords and Categorys, for example). 
   
   Also see is5_IPTCCopyTagsFromInputToOutput

   Param             Use
   ------------------------------------
   hIPTC             valid IPTC writer object, created with is5_IPTCWriteInit
   uSection          generally "2"
   uID               tag ID (see is5_IPTCGetTag)
   pData             data to write
   uDataLen          length of data to write

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_IPTCWriteAddTagData )(HISIPTCW hIPTC, UINT32 uSection, UINT32 uID, BYTE *pData, UINT32 uDataLen);

/*---------------------------------------------------------
   is5_IPTCExport / _is5_IPTCExport

   Purpose : 
   Generate an IPTC data block based on the strings added via
   is5_IPTCWriteAddTagData. This can be used as input to
   is5_AddPSDResource to create a Photoshop/IPTC resource 
   to a TIFF, JPG or PSD file.

   Note:
   When creating a resource block for is5_AddPSDResource, 
   all headers should be supressed (see uFlags).

   Param             Use
   ------------------------------------
   hIPTC             valid IPTC writer object, created with is5_IPTCWriteInit
   phOut             receives a handle to the allocated memory
   puSize            receives the size of the allocated memory block

   uFlags            bit      purpose
                     ---      -------
                      0       suppress "Photoshop 3.0" header
                      1       suppress "8BIM" header
                      2       suppress length header
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_IPTCExport )(HISIPTCW hIPTC, HGLOBAL *hOut, UINT32 *uSize, UINT32 uFlags);

/*---------------------------------------------------------
   is5_IPTCWriteDestroy / _is5_IPTCWriteDestroy

   Purpose : 
   Destroy the IPTC writer object.

   Param             Use
   ------------------------------------   
   valid IPTC writer object, created with is5_IPTCWriteInit.
   

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_IPTCWriteDestroy )(HISIPTCW hIPTC);

/*---------------------------------------------------------
   is5_IPTCCopyTagsFromInputToOutput / _is5_IPTCCopyTagsFromInputToOutput

   Purpose : 
   Copy all the IPTC strings from your IPTC parsing object 
   to an IPTC writer object. This removes any strings already 
   in the writer object.

   This is a quick way to copy the IPTC from one file to another.
   
   Param             Use
   ------------------------------------
   hIPTCW            valid IPTC writer object, created 
                     with is5_IPTCWriteInit

   hIPTCR            valid IPTC data reader object created 
                     with is5_IPTCInitialize

   uFlags            unused

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_IPTCCopyTagsFromInputToOutput )(HISIPTCW hIPTCW, HISIPTC hIPTCR, UINT32 uFlags);

#ifdef __cplusplus
}
#endif
#endif