/*---------------------------------------------------------
   ISImageEnhance.h : part of the ImgSource library.
   
   Image processing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is5_IMAGEHISTOH
#define is5_IMAGEHISTOH

// shut up, stupid compiler!
#pragma warning(disable:4786)

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif


/*---------------------------------------------------------
   is5_AdjustGamma / _is5_AdjustGamma

   Purpose : 
   Perform gamma adjustment on an image.

   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, etc.) 
   
   DO NOT use this on images that require a palette 
   unless the palette is arranged so that the visual 
   distance between two colors is directly proportional 
   to the distance between their color indecies, for all
   combinations of colors in the palette.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of BYTEs in a pixel.
   uRowStride        number of bytes in an image row
   fGamma            gamma correction factor
                
                     gamma must be >= 0.0
                     gamma < 1 , decrease brightness
                     gamma = 1 , no change
                     gamma > 1 , increase brightness

   uFlags            bit      meaning
                     ---      -------
                     0        if set, the fourth channel in 
                              a 32 bit image is modified. it is 
                              ignored. by default

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_AdjustGamma )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is5_AdjustBrightnessContrast / _is5_AdjustBrightnessContrast

   Purpose : 
   Modify the brightness and/or contrast of an image.

   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, etc.) 
   
   DO NOT use this on images that require a palette 
   unless the palette is arranged so that the visual 
   distance between two colors is directly proportional 
   to the distance between their color indecies, for all
   combinations of colors in the palette.

   Curve shapes:
   The uFlags parameter can select between a choice of 
   curve shapes. The default is a five point curve; this 
   is the traditional ImgSource adjustment. 
   
   The seven point curve forces pixels in the extreme dark 
   and light ranges to black or white faster than the five 
   point curve.

   The three point curve emphasizes mid-tones more than the
   five or seven point curves. This is closest to the 
   brightness / contrast that Photoshop uses.

   The smooth curve gives much nicer results: fewer blown-out
	pixels on the bright end and much less solid black.

	Applying the adjustment to the Luminance channel only (bit 8)
	reduces the chromatic shifts you can sometimes see if the
	adjustment is applied to the RGB channels separately.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel in the image (1 to 4)
   uRowStride        byte per pixel row
   fBrightness       brightness adjustment (-1..+1)
   fContrast         contrast adjustment (-1..+1)

   uFlags            bit      meaning
                     ---      -------
                     0        ignore 4th (alpha) channel
                     1        use 3 point curve
                     2        use 7 point curve

							8			apply changes to Luminance only.
										alpha channel is ignored.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_AdjustBrightnessContrast )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fBrightness, double fContrast, UINT32 uFlags);

/*---------------------------------------------------------
   is5_AdjustBrightnessContrast2 / _is5_AdjustBrightnessContrast2

   Purpose : 
   Modify the brightness and/or contrast of an image.

   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, etc.) 
   
   DO NOT use this on images that require a palette 
   unless the palette is arranged so that the visual 
   distance between two colors is directly proportional 
   to the distance between their color indecies, for all
   combinations of colors in the palette.

   This differs from is5_AdjustBrightnessContrast in that
	the contrast and brightness transfer function is a smooth
	curve instead of a series of series of line segments. Also,
	when calculating contrast, the endpoints of the curve remain
	anchored at 0 and 255 - this means that white pixels stay 
	white and black pixels stay black when decreasing 
	contrast (changing brightness may move these endpoints 
	points, however).

	The constrast curve is sigmoidal in shape: curved on 
	the ends and relatively straight in the middle. The 
	fContrastPower parameter controls the size of the middle 
	section. Lower values lengthen the middle section (by bringing
	the bends closer to the corners), larger values shorten the
	middle.

	Applying the adjustment to the Luminance channel only (bit 8)
	reduces the chromatic shifts you can sometimes see if the
	adjustment is applied to the RGB channels separately.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uRowStride        byte per pixel row
   fBrightness       brightness adjustment (-1..+1)
   fContrast         contrast adjustment (-1..+1)
	
	fContrastPower		controls the shape of the 
							contrast curve. 
							(0.1..0.9)
							.5 is a good default
							

   uFlags            bit      meaning
                     ---      -------
                     0        ignore 4th (alpha) channel

							8			apply changes to Luminance only.
										alpha channel is ignored.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_AdjustBrightnessContrast2 )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fBrightness, double fContrast, double fCurvePower, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ModifyImageSaturation / _is5_ModifyImageSaturation

   Purpose : 
   Adjust the saturation levels in an RGB or RGBA image.
   Will also work on BGR/BGRA.

   Param             Use
   ------------------------------------
   pInput            input image (24 or 32 bits)
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    3 or 4. if 4, the alpha channel is not modified.
   uRoStride         bytes per pixel row
   fSatLevel         saturation level (any value is valid)

                     guide:
                     1 = no change
                     0 = fully desaturated
                     < 0, image starts to resaturate, but photonegative
                     > 1, image gains saturation
                     
                     there are no limits on this value, though the useful 
                     results start to taper off around +/- 10.
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_ModifyImageSaturation )(BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double S);


/*---------------------------------------------------------
   is5_ApplyLUTsToImage / _is5_ApplyLUTsToImage

   Purpose : 
   Apply LUTs (look-up-tables) to pixels in an image.

   For each channel in each pixel the following function is
   performed :
      
      v = pImage[pixel][channel]
      pImage[pixel][channel] = LUT[v][channel]

   Ex.
   BYTE LUT[256];
   BYTE *LUTArray[1];
   LUTArray[0] = LUT;
   is5_ApplyLUTsToImage(pImage, w, h, 1, w, LUTArray, 0);

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   
   uChannelsPerPixel number of channels per pixel.
                     this also determines the number of LUTs 
                     in the pLUTs array.

   uRowStride        number of pixel components in an image row

   pLUTs             if image is 8 bits per component, this is an 
                     array of arrays of 256 BYTE values.

                     if image is 16 bits per component, this is an 
                     array of arrays of 65536 ISUINT16 values.

                     pLUTs[0] is the LUT used for the first channel,
                     pLUTs[1] is the LUT used for the second channel,
                     etc..
                     if you don't want to apply a LUT to a particular 
                     channel, put a NULL in the corresponding 
                     pLUTs place. 

                     pLUTs must contain uChannelsPerPixel elements!

   uFlags            bit      meaning
                     ---      -------
                      9       image is 16 bits per channel
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_ApplyLUTsToImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, BYTE **pLUTs, UINT32 uFlags);

// experimental
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_ApplyBrightnessLUT )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, BYTE *pLUT);

/*---------------------------------------------------------
    is5_AutoBrightnessRGB / _is5_AutoBrightnessRGB

    Purpose : 
    Automagically modify the brightness of an RGB image.

    Param                   Use
    ------------------------------------
    pRGB                    source RGB image
    uWidth                  width in pixels
    uHeight                 height
    uRowStride              number of bytes in an image row
    uLowThreshold           pixels below this level are set to black
    uHighThreshold          pixels above this level are set to white

    uMode                   0 : stretch brightness values to fit 
                            the specified range (histogram stretch)

                            1 : evenly distribute brightness values
                            across the specified range (histogram
                            equalization)
    Return
    ------
    FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_AutoBrightnessRGB )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uLowThreshold, UINT32 uHighThreshold, UINT32 uMode);

/*---------------------------------------------------------
   is5_GetBrightnessHistogram / _is5_GetBrightnessHistogram

   Purpose : 
   Find the brightness histogram of an image. Should not
   be used on images larger than 2 ^ 32 pixels.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel 3 = RGB (or BGR), 4 = RGBA (or BGRA)
   uRowStride        number of pixel components in a pixel row
   pHisto            array of 255 UINT32s (8 bpc) or 65536 UINT32s (16 bpc)

   uFlags            bit      meaning
                     ---      -------
                     0        set when image is in BGR (or BGRA) order
                     1        set when image is RGBA or BGRA
                     9        images have 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      	_ISCConv _ISFn( is5_GetBrightnessHistogram )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 *pHisto, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GetImageChannelHistograms / _is5_GetImageChannelHistograms

   Purpose : 
   Find the histograms for each channel in an image.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   
   uChannelsPerPixel number of channels in image. 
                     also number of images in pHisto.
   
   uRowStride        number of pixel components in a pixel row


   pHisto            array of pointers to arrays of 255 UINT32s each (8bpc) or 65535 UINT32s (16bpc).
                     the histogram for the first channel goes into pHisto[0]
                     the histogram for the second channel goes into pHisto[1]
                     etc..

                     if you don't want the histogram for a given channel, put a NULL
                     in the appropriate spot in pHisto

   uFlags            bit      meaning
                     ---      -------
                     9        images have 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetImageChannelHistograms )(BYTE *pImage, UINT32 nWidth, UINT32 nHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, UINT32 **pHisto, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GetImageChannelHistogramsDouble / _is5_GetImageChannelHistogramsDouble

   Purpose : 
   Find the histograms for each channel in an image made of doubles.

   Notes:

   Mode 1:
   
        1. find the minimum and maximum value used in the image for each pixel channel, 

        2. for each pixel in a channel, perform the following calculation:

                     pixel - min
                 v = -----------  x (uHistoSize - 1)
                      min - max

        3. increase pHisto[channel][v]

        In other words, the range of pixel values in the image is sub-divided 
        into uHistoSize buckets.

        On exit, pfMinVal and pfMaxVal are set to the min and max pixel values, respectively.

   Mode 2:

        1. for each pixel in a channel, apply the following calculation:

                clip the pixel value to the range fMinVal..fMaxVal .

           Where pfMinVal and pfMaxVal are passed into the function by the caller.

        2. perform the following calculation:

                       pixel - pfMinVal
                 v = --------------------  x (uHistoSize - 1)
                      pfMinVal - pfMaxVal


        3. increase pHisto[channel][v]

        In other words, the range of pixel values supplied by the caller 
        is sub-divided into uHistoSize buckets. 

        Note that if you set fMinVal and fMaxVal to the actual min/max pixel value 
        of the input image, this mode is also identical to Mode 1.

   Mode 3:

        1. for each pixel in a channel where the pixel value is 
        in the range fMinVal..fMaxVal, apply the following calculation:

                       pixel - pfMinVal
                 v = --------------------  x (uHistoSize - 1)
                      pfMinVal - pfMaxVal

        Where pfMinVal and pfMaxVal are passed into the function by the caller.

        2. increase pHisto[channel][v]

        In other words, the range of pixel values supplied by the caller 
        is sub-divided into uHistoSize buckets, and any pixel outside that range
        is not counted in the calculation. 
        
        Note that if you set fMinVal and fMaxVal to the actual min/max pixel value 
        of the input image, this mode is also identical to Mode 1.

   Param             Use
   ------------------------------------
   pImage            source image, doubles
   uWidth            width in pixels
   uHeight           height
   
   uChannelsPerPixel number of channels in image. 
   
   uRowStride        number of pixel components in a pixel row

   uHistoSize		 number of entries in the histogram

   pHisto            array of uChannelsPerPixel pointers to arrays of uHistoSize UINT32s.
                     the histogram for the first channel goes into pHisto[0]
                     the histogram for the second channel goes into pHisto[1]
                     etc..

                     if you don't want the histogram for a given channel, put a NULL
                     in the appropriate spot in pHisto

   pfMinVal          array of uHistoSize doubles. 
   
                     in mode 1, these receive the mininum pixel value for each channel.
                     in modes 2,3, these specify the low end of the range for each channel

   pfMaxVal          array of uHistoSize doubles. 
   
                     in mode 1, these receive the maximum pixel value for each channel.
                     in modes 2,3, these specify the high end of the range for each channel
  
   uFlags            bits        purpose
                     ---         -------
                     0,1         if both are 0, process in Mode 1 (uFlags = 0)
                                 if only bit 0 is set, process in Mode 2 (uFlags = 1)
                                 if both are set, process in Mode 3 (uFlags = 3)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetImageChannelHistogramsDouble )(double *pImage, UINT32 nWidth, UINT32 nHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, UINT32 uHistoSize, UINT32 **pHisto, double *pfMinVal, double *pfMaxVal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GetOneBitHistogram / _is5_GetOneBitHistogram

   Purpose : 
   Find the histogram for a one bit image.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   
   uRowStride        number of bytes in a pixel row

   pHisto            array of 2 UINT32
                     the number of 0 bits goes into pHisto[0]
                     the number of 1 bits goes into pHisto[1]

   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetOneBitHistogram )(BYTE *pImage, UINT32 nWidth, UINT32 nHeight, UINT32 uRowStride, UINT32 *pHisto, UINT32 uFlags) ;

/*---------------------------------------------------------
   is5_BrightnessHistogramStretchImage / _is5_BrightnessHistogramStretchImage

   Purpose : 
   Perform histogram stretch on an RGB or RGBA image
   The alpha channel is ignored. Also see 
   is5_BrightnessHistogramStretchImageInHSL.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    3 = RGB (or BGR), 4 = RGBA (or BGRA)
   uRowStride        number of bytes in a pixel row
   fLowLimit         percent of pixels to set to 0
   fHiLimit          percent of pixels to set to 255 
   uMidPoint         pixel value to set to middle intensity (1..254)

   uFlags            bit         purpose
                     ---         -------
                      8          if set, use mid-point to divide
                                 output range: 
                                 half the intensity values will 
                                 be scaled to 0...mid-point ,
                                 and the other half will be scaled
                                 mid-point...maxval .

                      9          do stretch in HSL colorspace. this
                                 will preserve colors better than 
                                 doing the stretch in the RGB space.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      	_ISCConv _ISFn( is5_BrightnessHistogramStretchImage )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fLowLimit, double fHighLimit, UINT32 uMidPoint, UINT32 uFlags);

/*---------------------------------------------------------
   is5_SaturationHistogramStretchImage / _is5_SaturationHistogramStretchImage

   Purpose : 
   Perform a histogram stretch on the saturation values in
   an RGB or RGBA image (using an HSV conversion).

   The alpha channel is ignored. 

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    3 = RGB (or BGR), 4 = RGBA (or BGRA)
   uRowStride        number of bytes in a pixel row
   fLowLimit         percent of pixels to set to 0
   fHiLimit          percent of pixels to set to 255
   uMidPoint         pixel value to set to middle intensity (1..254)

   uFlags            bit         purpose
                     ---         -------
                      8          if set, use mid-point to divide
                                 output range: 
                                 half the intensity values will 
                                 be scaled to 0...mid-point ,
                                 and the other half will be scaled
                                 mid-point...maxval .

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      	_ISCConv _ISFn( is5_SaturationHistogramStretchImage )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fLowLimit, double fHighLimit, UINT32 uMidPoint, UINT32 uFlags);

/*---------------------------------------------------------
   is5_HistogramStretchImage / _is5_HistogramStretchImage

   Purpose : 
   Perform histogram stretch on an image.

   Note:
   Color channels are modified independently of each other, 
   so some shifts in color may occur. 
   
   See is5_BrightnessHistogramStretchImage if you want to 
   stretch the brightness of the image.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel max 8
   uRowStride        number of pixel components in a pixel row
   fLowLimit         percent of pixels to set to 0
   fHiLimit          percent of pixels to set to max component value
   uMidPoint         pixel value to set to middle intensity (1..max-1)

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      8       if set, use mid-point to divide
                              output range: 
                              half the intensity values will 
                              be scaled to 0...mid-point ,
                              and the other half will be scaled
                              mid-point...maxval .

                      9       images have 16 bits per component
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_HistogramStretchImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, double fLowLimit, double fHighLimit, UINT32 uMidPoint, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GetStretchedHistogramLUT / _is5_GetStretchedHistogramLUT

	Given a histogram, generate the LUT (Look-Up Table) that would
	give a histogram "stretch", if applied to an image. 
	
	The following steps are equivalent to is5_HistogramStretchImage:
		
		1. is5_GetImageChannelHistograms
		2. For each channel in the image:
			
			  is5_GetStretchedHistogramLUT
		3. is5_ApplyLUTsToImage

   Param             Use
   ------------------------------------
	pHisto				histogram of 255 (or 65535, for 16-bit) values
	uTotalElements		number of items counted in the histogram (w*h, if from an image)
   fLowLimit         percent of pixels to set to 0
   fHiLimit          percent of pixels to set to max component value
   uMidPoint         pixel value to set to middle intensity (1..max-1)

	pLUT					receives LUT of 255 (or 65535, for 16-bit) values,
							representing the mapping out input values to output values
							of the stretched histogram.
							 
   uFlags            bit      purpose
                     ---      -------
							0-7		unused

                      8       if set, use mid-point to divide
                              output range: 
                              half the intensity values will 
                              be scaled to 0...mid-point ,
                              and the other half will be scaled
                              mid-point...maxval .

                      9       images have 16 bits per component
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL			_ISCConv  _ISFn(is5_GetStretchedHistogramLUT)(UINT32 *pHisto, UINT32 uTotalElements, double fLowPercent, double fHiPercent, UINT32 midPoint, BYTE *pLUT, UINT32 uFlags);

/*---------------------------------------------------------
   is5_BrightnessHistogramEqualizeImage / _is5_BrightnessHistogramEqualizeImage

   Purpose : 
   Perform histogram equalization on an RGB or RGBA image.
   If RGBA, the A channel is ignored. 

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel. 3 for RGB, 4 for RGBA.
   uRowStride        bytes in an image row
   uLoThresh         low threshold
   uHiThresh         high threshold
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      	_ISCConv _ISFn( is5_BrightnessHistogramEqualizeImage )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uLoThresh, UINT32 uHiThresh, UINT32 uFlags);

/*---------------------------------------------------------
   is5_HistogramEqualizeImage / _is5_HistogramEqualizeImage

   Purpose : 
   Perform histogram equalization on an image.

   Note:
   Color channels are modified independently of each other, 
   so some shifts in color may occur. See 
   is5_BrightnessHistogramEqualizeImage if you want to 
   equalize the brightness of the image.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel max 8.
   uRowStride        pixel components in an image row
   uLoThresh         low threshold
   uHiThresh         high threshold

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      9       images have 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_HistogramEqualizeImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, UINT32 uLoThresh, UINT32 uHiThresh, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ContrastMaskImage / _is5_ContrastMaskImage

   Purpose : 
   Apply a contrast mask to the image to bring out shadow and 
   highlight details. This is much more effective than traditional
   brightness / contrast adjustments for most images.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   pOut              destination image
   uBytesPerPixel    number of bytes per pixel in the image. (3 or 4)
   uInRowStride      bytes per input pixel row

   uBlendMode        0 : softer contrast
							1 : higher contrast (brighter highlights, darker shadows)

	fSigma            amount of blur in the mask image.
							all positive values are valid, but the best results
							are typically between 2.0 and 8.0. too high and
							there may be ringing around details. too low and 
							the image will start to blur.

   uFlags            unused

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is5_ContrastMaskImage)(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, UINT32 uBlendMode, double fSigma, UINT32 uFlags);


#define IS_HISTOGRAM_16BPC       (1<<9)
#define IS_BRIGHTNESS_HISTOGRAM_STRETCH_HSL  (1<<9)
#ifdef __cplusplus
}
#endif
#endif