/*---------------------------------------------------------
   ISImageFFT.h : part of the ImgSource library.
   
   Image processing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#pragma once

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Do not include this file directly. Include ISource.h
#endif

#ifdef __cplusplus
extern "C" {
#endif


/*---------------------------------------------------------
   is5_AllocFFTGrid

   Purpose : 
   Allocate a working space for the FFT functions. 

   Note:
   The dimensions you pass should be the dimensions of the image
   you are going to be working on. 

   Note:
   Image dimensions must be equal to a power of 2:
   (ex. 128, 256, 512, 1024, etc.). All FFT operations require 
   this. Use is5_NextPowerOfTwo to find the next largest power of
   two for each dimension. Then prepare your image accordingly (overlay
   onto a background of the correct size or resize the input image).

   Note:
   This buffer will be very large. (w * h * 16 BYTEs)

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height

   Return
   ------
      HGLOBAL handle to FFT grid
      Caller must call GlobalFree to release this memory.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL    _ISCConv _ISFn( is5_AllocFFTGrid )(UINT32 uWidth, UINT32 uHeight);

/*---------------------------------------------------------
   is5_DoFFT

   Purpose : 
   Fill the FFT grid based on a single channel (ie. grayscale) 
   input image. This is the first step in an FFT operation.

   This grid must be (w * h * 2) doubles. You can use 
   is5_AllocFFTGrid to allocate this if you wish.

   Note:
   This function can use images composed of both BYTEs and doubles,
   but, it only works with one at a time. You must pass NULL for 
   the buffer you are not using: ie. pass NULL for pInputD
   if you are passing a BYTE image and pass NULL for pInputB
   if you are passing a double image.

   Param             Use
   ------------------------------------
   pInputB           BYTE input
   pInputD           double input
   uWidth            width in pixels
   uHeight           height

   uRowStride        number of bytes per pixel row (if using pInputB)
                        or
                     number of double per pixel row (if using pInputD)

   grid              FFT grid (caller allocates)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn( is5_DoFFT )(BYTE *pInputB, double *pInputD, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, double *grid);

/*---------------------------------------------------------
   is5_FilterFFT

   Purpose : 
   Apply one of the pre-defined FFT filters to the FFT grid.
   The FFT grid must have been filled with is5_DoFFT.

   This is the second step in an FFT operation.

   Note:
   The values required in the pParams array depend on the filter
   type you've chosen.

   Filter type                Parameter list
   ----------------------     --------------
   1 - low pass, ideal        pParam[0] = cutoff (d0)

   2 - high pass, ideal       pParam[0] = cutoff (d0)

   3 - band pass              pParam[0] = low cutoff
                              pParam[1] = high cutoff

   4 - 1/f^p filter           pParam[0] = power (p)

   5 - low pass butterworth   pParam[0] = cutoff
                              pParam[1] = order (try 2)

   6 - high pass butterworth  pParam[0] = cutoff
                              pParam[1] = order (try 2)

   7 - low pass gaussian      pParam[0] = cutoff (d0)

   8 - high pass gaussian     pParam[0] = cutoff (d0)

   9 - high-boost butterworth 
                              v = offset + factor * HighPassButterworth()
                              pParam[0] = cutoff (d0)
                              pParam[1] = offset (try .25 to .5) 
                              pParam[2] = factor (try 1.5 to 2.0)

   10 - high-boost gaussian   
                              v = offset + factor * HighPassGaussian()
                              pParam[0] = cutoff (d0)
                              pParam[1] = offset (try .25 to .5) 
                              pParam[2] = factor (try 1.5 to 2.0)

   Param             Use
   ------------------------------------
   grid              FFT grid (filled by is5_DoFFT)
   uWidth            width in pixels
   uHeight           height

   uFilterType       see above

   pParams           array of doubles. these are the parameters
                     for the various filters.

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn( is5_FilterFFT )(double *grid, UINT32 width, UINT32 height, UINT32 uFilterType, double *params);

/*---------------------------------------------------------
   is5_DoFFTInverse

   Purpose : 
   Apply the inverse FFT to the FFT grid. This prepares the
   grid for conversion back to the final grayscale image. This
   must be called AFTER all filtering operations.

   This is the third step in an FFT operation.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   grid              FFT grid (caller allocates)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn( is5_DoFFTInverse )(UINT32 uWidth, UINT32 uHeight, double *grid);

/*---------------------------------------------------------
   is5_MakeGrayscaleFromFFT

   Purpose : 
   Generate the final grayscale image from the FFT grid.

   This is the final step in an FFT operation.

   Note:
   This function can return images as BYTEs or as doubles.
   If you want BYTEs, provide a non-NULL pointer for pOutB.
   If you want doubles, provide a non-NULL pointer for pOutD.
   One of the output buffers must be NULL and the other must
   be non-NULL.

   Param             Use
   ------------------------------------
   pOutB             output image, BYTEs
   pOutD             output image, doubles
   grid              FFT grid (caller allocates)
   uWidth            width in pixels
   uHeight           height    

   uRowStride        number of bytes per pixel row (if using pInputB)
                        or
                     number of double per pixel row (if using pInputD)

   bTranformed       if TRUE, the time signal assumed to be in the real 
                     part of the grid.

                     if FALSE, the frequency case is split to 
                     put DC in the middle of the image.

   bScaled           if TRUE, pixel value will be scaled between 0 and 255.
                     if FALSE, pixels values will be in the same range as 
                     the input.

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn( is5_MakeGrayscaleFromFFT )(BYTE *pOutB, double *pOutD, double *grid, UINT32 width, UINT32 height, UINT32 uRowStride, BOOL bTransformed, BOOL bSCaled);

/*---------------------------------------------------------
   is5_NextPowerOfTwo

   Purpose : 
   Find the next-highest power of two (2^x), from the input value.

   Ex.
   is5_NextPowerOfTwo(3) = 4
   is5_NextPowerOfTwo(18) = 32
   is5_NextPowerOfTwo(255) = 256
   is5_NextPowerOfTwo(256) = 512

   Use this to determine the appropriate input image size 
   for the FFT operations.

   Param             Use
   ------------------------------------
   val

   Return
   ------
      next highest power of 2
---------------------------------------------------------*/
_ISDeclSpec UINT32 _ISCConv _ISFn( is5_NextPowerOfTwo )(UINT32 val);

#ifdef __cplusplus
}
#endif
