/*---------------------------------------------------------

   ISImageFilter.h : part of the ImgSource library.

   Image filters

	This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

	No portion of this code may be copied or distributed without 
	permission of Smaller Animals Software, Inc..
	http://www.smalleranimals.com

---------------------------------------------------------*/

#pragma once

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Do not include this file directly. Include ISource.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

	
/*---------------------------------------------------------
   is5_GaussianBlurImage / _is5_GaussianBlurImage

   Purpose : 
   Apply a Gaussian blur to an image.

   Note:
   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, BGR, etc.) 

   Note: 
   For values of fSigma greater than 1.0, this function
   will use an infinite impulse response (IIR) approximation 
   of a Gaussian blur (as described by Rachid Deriche, 1993). 
   This runs in constant time with respect to fSigma, as 
   compared to linear time with the traditional method which 
   does a 2x1D convolution on the image. In other words, 
   fSigma has no effect on the execution time with the IIR 
   method. So, you can use high values for fSigma (large 
   blurs) and there is no performance penalty.

   If you need the traditional (slower but more accurate) 
   version, you can force this call to use it by setting bit 8
   in uFlags. 

   All ImgSource functions which need a Gaussian blur 
   internally will use the IIR version.
   
   Note:
   Bit 10 of uFlags can be used to disable the rounding
   previous versions used. This will give better results
   on the edges of large blurs.

   pInput != pOutput

   Note:
   Callback is called twice per color channel.

   Param             Use
   ------------------------------------
   pInput            source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel color components per pixel. max 8.
   uInRowStride      number of components in an input pixel row
   pOutput           destination image
   uOutRowStride     number of components in an output pixel row

   fSigma            standard dev. of the gaussian kernel.
                     in practical terms, this controls the
                     amount of blur. good values to start 
                     with are between 0.6 and 2.4. higher
                     values increase processing time.

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      8       use 2x1D convolution, regardless of fSigma.
                      9       images have 16 bits per component
							 10		do not round results up (2x1d convolution, only)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_GaussianBlurImage )(void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOutput, UINT32 uOutRowStride, double fSigma, UINT32 uFlags);

/*---------------------------------------------------------
   is5_UnsharpMaskImage / _is5_UnsharpMaskImage

   Purpose : 
   Apply an unsharp mask effect to an image. This is a 
   high-frequency enhancement effect. Similar in effect 
   to a sharpening filter, it brings out edges
   and point details. 
   
   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, BGR, etc.) 
   
   DO NOT use this on images that require a palette 
   unless the palette is arranged so that the visual 
   distance between two colors is directly proportional 
   to the distance between their color indecies, for all
   combinations of colors in the palette.

   pInput != pOutput

   This function works by first doing a Gaussian blur on 
   the image, then finding the difference between pixels
   in the blurred image and in the original. For each pixel
   in the blurred and the original images, if the pixel 
   differs by greater than a given threshold value, a 
   percentage of the difference is added to the original 
   pixel.

   Param             Use
   ------------------------------------
   pInput            source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel color components per pixel. max 8.
   uInRowStride      number of components in an input pixel row
   pOutput           destination image
   uOutRowStride     number of components in an output pixel row

   
   uThreshold        controls the depth of the effect. 
                     any value between 0 and 255 is valid.
                     the lower the value, the greater the effect.

   fAmount           controls the intensity of the effect.
                     values should be between 0.0 (no effect) 
                     and 1.0 (full effect). higher values will 
                     give unpredictable results.

   fSigma            standard dev. of the gaussian kernel.
                     this has an effect similar to the uThreshold 
                     value, but much broader (effects more pixels).
                     good values to start with are between 0.6 
                     and 2.4. 

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      9       images have 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_UnsharpMaskImage )(void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOutput, UINT32 uOutRowStride, UINT32 uThreshold, double fAmount, double fSigma, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ApplyConvolutionFilterToImage / _is5_ApplyConvolutionFilterToImage

   Purpose : 
   Apply a convolution filter to an image. This
   should only be used on continuous-tone images (RGBA,
   grayscale, RGB, etc.) 
   
   DO NOT use this on images that require a palette 
   unless the palette is arranged so that the visual 
   distance between two colors is directly proportional 
   to the distance between their color indecies, for all
   combinations of colors in the palette.

   If bTraditional is true, this can be used to do things
   like a Gaussian blur, etc., depending on the matrix used.

  (see also is5_ApplyMatrixToImage)

   The number of columns and the number of rows in the 
   matrix must both be odd.

   The filter is applied as follows :

      M = matrix ( 3 x 3, in this example)

		M = 
      1  1  1
      1 -8  1       this is a sharpening matrix
      1  1  1

      the matrix is centered on the current pixel. in 
      this example, M[4] is the center of the matrix.

      k is a scaling factor.

      P = pixels . since the matrix is centered on the 
      current pixel, P(x,y), the pixels used in this 
      calculation will be :
      
         P(x-1,y-1),    P(x, y-1),     P(x+1, y-1)
         P(x-1,y),      P(x, y),       P(x+1, y)
         P(x-1,y+1),    P(x, y+1),     P(x+1, y+1)

      t = sum ( M[i] * P[i] )  [for all elements of the matrix]


      if "traditional" processing (see uFlags)

         outPixel = k * t

      else

         outPixel = curPixel - (k * t)

      outPixel is then clipped to 0..255 (or 0.65535, if bpc=16)

   Matrix M in this example demonstrates the sharpening filter 
   used by this library, if bTraditional is FALSE.

   besides the ability to use matricies of arbitrary size,
   this function differs from is5_ApplyMatrixToRGB in the way 
   the final caluclation is performed. 

   Param             Use
   ------------------------------------
   pImage            input image
   uWidth            width in pixels
   uHeight           height 
   uChannelsPerPixel number of components in a pixel
   uInRowStride      number of components in an input pixel row
   pImageOut         output image
   uOutRowStride     number of components in an output pixel row
   uFilterCols       columns in filter matrix
   uFilterRows       rows in filter matrix
   pFilter           filter matrix
   k                 filter level

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      8       if set use the "traditional" method. see 
                              description above.

							 9       images have 16 bits per component
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_ApplyConvolutionFilterToImage )(void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pImageOut, UINT32 uOutRowStride, UINT32 uFilterCols, UINT32 uFilterRows, double * pFilter, double k, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ApplyMatrixToImage / _is5_ApplyMatrixToImage

   Purpose : 
   Apply a convolution matrix to an image.
                           
   (see also is5_ApplyConvolutionFilterToImage)

   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, BGR, etc.) 
   
   DO NOT use this on images that require a palette 
   unless the palette is arranged so that the visual 
   distance between two colors is directly proportional 
   to the distance between their color indecies, for all
   combinations of colors in the palette.

   pImage != pImageOut

   The matrix is an array of _9_ doubles, arranged as :

         0  1  2
         3  4  5
         6  7  8

      where "4" is the source pixel.

   It is applied for pixel(x,y) as :

   Step 1
   double tempSum = (
         pMatrix[0] * pixel(x-1, y-1) +
         pMatrix[1] * pixel(x,   y-1) +
         pMatrix[2] * pixel(x+1, y-1) +
         pMatrix[3] * pixel(x-1, y )  +
         pMatrix[4] * pixel(x,   y )  +
         pMatrix[5] * pixel(x+1, y )  +
         pMatrix[6] * pixel(x-1, y+1) +
         pMatrix[7] * pixel(x,   y+1) +
         pMatrix[8] * pixel(x+1, y+1));

   Step 2
         tempSum = tempSum * k;

   Step 3
         tempSum = min(tempSum, 255);

   Step 4
         tempSum = max(tempSum, 0);

   Step 5
         outval = (BYTE) tempSum;


   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel. max 8.
   uInRowStride      number of bytes in an input pixel row

   pImageOut         destination image

   uDoFeedback       if the bDoFeedback options is 1, 
                     Step 2 becomes :
                     tempSum = pixel(x,y) * (1.0 - k) + 
                     (k * tempSum);

   k                 intensity factor

   pMatrix           array of doubles

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      9       images have 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_ApplyMatrixToImage )(void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pImageOut, UINT32 uOutRowStride, UINT32 uDoFeedback, double k, double *pMatrix, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ApplyMatrixToImageDouble / _is5_ApplyMatrixToImageDouble

   Purpose : 
   Apply a convolution matrix to an image of doubles.

   This is basically the same as is5_ApplyMatrixToImage,
	except that the image pixels are doubles, and you must
	specify minimum and maximum values for the pixel values.
	In integer-based versions, these limits are inherent in
	the data types, with doubles, there are no obvious limits.

   Param             Use
   ------------------------------------
   pImage            source image, doubles
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of components per pixel. max 8.
   uInRowStride      number of components in an input pixel row

   pImageOut         destination image, doubles

   uDoFeedback       if the bDoFeedback options is 1, 
                     Step 2 becomes :
                     tempSum = pixel(x,y) * (1.0 - k) + 
                     (k * tempSum);

   fFeedback         intensity factor

   pMatrix           array of doubles

   fMinVal				minimum value allowed in the output image
	fMaxVal				maximum value allowed in the output image

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      9       images have 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     _ISCConv  _ISFn(is5_ApplyMatrixToImageDouble)(double *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, double *pImageOut, UINT32 uOutRowStride, UINT32 uDoFeedback, double k, double *pMatrix, double minVal, double maxVal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ApplyLUTAndCrossoverMatrixToRGB / _is5_ApplyLUTAndCrossoverMatrixToRGB

   Purpose : 
   Apply a crossover matrix and LUT to an RGB image.

   The matrix is a 12 item (3 rows, 4 columns) array of
   floating point values. 

   The LUT is an array of 3 * 256 BYTEs. 

   LUT[0]   -> LUT[255] are the values for the red channel (LUT_rd).
   LUT[256] -> LUT[511] are the values for the green channel (LUT_gr).
   LUT[512] -> LUT[767] are the values for the blue channel (LUT_bl).

   If pLUT is NULL, an identity LUT will be used.


   The matrix and LUT are applied as follows :

     -------------

     For each pixel, "cur", 

     // LUT
     cur.rd = LUT_rd[cur.rd]
     cur.gr = LUT_gr[cur.gr]
     cur.bl = LUT_bl[cur.bl]

     // matrix
     out.rd =(cur.rd * matrix[0, 0] + 
            cur.gr * matrix[0, 1] +
            cur.bl * matrix[0, 2])
         
     out.rd = out.rd + matrix[0, 3]

     out.gr =(cur.rd * matrix[1, 0] + 
            cur.gr * matrix[1, 1] +
            cur.bl * matrix[1, 2]) 

     out.gr = out.gr + matrix[1, 3]

     out.bl =(cur.rd * matrix[2, 0] + 
            cur.gr * matrix[2, 1] +
            cur.bl * matrix[2, 2])

     out.bl = out.bl + matrix[2, 3]
     
     ------------

     Matrix samples : 

     This matrix has no effect on the image :
       1.0, 0.0, 0.0, 0.0
       0.0, 1.0, 0.0, 0.0
       0.0, 0.0, 1.0, 0.0

     Increase the saturation :
        1.2, -0.1, -0.1, 0.0
       -0.1,  1.2, -0.1, 0.0
       -0.1, -0.1,  1.2, 0.0

     Increases the brightness :
       1.0, 0.0, 0.0, 20.0
       0.0, 1.0, 0.0, 20.0
       0.0, 0.0, 1.0, 20.0

     Change to grayscale :
       1/3, 1/3, 1/3, 0
       1/3, 1/3, 1/3, 0
       1/3, 1/3, 1/3, 0

     General rule : the sum of the first 3 items in 
     a row should add to 1. 

   Param             Use
   ------------------------------------
   pRGB              RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per pixel row
   pLUT              look-up table (LUT) - 768 values
   pMatrix           12 element crossover matrix

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_ApplyLUTAndCrossoverMatrixToRGB )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, BYTE *pLUT, double *pMatrix);

/*---------------------------------------------------------
   is5_CannyEdgeDetector

   Purpose : 
   Find edges in an image, using the Canny algorithm

   Param             Use
   ------------------------------------
   pIn               source image (8-bit grayscale)
   uWidth            width in pixels
   uHeight           height
   uInRowStride      number of pixels in input image row
   pOut              output image (8-bit grayscale)
   uOutRowStride     number of pixels in output image row

   fSigma            standard dev. of the gaussian kernel.
							controls pre-detection smoothing.
							larger values increase smoothing and 
							decrease the amount of edges found.

                     good values to start with are 
                     between 0.6 and 2.4. 

							larger values increase processing time.

   tlow              Specifies the low value to use in hysteresis. This is a 
                     fraction (0-1) of the computed high threshold edge 
                     strength value. (0.20-0.50 are good starting values)

   tHigh             Specifies the high value to use in hysteresis. 
                     This fraction (0-1) specifies the percentage point 
                     in a histogram of the gradient of the magnitude. 
                     Magnitude values of zero are not counted in the histogram.
                     (0.60-0.90 are good starting values)

   uFlags            unused

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_CannyEdgeDetector )(BYTE *pIn, UINT32 w, UINT32 h, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, double fSigma, double tLow, double tHigh, UINT32 uFlags);

/*---------------------------------------------------------
   is5_MotionBlur

   Purpose : 
   Apply a motion blur effect to an image

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uBpp              number of bytes per pixel
   uInRowStride      bytes per input pixel row
   pOut              output image
   uOutRowStride     bytes per output pixel row
   fAngle            angle of motion, for linear blur
   fDistance         distance between original image and final blur step
   length            number of blur steps
   eStyle            0 : linear. image is blurred in a line (traditional motion blur)
                     1 : radial. image is blurred in a circle around the center point
                     2 : zoom. image is blurred as if zooming in

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_MotionBlur )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, UINT32 uBpp, BYTE *pOut, UINT32 uOutRowStride, double fAngle, double fDistance, int length, int eStyle);

/*---------------------------------------------------------
   is5_ArithmeticMeanFilter

   Purpose : 
   Apply an arithmetic mean filter to an image
   (find average all values under filter)

   This is equivalent to is5_BlurImage with an opacity of 1.0.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_ArithmeticMeanFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uFilterSize, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GeometricMeanFilter

   Purpose : 
   Apply a geometric mean filter to an image

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_GeometricMeanFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uFilterSize, UINT32 uFlags);

/*---------------------------------------------------------
   is5_HarmonicMeanFilter

   Purpose : 
   Apply a harmonic mean filter to an image

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_HarmonicMeanFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, UINT32 uFlags);

/*---------------------------------------------------------
   is5_YpMeanFilter

   Purpose : 
   Apply a Yp Mean filter to an image

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect

   iP                controls power.
                        iP < 0, removes positive outliers
                        iP > 0, removes negative outliers

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_YpMeanFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, int iP, UINT32 uFlags);

/*---------------------------------------------------------
   is5_MidpointFilter

   Purpose : 
   Apply a midpoint filter to an image
   (find average of minimum and maximum values under filter)

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel   number of components per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_MidpointFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, UINT32 uFlags);

/*---------------------------------------------------------
   is5_MaximumFilter

   Purpose : 
   Apply a maximum filter to an image
   (find maximum value under filter)

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_MaximumFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, UINT32 uFlags);

/*---------------------------------------------------------
   is5_MinimumFilter

   Purpose : 
   Apply a minimum filter to an image. 
   (find minimum value under filter)

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_MinimumFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, UINT32 uFlags);

/*---------------------------------------------------------
   is5_SharpenImage

   Purpose : 
   Sharpen an image.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect
   k                 filter intensity (0 to 1)

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_SharpenImage )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, double k, UINT32 uFlags);

/*---------------------------------------------------------
   is5_BlurImage

   Purpose : 
   Blur an image.

   Note: 
	If fOpacity = 1.0, this funstion is equivalent to is5_ArithmeticMeanFilter.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect
   k                 filter intensity (0 to 1)

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_BlurImage )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, double k, UINT32 uFlags);

/*---------------------------------------------------------
   is5_MedianFilterImage

   Purpose : 
   Apply a median filter to an image.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect
	fAmount				controls opacity of effect (0=no effect, 1.0 = full effect)

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

  							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_MedianFilterImage )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, double fAmount, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ThresholdFilter

   Purpose:
   Set pixel components above a high threshold value to 255, set pixels
   below a low threshold value to 0.

   Param             Use
   ------------------------------------
   pIn               input image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   uMinVal           pixel values below this will be set to 0
   uMaxVal           pixel values above this will be set to 255

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_ThresholdFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, UINT32 uMinVal, UINT32 uMaxVal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_AdaptiveUnsharpMask

   Purpose:
   Perform an adaptive unsharp mask on the input image. This
   is a sophisticated sharpening function.

   Param             Use
   ------------------------------------
   pIn               input image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image
   uInRowStride      bytes per input pixel row
   pOut              output image
   uOutRowStride     bytes per output pixel row
   uThreshold        0 to 255. lower numbers yiled greater effect
   fAmount           amount of sharpened signal to add to the output image.
                     0 = no change. 1 = full change. you can use any number
                     (even negative), but the intended use is between 0 and 1.0.

   uMode					0: fast, less accurate
							1: good results, slower
							2: best results, very slow

   uFlags				bit			use
							---			---
							 2				if set, skip alpha channel
							 9				images have 16 bits per component

  

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AdaptiveUnsharpMask )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uThreshold, double fAmount, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is5_FindImageMean

   Purpose:
   Find the mean (arithmetic average) pixel value for one
   or more channels in an image.

   Param             Use
   ------------------------------------
   pIn               input image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image (max 8)

   pMeans            array of doubles, uBytesPerPixel elements.
                     these receive the calculated mean values.

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

  

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_FindImageMean )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, double *pMean, UINT32 uFlags);

/*---------------------------------------------------------
   is5_FindImageStandardDeviation

   Purpose:
   Find the standard deviation of the pixel values for one
   or more channels in an image.

   Param             Use
   ------------------------------------
   pIn               input image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image (max 8)

   pStdDeviation     array of doubles, uBytesPerPixel elements.
                     these receive the calculated values.

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_FindImageStandardDeviation )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, double *pStdDeviation, UINT32 uFlags);

/*---------------------------------------------------------
   is5_LocalHistogramEnhancement

   Purpose:
   Adjust the relative brightness of local region areas
   based on global image statistics. This can be used to
   raise or darken image areas with brightness levels that 
   differ significantly from the overall image.

   This locates areas of the image where the local brightness 
   is different (either above or below) from the global 
   brightness and adjusts the brightness accordingly. "Difference"
   is determined by calculating the mean and standard deviation for
   pixels under the filter and comparing them to the mean and 
   std dev for the entire image. The input parameters allow you
   to adjust the sensitivity and range of the comparison.

   Each output pixel is determined in this fashion:

   if ((local_mean <= k0 * global_mean) AND
       (k1 * global_std_dev <= local_std_dev <= k2 * global_std_dev))
   then
      out = input * k3
   else
      out = input

   When you have selected the "decrease brightness" option (via uFlags),
   the calculation becomes:

   if ((local_mean >= k0 * global_mean) AND
       (k1 * global_std_dev <= local_std_dev <= k2 * global_std_dev))
   then
      out = input * k3
   else
      out = input


   The local_mean and local_std_dev are calculated for each local 
   region of size uFliterSize x uFliterSize. Global mean and standard
   deviation are calculated with is5_FindImageMean and 
   is5_FindImageStandardDeviation. These calculation only need to be done
   once, if the input image is not changed. 

   Note:
   Use of this function will almost always require user input, as it 
   is extremely sensitive to image statistics and the parameters given.
   
   Reasonable starting values for k0, k1, k2 and k3 are :

      k0 : 0.5
      k1 : 0.02
      k2 : 0.5
      k3 : 3.0

   Param             Use
   ------------------------------------
   pIn               input image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image (max 8)
   uInRowStride      bytes per pixel row, input
   pOut              output image
   uOutRowStride     bytes per output row

   uFilterSize       must be odd. controls the size of the local area
                     inspected for adjustment. 3 is a good starting value.

   pStdDeviation     array of doubles, uBytesPerPixel elements.
                     these receive the calculated values.

   pMeans            array of mean values calculated in advance. see
                     is5_FindImageMean. 

   pStdDeviations    array of standard deviation values calculated in 
                     advance. see is5_FindImageStandardDeviation.

   k0                positive with a value generally less than 1.
                     this determines the lower limit for affected 
                     pixel values. 

   k1                this determines the lower limit with respect to the
                     local standard deviation for affected pixels. 0 < k1 < k2.

   k2                determines the upper limit with respect to the
                     local standard deviation for affected pixels. 0 < k1 < k2.

   k3                factor that determines the strength of the effect.

  uFlags             bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_LocalHistogramEnhancement )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uFilterSize, double *pMeans, double *pStdDeviations, double k0, double k1, double k2, double k3, UINT32 uFlags);

// experimental
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AdaptiveDWMTMFilter )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, double fNSTD, double K, UINT32 uFlags);

/*---------------------------------------------------------
   is5_MedianFilter1Bit

   Purpose : 
   Apply a median filter to a 1-bit image.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   N                 larger filter yields stronger effect (start with 3)
   uFlags            unused

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_MedianFilter1Bit )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, UINT32 N, UINT32 uFlags);

/*---------------------------------------------------------
   is5_DustAndScratches

   Purpose : 
   Removes dust and scratches (or any other small sharp 
   details) from the image. This is similar in effect
   to Photoshop's "Dust and Scratches" filter.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   uInRowStride      bytes per input pixel row
   pOut              destination image
   uOutRowStride     bytes per output pixel row
   uFilterSize       larger filter yields stronger effect

   uThreshold        0 to 255. controls the minimum detail
                     intensity required to trigger removal.

   fAmount           0.0 to 1.0. controls blending of original
                     image with the removed image detail. 
                     higher values obscure more of the source image.

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

							 9       images have 16 bits per component

  

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_DustAndScratches )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uFilterSize, UINT32 uThreshold, double fAmount, UINT32 uFlags);

/*---------------------------------------------------------
   is5_BilateralFilterImage

   Purpose : 
   Apply a bilateral filter to the image. 

   A bilateral filter is a detail-preserving denoising filter.
   It tends to reduce noise in smooth areas while leaving strong
   image details unaffected.

   Because this can be a very slow process, we have implemented a 
   quick and dirty approximation of the bilateral filter that
   runs at least an order of magnitude faster than the default 
   method while providing results that are similar to a true 
   bilateral filter. See uFlags, bit 8.

   Callback is called once per channel for the fast method, and
   one per row per channel, for the true bilateral method.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   uComponentsPerPixel    number of bytes per pixel in the image. max 8.
   pOut              destination image
   uOutRowStride     bytes per output pixel row

   uFilterSize       larger filter yields stronger effect, but 
                     requires more processing time. try 15 to start.

   uFilterType       select one of the following shapes for 
                     the smoothing kernel:
                        0 = Gaussian
                        1 = linear
                        2 = cosine
                        3 = constant

   fSigma            controls intensity of the effect. try 20 to start.

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      8       use the quick-and-dirty method. this is
                              not strictly a bi-lateral filter, but the results
                              are very close, and the function can be over 
                              ten times faster than the true bi-lateral filter.

 					  9       images have 16 bits per component

					  20	  disable multithreaded processing




   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_BilateralFilterImage )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uRowOutStride, UINT32 uFilterSize, UINT32 uFilterType, double fSigma, UINT32 uFlags);

/*---------------------------------------------------------
   is5_SobelFilter

   Purpose : 
   Apply a Sobel filter to the image. 

   This is an edge-highlighting filter.

   Because the output of this filter can be larger than 8 bits per
   component (for 8 bit per component input), the output image must 
   be 16 bits per pixel component. For 16-bit input, the output
   image must be 32 bits per component.

   This function has two modes: Seperate and Combined.
   The "Seperate" mode will filter up to 8 channels of data, each 
   channel processed independently of the others - as is typical of
   ImgSource filters.
   
   The "Combined" mode will process three input channels (RGB or RGBA, 
   ignoring alpha) into a single output channel; it does this by 
   finding the individual Sobel result for the 3 input channels, 
   then outputs the average of the three channels' results. In effect, 
   it does a simple RGB -> grayscale conversion on-the-fly. 
   
   See uFlags, bit 8.
	
   Callback is called once per row.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
	  *  Combined mode can only work on 3 or 4 bpp pixels (does not touch the alpha channel)

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uComponentsPerPixel    number of components per pixel in the images 
   uInRowStride      bytes per input pixel row

   pOut              destination image. for Separate mode, this
                     must have the number of components as the input image.
                     for Combined mode, this is a single channel image.

   uOutRowStride     bytes per output pixel row

   uFlags            bit      purpose
                     ---      -------
                     0-7      Used only in "Separate" mode.

                              These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      8       If clear, use 8->16, Seperate processing.
							  If set, use 3x8->16, Combined processing.

                      9       If set, source image is 16 bits per channel and
                              the output image is 32 bits per channel. Otherwise
                              input is 8-bpc and output is 16-bpc.

					  20	  disable multithreaded processing

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_SobelFilter )(const void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uFlags);

#define IS_MATRIX_16BPC (1<<9)

// currently these only apply to is5_SobelFilter	and is5_BilateralFilterImage
#define IS_FILTER_NO_LEFT_BOUNDSCHECK    (1<<20)
#define IS_FILTER_NO_RIGHT_BOUNDSCHECK   (1<<21)
#define IS_FILTER_NO_TOP_BOUNDSCHECK     (1<<22)
#define IS_FILTER_NO_BOTTOM_BOUNDSCHECK  (1<<23)

_ISDeclSpec BOOL _ISCConv _ISFn( is5_MedianFilterImage_O1 )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride,UINT32 uFilterSize, UINT32 uFlags);
_ISDeclSpec BOOL _ISCConv _ISFn( is5_DustAndScratches_O1 )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, UINT32 uFilterSize, UINT32 uThreshold, double fAmount, UINT32 uFlags);

#ifdef __cplusplus
}
#endif
