/*---------------------------------------------------------
   ISImageOverlay.h : part of the ImgSource library.
   
   Image processing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is5_IMAGEOVERLAYH
#define is5_IMAGEOVERLAYH

// shut up, stupid compiler!
#pragma warning(disable:4786)


#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_OverlayImage / _is5_OverlayImage

   Purpose : 
   Overlay one image onto another.

   Note: this should only be used on continuous-tone images 
   (RGBA, grayscale, RGB, etc.) 

   Note:
   Some overlay modes can be done using MMX functions. The MMX
   versions will be faster in general than the normal versions -
   many times faster, depending on overlay mode and buffer size. 

   ** As of v4.0.2.0, MMX versions are automatically used, on processors
   that support MMX **

   Param             Use
   ------------------------------------
   pBottom           bottom image
   uBottomWidth      bottom width in pixels
   uBottomHeight     bottom height 
   uChannelsPerPixel components per pixel (both images)
   uBottomRowStride  components per pixel row, bottom image
   
   pTop              top image
   uTopWidth         top width in pixels
   uTopHeight        top height 
   uTopRowStride     components per pixel row, top image   

   iXPos             position of top on bottom (may be < 0)
   iYPos             position of top on bottom (may be < 0)
   fOpacity          opacity of blended image. 0 to 1; 0 = transparent, 1 = opaque
   fK                double parameter. use depends on uMode

   uMode            values:
                     ( note * = MMX version available )

                     0 - fast blend
                     1 - slower, more accurate blend
                     2 - pixels are combined with logical AND     *
                     3 - pixels are combined with logical OR      *
                     4 - pixels are combined with logical XOR     *
                     5 - output pixel is max of top and bottom  (a.k.a. Photoshop 'Lighten')  *
                     6 - output pixel is min of top and bottom   (a.k.a. Photoshop 'Darken') *
                     7 - output pixel is the sum of top and bottom   *
                     8 - output pixel is difference of top and bottom   *
                     9 - if top is > fK, output top *
                     10 - if top is < fK, output top 
                     11 - absolute value of the difference of top and bottom  *
                     12 - multiply: (top * bottom) / maxComponent
                     13 - multiply with factor (top * bottom * fK) / maxComponent
                     14 - screen
                     15 - bottom = bottom + top - fK
                     16 - bottom = bottom - top - fK
                     17 - pixels are combined with logical NAND
                     18 - pixels are combined with logical NOR
                     19 - pixels are combined with logical NXOR/XNOR
                     20 - color dodge
                     21 - color burn
                     22 - soft dodge
                     23 - soft burn
                     24 - Photoshop "overlay"
                     25 - soft light (similar, but not identical to Photoshop's)
                     26 - hard light
                     27 - XFader reflect
                     28 - XFader glow
                     29 - XFader freeze
                     30 - XFader heat

   uFlags            bit         purpose
                     ---         -------
                      0				image pixel components have 16 bits
                      2          vertically flip top image (mode 0 and 1 only)          
							 20	      disable multithreaded processing
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_OverlayImage )(void *pBottom, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uChannelsPerPixel, UINT32 uBottomRowStride, void *pTop, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, double fOpacity, double fK, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is5_Overlay8Bit / _is5_Overlay8Bit

   Purpose : 
   Overlay one 8-bit image onto another

   Note:
   Both palettes are NULL:
   The images are assumed to have the same palette and the 
   top image is simply copied onto the bottom image with 
   no color matching. Opacity is ignored!

   pPalBottom is not NULL but pPalTop is NULL:
   Both images use the bottom palette.
   Output pixels are determined by looking up the RGB values 
   for both top and bottom pixels in the bottom palette and 
   combining these according to the opacity value. The 
   closest color to the resulting ideal RGB value will be 
   used from the bottom palette.

   Both palettes are not NULL:
   Output pixels are determined by looking up the RGB from 
   the bottom pixel in the bottom palette, the top pixel in 
   the top palette and combining these according to the opacity 
   value. The closest color to the resulting ideal RGB value 
   will be used from the bottom palette.

   Param             Use
   ------------------------------------
   pImageBottom         bottom image
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height
   uBottomRowStride     bytes per pixel row, bottom image
   uBottomPalColors     number of colors in bottom palette
   pPalBottom           palette for bottom image. 
   pImageTop            top image
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        bytes per pixel row, top image
   pPalTop              palette for top image
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)

   fOpacity             opacity of top on bottom (> 0, < 1)
                        (0 = transparent, 1 = opaque)

   uTransColorIdx       if you are using a transparent color,
                        this is the palette index of that color.

   uFlags               bit         purpose
                        ---         -------
                         0          use transparency

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_Overlay8Bit )(BYTE *pImageBottom, UINT32 uWidthBottom, UINT32 uHeightBottom, UINT32 uBottomRowStride, UINT32 uBottomPalColors, RGBQUAD *pPalBottom, BYTE *pImageTop, UINT32 uWidthTop, UINT32 uHeightTop, UINT32 uTopRowStride, RGBQUAD *pPalTop, __int32 iXPos, __int32 iYPos, double fOpacity, UINT32 uTransColorIdx, UINT32 uFlags);
           
/*---------------------------------------------------------
   is5_OverlayImageColor / _is5_OverlayImageColor

   Purpose : 
   Overlay the Hue, Saturation or Luminance values from pixels
	of one RGB/RGBA image onto another.

   Pixels from top and bottom are converted to the HLS colorspace, 
	combined, then converted back to RGB. Top image is not modified.

	Note:
	Alpha channel is not affected, for RGBA images.

   Param             Use
   ------------------------------------
   pBottom           bottom image (RGB, RGBA)
   uBottomWidth      bottom width in pixels
   uBottomHeight     bottom height 
	uBottomBytesPerPixel	bytes per pixel, bottom image (3 or 4)
   uBottomRowStride  bytes per pixel row, bottom image
   pTop              top image (RGB RGBA)
   uTopWidth         top width in pixels
   uTopHeight        top height
   uTopRowStride     bytes per pixel row, top image
   iXPos             position of top on bottom (may be < 0)
   iYPos             position of top on bottom (may be < 0)

   uMode					0: overlay Hue from top image
							1: overlay Saturation from top image
							2: overlay Color (Hue and Saturation) from top image
							3: overlay Luminosity from top image

   uFlags            unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is5_OverlayImageColor)(BYTE *pBottom, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBytesPerPixel, UINT32 uBottomRowStride, BYTE *pTop, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, double fOpacity, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is5_Overlay8BitOnImage / _is5_Overlay8BitOnImage

   Purpose : 
   Overlay an 8-bit image onto an RGB or RGBA image.

	Note:
	Alpha channel is not affected, for RGBA images.

   Param             Use
   ------------------------------------
   pRGB              bottom image
   uBottomWidth      bottom width in pixels
   uBottomHeight     bottom height 
	uBottomBytesPerPixel	bytes per pixel, bottom image (3 or 4)
   uBottomRowStride  bytes per pixel row, bottom image
   p8Bit             top image
   uTopWidth         top width in pixels
   uTopHeight        top height
   uTopRowStride     bytes per pixel row, top image
   pPal              palette of top image
   iXPos             position of top on bottom (may be < 0)
   iYPos             position of top on bottom (may be < 0)
   fOpacity          opacity of top on bottom (> 0, < 1)
                     (0 = transparent, 1 = opaque)

   uTransColorIdx    transparent color index

   uFlags               bit         purpose
                        ---         -------
                         0          favor speed over accuracy
                         1          use BGR color order
                         2          use transparent color index

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_Overlay8BitOnImage )(BYTE *pRGB1, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBottomBytesPerPixel, UINT32 uBottomRowStride, BYTE *p8Bit, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, double fOpacity, UINT32 uTransColorIdx, UINT32 uFlags);

/*---------------------------------------------------------
   is5_OverlayImage8BitAlphaOnRGB / _is5_OverlayImage8BitAlphaOnRGB

   Purpose : 
   Overlay an 8-bit image onto an RGB image using
   the rgbReserved values in the palette as alpha
   values.

   Param             Use
   ------------------------------------
   pRGB              bottom image
   uBottomWidth      bottom width in pixels
   uBottomHeight     bottom height
   uBottomRowStride  bytes per pixel row, bottom image
   p8Bit             top image
   uTopWidth         top width in pixels
   uTopHeight        top height
   uTopRowStride     bytes per pixel row, top image
   pPal              palette of top image
   iXPos             position of top on bottom (may be < 0)
   iYPos             position of top on bottom (may be < 0)

   uFlags            bit         purpose
                     ---         -------
                      0          set if bottom image is in BGR order
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_OverlayImage8BitAlphaOnRGB )(BYTE *pBottomImg, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBottomRowStride, BYTE *pTopImg, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, RGBQUAD *pPalTop, __int32 iXPos, __int32 iYPos, UINT32 uFlags);

/*---------------------------------------------------------
   is5_Overlay8BitTransIdxOnRGBA / _is5_Overlay8BitTransIdxOnRGBA

   Purpose : 
   Overlay an 8-bit image onto an RGBA image, one color
   index of the 8-bit image can be used as a 'transparent'
   color - the A channel of all transparent pixels is set
   to 0, all other pixels have their A channel set to 255.

   Param             Use
   ------------------------------------
   pRGB              bottom image
   uBottomWidth      bottom width in pixels
   uBottomHeight     bottom height
   uBottomRowStride  bytes per pixel row, bottom image
   p8Bit             top image
   uTopWidth         top width in pixels
   uTopHeight        top height
   uTopRowStride     bytes per pixel row, top image
   pPal              palette of top image
   iXPos             position of top on bottom (may be < 0)
   iYPos             position of top on bottom (may be < 0)
   uTransColorIdx    transparent color index

   uFlags            bit         purpose
                     ---         -------
                      0          set if bottom image is in BGR order
                      2          use transparency index

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is5_Overlay8BitTransIdxOnRGBA )(BYTE *pBottomImg, UINT32 uBottomImageWidth, UINT32 uBottomImageHeight, UINT32 uBottomRowStride, BYTE *pTopImg, UINT32 uTopImageWidth, UINT32 uTopImageHeight, UINT32 uTopRowStride, RGBQUAD *pPalTop, __int32 iXPos, __int32 iYPos, UINT32 uTransColorIdx, UINT32 uFlags);
                  
/*---------------------------------------------------------
   is5_OverlayImage8BitAlpha / _is5_OverlayImage8BitAlpha

   Purpose : 
   Overlay one 8-bit image onto another using the
   rgbReserved values in the palette as alpha values.

   Note:
   The images are assumed to have the same palette.

   Param                Use
   ------------------------------------
   pImageBottom         bottom image
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height
   uBottomRowStride     bytes per pixel row, bottom
   pImageTop            top image
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        bytes per pixel row, top
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)
   uPalColors           number of colors in the palette
   pPal                 palette
   uFlags               unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_OverlayImage8BitAlpha )(BYTE *pBottomImg, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBottomRowStride, BYTE *pTopImg, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, UINT32 uPalColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is5_Overlay1Bit / _is5_Overlay1Bit

   Purpose : 
   Overlay one 1 bit image onto another.

   Pixel rows must start on byte boundaries.

   Param                Use
   ------------------------------------
   pImageBottom         bottom image
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height 
   uBottomRowStride     bytes per pixel row, bottom
   pImageTop            top image
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        bytes per pixel row, top
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)

   uMode                overlay mode:
                        0 - set bottom to top 
                        1 - logical AND
                        2 - logical OR
                        3 - logical XOR
                        4 - logical NOR
                        5 - logical NAND
                        6 - logical XNOR

   uFlags               bit         purpose
                        ---         -------
                         0          if set, use transparency
                         1          if bit 0 is set, this bit is the 
                                    transparent color index:
                                    clear = color 0 transparent (eq to logical OR)
                                    set = color 1 transparent   (eq to logical AND)

                         3          vertical flip top image
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_Overlay1Bit )(BYTE *pImageBottom, UINT32 uWidthBottom, UINT32 uHeightBottom, UINT32 uBottomRowStride, BYTE *pImageTop, UINT32 uWidthTop, UINT32 uHeightTop, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is5_Overlay1BitMasked / _is5_Overlay1BitMasked

   Purpose : 
   Overlay one 1 bit image onto another, using a mask image
   to control which parts of the top are overlayed on the 
   bottom image.

   Pixel rows must start on byte boundaries.

   Param                Use
   ------------------------------------
   pImageBottom         bottom image
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height 
   uBottomRowStride     bytes per pixel row, bottom
   pImageTop            top image
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        bytes per pixel row, top

   pMask                1-bit mask image. must have the same
                        dimensions as the top image. the mask
                        image controls opacity in the top image. 
                        the top image will only show through
                        in places where a mask image pixel
                        has a value of 1.

   uMaskRowStride       number of bytes per mask image row

   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)

   uFlags               bit         purpose
                        ---         -------
                         0          if set, use transparency. this
                                    is applied after the mask.

                         1          if bit 0 is set, this bit is the 
                                    transparent color index:
                                    clear = color 0 transparent
                                    set = color 1 transparent

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_Overlay1BitMasked )(BYTE *pImageBottom, UINT32 uWidthBottom, UINT32 uHeightBottom, UINT32 uBottomRowStride, BYTE *pImageTop, UINT32 uWidthTop, UINT32 uHeightTop, UINT32 uTopRowStride, BYTE *pMask, UINT32 uMaskRowStride, __int32 iXPos, __int32 iYPos, UINT32 uFlags);

/*---------------------------------------------------------
   is5_OverlayRGBTrans / _is5_OverlayRGBTrans

   Purpose : 
   Overlay one RGB onto another with a transparency
   color. 
   
   Can also use RGBA images for top (or bottom), but alpha
   data is ignored.

   Param                Use
   ------------------------------------
   pRGB1                bottom image. RGB-24 (or RGBA-32)
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height 
   uBottomRowStride     bytes per pixel row, bottom
   pRGB2                top image. RGB-24 (or RGBA-32)
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        bytes per pixel row, top
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)
   fOpacity             opacity of top on bottom (> 0, < 1)
   clrTrans             color in top to make transparent
   uTolerance           maximum color distance to match

                        0 - only match specified color 
                           (strict color matching)

                        3 * (255 * 255) - maximum tolerance
                           (black matches white)

   uFlags               bit         purpose
                        ---         -------
                         0          favor speed over accuracy
                         1          images are in BGR order
                         2          bottom image is RGBA-32
                         3          top image is RGBA-32
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_OverlayRGBTrans )(BYTE *pRGB1, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBottomRowStride, BYTE *pRGB2, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, double fOpacity, COLORREF clrTrans, UINT32 uTolerance, UINT32 uFlags);

/*---------------------------------------------------------
   is5_OverlayRGBOnRGBA / _is5_OverlayRGBOnRGBA

   Purpose : 
   Overlay an RGB onto an RGBA image

   Param                Use
   ------------------------------------
   pRGBA                bottom image
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height 
   uBottomRowStride     bytes per pixel row, bottom
   pRGB                 top image
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        bytes per pixel row, top
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)

   fOpacity             opacity of top on bottom (> 0, < 1)
                        (0 = transparent, 1 = opaque)

   uFlags               unused
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_OverlayRGBOnRGBA )(BYTE *pRGBA, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBottomRowStride, BYTE *pRGB, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, double fOpacity, UINT32 uFlags);

/*---------------------------------------------------------
   is5_AlphaBlendRGB / _is5_AlphaBlendRGB

   Purpose : 
   Overlay one RGB onto another using an alpha mask, and a global
	transparency value, to control transparency.


   Param                Use
   ------------------------------------
   pRGB1                bottom image
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height
   uBottomRowStride     bytes per pixel row, bottom
   pRGB2                top image
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        bytes per pixel row, top
   pAlphaMask           8 bit alpha mask (uTopWidth x uTopHeight pixels)
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)

   fOpacity             opacity of top on bottom (> 0, < 1)
                        (0 = transparent, 1 = opaque)

   uFlags               unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_AlphaBlendRGB )(BYTE *pRGB1, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBottomRowStride, BYTE *pRGB2, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, BYTE *pAlphaMask, __int32 iXPos, __int32 iYPos, double fOpacity, UINT32 uFlags);

/*---------------------------------------------------------
   is5_AlphaBlendRGBA / _is5_AlphaBlendRGBA

   Purpose : 
   Alpha blend an RGBA image onto an RGB/RGBA image, with variable
	opacity. This function allows you to control the way the 
	bottom alpha channel is modified.

   Note:
	Mode 4 is probably the best choice when merging RGBA onto RGBA and 
	you want the resulting alpha to best reflect the combination of
	the two original alpha channels.

   Param                Use
   ------------------------------------
   pRGB                 bottom image
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height 
	uBottomChannels      channels per pixel row, bottom: 3 or 4 (RGB or RGBA)
   uBottomRowStride     bytes per pixel row, bottom
   pRGBA                top image (RGBA)
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        bytes per pixel row, top
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)

   fOpacity             opacity of top on bottom (> 0, < 1)
                        (0 = transparent, 1 = opaque)

	uMode				controls how the bottom image alpha values are modified
                        (if the bottom image has an alpha channel):
							0 : bottom alpha is unchanged
							1 : bottom alpha is set to top alpha
							2 : bottom alpha is the max of bottom and top alpha
							3 : bottom alpha is the sum of bottom and top alpha
							4 : alpha values are composited 
							5 : bottom alpha =  255 - ((255 - bottom_a)) * (255 - top_a)) / 255

   uFlags               unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL			_ISCConv _ISFn( is5_AlphaBlendRGBA )(BYTE *pImage1, UINT32 uWidth1, UINT32 uHeight1, UINT32 uBottomBPP, UINT32 uBottomRowStride, BYTE *pRGBA, UINT32 uWidth2, UINT32 uHeight2, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, double fOpacity, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is5_AlphaBlendColor / _is5_AlphaBlendColor

   Purpose : 
   Use an 8-bit mask image to alpha blend a solid color
   onto an output image.

   Note:
   Bits 1-5 of uFlags are mutually exclusive. 
   If more than one of bits 1-5 are set, the mode 
   associated with the highest set bit is used.

   Param                Use
   ------------------------------------
   pImage               image
   uWidth               width in pixels
   uHeight              height 
   uBytesPerPixel       bytes per pixel. 3 or 4!
   uRowStride           bytes per pixel row, bottom
   pAlpha               8-bit mask image. 0 = transparent, 255=opaque
   uAlphaWidth          mask width in pixels
   uAlphaHeight         mask height 
   uAlphaRowStride      bytes per pixel row, alpha
   clrBlend             color to blend. 
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)

   uFlags               bit      purpose
                        ---      -------
                         0       assume BGR byte order
                         1       blend alpha channel
                         2       set alpha channel to alpha of clrBlend where mask != 0
                         3       set alpha channel to mask value
								 4			set alpha to mask value. set RGB to mask color, not blended.
                         5       alpha values are composited

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_AlphaBlendColor )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, BYTE *pAlpha, UINT32 uAlphaWidth, UINT32 uAlphaHeight, UINT32 uAlphaRowStride, COLORREF clrBlend, __int32 iXPos, __int32 iYPos, UINT32 uFlags);

/*---------------------------------------------------------
   is5_AlphaBlend8BitGray / _is5_AlphaBlend8BitGray

   Purpose : 
   Overlay a Gray+Alpha (16 bit) image onto an 8-bit grayscale 
   image, or onto another Gray+Alpha 16-bit image

   Param                Use
   ------------------------------------
   pRGB                 bottom image
   uWidthRGB            bottom width in pixels
   uHeightRGB           bottom height 
   uBottomRowStride     bytes per pixel row, bottom
   pRGBA                top image (RGBA)
   uWidthRGBA           top width in pixels
   uHeightRGBA          top height 
   uTopRowStride        bytes per pixel row, top
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)

   uFlags               bit      purpose
                        ---      -------
                         0       image pixel components have 16 bits


   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_AlphaBlend8BitGray )(BYTE *pImage1, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBottomRowStride,BYTE *pImage2, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, UINT32 uFlags);

/*---------------------------------------------------------
   is5_OverlayImageDouble / _is5_OverlayImageDouble

   Purpose : 
   Overlay an image composed of doubles (not BYTEs) onto 
   another image (also of doubles), using various blending modes.

   Param                Use
   ------------------------------------
   pImageBottom         bottom image
   uBottomWidth         bottom width in pixels
   uBottomHeight        bottom height 
   uBottomRowStride     doubles per pixel row, bottom
   pTop                 top image (RGBA)
   uTopWidth            top width in pixels
   uTopHeight           top height 
   uTopRowStride        doubles per pixel row, top
   iXPos                position of top on bottom (may be < 0)
   iYPos                position of top on bottom (may be < 0)
   uComponentsPerPixel  number of doubles in a pixel
   fOpacity					opacity of blended image. 0 to 1; 0 = transparent, 1 = opaque
   fK							double parameter. use depends on uMode

   uMode					values:
								0,1 : combined with variable opacity
								5 - output pixel is max of top and bottom  (a.k.a. Photoshop 'Lighten')
								6 - output pixel is min of top and bottom   (a.k.a. Photoshop 'Darken')
								7 - output pixel is the sum of top and bottom
								8 - output pixel is difference of top and bottom
								9 - if top is > fK, output top
								10 - if top is < fK, output top 
								11 - absolute value of the difference of top and bottom
								12 - multiply: (top * bottom) / maxComponent
								13 - multiply with factor (top * bottom * fK) / maxComponent
								14 - screen
								15 - bottom = bottom + top - fK
								16 - bottom = bottom - top - fK

   uFlags               (unused)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_OverlayImageDouble )(double *pImageBottom, UINT32 uBottomWidth, UINT32 uBottomHeight, UINT32 uBottomRowStride,double *pTop, UINT32 uTopWidth, UINT32 uTopHeight, UINT32 uTopRowStride, __int32 iXPos, __int32 iYPos, UINT32 uComponentsPerPixel, double fOpacity, double fK, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is5_OverlayDIBOnDIB / _is5_OverlayDIBOnDIB

   Purpose : 
   Overlay one DIB on another.

   This can overlay a 1,4,8,15/16,24 or 32 bpp DIB onto
   a 24 or 32 bpp DIB. It can also overlay a 1-bit DIB onto
   another 1-bit DIB, or an 8-bit DIB onto another 8-bit DIB.

   No other bit depth combinations are supported.

   RLE-compressed DIBs are not supported.

   pInImage != pOutImage

   Param                Use
   ------------------------------------
   pDIBBottom           bottom DIB. 

   pDIBTop              top DIB. if 32 bits, the alpha channel in the top image is
                        used to control transparency.

   iXPos                x pos of top-left corner of top image on bottom image
   iYPos                y pos of top-left corner of top image on bottom image

   uFlags               bit      purpose
                        ---      -------
                         0       use alpha information in the top image (32 bits only)

   uFlags2              unused

   fK                   transparency 0..1 (0 = transparent, 1 = opaque)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_OverlayDIBOnDIB )(BITMAPINFOHEADER *pDIBBottom, BITMAPINFOHEADER *pDIBTop, __int32 iXPos, __int32 iYPos, UINT32 uFlags1, UINT32 uFlags2, double fK);


#ifdef __cplusplus
}
#endif

#endif