/*---------------------------------------------------------
   ISImageRotate.h : part of the ImgSource library.
   
   Image processing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is5_IMAGEROTATEH
#define is5_IMAGEROTATEH

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_RotateImage / _is5_RotateImage

   Purpose : 
   Rotate an image. Output buffer must be allocated by the 
   caller. See is5_GetRotatedBufSize and/or is5_GetRotatedPixelSize 
   to determine the size of the output buffer.

   Param                Use
   ------------------------------------
   pInBuf               source image
   uWidth               width in pixels
   uHeight              height
   
   uBytesPerPixel       1 = 8 bit colormapped
                        3 = RGB
                        4 = RGBA
   
   uInRowStride         bytes per pixel row, input

   pOut                 output buffer. allocated by the caller.

                        the caller should fill this buffer with a
                        color or pattern before calling this function.
                        that color or pattern will appear on the corners
                        of the image (where the rotated image doesn't cover).

   uOutRowStride        number of bytes in an output pixel row

   pPal                 palette for 8-bit
                        (not used when in "fast" mode)

   uPalColors           number of colors in pPal
   
   fRadians             rotation angle, in radians

   uFlags               bit         purpose
                        ---         -------
                        1-0         00 (uFlags = 0) - use a fast but 
                                    sloppy rotation algorithm. 
                                    
                                    01 (uFlags = 1) - use bi-linear interpolation.
                                    the results are far better than the
                                    fast method. however, this is much 
                                    slower, especially with 8-bit images.

                                    10 (uFlags = 2) - use bi-cubic interpolation.
                                    the results are better than bi-linear,
                                    but require even more time to process. this 
                                    method cannot be used with 8-bit images.

								20			disable multithreaded processing

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_RotateImage )(BYTE *inBuf, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, RGBQUAD *pPal, UINT32 uPalColors, double radians, UINT32 uFlags);

/*---------------------------------------------------------
   is5_RotateImageDouble / _is5_RotateImageDouble

   Purpose : 
   Rotate an image of doubles. 
   
   Output buffer must be allocated by the caller. See 
   is5_GetRotatedPixelSize to determine the size of the output buffer.

	Ex.
      UINT32 rotatedWidth = 0;
      UINT32 rotatedHeight = 0 ;
      is5_GetRotatedPixelSize(rads, sourceW, sourceH, &rotatedWidth, &rotatedHeight);

      UINT32 rotatedRowStride = rotatedWidth * uComponentsPerPixel;

      // alloc output
      double *pOut = new double[rotatedRowStride * rotatedHeight];

      // fill the buffer at "pOut" with 0's (black)
      memset(pOut, 0, rotatedRowStride * rotatedHeight * sizeof(double));

      // rotate
      is5_RotateImageDouble(pSource, sourceW, sourceH, 3, sourceW * 3, 
                     pOut, rotatedRowStride, 
                     rads, 0);


   Param                Use
   ------------------------------------
   pInBuf               source image
   uWidth               width in pixels
   uHeight              height
   
   uComponentsPerPixel  number of components per pixel
   
   uInRowStride         components per pixel row, input

   pOut                 output buffer. allocated by the caller.

                        the caller should fill this buffer with a
                        color or pattern before calling this function.
                        that color or pattern will appear on the corners
                        of the image (where the rotated image doesn't cover).

   uOutRowStride        number of components in an output pixel row

   fRadians             rotation angle, in radians

   uFlags               bit         purpose
                        ---         -------
                        1-0         00 (uFlags = 0) - use a fast but 
                                    sloppy rotation algorithm. 
                                    
                                    01 (uFlags = 1) - use bi-linear interpolation.
                                    the results are far better than the
                                    fast method. however, this is much 
                                    slower.

                                    10 (uFlags = 2) - use bi-cubic interpolation.
                                    the results are better than bi-linear,
                                    but require even more time to process.

								20			disable multithreaded processing
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_RotateImageDouble )(double *inBuf, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double radians, UINT32 uFlags);

/*---------------------------------------------------------
   is5_RotateOneBitImage / _is5_RotateOneBitImage

   Purpose : 
   Rotate a one-bit image. Output buffer must be allocated 
   by the caller. See is5_GetRotatedPixelSize to determine the 
   size of the output buffer.

   Param                Use
   ------------------------------------
   pInBuf               source image
   uWidth               width in pixels
   uHeight              height
   
   uInRowStride         bytes per pixel row, input

   pOut                 output buffer. allocated by the caller
   uOutRowStride        number of bytes in an output pixel row

   fRadians             rotation angle, in radians

   uFlags               bit         purpose
                        ---         -------
                        1-0         00 (uFlags = 0) - use bi-linear interpolation.
                                    the results are slightly better than the
                                    fast method. 

                                    01 (uFlags = 1) - use a fast but 
                                    sloppy rotation algorithm. 

								20			disable multithreaded processing
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_RotateOneBitImage )(BYTE *inBuf, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, double radians, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GetRotatedBufSize / _is5_GetRotatedBufSize

   Purpose : 
   Determine the number of bytes required to hold an image
   rotated by is5_RotateImage. 

   Param                Use
   ------------------------------------
   fRadians             rotation angle, in radians
   uWidth               source width in pixels
   uHeight              source height
   uBytesPerPixel       bytes per image pixel

   Return
   ------
   number of bytes
---------------------------------------------------------*/
_ISDeclSpec UINT32  	_ISCConv _ISFn( is5_GetRotatedBufSize )(double radians, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel);

/*---------------------------------------------------------
   is5_GetRotatedPixelSize / _is5_GetRotatedPixelSize

   Purpose : 
   Determine the number of pixels required to hold an image
   rotated by is5_RotateImage. 

   ex.
      UINT32 rotatedWidth = 0;
      UINT32 rotatedHeight = 0 ;
      is5_GetRotatedPixelSize(rads, sourceW, sourceH, &rotatedWidth, &rotatedHeight);

      UINT32 rotatedRowStride = rotatedWidth * uBytesPerPixel;

      // alloc output
      BYTE *p = new BYTE[rotatedRowStride * rotatedHeight];

      // fill the buffer at "p" with 0's (black)
      memset(p, 0, rotatedRowStride * rotatedHeight);

      // rotate
      is5_RotateImage(pSource, sourceW, sourceH, 3, sourceW * 3, 
                     p, rotatedRowStride, 
                     NULL, 0, rads, 0);

   Param                Use
   ------------------------------------
   fRadians             rotation angle, in radians
   uWidth               source width in pixels
   uHeight              source height
   pWidth               receives output width (in pixels)
   pHeight              recieves output height

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_GetRotatedPixelSize )(double radians, UINT32 uWidth, UINT32 uHeight, UINT32 *pWidth, UINT32 *pHeight);

/*---------------------------------------------------------
   is5_QuickRotateImageInPlace / _is5_QuickRotateImageInPlace

   Purpose : 
   Rotate an image in multiples of 90 degrees
   This performs the rotation in-place. Be sure to 
   note that the image dimensions will change when
   rotating 90 or 270 degrees! (width and height will 
   reverse)

   Param                Use
   ------------------------------------
   pImage               source image
   uWidth               width in pixels
   uHeight              height

   uRotate              rotation selector
                           0 - 90 degrees cw
                           1 - 180 deg
                           2 - 270 deg cw

   uChannelsPerPixel    components per image pixel

   uFlags            bit      meaning
                     ---      -------
                     9        images have 16 bits per component


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_QuickRotateImageInPlace )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uRotate, UINT32 uChannels, UINT32 uFlags);

/*---------------------------------------------------------
   is5_QuickRotateImage / _is5_QuickRotateImage

   Purpose : 
   Rotate an image in multiples of 90 degrees.
   
     
   Be sure to note that the image dimensions will change 
   when rotating 90 or 270 degrees! (width and height will 
   reverse)

   Param                Use
   ------------------------------------
   pImage               source image
   uWidth               width in pixels
   uHeight              height
   uInRowStride         components per input pixel row
   pOutput              output image, allocated by the caller
   uOutRowStride        components per pixel row, output

   uRotate              rotation selector
                           0 - 90 degrees cw
                           1 - 180 deg
                           2 - 270 deg cw

   uChannelsPerPixel    components per image pixel

   uFlags            bit      meaning
                     ---      -------
                     9        images have 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      	_ISCConv _ISFn( is5_QuickRotateImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOutput, UINT32 uOutRowStride, UINT32 uRotate, UINT32 uChannelsPerPixel, UINT32 uFlags);

/*---------------------------------------------------------
   is5_QuickRotateImageInPlaceDouble / _is5_QuickRotateImageInPlaceDouble

   Purpose : 
   Rotate an image made of doubles in multiples of 90 degrees
   This performs the rotation in-place. Be sure to 
   note that the image dimensions will change when
   rotating 90 or 270 degrees! (width and height will 
   reverse)

   Param                Use
   ------------------------------------
   pImage               source image, doubles
   uWidth               width in pixels
   uHeight              height

   uRotate              rotation selector
                           0 - 90 degrees cw
                           1 - 180 deg
                           2 - 270 deg cw

   uChannelsPerPixel    components per image pixel

   uFlags				unused


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_QuickRotateImageInPlaceDouble )(double *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uRotate, UINT32 uChannels, UINT32 uFlags);

/*---------------------------------------------------------
   is5_QuickRotateImageDouble / _is5_QuickRotateImageDouble

   Purpose : 
   Rotate an image made of doubles in multiples of 90 degrees.
   
   Be sure to note that the image dimensions will change 
   when rotating 90 or 270 degrees! (width and height will 
   reverse)

   Param                Use
   ------------------------------------
   pImage               source image, doubles
   uWidth               width in pixels
   uHeight              height
   uInRowStride         components per input pixel row
   pOutput              output image, allocated by the caller
   uOutRowStride        components per pixel row, output

   uRotate              rotation selector
                           0 - 90 degrees cw
                           1 - 180 deg
                           2 - 270 deg cw

   uChannelsPerPixel    components per image pixel

   uFlags				unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      	_ISCConv _ISFn( is5_QuickRotateImageDouble )(double *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOutput, UINT32 uOutRowStride, UINT32 uRotate, UINT32 uChannelsPerPixel, UINT32 uFlags);

/*---------------------------------------------------------
   is5_QuickRotate1Bit / _is5_QuickRotate1Bit

   Purpose : 
   Rotate a 1-bit image multiples of 90 degrees.

   Note:
   When rotating 90 or 270 degrees, output width
   and height are swapped, with respect to input
   width & height.

   Note:
   The source and destination image buffers will not
   be the same size.
   

   Param                Use
   ------------------------------------
   pInBuf               source image
   uWidth               source width in pixels
   uHeight              source height
   uInRowStride         bytes per input pixel row
   pOutBuf              output image
   uOutRowStride        bytes per output pixel row

   uRotate              rotation mode
                           0 - 90 degrees cw
                           1 - 180 deg
                           2 - 270 deg cw

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_QuickRotate1Bit )(BYTE *pInBuf, UINT32 nInWidth, UINT32 nInHeight, UINT32 uInRowStride, BYTE *pOutBuf, UINT32 uOutRowStride, UINT32 uRotate);

/*---------------------------------------------------------
   is5_ShearImageX / _is5_ShearImageX

   Purpose : 
   Shear an image. This slides the top (or bottom)
   of an image by xOffset pixels. (width increases)

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel in source image (1, 3 or 4)
   uInRowStride      number of bytes in source image row

   pOut              output buffer - allocated by caller
                     buffer must be at least (uWidth + abs(xOffset)) * uHeight * uBytesPerPixel  BYTEs 

   xOffset           if > 0, the top of the image is offset and this value is
                     the distance to offset the top row of the output 
                     image from the top-left corner.
                     if < 0, the bottom of the image is offset and this value 
                     is the distance to offset the bottom row from the 
                     bottom-left corner.

   uOutRowStride     number of bytes in output image row

   clrBack           color to fill the background
   bFast             if TRUE, use a fast but less accurate method
   pPal              palette, for 8-bit images. ignored if bFast is TRUE
   uPalColors        colors in pPal

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_ShearImageX )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, __int32 xOffset, UINT32 uOutRowStride, COLORREF clrBack, BOOL bFast, RGBQUAD *pPal, UINT32 uPalColors);

/*---------------------------------------------------------
   is5_ShearImageY / _is5_ShearImageY

   Purpose : 
   Shear an image. This slides the left (or right)
   of an image by yOffset pixels. (height increases)


   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel in source image (1, 3 or 4)
   uInRowStride      number of bytes in source image row

   pOut              output buffer - allocated by caller
                     buffer must be uInRowStride * (uHeight + abs(yOffset)) BYTEs 

   yOffset           if > 0, the left of the image is offset and this value is
                     the distance to offset the leftmost column of the output 
                     image from the top-left corner.

                     if < 0, the right of the image is offset and this value 
                     is the distance to offset the rightmost column from the 
                     top-right corner.
               
   uOutRowStride     number of bytes in output image row

   clrBack           color to fill the background
   bFast             if TRUE, use a fast but less accurate method
   pPal              palette, for 8-bit images. ignored if bFast is TRUE
   uPalColors        colors in pPal

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_ShearImageY )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, __int32 yOffset, UINT32 uOutRowStride, COLORREF clrBack, BOOL bFast, RGBQUAD *pPal, UINT32 uPalColors);

/*---------------------------------------------------------
   is5_ShearOneBitX / _is5_ShearOneBitX

   Purpose : 
   Shear an image. This slides the top (or bottom)
   of an image by xOffset pixels. (width increases)

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      number of bytes in source image row

   pOut              output buffer - allocated by caller
                     buffer must be at least (uWidth + abs(xOffset)) * uHeight * uBytesPerPixel  BYTEs 

   xOffset           if > 0, the top of the image is offset and this value is
                     the distance to offset the top row of the output 
                     image from the top-left corner.
                     if < 0, the bottom of the image is offset and this value 
                     is the distance to offset the bottom row from the 
                     bottom-left corner.

   uOutRowStride     number of bytes in output image row

   uFlags            bit         purpose
                     ---         -------
                      0          value to put in image background
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_ShearOneBitX )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, __int32 xOffset, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is5_ShearOneBitY / _is5_ShearOneBitY

   Purpose : 
   Shear a one bit per pixel image. This slides the left (or right) side
   of an image by yOffset pixels. (height increases)

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      number of bytes in source image row

   pOut              output buffer - allocated by caller
                     buffer must be uInRowStride * (uHeight + abs(yOffset)) BYTEs 

   yOffset           if > 0, the left of the image is offset and this value is
                     the distance to offset the leftmost column of the output 
                     image from the top-left corner.

                     if < 0, the right of the image is offset and this value 
                     is the distance to offset the rightmost column from the 
                     top-right corner.
               
   uOutRowStride     number of bytes in output image row

   uFlags            bit         purpose
                     ---         -------
                      0          value to put in image background
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     	_ISCConv _ISFn( is5_ShearOneBitY )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, __int32 yOffset, UINT32 uOutRowStride, UINT32 uFlags);

#define IS_ROTATE_16BPC (1<<9)

#ifdef __cplusplus
}
#endif
#endif