/*---------------------------------------------------------

   ISTransformEffects : part of the ImgSource library.

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef ISVECTOREFFECTSH
#define ISVECTOREFFECTSH

#if !defined is5_SOURCEH && !defined is5_INTERNAL
#error Do not include this file directly. Include ISource.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is5_AddSpinTransform
                                 
   Purpose : 
   Add a spin tranform object to the transformation chain.
   
   A spin transform rotates the image about the center with
   a constant angular distance. 

   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMaxRadius        maximum distance from the center for this effect.
                     pixels past this distance will not be affected.

   fMaxAngleOffset   angular distance to spin pixels (degrees)
   iOffsetX          pixels to offset from center
   iOffsetY          pixels to offset from center


   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddSpinTransform )(UINT32 uWidth, UINT32 uHeight, double fMaxRadius, double fMaxAngleOffsetDeg, int iOffsetX, int iOffsetY);

/*---------------------------------------------------------
   is5_AddSpinWaveTransform

   Purpose : 
   Add a spin wave tranform object to the transformation chain.
   
   A spin wave transform rotates the image about the center with
   a variable angular distance. The distance varies in a sine wave
   pattern.

   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMaxRadius        maximum distance from the center for this effect.
                     pixels past this distance will not be affected.

   fMaxAngleOffset   angular distance to spin pixels (degrees)
   
   fPeriods          number of periods in the distance wave
   iOffsetX          pixels to offset from center
   iOffsetY          pixels to offset from center

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddSpinWaveTransform )(UINT32 uWidth, UINT32 uHeight, double fMaxRadius, double fMaxAngleOffsetDeg, double fPeriods, int iOffsetX, int iOffsetY);

/*---------------------------------------------------------
   is5_AddEllipticalTransform

   Purpose : 
   Add a elliptic tranform object to the transformation chain.
   
   This will compress the image into the largest ellipse that fits
   the image dimensions. It can compress in the horizontally, verticaly
   or both. If both, and the image is square, this acts as a lens.
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fRadiusX          horizontal radius
   fRadiusY          vertical radius
  
   iOffsetX          pixels to offset from center
   iOffsetY          pixels to offset from center

   uMode             one of:
                     0: compress horizontally
                     1: compress vertically
                     2: compress in both directions

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddEllipticalTransform )(UINT32 uWidth, UINT32 uHeight, double fRadiusX, double fRadiusY, int iXoffset, int iYOffset, UINT32 uMode);

/*---------------------------------------------------------
   is5_AddLensTransform

   Purpose : 
   Add a lens tranform object to the transformation chain.
   
   A lens transform performs a roughly spherical distortion
   of the image.
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMaxRadius        maximum distance from the center for this effect.
                     pixels past this distance will not be affected.
  
   iOffsetX          pixels to offset from center
   iOffsetY          pixels to offset from center

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddLensTransform )(UINT32 uWidth, UINT32 uHeight, double fMaxRadius, int iOffsetX, int iOffsetY);

/*---------------------------------------------------------
   is5_AddPinchTransform

   Purpose : 
   Add a pinch tranform object to the transformation chain.
   
   A pinch transform attracts pixels towards the center of the image.
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMaxRadius        maximum distance from the center for this effect.
                     pixels past this distance will not be affected.   

   fAmount           intensity  (1 = no effect)

   iOffsetX          pixels to offset from center
   iOffsetY          pixels to offset from center

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddPinchTransform )(UINT32 uWidth, UINT32 uHeight, double fMaxRadius, double fAmount, int iOffsetX, int iOffsetY);

_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddPointAttractorTransform )(UINT32 uWidth, UINT32 uHeight, double fAmount, int iOffsetX, int iOffsetY, int iMode, double fK);

/*---------------------------------------------------------
   is5_AddRippleTransform

   Purpose : 
   Add an ripple tranform object to the transformation chain.
   
   A ripple transform creates circular waves in the output image
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMaxRadius        maximum distance from the center for this effect.
                     pixels past this distance will not be affected.   

   fAmount           intensity (1 = no effect)
   fPeriods          number of periods in the distance wave
   iOffsetX          pixels to offset from center
   iOffsetY          pixels to offset from center

   iDampenStyle      0: waves remain at full intensity
                     1: waves diminish linearly with their distance from center
                     2: waves diminish with the square of their distance
                     3: waves diminish with the sine of their distance (inverse!)
                     4: waves diminish with the cosine of their distance

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddRippleTransform )(UINT32 uWidth, UINT32 uHeight, double fMaxRadius, double fAmount, double fPeriods, double fOffsetX, double fOffsetY, int iDampenStyle);

/*---------------------------------------------------------
   is5_AddOffsetTransform

   Purpose : 
   Add an offset tranform object to the transformation chain.
   
   The offset transform slides the image by a specified amount.
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   xOffset           amount to slide in x direction
   yOffset           amount to slide in y direction
   iStyle            0: don't wrap the image at the edges
                     1: wrap in both X and Y
                     2: wrap only in X
                     3: wrap only in Y

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddOffsetTransform )(UINT32 uWidth, UINT32 uHeight, double xOffset, double yOffset, int iStyle);

/*---------------------------------------------------------
   is5_AddRandomTransform

   Purpose : 
   Add a random tranform object to the transformation chain.
   
   A random transform offsets pixels by a random amount.
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fAmount           maximum number of pixels allowed from original

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddRandomTransform )(UINT32 uWidth, UINT32 uHeight, double fAmount);

/*---------------------------------------------------------
   is5_AddLineSliceTransform

   Purpose : 
   Add a line slice tranform object to the transformation chain.
   
   A line slice is the simplest of the center-shift transforms. It
   produces a center-shift transform which runs from one side of the
   image to the other, in straight line.

   A center shift transform works by running a function through 
   the image rectangle in order to produce one point per line 
   (or column, if running in the Y direction). This point is 
   then used as a new center for each line. Pixels rows are split 
   down the middle and the left half are mapped to the left side 
   of the new center point, pixels from the right side are mapped 
   to the new right side. 
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   iOffset1          line endpoint 1
   iOffset2          line endpoint 2
   eStyle            0 or 1
                     0: the line runs from the left edge to the right edge.
                     iOffset1 is the Y coordinate of the left endpoint
                     iOffset2 is the Y coordinate of the right endpoint
  
                     1: the line runs from the top edge to the bottom edge.
                     iOffset1 is the X coordinate of the top endpoint
                     iOffset2 is the X coordinate of the bottom endpoint

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddLineSliceTransform )(UINT32 uWidth, UINT32 uHeight, int iOffset1, int iOffset2, int eStyle);

/*---------------------------------------------------------
   is5_AddZigZagTransform

   Purpose : 
   Add a zig zag tranform object to the transformation chain.
   
   A zig zag transform moves across the image with a triangle 
   wave pattern.

   A center shift transform works by running a function through 
   the image rectangle in order to produce one point per line 
   (or column, if running in the Y direction). This point is 
   then used as a new center for each line. Pixels rows are split 
   down the middle and the left half are mapped to the left side 
   of the new center point, pixels from the right side are mapped 
   to the new right side. 
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMaxOffsetPercent maximum offset percent for the wave. 
                     50% will reach the edges of the image

   iPeriods          number of periods in the wave

   eStyle            0 or 1
                        0: The wave moves from the left side to the right side.
                           iPeriods must be less than uWidth

                        1: The wave moves from top to bottom
                           iPeriods must be less than uHeight
   
   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddZigZagTransform )(UINT32 uWidth, UINT32 uHeight, double fMaxOffsetPercent, int iPeriods, int eStyle);

/*---------------------------------------------------------
   is5_AddWaveTransform

   Purpose : 
   Add a wave tranform object to the transformation chain.
   
   A wave transform moves across the image with a sine 
   wave pattern.

   A center shift transform works by running a function through 
   the image rectangle in order to produce one point per line 
   (or column, if running in the Y direction). This point is 
   then used as a new center for each line. Pixels rows are split 
   down the middle and the left half are mapped to the left side 
   of the new center point, pixels from the right side are mapped 
   to the new right side. 
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMaxOffsetPercent maximum offset percent for the wave. 
                     50% will reach the edges of the image

   fPeriods          number of periods in the wave

   eStyle            0 to 3
                     0: The wave moves from left to right as a sine wave
                     1: The wave moves from top to bottom as a sine wave
                     2: The wave moves from left to right as a cosine wave
                     3: The wave moves from top to bottom as a cosine wave

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddWaveTransform )(UINT32 uWidth, UINT32 uHeight, double fMaxOffsetPercent, double fPeriods, int iStyle);

/*---------------------------------------------------------
   is5_AddWowTransform

   Purpose : 
   Add a "wow" tranform object to the transformation chain.
   
   A wow transform is a transform that can't be described any other way.
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMaxRadius        maximum distance from the center for this effect.
                     pixels past this distance will not be affected.
   eStyle            0 or 1

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddWowTransform )(UINT32 uWidth, UINT32 uHeight, double fMaxRadius, int eStyle);

/*---------------------------------------------------------
   is5_AddMarbleTransform

   Purpose : 
   Add a marble tranform object to the transformation chain.
   
   A marble transform creates a marbled, wavy, glassy, rippled effect.
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fAmount           controls smoothness of the effect (try between .5 and 1.5 to start)
   fXScale           controls size of the perlin noise (try between 3 and 7)
   fYScale           controls size of the perlin noise

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddMarbleTransform )(UINT32 uWidth, UINT32 uHeight, double fAmount, double fXScale, double fYScale);

/*---------------------------------------------------------
   is5_AddPerlinDisplacementTransform

   Purpose : 
   Add a Perlin displacement tranform object to the transformation chain.
   
   This effect is similar to the marble transform, but the image tends to break up less
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fAmount           controls smoothness of the effect (try between 5 and 15 to start)
   fXScale           controls size of the Perlin noise (try between 10 and 20 to start)
   fYScale           controls size of the Perlin noise

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddPerlinDisplacementTransform )(UINT32 uWidth, UINT32 uHeight, double fAmount, double fXScale, double fYScale);

/*---------------------------------------------------------
   is5_AddBumpMapDisplacementTransform

   Purpose : 
   Add a bump map displacement tranform object to the transformation chain.
   
   This effect uses an 8-bit image (supplied by the caller) as a displacement
   map. The marble and Perlin displacement transforms used Perlin noise and/or sine 
   waves to generate a displacement map internally. This function lets you
   supply your own. The output image is displaced based on the light/dark transitions
   in the bump map image.
   
   The bump map image is tiled over the input image.

   * The bump map image must remain valid until is5_TransformProcess is called.

   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fAmount           controls smoothness of the effect (try between 10 and 15 to start)
   pBMBuf            bump map image (8 bit grayscale)
   uBMWidth          size of the bump map image.
   uBMHeight

   
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddBumpMapDisplacementTransform )(UINT32 uWidth, UINT32 uHeight, double fAmount, BYTE *pBMBuf, UINT uBMWidth, UINT uBMHeight);

/*---------------------------------------------------------
   is5_AddPolygonTransform

   Purpose : 
   Add a polygon tranform object to the transformation chain.

   This effect resizes image rows (or columns) to fit inside the specified
   polygon. This polygon must be convex. If it is concave, the largest convex
   polygon that can be formed from the points will be used.
   
   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.


   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   pPoints           array of POINTs
   uNumPoints        number of points in the array

   uMode             one of:
                     0     compress horizontally
                     1     compress vertically
                     2     compress in both directions

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddPolygonTransform )(UINT32 uWidth, UINT32 uHeight, POINT *pPoints, UINT32 uNumPoints, UINT32 uMode);

/*---------------------------------------------------------
   is5_AddGlassBlockTransform

   Purpose : 
   Add a glass block tranform object to the transformation chain.

   This effect simulates looking at the image through a wall of glass blocks.
   Each block shows a small part of the source image, but distorts it enough
   so that it's difficult to see details in the source image.
   
   Blocks can have "beveled" edges, that simulate the refraction of angled
   glass.

   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   uBlockWidth       width of blocks
   uBlockHeight      height of blocks
   uBevelSize        bevel width
   uBevelOffset      controls bevel refraction (20 is a good start)
   bSmoothBevels     don't apply the distortion to the edge bevel

   fBlockMag               controls how much of the source image each 
                           block shows (0..1). higher values show smaller
                           portions of the image in each block.

   fDistortionFreq         controls the amount of distortion (0.03 is a good start)

   fDistortionIntensity    controls how much the distortion affects the image
                           in each block.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddGlassBlockTransform )(UINT32 uWidth, UINT32 uHeight, UINT32 uBlockWidth, UINT32 uBlockHeight, UINT32 uBevelSize, UINT32 uBevelOffset, BOOL bSmoothBevels, double fBlockMagnification, double fDistortionFrequency, double fDistortionIntensity);

/*---------------------------------------------------------
   is5_AddUserTransform

   Purpose : 
   Add a user callback tranform object to the transformation chain.

   This allows you to write your own transform functions and add them to the
   transform chain, just like the built-in transforms. To do this, you
   must supply a callback function that looks like this:

   // this will offset the image by 10 pixels in the X direction:
   BOOL CALLBACK MyXForm(const double fCurX, const double fCurY, double *pFromX, double *pFromY, const IS_CALLBACK_DATA uUserData)
   {
      // this says "the output pixel should be the source pixel at (fCurX - 10.5, fCurY)"
      *pFromX = fCurX - 10.5;
      *pFromY = fCurY;

      return TRUE;
   }

   This function will be called once for each pixel in the output image.

   You need to set the pFromX and pFromY variables to the pixels (may be fractional)
   that the input X and and Y values will be pulled from. This is an inverse transform.
   Again, you tell the transform processor where to pull the source pixel *from*.
   
   Your function should be able to handle any kind of input range; inputs
   may be outside the image rectangle, if you are chaining tranforms. Likewise,
   your function can output values that are outside the image rectangle. If your
   function is the last in the chain, values outside the image rect will be set to the
   background color.

   Notes:
      * The transform is performed when you call is5_TransformProcess
      * The width & height you specify when you add the transforms
      much match the width & height of the image used when you call 
      is5_TransformProcess.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   pCallbackFn       user-defined callback function
   uUserData         32-bit user-defined data

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_AddUserTransform )(UINT32 uWidth, UINT32 uHeight, ISXFORMCALLBACK pCallbackFn, IS_CALLBACK_DATA uUserData);
                         
/*---------------------------------------------------------
   is5_TransformProcess

   Purpose : 
   Apply all transformations in the transform chain to the image.

   Notes:
      * pIn != pOut
      * Output is allocated by the caller
      * Each transform that you have added will be applied.
      * You can add as many transforms as you like.
      * To remove transforms, use is5_ClearAllTransforms.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.
      * The width & height you specify when you add the transforms
      must match the width & height of the image used here.
      * It is generally faster to chain transformations than perform them
      one at a time.
   
      * Not all chains make sense. You can safely chain multiple 
      center-slice transforms with no strange effects. But, if you 
      chain the other effects (spin, spin-wave, wow, lens, pinch, etc.)
      you may get results that aren't what you expected - pixels will be
      pulled from unexpected areas on the image, in places where the
      transform has run off the edge. To do multiple non-center-slice
      transforms, run each transform in its own call to 
      is5_TransformProcess.

   Param             Use
   ------------------------------------
   pIn               input image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of pixels in image
   uInRowStride      number of bytes in an input image row
   pOut              output image
   uOutRowStride     number of bytes in an output image row
   uMode             0 to 2
                     0: no interpolation - fast, ugly
                     1: bi-linear interpolation. better
                     2: bi-cubic interpolation. slow. best.
                     
                     if you set bit #3 (logical OR'd with the mode), interpolated pixels 
                     will be clipped to the image dimensions. this will limit the
                     use of the background color on edges.

   clrBack           background color if the transform goes off the image.
                     if uBytesPerPixel = 1, the background color is the Red value
                     if uBytesPerPixel = 2, the background colors are the R and G values
                     if uBytesPerPixel = 3, the background colors are the R, G B values
                     if uBytesPerPixel = 4, the background colors are the R, G, B, Reserved values
                     if uBytesPerPixel > 4, the background color is 0

   uFlags          bit     purpose
                   ---     -------
                   20      disable multithreaded processing
                
   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_TransformProcess )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, UINT32 uMode, COLORREF clrBack, UINT32 uFlags);

/*---------------------------------------------------------
   is5_GetTransformDimensions

   Purpose : 
   Find the range of pixels that will be accessed by the current 
   transformation chain. 
   
   Some transforms may look at pixel locations outside the image 
   rectangle. When this happens, the is5_TransformProcess renderer 
   will use the background color as the input pixel. But, if you
   want to do something more complex than just a single background
   color, you can overlay your input image onto a background image
   of your own creation. This function can tell you how large to make
   the background image, and where to position your input image.

   Notes:
      * pIn != pOut
      * Output is allocated by the caller
      * Each transform that you have added will be applied.
      * You can add as many transforms as you like.
      * To remove transforms, use is5_ClearAllTransforms.
      * You can apply the same transform to multiple images, as long
      the images are of the same width/height.
      * The width & height you specify when you add the transforms
      much match the width & height of the image used here.
      * It is generally faster to chain transformations than perform them
      one at a time.
   
      * Not all chains make sense. You can safely chain multiple 
      center-slice transforms with no strange effects. But, if you 
      chain the other effects (spin, spin-wave, wow, lens, pinch, etc.)
      you may get results that aren't what you expected - pixels will be
      pulled from unexpected areas on the iamge, in places where the
      transform has run off the edge. To do multiple non-center-slice
      transforms, run each transform in its own call to 
      is5_TransformProcess.

   Param             Use
   ------------------------------------
   uWidth            width in pixels
   uHeight           height
   fMinX             pixels locations that were accessed by the transform chain
   fMinY
   fMaxX
   fMaxY

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is5_GetTransformDimensions )(UINT32 uWidth, UINT32 uHeight, double *fMinX, double *fMinY, double *fMaxX, double *fMaxY);

/*---------------------------------------------------------
   is5_ClearAllTransforms

   Purpose : 
   Remove all transforms from the chain. You should call 
   this when you're done with the transforms. This will
   release the memory associated with the transform objects.

   
   
---------------------------------------------------------*/
_ISDeclSpec void _ISCConv _ISFn( is5_ClearAllTransforms )();

#ifdef __cplusplus
}
#endif

#endif