/*---------------------------------------------------------
   ISource.h : part of the ImgSource library.
   
   Main ImgSource header file

   This is the only file an application needs to #include

   C/C++ users, see note below about Exported Symbols.

   For ImgSource build configuration options, see ISConfig.h and ISTypes.h.

   Credits:
      Color quantization code modified from DL1Quant, by 
      Dennis Lee (denlee@ecf.utoronto.ca)

      JPEG code from IJG v6b.
      PNG code from LibPng
      TIFF code from LibTiff v3.5.7
      Compression from ZLib
      TGA code from Timothy Bish
      PCX code from Bob Johnson's PCXLib
      Some crypto from Wei Dai's CryptLib 3.1
      PAX code from Smaller Animals Software's PALib
      TLA code from Smaller Animals Software's TLALib
      RGBA Color quantization code modified from PNGQuant, by Greg Roelofs.

      All other code, Smaller Animals Software. 

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is5_SOURCEH
#define is5_SOURCEH

///////////////////////////////////////////////////////
// Exported Symbols
//
// C/C++ users of the DLL must #define IS50_USING_DLL (see below).
//
// Also, when using the DLL, all function names will start with "_" : 
//    Example: _is5_Initialize
//
// Users of the static libs should #undef IS50_USING_DLL, or simply comment-out the line below.
// When using the static libraries, function names will NOT start with "_"
//    Example: is5_Initialize
//
// If you don't define IS50_USING_DLL correctly, you will get undefined symbol errors
// when you try to link.
//
///////////////////////////////////////////////////////

//#define IS50_USING_DLL

#ifdef __cplusplus
extern "C" {
#endif

///////////////////////////////////////////////////////
// source / destination manager function defintions
#include "ISTypes.h"

///////////////////////////////////////////////////////
// source / destination manager function defintions
#include "ISErrorCodes.h"

///////////////////////////////////////////////////////
// source / destination manager function defintions
#include "ISIOManagers.h"

////////////////////////////////////////////////////////
// file write function defintions
#include "ISFilesW.h"

////////////////////////////////////////////////////////
// single line file write function defintions
#include "ISFilesWL.h"

////////////////////////////////////////////////////////
// file read function defintions
#include "ISFilesR.h"

////////////////////////////////////////////////////////
// extended TIFF function defintions
#include "ISFilesTIFFExt.h"

////////////////////////////////////////////////////////
// single line file read function defintions
#include "ISFilesRL.h"

////////////////////////////////////////////////////////
// file dimension / bit depth function defintions
#include "ISFileDims.h"

////////////////////////////////////////////////////////
// file utilities (DPI, text, etc) function defintions
#include "ISFileUtil.h"

////////////////////////////////////////////////////////
// file utilities (DPI, text, etc) function defintions for JPEG
#include "ISFileUtilJPG.h"

////////////////////////////////////////////////////////
// image drawing function defintions
#include "ISImageDraw.h"

////////////////////////////////////////////////////////
// image processing function defintions
#include "ISImageProcess.h"

////////////////////////////////////////////////////////
// image histogram processing function defintions
#include "ISImageEnhance.h"

////////////////////////////////////////////////////////
// image resizing function defintions
#include "ISImageResize.h"

////////////////////////////////////////////////////////
// image rotation function defintions
#include "ISImageRotate.h"

////////////////////////////////////////////////////////
// image overlay function defintions
#include "ISImageOverlay.h"

////////////////////////////////////////////////////////
// image text draw function defintions
#include "ISImageText.h"

////////////////////////////////////////////////////////
// image effects function defintions
#include "ISImageEffects.h"
	
////////////////////////////////////////////////////////
// image FFT defintions
#include "ISImageFFT.h"
	
////////////////////////////////////////////////////////
// image morphology function defintions
#include "ISImageMorpho.h"

////////////////////////////////////////////////////////
// noise generation 
#include "ISImageNoise.h"

////////////////////////////////////////////////////////
// image filter function defintions
#include "ISImageFilter.h"

////////////////////////////////////////////////////////
// image transformation function defintions
#include "ISImageTransformEffects.h"

////////////////////////////////////////////////////////
// EXIF data parser
#include "ISEXIF.h"

////////////////////////////////////////////////////////
// IPTC data parser
#include "ISIPTC.h"

////////////////////////////////////////////////////////
// PSD resource manager
#include "ISPSDResources.h"

////////////////////////////////////////////////////////
// Fast Fourier Transform functions
#include "ISImageFFT.h"
										
////////////////////////////////////////////////////////
// gradient fill function defintions
#include "ISGradientFill.h"

////////////////////////////////////////////////////////
// Generic zLib compression
#include "ISZLibComp.h"

////////////////////////////////////////////////////////
// general library function defintions

/*---------------------------------------------------------
   is5_GetImgSourceVersion / _is5_GetImgSourceVersion

   Purpose : 
   Get ImgSource version string

   Param             Use
   ------------------------------------
   pVers             receives version string   
                     must be at least 20 chars.

---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_GetImgSourceVersion )(char *buf);

/*---------------------------------------------------------
   is5_GetLastError / _is5_GetLastError

   Purpose : 
   Get error code set by last function.

   All functions in ImgSource set the global error value.
   It is recommended that you test the error value often.

   Return
   ------
   Error value
---------------------------------------------------------*/
_ISDeclSpec UINT32    	_ISCConv _ISFn( is5_GetLastError )();

/*---------------------------------------------------------
   is5_SetLastError / _is5_SetLastError

   Purpose : 
   Set the global ImgSource error code.

   Param                Use
   ------------------------------------
   error                error value to set
   pCallerFunctionName  name of calling function
---------------------------------------------------------*/
_ISDeclSpec void      	_ISCConv _ISFn( is5_SetLastError )(int error, const char *pCallerFunctionName);

/*---------------------------------------------------------
   is5_GetErrorList / _is5_GetErrorList

   Purpose : 
   ImgSource maintains a list of the last 50 errors it encountered.
   This function retrieves that list as a single string with
   errors separated by new lines.

    The string is of this form:

        Error 2 in is5_OpenFileSource (12:32:02)
        Error 5 in is5_DrawRGB (12:32:02)

    Also, there may be an optional bit of text on the end of the string.
    Ex.:
        Error #5 in is5_ReadJPG (04:22:04) [unsupported bit depth]

    The information in the parenthesis is the current system
    time in HH:MM:SS format.

    Param               Use
    ------------------------------------
    hOutData            receives HGLOBAL for the error string
    puLen               receives number of bytes allocated to the error string

    uFlags              unused
---------------------------------------------------------*/
_ISDeclSpec void  	    _ISCConv _ISFn( is5_GetErrorList )(HGLOBAL *hOutData, UINT32 *puLen, UINT32 uFlags);

/*---------------------------------------------------------
   is5_SetCallback / _is5_SetCallback

   Purpose : 
   Set global callback function. This function is called
   by all file read / write operations, and most image
   processing calls. It is not called by image display
   functions.

   Param             Use
   ------------------------------------
   pCallback         pointer to callback function

   The callback function must be a function of this type :
   
      BOOL CALLBACK MyCallbackFunction(const UINT32 uCurRow, 
                              const UINT32 uTotalRows,
                              const IS_CALLBACK_DATA uUserData)
      {
         if (time to quit)
         {
            return FALSE;
         }

         return TRUE;
      }

   Set it with:

      is5_SetCallback(MyCallbackFunction, uUserData);

   The library will also call the callback function once 
   per row for most image processing calls. For some complex
   operations, it may be called a fixed number of times, 
   regardless of image size.
   
   Operations which do not operate on image buffers are not 
   "image processing calls" and so, these won't use the callback 
   function. This includes calls such as JPG and PNG text calls, 
   PNG gamma and background operations, DPI queries, error 
   queries, etc.. 
   
   Basically, if there is no loop over pixels, there is no 
   callback call. However, this does not mean that *all* 
   operations which do have a pixel loop use the callback 
   function; some don't. Check the function comments in this 
   file, to be sure.

   If the callback function returns FALSE the operation aborts and 
   is5_GetLastError will return IS_ERR_CALLBACKCANCEL

   You can use this to monitor the progress of ImgSource operations.

   It is not neccessary for you to use this function. If you 
   do not set the callback function or pass a NULL to this call, 
   no callback will be performed.

   The callback is thread safe. Each thread can have its own 
   callback function.

   Parameter            Use
   ------------------------------------
   pCallback            pointer to callback function
   uUserData            user data. this data is thread safe.

   Return
   ------
   Error value
---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_SetCallback )(ISCALLBACKPTR pCallback, IS_CALLBACK_DATA uUserData);

/*---------------------------------------------------------
   is5_GetCallback / _is5_GetCallback

   Purpose : 
   Retrieve the callback state.

   Parameter            Use
   ------------------------------------
   puUserData           receives user data. 

   Return
   ------
   pointer to callback function
---------------------------------------------------------*/
_ISDeclSpec ISCALLBACKPTR     	_ISCConv _ISFn( is5_GetCallback )(IS_CALLBACK_DATA *puUserData);

/*---------------------------------------------------------
   is5_Initialize / _is5_Initialize

   Purpose : 
   Initialize the library. 
   
   You should call is5_Exit to free any resources allocated
   by this library.

   Param             Use
   ------------------------------------
   pKey              library unlock key
---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_Initialize )(const char *pKey);

/*---------------------------------------------------------
   is5_ThreadExit / _is5_ThreadExit

   Purpose : 
   Free any ImgSource resources associated with this
   thread.
   
   These resources include JPG and PNG metadata, error
   state info and thread-local storage areas (TLS).
   
   This function is useful in applications that are heavily 
   multithreaded. If you do not call this when your threads
   ( *threads* , not process) exit, the resources associated
   with them will remain in memory until your process exits.

   To repeat: 
   It it not a fatal mistake to not call this when a thread
   exits. It simply means any resources allocated for the thread 
   will remain allocated until the parent process exits.

   It is safe to call this even if the thread has not yet called
   any ImgSource functions.

   This should be the last ImgSource call that your thread makes.

   This function is not available in the DLL. The DLL
   will handle thread cleanup by itself.
---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_ThreadExit )();

/*---------------------------------------------------------
   is5_Exit / _is5_Exit

   Purpose : 
   Free any resources allocated by this library. This is
   a process-level function. For thread resource release,
   see is5_ThreadExit.
   
   Do not call this unless you've called is5_Initialize first!
   
   This is not required for the DLL version, as this is
   called on exit in DllMain.

---------------------------------------------------------*/
_ISDeclSpec void     	_ISCConv _ISFn( is5_Exit )();

/*---------------------------------------------------------
   is5_GetExtHook / _is5_GetExtHook

   Purpose : 
   Get a hook object to initialize an add-on library. 
   
   The hook object allows the add-on library to access ImgSource's
   I/O functions (to allow files to be read and written), 
   provides access to the progress callbacks (is5_SetCallback),
   provides access to the file reading memory allocation callbacks
   (is5_SetFileReadAllocCallbacks) and provides access to the
   ImgSource global error value. Add-on libraries will use 
   these services in their own operation.

   Usage:

   HISEXTHK hHook = is5_GetExtHook(0);
   AddOnLibrary_SetImgSourceHook(hHook);
   GlobalFree(hHook);

   Param             Use
   ------------------------------------
   uFlags            unused

   Return
   ------
   A HISEXTHK object to be passed to an add-on library.
   After passing this to the add-on library, you must free this
   memory by using GlobalFree.

---------------------------------------------------------*/
_ISDeclSpec HISEXTHK  	_ISCConv _ISFn( is5_GetExtHook )(UINT32 uFlags);

/*---------------------------------------------------------
   is5_SetDebugMessageLevel / _is5_SetDebugMessageLevel

   Purpose : 
   Control the level of debug messages shown by the library.
   These messages are written to the debug console (they are
   not popup windows or logfile messages).

   This only affects debug builds, since no debug messages
   are shown in release builds.

   Param             Use
   ------------------------------------
   uLevel            0: show all messages - verbose
                     1: show informational, warning and critical messages
                     2: show warning and critical messages (default)
                     3: show critical messages only
   
---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SetDebugMessageLevel )(UINT32 uLevel);

/*---------------------------------------------------------
   is5_SwapLong / _is5_SwapLong

   Purpose : 
   Convert a 32-bit integer from Intel byte order to Motorola
   byte order, and vice versa.
   
   Param             Use
   ------------------------------------
   pLong             32-bit integer to swap

   Return
   ------
   none
---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SwapLong )(UINT32 *pLong);

/*---------------------------------------------------------
   is5_SwapShort / _is5_SwapShort

   Purpose : 
   Convert a 16-bit integer from Intel byte order to Motorola
   byte order, and vice versa.
   
   Param             Use
   ------------------------------------
   pShort            16-bit integer to swap

   Return
   ------
   none
---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_SwapShort )(ISUINT16 *pShort);

/*---------------------------------------------------------
   is5_DisableMMXFunctions / _is5_DisableMMXFunctions

   Purpose : 
   By default, ImgSource will use MMX/SSE-optimized functions
   wherever possible. This function will disable those optimizations.
   
   Param             Use
   ------------------------------------
   bUseMMX           TRUE (1) to enable MMX, 0 to disable

   Return
   ------
   none
---------------------------------------------------------*/
_ISDeclSpec void  	_ISCConv _ISFn( is5_DisableMMXFunctions )(BOOL bUseMMX);

/*---------------------------------------------------------
   is5_IsMMXSupported / _is5_IsMMXSupported

   Purpose : 
   Test to see if the CPU supports Intel's MMX instructions.
   
   Param             Use
   ------------------------------------
   none

   Return
   ------
   TRUE if MMX is supported
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_IsMMXSupported )();

/*---------------------------------------------------------
   is5_UseMMXMemcopy / _is5_UseMMXMemcopy

   Purpose : 
   Force ImgSource to use an MMX-enabled memcopy function, internally.

   This may give better performance, in some cases.
   
   Param             Use
   ------------------------------------
   bUseMMXMemcopy   TRUE to use MMX

   Return
   ------
   TRUE if MMX is supported
---------------------------------------------------------*/
_ISDeclSpec BOOL  	_ISCConv _ISFn( is5_UseMMXMemcopy )(BOOL bUseMMXMemcopy);

/*---------------------------------------------------------
   is5_MemCopy / _is5_MemCopy

   Purpose : 
   Copy memory between buffers. By default, this is
   C's memcpy. If you've enabled MMX Memcopy (is5_UseMMXMemcopy)
   this uses an MMX-enabled function do the copy, which 
   may be faster than memcpy.
   
   Param             Use
   ------------------------------------
   pDest            output buffer
   pSrc             input buffer
   n                number of bytes to copy

   Return
   ------
   pDest
---------------------------------------------------------*/
_ISDeclSpec void*  	_ISCConv _ISFn( is5_MemCopy )(void *pDest, const void *pSrc, size_t n);

/*---------------------------------------------------------
   is5_SetWorkerThreadCount / _is5_SetWorkerThreadCount

   Purpose : 
   This sets the number of worker threads for functions 
   with multithreaded processing options. The default is
   four.
   
   See is5_SetWorkerThreadCallbackFrequency, is5_SetWorkerThreadTimeout.

   Param             Use
   ------------------------------------
   n                 number of worker threads

---------------------------------------------------------*/
_ISDeclSpec void  _ISCConv  _ISFn(is5_SetWorkerThreadCount)(UINT32 n);

/*---------------------------------------------------------
   is5_SetWorkerThreadCallbackFrequency / _is5_SetWorkerThreadCallbackFrequency

   Purpose : 
   Functions with multithreaded options do not call the 
   global progress callback (is5_SetCallback) once for every
   line processed, rather they are, in effect, called on a timer. 
   This function controls how frequently the callbacks happen.
      
   The default is 200ms. 
   
   Note:
   It's entirely likely that a mutli-threaded function will 
   finish before any callbacks are called. 

   See is5_SetWorkerThreadCount, is5_SetWorkerThreadTimeout.

   Param             Use
   ------------------------------------
   ms                number of milliseconds between callbacks

   Return
   ------
   pDest
---------------------------------------------------------*/
_ISDeclSpec void  _ISCConv  _ISFn(is5_SetWorkerThreadCallbackFrequency)(UINT32 ms);

/*---------------------------------------------------------
   is5_SetWorkerThreadTimeout / _is5_SetWorkerThreadTimeout

   Purpose : 
   This sets the thread timeout length for functions with
   multithreaded options. 
   
   Note:
   This is only used when there is no global progress 
   callback set (is5_SetCallback).
      
   The default is infinity. 
   
   See is5_SetWorkerThreadCount, is5_SetWorkerThreadCallbackFrequency.

   Param             Use
   ------------------------------------
   ms                number of milliseconds for the timeout

   Return
   ------
   pDest
---------------------------------------------------------*/
_ISDeclSpec void  _ISCConv  _ISFn(is5_SetWorkerThreadTimeout)(UINT32 ms);

#ifdef __cplusplus
}
#endif


#endif