// ISImageDemoDoc.cpp : implementation of the CISImageDemoDoc class
//

#include "stdafx.h"
#include "ISImageDemo.h"
#include "ResizeDlg.h"
#include "ColorDepth.h"
#include "RotationDlg.h"
#include "ISImageDemoDoc.h"
#include "MetadataDisplay.h"
#include "ISImageEx.h"
#include "ISLibSetup.h"
#include <fstream>
#include <algorithm>
#include <math.h>
#include "OverlayPickerDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CISImageDemoDoc

IMPLEMENT_DYNCREATE(CISImageDemoDoc, CDocument)

BEGIN_MESSAGE_MAP(CISImageDemoDoc, CDocument)
    //{{AFX_MSG_MAP(CISImageDemoDoc)
    ON_COMMAND(ID_FILE_SAVE_AS, OnFileSaveAs)
    ON_UPDATE_COMMAND_UI(ID_FILE_SAVE, OnUpdate_Valid)
    ON_COMMAND(ID_EDIT_RESIZE, OnEditResize)
    ON_UPDATE_COMMAND_UI(ID_EDIT_RESIZE, OnUpdate_Valid)
    ON_COMMAND(ID_EDIT_HORIZONTALFLIP, OnEditHorizontalflip)
    ON_UPDATE_COMMAND_UI(ID_EDIT_HORIZONTALFLIP, OnUpdate_Valid)
    ON_COMMAND(ID_EDIT_VERTICALFLIP, OnEditVerticalflip)
    ON_UPDATE_COMMAND_UI(ID_EDIT_VERTICALFLIP, OnUpdate_Valid)
    ON_COMMAND(ID_SHARPEN, OnSharpen)
    ON_UPDATE_COMMAND_UI(ID_SHARPEN, OnUpdate_24)
    ON_COMMAND(ID_BLUR, OnBlur)
    ON_UPDATE_COMMAND_UI(ID_BLUR, OnUpdate_24)
    ON_COMMAND(ID_EXIF, OnExif)
    ON_COMMAND(ID_IPTC, OnIptc)
    ON_COMMAND(ID_ROTATE, OnRotate)
    ON_UPDATE_COMMAND_UI(ID_ROTATE, OnUpdate_Valid)
    ON_COMMAND(ID_AUTOCROP, OnAutocrop)
    ON_UPDATE_COMMAND_UI(ID_AUTOCROP, OnUpdate_Valid)
    ON_COMMAND(ID_CENTERSHARP, OnCentersharp)
    ON_UPDATE_COMMAND_UI(ID_CENTERSHARP, OnUpdate_24)
    ON_COMMAND(ID_EDIT_PASTE, OnEditPaste)
    ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE, OnUpdate_Valid)
    ON_COMMAND(ID_COLORS, OnColors)
    ON_UPDATE_COMMAND_UI(ID_COLORS, OnUpdate_Valid)
    ON_COMMAND(IDS_DESKEW, OnDeskew)
    ON_UPDATE_COMMAND_UI(IDS_DESKEW, OnUpdate_Valid)
    ON_COMMAND(ID_FILE_SAVE, OnFileSaveAs)
    //}}AFX_MSG_MAP
    ON_COMMAND(ID_MOREIMAGESTUFF_WAVELETDENOISE, OnMoreimagestuffWaveletdenoise)
    ON_COMMAND(ID_GAUSSIANBLUR_SMALL, OnGaussianblurSmall)
    ON_COMMAND(ID_GAUSSIANBLUR_MEDIUM, OnGaussianblurMedium)
    ON_COMMAND(ID_GAUSSIANBLUR_HUGE, OnGaussianblurHuge)
    ON_COMMAND(ID_BLURS_RADIALBLUR, OnBlursRadialblur)
    ON_COMMAND(ID_BLURS_ZOOMBLUR, OnBlursZoomblur)
    ON_COMMAND(ID_BLURS_MOTIONBLUR, OnBlursMotionblur)
    ON_COMMAND(ID_FILTERS_UNSHARPMASK, OnFiltersUnsharpMask)
    ON_COMMAND(ID_FILTERS_MEDIAN, OnFiltersMedian)
    ON_COMMAND(ID_FILTERS_MIN, OnFiltersMin)
    ON_COMMAND(ID_FILTERS_MAX, OnFiltersMax)
    ON_COMMAND(ID_FILTERS_BILATERAL, OnFiltersBilateral)
    ON_COMMAND(ID_FILTERS_MIDPOINT, OnFiltersMidpoint)
    ON_COMMAND(ID_FILTERS_ARITHMETICMEAN, OnFiltersArithmeticmean)
    ON_COMMAND(ID_FILTERS_SOBEL, OnFiltersSobel)
    ON_COMMAND(ID_MAKESOMENOISE_GAUSSIAN, OnMakesomenoiseGaussian)
    ON_COMMAND(ID_MAKESOMENOISE_RAYLEIGH, OnMakesomenoiseRayleigh)
    ON_COMMAND(ID_MAKESOMENOISE_PERLIN, OnMakesomenoisePerlin)
    ON_COMMAND(ID_PRETTYCOLORS_DOUTONE, OnPrettycolorsDoutone)
    ON_COMMAND(ID_PRETTYCOLORS_POSTERIZE, OnPrettycolorsPosterize)
    ON_COMMAND(ID_PRETTYCOLORS_MOSAIC, OnPrettycolorsMosaic)
    ON_COMMAND(ID_PRETTYCOLORS_GAUZY, OnPrettycolorsGauzy)
    ON_COMMAND(ID_PRETTYCOLORS_CRACKLE, OnPrettycolorsCrackle)
    ON_COMMAND(ID_MORPHOLOGICAL_EROSION, OnMorphologicalErosion)
    ON_COMMAND(ID_MORPHOLOGICAL_DILATION, OnMorphologicalDilation)
    ON_COMMAND(ID_MORPHOLOGICAL_OPEN, OnMorphologicalOpen)
    ON_COMMAND(ID_MORPHOLOGICAL_CLOSE, OnMorphologicalClose)
    ON_COMMAND(ID_MORPHOLOGICAL_TOPHAT, OnMorphologicalTophat)
    ON_COMMAND(ID_MORPHOLOGICAL_GRADIENT, OnMorphologicalGradient)
    ON_COMMAND(ID_MORPHOLOGICAL_THINNING, OnMorphologicalThinning)
    ON_COMMAND(ID_MORPHOLOGICAL_BOUNDARYEXTRACTION, OnMorphologicalBoundaryextraction)
    ON_COMMAND(ID_MORPHOLOGICAL_HITORMISS, OnMorphologicalHitormiss)
    ON_COMMAND(ID_PRETTYCOLORS_INVERT, OnPrettycolorsInvert)
    ON_COMMAND(ID_GEOMETRY_LINE, OnGeometryLine)
    ON_COMMAND(ID_GEOMETRY_THICKLINE, OnGeometryThickline)
    ON_COMMAND(ID_GEOMETRY_ELLIPSE, OnGeometryEllipse)
    ON_COMMAND(ID_GEOMETRY_POLYGON, OnGeometryPolygon)


    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_EROSION, OnUpdate_8G_1)
    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_DILATION, OnUpdate_8G_1)
    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_OPEN, OnUpdate_8G_1)
    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_CLOSE, OnUpdate_8G_1)
    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_TOPHAT, OnUpdate_8G)
    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_GRADIENT, OnUpdate_8G)
    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_THINNING, OnUpdate_1)
    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_BOUNDARYEXTRACTION, OnUpdate_1)
    ON_UPDATE_COMMAND_UI(ID_MORPHOLOGICAL_HITORMISS, OnUpdate_1)

    ON_UPDATE_COMMAND_UI(ID_MOREIMAGESTUFF_WAVELETDENOISE, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_GAUSSIANBLUR_SMALL, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_GAUSSIANBLUR_MEDIUM, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_GAUSSIANBLUR_HUGE, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_BLURS_RADIALBLUR, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_BLURS_ZOOMBLUR, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_BLURS_MOTIONBLUR, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_FILTERS_UNSHARPMASK, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_FILTERS_MEDIAN, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_FILTERS_MIN, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_FILTERS_MAX, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_FILTERS_BILATERAL, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_FILTERS_MIDPOINT, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_FILTERS_ARITHMETICMEAN, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_FILTERS_SOBEL, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_MAKESOMENOISE_GAUSSIAN, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_MAKESOMENOISE_RAYLEIGH, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_MAKESOMENOISE_PERLIN, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_PRETTYCOLORS_DOUTONE, OnUpdate_32_24)
    ON_UPDATE_COMMAND_UI(ID_PRETTYCOLORS_POSTERIZE, OnUpdate_24)
    ON_UPDATE_COMMAND_UI(ID_PRETTYCOLORS_MOSAIC, OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_PRETTYCOLORS_GAUZY, OnUpdate_24)
    ON_UPDATE_COMMAND_UI(ID_PRETTYCOLORS_CRACKLE, OnUpdate_24)

    ON_UPDATE_COMMAND_UI(ID_PRETTYCOLORS_INVERT,    OnUpdate_Valid)
    ON_UPDATE_COMMAND_UI(ID_GEOMETRY_LINE,          OnUpdate_Valid)
    ON_UPDATE_COMMAND_UI(ID_GEOMETRY_THICKLINE,     OnUpdate_32_24_8G)
    ON_UPDATE_COMMAND_UI(ID_GEOMETRY_ELLIPSE,       OnUpdate_Valid)
    ON_UPDATE_COMMAND_UI(ID_GEOMETRY_POLYGON,       OnUpdate_Valid)

    ON_COMMAND(ID_PRETTYCOLORS_RETINEX, OnPrettycolorsRetinex)
    ON_COMMAND(ID_SATURATION_MORE, OnSaturationMore)
    ON_COMMAND(ID_SATURATION_LESS, OnSaturationLess)
    ON_COMMAND(ID_VIBRANCE_MORE, OnVibranceMore)
    ON_COMMAND(ID_VIBRANCE_LESS, OnVibranceLess)

    ON_UPDATE_COMMAND_UI(ID_PRETTYCOLORS_RETINEX,   OnUpdate_Valid)
    ON_UPDATE_COMMAND_UI(ID_SATURATION_MORE,        OnUpdate_32_24)
    ON_UPDATE_COMMAND_UI(ID_SATURATION_LESS,        OnUpdate_32_24)
    ON_UPDATE_COMMAND_UI(ID_VIBRANCE_MORE,          OnUpdate_32_24)
    ON_UPDATE_COMMAND_UI(ID_VIBRANCE_LESS,          OnUpdate_32_24)
    
    ON_COMMAND(ID_MOREIMAGESTUFF_OVERLAY,           OnMoreimagestuffOverlay)
    ON_UPDATE_COMMAND_UI(ID_MOREIMAGESTUFF_OVERLAY, OnUpdate_Valid)
    END_MESSAGE_MAP()

//////////////////////////////////////////////////////////////////////
// control UI items via image bit depth


void CISImageDemoDoc::OnUpdate_Valid(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(m_image.ISValid());
}

void CISImageDemoDoc::OnUpdate_32_24_8G(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable
        (
        m_image.ISValid() && 
        (
        m_image.BitDepth()==32 || 
        m_image.BitDepth()==24 || 
        (m_image.BitDepth()==8 && m_image.Is8BitGray())
        )
        );
}

void CISImageDemoDoc::OnUpdate_32_24(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable
        (
        m_image.ISValid() && 
        (
        m_image.BitDepth()==32 || 
        m_image.BitDepth()==24
        )
        );
}

void CISImageDemoDoc::OnUpdate_24(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable
        (
        m_image.ISValid() && 
        (
        m_image.BitDepth()==24
        )
        );
}

void CISImageDemoDoc::OnUpdate_1(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable
        (
        m_image.ISValid() && 
        (
        m_image.BitDepth()==1
        )
        );
}

void CISImageDemoDoc::OnUpdate_8G_1(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable
        (
        m_image.ISValid() && 
        (
        m_image.BitDepth()==1 || 
        (m_image.BitDepth()==8 && m_image.Is8BitGray())
        )
        );
}

void CISImageDemoDoc::OnUpdate_8G(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable
        (
        m_image.ISValid() && 
        (
        (m_image.BitDepth()==8 && m_image.Is8BitGray())
        )
        );
}


/////////////////////////////////////////////////////////////////////////////
// CISImageDemoDoc construction/destruction

CISImageDemoDoc::CISImageDemoDoc()
{

}

CISImageDemoDoc::~CISImageDemoDoc()
{
}

BOOL CISImageDemoDoc::OnNewDocument()
{
    if (!CDocument::OnNewDocument())
        return FALSE;

    return TRUE;
}

void reportRes(UINT res)
{
    if (res != IS_ERR_OK)
    {
        TRACE("Got an error code : %d\n", res);
    }
}

/////////////////////////////////////////////////////////////////////////////
// CISImageDemoDoc serialization

void CISImageDemoDoc::Serialize(CArchive& ar)
{
    if (ar.IsStoring())
    {
        // TODO: add storing code here
    }
    else
    {
        // TODO: add loading code here
    }
}

/////////////////////////////////////////////////////////////////////////////
// CISImageDemoDoc diagnostics

#ifdef _DEBUG
void CISImageDemoDoc::AssertValid() const
{
    CDocument::AssertValid();
}

void CISImageDemoDoc::Dump(CDumpContext& dc) const
{
    CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CISImageDemoDoc commands


BOOL CISImageDemoDoc::OnOpenDocument(LPCTSTR pFilename) 
{
    CWaitCursor waitHere;

    // read it
    UINT res = m_image.LoadImage(pFilename);

    if (m_cBmp.GetSafeHandle())
    {
        m_cBmp.DeleteObject();
    }

    if (res!=IS_ERR_OK)
    {
        AfxMessageBox(CISImageEx::ErrorString(res));
        if (res!=IS_ERR_TRIALVERSION)
        {
            return FALSE;
        }

    }    

    CSize szImg = m_image.Size();
    CRect rcCrop(CPoint(0, 0), szImg);
    rcCrop.top = 0;
    m_image.Crop(rcCrop);

    UpdateImage();

    SetTitle(pFilename);

    return TRUE;
}

///////////////////////////////////////////////////

COLORREF CISImageDemoDoc::GetBackgroundColor()
{
   return RGB(64,32,32);
}

void CISImageDemoDoc::UpdateImage()
{
    //
    // copy to a CBitmap for display
    //
    UINT res = m_image.CopyToCBitmap(m_cBmp, NULL, GetBackgroundColor());

    if (m_cBmp.GetSafeHandle()==NULL && res!=IS_ERR_OK)
    {
        AfxMessageBox(CISImageEx::ErrorString(res));
    }
    else
    {
        // make a windows palette
        res = m_image.MakeImagePalette(m_pal);
        if (res!=IS_ERR_OK)
        {
            AfxMessageBox(CISImageEx::ErrorString(res));
        }
    }

    // force a redraw and a scroll bar recalc
    UpdateAllViews(NULL, eHintNewImage, NULL);

}


///////////////////////////////////////////////////

void CISImageDemoDoc::OnFileSaveAs() 
{
    // big, scary output filter
    CString filt = _T("Windows BMP Files (*.BMP)|*.BMP|\
                      GIF Files (*.GIF)|*.GIF|\
                      JPEG Files (*.JPG)|*.JPG|\
                      ZSoft PCX Files (*.PCX)|*.PCX|\
                      Adobe Photoshop Files (*.PSD)|*.PSD|\
                      PNG Files (*.PNG)|*.PNG|\
                      Targa TGA Files (*.TGA)|*.TGA|\
                      Wireless Bitmap Files (*.WBMP)|*.WBMP|\
                      TIFF Files (*.TIF)|*.TIF|\
                      All files (*.*)|*.*||");

    CFileDialog fileDlg(FALSE, NULL, NULL, NULL, filt, NULL);

    fileDlg.m_ofn.lpstrTitle = _T("Save current image to this file");

    if (fileDlg.DoModal()==IDOK) 
    {
        CString csOutPath = fileDlg.GetPathName();

        // get input file extension
        CString ext;
        int iDotPos = csOutPath.ReverseFind('.');
        if (iDotPos!=-1) 
        {
            ext = csOutPath.Mid(iDotPos);
        }
        else
        {
            AfxMessageBox(_T("Unable to determine output format"));
            return;
        }

        // find the base output format
        CISImageEx::eImgFmt eSaveFmt = CISImageEx::FmtFromExt(ext);

        eSaveFmt = m_image.FindBestOutputFormat(eSaveFmt);

        CWaitCursor wait_here_please;

        UINT32 res = m_image.SaveImage(csOutPath, eSaveFmt);

        SetPathName(csOutPath);
    }

}

///////////////////////////////////////////////////

// resize the image
void CISImageDemoDoc::OnEditResize() 
{
    CResizeDlg d;

    d.m_uX = m_image.Size().cx;
    d.m_uY = m_image.Size().cy;
    d.m_uBitDepth = m_image.BitDepth();

    if (d.DoModal()==IDOK)
    {
        UINT32 uType = d.m_iResizeMethod;

        switch (uType)
        {
        case 0:
            uType = is6_RESIZE_MODE_BILINEAR;
            break;
        case 1:
            uType = is6_RESIZE_MODE_BICUBIC;
            break;
        case 2:
            uType = is6_RESIZE_MODE_AREAAVG;
            break;
        case 3:
            uType = is6_RESIZE_MODE_NEAREST_NEIGHBOR;
            break;
        case 4:
            uType = is6_RESIZE_MODE_LANCZOS3;
            break;
        case 5:
            uType = is6_RESIZE_MODE_CATROM;
            break;
        case 6:
            uType = is6_RESIZE_MODE_MITCHELL;
            break;
        }

        CWaitCursor wait;
        UINT res = m_image.Resize(CSize(d.m_uX,d.m_uY), uType, d.m_1to8 ? true : false);
        reportRes(res);
        UpdateImage();
    }
}

///////////////////////////////////////////////////

// flip and update
void CISImageDemoDoc::OnEditHorizontalflip() 
{
    CWaitCursor wait;
    UINT res = m_image.Flip(CISImageEx::eFlipHorizontal);
    reportRes(res);
    UpdateImage();
}


///////////////////////////////////////////////////

// flip and update
void CISImageDemoDoc::OnEditVerticalflip() 
{
    CWaitCursor wait;
    UINT res = m_image.Flip(CISImageEx::eFlipVertical);
    reportRes(res);
    UpdateImage();
}

///////////////////////////////////////////////////

void CISImageDemoDoc::OnSharpen() 
{
    CWaitCursor wait;
    UINT res = m_image.Sharpen(30);
    reportRes(res);

    UpdateImage();
}

///////////////////////////////////////////////////

void CISImageDemoDoc::OnCentersharp() 
{
    CWaitCursor wait;

    CRect subRect;
    subRect.left = m_image.Width() / 4;
    subRect.right = 3 * m_image.Width() / 4;
    subRect.top = m_image.Height() / 4;
    subRect.bottom = 3 * m_image.Height() / 4;

    UINT res = m_image.Sharpen(30, subRect);
    reportRes(res);

    UpdateImage();
}

///////////////////////////////////////////////////

void CISImageDemoDoc::OnBlur() 
{
    CWaitCursor wait;
    UINT res = m_image.Blur(50);
    reportRes(res);
    UpdateImage();
}

///////////////////////////////////////////////////

BOOL        CISImageDemoDoc::Draw(const CDC *pDC, const CRect & outRect)  
{
    return _ISFn(is6_DrawHBITMAP)(pDC->m_hDC, 
        (HBITMAP)m_cBmp.GetSafeHandle(), 
        outRect.left, 
        outRect.top, 
        outRect.Width(), 
        outRect.Height(), 
        (HPALETTE)m_pal.GetSafeHandle());
}

///////////////////////////////////////////////////

BOOL        CISImageDemoDoc::StretchDraw(CDC *pDC, const CRect & outRect)  
{
    return m_image.StretchDraw(pDC, outRect);
}

///////////////////////////////////////////////////

void CISImageDemoDoc::OnExif() 
{
    // only care about JPG and TIFF, for EXIF info
    CString filt="JPEG Files (*.JPG)|*.JPG|TIFF Files (*.TIF)|*.TIF|All files (*.*)|*.*||";

    CString defFilt = "*.Jpg;*.Tif;";

    CFileDialog fileDlg(TRUE, defFilt, defFilt, NULL, filt, NULL);

    fileDlg.m_ofn.Flags|=OFN_FILEMUSTEXIST;
    fileDlg.m_ofn.lpstrTitle = _T("Extract EXIF info from file");

    if (fileDlg.DoModal()!=IDOK)
    {
        return;
    }

    CStringArray csa;
    CString csFileName= fileDlg.GetPathName();

    CISImageEx temp;
    bool ok = false;

    AfxGetApp()->BeginWaitCursor();

    ok  = temp.GetEXIFTagList(csFileName, csa);

    const char *pSections[] = {"Main IFD", "EXIF IFD", "Interoperability IFD", "Thumbnail IFD", "GPS IFD"};

    if (ok)
    {
        CString csMsg; 
        csMsg.Format(_T("EXIF information for %s:\n"), csFileName);

        for (int i=0;i<min(5, csa.GetSize());i++)
        {
            CString t = csa.GetAt(i);

            // chop these down, if the tags are really big.
            if (t.GetLength() > 300)
            {
                t = t.Left(300) + "   [... chopped]\r\n";
            }

            if (!t.IsEmpty())
            {
                csMsg+="\r\n\r\n";
                csMsg+=pSections[i];
                csMsg+="\r\n";
                csMsg+="--------------------------\n";
                csMsg+=t;
            }
        }

        AfxGetApp()->EndWaitCursor();

        CMetadataDisplay dlg;
        dlg.m_csDisplayText = csMsg;
        dlg.DoModal();
    }
    else
    {
        AfxGetApp()->EndWaitCursor();
        CString csMsg; 
        csMsg.Format(_T("%s did not contain any EXIF information"), csFileName);
        AfxMessageBox(csMsg);
    }
}

///////////////////////////////////////////////////

void CISImageDemoDoc::OnIptc() 
{
    // only care about JPG and TIFF, for IPTC info
    CString filt = _T("JPEG Files (*.JPG)|*.JPG|TIFF Files (*.TIF)|*.TIF|All files (*.*)|*.*||");

    CString defFilt = _T("*.Jpg;*.Tif;");

    CFileDialog fileDlg(TRUE, defFilt, defFilt, NULL, filt, NULL);

    fileDlg.m_ofn.Flags|=OFN_FILEMUSTEXIST;
    fileDlg.m_ofn.lpstrTitle = _T("Extract IPTC info from file");

    if (fileDlg.DoModal()!=IDOK)
    {
        return;
    }

    CStringArray csa;
    CString csFileName= fileDlg.GetPathName();

    CISImageEx temp;
    bool ok = temp.GetIPTCTagList(csFileName, csa);

    if (ok)
    {
        CString csMsg; 
        csMsg.Format(_T("IPTC information for %s:\n"), csFileName);

        for (int i=0;i<csa.GetSize();i++)
        {
            CString t = csa.GetAt(i);
            if (!t.IsEmpty())
            {
                csMsg+="\n";
                csMsg+="--------------------------\n";
                csMsg+=csa.GetAt(i);
            }
        }

        CMetadataDisplay dlg;
        dlg.m_csDisplayText = csMsg;
        dlg.DoModal();
    }
    else
    {
        CString csMsg; 
        csMsg.Format(_T("%s did not contain any IPTC information"), csFileName);
        AfxMessageBox(csMsg);
    }

}

///////////////////////////////////////////////////

void CISImageDemoDoc::OnRotate() 
{
    CRotationDlg dlg;

    if (dlg.DoModal())
    {
       COLORREF clr = GuessImageBackgroundColor();

        double pi = acos((double)-1);
        double rads = pi * dlg.m_degs / 180.0;

        CWaitCursor wait;
        UINT res = m_image.Rotate(rads, clr, dlg.m_bQuality ? false : true);
        reportRes(res);

        UpdateImage();
    }
}

///////////////////////////////////////////////////

void CISImageDemoDoc::OnAutocrop() 
{
    CWaitCursor wait;
    UINT res = m_image.AutoCrop();
    reportRes(res);
    UpdateImage();
}


void CISImageDemoDoc::OnEditPaste() 
{
    if (::OpenClipboard(AfxGetMainWnd()->m_hWnd))
    {
        HANDLE hDIB = GetClipboardData(CF_DIB);
        if (hDIB)
        {
            BITMAPINFOHEADER *pDIB = (BITMAPINFOHEADER*)GlobalLock(hDIB);

            if (pDIB)
            {
                UINT32 w;
                __int32 h;
                _ISFn(is6_DIBWidth)(pDIB, &w);
                _ISFn(is6_DIBHeight)(pDIB, &h);

                h = abs(h);

                CISImageEx tmp;
                if (tmp.AllocBlank(w, h, 24) == IS_ERR_OK)
                {
                    _ISFn(is6_DIBToRGB)(pDIB, tmp.ImageData());
                }
                GlobalUnlock(hDIB);

                UINT res = m_image.OverlayImage(tmp, 0, 0, 1.0);
                reportRes(res);

                UpdateImage();
            }
        }
        else
        {
            // no DIBs to paste. try HBITMAP
            HBITMAP hBmp = (HBITMAP)GetClipboardData(CF_BITMAP);
            if (hBmp)
            {
                CISImageEx tmp;
                if (tmp.CopyFromCBitmap(hBmp) == IS_ERR_OK)
                {
                    UINT res = m_image.OverlayImage(tmp, 0, 0, 1.0);
                    reportRes(res);

                    UpdateImage();
                }
            }

        }
        CloseClipboard();
    }
}

void CISImageDemoDoc::OnColors() 
{
    CColorDepth	d;
    d.m_uBitDepth = m_image.BitDepth();

    if (d.DoModal()==IDOK)
    {
        switch (d.m_uMakeType)
        {
        case 0: // make 24
            m_image.Make24Bit();
            break;
        case 1: // make 8 bit
            {
                RGBQUAD pal[256];
                int i = 0;
                switch (d.m_8BitType)
                {
                case 0:
                   m_image.Make8BitGray();
                   break;
                case 1:
                    // 16 color pal
                    pal[i].rgbRed = 0x00; pal[i].rgbGreen = 0x00; pal[i++].rgbBlue = 0x00;
                    pal[i].rgbRed = 0x00; pal[i].rgbGreen = 0x00; pal[i++].rgbBlue = 0xA8;
                    pal[i].rgbRed = 0x00; pal[i].rgbGreen = 0xA8; pal[i++].rgbBlue = 0x00;
                    pal[i].rgbRed = 0x00; pal[i].rgbGreen = 0xA8; pal[i++].rgbBlue = 0xA8;

                    pal[i].rgbRed = 0xa8; pal[i].rgbGreen = 0x00; pal[i++].rgbBlue = 0x00;
                    pal[i].rgbRed = 0xa8; pal[i].rgbGreen = 0x00; pal[i++].rgbBlue = 0xA8;
                    pal[i].rgbRed = 0xA8; pal[i].rgbGreen = 0x54; pal[i++].rgbBlue = 0x00;
                    pal[i].rgbRed = 0xA8; pal[i].rgbGreen = 0xA8; pal[i++].rgbBlue = 0xA8;

                    pal[i].rgbRed = 0x54; pal[i].rgbGreen = 0x54; pal[i++].rgbBlue = 0x54;
                    pal[i].rgbRed = 0x54; pal[i].rgbGreen = 0x54; pal[i++].rgbBlue = 0xFE;
                    pal[i].rgbRed = 0x54; pal[i].rgbGreen = 0xFE; pal[i++].rgbBlue = 0x54;
                    pal[i].rgbRed = 0x54; pal[i].rgbGreen = 0xFE; pal[i++].rgbBlue = 0xFE;

                    pal[i].rgbRed = 0xFe; pal[i].rgbGreen = 0x54; pal[i++].rgbBlue = 0x54;
                    pal[i].rgbRed = 0xFE; pal[i].rgbGreen = 0x54; pal[i++].rgbBlue = 0xFE;
                    pal[i].rgbRed = 0xFE; pal[i].rgbGreen = 0xFE; pal[i++].rgbBlue = 0x54;
                    pal[i].rgbRed = 0xFE; pal[i].rgbGreen = 0xFE; pal[i++].rgbBlue = 0xFE;
                    m_image.Make8Bit(pal, 16);
                    break;

                case 2:
                    {
                        // the "web-safe" palette
                        RGBQUAD webPal[] =
                        {
                            {255,255,255,0},{255,255,204,0},{255,255,153,0},{255,255,102,0},{255,255,51,0},{255,255,0,0},{255,204,255,0},{255,204,204,0},
                            {255,204,153,0},{255,204,102,0},{255,204,51,0},{255,204,0,0},{255,153,255,0},{255,153,204,0},{255,153,153,0},{255,153,102,0},
                            {255,153,51,0},{255,153,0,0},{255,102,255,0},{255,102,204,0},{255,102,153,0},{255,102,102,0},{255,102,51,0},{255,102,0,0},
                            {255,51,255,0},{255,51,204,0},{255,51,153,0},{255,51,102,0},{255,51,51,0},{255,51,0,0},{255,0,255,0},{255,0,204,0},
                            {255,0,153,0},{255,0,102,0},{255,0,51,0},{255,0,0,0},{204,255,255,0},{204,255,204,0},{204,255,153,0},{204,255,102,0},
                            {204,255,51,0},{204,255,0,0},{204,204,255,0},{204,204,204,0},{204,  204,153,0},{204,204,102,0},{204,204,51,0},{204,204,0,0},
                            {204,153,255,0},{204,153,204,0},{204,153,  153,0},{204,153,102,0},{204,153,51,0},{204,153,0,0},{204,102,255,0},{204,102,204,0},
                            {204,102,153,0},{204,102,102,0},{204,102,51,0},{204,102,0,0},{204,51,255,0},{204,51,204,0},{204,51,153,0},{204, 51,102,0},
                            {204,51,51,0},{204,51,0,0},{204,0,255,0},{204,0,204,0},{204,0,153,0},{204,0,  102,0},{204,0,51,0},{204,0,0,0},
                            {153,255,255,0},{153,255,204,0},{153,255,153,0},{153,255,102,0},{153,255,51,0},{153,255,0,0},{153,204,255,0},{153,204,204,0},
                            {153,204,153,0},{153,204,102,0},{153,204,51,0},{153,204,0,0},{153,153,255,0},{153,153,204,0},{153,153,153,0},{153,153,102,0},
                            {153,153,51,0},{153,153,0,0},{153,102,255,0},{153,102,204,0},{153,102,153,0},{153,102,102,0},{153,102,51,0},{153,102,0,0},
                            {153,51,255,0},{153,51,204,0},{153,51,153,0},{153,51,102,0},{153,51,51,0},{153,  51,0,0},{153,0,255,0},{153,0,204,0},
                            {153,0,153,0},{153,0,102,0},{153,0,51,0},{153,0,  0,0},{102,255,255,0},{102,255,204,0},{102,255,153,0},{102,255,102,0},
                            {102,255,51,0},{102,255,0,0},{102,204,255,0},{102,204,204,0},{102,204,153,0},{102,204,102,0},{102,204,51,0},{102,204,0,0},
                            {102,153,255,0},{102,153,204,0},{102,153,153,0},{102,153,102,0},{102,153,51,0},{102,153,0,0},{102,102,255,0},{102,102,204,0},
                            {102,102,153,0},{102,102,102,0},{102,102,51,0},{102,102,0,0},{102,51,255,0},{102,51,204,0},{102,51,153,0},{102,51,102,0},
                            {102,51,51,0},{102,51,0,0},{102,0,255,0},{102,0,204,0},{102,0,153,0},{102,0,102,0},{102,0,51,0},{102,0,0,0},
                            {51,255,255,0},{51,255, 204,0},{51,255,153,0},{51,255,102,0},{51,255,51,0},{51,255,0,0},{51,204,255,0},{51,204,204,0},
                            {51,204,153,0},{51,204,102,0},{51,204,51,0},{51,204,0,0},{51,153,255,0},{51,153,204,0},{51,153,153,0},{51,153,102,0},
                            {51,153,51,0},{51,153,0,0},{51,102,255,0},{51,102,204,0},{51,102,153,0},{51,102,102,0},{51,102,51,0},{51,102,0,0},
                            {51,51,255,0},{51,51,204,0},{51,51,153,0},{51,51,102,0},{51,51,51,0},{51,51,0,0},{51,0,255,0},{51,0,204,0},
                            {51,0,153,0},{51,0,102,0},{51,0,51,0},{51,0,0,0},{0,255,255,0},{0,255,204,0},{0,255,153,0},{0,255,102,0},
                            {0,255,51,0},{0,255,0,0},{0,204,255,0},{0,204,204,0},{0,204,153,0},{0,204,102,0},{0,204,51,0},{0,204,0,0},
                            {0,153,255,0},{0,153,204,0},{0,153,153,0},{0,153,102,0},{0,153,51,0},{0,153,0,0},{0,102,255,0},{0,102,204,0},
                            {0,102,153,0},{0,102,102,0},{0,102,51,0},{0,102,0,0},{0,51,255,0},{0,51,204,0},{0,51,153,0},{0,51,102,0},
                            {0,51,51,0},{0,51,0,0},{0,0,255,0},{0,0,204,0},{0,0,153,0},{0,0,102,0},{0,0,51,0},{0,0,0,0}
                        };					   
                        m_image.Make8Bit(webPal, 256);
                    }
                    break;
                case 3:
                    // best case
                    m_image.Make8Bit();
                    break;
                }
            }
            break;
        case 2: // make 1-bit
            {
                UINT32 uMode = 0;
                switch (d.m_1bitType)
                {
                case 0:  // tresh = 128
                    uMode = 0;
                    break;
                case 1:  // two-peaks
                    uMode = 1;
                    break;
                case 2:  // otsu
                    uMode = 4;
                    break;
                case 3:  // balanced avg
                    uMode = 5;
                    break;
                case 4:  // color match
                    uMode = 10;
                    break;
                }
                m_image.Make1Bit(128, uMode);
            }
            break;
        }
    }

    UpdateImage();

}

COLORREF CISImageDemoDoc::GuessImageBackgroundColor()
{
   // when rotating, the background has to be filled with some color.
   // ideally, it should be the 'background' color of the source
   // image... but that's not a trivial problem. so, we'll just
   // look at the pixel(s) in the top-left of the image.
   // this is not very smart, but it might be better than nothing.

   COLORREF clr = 0;

   if (m_image.BitDepth()==1)
   {
      // first 8 pixels
      if (m_image.ImageDataConst()[0]==255) clr = 1;
   }
   else if (m_image.BitDepth() == 8)
   {
      // first pixel
      clr = m_image.ImageDataConst()[0];
   }
   else if (m_image.BitDepth() == 24)
   {
      // first
      clr = RGB(m_image.ImageDataConst()[0], m_image.ImageDataConst()[1], m_image.ImageDataConst()[2]);
   }
   else if (m_image.BitDepth() == 32)
   {
      clr = RGBA(m_image.ImageDataConst()[0], m_image.ImageDataConst()[1], m_image.ImageDataConst()[2], m_image.ImageDataConst()[3]);
   }
   return clr;
}

void CISImageDemoDoc::OnDeskew() 
{
    CWaitCursor wait;

    COLORREF clr = GuessImageBackgroundColor();

    UINT res = m_image.Deskew(clr);
    reportRes(res);

    UpdateImage();
}


void CISImageDemoDoc::OnFileSave() 
{
    // TODO: Add your command handler code here

}
void CISImageDemoDoc::OnMoreimagestuffWaveletdenoise()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.WaveletDenoise(5, .9);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnGaussianblurSmall()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.GaussianBlur(2.0);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnGaussianblurMedium()
{   
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.GaussianBlur(8.0);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnGaussianblurHuge()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.GaussianBlur(50.0);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnBlursRadialblur()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.RadialBlur(.5, 10, CPoint(m_image.Width() / 2, m_image.Height() / 2));
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnBlursZoomblur()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.ZoomBlur(.5, CPoint(m_image.Width() / 2, m_image.Height() / 2));
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnBlursMotionblur()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.MotionBlur(.5, 10);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnFiltersArithmeticmean()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.Filter(CISImageEx::ArithMean, 5);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnFiltersUnsharpMask()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.UnsharpMask(15, 1, 1.5);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnFiltersBilateral()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.BilateralFilter(15, 0, 20);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnFiltersMedian()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.Filter(CISImageEx::Median, 5);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnFiltersMin()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.Filter(CISImageEx::Min, 5);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnFiltersMax()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.Filter(CISImageEx::Max, 5);
        reportRes(res);
        UpdateImage();
    }
}


void CISImageDemoDoc::OnFiltersMidpoint()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        UINT res = m_image.Filter(CISImageEx::Midpoint, 5);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnFiltersSobel()
{
    CWaitCursor wait;

    if (m_image.BitDepth() == 24 || m_image.BitDepth() == 32)
    {
        CISImageEx temp;
        UINT res = m_image.Sobel(temp);
        reportRes(res);
        m_image.Steal(temp);
        res = m_image.HistogramStretch(0, 0);
        reportRes(res);
        UpdateImage();
    }
}

void CISImageDemoDoc::OnMakesomenoiseGaussian()
{
    UINT res = m_image.AddNoise(CISImageEx::Gaussian, 1000, 128, false);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMakesomenoiseRayleigh()
{
    UINT res = m_image.AddNoise(CISImageEx::Rayleigh, 1000, 128, false);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMakesomenoisePerlin()
{
    UINT res = m_image.PerlinNoise(m_image.Width(), m_image.Height(), .05);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnPrettycolorsDoutone()
{
    UINT res = m_image.Duotone(RGB(255,192,0));
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnPrettycolorsPosterize()
{
    UINT res = m_image.Posterize();
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnPrettycolorsMosaic()
{
    UINT res = m_image.Mosaic(30);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnPrettycolorsGauzy()
{
    UINT res = m_image.Gauzy(.5, .5, .5, 0);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnPrettycolorsCrackle()
{
    UINT res = m_image.Crackle(0, 50, true);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalErosion()
{
    int mask[9] = 
    {
        1,1,1,
        1,1,1,
        1,1,1,
    };
    UINT res = m_image.Morpho(CISImageEx::Erosion, mask, 3);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalDilation()
{
    int mask[25] = 
    {
        0, 0, 1, 0, 0,
        0, 1, 1, 1, 0,
        1, 1, 1, 1, 1,
        0, 1, 1, 1, 0,
        0, 0, 1, 0, 0,
    };
    UINT res = m_image.Morpho(CISImageEx::Dilation, mask, 5);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalOpen()
{
    int mask[25] = 
    {
        0, 0, 1, 0, 0,
        0, 1, 1, 1, 0,
        1, 1, 1, 1, 1,
        0, 1, 1, 1, 0,
        0, 0, 1, 0, 0,
    };
    UINT res = m_image.Morpho(CISImageEx::Open, mask, 5);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalClose()
{
    int mask[25] = 
    {
        0, 0, 1, 0, 0,
        0, 1, 1, 1, 0,
        1, 1, 1, 1, 1,
        0, 1, 1, 1, 0,
        0, 0, 1, 0, 0,
    };
    UINT res = m_image.Morpho(CISImageEx::Close, mask, 5);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalTophat()
{
    int mask[25] = 
    {
        0, 0, 1, 0, 0,
        0, 1, 1, 1, 0,
        1, 1, 1, 1, 1,
        0, 1, 1, 1, 0,
        0, 0, 1, 0, 0,
    };
    UINT res = m_image.TopHat(mask, 5);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalGradient()
{
    int mask[9] = 
    {
        1, 1, 1,
        1, 1, 1,
        1, 1, 1,
    };
    UINT res = m_image.Gradient(mask, 3);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalThinning()
{
    CWaitCursor wait;

    UINT res = m_image.Thinning(100);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalBoundaryextraction()
{
    int mask[9] = 
    {
        1, 1, 1,
        1, 1, 1,
        1, 1, 1,
    };
    UINT res = m_image.BoundaryExtraction(mask, 3);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnMorphologicalHitormiss()
{
    int Amask[9] = 
    {
        0, 0, 0,
        0, 1, 1, 
        0, 0, 0,

    };
    int Bmask[9] = 
    {
        0,0,0,
        1,0,0,
        0,0,0
    };

    UINT res = m_image.HitOrMiss(Amask, 3, Bmask, 3);

    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnPrettycolorsInvert()
{
    UINT res = m_image.Inverse();
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnGeometryLine()
{
    CPoint a, b;
    a.x = rand() % m_image.Width();
    a.y = rand() % m_image.Height();
    b.x = rand() % m_image.Width();
    b.y = rand() % m_image.Height();

    UINT res = m_image.Line(a,b, RGB(255,0,0), 1.0, true, 1, CISImageEx::noArrow, CISImageEx::noArrow);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnGeometryThickline()
{
    CPoint a, b;
    a.x = rand() % m_image.Width();
    a.y = rand() % m_image.Height();
    b.x = rand() % m_image.Width();
    b.y = rand() % m_image.Height();

    UINT res = m_image.Line(a,b, RGB(rand()%255,rand()%255,rand()%255), 1.0, true, 2 + rand() % 10, CISImageEx::arrow, CISImageEx::diamond);
    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnGeometryEllipse()
{
    CPoint c;
    c.x = rand() % (m_image.Width() / 2) + 10;
    c.y = rand() % (m_image.Height() / 2 ) + 10;

    UINT32 arad = rand() % (m_image.Width() / 3);
    UINT32 brad = rand() % (m_image.Height() / 3);

    UINT res = m_image.Ellipse(c, arad, brad, RGB(rand()%255,rand()%255,rand()%255), true, (double)(rand() % 100) / 100);

    reportRes(res);
    UpdateImage();
}

void CISImageDemoDoc::OnGeometryPolygon()
{
    std::vector<XYdouble> points;

    XYdouble t;

    t.x = (rand() % m_image.Width() / 4);
    t.y = (rand() % m_image.Height() / 4);
    points.push_back(t);

    t.x = (rand() % m_image.Width() / 4) + (3 * m_image.Width() / 4);
    t.y = (rand() % m_image.Height() / 4);
    points.push_back(t);

    t.x = (rand() % m_image.Width() / 4) + (3 * m_image.Width() / 4);
    t.y = (rand() % m_image.Height() / 4) + (3 * m_image.Height() / 4);
    points.push_back(t);

    t.x = (rand() % m_image.Width() / 4);
    t.y = (rand() % m_image.Height() / 4) + (3 * m_image.Height() / 4);
    points.push_back(t);

    UINT res = m_image.PolygonFill(RGB(rand()%255,rand()%255,rand()%255), .7, points, true);

    reportRes(res);
    UpdateImage();
}


void CISImageDemoDoc::OnPrettycolorsRetinex()
{
   CWaitCursor wait;

   m_image.Retinex(40,1,20,1.02);
   UpdateImage();
}


void CISImageDemoDoc::OnSaturationMore()
{
   m_image.AdjustSaturation(2.0, false);
   UpdateImage();
}


void CISImageDemoDoc::OnSaturationLess()
{
   m_image.AdjustSaturation(0.5, false);
   UpdateImage();
}


void CISImageDemoDoc::OnVibranceMore()
{
   m_image.AdjustSaturation(4.0, true);
   UpdateImage();
}


void CISImageDemoDoc::OnVibranceLess()
{
   m_image.AdjustSaturation(0.5, true);
   UpdateImage();
}


void CISImageDemoDoc::OnMoreimagestuffOverlay()
{
   // collect  all the other loaded images which aren't this one...
   std::vector< CISImageDemoDoc* > listDocs;
   bool ok = false;
   if (m_image.ISValid())
   {
      CISImageDemoApp * theApp = (CISImageDemoApp*)AfxGetApp();
      
      theApp->getImageDocs(listDocs, this);

      for (std::vector< CISImageDemoDoc* >::iterator it = listDocs.begin();
         it != listDocs.end();
         it++)
      {
         if ((*it)->m_image.ISValid())
         {
            ok = true;
            break;
         }
      }
   }

   if (!ok)
   {
      AfxMessageBox("You need to open another (non-animated GIF) image, so that I have something to overlay!");
   }
   else
   {
      COverlayPickerDlg dlg(listDocs);

      if (dlg.DoModal() == IDOK)
      {
         int docIdx = dlg.m_docIdx;

         double op = 1.0;
         CISImageEx::overlayMode ovl =  dlg.m_mode;

         if (ovl == CISImageEx::overlayMode::eBlendFast)
         {
            op = .5;
         }

         CISImageDemoDoc *pTopDoc = listDocs[docIdx];
         m_image.Merge(pTopDoc->m_image, 0, 0, ovl, op);
      }
   }

   UpdateImage();
}
