/************************************************************

   class CISImageEx

   For ImgSource v5.0

   This is a wrapper class for the ImgSource functions. This
   only uses a very small portion of the total ImgSource function
   list, but it should be simple enough to add whichever function
   you need, by looking at similar functions here.

   Sample use (load a BMP, reduce by 1/2, save as JPG) :

   CISImageEx image;
   image.LoadImage("C:\\input.bmp");

   CSize csSize = image.Size();
   csSize.cx/=2; csSize.cy/=2;
   image.Resize(csSize);

   image.SaveImage("C:\\output.jpg", eFmtJPG);

   ----------

   This uses most of the ImgSource functions, with a few major exceptions:

    * the colorspace conversions are not represented here because this
      class deals with BYTE data, and most colorspaces use floating point
      data for their pixel values.
    
    * there are many more overlay and alpha-blend functiosn in ImgSource

    * only a handful of the 'artistic' effects are shown here. 

    * only the basics of file reading are shown here (no line-by-line
      processing, none of the extended TIFF or JPG options, little of the 
      EXIF handling, etc..)

   ----------

   This class may be used without restriction.

   ImgSource and CISImageEx are copyright 2010-2014, Smaller Animals Software.

   All questions to : smallest@smalleranimals.com

   ************************************************************/

#include "stdafx.h"
#include "ISImageEx.h"
#include "math.h"
#include <algorithm>
#include <istream>
using namespace std;

#ifndef ISNEWSPEC
#if _MSC_VER <= 1200
#define ISNEWSPEC 
#else
#define ISNEWSPEC (std::nothrow)
#endif
#endif

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////

CISImageEx::CISImageEx()
{
    m_uWidth = m_uHeight = 0;
    m_uBitDepth = 0;
    m_uPalColors = 0;
    m_bHGLOBAL = true;

    m_pData = NULL;
    m_hData = NULL;

    m_fDPIX = m_fDPIY = 0.0;
    m_eDPIUnits = eDPINone;
    m_eSrcFmt = eFmtNone;
    m_iFrameIdx = 0;


    m_outDPI.uDPIUnits = eDPINone;
    m_outDPI.fDPIX = DEF_DPIX;
    m_outDPI.fDPIY = DEF_DPIY;
    m_JPGQuality = DEF_JPG_QUALITY;
    m_bProgressiveJPG = false;

    m_uBWTolerance = 128;

    m_bBmp8RLE = false;

    m_uTiffComp = eTiffCompNone;
}

//////////////////////////////////////////////////////////////////////

CISImageEx::~CISImageEx()
{
    Clear();
}

//////////////////////////////////////////////////////////////////////

void CISImageEx::Clear()
{
    if (m_bHGLOBAL)
    {
        _ASSERT(m_pData==NULL);
        if (m_hData)
        {
            GlobalFree(m_hData);
            m_hData = NULL;
            m_pData = NULL;
        }
    }
    else
    {
        _ASSERT(m_hData==NULL);
        if (m_pData)
        {
            delete [] m_pData;
            m_pData = NULL;
            m_hData = NULL;
        }
    }

    m_uWidth = m_uHeight = 0;
    m_uBitDepth = 0;
    m_uPalColors = 0;
}

//////////////////////////////////////////////////////////////////////

UINT32   CISImageEx::RowStride() const 
{
    switch (BitDepth())
    {
    case 1:
        return OneBitRowStride(Width());
        break;
    case 8:
        return Width();
        break;
    case 24:
        return Width() * 3;
        break;
    case 32:
       return Width() * 4;
       break;
    default:
        ASSERT(0);
        break;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::AllocBlank(const CISImageEx & target)
{
    UINT32 res = AllocBlank(target.Size(), target.BitDepth());
    CopyAttribs(target);
    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::AllocBlank(const CSize &cz, UINT32 d)
{
    return AllocBlank(cz.cx, cz.cy, d);
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::AllocBlank(UINT32 w, UINT32 h, UINT32 d)
{
    Clear();

    UINT size = GetDataSize(w, h, d);

    m_pData = new ISNEWSPEC BYTE [size];
    if (m_pData)
    {
        m_bHGLOBAL = false;
        m_uWidth = w;
        m_uHeight = h;
        m_uBitDepth = d;

        m_uPalColors = 0;
        if (d <= 8)
            m_uPalColors = (1 << d);

        return IS_ERR_OK;
    }

    return IS_ERR_MEM;
}

//////////////////////////////////////////////////////////////////////

UINT32     CISImageEx::GetDataSize(UINT w, UINT h, UINT bitdepth)
{
    UINT32 size = w * h * (bitdepth / 8);
    if (bitdepth == 1)
    {
        size = OneBitRowStride(w) * h;
    }

    return size;
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::Steal(CISImageEx &src)
{
    Clear();

    if (!src.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    // steal src's stuff
    if (src.m_bHGLOBAL)
    {
        m_bHGLOBAL = true;
        m_hData = src.m_hData;
        m_pData = NULL;

        // yoink
        src.m_bHGLOBAL = NULL;
    }
    else
    {
        m_bHGLOBAL = false;
        m_pData = src.m_pData;
        m_hData = NULL;

        // yoink
        src.m_pData = NULL;
    }

    m_uWidth = src.Width();
    m_uHeight = src.Height();
    m_uBitDepth = src.BitDepth();

    // copy the palette
    memcpy(Palette(), src.Palette(), src.PalColors() * sizeof(RGBQUAD));

    m_uPalColors = src.PalColors();

    m_eSrcFmt = src.m_eSrcFmt; 

    m_fDPIX = src.fDPIX();
    m_fDPIY = src.fDPIY();
    m_eDPIUnits = src.DPIU();

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::AssignBytes(BYTE *pData, UINT32 w, UINT32 h, UINT32 d, UINT32 uColors, RGBQUAD *pPal)
{
    Clear();

    if (pData==NULL)
    {
        return IS_ERR_BADPARAM;
    }

    m_pData = pData;
    m_bHGLOBAL = false;
    m_uWidth = w;
    m_uHeight = h;
    m_uBitDepth = d;
    m_uPalColors = uColors;

    if (pPal!=NULL)
    {
        // copy the palette
        memcpy(m_pal, pPal, uColors * sizeof(RGBQUAD));
    }

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::AssignHGLOBAL(HGLOBAL hData, UINT32 w, UINT32 h, UINT32 d, UINT32 uColors, RGBQUAD *pPal)
{
    Clear();

    if (hData==NULL)
    {
        return IS_ERR_BADPARAM;
    }

    m_hData = hData;
    m_bHGLOBAL = true;
    m_uWidth = w;
    m_uHeight = h;
    m_uBitDepth = d;
    m_uPalColors = uColors;

    if (pPal!=NULL)
    {
        // copy the palette
        memcpy(m_pal, pPal, uColors * sizeof(RGBQUAD));
    }

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::CopyAttribs(const CISImageEx& src)
{
    const RGBQUAD *pPal = src.ConstPalette();
    memcpy(m_pal, pPal, src.PalColors() * 4);
    m_uPalColors = src.PalColors();
    m_uBitDepth = src.BitDepth();

    return CopyMetaData(src);
}

//////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::CopyMetaData(const CISImageEx& src)
{
    // data that isn't affected by image processing operations
    m_fDPIX = src.fDPIX();
    m_fDPIY = src.fDPIY();
    m_eDPIUnits = src.DPIU();
    m_csFileName = src.m_csFileName;
    m_iFrameIdx = src.m_iFrameIdx;

    m_eSrcFmt = src.SrcFormat();

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::Copy(const CISImageEx& src)
{
    Clear();

    UINT res = AllocBlank(src);

    if (res == IS_ERR_OK)
    {
        memcpy(ImageData(), src.ImageDataConst(), GetDataSize(Width(), Height(), BitDepth()));

        CopyAttribs(src);
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::AddPalColor(COLORREF clr)
{
    if (BitDepth()!=8)
    {
        _ASSERT(0);
        return 0;
    }


    HISCOLORMATCH hcm = _ISFn(is6_CreateColorMatcher)(Palette(), PalColors());
    int match = _ISFn(is6_FindClosestColor)(hcm, GetRValue(clr), GetGValue(clr), GetBValue(clr));
    _ISFn(is6_DestroyColorMatcher)(hcm);

    if (m_uPalColors >= 256)
    {
        _ASSERT(0);
        return match;
    }

    COLORREF matchClr = RGB(m_pal[match].rgbRed, m_pal[match].rgbGreen, m_pal[match].rgbBlue);
    if (_ISFn(is6_ColorCompare)(matchClr, clr, 0))
    {
        return match;
    }

    RGBQUAD rgbq;
    rgbq.rgbRed = GetRValue(clr);
    rgbq.rgbGreen = GetGValue(clr);
    rgbq.rgbBlue = GetBValue(clr);
    *(DWORD *)&m_pal[m_uPalColors] = *(DWORD*)&rgbq;
    m_uPalColors++;

    return m_uPalColors - 1;
}

/////////////////////////////////////////////////////////////////////////////

UINT32   CISImageEx::SetPalColor(int idx, COLORREF clr)
{
    UINT32 res = IS_ERR_OK;

    if (idx >= (int)m_uPalColors)
    {
        res = IS_ERR_BADPARAM;
    }
    else
    {
        RGBQUAD rgbq;
        rgbq.rgbRed = GetRValue(clr);
        rgbq.rgbGreen = GetGValue(clr);
        rgbq.rgbBlue = GetBValue(clr);
        *(DWORD *)&m_pal[idx] = *(DWORD*)&rgbq;
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

void     CISImageEx::SetGrayPalette()
{
    m_uPalColors = 256;
    for (UINT i=0;i<256;i++)
    {
        m_pal[i].rgbBlue = m_pal[i].rgbGreen = m_pal[i].rgbRed = i;
        m_pal[i].rgbReserved = 0;
    }
}

/////////////////////////////////////////////////////////////////////////////

bool     CISImageEx::Is8BitGray() const
{
    if (ISValid() && BitDepth() == 8 && m_uPalColors == 256)
    {
        for (UINT i=0;i<256;i++)
        {
            if (m_pal[i].rgbRed!=i || m_pal[i].rgbGreen != i || m_pal[i].rgbRed != i)
                return false;
        }
        return true;
    }

    return false; 
}

/////////////////////////////////////////////////////////////////////////////

UINT32   CISImageEx::FromDC(HDC hDC, CRect rect)
{
    Clear();

    UINT res = AllocBlank(rect.Size(), 24);

    if (res == IS_ERR_OK)
    {
        _ISFn(is6_DCToRGB)(hDC, rect.left, rect.top, rect.Width(), rect.Height(), ImageData());

        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::CopyFromCBitmap(HBITMAP hBmp)
{
    Clear();

    if (!hBmp)
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 w, h, res = IS_ERR_OK;

    // get reference DC
    HDC hDC = GetDC(NULL);

    if (_ISFn(is6_GetHBITMAPDimensions(hBmp, &w, &h)))
    {
        res = AllocBlank(w,h,24);
        if (res==IS_ERR_OK)
        {
            res = _ISFn(is6_HBITMAPToRGB)(hBmp, hDC, NULL, ImageData());
        }
    }

    ReleaseDC(NULL, hDC);

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::CopyToCBitmap(CBitmap &cBmp, HPALETTE hPal, COLORREF alphaBackgroundColor)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    HPALETTE hUsePal = hPal;

    // try normal HBITMAPs first
    _ISFn(is6_SetHBITMAPCreateMethod)(0);

    // get reference DC
    HDC hDC = NULL;

    if (BitDepth() == 1)
    {
        hDC = CreateCompatibleDC(NULL);
    }
    else
    {
        hDC = GetDC(NULL);
    }

    HBITMAP hBmp = NULL;
    if (BitDepth()==32)
    {
       /*
       we'll just blend this RGBA onto a solid background and make that our RGB image
       */
       CISImageEx temp;

       temp.AllocBlank(Size(), 24);
       temp.Fill(alphaBackgroundColor);
       temp.AlphaBlendRGBA(*this, CPoint(0,0), bottomAlpha);

       hBmp = _ISFn(is6_RGBToHBITMAP)(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(), hUsePal, hDC);
    }
    else if (BitDepth()==24)
    {
        hBmp = _ISFn(is6_RGBToHBITMAP)(ImageData(), Width(), Height(), RowStride(), hUsePal, hDC);
    }
    else if (BitDepth()==8)
    {
        HGLOBAL hDIB = _ISFn(is6_8BitToDIB)(ImageData(), Width(), Height(), RowStride(), PalColors(), BitDepth(), Palette(), NULL, 0);

        if (hDIB)
        {
            hBmp = _ISFn(is6_DIBToHBITMAP)((BITMAPINFOHEADER*)hDIB, hUsePal, hDC);

            GlobalFree(hDIB);
        }
    }
    else if (BitDepth()==1)
    {
        HGLOBAL hDIB = _ISFn(is6_1BitToDIB)(ImageData(), Width(), Height(), RowStride(), Palette(), NULL, 0);

        if (hDIB)
        {
            hBmp = _ISFn(is6_DIBToHBITMAP)((BITMAPINFOHEADER*)hDIB, hUsePal, hDC);

            GlobalFree(hDIB);
        }
    }

    if (hBmp==NULL)
    {
        // try it with DIBSections
       _ISFn(is6_SetHBITMAPCreateMethod)(1);

       if (BitDepth()==32)
       {
          /*
          we'll just blend this RGBA onto a solid background and make that our RGB image
          */
          CISImageEx temp;

          temp.AllocBlank(Size(), 24);
          temp.Fill(alphaBackgroundColor);
          temp.AlphaBlendRGBA(*this, CPoint(0,0), bottomAlpha);

          hBmp = _ISFn(is6_RGBToHBITMAP)(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(), hUsePal, hDC);
       }
       else if (BitDepth()==24)
       {
            hBmp = _ISFn(is6_RGBToHBITMAP)(ImageData(), Width(), Height(), RowStride(), hUsePal, hDC);
        }
        else if (BitDepth()==8)
        {
            HGLOBAL hDIB = _ISFn(is6_8BitToDIB)(ImageData(), Width(), Height(), RowStride(), PalColors(), BitDepth(), Palette(), NULL, 0);

            if (hDIB)
            {
                hBmp = _ISFn(is6_DIBToHBITMAP)((BITMAPINFOHEADER*)hDIB, hUsePal, hDC);

                GlobalFree(hDIB);
            }
        }
        else if (BitDepth()==1)
        {
            HGLOBAL hDIB = _ISFn(is6_1BitToDIB)(ImageData(), Width(), Height(), RowStride(), Palette(), NULL, 0);

            if (hDIB)
            {
                hBmp = _ISFn(is6_DIBToHBITMAP)((BITMAPINFOHEADER*)hDIB, hUsePal, hDC);

                GlobalFree(hDIB);
            }
        }

        _ISFn(is6_SetHBITMAPCreateMethod)(0);
    }

    ReleaseDC(NULL, hDC);

    if (hBmp)
    {
        if (cBmp.GetSafeHandle())
            cBmp.DeleteObject();

        cBmp.Attach(hBmp);
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////////////////////

CSize CISImageEx::GetBestFitDims(CSize czPix)
{
    if (!ISValid())
    {
        ASSERT(0);
        return CSize(0,0);
    }

    //g_statusLog.StatusOut("%s, GetBestFItDims (%d, %d)", m_csName, czPix.cx, czPix.cy);

    double ratx = (double)Width() / (double)czPix.cx;
    double raty = (double)Height() / (double)czPix.cy;

    if ((ratx==0) || (raty==0))
    {
        return CSize(0,0);
    }

    CSize newSize = czPix;

    if (ratx > raty) 
    {
        newSize.cy = (UINT)((double)Height() / ratx);
    } 
    else if (ratx < raty)
    {
        newSize.cx = (UINT)((double)Width() / raty);
    } 

    if (newSize.cx == 0) newSize.cx = 1;
    if (newSize.cy == 0) newSize.cy = 1;

    return newSize;
}

//////////////////////////////////////////////////////////////////////

UINT CISImageEx::LoadImage(const TCHAR * pFullPath, eImgFmt inFmt, UINT uTIFFFrameIdx)
{
    //TRACE("Loading %s...\n", pFullPath);

    // clean up
    Clear();

    // no name???
    if (_tcslen(pFullPath)==0)
    {
        return IS_ERR_FILEOPEN;
    }

    eImgFmt eFmt = eFmtNone;

    if (inFmt != eFmtNone)
    {
        // the caller has decided to tell us how to read it...
        eFmt = inFmt;
    }
    else
    {
        eFmt = GuessFormat(pFullPath);
    }

    if (eFmt==eFmtNone)
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 uRes = IS_ERR_OK;

    // metafiles can only be read from disk, so we have to bypass the
    // ISource source manager stuff
    if ((eFmt==eFmtWmf) || (eFmt==eFmtEmf))
    {
        UINT32 uWidth, uHeight;
        HGLOBAL hRGB = _ISFn(is6_ReadMetafile)((const char *)pFullPath, &uWidth, &uHeight, 100, 100, RGB(0,0,0), 0);

        uRes = _ISFn(is6_GetLastError)();

        if (hRGB!=NULL)
        {
            m_bHGLOBAL = true;
            m_hData = hRGB;
            m_uWidth = uWidth;
            m_uHeight = uHeight;
            m_uBitDepth = 24;

            m_eSrcFmt = eFmt; 

            m_csFileName = pFullPath;

            m_iFrameIdx = 0;
        }
    }   
    else
    {
        // open it...
#ifdef _UNICODE
        HISSRC hSource = _ISFn(is6_OpenFileSourceW)(pFullPath);
#else
        HISSRC hSource = _ISFn(is6_OpenFileSource)(pFullPath);
#endif

        if (hSource==NULL)
        {
            return _ISFn(is6_GetLastError)();
        }

        // read it...
        uRes = LoadFile(hSource, eFmt, uTIFFFrameIdx);

        m_csFileName = pFullPath;

        // close it
        _ISFn(is6_CloseSource)(hSource);
    }

    return uRes;

}

//////////////////////////////////////////////////////////////////////

UINT   CISImageEx::LoadImage(BYTE *pMem, UINT32 uSize, eImgFmt inFmt, UINT32 uTIFFFrameIdx)
{
    HISSRC hSource = _ISFn(is6_OpenMemorySource)(pMem, uSize);
    if (hSource==NULL)
    {
        return _ISFn(is6_GetLastError)();
    }

    eImgFmt eFmt = eFmtNone;

    if (inFmt != eFmtNone)
    {
        // the caller has decided to tell us how to read it...
        eFmt = inFmt;
    }
    else
    {
        eFmt = GuessFormat(hSource);
    }

    if (eFmt==eFmtNone)
    {
        ASSERT(0);
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_Seek)(hSource, 0, 0);

    // read it...
    UINT res = LoadFile(hSource, eFmt, uTIFFFrameIdx);

    m_csFileName = "memory";

    // close it
    _ISFn(is6_CloseSource)(hSource);

    return res;

}

//////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::LoadFile(HISSRC hSource, eImgFmt eFmt, UINT uTIFFFrameIdx)
{
    UINT32 uWidth = 0, 
        uHeight = 0;

    HGLOBAL hImg = NULL;

    UINT32 uRes = IS_ERR_OK;

    // just drop into the appropriate reading function, based on the format

    // if you don't care about the detail of bit-depth and 
    // palettes, etc. you could just use is6_readImage.

    switch (eFmt)
    {
    case eFmtBmp:
        hImg = _ISFn(is6_ReadBMP)(hSource, &uWidth, &uHeight, 24, NULL, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 24;
        m_uPalColors=0;
        break;
    case eFmtJpg:
        hImg = _ISFn(is6_ReadJPG)(hSource, &uWidth, &uHeight, 24, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 24;
        m_uPalColors=0;
        break;
    case eFmtPng:
        hImg = _ISFn(is6_ReadPNG)(hSource, &uWidth, &uHeight, 24, NULL, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 24;
        m_uPalColors=0;
        break;
    case eFmtPng32:
        hImg = _ISFn(is6_ReadPNG)(hSource, &uWidth, &uHeight, 32, NULL, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 32;
        m_uPalColors=0;
        break;
    case eFmtPcx:
        hImg = _ISFn(is6_ReadPCX)(hSource, &uWidth, &uHeight, 24, NULL, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 24;
        m_uPalColors=0;
        break;
    case eFmtPsd:
        hImg = _ISFn(is6_ReadPSD)(hSource, &uWidth, &uHeight, 24, NULL, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 24;
        m_uPalColors=0;
        break;
    case eFmtPCD:
        hImg = _ISFn(is6_ReadPCD)(hSource, &uWidth, &uHeight, DEF_PHOTOCD_RES, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 24;
        m_uPalColors=0;
        break;
    case eFmtTif:
        hImg = _ISFn(is6_ReadTIFF)(hSource, &uWidth, &uHeight, 24, NULL, uTIFFFrameIdx, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_iFrameIdx = uTIFFFrameIdx;
        m_uBitDepth = 24;
        m_uPalColors=0;
        break;
    case eFmtTif32:
        hImg = _ISFn(is6_ReadTIFF)(hSource, &uWidth, &uHeight, 32, NULL, uTIFFFrameIdx, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_iFrameIdx = uTIFFFrameIdx;
        m_uBitDepth = 32;
        m_uPalColors=0;
        break;
    case eFmtTif8:
        hImg = _ISFn(is6_ReadTIFF)(hSource, &uWidth, &uHeight, 8, Palette(), uTIFFFrameIdx, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uPalColors = 256;
        m_uBitDepth = 8;
        m_iFrameIdx = uTIFFFrameIdx;
        break;
    case eFmtTif1:
        hImg = _ISFn(is6_ReadTIFF)(hSource, &uWidth, &uHeight, 1, Palette(), uTIFFFrameIdx, 0);
        uRes = _ISFn(is6_GetLastError)();

        // force this palette. 
        Palette()[0].rgbBlue = Palette()[0].rgbGreen = Palette()[0].rgbRed = 255;
        Palette()[1].rgbBlue = Palette()[1].rgbGreen = Palette()[1].rgbRed = 0;

        m_uPalColors = 2;
        m_uBitDepth = 1;
        m_iFrameIdx = uTIFFFrameIdx;
        break;
    case eFmtGIF:
        {
            __int32 tc, bc; // we're ignoring the transparent color in this class
            BOOL in;
            UINT32 bd;
            hImg = _ISFn(is6_ReadGIF)(hSource, &uWidth, &uHeight, &tc, Palette(), 0);
            uRes = _ISFn(is6_GetLastError)();
            if (hImg)
            {
                // we need the number of colors in the palette. so, rewind and read it again
                _ISFn(is6_Seek)(hSource, 0, 0);

                UINT32 dummyW, dummyH; // don't care about these
                _ISFn(is6_GetGIFDims)(hSource, &dummyW, &dummyH, &bd, &bc, &tc, Palette(), &in, 0);

                m_uPalColors = 1 << bd;
                m_uBitDepth = 8;
            }
        }
        break;
    case eFmtTga:
        hImg = _ISFn(is6_ReadTGA)(hSource, &uWidth, &uHeight, 24, NULL, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 24;
        m_uPalColors=0;
        break;
    case eFmtTga32:
        hImg = _ISFn(is6_ReadTGA)(hSource, &uWidth, &uHeight, 32, NULL, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 32;
        m_uPalColors=0;
        break;
    case eFmtBmp8:
        hImg = _ISFn(is6_ReadBMP)(hSource, &uWidth, &uHeight, 8, Palette(), 0);
        uRes = _ISFn(is6_GetLastError)();
        if (hImg)
        {
            // we need the number of colors in the palette. so, rewind and read it again
            _ISFn(is6_Seek)(hSource, 0, 0);
            _ISFn(is6_GetBMPPalette)(hSource, &m_uPalColors, Palette(), 0);
            m_uBitDepth = 8;
        }
        break;
    case eFmtBmp1:
        hImg = _ISFn(is6_ReadBMP)(hSource, &uWidth, &uHeight, 1, Palette(), 0);
        uRes = _ISFn(is6_GetLastError)();
        if (hImg)
        {
            // we need the number of colors in the palette. so, rewind and read it again
            _ISFn(is6_Seek)(hSource, 0, 0);
            _ISFn(is6_GetBMPPalette)(hSource, &m_uPalColors, Palette(), 0);
            m_uBitDepth = 1;
        }
        break;
    case eFmtPcx8:
        hImg = _ISFn(is6_ReadPCX)(hSource, &uWidth, &uHeight, 8, Palette(), 0);  
        uRes = _ISFn(is6_GetLastError)();
        if (hImg)
        {
            // we need the number of colors in the palette. so, rewind and read it again
            _ISFn(is6_Seek)(hSource, 0, 0);
            _ISFn(is6_GetPCXPalette)(hSource, &m_uPalColors, Palette(), 0);
            m_uBitDepth = 8;
        }
        break;
    case eFmtPng8:					    
        hImg = _ISFn(is6_ReadPNG)(hSource, &uWidth, &uHeight, 8, Palette(), 0);
        uRes = _ISFn(is6_GetLastError)();
        if (hImg)
        {
            // we need the number of colors in the palette. so, rewind and read it again
            _ISFn(is6_Seek)(hSource, 0, 0);
            _ISFn(is6_GetPNGPalette)(hSource, &m_uPalColors, Palette(), 0);
            m_uBitDepth = 8;
        }
        break;
    case eFmtPng1:					    
        hImg = _ISFn(is6_ReadPNG)(hSource, &uWidth, &uHeight, 1, Palette(), 0);
        uRes = _ISFn(is6_GetLastError)();
        if (hImg)
        {
            // we need the number of colors in the palette. so, rewind and read it again
            _ISFn(is6_Seek)(hSource, 0, 0);
            _ISFn(is6_GetPNGPalette)(hSource, &m_uPalColors, Palette(), 0);
            m_uBitDepth = 1;
        }
        break;
    case eFmtTga8:		
        hImg = _ISFn(is6_ReadTGA)(hSource, &uWidth, &uHeight, 8, Palette(), 0);
        uRes = _ISFn(is6_GetLastError)();
        if (hImg)
        {
            // we need the number of colors in the palette. so, rewind and read it again
            _ISFn(is6_Seek)(hSource, 0, 0);
            _ISFn(is6_GetTGAPalette)(hSource, &m_uPalColors, Palette(),0 );
            m_uBitDepth = 8;
        }
        break;
    case eFmtPsd8:
        hImg = _ISFn(is6_ReadPSD)(hSource, &uWidth, &uHeight, 8, Palette(), 0);
        uRes = _ISFn(is6_GetLastError)();
        if (hImg)
        {
            // we need the number of colors in the palette. so, rewind and read it again
            _ISFn(is6_Seek)(hSource, 0, 0);
            _ISFn(is6_GetPSDPalette)(hSource, &m_uPalColors, Palette(), 0);
            m_uBitDepth = 8;
        }
        break;
    case eFmtJpg8:	
        hImg = _ISFn(is6_ReadJPG)(hSource, &uWidth, &uHeight, 8, 0);
        uRes = _ISFn(is6_GetLastError)();

        // grayscale palette
        {for (int i=0;i<256;i++) memset(&m_pal[i], i, sizeof(RGBQUAD));}
        m_uBitDepth = 8;
        m_uPalColors=256;
        break;
    case eFmtWBmp:	
        hImg = _ISFn(is6_ReadWBMP)(hSource, &uWidth, &uHeight, m_pal, 8, 0);
        uRes = _ISFn(is6_GetLastError)();
        m_uBitDepth = 8;
        m_uPalColors = 2;
        break;
    default:
        // unsupported format type
        // (metafiles are handled in the LoadFromFile functions)
        _ASSERT(FALSE);
        break;
    }

    if (hImg!=NULL)
    {                                                     
        GetDPI(hSource, eFmt);
        m_outDPI.uDPIUnits = DPIU();
        m_outDPI.fDPIX = DPIX();
        m_outDPI.fDPIY = DPIY();

        m_hData = hImg;
        m_bHGLOBAL = true;
        m_uWidth = uWidth;
        m_uHeight = uHeight;

        m_eSrcFmt = eFmt; 

        // DPI compensation
        // some images have non-equal DPI for the X and Y directions. we have to resize
        // to get a correct aspect ration for display
        if ((eFmt==eFmtTif) || (eFmt==eFmtTif8))
        {
            if (m_fDPIX != m_fDPIY)
            {
                if (m_fDPIX != 0)
                {
                    double fRat = (double)m_fDPIY / (double)m_fDPIX;
                    if (fRat!=0)
                    {
                        if (fabs(1.0 - fRat) > 0.25)
                        {
                            UINT32 uNewY = (UINT)((double)m_uHeight / fRat);
                            CISImageEx t;

                            t.ResizeCopy(*this, m_uWidth, uNewY, 0);

                            Steal(t);
                        }
                    }
                }
            }
        }
    }

    return uRes;
}

//////////////////////////////////////////////////////////////////////

CISImageEx::eImgFmt CISImageEx::GuessFormat(const TCHAR *pFilename)
{
#ifdef _UNICODE
    HISSRC hSource = _ISFn(is6_OpenFileSourceW)(pFilename);
#else
    HISSRC hSource = _ISFn(is6_OpenFileSource)(pFilename);
#endif

    if (hSource==NULL)
    {
        return eFmtNone;
    }

    eImgFmt eFmt = GuessFormat(hSource, pFilename);

    _ISFn(is6_CloseSource)(hSource);

    return eFmt;
}

//////////////////////////////////////////////////////////////////////

CISImageEx::eImgFmt CISImageEx::GuessFormat(HISSRC hSource, const TCHAR *pFilename)
{
    // this will tell us the format
    UINT32 uFmt = _ISFn(is6_GuessFileType)(hSource);

    UINT32 w, h, d, i = 0, t = 0, f, m;
    eImgFmt eFmt;

    switch (uFmt)
    {
    default:
    case 0:
        {
            if (pFilename!=NULL)
            {
                // get input file extension
                CString ext, csFullPath(pFilename);
                int iDotPos = csFullPath.ReverseFind('.');
                if (iDotPos!=-1) 
                {
                    ext = csFullPath.Mid(iDotPos);
                }

                // ImgSource couldn't tell us. we'll have to try the file extension
                eFmt = FmtFromExt(ext);
            }
        }
        break;
    case 1: eFmt = eFmtBmp; break;
    case 2: eFmt = eFmtGIF; break;
    case 3: eFmt = eFmtJpg; break;
    case 4: eFmt = eFmtPng; break;
    case 5: eFmt = eFmtPcx; break;
    case 6: eFmt = eFmtTif; break;
    case 7: eFmt = eFmtWmf; break;
    case 8: eFmt = eFmtEmf; break;
    case 10: eFmt = eFmtPsd; break;
    case 13: eFmt = eFmtWBmp; break;
    case 14: eFmt = eFmtTga; break;
    case 16: eFmt = eFmtPCD; break;
    }

    // now we'll try to get bit depth info

    // rewind
    _ISFn(is6_Seek)(hSource, 0,0);

    switch (eFmt)
    {
    default:
        _ASSERT(0);
        break;
    case eFmtWmf:
    case eFmtEmf:
        // nothing to do here.
        // these are always 24 bits
        break;
        // for the rest, we'll read to the lowest bit depth we can (out of 8 or 1)
    case eFmtBmp:
        if (_ISFn(is6_GetBMPDims)(hSource, &w, &h, &d, NULL, 0))
            if (d <= 8)
            {
                eFmt = eFmtBmp8;
                if (d == 1)
                {
                    eFmt = eFmtBmp1;
                }
            }
            break;

    case eFmtGIF:
        {
            __int32 tc, bc; // we're ignoring the transparent color in this class
            BOOL in;
            if (_ISFn(is6_GetGIFDims)(hSource, &w, &h, &d, &bc, &tc, NULL, &in, 0))
            {
                eFmt = eFmtGIF;
            }
        }
        break;

    case eFmtJpg:
        if (_ISFn(is6_GetJPGDims)(hSource, &w, &h, &d, &t, &f, NULL, 0))
            if (d <= 8)
            {
                eFmt = eFmtJpg8;
            }
            break;
    case eFmtPng:
        {
            UINT32 tr,tg,tb,tgr, tidx;
            if (_ISFn(is6_GetPNGDims)(hSource, &w, &h, &d, &t, &i, &tidx, &tr, &tg, &tb, &tgr, NULL, 0))
            {
               if (d==8 && t==6)
               {
                  eFmt = eFmtPng32;
               }
               else if ((d == 1))
               {
                  eFmt = eFmtPng1;
               }
               else if ((d <= 8) && ((t==3) || (t==0))) // if paletted, or grayscale
               {
                  eFmt = eFmtPng8;
               }
            }
        }
        break;
    case eFmtPcx:
       if (_ISFn(is6_GetPCXDims)(hSource, &w, &h, &d, NULL, 0))
          if (d <= 8)
          {
             eFmt = eFmtPcx8;
          }
          break;
    case eFmtTga:
       if (_ISFn(is6_GetTGADims)(hSource, &w, &h, &d, 0))
       {
          if (d==32)
          {
             eFmt = eFmtTga32;
          }
          else if (d <= 8)
          {
             eFmt = eFmtTga8;
          }
       }
       break;
    case eFmtTif:
       if (_ISFn(is6_GetTIFFDims)(hSource, 0, &w, &h, &d, NULL, 0))
       {
          if (d==32)
          {
             eFmt = eFmtTif32;
          }
          else if (d <= 8)
          {
             eFmt = eFmtTif8;

             if (d==1)
             {
                eFmt = eFmtTif1;
             }
          }
       }
            break;
    case eFmtPsd:
        if (_ISFn(is6_GetPSDDims)(hSource, &w, &h, &d, &m, NULL, 0))
            if ((d == 0) || (d == 1) || (d == 2))
            {
                eFmt = eFmtPsd8;
            }
            break;
    case eFmtWBmp:
        _ISFn(is6_GetWBMPDims)(hSource, &w, &h, 0);
        break;
    case eFmtPCD:
        break;
    }

    return eFmt;
}

/*
decide which output format best fits the image we have
*/
CISImageEx::eImgFmt CISImageEx::FindBestOutputFormat(CISImageEx::eImgFmt eSaveFmt)
{
   switch (eSaveFmt)
   {
   default:       // default to JPG
      ASSERT(0);
   case CISImageEx::eFmtNone:
   case CISImageEx::eFmtPCD:  // can't write PCD
   case CISImageEx::eFmtJpg8: // don't know how to handle this one
   case CISImageEx::eFmtJpg:
      // only one option here... JPG-24
      eSaveFmt = CISImageEx::eFmtJpg;
      break;
   case CISImageEx::eFmtBmp:
   case CISImageEx::eFmtBmp1:
   case CISImageEx::eFmtBmp8:
      switch (BitDepth())
      {
      case 1:
         // if the image is 1-bit, save as 1-bit
         eSaveFmt = CISImageEx::eFmtBmp1;
         break;
      case 8:
         // if the image is 8-bit, save as 8-bit
         eSaveFmt = CISImageEx::eFmtBmp8;
         break;
      default:
         break;
      }
      break;
   case CISImageEx::eFmtPcx:   
   case CISImageEx::eFmtPcx8:   
      switch (BitDepth())
      {
      case 8:
      case 1:
         eSaveFmt = CISImageEx::eFmtPcx8;
         break;
      default:
         break;
      }
      break;
   case CISImageEx::eFmtPng:
   case CISImageEx::eFmtPng1:
   case CISImageEx::eFmtPng8:
   case CISImageEx::eFmtPng32:
      switch (BitDepth())
      {
      case 1:
         eSaveFmt = CISImageEx::eFmtPng1;
         break;
      case 8:
         eSaveFmt = CISImageEx::eFmtPng8;
         break;
      case 32:
         eSaveFmt = CISImageEx::eFmtPng32;
         break;
      default:
         break;
      }
      break;
   case CISImageEx::eFmtTif:
   case CISImageEx::eFmtTif1:
   case CISImageEx::eFmtTif8:
   case CISImageEx::eFmtTif32:
      switch (BitDepth())
      {
      case 1:
         eSaveFmt = CISImageEx::eFmtTif1;
         break;
      case 8:
         eSaveFmt = CISImageEx::eFmtTif8;
         break;
      case 32:
         eSaveFmt = CISImageEx::eFmtTif32;
         break;
      default:
         break;
      }
      break;
   case CISImageEx::eFmtTga:	
   case CISImageEx::eFmtTga8:
   case CISImageEx::eFmtTga32:
      switch (BitDepth())
      {
      case 1:
      case 8:
         eSaveFmt = CISImageEx::eFmtTga8;
         break;
      case 32:
         eSaveFmt = CISImageEx::eFmtTga32;
         break;
      default:
         break;
      }
      break;
   case CISImageEx::eFmtPsd:
   case CISImageEx::eFmtPsd8:
      switch (BitDepth())
      {
      case 1:
      case 8:
         eSaveFmt = CISImageEx::eFmtPsd8;
         break;
      default:
         break;
      }
      break;
   case CISImageEx::eFmtWmf:
   case CISImageEx::eFmtEmf:
   case CISImageEx::eFmtWBmp:
   case CISImageEx::eFmtGIF:
      // nothing to do here
      break;
   }

   return eSaveFmt;
}

//////////////////////////////////////////////////////////////////////

int     CISImageEx::GetMultiFrameCount(const TCHAR *pFile)
{
    eImgFmt eFmt = GuessFormat(pFile);

    switch (eFmt)
    {
    case eFmtTif:
    case eFmtTif8:
    case eFmtTif1:
        {
#ifdef _UNICODE
            HISSRC hSource = _ISFn(is6_OpenFileSourceW)(pFile);
#else
            HISSRC hSource = _ISFn(is6_OpenFileSource)(pFile);
#endif
            if (hSource==NULL)
            {
                return 0;
            }

            UINT cnt = _ISFn(is6_GetTIFFPageCount)(hSource);

            _ISFn(is6_CloseSource)(hSource);

            return cnt;
        }
        break;
    default:
        break;
    }

    return 0;
}

/////////////////////////////////////////////////////////////////////////////
/*
set the DPI values in dpi according to the supported units of the 
file format fmt, and the value in m_outDPI
*/
void CISImageEx::SetDPI(eImgFmt fmt, IS4DPIStruct &dpi)
{
    switch (fmt)
    {
    case eFmtJpg:
    case eFmtJpg8:
        dpi.fDPIX = m_outDPI.fDPIX;
        dpi.fDPIY = m_outDPI.fDPIY;
        switch (m_outDPI.uDPIUnits)
        {
        case eDPINone: 
            dpi.uDPIUnits = 0;
            break;
        case eDPIInch: 
            dpi.uDPIUnits = 1;
            break;
        case eDPIcm: 
            dpi.uDPIUnits = 2;
            break;
        }
        break;
    case eFmtTif:
    case eFmtTif8:
       case eFmtTif32:
    case eFmtTif1:
        dpi.fDPIX = m_outDPI.fDPIX;
        dpi.fDPIY = m_outDPI.fDPIY;
        switch (m_outDPI.uDPIUnits)
        {
        case eDPINone: 
            dpi.uDPIUnits = 1;
            break;
        case eDPIInch: 
            dpi.uDPIUnits = 2;
            break;
        case eDPIcm: 
            dpi.uDPIUnits = 3;
            break;
        }
        break;
    case eFmtPng:
    case eFmtPng8:
    case eFmtPng1:
       case eFmtPng32:
        // png has dots per meter
        switch (m_outDPI.uDPIUnits)
        {
        case eDPINone: 
            dpi.fDPIX = 0; dpi.fDPIY = 0; dpi.uDPIUnits = 0;
            break;
        case eDPIInch: 
            dpi.fDPIX = m_outDPI.fDPIX * 39.37; // inches per meter
            dpi.fDPIY = m_outDPI.fDPIY * 39.37;
            dpi.uDPIUnits = 1;
            break;
        case eDPIcm: 
            dpi.fDPIX = m_outDPI.fDPIX * 100;   // cm per meter
            dpi.fDPIY = m_outDPI.fDPIY * 100;
            dpi.uDPIUnits = 1;
            break;
        }
        break;
    case eFmtBmp:
    case eFmtBmp1:
    case eFmtBmp8:
        // bmp has dots per meter
        switch (m_outDPI.uDPIUnits)
        {
        case eDPINone: 
            dpi.fDPIX = 0; dpi.fDPIY = 0; dpi.uDPIUnits = 0;
            break;
        case eDPIInch: 
            dpi.fDPIX = m_outDPI.fDPIX * 39.37; // inches per meter
            dpi.fDPIY = m_outDPI.fDPIY * 39.37;
            dpi.uDPIUnits = 1;
            break;
        case eDPIcm: 
            dpi.fDPIX = m_outDPI.fDPIX * 100;   // cm per meter
            dpi.fDPIY = m_outDPI.fDPIY * 100;
            dpi.uDPIUnits = 1;
            break;
        }
        break;
    default:
        dpi.fDPIX = dpi.fDPIY = 0; dpi.uDPIUnits = eDPINone;
        break;
    }
}

/////////////////////////////////////////////////////////////////////////////

void CISImageEx::GetDPI(HISSRC hSource, eImgFmt eFmt)
{
    // DPI
    IS4DPIStruct dpi;
    UINT32 d =0;

    // DPI
    switch (eFmt)
    {
    case eFmtJpg:
    case eFmtJpg8:
        _ISFn(is6_Seek)(hSource, 0, 0);
        _ISFn(is6_GetJPGDims)(hSource, &d, &d, &d, &d, &d, &dpi, 0);
        m_fDPIX = dpi.fDPIX;
        m_fDPIY = dpi.fDPIY;
        switch (dpi.uDPIUnits)
        {
        default:
        case 0: m_eDPIUnits = eDPINone; break;
        case 1: m_eDPIUnits = eDPIInch; break;
        case 2: m_eDPIUnits = eDPIcm; break;
        }
        break;
    case eFmtPng:
    case eFmtPng8:
    case eFmtPng1:
    case eFmtPng32:
        // DPI
        _ISFn(is6_Seek)(hSource, 0, 0);
        _ISFn(is6_GetPNGDims)(hSource, &d, &d, &d, &d, &d, &d, &d, &d, &d, &d, &dpi, 0);
        m_fDPIX = dpi.fDPIX;
        m_fDPIY = dpi.fDPIY;
        switch (dpi.uDPIUnits)
        {
        default:
        case 0: m_eDPIUnits = eDPINone; break;
        case 1: m_eDPIUnits = eDPIcm; m_fDPIX/=100; m_fDPIY/=100; break;
        }
        break;
    case eFmtTif:
    case eFmtTif8:
    case eFmtTif1:
    case eFmtTif32:
        // DPI
        _ISFn(is6_Seek)(hSource, 0, 0);
        _ISFn(is6_GetTIFFDims)(hSource, 0, &d, &d, &d, &dpi, 0);
        m_fDPIX = dpi.fDPIX;
        m_fDPIY = dpi.fDPIY;
        switch (dpi.uDPIUnits)
        {
        default:
        case 1: m_eDPIUnits = eDPINone; break;
        case 2: m_eDPIUnits = eDPIInch; break;
        case 3: m_eDPIUnits = eDPIcm; break;
        }
        break;
    case eFmtBmp:
    case eFmtBmp8:
        // DPI
        _ISFn(is6_Seek)(hSource, 0, 0);
        _ISFn(is6_GetBMPDims)(hSource, &d, &d, &d, &dpi, 0);
        m_fDPIX = dpi.fDPIX;
        m_fDPIY = dpi.fDPIY;
        switch (dpi.uDPIUnits)
        {
        default:
            m_eDPIUnits = eDPINone; 
            break;
        case 2: 
            m_eDPIUnits = eDPIcm; 
            m_fDPIX/=100; 
            m_fDPIY/=100; 
            break;
        }
        break;

    default:
        m_fDPIX = DEF_DPIX; m_fDPIY = DEF_DPIY; m_eDPIUnits = eDPINone;
        break;
    }
}

//////////////////////////////////////////////////////////////////////

CISImageEx::eImgFmt CISImageEx::FmtFromExt(const TCHAR * fileName)
{
    const TCHAR *pDot = _tcsrchr(fileName, _T('.'));
    if (pDot==NULL)
    {
        return CISImageEx::eFmtNone;
    }

    if (_tcsicmp(pDot, _T(".bmp"))==0) 
    {
        return CISImageEx::eFmtBmp;
    }

    if (_tcsicmp(pDot, _T(".wbmp"))==0) 
    {
        return CISImageEx::eFmtWBmp;
    }

    if (_tcsicmp(pDot, _T(".wbm"))==0) 
    {
        return CISImageEx::eFmtWBmp;
    }

    if (_tcsicmp(pDot, _T(".tga"))==0) 
    {
        return CISImageEx::eFmtTga;
    }

    if ((_tcsicmp(pDot, _T(".jpeg"))==0) || (_tcsicmp(pDot, _T(".jpg"))==0))
    {
        return CISImageEx::eFmtJpg;
    }																	    

    if ((_tcsicmp(pDot, _T(".tif"))==0) || (_tcsicmp(pDot, _T(".tiff"))==0))
    {
        return CISImageEx::eFmtTif;
    }

    if (_tcsicmp(pDot, _T(".pcx"))==0)
    {
        return CISImageEx::eFmtPcx;
    }

    if (_tcsicmp(pDot, _T(".gif"))==0)
    {
        return CISImageEx::eFmtGIF;
    }

    if (_tcsicmp(pDot, _T(".png"))==0)
    {
        return CISImageEx::eFmtPng;
    }

    if (_tcsicmp(pDot, _T(".wmf"))==0)
    {
        return CISImageEx::eFmtWmf;
    }

    if (_tcsicmp(pDot, _T(".emf"))==0)
    {
        return CISImageEx::eFmtEmf;
    }

    if (_tcsicmp(pDot, _T(".psd"))==0)
    {
        return CISImageEx::eFmtPsd;
    }

    return CISImageEx::eFmtNone;
}

//////////////////////////////////////////////////////////////////////

UINT CISImageEx::SaveImage(const TCHAR *csPath, eImgFmt eFmt)
{
    if (!ISValid())
        return IS_ERR_BADPARAM;

    UINT res = IS_ERR_OK;
#ifdef _UNICODE
    HISDEST hDest = _ISFn(is6_OpenFileDestW)(csPath, FALSE);
#else
    HISDEST hDest = _ISFn(is6_OpenFileDest)(csPath, FALSE);
#endif
    if (hDest==NULL)
        return _ISFn(is6_GetLastError)();

    if (eFmt==CISImageEx::eFmtNone)
    {
        // the caller didn't specify an output type.
        // we'll just guess from the file extension.
        eFmt = FmtFromExt(csPath);
        if (eFmt==CISImageEx::eFmtNone)
            return IS_ERR_BADPARAM;
    }

    res = SaveFile(hDest, eFmt);

    _ISFn(is6_CloseDest)(hDest);

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT CISImageEx::SaveFile(HISDEST hDest, eImgFmt eFmt)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    switch (eFmt)
    {
    case eFmtPng32:
    case eFmtTif32:
    case eFmtTga32:
        // expand
        if (BitDepth() != 32)
        {
            CISImageEx temp;
            Make32BitCopy(temp);
            res = temp.Save32Bit(hDest, eFmt);
        }
        else
        {
            res = Save32Bit(hDest, eFmt);
        }
        break;
    case eFmtBmp:
    case eFmtJpg:
    case eFmtPcx:
    case eFmtPng:
    case eFmtTif:
    case eFmtTga:
    case eFmtWmf:
    case eFmtEmf:
    case eFmtPsd:
        // expand
        if (BitDepth() != 24)
        {
            CISImageEx temp;
            Make24BitCopy(temp);
            res = temp.Save24Bit(hDest, eFmt);
        }
        else
        {
            res = Save24Bit(hDest, eFmt);
        }
        break;

        // we'll get to the other formats later
    default:
        break;
    }

    // save to the appropriate bit depth
    switch (eFmt)
    {
    case eFmtTif1:
    case eFmtPng1:
    case eFmtBmp1:
        if (BitDepth() != 1)
        {
            CISImageEx temp;
            Make1BitCopy(temp, 0, 8);
            res = temp.Save1Bit(hDest, eFmt);
        }
        else
        {
            res = Save1Bit(hDest, eFmt);
        }

        break;

    case eFmtBmp8:
    case eFmtPng8:
    case eFmtPcx8:
    case eFmtTif8:
    case eFmtTga8:
    case eFmtPsd8:
    case eFmtGIF:
        if (BitDepth() != 8)
        {
            CISImageEx temp;
            Make8BitCopy(temp);
            res = temp.Save8Bit(hDest, eFmt);
        }
        else
        {
            res = Save8Bit(hDest, eFmt);
        }
        break;

    case eFmtWBmp:
        {
            CISImageEx temp;
            res = Make8BitBWCopy(temp, m_uBWTolerance);
            res = temp.Save8Bit(hDest, eFmt);
        }
        break;

    case eFmtJpg8:
        {
            CISImageEx temp;
            res = Make8BitGrayCopy(temp);
            res = temp.Save8Bit(hDest, eFmt);
        }
        break;
    }

    // how'd we do?
    res = _ISFn(is6_GetLastError)();
    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Save32Bit(HISDEST hDest, eImgFmt eFmt)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 32)
    {
        // need to convert image to 32bpp first
        return IS_ERR_BADPARAM;
    }

    IS4DPIStruct dpi;
    SetDPI(eFmt, dpi);

    switch (eFmt)
    {
    case eFmtPng32:
        _ISFn(is6_ClearPNGOutputText)();
        _ISFn(is6_WritePNG)(hDest, ImageData(), Width(), Height(), RowStride(), 8, 0, NULL, 6, DEF_PNG_GAMMA, &dpi, 0);
        break;
    case eFmtTif32:
        _ISFn(is6_WriteTIFF)(hDest, ImageData(), Width(), Height(), 32, RowStride(), NULL, m_uTiffComp, &dpi, 0);
        break;
    case eFmtTga32:
        _ISFn(is6_WriteTGA)(hDest, ImageData(), Width(), Height(), 32, RowStride(), NULL, NULL, 0);
        break;
    }

    // how'd we do?
    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Save24Bit(HISDEST hDest, eImgFmt eFmt)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24)
    {
        // need to convert image to 24bpp first
        return IS_ERR_BADPARAM;
    }

    IS4DPIStruct dpi;
    SetDPI(eFmt, dpi);

    switch (eFmt)
    {
    case eFmtBmp:
        _ISFn(is6_WriteBMP)(hDest, ImageData(), Width(), Height(), 24, RowStride(), 1 << 24, NULL, &dpi, 0);
        break;
    case eFmtJpg:
        _ISFn(is6_ClearJPGOutputMarkers)();
        _ISFn(is6_ClearJPGOutputText)();
        _ISFn(is6_WriteJPG)(hDest, ImageData(), Width(), Height(), 24, RowStride(), m_JPGQuality, &dpi, m_bProgressiveJPG ? 1 : 0);
        break;
    case eFmtPcx:
        _ISFn(is6_WritePCX)(hDest, ImageData(), Width(), Height(), 24, RowStride(), NULL, &dpi, 0);
        break;
    case eFmtPng:
        _ISFn(is6_ClearPNGOutputText)();
        _ISFn(is6_WritePNG)(hDest, ImageData(), Width(), Height(), RowStride(), 8, 0, NULL, 2, DEF_PNG_GAMMA, &dpi, 0);
        break;
    case eFmtTif:
        _ISFn(is6_WriteTIFF)(hDest, ImageData(), Width(), Height(), 24, RowStride(), NULL, m_uTiffComp, &dpi, 0);
        break;
    case eFmtTga:
        _ISFn(is6_WriteTGA)(hDest, ImageData(), Width(), Height(), 24, RowStride(), NULL, NULL, 0);
        break;
    case eFmtWmf:
        _ISFn(is6_WriteRGBToMetafile)(hDest, ImageData(), Width(), Height(), RowStride(), 0, 0);
        break;
    case eFmtEmf:
        _ISFn(is6_WriteRGBToMetafile)(hDest, ImageData(), Width(), Height(), RowStride(), 1, 0);
        break;
    case eFmtPsd:
        _ISFn(is6_WritePSD)(hDest, ImageData(), Width(), Height(), 24, RowStride(), NULL, 0);
        break;
    }

    // how'd we do?
    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Save8Bit(HISDEST hDest, eImgFmt eFmt)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    IS4DPIStruct dpi;
    SetDPI(eFmt, dpi);

    BYTE *pImageData = ImageData();

    if (BitDepth() != 8)
    {
        // need to convert image to 8bpp first
        return IS_ERR_BADPARAM;
    }

    // switch on output type
    switch (eFmt)
    {
    case eFmtJpg8:
        {
            _ISFn(is6_ClearJPGOutputText)();
            _ISFn(is6_ClearJPGOutputMarkers)();
            _ISFn(is6_WriteJPG)(hDest, pImageData, Width(), Height(), 8, RowStride(), m_JPGQuality, &dpi, m_bProgressiveJPG ? 1 : 0);
        }
        break;

    case eFmtPng8:
        {
            _ISFn(is6_ClearPNGOutputText)();
            _ISFn(is6_WritePNG)(hDest, pImageData, Width(), Height(), RowStride(), 8, PalColors(), Palette(), 3, DEF_PNG_GAMMA, &dpi, 0);
        }
        break;

    case eFmtPcx8:
        {
            _ISFn(is6_WritePCX)(hDest, pImageData, Width(), Height(), 8, RowStride(), Palette(), &dpi, 0);
        }
        break;

    case eFmtPsd8:
        {
            _ISFn(is6_WritePSD)(hDest, pImageData, Width(), Height(), 8, RowStride(), Palette(), 0);
        }
        break;

    case eFmtTif8:
        {
            _ISFn(is6_WriteTIFF)(hDest, pImageData, Width(), Height(), 8, RowStride(), Palette(), m_uTiffComp, &dpi, 0);
        }
        break;

    case eFmtTga8 :
        {
            _ISFn(is6_WriteTGA)(hDest, pImageData, Width(), Height(), 8, RowStride(), Palette(), &dpi, 0);
        }
        break;

    case eFmtBmp8:
        {
            _ISFn(is6_WriteBMP)(hDest, pImageData, Width(), Height(), 8, RowStride(), PalColors(), Palette(), &dpi, (m_bBmp8RLE ? 1 : 0));
        }
        break;
    case eFmtGIF:
        {
            _ISFn(is6_WriteGIF)(hDest, pImageData, Width(), Height(), RowStride(), 0, -1, 8, Palette(), 0);
        }
        break;
    case eFmtWBmp:
        {
            _ISFn(is6_WriteWBMP)(hDest, pImageData, Width(), Height(), RowStride(), 0);
        }
        break;
    }

    return _ISFn(is6_GetLastError());
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Save1Bit(HISDEST hDest, eImgFmt eFmt)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    IS4DPIStruct dpi;
    SetDPI(eFmt, dpi);

    if (BitDepth() != 1)
    {
        // need to convert image to 1bpp first
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    switch (eFmt)
    {
    case eFmtBmp1:
        {
            _ISFn(is6_WriteBMP)(hDest, ImageData(), Width(), Height(), 1, RowStride(), 2, Palette(), &dpi, 0);
        }
        break;
    case eFmtPng1:
        {
            _ISFn(is6_WritePNG)(hDest, ImageData(), Width(), Height(), RowStride(), 1, 2, Palette(), 3, 0, &dpi, 0);
        }
        break;
    case eFmtTif1:
        {
            // is Palette[0] = black (0) ?
            UINT32 uFlags = 0;
            if (_ISFn(is6_ColorCompare)(RGB(Palette()[0].rgbRed, Palette()[0].rgbGreen, Palette()[0].rgbBlue), 0, 0))
            {
                // set the flag that tells ImgSource which color is which (1-bit TIFF writing doesn't use the palette)
                uFlags = (1 << 5);
            }

            _ISFn(is6_WriteTIFF)(hDest, ImageData(), Width(), Height(), 1, RowStride(), NULL, m_uTiffComp, &dpi, uFlags);
        }
        break;
    default:
        break;
    }

    res = _ISFn(is6_GetLastError());
    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32   CISImageEx::Make1Bit(UINT uBWTolerance, UINT uMode)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() == 1)
    {
        // done
        return IS_ERR_OK;
    }

    CISImageEx t;

    UINT res = Make1BitCopy(t, uBWTolerance, uMode);

    if (res == IS_ERR_OK)
    {
        Steal(t);
    }

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32   CISImageEx::Make1BitCopy(CISImageEx &out, UINT uBWTolerance, UINT uMode) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() == 1)
    {
        // copy
        return out.Copy(*this);
    }

    RGBQUAD rgbPal[256];
    memset(&rgbPal[0], 0, sizeof(RGBQUAD));
    memset(&rgbPal[1], 255, sizeof(RGBQUAD));

    // bytes per row
    UINT32 rowStride1Bit = OneBitRowStride(Width());

    UINT res = out.AllocBlank(Size(), 1);

    CISImageEx oneRow;
    oneRow.AllocBlank(Width(), 1, 8);

    UINT32 uBWToleranceLevel = uBWTolerance * uBWTolerance * 3;

    if (res == IS_ERR_OK)
    {
        if (BitDepth() == 8)
        {
            switch (uMode)
            {
            default:
                {
                    CISImageEx temp8Gray;
                    res = Make8BitGrayCopy(temp8Gray);
                    if (res == IS_ERR_OK)
                    {
                        _ISFn(is6_GrayscaleTo1Bit)(temp8Gray.ImageData(), temp8Gray.Width(), temp8Gray.Height(), temp8Gray.RowStride(), out.ImageData(), out.RowStride(), uMode, uBWTolerance, 0);
                    }
                }
                break;
            case 100:
                {
                    // map the current palette to B/W
                    BYTE lut[256];
                    memset(lut, 0, 256); // set all to zero (black)
                    for (UINT c = 0; c < PalColors(); c++)
                    {
                        if (_ISFn(is6_ColorCompare)(RGB(ConstPalette()[c].rgbRed, ConstPalette()[c].rgbGreen, ConstPalette()[c].rgbBlue), RGB(255,255,255), uBWToleranceLevel))
                        {
                            // if this color is closer to white than black, use white in the output
                            lut[c] = 1;
                        }
                    }

                    for (UINT row = 0; row < Height(); row++)
                    {
                        // convert one row into 8-bit B&W

                        for (UINT32 col=0; col < Width(); col++)
                        {
                            BYTE pix = ImageDataConst()[col + row * Width()];

                            // use out LUT to map input to output
                            oneRow.ImageData()[col] = lut[pix];
                        }

                        // squeeze that 8-bpp row into the output buffer
                        _ISFn(is6_Pack8BitBuffer)(oneRow.ImageData(), Width(), 1, Width(), out.ImageData() + rowStride1Bit * row, 1, TRUE, rowStride1Bit, 0);
                    }
                }
                break;
            }
        }
        else if (BitDepth() == 24 || BitDepth() == 32)
        {
           Make8BitGrayCopy(out);
           out.Make1Bit(uBWToleranceLevel, uMode);
        }

        memcpy(out.Palette(), rgbPal, 2 * sizeof(RGBQUAD));
        out.m_uPalColors = 2;
        out.m_uBitDepth = 1;

        out.CopyMetaData(*this);
    }

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Make8Bit(RGBQUAD *pPal, UINT32 uPalColors)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==8)
    {
        return IS_ERR_OK;
    }

    CISImageEx temp;
    UINT32 res = Make8BitCopy(temp, pPal, uPalColors);

    if (res == IS_ERR_OK)
    {
       Steal(temp);
    }

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Make8BitCopy(CISImageEx &out, RGBQUAD *pPal, UINT32 uPalColors) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==8)
    {
        return out.Copy(*this);
    }

    UINT res = out.AllocBlank(Size(), 8);

    if (res == IS_ERR_OK)
    {
       if (BitDepth()==32)
       {
          Make24BitCopy(out);
          out.Make8Bit(pPal, uPalColors);
       }
       else if (BitDepth()==24)
        {
            if (pPal==NULL)
            {
                _ISFn(is6_QuantizeRGBTo8Bit)((BYTE*)ImageDataConst(), Width(), Height(), RowStride(), out.ImageData(), out.RowStride(), 256, out.Palette(), 1, 20, 0);
                out.m_uPalColors = 256;
            }
            else
            {
                uPalColors = max(uPalColors, 2);

                _ISFn(is6_RGBTo8BitDithered)((BYTE*)ImageDataConst(), Width(), Height(), RowStride(), out.ImageData(), out.RowStride(), uPalColors, pPal, 1, 20, 0);
                out.m_uPalColors = uPalColors;
                memcpy(out.m_pal, pPal, uPalColors * sizeof(RGBQUAD));
            }

            res = _ISFn(is6_GetLastError)();
        }
        else if (BitDepth() == 1)
        {
            UINT32 rowStride1 = RowStride();

            for (UINT z=0; z < Height(); z++)
            {
                _ISFn(is6_UnpackBufferTo8Bit)((BYTE*)ImageDataConst() + (z * rowStride1), Width(), 1, rowStride1, 1, out.ImageData() + (z * Width()), out.RowStride(), 0);
            }

            res = _ISFn(is6_GetLastError)();

            out.m_uPalColors = 256;

            memcpy(out.Palette(), ConstPalette(), PalColors() * sizeof(RGBQUAD));
        }

        out.CopyMetaData(*this);
    }

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Make8BitBW(UINT32 uBWTolerance)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() == 1)
    {
        // done
        return IS_ERR_OK;
    }

    CISImageEx t;

    UINT res = Make8BitBWCopy(t, uBWTolerance);

    if (res == IS_ERR_OK)
    {
        Steal(t);
    }

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Make8BitBWCopy(CISImageEx &out, UINT32 uBWTolerance) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    RGBQUAD rgbPal[256];
    memset(&rgbPal[0], 0, sizeof(RGBQUAD));
    memset(&rgbPal[1], 255, sizeof(RGBQUAD));

    UINT32 n = Width() * Height();

    UINT32 uBWToleranceLevel = uBWTolerance * uBWTolerance * 3;

    // map the current palette to B/W
    BYTE lut[256];
    memset(lut, 0, 256); // set all to zero (black)
    for (UINT c = 0; c < PalColors(); c++)
    {
        if (_ISFn(is6_ColorCompare)(RGB(ConstPalette()[c].rgbRed, ConstPalette()[c].rgbGreen, ConstPalette()[c].rgbBlue), RGB(255,255,255), uBWToleranceLevel))
        {
            // if this color is closer to white than black, use white in the output
            lut[c] = 1;
        }
    }

    UINT res = out.AllocBlank(Width(), Height(), 8);

    if (res == IS_ERR_OK)
    {
        if (BitDepth()==1)
        {
            // expand this to 8-bit, then apply a threshhold to the data
            UINT32 rowStride1 = OneBitRowStride(Width());

            for (UINT z=0; z < Height(); z++)
            {
                BYTE *pCurOutRow = out.ImageData() + z * Width();

                // expand this line to 8-bit
                _ISFn(is6_UnpackBufferTo8Bit)((BYTE*)ImageDataConst() + (z * rowStride1), Width(), 1, rowStride1, 1, pCurOutRow, out.RowStride(), 0);

                // map those pixels to the new palette
                for (UINT p=0; p<Width(); p++)
                {
                    BYTE pix = pCurOutRow[p];
                    pCurOutRow[p] = lut[pix];
                }
            }
        } 
        else if (BitDepth()==8)
        {
            // apply a threshold
            for (UINT y=0; y < Height(); y++)
            {
               for (UINT x = 0; x < Width(); x++)
               {
                  BYTE pix = ImageDataConst()[y * RowStride() + x];
                  out.ImageData()[y * out.RowStride() + x] = lut[pix];
               }
            }
        } 
        else if (BitDepth()==24)
        {
            // apply threshold, make a 2-color palette

            for (UINT y=0; y < Height(); y++)
            {
               for (UINT x = 0; x < Width(); x++)
               {
                  const BYTE * pPix = ImageDataConst() + (y * RowStride() * 3 + x * 3);
                  if (_ISFn(is6_ColorCompare)(RGB(*pPix, *(pPix + 1), *(pPix + 2)), 0, uBWToleranceLevel))
                  {
                     out.ImageData()[y * out.RowStride() + x] = 1;
                  }
                  else
                  {
                     out.ImageData()[y * out.RowStride() + x] = 0;
                  }
               }
            }
        }
        else if (BitDepth()==32)
        {
            for (UINT y=0; y < Height(); y++)
            {
               for (UINT x = 0; x < Width(); x++)
               {
                  const BYTE * pPix = ImageDataConst() + (y * RowStride() * 4 + x * 4);
                  if (_ISFn(is6_ColorCompare)(RGB(*pPix, *(pPix + 1), *(pPix + 2)), 0, uBWToleranceLevel))
                  {
                     out.ImageData()[y * out.RowStride() + x] = 1;
                  }
                  else
                  {
                     out.ImageData()[y * out.RowStride() + x] = 0;
                  }
               }
            }
        }

        out.CopyMetaData(*this);

        memcpy(out.Palette(), rgbPal, 2 * sizeof(RGBQUAD));
        out.m_uPalColors = 2;

    }   

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::Make8BitGray()
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() == 1)
    {
        // done
        return IS_ERR_OK;
    }

    CISImageEx t;

    UINT res = Make8BitGrayCopy(t);

    if (res == IS_ERR_OK)
    {
        Steal(t);
    }

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Make8BitGrayCopy(CISImageEx &out) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 n = Width() * Height();

    UINT res = out.AllocBlank(Size(), 8);

    if (res == IS_ERR_OK)
    {
        // map the current palette to B/W
        BYTE lut[256];
        memset(lut, 0, 256); // set all to zero (black)

        if (BitDepth() < 24)
        {
            for (UINT c = 0; c < PalColors(); c++)
            {
                int iLum = (int)(.299 * (double)ConstPalette()[c].rgbRed + .587 * (double)ConstPalette()[c].rgbGreen + .114 * (double)ConstPalette()[c].rgbBlue);
                {
                    lut[c] = (BYTE)iLum;
                }
            }
        }

        if (BitDepth()==1)
        {
            UINT32 rowStride1 = (Width() + 7) / 8;
            const BYTE *pCurIn = ImageDataConst();
            BYTE *pCurOut = out.ImageData();

            for (UINT z=0; z < Height(); z++)
            {
                BYTE *pCurOutRow = pCurOut + z * Width();

                // expand this line to 8-bit
                _ISFn(is6_UnpackBufferTo8Bit)((BYTE*)pCurIn + (z * rowStride1), Width(), 1, 1, rowStride1, pCurOutRow, out.RowStride(), 0);

                // map those pixels to the new palette
                for (UINT p=0; p<Width(); p++)
                {
                    BYTE pix = pCurOutRow[p];
                    pCurOutRow[p] = lut[pix];
                }
            }
        }

        else if (BitDepth()==8)
        {
           // map input pix from palette color to luminance via the LUT
           for (UINT y=0; y < Height(); y++)
           {
              for (UINT x = 0; x < Width(); x++)
              {
                 BYTE pix = ImageDataConst()[y * RowStride() + x];
                 out.ImageData()[y * out.RowStride() + x] = lut[pix];
              }
           }
        }
        else if (BitDepth()==24 || BitDepth()==32)
        {
            // ImgSource can do this one for us
            _ISFn(is6_ImageToGrayScaleSingle)((BYTE*)ImageDataConst(), Width(), Height(), BitDepth() / 8, RowStride(), out.ImageData(), out.RowStride(), 0);
        }


        // 256 color grayscale palette
        out.SetGrayPalette();

        out.CopyMetaData(*this);
    }

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Make24Bit()
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() == 24)
    {
        // done
        return IS_ERR_OK;
    }

    CISImageEx t;

    UINT res = Make24BitCopy(t);

    if (res == IS_ERR_OK)
    {
        Steal(t);
    }

    return res;

}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::Make24BitCopy(CISImageEx &out) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==24)
    {
        return out.Copy(*this);
    }

    UINT32 res = out.AllocBlank(Width(), Height(), 24);
    if (res == IS_ERR_OK)
    {
        if (BitDepth() == 1)
        {
            CISImageEx t;
            res = Make8BitCopy(t);

            if (res == IS_ERR_OK)
            {
                return t.Make24BitCopy(out);
            }
        }
        else if (BitDepth() == 8)
        {
            _ISFn(is6_8BitToRGB)((BYTE*)ImageDataConst(), Width(), Height(), RowStride(), out.ImageData(), out.RowStride(), PalColors(), (RGBQUAD*)ConstPalette(), 0);
            res = _ISFn(is6_GetLastError)();
        }
        else if (BitDepth() == 32)
        {
           CISImageEx a; // we'll just discard this
           res = SplitRGBAToRGBAAndA(out, a);
        }

        out.CopyMetaData(*this);
    }

    return res;
}


//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Make32Bit()
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() == 32)
    {
        // done
        return IS_ERR_OK;
    }

    CISImageEx t;

    UINT res = Make32BitCopy(t);

    if (res == IS_ERR_OK)
    {
        Steal(t);
    }

    return res;

}

//////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::Make32BitCopy(CISImageEx &out) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==32)
    {
        return out.Copy(*this);
    }

    UINT32 res = out.AllocBlank(Size(), 32);
    if (res == IS_ERR_OK)
    {
        if (BitDepth() == 1)
        {
            CISImageEx t;
            res = Make8BitCopy(t);

            if (res == IS_ERR_OK)
            {
                return t.Make32BitCopy(out);
            }
        }
        else if (BitDepth() == 8)
        {
            _ISFn(is6_8BitToRGBA)((BYTE*)ImageDataConst(), Width(), Height(), RowStride(), out.ImageData(), out.RowStride(), PalColors(), (RGBQUAD*)ConstPalette(), 2);
            res = _ISFn(is6_GetLastError)();
        }
        else if (BitDepth() == 24)
        {
            _ISFn(is6_RGBToRGBA)((BYTE*)ImageDataConst(), Width(), Height(), RowStride(), out.ImageData(), out.RowStride(), 255);
            res = _ISFn(is6_GetLastError)();
        }

        out.CopyMetaData(*this);
    }

    return res;
}

//////////////////////////////////////////////////////////////////////

UINT32  CISImageEx::MakeCopy(CISImageEx &out, int targetDepth) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    switch (targetDepth)
    {
    case 1:
       return Make1BitCopy(out, 128);
    case 8:
       if (Is8BitGray())
          return Make8BitGrayCopy(out);
       else
          return Make8BitCopy(out);
    case 24:
       return Make24BitCopy(out);
    case 32:
       return Make32BitCopy(out);
    }

    return IS_ERR_BADPARAM;
}

//////////////////////////////////////////////////////////////////////

UINT CISImageEx::Matte(CSize matteSize, COLORREF clr, bool bBorder, COLORREF borderClr)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CSize bfSize = GetBestFitDims(matteSize);

    if (bBorder && ((bfSize.cx < 2) || (bfSize.cy < 2)))
    {
        TRACE(_T("image too small for a border\n"));
        bBorder = false;
    }

    if (bBorder)
    {
        bfSize.cx -= 2;
        bfSize.cy -= 2;
    }

    res = Resize(bfSize);

    if (res == IS_ERR_OK)
    {
        CISImageEx temp;
        res = temp.AllocBlank(matteSize, BitDepth());

        if (res == IS_ERR_OK)
        {

            if ((BitDepth()==8) || (BitDepth()==24))
            {
                if (BitDepth()==8)
                {
                    //TrimPaletteToUsed();

                    if (m_uPalColors < 256)
                    {
                        AddPalColor(clr);
                    }
                    if (m_uPalColors < 256)
                    {
                        AddPalColor(borderClr);
                    }

                    HISCOLORMATCH hcm = _ISFn(is6_CreateColorMatcher)(Palette(), PalColors());
                    clr = _ISFn(is6_FindClosestColor)(hcm, GetRValue(clr), GetGValue(clr), GetBValue(clr));
                    borderClr = _ISFn(is6_FindClosestColor)(hcm, GetRValue(borderClr), GetGValue(borderClr), GetBValue(borderClr));
                    _ISFn(is6_DestroyColorMatcher)(hcm);
                }

                temp.CopyAttribs(*this);

                CRect fillRect(0, 0, temp.Width() - 1, temp.Height() - 1);
                UINT32 bpp = BitDepth() / 8;

                _ISFn(is6_FillSolidRect)(temp.ImageData(), 
                    temp.Width(), temp.Height(), 
                    bpp, temp.RowStride(), fillRect, clr, 0);

                int iXPos = (matteSize.cx - bfSize.cx) / 2;
                int iYPos = (matteSize.cy - bfSize.cy) / 2;

                _ISFn(is6_OverlayImage)(temp.ImageData(), temp.Width(), temp.Height(), bpp, temp.RowStride(),
                    ImageData(), Width(), Height(), RowStride(),
                    iXPos , iYPos, 
                    1, 0, 0, 0);

                res = _ISFn(is6_GetLastError)();

                if (bBorder)
                {
                    CPoint p0(iXPos - 1, iYPos - 1);
                    CPoint p1(iXPos + Width(), iYPos - 1);
                    CPoint p2(iXPos + Width(), iYPos + Height());
                    CPoint p3(iXPos - 1, iYPos + Height());

                    _ISFn(is6_DrawLineOnImage)(temp.ImageData(), temp.Width(), temp.Height(), bpp, temp.Width() * bpp, borderClr, &p0, &p1, 0);
                    _ISFn(is6_DrawLineOnImage)(temp.ImageData(), temp.Width(), temp.Height(), bpp, temp.Width() * bpp, borderClr, &p1, &p2, 0);
                    _ISFn(is6_DrawLineOnImage)(temp.ImageData(), temp.Width(), temp.Height(), bpp, temp.Width() * bpp, borderClr, &p2, &p3, 0);
                    _ISFn(is6_DrawLineOnImage)(temp.ImageData(), temp.Width(), temp.Height(), bpp, temp.Width() * bpp, borderClr, &p3, &p0, 0);

                    res = _ISFn(is6_GetLastError)();
                }

            }
            else if (BitDepth()==1)
            {
                HISCOLORMATCH hcm = _ISFn(is6_CreateColorMatcher)(Palette(), PalColors());
                clr = _ISFn(is6_FindClosestColor)(hcm, GetRValue(clr), GetGValue(clr), GetBValue(clr));
                borderClr = _ISFn(is6_FindClosestColor)(hcm, GetRValue(borderClr), GetGValue(borderClr), GetBValue(borderClr));
                _ISFn(is6_DestroyColorMatcher)(hcm);

                temp.CopyAttribs(*this);

                _ISFn(is6_FillSolidRect1Bit)(temp.ImageData(), 
                    temp.Width(), temp.Height(), temp.RowStride(),
                    CRect(0, 0, temp.Width() - 1, temp.Height() - 1), clr);

                int iXPos = (matteSize.cx - bfSize.cx) / 2;
                int iYPos = (matteSize.cy - bfSize.cy) / 2;

                _ISFn(is6_Overlay1Bit)(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(),
                    ImageData(), Width(), Height(), RowStride(),
                    iXPos , iYPos, 0, 0);

                res = _ISFn(is6_GetLastError)();

                if (bBorder)
                {
                    CPoint p0(iXPos - 1, iYPos - 1);
                    CPoint p1(iXPos + Width(), iYPos - 1);
                    CPoint p2(iXPos + Width(), iYPos + Height());
                    CPoint p3(iXPos - 1, iYPos + Height());

                    _ISFn(is6_DrawLineOn1Bit)(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(), borderClr, &p0, &p1, 0);
                    _ISFn(is6_DrawLineOn1Bit)(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(), borderClr, &p1, &p2, 0);
                    _ISFn(is6_DrawLineOn1Bit)(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(), borderClr, &p2, &p3, 0);
                    _ISFn(is6_DrawLineOn1Bit)(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(), borderClr, &p3, &p0, 0);

                    res = _ISFn(is6_GetLastError)();
                }

            }

            Steal(temp);

        }
    }

    return res;
}


//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Draw(const CDC *pDC, const CPoint & ptTopLeft, const HPALETTE hPal)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    BYTE *pRGB = ImageData();
    if (pRGB==NULL)
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==32)
    {
        BOOL ok = _ISFn(is6_DrawRGBA)(pDC->m_hDC, pRGB, Width(), Height(), RowStride(), ptTopLeft.x, ptTopLeft.y, hPal, 0);
        ASSERT(ok);
    }
    else if (BitDepth()==24)
    {
        BOOL ok = _ISFn(is6_DrawRGB)(pDC->m_hDC, pRGB, Width(), Height(), RowStride(), ptTopLeft.x, ptTopLeft.y, hPal);
        ASSERT(ok);
    }
    else if (BitDepth()==8)
    {
        BOOL ok = _ISFn(is6_Draw8Bit)(pDC->m_hDC, pRGB, Width(), Height(), RowStride(), Palette(), ptTopLeft.x, ptTopLeft.y, hPal);
        ASSERT(ok);
    }
    else if (BitDepth()==1)
    {
        BOOL ok = _ISFn(is6_Draw1Bit)(pDC->m_hDC, ImageData(), Width(), Height(), RowStride(), Palette(),ptTopLeft.x, ptTopLeft.y, hPal);
        ASSERT(ok);
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::StretchDraw(const CDC *pDC, const CRect & outRect)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    BYTE *pRGB = ImageData();
    if (pRGB==NULL)
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==32)
    {
       return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==24)
    {
        BOOL ok = _ISFn(is6_StretchDrawRGB)(pDC->m_hDC, pRGB, Width(), Height(), RowStride(), outRect.left, outRect.top, outRect.Width(), outRect.Height());
        ASSERT(ok);
    }
    else if (BitDepth()==8)
    {
        BOOL ok = _ISFn(is6_StretchDraw8Bit)(pDC->m_hDC, pRGB, Width(), Height(), RowStride(), Palette(), outRect.left, outRect.top, outRect.Width(), outRect.Height());
        ASSERT(ok);
    }
    else if (BitDepth()==1)
    {
        BOOL ok = _ISFn(is6_StretchDraw1Bit)(pDC->m_hDC, ImageData(), Width(), Height(), RowStride(), Palette(), outRect.left, outRect.top, outRect.Width(), outRect.Height());
        ASSERT(ok);
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////////////////////

UINT32      CISImageEx::Resize(const CSize &czPix, int iMethod, bool bResize1BitToGrayscale)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx temp;

    UINT32 uRes = temp.ResizeCopy(*this, czPix.cx, czPix.cy, iMethod, bResize1BitToGrayscale);
    if (uRes!=IS_ERR_OK)
    {
        return uRes;
    }

    return Steal(temp);
}

/////////////////////////////////////////////////////////////////////////////

UINT32	CISImageEx::ResizeCopy(const CISImageEx &src, const CSize &cz, UINT32 uType, bool bResize1BitToGrayscale) 
{
    return ResizeCopy(src, cz.cx, cz.cy, uType, bResize1BitToGrayscale);
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::ResizeCopy(const CISImageEx &src, UINT32 w, UINT32 h, UINT32 uType, bool bResize1BitToGrayscale) 
{
    if (!src.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (CSize(w,h) == src.Size())
    {
        return Copy(src);
    }

    // allocate room in this for a resized copy of src
    UINT32 res = IS_ERR_OK;

    if (bResize1BitToGrayscale && src.BitDepth() == 1)
    {
        res = AllocBlank(w, h, 8);
    }
    else
    {
        res = AllocBlank(w, h, src.BitDepth());
    }

    if (res!=IS_ERR_OK)
    {
        Clear();
        return res;
    }

    // switch on bit depth
    BOOL ok = FALSE;
    switch (src.BitDepth())
    {
    case 32:
    case 24:

        if ((src.Width() < w) || (src.Height() < h))
        {
            switch (uType)
            {
            case is6_RESIZE_MODE_AREAAVG_FAST:  // these two can't enlarge an image
            case is6_RESIZE_MODE_AREAAVG:
                uType = is6_RESIZE_MODE_BICUBIC;   // so, we'll pick a mode that can
                break;
            }
        }

        ok = _ISFn(is6_ResizeImage)((BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), ImageData(), w, h, RowStride(), BitDepth() / 8, uType, 0);

        break;
    case 8:
        // make sure we use a sizing method ImgSource supports for this bitdepth
        switch (uType)
        {
        case is6_RESIZE_MODE_AREAAVG_FAST:
        case is6_RESIZE_MODE_BILINEAR:
        case is6_RESIZE_MODE_NEAREST_NEIGHBOR:
            break;
        case is6_RESIZE_MODE_AREAAVG:
            uType = is6_RESIZE_MODE_AREAAVG_FAST;
            break;
        default:
            uType = is6_RESIZE_MODE_BILINEAR;
            break;
        }

        if ((src.Width() < w) || (src.Height() < h))
        {
            switch (uType)
            {
            case is6_RESIZE_MODE_AREAAVG_FAST:  // these two can't enlarge an image
            case is6_RESIZE_MODE_AREAAVG:
                uType = is6_RESIZE_MODE_BILINEAR;   // so, we'll pick a mode that can
                break;
            }
        }
        ok = _ISFn(is6_ResizeImage8)((BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), ImageData(), w, h, RowStride(), (RGBQUAD*)src.ConstPalette(), src.PalColors(), uType, 0);

        break;
    case 1:
        {                                          
            // make sure we use a sizing method ImgSource supports for this bitdepth
            if (bResize1BitToGrayscale)
            {
                switch (uType)
                {
                case is6_RESIZE_MODE_BOX: 
                case is6_RESIZE_MODE_TRIANGLE: 
                case is6_RESIZE_MODE_HAMMING: 
                case is6_RESIZE_MODE_GAUSSIAN: 
                case is6_RESIZE_MODE_BELL: 
                case is6_RESIZE_MODE_BSPLINE: 
                case is6_RESIZE_MODE_CUBIC1: 
                case is6_RESIZE_MODE_CUBIC2: 
                case is6_RESIZE_MODE_LANCZOS3: 
                case is6_RESIZE_MODE_MITCHELL: 
                case is6_RESIZE_MODE_SINC: 
                case is6_RESIZE_MODE_HERMITE:
                case is6_RESIZE_MODE_HANNING: 
                case is6_RESIZE_MODE_CATROM:
                case is6_RESIZE_MODE_AREAAVG_FAST:
                case is6_RESIZE_MODE_BILINEAR:
                    break;
                case is6_RESIZE_MODE_AREAAVG:
                    uType = is6_RESIZE_MODE_AREAAVG_FAST;
                    break;
                default:
                    uType = is6_RESIZE_MODE_BILINEAR;
                    break;
                }

                // are we black=0 or black = 1?
                bool blackis0 = src.ConstPalette()[0].rgbRed==0;

                ok = _ISFn(is6_ResizeImage1To8Gray)((BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), ImageData(), w, h, RowStride(), uType, blackis0? 1 : 0);
            }
            else
            {
                switch (uType)
                {
                case is6_RESIZE_MODE_BILINEAR:
                case is6_RESIZE_MODE_AREAAVG_FAST:
                    break;
                case is6_RESIZE_MODE_AREAAVG:
                    uType = is6_RESIZE_MODE_AREAAVG_FAST;
                    break;
                default:
                    uType = is6_RESIZE_MODE_BILINEAR;
                    break;
                }

                UINT32 uCutOff = 192;
                if ((src.Width() < w) || (src.Height() < h))
                {
                    switch (uType)
                    {
                    case is6_RESIZE_MODE_AREAAVG_FAST:  // these two can't enlarge an image
                    case is6_RESIZE_MODE_AREAAVG:
                        uType = is6_RESIZE_MODE_BILINEAR;   // so, we'll pick a mode that can
                        break;
                    }
                }
                else
                {
                    // try to determine background vs foreground based on histograms
                    UINT32 histo[2];
                    _ISFn(is6_GetOneBitHistogram)((BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), histo, 0);

                    if (histo[0] > histo[1])
                    {
                        uCutOff = 64;
                    }
                }

                ok = _ISFn(is6_ResizeImage1)((BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), ImageData(), w, h, RowStride(), uCutOff, uType, 0);
            }
        }  
        break;
    }

    if (!ok)
    {
       return _ISFn(is6_GetLastError)();
    }


    if (bResize1BitToGrayscale && src.BitDepth()==1)
    {
       SetGrayPalette();
    }
    else
    {
       // copy the palette
       memcpy(Palette(), src.ConstPalette(), src.PalColors() * sizeof(RGBQUAD));
       m_uPalColors = src.PalColors();
    }

    // copy the rest
    m_eSrcFmt = src.SrcFormat(); 

    m_fDPIX = src.DPIX();
    m_fDPIY = src.DPIY();
    m_eDPIUnits = src.DPIU();

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::QuickRotate(int iDeg)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    int iRotFlag = 0;
    switch (iDeg)
    {
    case 0:
        return IS_ERR_OK;
        break;
    case 180:
        iRotFlag = 1;
        break;
    case 90:
        iRotFlag = 0;
        break;
    case 270:
        iRotFlag = 2;
        break;
    default:
        ASSERT(0);
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    switch (BitDepth())
    {
    case 1:
        {
            // can't do this in-place
            CISImageEx t;
            res = QuickRotateCopy(iDeg, t);
            if (res == IS_ERR_OK)
            {
                Steal(t);
            }
        }
        break;
    case 8:
    case 24:
    case 32:
       _ISFn(is6_QuickRotateImageInPlace)(ImageData(), Width(), Height(), iRotFlag, BitDepth() / 8, 0);
       if (iRotFlag!=1)
          std::swap(m_uWidth, m_uHeight);
       break;

    }

    return res;

}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::QuickRotateCopy(int iDeg, CISImageEx &out)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;
    int iRotFlag = 0;
    switch (iDeg)
    {
    case 180:
        iRotFlag = 1;
        res = out.AllocBlank(Width(), Height(), BitDepth());
        break;
    case 90:
        iRotFlag = 0;
        res = out.AllocBlank(Height(), Width(), BitDepth());
        break;
    case 270:
        iRotFlag = 2;
        res = out.AllocBlank(Height(), Width(), BitDepth());
        break;
    default:
        ASSERT(0);
        return IS_ERR_BADPARAM;
    }

    if (res==IS_ERR_OK)
    {
        switch (BitDepth())
        {
        case 1:
            _ISFn(is6_QuickRotate1Bit)(ImageData(), Width(), Height(), RowStride(), out.ImageData(), out.RowStride(), iRotFlag);
            break;
        case 8:
        case 24:
        case 32:
            _ISFn(is6_QuickRotateImage)(ImageData(), Width(), Height(), RowStride(), out.ImageData(), out.RowStride(), iRotFlag, BitDepth() / 8, 0);
            break;
        }

        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::Flip(int eFlip)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (eFlip==0)
    {
        return IS_ERR_OK;
    }

    switch (BitDepth())
    {
    case 1:
        switch (eFlip)
        {
        case 1:
            _ISFn(is6_FlipHorizontal1Bit)(ImageData(), Width(), Height(), RowStride());
            break;
        case 2:
            _ISFn(is6_VerticalFlipImage)(ImageData(), RowStride(), Height());
            break;
        }

        break;
    case 8:
    case 24:
    case 32:
        switch (eFlip)
        {
        case 1:
            _ISFn(is6_HorizontalFlipImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 0);
            break;
        case 2:
            _ISFn(is6_VerticalFlipImage)(ImageData(), RowStride(), Height());
            break;
        }
        break;
    }            

    return _ISFn(is6_GetLastError)();
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::Sharpen(UINT32 uLevel)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth()!=24 && BitDepth()!=32 && BitDepth()!=8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_SharpenImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            t.ImageData(), RowStride(), 
            3,
            (double)uLevel / 100.0,  
            1 | 2 | 4);

        Steal(t);
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::Sharpen(UINT32 uLevel, const CRect &r)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth()!=24 && BitDepth()!=32 && BitDepth()!=8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    // crop to the image rect
    CRect imgRect(0,0, Width(), Height());
    if (!imgRect.IntersectRect(imgRect, r))
    {
        // no pixels are selected
        return IS_ERR_BADPARAM;
    }

    // need to output to a temp image
    CISImageEx t;
    res = t.AllocBlank(imgRect.Size(), BitDepth());

    if (res == IS_ERR_OK)
    {
        // find the address of the pixel at the top of the crop rect
        BYTE *pStart = ImageData() +                  // base address
            imgRect.top * RowStride() +               // row offset
            imgRect.left * (BitDepth()/8);            // column offset

        _ISFn(is6_SharpenImage)(pStart,                            // start here
            imgRect.Width(),                    // process this many columns
            imgRect.Height(),                   // process this many rows
            BitDepth()/8,                       // bytes in each pixel
            RowStride(),                        // each row has this many bytes
            t.ImageData(), t.RowStride(),       // output image
            3,												// filter size. 3 is the default
            (double)uLevel / 100.0, 
            1 | 2 | 4);                         // process channels 0, 1 and 2 (R,G and B)

        // now overlay the temp image onto "this".
        _ISFn(is6_OverlayImage)(ImageData(), Width(), Height(), BitDepth()/8, RowStride(), t.ImageData(), t.Width(), t.Height(), t.RowStride(), imgRect.left, imgRect.top, 1, 0, 0, 0);
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32   CISImageEx::GaussianBlur(double sigma, bool modifyAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth()!=24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_GaussianBlurImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), sigma, sigma, (1 | 2 | 4) | (modifyAlpha?7:0));
        Steal(t);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::Blur(UINT32 uLevel)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth()!=24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_BlurImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), RowStride(), 3, (double)uLevel / 100.0, 1 | 2 | 4);
        Steal(t);
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::Blur(UINT32 uLevel, const CRect &r)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth()!=24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }
    // crop to the image rect
    CRect imgRect(0,0, Width(), Height());
    if (!imgRect.IntersectRect(imgRect, r))
    {
        // no pixels are selected
        return IS_ERR_BADPARAM;
    }

    // need to output to a temp image
    CISImageEx t;
    res = t.AllocBlank(imgRect.Size(), BitDepth());

    if (res == IS_ERR_OK)
    {
        // find the address of the pixel at the top of the crop rect
        BYTE *pStart = ImageData() +                 // base address
            imgRect.top * RowStride() +   // row offset
            imgRect.left * (BitDepth() / 8);             // column offset

        // this is a simple box blur. if you need a high-quality blur, see is6_GaussianBlurImage

        _ISFn(is6_BlurImage)(pStart,                            // start here
            imgRect.Width(),                    // process this many columns
            imgRect.Height(),                   // process this many rows
            BitDepth() / 8,                     // bytes in a pixel
            RowStride(),                        // each row has this many bytes
            t.ImageData(), t.RowStride(),       // output image
            3,												// filter size
            (double)uLevel / 100.0, 
            1 | 2 | 4);                         // process channels 0, 1 and 2 (R,G and B)

        // now overlay the temp image onto "this".
        _ISFn(is6_OverlayImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.Width(), t.Height(), t.RowStride(), imgRect.left, imgRect.top, 1, 0, 0, 0);
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::DrawTextOnImage(const TCHAR *pText, const char *pFontName, UINT32 uFontPixels, int iXPos, int iYPos, COLORREF clrText)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if ((pText==NULL) || (pFontName==NULL) || (uFontPixels==0))
    {
        return IS_ERR_OK;
    }

    // do we need to enable Unicode in ImgSource?
#ifdef _UNICODE
    UINT32 uUnicodeFlag = (1 << 2);
#else
    UINT32 uUnicodeFlag = 0;
#endif

    UINT32 uSmoothingFlag = (1 << 6);

    UINT res = IS_ERR_OK;

    CRect tr(iXPos, iYPos, Width() - 1, Height() -1);
    switch (BitDepth())
    {
    case 1:
        {
            HISCOLORMATCH hcm = _ISFn(is6_CreateColorMatcher)(Palette(), PalColors());
            int idx = _ISFn(is6_FindClosestColor)(hcm, GetRValue(clrText), GetGValue(clrText), GetBValue(clrText));
            _ISFn(is6_DestroyColorMatcher)(hcm);

            _ISFn(is6_DrawTextOn1Bit)(ImageData(), Width(), Height(), RowStride(), (const char *)pText, pFontName, uFontPixels, tr, 0, 0 | uUnicodeFlag, idx);
        }
        break;
    case 8:
        {
            HISCOLORMATCH hcm = _ISFn(is6_CreateColorMatcher)(Palette(), PalColors());
            int idx = _ISFn(is6_FindClosestColor)(hcm, GetRValue(clrText), GetGValue(clrText), GetBValue(clrText));
            _ISFn(is6_DestroyColorMatcher)(hcm);

            _ISFn(is6_DrawTextOn8Bit)(ImageData(), Width(), Height(), RowStride(), (const char *)pText, pFontName, uFontPixels, tr, 0, 0 | uUnicodeFlag, idx, Palette(), PalColors());
        }
        break;
    case 24:
        _ISFn(is6_DrawTextOnRGB)(ImageData(), Width(), Height(), RowStride(), (const char *)pText, pFontName, uFontPixels, tr, 0, 0 | uUnicodeFlag | uSmoothingFlag, clrText);
        break;
    case 32:
        _ISFn(is6_DrawTextOnRGBA)(ImageData(), Width(), Height(), RowStride(), (const char *)pText, pFontName, uFontPixels, tr, 0, 0 | uUnicodeFlag | uSmoothingFlag, clrText);
        break;
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::OverlayImage(const CISImageEx &src, int ixpos, int iypos, double fOpacity)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    CISImageEx t;

    switch (BitDepth())
    {
    case 1:

        if (src.BitDepth()!=1)
        {
            src.Make1BitCopy(t, 128);
            _ISFn(is6_Overlay1Bit)(ImageData(), Width(), Height(), RowStride(), (BYTE*)t.ImageDataConst(), t.Width(), t.Height(), t.RowStride(), ixpos, iypos, 0, 0);
        }
        else
        {
            _ISFn(is6_Overlay1Bit)(ImageData(), Width(), Height(), RowStride(), (BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), ixpos, iypos, 0, 0);
        }

        break;
    case 8:

        if (src.BitDepth()!=8)
        {
            src.Make8BitCopy(t);
            _ISFn(is6_Overlay8Bit)(ImageData(), Width(), Height(), RowStride(), PalColors(), Palette(), (BYTE*)t.ImageDataConst(), t.Width(), t.Height(), t.RowStride(), t.Palette(), ixpos, iypos, fOpacity, 0, 0);
        }
        else
        {
            _ISFn(is6_Overlay8Bit)(ImageData(), Width(), Height(), RowStride(), PalColors(), Palette(), (BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), (RGBQUAD*)src.ConstPalette(), ixpos, iypos, fOpacity, 0, 0);
        }

        break;
    case 24:

        if (src.BitDepth()!=24)
        {
            src.Make24BitCopy(t);
            _ISFn(is6_OverlayImage)(ImageData(), Width(), Height(), 3, RowStride(), (BYTE*)t.ImageDataConst(), t.Width(), t.Height(), t.RowStride(), ixpos, iypos, fOpacity, 0, 0, 0);
        }
        else
        {
            _ISFn(is6_OverlayImage)(ImageData(), Width(), Height(), 3, RowStride(), (BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), ixpos, iypos, fOpacity, 0, 0, 0);
        }
        break;
    case 32:
        // overlay onto RGBA, do not modify alpha
        if (src.BitDepth()!=24)
        {
            src.Make24BitCopy(t);
            _ISFn(is6_OverlayRGBOnRGBA)(ImageData(), Width(), Height(), RowStride(), (BYTE*)t.ImageDataConst(), t.Width(), t.Height(), t.RowStride(), ixpos, iypos, fOpacity, 0);
        }
        else
        {
            _ISFn(is6_OverlayRGBOnRGBA)(ImageData(), Width(), Height(), RowStride(), (BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), ixpos, iypos, fOpacity, 0);
        }
        break;
    }

    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::Merge(const CISImageEx &src, int ixpos, int iypos, overlayMode mode, double fOpacity, double param)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth() != src.BitDepth())
    {
       // make a copy of src at out bit depth
       CISImageEx tmp;
       res = src.MakeCopy(tmp, BitDepth());

       if (res == IS_ERR_OK)
       {
          _ISFn(is6_OverlayImage)(ImageData(), Width(), Height(), 3, RowStride(), (BYTE*)tmp.ImageDataConst(), tmp.Width(), tmp.Height(), tmp.RowStride(), ixpos, iypos, fOpacity, param, (int)mode, 0);
          res = _ISFn(is6_GetLastError)();
       }
    }
    else
    {
       _ISFn(is6_OverlayImage)(ImageData(), Width(), Height(), 3, RowStride(), (BYTE*)src.ImageDataConst(), src.Width(), src.Height(), src.RowStride(), ixpos, iypos, fOpacity, param, (int)mode, 0);
       res = _ISFn(is6_GetLastError)();
    }

    
    return res;
}

/////////////////////////////////////////////////////////////////////////////

UINT32     CISImageEx::MakeImagePalette(CPalette &pal)
{ 
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    LPLOGPALETTE     lpPal;
    BYTE *           pLogPal;
    HPALETTE         hPal = NULL;
    int              i;

    try 
    {
        pLogPal = new BYTE [sizeof (LOGPALETTE) + (sizeof (PALETTEENTRY) * (256))];
    } 
    catch (CMemoryException *e) 
    {
        e->ReportError();
        e->Delete();
        pLogPal=NULL;
        return IS_ERR_MEM;;
    }

    lpPal = (LPLOGPALETTE) pLogPal;
    lpPal->palVersion = 0x300;
    lpPal->palNumEntries = 256;

    RGBQUAD imgPal[256];
    GetPalette(imgPal);

    for (i=0;i<256;i++) 
    {
        lpPal->palPalEntry[i].peRed   = imgPal[i].rgbRed;
        lpPal->palPalEntry[i].peGreen = imgPal[i].rgbGreen;
        lpPal->palPalEntry[i].peBlue  = imgPal[i].rgbBlue;
        lpPal->palPalEntry[i].peFlags = 0;
    }

    if (pal.GetSafeHandle())
    {
        pal.DeleteObject();
    }

    pal.CreatePalette (lpPal);

    delete [] pLogPal;

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////

UINT32 CISImageEx::GetPalette(RGBQUAD *pPal)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 cf=0;
    switch (BitDepth())
    {
    case 32:
       return IS_ERR_DEPTHMISMATCH;
    case 24:
        _ISFn(is6_Get8BitPaletteFromRGB)(ImageData(), Width(),  Height(), Width() * 3, 256, pPal, &cf, 0);
        break;
    case 1:
    case 8:
        memcpy(pPal, Palette(), 256 * sizeof(RGBQUAD));
        break;
    }

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////

CString CISImageEx::ErrorString(UINT32 uRes)
{

    CString csErr;
    csErr.Format(_T("Error #%u!"), uRes);

    switch (uRes)
    {
    case IS_ERR_TRIALVERSION:     	csErr = "The ImgSource library was not initialized with a registered key.\nAll images read or written will have a big red \"X\" placed on them."; break;
    case IS_ERR_OK:			   		csErr = "no err"; break;
    case IS_ERR_MEM:			      	csErr = "out of memory"; break;
    case IS_ERR_FILEOPEN:				csErr = "error on file open"; break;
    case IS_ERR_FILEREAD:				csErr = "error on file read"; break;
    case IS_ERR_FILEWRITE:				csErr = "error on file write"; break;
    case IS_ERR_BADPARAM:				csErr = "bad user param"; break;
    case IS_ERR_INVALIDBMP:		   	csErr = "bad BMP file"; break;
    case IS_ERR_BMPRLE:					csErr = "some RLE variations are not supported"; break;
    case IS_ERR_INVALIDJPG:		   	csErr = "bad JPG file"; break;
    case IS_ERR_DC:						csErr = "error with device context"; break;
    case IS_ERR_DIB:						csErr = "problem with a GetDIBits call"; break;
    case IS_ERR_INVALIDGIF:				csErr = "bad GIF file"; break;
    case IS_ERR_NORESOURCE:		   	csErr = "resource not found"; break;
    case IS_ERR_CALLBACKCANCEL:		csErr = "callback returned FALSE - operation aborted"; break;
    case IS_ERR_INVALIDPNG:		   	csErr = "bad PNG file"; break;
    case IS_ERR_PNGCREATE:				csErr = "internal PNG lib behavior - contact smaller animals s.w."; break;
    case IS_ERR_INTERNAL:				csErr = "misc unexpected behavior error - contact smaller animals s.w."; break;
    case IS_ERR_FONT:				   	csErr = "trouble creating a font object"; break;
    case IS_ERR_INTTIFF:			   	csErr = "misc internal TIFF error"; break;
    case IS_ERR_INVALIDTIFF:			csErr = "invalid TIFF file"; break;
    case IS_ERR_NOTIFFLZW:				csErr = "this will not read TIFF-LZW images"; break;
    case IS_ERR_INVALIDPCX:		   	csErr = "invalid PCX image"; break;
    case IS_ERR_CREATEBMP:				csErr = "a call to the fn CreateCompatibleBitmap failed"; break;
    case IS_ERR_NOLINES:			   	csErr = "end of an image while using single-line de/compression"; break;
    case IS_ERR_GETDIB:					csErr = "error during a call to GetDIBits"; break;
    case IS_ERR_NODEVOP:			   	csErr = "device does not support an operation required by this function"; break;
    case IS_ERR_INVALIDWMF:		   	csErr = "invalid windows metafile"; break;
    case IS_ERR_DEPTHMISMATCH:	   	csErr = "the image was not of the required bit-depth"; break;
    case IS_ERR_BITBLT:					csErr = "a call to BitBlt failed."; break;
    case IS_ERR_BUFTOOSMALL:			csErr = "output buffer is too small for this operation"; break;
    case IS_ERR_TOOMANYCOLORS:	   	csErr = "not enough room in the output palette to store the colors from this image"; break;
    case IS_ERR_INVALIDTGA:			   csErr = "bad TGA File"; break;
    case IS_ERR_CREATEDIB:				csErr = "a call to the fn CreateDIBitmap failed"; break;
    }

    return csErr;
}

//////////////////////////////////////////////////////

UINT CISImageEx::OneBitRowStride(UINT32 w)
{
    return (w + 7) / 8;
}

//////////////////////////////////////////////////////

bool CISImageEx::GetEXIFTagList(const TCHAR *pFile, CStringArray& list)
{
    bool ok = false;

    // open the file
#ifdef _UNICODE
    HISSRC hSrc = _ISFn(is6_OpenFileSourceW)(pFile);
#else
    HISSRC hSrc = _ISFn(is6_OpenFileSource)(pFile);
#endif
    if (hSrc)
    {
        // this will grab any EXIF data that it can find
        HISEXIFREADER hEXIF = _ISFn(is6_EXIFCreateReader)(hSrc, 0, 0);
        if (hEXIF)
        {
            HGLOBAL hStr = NULL;
            UINT32 uLen = 0;

            // get "Main IFD" tag list 
            _ISFn(is6_EXIFGetTagList)(hEXIF, &hStr, &uLen, 1 | 256);

            if (hStr)
            {
                ok = true;

                char *buf = new ISNEWSPEC char[uLen + 1];
                memcpy(buf, (BYTE*)hStr, uLen);
                buf[uLen] = 0;

                list.Add(CString(buf));

                GlobalFree(hStr);
                delete [] buf;
            }

            hStr = NULL;
            uLen = 0;

            // get "EXIF IFD" tag list
            _ISFn(is6_EXIFGetTagList)(hEXIF, &hStr, &uLen, 2 | 256);
            if (hStr)
            {
                ok = true;

                char *buf = new ISNEWSPEC char[uLen + 1];
                memcpy(buf, (BYTE*)hStr, uLen);
                buf[uLen] = 0;

                list.Add(CString(buf));

                GlobalFree(hStr);
                delete [] buf;
            }

            hStr = NULL;
            uLen = 0;

            // get "Interop IFD" tag list
            _ISFn(is6_EXIFGetTagList)(hEXIF, &hStr, &uLen, 4 | 256);
            if (hStr)
            {
                ok = true;

                char *buf = new ISNEWSPEC char[uLen + 1];
                memcpy(buf, (BYTE*)hStr, uLen);
                buf[uLen] = 0;

                list.Add(CString(buf));

                GlobalFree(hStr);
                delete [] buf;
            }

            hStr = NULL;
            uLen = 0;

            // get "Thumbnail IFD" tag list
            _ISFn(is6_EXIFGetTagList)(hEXIF, &hStr, &uLen, 8 | 256);
            if (hStr)
            {
                ok = true;

                char *buf = new ISNEWSPEC char[uLen + 1];
                memcpy(buf, (BYTE*)hStr, uLen);
                buf[uLen] = 0;

                list.Add(CString(buf));

                GlobalFree(hStr);
                delete [] buf;
            }

            hStr = NULL;
            uLen = 0;

            // get "GPS IFD" tag list
            _ISFn(is6_EXIFGetTagList)(hEXIF, &hStr, &uLen, 16 | 256);
            if (hStr)
            {
                ok = true;

                char *buf = new ISNEWSPEC char[uLen + 1];
                memcpy(buf, (BYTE*)hStr, uLen);
                buf[uLen] = 0;

                list.Add(CString(buf));

                GlobalFree(hStr);
                delete [] buf;
            }

            // clean up
            _ISFn(is6_EXIFDestroyReader)(hEXIF);
        }

        // close the file
        _ISFn(is6_CloseSource)(hSrc);
    }

    return ok;
}

//////////////////////////////////////////////////////

bool CISImageEx::GetIPTCTagList(const TCHAR *pFile, CStringArray& list)
{
    bool ok = false;

    // open the file
#ifdef _UNICODE
    HISSRC hSrc = _ISFn(is6_OpenFileSourceW)(pFile);
#else
    HISSRC hSrc = _ISFn(is6_OpenFileSource)(pFile);
#endif
    if (hSrc)
    {
        // this will grab any IPTC data that it can find
        HISIPTC hIPTC = _ISFn(is6_IPTCInitialize)(hSrc, 0, 0);
        if (hIPTC)
        {
            HGLOBAL hStr = NULL;
            UINT32 uLen = 0;

            // get tag list
            _ISFn(is6_IPTCGetTagList)(hIPTC, &hStr, &uLen, 0x100);

            CString csList;
            if (hStr)
            {
                ok = true;

                char *buf = new ISNEWSPEC char[uLen + 1];
                memcpy(buf, (BYTE*)hStr, uLen);
                buf[uLen] = 0;

                list.Add(CString(buf));

                GlobalFree(hStr);
                delete [] buf;
            }

            hStr = NULL;
            uLen = 0;

            // clean up
            _ISFn(is6_IPTCRelease)(hIPTC);
        }

        // close the file
        _ISFn(is6_CloseSource)(hSrc);

    }

    return ok;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Fill(COLORREF clr)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    switch (BitDepth())
    {
    case 32:
    case 24:
        _ISFn(is6_FillSolidRect)(ImageData(), Width(), Height(), BitDepth()/8, RowStride(), CRect(0,0,Width(), Height()), clr, 0);
        break;
    case 8:
        {
            HISCOLORMATCH hcm = _ISFn(is6_CreateColorMatcher)(Palette(), PalColors());
            int idx = _ISFn(is6_FindClosestColor)(hcm, GetRValue(clr), GetGValue(clr), GetBValue(clr));
            _ISFn(is6_DestroyColorMatcher)(hcm);

            _ISFn(is6_FillSolidRect)(ImageData(), Width(), Height(), 1, RowStride(), CRect(0,0,Width(), Height()), idx, 0);
        }
        break;
    case 1:
        _ISFn(is6_FillSolidRect1Bit)(ImageData(), Width(), Height(), RowStride(), CRect(0,0,Width(), Height()), clr);
        break;
    }

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Rotate(double radians, COLORREF backColor, bool bFast)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (radians==0)
    {
        return IS_ERR_OK;
    }

    // how big is the rotated version ?
    UINT32 rw, rh;
    _ISFn(is6_GetRotatedPixelSize)(radians, Width(), Height(), &rw, &rh);

    // allocate that much room
    CISImageEx temp;
    UINT32 res = temp.AllocBlank(rw, rh, BitDepth());
    if (res!=IS_ERR_OK)
    {
        return res;
    }

    // copy palette, etc.
    temp.CopyAttribs(*this);

    // make a solid background
    temp.Fill(backColor);

    // rotate into temp
    switch (BitDepth())
    {
    case 32:
    case 24:
    case 8:
        _ISFn(is6_RotateImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), temp.ImageData(), temp.RowStride(), Palette(), PalColors(), radians, bFast ? 1 : 0);
        break;
    case 1:
        _ISFn(is6_RotateOneBitImage)(ImageData(), Width(), Height(), RowStride(), temp.ImageData(), temp.RowStride(), radians, bFast ? 1 : 0);
        break;
    }

    Steal(temp);

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::AutoCrop()
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    RECT rc;

    BOOL ok = FALSE;
    switch (BitDepth())
    {
    case 1:
        {
            // try to determine background vs foreground based on histograms
            UINT32 histo[2];
            _ISFn(is6_GetOneBitHistogram)(ImageData(), Width(), Height(), RowStride(), histo, 0);
            ok = _ISFn(is6_GetBorder)(ImageData(), Width(), Height(), BitDepth(), RowStride(), Palette(), 
                0, 
                histo[0] > histo[1] ? 0 : 1, // if there are more 0 pixels than 1, assume 0 is the background
                &rc, 1);
        }
        break;
    case 8:
    case 24:
    case 32:
        ok = _ISFn(is6_GetBorder)(ImageData(), Width(), Height(), BitDepth(), RowStride(), Palette(), 1024, RGB(0,0,0), &rc, 0);
        if (!ok)
        {
            UINT32 res = _ISFn(is6_GetLastError)();
            if (res == IS_ERR_CANTFINDBORDER)
            {
                AfxMessageBox("Couldn't find a suitable border to crop. Sorry!");
            }
        }
        break;
    }

    if (ok)
    {
        Crop(rc);
    }

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Crop(CRect cr)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    cr.NormalizeRect();

    // we need to add one to bottom and right because
    // the crop area does not include the bottom or right edges.
    cr.right += 1;
    cr.bottom += 1;

    // allocate a temp image
    CISImageEx temp;
    UINT32 res = temp.AllocBlank(cr.Size(), BitDepth());
    if (res!=IS_ERR_OK)
    {
        return res;
    }

    // copy palette, etc.
    temp.CopyAttribs(*this);

    // crop into temp
    switch (BitDepth())
    {
    case 32:
    case 24:
    case 8:
        _ISFn(is6_CropImage)(ImageData(), Width(), Height(), RowStride(), temp.ImageData(), temp.RowStride(), cr, BitDepth() / 8);
        break;
    case 1:
        _ISFn(is6_Crop1Bit)(ImageData(), Width(), Height(), RowStride(), temp.ImageData(), temp.RowStride(), cr);
        break;
    }

    res = _ISFn(is6_GetLastError());

    Steal(temp);

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Deskew(COLORREF backClr)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CISImageEx temp1;
    CISImageEx *pWhichToDeskew = NULL;

    double fSkewAngle = 0;
    if (BitDepth() != 1)
    {
        temp1.CopyAttribs(*this);
        res = Make1BitCopy(temp1, 128, 4);
        if (res==IS_ERR_OK)
        {
            // copy palette, etc.
            pWhichToDeskew = &temp1;
        }
    }
    else
    {
        pWhichToDeskew = this;
    }

    if (res==IS_ERR_OK && pWhichToDeskew!=NULL)
    {
        res = pWhichToDeskew->FindSkewAngle1Bit(fSkewAngle);

        if (res == IS_ERR_OK)
        {
            res = Rotate(fSkewAngle, backClr, false);
        }
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::FindSkewAngle1Bit(double &rads)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 1)
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    rads = 0;
    double deg = 0;

    CISImageEx temp;
    res = temp.Copy(*this);
    if (res!=IS_ERR_OK) return res;

    // get rid of anything on the edges
    temp.AutoCrop();

    // do a morphological boundary extraction to isolate edges
    int mask[9] = 
    {
        1, 1, 1,
        1, 1, 1,
        1, 1, 1,
    };
    temp.BoundaryExtraction(mask, 3);

    // try to determine background vs foreground based on histograms.
    // by default, the deskew function assumes background is 0 and foreground is 1
    UINT32 histo[2];
    _ISFn(is6_GetOneBitHistogram)(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(), histo, 0);

    // if we need to change that assumption, set the flag
    UINT32 uDeskewFlag = 0;
    if (histo[1] > histo[0])
        uDeskewFlag = 1;

    // look at half of the pixels. use 1/2 degree resolution
    BOOL ok = _ISFn( is6_FindDeskewAngleOneBit )(temp.ImageData(), temp.Width(), temp.Height(), temp.RowStride(), &deg, 2, 45, temp.Width() * temp.Height() / 2, uDeskewFlag);

    if (ok)
    {                                                       
        // degrees to radians 
        double pi = acos((double)-1);
        rads = (pi / 180.0) * (double)-deg; // -deg, because we want to rotate in the opposite direction
    }
    else
    {
        res = _ISFn(is6_GetLastError());
    }

    return res;
}

//////////////////////////////////////////////////////

CSize	CISImageEx::GetDPIInches() const
{
    switch (m_eDPIUnits) 
    {
    default:
    case eDPINone:
        return CSize(0,0);
    case eDPIcm:
        return CSize((int)(m_fDPIX * 2.54), (int)(m_fDPIY * 2.54));
    case eDPIInch:
        return CSize(DPIX(), DPIY());
    }
}

//////////////////////////////////////////////////////

BYTE *   CISImageEx::ToDIB()
{
    if (!ISValid())
    {
        return NULL;
    }

    UINT32 dibSize = _ISFn(is6_GetISDIBSize)(Width(), Height(), BitDepth(), PalColors());

    BYTE *pOut = new ISNEWSPEC BYTE[dibSize];

    if (pOut)
    {
        switch (BitDepth())
        {
        case 32:
            _ISFn(is6_RGBAToDIB)(ImageData(), Width(), Height(), RowStride(), pOut, 0);
            break;

        case 24:
            _ISFn(is6_RGBToDIB)(ImageData(), Width(), Height(), RowStride(), pOut, 0);
            break;

        case 8:
            _ISFn(is6_8BitToDIB)(ImageData(), Width(), Height(), RowStride(), PalColors(), 8, Palette(), pOut, 0);
            break;

        case 1:
            _ISFn(is6_1BitToDIB)(ImageData(), Width(), Height(), RowStride(), Palette(), pOut, 0);
            break;

        default:
            delete [] pOut;
            pOut = NULL;
            break;
        }
    }


    return pOut;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::FromDIB(BITMAPINFOHEADER * pBMIH, BYTE defAlpha /*=255*/)
{
    Clear();
    UINT32 res = IS_ERR_OK;

    CISImageEx temp;

    UINT32 bc, w;
    __int32 h;

    BOOL ok = _ISFn(is6_DIBBitCount)(pBMIH, &bc);
    if (ok)
    {
        _ISFn(is6_DIBWidth)(pBMIH, &w);
        _ISFn(is6_DIBHeight)(pBMIH, &h);

        UINT32 colors = 0;
        _ISFn(is6_GetDIBPalette)(pBMIH, &colors);

        if (bc==32)
        {
            temp.AllocBlank(w, h, 32);
            ok = _ISFn(is6_DIBToRGBA)(pBMIH, defAlpha, temp.ImageData());
        }
        else if (bc > 8)
        {
            temp.AllocBlank(w, h, 24);
            ok = _ISFn(is6_DIBToRGB)(pBMIH, temp.ImageData());
        }
        else if (bc > 1)
        {
            temp.AllocBlank(w, h, 8);
            ok = _ISFn(is6_DIBToColormapped)(pBMIH, 8, &colors, temp.Palette(), temp.ImageData());
            temp.m_uPalColors = colors;

        }
        else if (bc == 1)
        {
            temp.AllocBlank(w, h, 1);
            ok = _ISFn(is6_DIBToColormapped)(pBMIH, 1, &colors, temp.Palette(), temp.ImageData());
            temp.m_uPalColors = colors;
        }
        else
        {
            res = IS_ERR_BADPARAM;
            ok = FALSE;
        }

    }

    if (ok)
    {
        Steal(temp);
    }
    else
    {
        res = _ISFn(is6_GetLastError());
    }

    return res;

}

//////////////////////////////////////////////////////

UINT32   CISImageEx::AdjustColorBalance(int R, int G, int B,int Rshadow, int Gshadow, int Bshadow,int Rhighlight,int Ghighlight,int Bhighlight,int Density)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_AdjustImageColorBalance)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), R, G, B, Rshadow, Gshadow, Bshadow, Rhighlight, Ghighlight, Bhighlight, Density, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Duotone(COLORREF clr)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_DuotoneImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), clr, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Invert(bool noModAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_InvertImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), noModAlpha ? 1 : 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Posterize()
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_PosterizeImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Halftone(UINT32 uMaxDotSize, bool bSmoothDots)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_HalftoneImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), uMaxDotSize, bSmoothDots ? 1 : 0);

    return _ISFn(is6_GetLastError)();
}


//////////////////////////////////////////////////////

UINT32   CISImageEx::Mosaic(UINT32 uSquareSize)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() == 1)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_MosaicImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), uSquareSize);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Solarize()
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_SolarizeImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 1);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::AdjustTint(int tint)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_AdjustTint)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), tint, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Crackle(int mode, UINT32 uDefn, bool highContrast)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24)
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx temp;
    temp.AllocBlank(Size(), 24);

    if (!temp.ISValid())
    {
        return IS_ERR_MEM;
    }

    _ISFn(is6_CrackleRGB)(ImageData(), Width(), Height(), RowStride(), temp.ImageData(), temp.RowStride(), mode, uDefn, highContrast ? 0 : 1);

    Steal(temp);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Gauzy(double blur, double sharp, double clarity, UINT32 mode)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx temp;
    temp.AllocBlank(*this);

    if (!temp.ISValid())
    {
        return IS_ERR_MEM;
    }

    _ISFn(is6_GauzyImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), temp.ImageData(), temp.RowStride(), blur, sharp, clarity, mode);

    Steal(temp);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::InnerGlow(UINT32 size, double opacity, COLORREF glowClr)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_InnerGlowRGBA)(ImageData(), Width(), Height(), RowStride(), size, opacity, glowClr, NULL, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::OuterGlow(UINT32 size, double opacity, COLORREF glowClr)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_OuterGlowRGBA)(ImageData(), Width(), Height(), RowStride(), size, opacity, glowClr, NULL, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::DropShadow(int offsetMax, int diffusion, COLORREF clrBackground, COLORREF clrShadow, int angle, bool useAlphaAsShadow, bool modAlphaOnly )
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    // can only use one of these!
    if (modAlphaOnly && useAlphaAsShadow)
    {
        return IS_ERR_BADPARAM;
    }

    HGLOBAL hTemp=NULL;
    UINT32 w=0, h=0;
    POINT op, sp;

    _ISFn(is6_DropShadow)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), &hTemp, &w, &h, offsetMax, diffusion, clrBackground, clrShadow, angle, &op, &sp, (useAlphaAsShadow ? 1 : 0) | (modAlphaOnly ? 2 : 0));

    if (hTemp)
    {
        CISImageEx temp;
        temp.AssignHGLOBAL(hTemp, w, h, BitDepth(), 0, NULL);
        Copy(temp);
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::AdjustGamma(double fGamma)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 32 && BitDepth() != 24 && BitDepth() != 8)
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_AdjustGamma)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), fGamma, 0);

    return _ISFn(is6_GetLastError)();
}


//////////////////////////////////////////////////////

UINT32   CISImageEx::AdjustBrightnessContrast(double brightness, double contrast, bool luminance, bool sigmoidal, double power)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 32 && BitDepth() != 24 && BitDepth() != 8)
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    if (sigmoidal)
    {
        _ISFn(is6_AdjustBrightnessContrast)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), brightness, contrast, power, 0, luminance ? (1<<8) : 1);
    }
    else
    {
        _ISFn(is6_AdjustBrightnessContrast)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), brightness, contrast, 0, 1, luminance ? (1<<8) : 1);
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::AplyLUTsToImage(BYTE **pLUTs)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && pLUTs==NULL)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_ApplyLUTsToImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), (const void**)pLUTs, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::AutoBrightness(UINT32 lowThresh, UINT32 highThresh, bool stretch)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_AutoBrightnessRGB)(ImageData(), Width(), Height(), RowStride(), lowThresh, highThresh, stretch ? 0:1);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::GetBrightnessHistogram(UINT32 histo[256])
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() !=32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_GetBrightnessHistogram)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), histo, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::GetChannelHistograms(UINT32 **pHistos)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() !=32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_GetImageChannelHistograms)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), pHistos, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Get1BitHistogram(UINT32 histo[2])
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 1)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_GetOneBitHistogram)(ImageData(), Width(), Height(), RowStride(), histo, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::BrightnessHistogramStretch(double fLowLimit, double fHiLimit, UINT32 midPoint, bool useHSL, bool useMidpoint)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() !=32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_BrightnessHistogramStretchImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), fLowLimit, fHiLimit, midPoint, (useHSL ? (1<<9):0) | (useMidpoint? (1<<8):0));

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::HistogramStretch(double fLowLimit, double fHiLimit, UINT32 midPoint, bool useMidpoint, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_HistogramStretchImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), fLowLimit, fHiLimit, midPoint, 1 | 2 | 4 | (modAlpha?7:0) | (useMidpoint? (1<<8):0));

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::GetStretchedHistogramLUT(UINT32 histo[256], BYTE LUT[256], double fLowLimit, double fHiLimit, UINT32 midPoint, bool useMidpoint)
{
    UINT32 total = 0;
    for (UINT32 i=0;i<256;i++)
    {
        total+=histo[i];
    }

    _ISFn(is6_GetStretchedHistogramLUT)(histo, total, fLowLimit, fHiLimit, midPoint, LUT, (useMidpoint? (1<<8):0));

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::BrightnessHistogramEqualize(BYTE lowThreshold, BYTE highThreshold)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() !=32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_BrightnessHistogramEqualizeImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), lowThreshold, highThreshold, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::HistogramEqualize(BYTE lowThreshold, BYTE highThreshold, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_HistogramEqualizeImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), lowThreshold, highThreshold, 1 | 2 | 4 | (modAlpha?7:0));

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::HistogramSpecification(UINT32 histo[256], bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_HistogramSpecification)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), histo, 1 | 2 | 4 | (modAlpha?7:0));

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::ContrastMask(double sigma, bool highContrast)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_ContrastMaskImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), highContrast?1:0, sigma, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::InpaintImage(CISImageEx &mask8, UINT32 sigma, bool better, bool noModAlpha)
{
    if (!ISValid() || !mask8.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() !=32 && BitDepth() != 8)
    {
        return IS_ERR_BADPARAM;
    }

    if (mask8.BitDepth() != 8)
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx temp;
    temp.AllocBlank(*this);
    if (!temp.ISValid())
    {
        return IS_ERR_MEM;
    }

    _ISFn(is6_InpaintImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
        temp.ImageData(), temp.RowStride(),
        mask8.ImageData(), mask8.RowStride(),
        sigma, (noModAlpha?(1<<1):0) | (better?1:0));

    Steal(temp);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::DetermineImageTranslation(CISImageEx &img_sub, CPoint & pt, bool applyHammingWindowFirst)
{
    if (!ISValid() || !img_sub.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    // this operates on 8-bit grayscale images
    CISImageEx *pA = this, *pB = &img_sub;

    CISImageEx thisCopy, thatCopy;

    if (BitDepth() != 8)
    {
        UINT32 res = Make8BitGrayCopy(thisCopy);
        if (res != IS_ERR_OK)
            return res;

        pA = &thisCopy;
    }

    if (img_sub.BitDepth() != 8)
    {
        UINT32 res = img_sub.Make8BitGrayCopy(thatCopy);
        if (res != IS_ERR_OK)
            return res;

        pB = &thatCopy;
    }

    int x, y;
    _ISFn(is6_DetermineImageTranslation)(pA->ImageData(), pA->Width(), pA->Height(), pA->RowStride(),
        pB->ImageData(), pB->Width(), pB->Height(), pB->RowStride(),
        &x, &y, applyHammingWindowFirst ? 1 : 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::UnsharpMask(UINT32 uThreshold, double sigma, double amount, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth()!=24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_UnsharpMaskImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), uThreshold, amount, sigma, (1 | 2 | 4) | (modAlpha?7:0));
        Steal(t);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::AdaptiveUnsharpMask(UINT32 uThreshold, double amount, int mode, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_AdaptiveUnsharpMask)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), uThreshold, amount, mode, (1 | 2 | 4) | (modAlpha?7:0));
        Steal(t);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Convolve(double *pKernelNxN, UINT32 N, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
       _ISFn( is6_Convolve )(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            t.ImageData(), t.RowStride(), 
            0, 
            1.0, pKernelNxN, N, N, (1 | 2 | 4) | (modAlpha?7:0));

        Steal(t);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::CannyEdgeDetector(double sigma, double low, double high, CISImageEx & edges, bool fast)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    // this operates on 8-bit grayscale images
    CISImageEx *pA = this;

    CISImageEx thisCopy;

    if (BitDepth() != 8)
    {
        UINT32 res = Make8BitGrayCopy(thisCopy);
        if (res != IS_ERR_OK)
            return res;

        pA = &thisCopy;
    }

    res = edges.AllocBlank(Size(), 8);
    edges.SetGrayPalette();

    if (res == IS_ERR_OK)
    {
        _ISFn(is6_CannyEdgeDetector)(ImageData(), Width(), Height(), RowStride(), 
            edges.ImageData(), edges.RowStride(), 
            sigma, low, high, fast ? 1 : 0);

        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::MotionBlur(double angle, double sigma, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_MotionBlur)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            t.ImageData(), t.RowStride(), 
            angle, sigma, (1 | 2 | 4) | (modAlpha?7:0));
        Steal(t);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::RadialBlur(double angle, double quality, CPoint center, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_RadialBlur)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            t.ImageData(), t.RowStride(), 
            angle, quality, &center, (1 | 2 | 4) | (modAlpha?7:0));
        Steal(t);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::ZoomBlur(double amount, CPoint center, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_ZoomBlur)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            t.ImageData(), t.RowStride(), 
            amount, &center, (1 | 2 | 4) | (modAlpha?7:0));
        Steal(t);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Filter(filterType type, UINT32 filterSize, double param, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = IS_ERR_OK;

    CISImageEx t;

    if (BitDepth()==1 && type==Median)
    {
        res = t.AllocBlank(*this);
        if (res == IS_ERR_OK)
        {
            _ISFn(is6_MedianFilter1Bit)(ImageData(), Width(), Height(), RowStride(), t.ImageData(), t.RowStride(), filterSize, 0);
            Steal(t);
            res = _ISFn(is6_GetLastError)();
        }
        return res;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        switch (type)
        {
        case ArithMean:
            _ISFn(is6_Filter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), filterSize, 0, 0, (1 | 2 | 4) | (modAlpha?7:0));
            break;
        case GeoMean:
            _ISFn(is6_Filter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), filterSize, 1, 0, (1 | 2 | 4) | (modAlpha?7:0));
            break;
        case HarmonicMean:
            _ISFn(is6_Filter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), filterSize, 2, 0, (1 | 2 | 4) | (modAlpha?7:0));
            break;
        case YpMean:
            _ISFn(is6_Filter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), filterSize, 3, (UINT32)param, (1 | 2 | 4) | (modAlpha?7:0));
            break;
        case Midpoint:
            _ISFn(is6_Filter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), filterSize, 4, 0, (1 | 2 | 4) | (modAlpha?7:0));
            break;
        case Min:
            _ISFn(is6_Filter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), filterSize, 7, 0, (1 | 2 | 4) | (modAlpha?7:0));
            break;
        case Max:
            _ISFn(is6_Filter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), filterSize, 6, 0, (1 | 2 | 4) | (modAlpha?7:0));
            break;
        case Median:
            _ISFn(is6_MedianFilterImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), t.ImageData(), t.RowStride(), filterSize, param, (1 | 2 | 4) | (modAlpha?7:0));
            break;
        }

        Steal(t);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Threshold(UINT32 lowThreshold, UINT32 highThreshold, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_ThresholdFilter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), lowThreshold, highThreshold, (1 | 2 | 4) | (modAlpha?7:0));

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::DustAndScratches(UINT32 filterSize, UINT32 threshold, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_DustAndScratches)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            t.ImageData(), t.RowStride(),
            filterSize, threshold, 1.0, (1 | 2 | 4) | (modAlpha?7:0));

        res = _ISFn(is6_GetLastError)();

        Steal(t);
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::BilateralFilter(UINT32 filterSize, UINT32 filterType, double sigma, bool fast, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_BilateralFilterImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            t.ImageData(), t.RowStride(),
            filterSize, filterType, sigma, (1 | 2 | 4) | (modAlpha?7:0) | (fast ? (1<<8) : 0));

        Steal(t);

        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::Sobel(CISImageEx & out8, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    res = out8.AllocBlank(Size(), 8);
    out8.SetGrayPalette();

    if (res != IS_ERR_OK) return res;

    // output is 16bpc
    UINT32 trs = Width();
    ISUINT16 * pS16 = new ISNEWSPEC ISUINT16[trs * Height()];

    if (pS16)
    {
        _ISFn(is6_SobelFilter)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), pS16, trs, (1<<8));

        res = _ISFn(is6_GetLastError)();

        if (res == IS_ERR_OK)
        {
            _ISFn(is6_Convert16bpcTo8bpc)(pS16, Width(), Height(), 1, 16, trs, out8.ImageData(), out8.RowStride(), 0);

            res = _ISFn(is6_GetLastError)();
        }

        delete [] pS16;
    }
    else
    {
        res = IS_ERR_MEM;
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::WaveletDenoise8BitGrayscale(double fThreshold, double fCurve)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()!=8)
    {
        return IS_ERR_BADPARAM;
    }

    CISImageEx t;
    UINT res = t.AllocBlank(*this);
    if (res == IS_ERR_OK)
    {
        _ISFn(is6_MultiscaleDenoise)(ImageData(), Width(), Height(), 1, RowStride(), t.ImageData(), t.RowStride(), fThreshold, fCurve, 0);

        Steal(t);

        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32   CISImageEx::WaveletDenoise(double fThreshold, double fCurve, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==8 && Is8BitGray())
    {
        return WaveletDenoise8BitGrayscale(fThreshold, fCurve);
    }

    UINT32 res = IS_ERR_OK;

    // split, process each channel separately

    CISImageEx r,g,b,a;
    res = SplitToChannels(r,g,b,a);
    if (res != IS_ERR_OK) return res;

    res = r.WaveletDenoise8BitGrayscale(fThreshold, fCurve);
    if (res != IS_ERR_OK) return res;

    res = g.WaveletDenoise8BitGrayscale(fThreshold, fCurve);
    if (res != IS_ERR_OK) return res;

    res = b.WaveletDenoise8BitGrayscale(fThreshold, fCurve);
    if (res != IS_ERR_OK) return res;

    if (a.ISValid() && modAlpha) res = a.WaveletDenoise8BitGrayscale(fThreshold, fCurve);
    if (res != IS_ERR_OK) return res;

    // recombine
    res = CombineChannels(r, g, b, a, BitDepth()==32);

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::SplitToChannels(CISImageEx & r, CISImageEx & g, CISImageEx & b, CISImageEx & a) const 
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    UINT32 res = IS_ERR_OK;

    res = r.AllocBlank(Size(), 8);
    if (res != IS_ERR_OK) return res;

    res = g.AllocBlank(Size(), 8);
    if (res != IS_ERR_OK) return res;

    res = b.AllocBlank(Size(), 8);
    if (res != IS_ERR_OK) return res;

    if (BitDepth() == 32)
    {
        res = a.AllocBlank(Size(), 8);
        if (res != IS_ERR_OK) return res;
    }

    BYTE *pCh[4];
    pCh[0] = r.ImageData();
    pCh[1] = g.ImageData();
    pCh[2] = b.ImageData();
    pCh[3] = a.ImageData();

    _ISFn(is6_SplitImage)(ImageDataConst(), Width(), Height(), BitDepth() / 8, RowStride(), (void**)pCh, r.RowStride(), 0);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::CombineChannels(const CISImageEx & r, const CISImageEx & g, const CISImageEx & b, const CISImageEx & a, bool bIs32Bit)
{
    if (!r.ISValid() || !g.ISValid() || !b.ISValid() || (bIs32Bit && !a.ISValid()))
    {
        return IS_ERR_BADPARAM;
    }

    Clear();

    const BYTE *pCh[4];
    pCh[0] = r.ImageDataConst();
    pCh[1] = g.ImageDataConst();
    pCh[2] = b.ImageDataConst();
    pCh[3] = a.ImageDataConst();

    UINT res = AllocBlank(r.Size(), bIs32Bit ? 32 : 24);
    if (res==IS_ERR_OK)
    {
        _ISFn(is6_CombineImageChannels)((const void**)pCh, r.Width(), r.Height(), bIs32Bit ? 4 : 3, r.RowStride(), ImageData(), RowStride(), 0);
        res = _ISFn(is6_GetLastError)();
    }

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::WaveletDecompose8BitGrayscale(UINT32 levelCount, std::vector<CISImageEx*> levels)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 8 || levelCount < 2)
    {
        return IS_ERR_BADPARAM;
    }

    levels.clear();

    UINT32 res = IS_ERR_OK;

    // array of pointers to BYTE buffers
    BYTE **pL = new ISNEWSPEC BYTE*[levelCount];
    if (pL==NULL)
    {
        return IS_ERR_MEM;
    }

    // alloc a CISImageEx for each sublevel, add it to the array and to the vector
    for (UINT32 i=0;i<levelCount;i++)
    {
        CISImageEx * t = new ISNEWSPEC CISImageEx();

        if (t==NULL) 
        {
            for (UINT32 z=0;z<i-1;z++)
            {
                delete levels[z];
            }
            levels.clear();
            delete [] pL;
            return IS_ERR_MEM;
        }

        res = t->AllocBlank(Size(), 8);
        if (res==IS_ERR_MEM) 
        {
            for (UINT32 z=0;z<i;z++)
            {
                delete levels[z];
            }
            levels.clear();
            delete [] pL;
            return IS_ERR_MEM;
        }

        pL[i] = t->ImageData();
        levels.push_back(t);
    }   

    _ISFn(is6_MultiscaleDecompose)(ImageData(), Width(), Height(), RowStride(), levelCount - 1, (void**)pL, levels[0]->RowStride(), 0);
    res = _ISFn(is6_GetLastError)();

    delete [] pL;

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::WaveletRecompose8BitGrayscale(UINT32 levelCount, std::vector<CISImageEx*> levels)
{
    Clear();

    if (levels.size() != levelCount || levelCount < 2)
    {
        return IS_ERR_BADPARAM;
    }

    // sanity checks
	UINT32 i;
    for ( i = 0; i < levelCount; i++)
    {
        if (!levels[i]->ISValid())
            return IS_ERR_BADPARAM;

        if (levels[i]->BitDepth() != 8)
            return IS_ERR_BADPARAM;
    }

    for (i = 1; i < levelCount; i++)
    {
        if (levels[i]->Size() != levels[i-1]->Size())
            return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    // array of pointers to BYTE buffers
    BYTE **pL = new ISNEWSPEC BYTE*[levelCount];
    if (pL==NULL)
    {
        return IS_ERR_MEM;
    }

    for (i = 0; i < levelCount; i++)
    {
        pL[i] = levels[i]->ImageData();
    }

    res = AllocBlank(levels[0]->Size(), 8);
    if (res != IS_ERR_OK)
    {
        delete [] pL;
        return IS_ERR_MEM;
    }

    _ISFn(is6_MultiscaleRecompose)((const void**)pL, levels[0]->Width(), levels[0]->Height(), levels[0]->RowStride(), levelCount, ImageData(), RowStride(), 0);
    res = _ISFn(is6_GetLastError)();

    delete [] pL;

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::Morpho(morphoPrimitiveMode mode, const int *maskNxN, UINT32 N)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 1 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if ((N & 0x01)==0) // kernel must be odd and greater than 0
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
    {
        return IS_ERR_MEM;
    }

    if (BitDepth()==8)
    {
        t.SetGrayPalette();

        switch (mode)
        {
        case Erosion:
            _ISFn(is6_MorphoErosionGray)(ImageData(), Width(), Height(), RowStride(), maskNxN, N, t.ImageData(), t.RowStride(), 0);
            break;
        case Dilation:
            _ISFn(is6_MorphoDilationGray)(ImageData(), Width(), Height(), RowStride(), maskNxN, N, t.ImageData(), t.RowStride(), 0);
            break;
        case Open:
            _ISFn(is6_MorphoErosionGray)(ImageData(), Width(), Height(), RowStride(), maskNxN, N, t.ImageData(), t.RowStride(), 0);
            break;
        case Close:
            _ISFn(is6_CloseGray)(ImageData(), Width(), Height(), RowStride(), maskNxN, N, t.ImageData(), t.RowStride(), 0);
            break;
        }
    }
    else
    {
        switch (mode)
        {
        case Erosion:
            _ISFn(is6_MorphoErosion1Bit)(ImageData(), Width(), Height(), RowStride(), t.ImageData(), t.RowStride(), maskNxN, N, N, 0);
            break;
        case Dilation:
            _ISFn(is6_MorphoDilation1Bit)(ImageData(), Width(), Height(), RowStride(), t.ImageData(), t.RowStride(), maskNxN, N, N, 0);
            break;
        case Open:
            _ISFn(is6_MorphoOpen1Bit)(ImageData(), Width(), Height(), RowStride(), t.ImageData(), t.RowStride(), maskNxN, N, N, 0);
            break;
        case Close:
            _ISFn(is6_MorphoClose1Bit)(ImageData(), Width(), Height(), RowStride(), t.ImageData(), t.RowStride(), maskNxN, N, N, 0);
            break;
        }
    }

    Steal(t);

    res = _ISFn(is6_GetLastError)();

    return res;
}

// grayscale morpho
//////////////////////////////////////////////////////

UINT32  CISImageEx::TopHat(const int *maskNxN, UINT32 N, bool blackHat, bool addKernelValues)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if ((N & 0x01)==0) // kernel must be odd and greater than 0
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
    {
        return IS_ERR_MEM;
    }

    _ISFn(is6_TophatGray)(ImageData(), Width(), Height(), RowStride(), maskNxN, N, t.ImageData(), t.RowStride(), (addKernelValues?0:1) | (blackHat?2:0));

    Steal(t);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::Gradient(const int *maskNxN, UINT32 N, bool addKernelValues)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if ((N & 0x01)==0) // kernel must be odd and greater than 0
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
    {
        return IS_ERR_MEM;
    }

    t.SetGrayPalette();

    _ISFn(is6_MorphoGradientGray)(ImageData(), Width(), Height(), RowStride(), maskNxN, N, t.ImageData(), t.RowStride(), (addKernelValues?0:1));

    Steal(t);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::HitOrMiss(const int *AmaskNxN, UINT32 AN, const int *BmaskNxN, UINT32 BN)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 1)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (((AN & 0x01)==0) || ((BN & 0x01)==0)) // kernel must be odd and greater than 0
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
    {
        return IS_ERR_MEM;
    }

    _ISFn(is6_MorphoHitOrMiss1Bit)(ImageData(), Width(), Height(), RowStride(), t.ImageData(), t.RowStride(), AmaskNxN, AN, AN, BmaskNxN, BN, BN, 0);

    Steal(t);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::BoundaryExtraction(const int *maskNxN, UINT32 N)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 1)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if ((N & 0x01)==0) // kernel must be odd and greater than 0
    {
        return IS_ERR_BADPARAM;
    }

    UINT32 res = IS_ERR_OK;

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
    {
        return IS_ERR_MEM;
    }

    _ISFn(is6_MorphoBoundaryExtraction1Bit)(ImageData(), Width(), Height(), RowStride(), t.ImageData(), t.RowStride(), maskNxN, N, N, 0);

    Steal(t);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::Thinning(UINT32 maxIterations)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 1)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    // 8 masks, 3x3 each. allocate them in one array.
    int AMasks[8 * 9] =
    {
        0, 0, 0,
        1, 1, 1,   
        1, 1, 1,

        1, 0, 0,   
        1, 1, 0,   
        1, 1, 1,

        1, 1, 0,   
        1, 1, 0,   
        1, 1, 0,

        1, 1, 1,   
        1, 1, 0,   
        1, 0, 0,

        1, 1, 1,   
        1, 1, 1,   
        0, 0, 0,

        1, 1, 1,   
        0, 1, 1,   
        0, 0, 1,   

        0, 1, 1,   
        0, 1, 1,   
        0, 1, 1,   

        0, 0, 1,
        0, 1, 1,
        1, 1, 1
    };

    // Bs are inverse of As
    int BMasks[8 * 9];
    for (int i=0;i<8*9;i++)
        BMasks[i] = AMasks[i] == 1 ? 0 : 1;


    UINT32 res = IS_ERR_OK;

    CISImageEx t;
    res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
    {
        return IS_ERR_MEM;
    }

    _ISFn(is6_MorphoThinning1Bit)(ImageData(), Width(), Height(), RowStride(), t.ImageData(), AMasks, 3, BMasks, 3, 8, maxIterations, 0);

    Steal(t);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::Intersect(const CISImageEx &img)
{
    if (!ISValid() || !SameSize(img))
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != img.BitDepth())
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    CISImageEx t;
    UINT res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
        return res;

    _ISFn(is6_IntersectBuffers)(ImageData(), img.ImageDataConst(), t.ImageData(), RowStride() * Height());

    Steal(t);

    res = _ISFn(is6_GetLastError)();

    return res;

}

//////////////////////////////////////////////////////

UINT32  CISImageEx::Union(const CISImageEx &img)
{
    if (!ISValid() || !SameSize(img))
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != img.BitDepth())
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    CISImageEx t;
    UINT res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
        return res;

    _ISFn(is6_UnionBuffers)(ImageData(), img.ImageDataConst(), t.ImageData(), RowStride() * Height());

    Steal(t);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::XOR(const CISImageEx &img)
{
    if (!ISValid() || !SameSize(img))
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != img.BitDepth())
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    CISImageEx t;
    UINT res = t.AllocBlank(*this);
    if (res != IS_ERR_OK)
        return res;

    _ISFn(is6_XORBuffers)(ImageData(), img.ImageDataConst(), t.ImageData(), RowStride() * Height());

    Steal(t);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::Inverse()
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_InvertBuffer)(ImageData(), RowStride() * Height());

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::AddNoise(noiseType type, double fVariance, double fMean, bool monochrome, bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()!=24 && BitDepth() != 32 && BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_BADPARAM;
    }

    switch (type)
    {
    case Gaussian:
        _ISFn(is6_AddNoise)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 0, fVariance, fMean, monochrome, (1 | 2 | 4) | (modAlpha?7:0));
        break;
    case NegativeExponential:
        _ISFn(is6_AddNoise)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 1, fVariance, fMean, monochrome, (1 | 2 | 4) | (modAlpha?7:0));
        break; 
    case Rayleigh:
        _ISFn(is6_AddNoise)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 2, fVariance, fMean, monochrome, (1 | 2 | 4) | (modAlpha?7:0));
        break;
    case Uniform:
        _ISFn(is6_AddNoise)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 3, fVariance, fMean, monochrome, (1 | 2 | 4) | (modAlpha?7:0));
        break;
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::PerlinNoise(UINT32 width, UINT32 height, double scale)
{
    Clear();

    UINT res = AllocBlank(width, height, 8);
    SetGrayPalette();

    HISPERLIN hPerlin = _ISFn(is6_InitializePerlinNoiseVolume)(0);

    if (hPerlin)
    {
        _ISFn(is6_GetPerlinNoiseSlice)(hPerlin, ImageData(), Width(), Height(), RowStride(), scale, 0, 0, 0, 0, 0, 0, 0);

        res = _ISFn(is6_GetLastError)();

        _ISFn(is6_DestroyPerlinNoiseVolume)(hPerlin);
    }

    return res;
}

//////////////////////////////////////////////////////

// RGB onto RGB/RGBA using an 8-bit grayscale image as alpha mask
UINT32  CISImageEx::AlphaBlendRGBMask(const CISImageEx & srcRGB, const CISImageEx & mask8, CPoint pos)
{
    UINT32 res = IS_ERR_OK;

    if (!ISValid() || !srcRGB.ISValid() || !mask8.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (mask8.BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (srcRGB.BitDepth() != 24 && srcRGB.BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    _ISFn(is6_AlphaBlendRGB)(ImageData(), Width(), Height(), RowStride(), (BYTE*)srcRGB.ImageDataConst(), srcRGB.Width(), srcRGB.Height(), srcRGB.RowStride(), (BYTE*)mask8.ImageDataConst(), pos.x, pos.y, 1.0, 0);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

// RGBA onto RGB/RGBA
UINT32  CISImageEx::AlphaBlendRGBA(const CISImageEx & srcRGBA, CPoint pos, alphaModMode mode)
{
    UINT32 res = IS_ERR_OK;

    if (!ISValid() || !srcRGBA.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (srcRGBA.BitDepth()!=24 && srcRGBA.BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    _ISFn(is6_AlphaBlendRGBA)(ImageData(), Width(), Height(), BitDepth() / 8,  RowStride(), 
        (BYTE*)srcRGBA.ImageDataConst(), srcRGBA.Width(), srcRGBA.Height(), srcRGBA.RowStride(), pos.x, pos.y, 1.0, (UINT32)mode, 0);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

// solid color onto RGB/RGBA using an 8-bit grayscale image as alpha mask
UINT32  CISImageEx::AlphaBlendSolid(const CISImageEx & mask8, CPoint pos, COLORREF clr)
{
    UINT32 res = IS_ERR_OK;

    if (!ISValid() || !mask8.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (mask8.BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    _ISFn(is6_AlphaBlendColor)(ImageData(), Width(), Height(), BitDepth() / 8,  RowStride(), 
        (BYTE*)mask8.ImageDataConst(), mask8.Width(), mask8.Height(), mask8.RowStride(), clr, pos.x, pos.y, 0);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

// 8-bit with transparency info in the rgbReserved members of its palette onto RGB/RGBA
UINT32  CISImageEx::AlphaBlend8Trans(const CISImageEx & src8Bit, CPoint pos)
{
    UINT32 res = IS_ERR_OK;

    if (!ISValid() || !src8Bit.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (src8Bit.BitDepth() != 8)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    _ISFn(is6_OverlayImage8BitAlphaOnRGB)(ImageData(), Width(), Height(), RowStride(), 
        (BYTE*)src8Bit.ImageDataConst(), src8Bit.Width(), src8Bit.Height(), src8Bit.RowStride(), (RGBQUAD*)src8Bit.ConstPalette(), pos.x, pos.y, 0);

    res = _ISFn(is6_GetLastError)();

    return res;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::BGR()
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==24)
    {
        _ISFn(is6_RGBToBGR)(ImageData(), Width(), Height(), RowStride(), 0);
    }
    else
    {
        _ISFn(is6_RGBAToBGRA)(ImageData(), Width(), Height(), RowStride(), 0);
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::SetChannel(const CISImageEx & channel8, UINT32 channel) 
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (Size() != channel8.Size())
    {
        return IS_ERR_BADPARAM;
    }

    _ISFn(is6_SetImageChannel)((void *)channel8.ImageDataConst(), Width(), Height(), channel8.RowStride(), ImageData(), BitDepth() / 8, RowStride(), channel, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::SplitRGBAToRGBAAndA(CISImageEx & rgb, CISImageEx & a) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    UINT res;

    res = rgb.AllocBlank(Size(), 24);
    if (res != IS_ERR_OK) return res;

    res = a.AllocBlank(Size(), 8);
    if (res != IS_ERR_OK) return res;

    _ISFn(is6_SplitRGBAToRGBPlusA)((BYTE*)ImageDataConst(), Width(), Height(), RowStride(), rgb.ImageData(), rgb.RowStride(), a.ImageData(), a.RowStride(), 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::FillSolidRect(CRect rect, COLORREF clr, double opacity)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    switch (BitDepth())
    {
    case 32:
    case 24:
        if (opacity==1)
        {
            _ISFn(is6_FillSolidRect)(ImageData(), Width(), Height(), BitDepth()/8, RowStride(), &rect, clr, 0);
        }
        else
        {
            _ISFn(is6_FillTransparentRectOnImage)(ImageData(), Width(), Height(), BitDepth()/8, RowStride(), &rect, clr, opacity, 0);
        }
        break;
    case 8:
        _ISFn(is6_FillSolidRect)(ImageData(), Width(), Height(), BitDepth()/8, RowStride(), &rect, clr, 0);
        break;
    case 1:
        _ISFn(is6_FillSolidRect1Bit)(ImageData(), Width(), Height(), RowStride(), &rect, clr);
        break;
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::FloodFill(CPoint ptStart, COLORREF clr, UINT32 tolerance)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    _ISFn(is6_FloodFillImage)(ImageData(), Width(), Height(), BitDepth()/8, RowStride(), &ptStart, clr, 1, tolerance, 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::GetFillMask(CPoint ptStart, COLORREF clr, UINT32 tolerance, CISImageEx &mask8) const
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 8 && BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    UINT32 res = mask8.AllocBlank(Size(), 8);
    if (res !=IS_ERR_OK) return res;

    mask8.SetGrayPalette();

    _ISFn(is6_GetColorMatchMaskFromImage)((BYTE*)ImageDataConst(), Width(), Height(), BitDepth()/8, RowStride(), mask8.ImageData(), mask8.RowStride(), &ptStart, clr, 1, tolerance, 2);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::PolygonFill(COLORREF clr, double opacity, std::vector<XYdouble> & points, bool smooth)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    XYdouble * pts = new ISNEWSPEC XYdouble[points.size()];
    if (pts==NULL)
    {
        return IS_ERR_MEM;
    }

    for (size_t i = 0; i < points.size(); i++)
    {
        pts[i].x = points[i].x;
        pts[i].y = points[i].y;
    }

    switch (BitDepth())
    {
    case 32:
    case 24:
    case 8:
        _ISFn(is6_PolygonFillImage)(ImageData(), Width(), Height(), BitDepth()/8, RowStride(), pts, (UINT32)points.size(), NULL, 0, 0, 0, clr, opacity, Palette(), PalColors(), 1 | (smooth?4:0));
        break;
    case 1:
        _ISFn(is6_PolygonFill1Bit)(ImageData(), Width(), Height(), RowStride(), pts, (UINT32)points.size(), clr, 0);
        break;
    }

    delete [] pts;

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::Ellipse(CPoint center, UINT32 arad, UINT32 brad, COLORREF clr, bool fill, double opacity, bool smooth)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    switch (BitDepth())
    {
    case 32:
    case 24:
    case 8:
        _ISFn(is6_DrawEllipseOnImage)(ImageData(), Width(), Height(), BitDepth()/8, RowStride(), clr, &center, arad, brad, fill, opacity, Palette(), PalColors(), 2 | (smooth?8:0));
        break;
    case 1:
        _ISFn(is6_DrawEllipseOn1Bit)(ImageData(), Width(), Height(), RowStride(), clr, &center, arad, brad, fill, 0);
        break;
    }

    return _ISFn(is6_GetLastError)();
}

typedef enum {noArrow, arrow, barbedArrow, square, diamond, notched} lineEndStyle;

//////////////////////////////////////////////////////

UINT32  CISImageEx::Line(CPoint a, CPoint b, COLORREF clr, double opacity, bool smooth, UINT32 thickness, lineEndStyle end1, lineEndStyle end2)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    XYdouble xya, xyb;
    xya.x = a.x;
    xya.y = a.y;
    xyb.x = b.x;
    xyb.y = b.y;
    switch (BitDepth())
    {
    case 32:
    case 24:
    case 8:
        if (thickness > 1)
            _ISFn(is6_DrawThickLineOnImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), clr, Palette(), PalColors(), opacity, &xya, &xyb, thickness, (int)end1, (int)end2, 15, thickness+15, 1 | (smooth?4:0));
        else
            _ISFn(is6_DrawLineOnImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), clr, &a, &b, smooth?2:0);
        break;
    case 1:
        _ISFn(is6_DrawLineOn1Bit)(ImageData(), Width(), Height(), RowStride(), clr, &a, &b, 0);
        break;
    }

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::Despeckle(bool modAlpha)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==1) 
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    _ISFn(is6_DespeckleImage)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 1 | 2 | 4 | (modAlpha?7:0));

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::NormalizedDifference(const CISImageEx &target, CISImageEx &out8) const
{
    if (!ISValid() || !target.ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (!SameSize(target))
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth()==1) 
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (BitDepth()==8 && !Is8BitGray())
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    UINT res = out8.AllocBlank(Size(), 8);
    if (res != IS_ERR_OK) return res;
    out8.SetGrayPalette();

    _ISFn(is6_NormalizedDifferenceImage)((BYTE*)ImageDataConst(), (BYTE*)target.ImageDataConst(), Width(), Height(), BitDepth() / 8, RowStride(), out8.ImageData(), 0);

    return _ISFn(is6_GetLastError)();
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::SeamCarve(int seams, const CISImageEx &mask8, CISImageEx &out, UINT32 newSeamWeight)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    CSize newSize;
    newSize.cx = Width() + seams;
    newSize.cy = Height();

    if (newSize.cx < 1)
    {
        return IS_ERR_BADPARAM;
    }

    UINT res = out.AllocBlank(Size(), BitDepth());
    if (res != IS_ERR_OK) return res;

    if (!mask8.ISValid())
    {
        _ISFn(is6_SeamCarve)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            NULL, 0,
            out.ImageData(), out.RowStride(), seams, newSeamWeight, 0);
    }
    else
    {
        _ISFn(is6_SeamCarve)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), 
            (BYTE*)mask8.ImageDataConst(), mask8.RowStride(),
            out.ImageData(), out.RowStride(), seams, newSeamWeight, 0);
    }

    return _ISFn(is6_GetLastError)();
}

UINT32  CISImageEx::Retinex(double a, double b, double c, double d)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    // convert RGB to YCC 
    double * dblYCC = new double[Width() * Height() * 3];

    _ISFn(is6_RGBToYCC)(ImageDataConst(), Width(), Height(), RowStride(), dblYCC, Width() * 3, IS_IMGPROC_64_DOUBLE);
    
    // copy the Y channel to a temp buffer
    double * dblL = new double[Width() * Height()];
    _ISFn(is6_CopyImageChannel)(dblYCC, Width(), Height(), 3, Width() * 3, dblL, Width(), 0, IS_IMGPROC_64_DOUBLE);

    // perform the retinex oepration on the Y channel
    double * dblM = new double[Width() * Height()];

    UINT32 mode = 0;
    _ISFn(is6_RetinexImage)(dblL, Width(), Height(), Width(), dblM, Width(), mode, a, b, c, d, 0, 0, IS_IMGPROC_64_DOUBLE | 1);
    delete[] dblL;

    // copy the modified Y channel back to the YCC image
    _ISFn(is6_SetImageChannel)(dblM, Width(), Height(), Width(), dblYCC, 3, Width() * 3 , 0, IS_IMGPROC_64_DOUBLE);
    delete[] dblM;

    // YCC -> RGB
    _ISFn(is6_YCCToRGB)(dblYCC, Width(), Height(), Width()*3, ImageData(), RowStride(), IS_IMGPROC_64_DOUBLE);

    delete[] dblYCC;

    return IS_ERR_OK;
}

//////////////////////////////////////////////////////

UINT32  CISImageEx::AdjustSaturation(double level, bool bVibrance)
{
    if (!ISValid())
    {
        return IS_ERR_BADPARAM;
    }

    if (BitDepth() != 24 && BitDepth() != 32)
    {
        return IS_ERR_DEPTHMISMATCH;
    }

    if (bVibrance)
    {
       _ISFn(is6_Vibrance)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), level, 0);
    }
    else
    {
       _ISFn(is6_ModifyImageSaturation)(ImageData(), Width(), Height(), BitDepth() / 8, RowStride(), level);
    }

    return _ISFn(is6_GetLastError)();
}
