/************************************************************

class CISImageEx

For ImgSource v4.0 and higher !!!

This is a wrapper class for the ImgSource functions. This
only uses a very small portion of the total ImgSource function
list, but it should be simple enough to add whichever function
you need, by looking at similar functions here.

Sample use (load a BMP, reduce by 1/2, save as JPG) :

CISImageEx image;
image.LoadImage("C:\\input.bmp");

CSize csSize = image.Size();
csSize.cx/=2; csSize.cy/=2;
image.Resize(csSize);

image.SaveImage("C:\\output.jpg", eFmtJPG);


This class may be used without restriction.

ImgSource and CISImageEx are copyright 2002-2006, Smaller Animals Software.

All questions to : smallest@smalleranimals.com

************************************************************/

#ifndef CISIMAGEEXH
#define CISIMAGEEXH

#include <istream>
#include <vector>

///////////////////////////////////////////////////////////////////////

class CISImageEx
{

    ///////////////////////////////////////////////////////////////////////
    //
    // first, some constants
    //

public:
    typedef enum 
    {
        eDPINone = 1, 
        eDPIInch, 
        eDPIcm
    } eDPIUnit;

    enum 
    {
        eTiffCompNone = 1, 
        eTiffCompG2 = 2, 
        eTiffCompG3 = 3, 
        eTiffCompG4 = 4, 
        eTiffCompJPEG = 7, 
        eTiffCompPackbits = 32773, 
        eTiffCompDeflate = 32946
    };

    enum 
    {
        eFlipNone,
        eFlipHorizontal,
        eFlipVertical
    };

#define DEF_DPIU 1
#define DEF_DPIX 300
#define DEF_DPIY 300

    // default to the "base" (768 x 512) resolution (3)
#define DEF_PHOTOCD_RES       3

#define DEF_METAFILE_WIDTH    100
#define DEF_METAFILE_HEIGHT   100
#define DEF_METAFILE_BACKGROUND  RGB(128,128,128)  // gray

#define DEF_JPG_QUALITY       75    // decent quality

#define DEF_PNG_GAMMA         0.345

#pragma warning(disable : 4091)
    typedef enum eImgFmt
    {
        eFmtJpg,
        eFmtJpg8,
        eFmtBmp,
        eFmtBmp8,
        eFmtPcx,   
        eFmtPcx8,
        eFmtPng,
        eFmtPng32,
        eFmtPng24,
        eFmtPng8,
        eFmtTif,
        eFmtTif8,  
        eFmtTif24,  
        eFmtTif32,  
        eFmtTga,
        eFmtTga8,		
        eFmtTga32,		
        eFmtWmf,
        eFmtEmf,
        eFmtPsd,
        eFmtPsd8,
        eFmtWBmp,
        eFmtTif1,
        eFmtPng1,
        eFmtBmp1,
        eFmtPCD,
        eFmtGIF,
        eFmtNone,
    };

    ///////////////////////////////////////////////////////////////////////
    //
    // now some static functions
    //

    // guess file format from file extension
    static eImgFmt FmtFromExt(const TCHAR * fileName);

    // return a text version of the ImgSource error code
    static CString ErrorString(UINT32 uRes);

    // guess the image's file format 
    static eImgFmt  GuessFormat(const TCHAR *pFilename);

    // decide which format to save in, depending on image depth
    eImgFmt FindBestOutputFormat(eImgFmt eSaveFmt);

    // find the number of frames in this TIFF image
    static int      GetMultiFrameCount(const TCHAR *pFile);

    static UINT       OneBitRowStride(UINT32 w);

    static bool GetEXIFTagList(const TCHAR *pFile, CStringArray& list);
    static bool GetIPTCTagList(const TCHAR *pFile, CStringArray& list);

    ///////////////////////////////////////////////////////////////////////
    //
    // now some allocation and copying members
    //

    CISImageEx();
    virtual ~CISImageEx();

    // release all memory associated with this image
    virtual void	Clear();

    // simple validity test
    bool	ISValid() const	{if (m_bHGLOBAL) return (m_hData!=NULL); else return (m_pData!=NULL);}

    // you can use these to associate this with RGB, 8-bit or 1-bit image buffers
    UINT32	AssignBytes(BYTE *pData, UINT32 w, UINT32 h, UINT32 d, UINT32 uColors, RGBQUAD *pPal);
    UINT32	AssignHGLOBAL(HGLOBAL hData, UINT32 w, UINT32 h, UINT32 d, UINT32 uColors, RGBQUAD *pPal);

    // allocate a blank image of the appropriate size
    UINT32	AllocBlank(const CSize &cz, UINT32 d);
    UINT32	AllocBlank(UINT32 w, UINT32 h, UINT32 d);
    UINT32	AllocBlank(const CISImageEx &target);

    // associate this with the iamge data from src. renders src invalid.
    // this is a quick way to transfer one image to another (for instance, when src is a temp image)
    UINT32	Steal(CISImageEx &src);

    // make a duplicate of src
    UINT32  Copy(const CISImageEx &src);

    ///////////////////////////////////////////////////////////////////////
    //
    // loading and saving
    //

    // load a file-based image
    UINT	   LoadImage(const TCHAR * pFullPath, eImgFmt fmt = eFmtNone, UINT uTIFFFrameIdx = 0);

    // load a memory-based image
    UINT     LoadImage(BYTE *pMem, UINT32 uSize, eImgFmt fmt = eFmtNone, UINT32 uTiffIdx =0 );

    // save an image to a file
    UINT	   SaveImage(const TCHAR *csPath, eImgFmt fmt);

    UINT32  FromDC(HDC hDC, CRect rect);

    ///////////////////////////////////////////////////////////////////////
    //
    // access info after loading
    //

    // get various pieces of information about the image
    _inline CSize    Size() const      {return CSize(m_uWidth, m_uHeight);}
    _inline CSize    Pix() const       {return m_uWidth * m_uHeight;}
    _inline UINT32	Width() const		{return	m_uWidth;}
    _inline UINT32	Height() const		{return	m_uHeight;}
    _inline UINT32	BitDepth() const	{return m_uBitDepth;}
    _inline UINT32	PalColors() const	{return m_uPalColors;}
    _inline CString  GetFileName() const {return m_csFileName;}   // input file name
    _inline int      FrameIdx() const  {return m_iFrameIdx;}      // TIFF frame index

    UINT32	         RowStride() const ;

    CSize	         GetDPIInches() const;
    _inline CSize	   GetDPI() const		{return CSize(DPIX(), DPIY());}
    _inline double	fDPIX() const		{return m_fDPIX;}
    _inline double	fDPIY() const		{return m_fDPIY;}
    _inline UINT32	DPIX() const		{return (UINT32)m_fDPIX;}
    _inline UINT32	DPIY() const		{return (UINT32)m_fDPIY;}
    _inline eDPIUnit DPIU() const		{return m_eDPIUnits;} // DPI units

    _inline eImgFmt	SrcFormat() const	{return m_eSrcFmt;}

    _inline RGBQUAD * Palette()	{return m_pal;}
    _inline const RGBQUAD *ConstPalette() const	{return m_pal;}

    void    SetGrayPalette();
    bool    Is8BitGray() const;

    // how many bytes does the pixel data for this image require?
    UINT     GetDataSize(UINT w, UINT h, UINT bitdepth);

    // get a pointer to the pixel data
    BYTE *  ImageData() {if (m_bHGLOBAL) return (BYTE *)m_hData; else return m_pData;}
    const BYTE *  ImageDataConst() const  {if (m_bHGLOBAL) return (BYTE *)m_hData; else return m_pData;}

    bool    SameSize(const CISImageEx & b) const    {return Size() == b.Size() && RowStride()==b.RowStride() && BitDepth()==b.BitDepth();}

    ///////////////////////////////////////////////////////////////////////
    //
    // image modification
    //

    UINT32  Fill(COLORREF clr);

    // various bit depth conversions
    UINT32	Make8Bit(RGBQUAD *pPal = NULL, UINT32 uPalColors = 0);
    UINT32	Make8BitCopy(CISImageEx &out, RGBQUAD *pPal = NULL, UINT32 uPalColors = 0) const;

    UINT32  Make1Bit(UINT iTol, UINT uMode = 0);
    UINT32  Make1BitCopy(CISImageEx &out, UINT iTol, UINT uMode = 0) const;

    UINT32	Make8BitGray();
    UINT32	Make8BitGrayCopy(CISImageEx &out) const;

    UINT32	Make8BitBW(UINT32 uBWTolerance);
    UINT32  Make8BitBWCopy(CISImageEx &out, UINT32 uBWTolerance) const;

    UINT32	Make24Bit();
    UINT32	Make24BitCopy(CISImageEx &out) const;

    UINT32	Make32Bit();
    UINT32	Make32BitCopy(CISImageEx &out) const;

    UINT32  MakeCopy(CISImageEx &out, int targetDepth) const;

    // add a single color to the palette
    UINT32	AddPalColor(COLORREF clr);
    UINT32  SetPalColor(int idx, COLORREF clr);


    // return the size that most closely matches czPix without changing the current aspect ratio
    CSize   GetBestFitDims(CSize czPix);

    ///////////////////////////////////////////////////////////////////////
    //
    // Resizing
    //
    UINT32  Resize(const CSize &czPix, int iMethod = 0, bool bResize1BitToGrayscale = false);
    UINT32	ResizeCopy(const CISImageEx &src, const CSize &cz, UINT32 uType = 0, bool bResize1BitToGrayscale = false);
    UINT32	ResizeCopy(const CISImageEx &src, UINT32 w, UINT32 h, UINT32 uType, bool bResize1BitToGrayscale = false);

    UINT32  Crop(CRect cr);

    UINT32  SeamCarve(int seams /* < 0 = remove seams, > 0 = add seams*/, const CISImageEx &mask8 /*optional*/, CISImageEx &out, UINT32 newSeamWeight);

    // auto-detect a border and crop to it
    UINT32  AutoCrop();

    ///////////////////////////////////////////////////////////////////////
    //
    // Rotation / translation
    //

    // rotate 90 degrees
    UINT32  QuickRotate(int iDeg);
    UINT32  QuickRotateCopy(int iDeg, CISImageEx &out);

    // vertical / horizontal flip
    UINT32  Flip(int eFlip);

    // rotate an arbitrary number of degrees
    UINT32  Rotate(double radians, COLORREF backColor, bool bFast);

    // given a 1-bit image, find and correct for image rotation
    UINT32  Deskew(COLORREF backFillColor);

    UINT32  FindSkewAngle1Bit(double &angle);

    // if img_sub is contained within this image (ex. if it is a cropped version),
    // try to determine the location of img_sub within this image.
    UINT32  DetermineImageTranslation(CISImageEx &img_sub, CPoint & location, bool applyHammingWindowFirst);

    ///////////////////////////////////////////////////////////////////////
    //
    // Overlay and alpha blend
    //

    // draw another image on top of this
    UINT32  OverlayImage(const CISImageEx &src, int ixpos, int iypos, double fOpacity);

    // these are in order spec'd for is6_OverlayImage. don't change the order!
    typedef enum 
   {
      eBlendFast, eBlendSlow, eAnd, eOr, eXor, eMax, eMin, 
      eAdd, eSubBottomMinusTop, eIgnoreMin, eIgnoreMax, eAbsDiff,
      eMul, eMulFact, eScreen, eAddOffset, eSubOffset, eNAnd,
      eNOr, eXNor, eColorDodge, eColorBurn, eSoftDodge, eSoftBurn,
      ePSOverlay, eXFSoftLight, eHardLight, eReflect, eGlow,
      eFreeze, eHeat, eVividLight, eLinearLight, ePinLight, ePhoenix,
      eGrainExtract, eGrainMerge, eExclusion, ePSSubtract, eSubTopMinusBottom,eSolid
	} overlayMode;

    // use the blending modes in overlayMode to combine two images
    UINT32  Merge(const CISImageEx &src, int ixpos, int iypos, overlayMode=eBlendFast, double fOpacity=1.0, double param=0);

    // RGB onto RGB/RGBA using an 8-bit grayscale image as alpha mask
    UINT32  AlphaBlendRGBMask(const CISImageEx & srcRGB, const CISImageEx & mask8, CPoint pos);

    /* RGBA onto RGB/RGBA.      
    (if the bottom image has an alpha channel):

    0 : bottom alpha is unchanged
    1 : bottom alpha is set to top alpha
    2 : bottom alpha is the max of bottom and top alpha
    3 : bottom alpha is the sum of bottom and top alpha
    4 : alpha values are composited 
    5 : bottom alpha =  255 - ((255 - bottom_a)) * (255 - top_a)) / 255
    */
    typedef enum {bottomAlpha, topAlpha, maxAlpha, sumAlpha, blendAlpha, mixAlpha} alphaModMode;

    UINT32  AlphaBlendRGBA(const CISImageEx & srcRGBA, CPoint pos, alphaModMode mode);

    // solid color onto RGB/RGBA using an 8-bit grayscale image as alpha mask
    UINT32  AlphaBlendSolid(const CISImageEx & mask8, CPoint pos, COLORREF clr);

    // 8-bit with transparency info in the rgbReserved members of its palette onto RGB
    UINT32  AlphaBlend8Trans(const CISImageEx & src8Bit, CPoint pos);

    // given a rectangle, resize the image to fit, without changing the aspect ratio,
    // then place it on a solid background of color "clr", draw borders if required.
    UINT32  Matte(CSize matteSize, COLORREF clr, bool bBorder, COLORREF borderClr);

    UINT32  NormalizedDifference(const CISImageEx &target, CISImageEx &out8) const;

    ///////////////////////////////////////////////////////////////////////
    //
    // Filters
    //

    UINT32  Sharpen(UINT32 uLevel);

    UINT32  GaussianBlur(double sigma, bool modAlpha=false);

    UINT32  Blur(UINT32 uLevel);

    // demonstrates how to operate on a sub-region of an image
    UINT32  Sharpen(UINT32 uLevel, const CRect &r);
    UINT32  Blur(UINT32 uLevel, const CRect &r);

    // filtertype: 0 = Gaussian, 1 = linear, 2 = cosine, 3 = constant
    UINT32  BilateralFilter(UINT32 filterSize, UINT32 filterType, double sigma, bool fast=true, bool modAlpha=false);

    UINT32  UnsharpMask(UINT32 uThreshold, double sigma, double amount, bool modAlpha=false);
    UINT32  AdaptiveUnsharpMask(UINT32 uThreshold, double amount, int mode /*0=good,1=better,2=best*/, bool modAlpha=false);

    UINT32  Convolve(double *pKernelNxN, UINT32 N, bool modAlpha=false);

    UINT32  MotionBlur(double angle, double sigma, bool modAlpha=false);
    UINT32  RadialBlur(double angle, double sigma, CPoint center, bool modAlpha=false);
    UINT32  ZoomBlur(double amount, CPoint center, bool modAlpha=false);

    typedef enum { ArithMean, GeoMean, HarmonicMean, YpMean, Midpoint, Max, Min, Median} filterType;
    // for YpMean: param < 0 removes positive outliers, param > 0 removes negative outliers
    // for Median: param controls opacity of result (0..1)
    UINT32  Filter(filterType type, UINT32 filterSize, double param=1.0, bool modAlpha=false);

    UINT32  Threshold(UINT32 lowThreshold, UINT32 highThreshold, bool modAlpha=false);

    UINT32  DustAndScratches(UINT32 filterSize, UINT32 threshold, bool modAlpha=false);

    UINT32  WaveletDenoise(double fThreshold, double fCurve, bool modAlpha=false);
    UINT32  WaveletDenoise8BitGrayscale(double fThreshold, double fCurve);

    UINT32  Despeckle(bool modAlpha=false);

    // decomposes the image into a set of image, each with half the frequency of the previous
    // the first image contains the high-frequency details, the last contains the residual.
    // you must delete all the images in the vector, when you are done.

    // fills the vector with the sub-level images from the input 8 bit grayscale image
    UINT32  WaveletDecompose8BitGrayscale(UINT32 levelCount, std::vector<CISImageEx*> levels);

    // combines the sub-level images
    UINT32  WaveletRecompose8BitGrayscale(UINT32 levelCount, std::vector<CISImageEx*> levels);

    UINT32   Sobel(CISImageEx & out8, bool modAlpha=false);

    ///////////////////////////////////////////////////////////////////////
    //
    // brightness/contrast/saturation
    //
    UINT32  AdjustGamma(double fGamma);
    UINT32  AdjustBrightnessContrast(double brightness, double contrast, bool luminance=false, bool sigmoidal=false, double power=0);
    UINT32  AdjustColorBalance(int R, int G, int B,int Rshadow, int Gshadow, int Bshadow,int Rhighlight,int Ghighlight,int Bhighlight,int Density);
    UINT32  AutoBrightness(UINT32 lowThresh=5, UINT32 highThresh=250, bool stretch=true);

    UINT32  ContrastMask(double sigma, bool highContrast);

    UINT32  AdjustSaturation(double level, bool bVibrance);

    UINT32  Invert(bool noModAlpha);

    ///////////////////////////////////////////////////////////////////////
    //
    // general
    //

    UINT32  SplitToChannels(CISImageEx & r, CISImageEx & g, CISImageEx & b, CISImageEx & a) const ;
    UINT32  CombineChannels(const CISImageEx & r, const CISImageEx & g, const CISImageEx & b, const CISImageEx & a, bool bIs32Bit);
    UINT32  SetChannel(const CISImageEx & channel8, UINT32 channel);
    UINT32  SplitRGBAToRGBAAndA(CISImageEx & rgb, CISImageEx & a) const;

    // draw some text on the image
    UINT32  DrawTextOnImage(const TCHAR *pText, const char *pFontName, UINT32 uFontPixels, int iXPos, int iYPos, COLORREF clrText);

    // attempt to auto-correct the image in locations specified by the 8-bit mask image.
    // useful for correcting damage
    UINT32  InpaintImage(CISImageEx &mask8, UINT32 sigma, bool better=false, bool noModAlpha=true);

    UINT32  CannyEdgeDetector(double sigma, double low, double high, CISImageEx & edges, bool fast=false);

    ///////////////////////////////////////////////////////////////////////
    //
    // noise
    //
    typedef enum {Gaussian, Rayleigh, NegativeExponential, Uniform} noiseType;
    UINT32  AddNoise(noiseType type, double variance, double mean, bool monochrome, bool modAlpha=false);
    UINT32  PerlinNoise(UINT32 width, UINT32 height, double scale);

    ///////////////////////////////////////////////////////////////////////
    //
    // Histogram functions
    //
    UINT32  GetBrightnessHistogram(UINT32 histo[256]);
    UINT32  GetChannelHistograms(UINT32 **pHistos);
    UINT32  Get1BitHistogram(UINT32 histo[2]);

    UINT32  HistogramStretch(double fLowLimit, double fHiLimit, UINT32 midPoint=0, bool useMidpoint=false, bool modAlpha=false);
    UINT32  BrightnessHistogramStretch(double fLowLimit, double fHiLimit, UINT32 midPoint=0, bool useHSL=false, bool useMidpoint=false);

    UINT32  HistogramEqualize(BYTE lowThreshold=0, BYTE highThreshold=255, bool modAlpha=false);
    UINT32  BrightnessHistogramEqualize(BYTE lowThreshold=0, BYTE highThreshold=255);

    UINT32  HistogramSpecification(UINT32 histo[256], bool modAlpha=false);

    UINT32  Retinex(double a=40, double b=2, double c=20, double d=1.02);

    ///////////////////////////////////////////////////////////////////////
    //
    // LUT functions
    //
    UINT32  AplyLUTsToImage(BYTE **pLUTs);
    UINT32  GetStretchedHistogramLUT(UINT32 histo[256], BYTE LUT[256], double fLowLimit, double fHiLimit, UINT32 midPoint=0, bool useMidpoint=false);

    ///////////////////////////////////////////////////////////////////////
    //
    // effects
    //
    UINT32  Duotone(COLORREF clr);
    UINT32  Posterize();
    UINT32  Halftone(UINT32 uMaxDotSize, bool bSmoothDots);
    UINT32  Mosaic(UINT32 uSquareSize);
    UINT32  Solarize();
    UINT32  AdjustTint(int tint /* +/- 180 */);
    UINT32  Crackle(int mode, UINT32 uDefn, bool highContrast);
    UINT32  Gauzy(double blur, double sharp, double clarity, UINT32 mode);
    UINT32  BGR();

    UINT32  InnerGlow(UINT32 size, double opacity, COLORREF glowClr);
    UINT32  OuterGlow(UINT32 size, double opacity, COLORREF glowClr);
    UINT32  DropShadow(int offsetMax, int diffusion, COLORREF clrBackground, COLORREF clrShadow, int angle, bool useAlphaAsShadow, bool modAlphaOnly );

    ///////////////////////////////////////////////////////////////////////
    //
    // Morphological functions
    //

    // 1 bit or 8-bit grayscale only
    typedef enum {Erosion, Dilation, Open, Close} morphoPrimitiveMode;
    UINT32  Morpho(morphoPrimitiveMode mode, const int *maskNxN, UINT32 N);

    // grayscale only
    UINT32  TopHat(const int *maskNxN, UINT32 N, bool blackHat=false, bool addKernelValues=false);
    UINT32  Gradient(const int *maskNxN, UINT32 N, bool addKernelValues=false);

    // 1-bit only
    UINT32  HitOrMiss(const int *AmaskNxN, UINT32 AN, const int *BmaskNxN, UINT32 BN);
    UINT32  BoundaryExtraction(const int *maskNxN, UINT32 N);
    UINT32  Thinning(UINT32 maxIterations);

    // images must have identical bit depths, dimensions and rowStrides
    UINT32  Intersect(const CISImageEx &img);
    UINT32  Union(const CISImageEx &img);
    UINT32  XOR(const CISImageEx &img);

    UINT32  Inverse();

    ///////////////////////////////////////////////////////////////////////
    //
    // graphics
    //
    UINT32  FillSolidRect(CRect rect, COLORREF clr, double opacity);
    UINT32  FloodFill(CPoint ptStart, COLORREF clr, UINT32 tolerance);
    UINT32  GetFillMask(CPoint ptStart, COLORREF clr, UINT32 tolerance, CISImageEx &mask8) const;
    UINT32  PolygonFill(COLORREF clr, double opacity, std::vector<XYdouble> & points, bool smooth);
    UINT32  Ellipse(CPoint center, UINT32 arad, UINT32 brad, COLORREF clr, bool fill, double opacity=1, bool smooth=true);

    typedef enum {noArrow, arrow, barbedArrow, square, diamond, notched} lineEndStyle;
    UINT32  Line(CPoint a, CPoint b, COLORREF clr, double opacity, bool smooth, UINT32 thickness, lineEndStyle end1, lineEndStyle end2);


    ///////////////////////////////////////////////////////////////////////
    //
    // drawing and GDI functions
    //

    // draw or stretchdraw the image
    UINT32  Draw(const CDC *pDC, const CPoint &ptTopLeft, const HPALETTE hPal);
    UINT32  StretchDraw(const CDC *pDC, const CRect & rect);

    // copy this to a CBitmap
    UINT32  CopyToCBitmap(CBitmap &cBmp, HPALETTE hPal, COLORREF alphaBackgroundColor=RGB(255,255,255));

    // copy this from a CBitmap
    UINT32  CopyFromCBitmap(HBITMAP hBmp);

    BYTE *   ToDIB();
    UINT32  FromDIB(BITMAPINFOHEADER*, BYTE defAlpha=255);

    // copy the palette
    UINT32  GetPalette(RGBQUAD *pPal);

    // create a CPalette from the image's palette
    UINT32  MakeImagePalette(CPalette &pal);

    ///////////////////////////////////////////////////

protected:
    // various internal helper functions

    static eImgFmt  GuessFormat(HISSRC hSrc, const TCHAR *pFilename = NULL);

    UINT32	LoadFile(HISSRC hSource, eImgFmt eFmt, UINT uTIFFFrameIdx);

    UINT32	SaveFile(HISDEST hDest, eImgFmt fmt);
    UINT32  Save32Bit(HISDEST hDest, eImgFmt fmt);
    UINT32  Save24Bit(HISDEST hDest, eImgFmt fmt);
    UINT32  Save8Bit(HISDEST hDest, eImgFmt fmt);
    UINT32  Save1Bit(HISDEST hDest, eImgFmt fmt);

    void	   GetDPI(HISSRC hSource, eImgFmt eFmt);
    void	   SetDPI(eImgFmt fmt, IS4DPIStruct &dpi);

    void	   CalcDPI(eImgFmt eFmt, UINT32 dx, UINT32 dy, UINT32 u);

    UINT32	CopyAttribs(const CISImageEx& src);
    UINT32  CopyMetaData(const CISImageEx& src);

    ///////////////////////////////////////////////////

protected:
    double	m_fDPIX;
    double	m_fDPIY;
    eDPIUnit m_eDPIUnits;

    UINT32	m_uWidth, m_uHeight;
    UINT32  m_uBitDepth;

    UINT32	m_uPalColors;
    RGBQUAD	m_pal[256];

    eImgFmt	m_eSrcFmt;
    CString  m_csFileName;
    int      m_iFrameIdx;

    bool    m_bHGLOBAL;

    BYTE *  m_pData;
    HGLOBAL m_hData; 

    UINT32     m_uBWTolerance;

    UINT32     m_JPGQuality;
    bool        m_bProgressiveJPG;
    IS4DPIStruct m_outDPI;
    UINT32     m_uTiffComp;
    bool        m_bBmp8RLE;

};

////////////////////////////////////////////////////////

#endif // CISIMAGEEXH
