/*---------------------------------------------------------
   ISFilesTIFFExt.h : part of the ImgSource library.
   
   Specialized TIFF and GeoTIFF read/write functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is6_TIFF_EXTH
#define is6_TIFF_EXTH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {               
#endif

/*---------------------------------------------------------
   is6_CreateTIFFReader_TR / _is6_CreateTIFFReader_TR

   Purpose : 
   Create an ImgSource TIFF reading object. This allows
   you to read images, read tags and perform other operations on 
   a TIFF file without having to re-read the TIFF directory
   structure, as you would with the normal (non is6_*_TR) TIFF
   functions. This allows greater flexibility and performance.

   Example:
       HISTIFFREADER hTIFFR = is6_CreateTIFFReader_TR(hSrc, 0, 0);

       char *pStr = NULL;
       UINT32 wid;

       // get image description
       is6_GetTIFFTag_TR(hTIFFR, 270, 0, &pStr);

       // get image width
       is6_GetTIFFTag_TR(hTIFFR, 256, 0, &wid);

       // get RichIPTC data (count and pointer to data)
       UINT32 count;
       BYTE *pRichIPTC=NULL;
       is6_GetTIFFTag_TR(hTIFFR, 33723, 0, &count, &pRichIPTC);

       // read image 
       UINT32 w1,h1;
       HGLOBAL hImg1 = is6_ReadTIFF_TR(hTIFFR, &w, &h, 24, 0, 0, 0);

       // change to page 2
       is6_SetTIFFPage_TR(hTIFFR, 1);

       // read it
       UINT32 w2,h2;
       HGLOBAL hImg2 = is6_ReadTIFF_TR(hTIFFR, &w, &h, 24, 0, 0, 0);

       // read a single strip.
       // note that we don't have to call is6_Seek, or reopen the file, or use 
       // the TIFF tile/strip reader to do this.
       UINT32 w2s,h2s;
       HGLOBAL hImgStrip = is6_ReadTIFFTileOrStrip_TR(hTIFFR, &w2s, &h2s, 24, 0, 0, 0, 1 << 3);

       is6_DestroyTIFFReader_TR(hTIFFR, 0);

       is6_CloseSource(hSrc);

   Note:
   You don't need to use is6_Seek to rewind the file pointer, when using this.
   
   Param             Use
   ------------------------------------
   hSource           valid source object. this must remain
                     open until you call is6_DestroyTIFFReader_TR.

   uFrameIdx         initial frame/page. you can change the 
                     frame with is6_SetTIFFPage_TR.

   uFlags            unused

   Return
   ------
   HISTIFFREADER TIFF reader object. Use is6_DestroyTIFFReader_TR 
   when finished with this object.
---------------------------------------------------------*/
_ISDeclSpec HISTIFFREADER      _ISCConv _ISFn( is6_CreateTIFFReader_TR )(HISSRC hSource, UINT32 uFrameIdx, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadTIFF_TR / _is6_ReadTIFF_TR

   Purpose : 
   Read a single page from a TIFF file using a TIFF reader object.

   For single page TIFFs, use page index 0.

   Note:
    If reading to 24-bit RGB or 32-bit RGBA, this function will 
    automatically convert the input image data to 24-bit (or 32-bit) data.
    
    For other bit depths, no conversion is done, and you must read
    the TIFF to its native bit depth. For example: to get 1-bit data, you 
    must read a 1-bit TIFF; to get 8-bit data, you need an 8-bit TIFF; to 
    get 16-bit data, you need a 16-bit TIFF, etc.. 

   Note:
   Unless reading to 24 or 32-bits, if the file bit depth is not equal 
    to the requested bit depth, this function will fail with 
    IS_ERR_DEPTHMISMATCH.


   Param             Use
   ------------------------------------
   hReader           valid TIFF reader object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bit depth
                     1 - monochrome. image lines will be ((w + 7) / 8) bytes wide.
                     8 - colormapped
                     16 - read 16-bit grayscale (single 16-bit value per pixel)
                     24 - RGB 
                     32 - RGBA                                
                     48 - RGB with 16 bits per component
                     64 - RGBA with 16 bits per component
                     
   uFrameIndex       page number to read

   pPal              receives image palette

   uFlags            bit      purpose
                     ---      -------
                      0       ignore image "orientation" flag.
                              this can greatly speed up image reading, 
                              though the images may be rotated/flipped
                              differently than intended.

                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored. not applicable for 16-bit images.

                      2       return CMYK images as CMYK, instead of 
                              converting to RGBA

                      3       vertical flip 1-bit input images 

                      4       vertical flip 16-bit input images 
                              (overrides orientation correction)

                      5       DWORD-align 16-bit input images 
                              (overrides orientation correction)

                             6       swap red and blue in 24, 32, 48 and 64-bit images

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL            _ISCConv _ISFn( is6_ReadTIFF_TR )(HISTIFFREADER hReader, UINT32 *pWidth, UINT32 *pHeight, UINT32 uBitsPerPixel, UINT32 uFrameIdx, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadTIFFTileOrStrip_TR / _is6_ReadTIFFTileOrStrip_TR

   Purpose : 
   Read a single tile or strip from a TIFF file using a TIFF reader object. 

   To determine if a TIFF is stored as tiles or strips, use
   is6_GetTIFFTag_TR and test the TIFFTAG_TILEWIDTH and/or 
   TIFFTAG_TILELENGTH tags. If both are set, it's a tiled TIFF. 
   If none are set it's a stripped TIFF. If only one is set, 
   it's an error.

   Before you get a tile, you need to know the tile width 
   and height. Tiles are requested by the top-left corner of 
   the tile. Ex. if the tile dimensions are 64x64, you can get
   tiles at (0,0) , (64,0) , (128,0) , (64,0) , etc..

   Strips are requested by their index. Use TIFFTAG_ROWSPERSTRIP 
   to get the height of a strip. Strip #0 is at the top of the 
   image, strip #1 follows after TIFFTAG_ROWSPERSTRIP rows, etc.. 

   Note:
   This function DOES NOT correct for image orientation. Tiles are
   strips are returned as they are stored in the file. If you
   need to correct for orientation, you will need to do this manually.

   Note:
   Tiles along the right and bottom edges (and strips along the 
   bottom) will likely contain padding unless the image is a perfect
   multiple of the tile/strip size. This padding will be filled 
   with zeros. 

   Note:
   If the file bit depth is greater than the requested 
   bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hReader           valid TIFF reader object
   pWidth            receives image width (in pixels)
   pHeight           receives image height (in pixels)

   uBitsPerPixel     desired bit depth
                     1 - monochrome. image lines will be ((w + 7) / 8) bytes wide.
                     8 - colormapped        
                     16 - 16-bit grayscale (pixels are returned as 16-bit values)
                     24 - RGB 
                     32 - RGBA 
                     48 - RGB with 16 bits per component
                     64 - RGBA with 16 bits per component

   pPal              receives image palette

   xposOrStripIndex  tile left "x" coordinate, or strip index

   ypos              tile top "y" coordinate (ignored for strips)

   uFlags            bit      purpose
                     ---      -------
                      0       unused

                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored. not valid for 16, 48 or 64-bit images

                      2       return CMYK images as CMYK, instead of 
                              converting to RGBA

                      3       read strips, not tiles

                      4       read 16, 48 or 64-bit images to DWORD-aligned rows


   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL            _ISCConv _ISFn( is6_ReadTIFFTileOrStrip_TR )(HISTIFFREADER hReader, UINT32 *pWidth, UINT32 *pHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 xposOrStripIndex, UINT32 ypos, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetTIFFPageCount_TR / _is6_GetTIFFPageCount_TR

   Purpose : 
   Determine the number of frames/pages in the current TIFF file.

   Param             Use
   ------------------------------------
   hReader           valid TIFF reader object

   Return
   ------
   number of pages
---------------------------------------------------------*/
_ISDeclSpec UINT32             _ISCConv _ISFn( is6_GetTIFFPageCount_TR )(HISTIFFREADER hReader);

/*---------------------------------------------------------
   is6_GetTIFFTag_TR / _is6_GetTIFFTag_TR

   Purpose : 
   Read a TIFF tag.

   Note:
   This is a direct call to LibTiff's TIFFVGetField function.
   As the caller, you must know the number and type of all 
   parameter you pass in the variable length argument list.
   Passing the wrong number or type of parameters will almost
   certainly cause your program to crash.

   This uses C/C++'s variable argument list functionality. In C#, you
   can use the __arglist keyword to use this function. We do not
   know what is required to use it from other languages.

   This is the current set of tags LibTiff has built-in:
      Tag Name                            ID    Parameters
      ----------------------------------------------------
      TIFFTAG_SUBFILETYPE                 254   UINT32*
      TIFFTAG_IMAGEWIDTH                  256   UINT32*
      TIFFTAG_IMAGELENGTH                 257   UINT32*
      TIFFTAG_BITSPERSAMPLE               258   ISUINT16*
      TIFFTAG_COMPRESSION                 259   ISUINT16*
      TIFFTAG_PHOTOMETRIC                 262   ISUINT16*
      TIFFTAG_THRESHHOLDING               263   ISUINT16*
      TIFFTAG_FILLORDER                   266   ISUINT16*
      TIFFTAG_DOCUMENTNAME                269   char**
      TIFFTAG_ARTIST                      315   char**
      TIFFTAG_DATETIME                    306   char**
      TIFFTAG_HOSTCOMPUTER                316   char**
      TIFFTAG_IMAGEDESCRIPTION            270   char**
      TIFFTAG_MAKE                        271   char**
      TIFFTAG_MODEL                       272   char**
      TIFFTAG_COPYRIGHT                   273   char**
      TIFFTAG_ORIENTATION                 274   ISUINT16*
      TIFFTAG_SAMPLESPERPIXEL             277   ISUINT16*
      TIFFTAG_ROWSPERSTRIP                278   UINT32*
      TIFFTAG_MINSAMPLEVALUE              280   ISUINT16*
      TIFFTAG_MAXSAMPLEVALUE              281   ISUINT16*
      TIFFTAG_SMINSAMPLEVALUE             340   double*
      TIFFTAG_SMAXSAMPLEVALUE             341   double* 
      TIFFTAG_XRESOLUTION                 282   float*
      TIFFTAG_YRESOLUTION                 283   float*
      TIFFTAG_PLANARCONFIG                284   ISUINT16*
      TIFFTAG_XPOSITION                   286   float*
      TIFFTAG_YPOSITION                   287   float*
      TIFFTAG_PAGENAME                    285   char**
      TIFFTAG_RESOLUTIONUNIT              296   ISUINT16*
      TIFFTAG_PAGENUMBER                  297   ISUINT16*, ISUINT16*   
      TIFFTAG_HALFTONEHINTS               321   ISUINT16*, ISUINT16*   
      TIFFTAG_COLORMAP                    320   ISUINT16**, ISUINT16**, ISUINT16** (1 << bits per sample items in each array)
      TIFFTAG_MATTEING                    32995 ISUINT16*
      TIFFTAG_EXTRASAMPLES                338   ISUINT16*, ISUINT16**   
      TIFFTAG_TILEWIDTH                   322   UINT32*
      TIFFTAG_TILELENGTH                  323   UINT32*
      TIFFTAG_TILEDEPTH                   32998 UINT32*
      TIFFTAG_DATATYPE                    32996 ISUINT16*
      TIFFTAG_SAMPLEFORMAT                339   ISUINT16*   
      TIFFTAG_IMAGEDEPTH                  32997 UINT32*
      TIFFTAG_STONITS                     37439 double*
      TIFFTAG_PIXAR_IMAGEFULLWIDTH        33300 UINT32*
      TIFFTAG_PIXAR_IMAGEFULLLENGTH       33301 UINT32*
      TIFFTAG_PIXAR_TEXTUREFORMAT         33302 char**
      TIFFTAG_PIXAR_WRAPMODES             33303 char**
      TIFFTAG_PIXAR_FOVCOT                33304 float*
      TIFFTAG_PIXAR_MATRIX_WORLDTOSCREEN  33305 float**
      TIFFTAG_PIXAR_MATRIX_WORLDTOCAMERA  33306 float**
      TIFFTAG_SUBIFD                      330   ISUINT16*,UINT32*
      TIFFTAG_YCBCRCOEFFICIENTS           529   float**
      TIFFTAG_YCBCRPOSITIONING            531   ISUINT16*
      TIFFTAG_YCBCRSUBSAMPLING            530   ISUINT16*,ISUINT16*
      TIFFTAG_WHITEPOINT                  318   float**
      TIFFTAG_PRIMARYCHROMATICITIES       319   float**
      TIFFTAG_TRANSFERFUNCTION            301   ISUINT16** ( if samplesperpixel - extrasamples > 1: ISUINT16**, ISUINT16**, ISUINT16** )
      TIFFTAG_REFERENCEBLACKWHITE         532   float**
      TIFFTAG_INKSET                      332   ISUINT16*
      TIFFTAG_DOTRANGE                    336   ISUINT16*,ISUINT16*   
      TIFFTAG_INKNAMES                    333   char**
      TIFFTAG_NUMBEROFINKS                334   ISUINT16*
      TIFFTAG_TARGETPRINTER               337   char**
      TIFFTAG_ICCPROFILE                  34675 UINT32*, BYTE**   
      TIFFTAG_PHOTOSHOP                   34377 UINT32*, BYTE**
      TIFFTAG_RICHTIFFIPTC                33723 UINT32*, BYTE**
      TIFFTAG_XMLPACKET                   700   UINT32*, BYTE**
      TIFFTAG_STRIPOFFSETS                273   UINT32**
      TIFFTAG_TILEOFFSETS                 324   UINT32**
      TIFFTAG_STRIPBYTECOUNTS             279   UINT32**
      TIFFTAG_TILEBYTECOUNTS              325   UINT32**

   You can also read tags that aren't listed here, but parameter type
   and count will be determined by tag data type and count.

   Note:
   Where the parameter is a pointer to a pointer (ex: uint32**), you
   must pass the address of a pointer to the appropriate data type. Your
   pointer will be set to point to the image's tag data. You do not
   need to delete/free this memory. You should not modify the data.
   When the parameter is just a pointer, you must pass the address of
   a variable of the appropriate data type. 
   
   Note:
   There is NO WAY for this function to validate that you've passed the
   correct parameter types and count.

   Note:
   Please see the TIFF 6.0 specification, or a copy of the LibTiff's tiff.h
   for a list of what values are acceptable for each tag. 

   Param             Use
   ------------------------------------
   hReader           valid TIFF reader object
   uTiffTagID        TIFF tag ID
   uFlags            unused
   ...               variable length argument list

   Return
   ------
   FALSE on failure, or if tag not found
---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_GetTIFFTag_TR )(HISTIFFREADER hReader, UINT32 uTiffTagID, UINT32 uFlags, ...);

/*---------------------------------------------------------
    is6_GetTIFFTagV_TR / _is6_GetTIFFTagV_TR

    Same as is6_GetTIFFTag_TW, but accepts a varargs list.

---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_GetTIFFTagV_TR )(HISTIFFREADER hReader, UINT32 uTiffTagID, UINT32 uFlags, va_list ap);

/*---------------------------------------------------------
   is6_SetTIFFPage_TR / _is6_SetTIFFPage_TR

   Purpose : 
   Change to a different page/frame in the TIFF file. All
   subsequent image and tag reads will be performed on the 
   new page.

   Param             Use
   ------------------------------------
   hReader           valid TIFF reader object
   uFrameIdx         new frame index

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_SetTIFFPage_TR )(HISTIFFREADER hReader, UINT32 uFrameIdx);

/*---------------------------------------------------------
   is6_DestroyTIFFReader_TR / _is6_DestroyTIFFReader_TR

   Purpose : 
   Destroy the TIFF reader object.

   Param             Use
   ------------------------------------
   hReader           valid TIFF reader object
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_DestroyTIFFReader_TR )(HISTIFFREADER hReader, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CreateTIFFWriter_TW / _is6_CreateTIFFWriter_TW

   Purpose : 
   Create an ImgSource TIFF writing object. This allows
   you to write images, tags and perform other operations on 
   a TIFF file without having to re-read the TIFF directory
   structure, as you would with the normal (non is6_*_TW) TIFF
   functions. This allows greater flexibility and performance.
     
   After calling this, you need to call is6_InitializeTIFFWriter_TW
   to set up a new page/frame, before you write the image data or
   set tags.

   Example:
   HISDEST hDest = is6_OpenFileDest(filename, 0);
 
   HISTIFFWRITER hTIFF = is6_CreateTIFFWriter_TW(hDest, 0);
   
   // create the writer
   is6_InitializeTIFFWriter_TW(hTIFF, w1, h1, 1, pal, 0, 0, 0);

   // set the image description tag
   is6_SetTIFFTag_TW(hTIFF, 270, 0, "A lovely image!");

   // write a 1-bit image
   is6_WriteTIFF_TW(hTIFF,  p1, (w1+7)/8, 0);

   // start a new page by finishing this page. 
   // (this is not necessary if you are only writing a single page)
   is6_WriteDirectory_TW(hTIFF, 0);

   // initialize another image page
   is6_InitializeTIFFWriter_TW(hTIFF, w2, h2, 1, pal, 0, 0, 0);
   
   // set a different image description
   is6_SetTIFFTag_TW(hTIFF, 270, 0, "A Fantastic Image!!");

   is6_WriteTIFF_TW(hTIFF,  p2, (w2+7)/8, 0);

   // close it up.
   is6_DestroyTIFFWriter_TW(hTIFF, 0);

   is6_CloseDest(hDest);



   Param             Use
   ------------------------------------
   hDest             valid destination object. this must remain
                     open until you call is6_DestroyTIFFReader_TW.

   uFlags            bit         purpose
                     ---         -------
                      0          write image in append mode.
                                 hDest must have been created with 
                                 the "append" flag, and the file must 
                                 already exist.

   Return
   ------
   HISTIFFWRITER TIFF writer object. Use is6_DestroyTIFFWriter_TW 
   when finished with this object.
---------------------------------------------------------*/
_ISDeclSpec HISTIFFWRITER             _ISCConv _ISFn( is6_CreateTIFFWriter_TW )(HISDEST hDest, UINT32 uFlags);

/*---------------------------------------------------------
   is6_InitializeTIFFWriter_TW / _is6_InitializeTIFFWriter_TW

   Purpose : 
   Prepare a TIFF writer object for writing an image. This
   is the second step in writing a TIFF image with the image writer object.
   This is where you tell the object the bit depth, dimensions, color type,
   etc. of the image you're going to write. 

   All image writing will occur in the current TIFF directory (aka 'page').
   To write a multiple page TIFF, see is6_WriteDirectory_TW.

   Note:
   If uBitsPerPixel = 64, input image must be 64-bit RGBA or CMYK.
   If uBitsPerPixel = 48, input image must be 48-bit RGB.
   If uBitsPerPixel = 32, input image must be 32-bit RGBA or CMYK.
   If uBitsPerPixel = 24, input image must be 24-bit RGB.
   If uBitsPerPixel = 16, input image must be 16-bit grayscale (one 16-bit value per pixel)
   If uBitsPerPixel = 8, input image must be 8-bit.
   If uBitsPerPixel = 4, input image must be 4-bit, see notes below.
   If uBitsPerPixel = 1, input image must be 1-bit, see notes below.


   Note:
   For 1 bit images:
      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

      These rows must be (int)((uWidth + 7) / 8) bytes wide.

   For 4 bit images:
      Pixels must be packed 2 to a BYTE.

      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

      These rows must be (int)((uWidth + 1) / 2) bytes wide.

   Note:
      1-bit images do not use the palette. They must be arranged
      so that 0-pixels are white and 1-pixels are black (use
      uFlags to reverse this, see below).

    Note:
        For G2 (Huffman)/G3/G4 fax-compressed images, zero pixels
        are _always_ assumed to be white (1 = black). This is 
        regardless of what you set for uFlags bit #5.

   Note:
   See is6_SetTIFFOutputStripSize to control strip size.
   
   Note:
   You can control image orientation with is6_SetTIFFOrientation before this
   call, or by setting the orientation tag with is6_SetTIFFTag_TW after 
   this call. 
   
   is6_SetTIFFTag_TW can be used to override many tags, but it is 
   recommended that you do not override image dimension tags, bit depth tags 
   or other fundamental properties.

   Param             Use
   ------------------------------------
   hWriter           valid TIFF writing object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uBitsPerPixel     1 = write 1-bit image. see notes.
                     4 = write 4-bit image, pPal must be valid
                     8 = write 8-bit image, pPal must be valid
                     16 = write 16-bit grayscale image. 
                     24 = RGB image
                     32 = RGBA image
                     48 = RGB with 16 bits per component
                     64 = RGBA with 16 bits per component
   
   pPal              array of 256 RGBQUAD entries. the 
                     image palette.
   
   uCompMode         1 : no compression

                     2 : CCITT modified Huffman RLE (1-bit only. see notes)
                     3 : CCITT Group 3 fax encoding (1-bit only. see notes)
                     4 : CCITT Group 4 fax encoding (1-bit only. see notes)
                     5 : LZW
                     32771 : CCITT RLE / word aligned (1-bit only)

                     7 : JPEG. 24 or 8-bit only. if 8-bit, the image
                     must be 8-bit grayscale. See is6_SetTIFFJPGQuality.

                     8 : Deflate / ZIP (as used by Adobe)
                     
                     32773 : Macintosh PackBits

                     32946 :  Deflate / ZIP (same algorithm as 8, 
                     but with different compression ID)

                     32909 : Pixar companded 11-bit ZIP (8, 24 and 32-bit only)

   pDPIStruct        optional resolution info: (pass NULL if you don't care)

                     fDPIX      X pixel density
                     fDPIY      Y pixel density

                     uDPIUnits   1 - no units specified in file
                                 2 - dots / inch
                                 3 - dots / cm

   uFlags            bit         purpose
                     ---         -------
                      0          write image in append mode.
                                 hDest must have been created with 
                                 the "append" flag, and the file must 
                                 already exist.

                      1          write 4 and 8-bit image as grayscale (no palette).

                      2          input image is flipped vertically
                      3          input image is in BGR order (24 and 32 bpp only)

                      4          write 32-bit images as CMYK instead of RGBA

                      5          write 1-bit images where 0 pixels are black.
                                 default is 0=white. not applicable with G2/G3/G4
                                            compression.

                      6          enable TIFF predictor (only for LZW and Deflate 
                                 compression schemes. > 4 bits per pixel )

                      7          write tiled TIFF. see is6_SetTIFFOutputTileSize.



   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_InitializeTIFFWriter_TW )(HISTIFFWRITER hWriter, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, const RGBQUAD * pPal, UINT32 uCompMode, const IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteTIFF_TW / _is6_WriteTIFF_TW

   Purpose : 
   Write an image using a TIFF writer object.

   All image writing will occur in the current TIFF directory (aka 'page').
   To write a multi-page TIFF, see is6_WriteDirectory_TW.

   Note:
   Prior to calling this, you must call is6_InitializeTIFFWriter_TW to set up
   the image data for the page.

   Param             Use
   ------------------------------------
   hWriter           valid TIFF writer object
   pImage            image data. see is6_InitializeTIFFWriter_TW
   uRowStride        components per row of image data

  
   uFlags            bit         purpose
                     ---         -------
                      2          input image is flipped vertically
                      3          input image is in BGR order (24 and 32 bpp only)

   Return
   ------
   BYTEs written so far. the file may not be complete 
   until you call is6_DestroyTIFFWriter_TW.

---------------------------------------------------------*/
_ISDeclSpec UINT32             _ISCConv _ISFn( is6_WriteTIFF_TW )(HISTIFFWRITER hWriter, const BYTE *pImage, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_SetTIFFTag_TW / _is6_SetTIFFTag_TW

   Purpose : 
   Add a TIFF tag to be written to the next directory write
   (this happens after image data is written).

   Note:
   This is a direct call to LibTiff's TIFFVSetField function.
   As the caller, you must know the number and type of all 
   parameter you pass in the variable length argument list.
   Passing the wrong number or type of parameters will almost
   certainly cause your program to crash.

   This uses C/C++'s variable argument list functionality. In C#, you
   can use the __arglist keyword to use this function. We do not
   know what is required to use it from other languages.

   Note:
   ImgSource will set some of these tags as part of its normal operation
   when writing images (ex. TIFFTAG_IMAGEWIDTH, TIFFTAG_IMAGELENGTH, etc).
   Setting tags that describe the dimensions and configuration of image data
   to different value than ImgSource sets will probably cause serious errors
   in the files, and/or the operation of you application. Be careful.

   This is the current set of writeable tags LibTiff has built-in:

      Tag Name                            ID    Parameters
      ----------------------------------------------------
      TIFFTAG_SUBFILETYPE                 254   uint32
      TIFFTAG_IMAGEWIDTH                  256   uint32
      TIFFTAG_IMAGELENGTH                 257   uint32
      TIFFTAG_BITSPERSAMPLE               258   int
      TIFFTAG_COMPRESSION                 259   int
      TIFFTAG_PHOTOMETRIC                 262   int
      TIFFTAG_THRESHHOLDING               263   int
      TIFFTAG_FILLORDER                   266   int
      TIFFTAG_DOCUMENTNAME                269   char*
      TIFFTAG_ARTIST                      315   char*
      TIFFTAG_DATETIME                    306   char*
      TIFFTAG_HOSTCOMPUTER                316   char*
      TIFFTAG_IMAGEDESCRIPTION            270   char*
      TIFFTAG_MAKE                        271   char*
      TIFFTAG_MODEL                       272   char*
      TIFFTAG_COPYRIGHT                   273   char*
      TIFFTAG_ORIENTATION                 274   int
      TIFFTAG_SAMPLESPERPIXEL             277   int
      TIFFTAG_ROWSPERSTRIP                278   uint32
      TIFFTAG_MINSAMPLEVALUE              280   int
      TIFFTAG_MAXSAMPLEVALUE              281   int
      TIFFTAG_SMINSAMPLEVALUE             340   double
      TIFFTAG_SMAXSAMPLEVALUE             341   double
      TIFFTAG_XRESOLUTION                 282   double
      TIFFTAG_YRESOLUTION                 283   double
      TIFFTAG_PLANARCONFIG                284   int
      TIFFTAG_XPOSITION                   286   double
      TIFFTAG_YPOSITION                   287   double
      TIFFTAG_PAGENAME                    285   char*
      TIFFTAG_RESOLUTIONUNIT              296   int
      TIFFTAG_PAGENUMBER                  297   int, int
      TIFFTAG_HALFTONEHINTS               321   int, int
      TIFFTAG_COLORMAP                    320   uint16*, uint16*, uint16* (1 << bits per sample items in each array)
      TIFFTAG_EXTRASAMPLES:               32995 int, uint16*
      TIFFTAG_MATTEING                    338   int
      TIFFTAG_TILEWIDTH                   322   uint32
      TIFFTAG_TILELENGTH                  323   uint32
      TIFFTAG_TILEDEPTH                   32998 uint32
      TIFFTAG_DATATYPE                    32996 int
      TIFFTAG_SAMPLEFORMAT                339   int
      TIFFTAG_IMAGEDEPTH                  32997 uint32
      TIFFTAG_STONITS                     37439 double
      TIFFTAG_PIXAR_IMAGEFULLWIDTH        33300 uint32
      TIFFTAG_PIXAR_IMAGEFULLLENGTH       33301 uint32
      TIFFTAG_PIXAR_TEXTUREFORMAT         33302 char*
      TIFFTAG_PIXAR_WRAPMODES             33303 char*
      TIFFTAG_PIXAR_FOVCOT                33304 double
      TIFFTAG_PIXAR_MATRIX_WORLDTOSCREEN  33305 float*
      TIFFTAG_PIXAR_MATRIX_WORLDTOCAMERA  33306 float*
      TIFFTAG_SUBIFD                      330   int , uint32*
      TIFFTAG_YCBCRCOEFFICIENTS           529   float*
      TIFFTAG_YCBCRPOSITIONING            531   int
      TIFFTAG_YCBCRSUBSAMPLING            530   int, int
      TIFFTAG_WHITEPOINT                  318   float*
      TIFFTAG_PRIMARYCHROMATICITIES       319   float*
      TIFFTAG_TRANSFERFUNCTION            301   uint16*
      TIFFTAG_REFERENCEBLACKWHITE         532   float*
      TIFFTAG_INKSET                      332   int
      TIFFTAG_DOTRANGE                    336   int, int
      TIFFTAG_INKNAMES                    333   int, char*
      TIFFTAG_NUMBEROFINKS                334   int
      TIFFTAG_TARGETPRINTER               337   char*
      TIFFTAG_ICCPROFILE                  34675 uint32, void*
      TIFFTAG_PHOTOSHOP                   34377 uint32,  void*
      TIFFTAG_RICHTIFFIPTC                33723 uint32, uint32*
      TIFFTAG_XMLPACKET                   700   uint32, void*

   
   Note:
   There is NO WAY for this function to validate that you've passed the
   correct parameter types and count.

   Note:
   Please see the TIFF 6.0 specification, or a copy of the LibTiff's tiff.h
   for a list of what values are acceptable for each tag. 

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_SetTIFFTag_TW )(HISTIFFWRITER hWriter, UINT32 uTagID, UINT32 uFlags, ...);

/*---------------------------------------------------------
    is6_SetTIFFTagV_TW / _is6_SetTIFFTagV_TW

    Same as is6_SetTIFFTag_TW, but accepts a varargs list.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_SetTIFFTagV_TW )(HISTIFFWRITER hTIFF, UINT32 uTagID, UINT32 uFlags, va_list ap);

/*---------------------------------------------------------
   is6_WriteDirectory_TW / _is6_WriteDirectory_TW

   Purpose : 
   Write the current page/frame directory and start a new one.

   This must be called between each page in a multi-page TIFF file.
   It is not necessary to call it after the final (or only) page 
   in a file.

   Param             Use
   ------------------------------------
   hWriter           valid TIFF writer object
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_WriteDirectory_TW )(HISTIFFWRITER hWriter, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DestroyTIFFWriter_TW / _is6_DestroyTIFFWriter_TW

   Purpose : 
   Destroy the TIFF writer object.

   Param             Use
   ------------------------------------
   hWriter           valid TIFF writer object
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL               _ISCConv _ISFn( is6_DestroyTIFFWriter_TW )(HISTIFFWRITER hTIFF, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GeoTIFFCreateReader / _is6_GeoTIFFCreateReader

   Purpose : 
   Create an ImgSource GeoTIFF data reading object. This
   call creates the GeoTIFF reader object from your TIFFReader object
   and parses the TIFF file for GeoTIFF tag data. To fetch the 
   individual tags from the data, use is6_GeoTIFFGetTag.

   Use is6_DestroyGEOTIFFReader to release the memory associated with
   this object.

   Note:
   For a list of GeoTiff tags, see the web site : 
      http://www.remotesensing.org/geotiff/spec/geotiffhome.html

   Note:
   This looks in the current TIFF page/directory for GeoTIFF data
   (see is6_SetTIFFPage_TR to change page).

   Param             Use
   ------------------------------------
   hTIFFReader       valid TIFF reader object
   uFlags            unused

   Return
   ------
   HISGEOTIFFREADER GetTIFF reader object. NULL on failure.
---------------------------------------------------------*/
_ISDeclSpec HISGEOTIFFREADER  _ISCConv _ISFn( is6_GeoTIFFCreateReader )(HISTIFFREADER hTIFFReader, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GeoTIFFGetTag / _is6_GeoTIFFGetTag

   Purpose : 
   Get the tag data for a single tag from an ImgSource
   GeoTIFF data reader object.

   All data is returned as an array of values.

   Note:
   For a list of tags and the values associated with them, see:
      http://www.remotesensing.org/geotiff/spec/geotiffhome.html

   Note: 
   phData will point to an array of values of the data type 
   used by the tag:

       is6_GeoTIFFTYPE_BYTE        data is an array of BYTEs 
       is6_GeoTIFFTYPE_ASCII        data is an array of ASCII-encoded, zero-terminated string
       is6_GeoTIFFTYPE_SHORT        data in an array of 16-bit unsigned integers 
       is6_GeoTIFFTYPE_LONG        data in an array of 32-bit unsigned integers
       is6_GeoTIFFTYPE_SBYTE        data in an array of BYTEs
       is6_GeoTIFFTYPE_UNKNOWN     data in an array of BYTEs
       is6_GeoTIFFTYPE_SSHORT        data in an array of 16-bit signed integers 
       is6_GeoTIFFTYPE_SLONG        data in an array of 32-bit signed integers
       is6_GeoTIFFTYPE_FLOAT        data in an array of floats (32-bit IEEE floating point)
       is6_GeoTIFFTYPE_DOUBLE        data in an array of doubles (64-bit IEEE floating point)

       is6_GeoTIFFTYPE_RATIONAL     data in an array of 64-bit signed fractions,
                                  where each fraction is composed of two 32-bit values;
                                  the first 32-bit integer is the numerator, and the second
                                  is the denominator. so, a single tag value is two 32-bit values.

   Param             Use
   ------------------------------------
   hGeoTIFFReader    GeoTIFF data reader object created by
                     is6_GeoTIFFCreateReader.

   uTagID            tag ID

   phData            receives a handle to an array of data. see Notes.
                     you must use GlobalFree to release this
                     memory when you are finished with it.

   puCount           receives the number of elements in the data array.

   puTagType         receives tag data type. this determines the format
                     of the data returned in phData

   uFlags            unused
   
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL              _ISCConv _ISFn( is6_GeoTIFFGetTag )(HISGEOTIFFREADER hGeoTIFFReader, UINT32 uTagID, HGLOBAL *phData, UINT32 *puDataCount, UINT32 *puDataType, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GeoTIFFTranslateCoordinates / _is6_GeoTIFFTranslateCoordinates

   Purpose : 
   Using an ImgSource GeoTIFF data reader object, translate
   coordinates between image and PCS space (and vice versa).

   Param                Use
   ------------------------------------
   hGeoTIFFReader       GeoTIFF data reader object created by
                        is6_GeoTIFFCreateReader.

   x                    pointer to x coordinate to translate
   y                    pointer to y coordinate to translate

   uFlags               Bit      

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL              _ISCConv _ISFn( is6_GeoTIFFTranslateCoordinates )(HISGEOTIFFREADER hGeoTIFFReader, double*x, double *y, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DestroyGEOTIFFReader / _is6_DestroyGEOTIFFReader

   Purpose : 
   Destroy an ImgSource GeoTIFF data reader object. This
   will free all memory associated with the GeoTIFF reader object.

   Param                Use
   ------------------------------------
   hGeoTIFFReader       GeoTIFF data reader object created by
                        is6_GeoTIFFCreateReader.

   uFlags               bit      purpose
                        ---      -------
                         0       if clear, translate image to PCS
                                 if set, translate PCS to image
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL              _ISCConv _ISFn( is6_DestroyGEOTIFFReader )(HISGEOTIFFREADER hGeoTIFFReader, UINT32 uFlags);

#ifdef __cplusplus
}
#endif

#endif