/*---------------------------------------------------------
   ISFilesWL.h : part of the ImgSource library.
   
   Single lines file output

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is6_FILESWLH
#define is6_FILESWLH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_InitWriteBMPLines / _is6_InitWriteBMPLines

   Purpose : 
   Set up for single line BMP file writing.
   Use with is6_WriteNextBMPLine and is6_DestroyWriteBMPLines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   * If you set uBitsPerPixel to 24, pImg must be 24-bit RGB. 

   * If 16, pImg must be in the "x555" format (see is6_RGB24ToRGB16)

   * If 8, pImg must be 8-bit colormapped and pPal must contain 
     a valid palette. 
   
   * If 4, pImg must be four bits per pixel with input 
     rows ((w + 1) / 2) bytes wide. 
   
   * If 1, the image data must be 1 bit per pixel, a row must 
     be ((w + 7) / 8) bytes wide.

   Param             Use
   ------------------------------------
   hDest             valid dest object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uBitsPerPixel     1, 4, 8, 16, 24, 32
   uColors           colors in pPal. 0, if uBitsPerPixel
                     is 16 or 24.

   pPal              palette

   pDPIStruct        optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density (dots / meter)
                     fDPIY      Y pixel density (dots / meter)
                     uDPIUnits: ignored (only dots / meter allowed)

   uFlags            bit         purpose
                     ---         -------
                      3          input image is in BGR order (24 and 32 bpp only)

   Return
   ------
   HCBMP - BMP single line write object
---------------------------------------------------------*/
   _ISDeclSpec HCBMP         _ISCConv _ISFn(is6_InitWriteBMPLines)(HISDEST hDest, UINT32 uWidth, __int32 iHeight, UINT32 uBitsPerPixel, UINT32 uColors, const RGBQUAD * pPal, const IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteNextBMPLine / _is6_WriteNextBMPLine

   Purpose : 
   Write a single BMP line

   Param             Use
   ------------------------------------
   hSingle           single line HCBMP object

   pRow              image data. See is6_InitWriteBMPLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
   _ISDeclSpec BOOL         _ISCConv _ISFn(is6_WriteNextBMPLine)(HCBMP hSingle, const BYTE *pRow);

/*---------------------------------------------------------
   is6_DestroyWriteBMPLines / _is6_DestroyWriteBMPLines

   Purpose : 
   Destroy the HCBMP object

   Param             Use
   ------------------------------------
   hSingle           single line BMP object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyWriteBMPLines )(HCBMP hSingle);

/*---------------------------------------------------------
   is6_InitWritePCXLines / _is6_InitWritePCXLines

   Purpose : 
   Set up for single line PCX file writing.
   Use with is6_WriteNextPCXLine and is6_DestroyWritePCXLines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid dest object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uBitsPerPixel     8 or 24
   pPal              256-color palette, if uBitsPerPixel = 8

   uFlags            bit         purpose
                     ---         -------
                      0          if set, set the 'grayscale' flag in the file header
                                 this applies to 1 and 8 bit images only

                      3          input image is in BGR order (24 bpp only)


   Return
   ------
   HCPCX - PCX single line write object
---------------------------------------------------------*/
_ISDeclSpec HCPCX         _ISCConv _ISFn(is6_InitWritePCXLines)(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, const RGBQUAD * pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteNextPCXLine / _is6_WriteNextPCXLine

   Purpose : 
   Write a single PCX line

   Param             Use
   ------------------------------------
   hSingle           single line HCPCX object

   pRow              image data. See is6_InitWritePCXLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_WriteNextPCXLine )(HCPCX hSingle, const BYTE *pRow);

/*---------------------------------------------------------
   is6_DestroyWritePCXLines / _is6_DestroyWritePCXLines

   Purpose : 
   Destroy the HCPCX object

   Param             Use
   ------------------------------------
   hSingle           single line PCX object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyWritePCXLines )(HCPCX hSingle);

/*---------------------------------------------------------
   is6_InitWriteTGALines / _is6_InitWriteTGALines

   Purpose : 
   Set up for single line TGA file writing.
   Use with is6_WriteNextTGALine and is6_DestroyWriteTGALines.

   If pPal==NULL, is6_WriteNextTGALine expects RGB lines
   else 8 bit lines

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid dest object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   pPal              palette

   uFlags            bit         purpose
                     ---         -------
                      3          input image is in BGR order (24 bpp only)

   Return
   ------
   HCTGA - TGA single line write object
---------------------------------------------------------*/
_ISDeclSpec HCTGA         _ISCConv _ISFn(is6_InitWriteTGALines)(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, const RGBQUAD * pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteNextTGALine / _is6_WriteNextTGALine

   Purpose : 
   Write a single TGA line

   Param             Use
   ------------------------------------
   hSingle           single line HCTGA object

   pRow              image data. See is6_InitWriteTGALines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_WriteNextTGALine)(HCTGA hSingle, const BYTE *pRow);

/*---------------------------------------------------------
   is6_DestroyWriteTGALines / _is6_DestroyWriteTGALines

   Purpose : 
   Destroy the HCTGA object

   Param             Use
   ------------------------------------
   hSingle           single line HCTGA object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyWriteTGALines )(HCTGA hSingle);

/*---------------------------------------------------------
   is6_InitWriteTIFFLines / _is6_InitWriteTIFFLines

   Purpose : 
   Set up for single line TIFF file writing.
   Use with is6_WriteNextTIFFLine and is6_DestroyWriteTIFFLines.

   If uBitsPerPixel = 32, is6_WriteNextTIFFLine expects RGBA lines
   If uBitsPerPixel = 24, is6_WriteNextTIFFLine expects RGB lines
   If uBitsPerPixel = 16, input image must be 16-bit grayscale (one 16-bit value per pixel)
   If uBitsPerPixel = 8, is6_WriteNextTIFFLine expects 8-bit lines
   If uBitsPerPixel = 4, is6_WriteNextTIFFLine expects 4-bit lines, see notes below.
   If uBitsPerPixel = 1, input image must be 1-bit, see notes below.

   Notes:

   For 4 bit images:
      Pixels must be packed 2 to a BYTE.

      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

      These rows must be (int)((uWidth + 1) / 2) bytes wide.

   For 1 bit images:
      Each pixel row must start on a byte boundary and it must
      be a whole number of bytes wide.

      These rows must be (int)((uWidth + 7) / 8) bytes wide.

      Zero is white, one is black. 

   Note: Single line input/output functions are not 
   available for unregistered users.

   See is6_SetTIFFOutputStripSize to control strip size.

   See is6_SetTIFFOrientation to control orientation.

   Parameters
   ----------
   See is6_WriteTIFF for parameter descriptions.
 
   Return
   ------
   HISTIFFWRITER - TIFF single line write object

---------------------------------------------------------*/
_ISDeclSpec HISTIFFWRITER         _ISCConv _ISFn(is6_InitWriteTIFFLines)(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, const RGBQUAD * pPal, UINT32 uCompMode, const IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteNextTIFFLine / _is6_WriteNextTIFFLine

   Purpose : 
   Write a single TIFF line. 
   
   Note:
   If you are using the TIFF predictor, this function will 
   modify the image data for this row. You should make a copy of
   the row and pass that to this function.

   Param             Use
   ------------------------------------
   hSingle           single line TIFF object

   pRow              image data. See is6_InitWriteTIFFLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_WriteNextTIFFLine)(HISTIFFWRITER hSingle, const BYTE *pRow);

/*---------------------------------------------------------
   is6_DestroyWriteTIFFLines / _is6_DestroyWriteTIFFLines

   Purpose : 
   Destroy the HISTIFFWRITER object

   Param               Use
   ------------------------------------
   hSingle            single line HISTIFFWRITER object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyWriteTIFFLines )(HISTIFFWRITER hSingle);

/*---------------------------------------------------------
   is6_InitWritePNGLines / _is6_InitWritePNGLines

   Purpose : 
   Set up for single line PNG file writing.
   Use with is6_WriteNextPNGLine and is6_DestroyWritePNGLines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   p8Bit             pointer to image
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   uWidthBytes       image row width, in bytes
   uBitDepth         bits per component

   uColors           colors in image. if between 1 and 256,
                     you must supply a valid palette at pPal.
                     if 16 bits or higher, 0 is acceptable here.

   pPal              array of up to 256 RGBQUAD entries. 
                     the image palette.

   uColorType        0 = gray, 2 = RGB, 3 = palettized,
                     4 = gray-alpha, 6 = RGBA

   fGamma            file gamma value. 0 if don't care.
                     note that this value is usually 1/screen_gamma.
                     ie. it's generally less than 1.

    pDPIStruct       optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density
                     fDPIY      Y pixel density

                     uDPIUnits   0 - no units specified in file
                                 1 - dots / meter

   uFlags            bit         purpose
                     ---         -------
                      1          set, if palette contains alpha info.
                                 the PNG writer will use the rgbReserved 
                                 entries of the input palette as alpha values 
                                 (0 = transparent, 255 = opaque).

                      3          input pixels are in BGR order

                      4          suppress automatic time-stamping

   Return
   ------
   BYTEs written
---------------------------------------------------------*/
_ISDeclSpec HCPNG         _ISCConv _ISFn(is6_InitWritePNGLines)(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uWidthBytes, UINT32 uBitDepth, UINT32 uColors, const RGBQUAD * pPal, UINT32 uColorType, double fGammaVal, const IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteNextPNGLine / _is6_WriteNextPNGLine

   Purpose : 
   Write a single PNG line

   Param               Use
   ------------------------------------
   hSingle            single line PNG object
   pRow               image data. See is6_InitWritePNGLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_WriteNextPNGLine)(HCPNG hSingle, const BYTE *pRow);

/*---------------------------------------------------------
   is6_DestroyWritePNGLines / _is6_DestroyWritePNGLines

   Purpose : 
   Destroy the HCPNG object

   Param               Use
   ------------------------------------
   hSingle            single line HCPNG object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyWritePNGLines )(HCPNG hSingle);

/*---------------------------------------------------------
   is6_InitWriteJPGLines / _is6_InitWriteJPGLines

   Purpose : 
   Set up for single line JPG file writing.
   Use with is6_WriteNextJPGLine and is6_DestroyWriteJPGLines.

   If bGrayscale = 0, is6_WriteNextJPGLine expects RGB lines
   else 8 bit lines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid destination object
   pImg              pointer to image data
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)

   uQuality          1<->100 . quality is the inverse of
                     compression.

   uBitsPerPixel     8 = input image is 8 bit grayscale
                     24 = input image is 24-bit RGB
                     32 - input is 32-bit CMYK or YCCK

   pDPIStruct        optional resolution info.
                     (pass NULL if you don't care)
                     fDPIX      X pixel density
                     fDPIY      Y pixel density
                     uDPIUnits   0 - no units specified in file
                                 1 - dots / inch
                                 2 - dots / cm

   uFlags            bit         purpose
                     ---         -------
                      0          write "progressive"

                      1          if set, write 32-bit as YCCK, else 
                                 write 32-bit as CMYK.

                      3          input image is in BGR order (24 bpp only)

                      4          if set, input image is 24-bit YUV/YCbCr

                      5          if set, do not write JFIF (APP0) header


   Note: 
   DPI info is not saved for CMYK JPGs.
   
   Return
   ------
   HCJPG - JPG single line write object
---------------------------------------------------------*/
_ISDeclSpec HCJPG         _ISCConv _ISFn(is6_InitWriteJPGLines)(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uQuality, UINT32 uBitsPerPixel, const IS4DPIStruct *pDPIStruct, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteNextJPGLine / _is6_WriteNextJPGLine

   Purpose : 
   Write a single JPG line

   Param             Use
   ------------------------------------
   hSingle           single line JPG object

   pRow              image data. See is6_InitWriteJPGLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_WriteNextJPGLine)(HCJPG hSingle, const BYTE *pRow);

/*---------------------------------------------------------
   is6_DestroyWriteJPGLines / _is6_DestroyWriteJPGLines

   Purpose : 
   Destroy the HCJPG object

   Param             Use
   ------------------------------------
   hSingle           single line HCJPG object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyWriteJPGLines )(HCJPG hSingle);

/*---------------------------------------------------------
   is6_InitWritePAXLines / _is6_InitWritePAXLines

   Purpose : 
   Set up for single line PAX file writing.
   Use with is6_WriteNextPAXLine and is6_DestroyWritePAXLines.

    Note:
    Use is6_SetPAXOutputInfo to set the PAX ID
    for each output file - before calling this 
    function!

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hDest             valid dest object
   uWidth            image width (in pixels)
   uHeight           image height (in pixels)
   pPal              palette of 256 RGBQUADs 

   uBitsPerPixel     BYTEs per pixel (1 or 3)
                        8 : format is 8-bit PAX. pPal
                        must point to a palette. lines must 
                        be written with is6_WriteNextPAXLine8Bit.

                        24 : format is 24-bit PAX. pPal
                        may be NULL. lines must be written 
                        with is6_WriteNextPAXLineRGB.

   pPassword         password for this file
   uEncType          PAX encryption type (1 - 6)

   uFlags            bit         purpose
                     ---         -------
                      3          input image is in BGR order (24 bpp only)


   Return
   ------
   HCPAX - PAX single line write object
---------------------------------------------------------*/
_ISDeclSpec HCPAX         _ISCConv _ISFn(is6_InitWritePAXLines)(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, const RGBQUAD * pPal, UINT32 uBitsPerPixel, const char *pPassword, UINT32 uEncType, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteNextPAXLine / _is6_WriteNextPAXLine

   Purpose : 
   Write a single PAX line

   Param             Use
   ------------------------------------
   hSingle           single line HCPAX object

   pRow              image data. See is6_InitWritePAXLines 
                     for image format notes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_WriteNextPAXLine)(HCPAX hSingle, const BYTE *pRow);

/*---------------------------------------------------------
   is6_DestroyWritePAXLines / _is6_DestroyWritePAXLines

   Purpose : 
   Destroy the HCPAX object

   Param             Use
   ------------------------------------
   hSingle           single line PAX object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyWritePAXLines )(HCPAX hSingle);

/*---------------------------------------------------------
   is6_InitWriteTLALines / _is6_InitWriteTLALines

   Purpose : 
   Set up for single line TLA file writing.
   Use with is6_WriteNextTLALine and is6_DestroyWriteTLALines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   See comments for is6_WriteTLA for an explanation of
   the parameters.

   Return
   ------
   HCTLA - TLA single line write object
---------------------------------------------------------*/
_ISDeclSpec HCTLA         _ISCConv _ISFn(is6_InitWriteTLALines)(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, const char *pPassword, UINT32 uBitsPerPixel, UINT32 uStripSize, UINT32 uFilterType, UINT32 uPalColors, const RGBQUAD * pPal, BYTE **pDataArray, UINT32 *dataSizeArray, UINT32 uTags, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WriteNextTLALine / _is6_WriteNextTLALine

   Purpose : 
   Write a single TLA line

   Param               Use
   ------------------------------------
   hSingle             single line HCTLA object

   pRow                image data. See comments for 
                       is6_WriteTLA for an explanation 
                       of the image data format.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_WriteNextTLALine)(HCTLA hSingle, const BYTE *pRow);

/*---------------------------------------------------------
   is6_DestroyWriteTLALines / _is6_DestroyWriteTLALines

   Purpose : 
   Destroy the HCTLA object

   Param               Use
   ------------------------------------
   hSingle                single line TLA object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyWriteTLALines )(HCTLA hSingle);

#ifdef __cplusplus
}
#endif

#endif
