/*---------------------------------------------------------
   ISImageEnhance.h : part of the ImgSource library.
   
   Image processing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is6_IMAGEHISTOH
#define is6_IMAGEHISTOH

// shut up, stupid compiler!
#pragma warning(disable:4786)

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_ToneMapping / _is6_ToneMapping

   Purpose : 
   Perform an automatic tone enhancement on an RGB-24 image.

   Param             Use
   ------------------------------------
   pInput            input image (24 bits)
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel must be 3
   uInRowStride      components per input pixel row
   pOut              output image (may be the same as pInput)
   uOutRowStride     components per output pixel row

   fLevel            lower values yield lower contrast.
                     start with 50. there's no upper limit

   sz                controls size of the internal guided blur.
                     larger values increase contrast. start 
                     with 100

   eps               controls blur depth on the internal guided
                     blur. higher values reduce detail in output.
                     start with 0.1. max = 1.
                
   uFlags            bit      meaning
                     ---      -------
                     0        if set, use floats for 
                              internal calcs, else use doubles.
                              floats take up half the space, but 
                              the calcs are somewhat slower

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv  _ISFn(is6_ToneMapping)(const BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, double fLevel, UINT sz, double eps, UINT32 uFlags);

/*---------------------------------------------------------
   is6_EnhanceDetails / _is6_EnhanceDetails

   Purpose : 
   Increase image details of a 24-bit RGB image by local 
   contrast adjustment.

   Param             Use
   ------------------------------------
   pInput            input image (24 bits)
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel must be 3
   uInRowStride      components per input pixel row
   pOut              output image (may be the same as pInput)
   uOutRowStride     components per output pixel row

   uFilterSize       larger values increase details (start with 20)
   fSigma            larger values increase contrast (start with 10)
                
   uFlags            unused                     

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_EnhanceDetails)(const BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, UINT32 uFilterSize, double fSigma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_AdjustGamma / _is6_AdjustGamma

   Purpose : 
   Perform gamma adjustment on an image.

   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, etc.) 
   
   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of BYTEs in a pixel.
   uRowStride        number of bytes in an image row
   fGamma            gamma correction factor
                
                     gamma must be >= 0.0
                     gamma < 1 , decrease brightness
                     gamma = 1 , no change
                     gamma > 1 , increase brightness

   uFlags            bit      meaning
                     ---      -------
                     0        if set, the fourth channel in 
                              a 32 bit image is modified. it is 
                              ignored. by default

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_AdjustGamma )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_AdjustBrightnessContrast / _is6_AdjustBrightnessContrast

   Purpose : 
   Modify the brightness and/or contrast of an image.

   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, etc.) 
   
   Curve shapes:
   The uFlags parameter can select between a choice of 
   curve shapes. The default is a five point curve; this 
   is the traditional ImgSource adjustment. 
   
   The seven point curve forces pixels in the extreme dark 
   and light ranges to black or white faster than the five 
   point curve.

   The three point curve emphasizes mid-tones more than the
   five or seven point curves. This is closest to the 
   brightness / contrast that Photoshop uses.

   The smooth curve gives much nicer results: fewer blown-out
    pixels on the bright end and much less solid black.

   Applying the adjustment to the Luminance channel only (bit 8)
   reduces the chromatic shifts you can sometimes see if the
   adjustment is applied to the RGB channels separately.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel in the image (1 to 4)
   uRowStride        byte per pixel row
   fBrightness       brightness adjustment (-1..+1)
   fContrast         contrast adjustment (-1..+1)

   fContrastPower    controls the shape of the
                     spline contrast curve. 
                     0.1 ... 0.9
                     0.5 is a good default

   uMode             0        use an S-shaped spline
                     1        use 3 point 'curve'
                     2        use 5 point 'curve'
                     3        use 7 point 'curve'
                     
   uFlags            bit      meaning
                     ---      -------
                     0        ignore 4th (alpha) channel

                     8        apply changes to Luminance only.
                              alpha channel is ignored.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_AdjustBrightnessContrast)(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fBrightness, double fContrast, double fSplinePower, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ModifyImageSaturation / _is6_ModifyImageSaturation

   Purpose : 
   Adjust the saturation levels in an RGB or RGBA image.
   Will also work on BGR/BGRA.

   Param             Use
   ------------------------------------
   pInput            input image (24 or 32 bits)
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    3 or 4. if 4, the alpha channel is not modified.
   uRowStride        bytes per pixel row
   fSatLevel         saturation level (any value is valid)

                     guide:
                     1 = no change
                     0 = fully desaturated
                     < 0, image starts to resaturate, but photonegative
                     > 1, image gains saturation
                     
                     there are no limits on this value, though the useful 
                     results start to taper off around +/- 10.
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ModifyImageSaturation )(BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double S);

/*---------------------------------------------------------
   is6_Vibrance / _is6_Vibrance

   Purpose : 
   Adjust the saturation levels in an RGB or RGBA image such
   that areas with low or high saturation are modified less than
   areas with high saturation. 

   This is similar to, but not exactly the same as, the "Vibrance"
   control in Adobe products.

   Will also work on BGR/BGRA.

   Param             Use
   ------------------------------------
   pInput            input image (24 or 32 bits)
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel 3 or 4. if 4, the alpha channel is not modified.
   uRowStride        components per pixel row

   fLevel            saturation level (any value is valid)

                     guide:
                     0 = no change
                     < 0, desaturate
                     > 1, increase saturation
                
   uFlags            unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv  _ISFn(is6_Vibrance)(const void * pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, double fLevel, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ApplyLUTsToImage / _is6_ApplyLUTsToImage

   Purpose : 
   Apply LUTs (look-up-tables) to pixels in an image.

   For each channel in each pixel the following function is
   performed :
      
      v = pImage[pixel][channel]
      pImage[pixel][channel] = LUT[v][channel]

   Ex.
   BYTE LUT[256];
   BYTE *LUTArray[1];
   LUTArray[0] = LUT;
   is6_ApplyLUTsToImage(pImage, w, h, 1, w, LUTArray, 0);

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   
   uChannelsPerPixel number of channels per pixel.
                     this also determines the number of LUTs 
                     in the pLUTs array.

   uRowStride        number of pixel components in an image row

   pLUTs             if image is 8 bits per component, this is an 
                     array of arrays of 256 BYTE values.

                     if image is 16 bits per component, this is an 
                     array of arrays of 65536 ISUINT16 values.

                     pLUTs[0] is the LUT used for the first channel,
                     pLUTs[1] is the LUT used for the second channel,
                     etc..
                     if you don't want to apply a LUT to a particular 
                     channel, put a NULL in the corresponding 
                     pLUTs place. 

                     pLUTs must contain uChannelsPerPixel elements!

   uFlags            bit      meaning
                     ---      -------
                      9       image is 16 bits per component
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_ApplyLUTsToImage)(void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, const void **pLUTs, UINT32 uFlags);

// experimental
_ISDeclSpec BOOL      _ISCConv _ISFn(is6_ApplyBrightnessLUT)(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, const BYTE *pLUT);

/*---------------------------------------------------------
    is6_AutoBrightnessRGB / _is6_AutoBrightnessRGB

    Purpose : 
    Automagically modify the brightness of an RGB image.

    Param                   Use
    ------------------------------------
    pRGB                    source RGB image
    uWidth                  width in pixels
    uHeight                 height
    uRowStride              number of bytes in an image row
    uLowThreshold           pixels below this level are set to black
    uHighThreshold          pixels above this level are set to white

    uMode                   0 : stretch brightness values to fit 
                            the specified range (histogram stretch)

                            1 : evenly distribute brightness values
                            across the specified range (histogram
                            equalization)
    Return
    ------
    FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_AutoBrightnessRGB )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uLowThreshold, UINT32 uHighThreshold, UINT32 uMode);

/*---------------------------------------------------------
   is6_GetBrightnessHistogram / _is6_GetBrightnessHistogram

   Purpose : 
   Find the brightness histogram of an image. Should not
   be used on images larger than 2 ^ 32 pixels.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel 3 = RGB (or BGR), 4 = RGBA (or BGRA)
   uRowStride        number of pixel components in a pixel row
   pHisto            array of 255 UINT32s (8 bpc) or 65536 UINT32s (16 bpc)

   uFlags            bit      meaning
                     ---      -------
                     0        set when image is in BGR (or BGRA) order                     

					 9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL          _ISCConv _ISFn(is6_GetBrightnessHistogram)(const void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 *pHisto, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetImageChannelHistograms / _is6_GetImageChannelHistograms

   Purpose : 
   Find the histograms for each channel in an image.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   
   uChannelsPerPixel number of channels in image. 
                     also number of images in pHisto.
   
   uRowStride        number of pixel components in a pixel row


   pHisto            array of pointers to arrays of 255 UINT32s each (8bpc) or 65535 UINT32s (16bpc).
                     the histogram for the first channel goes into pHisto[0]
                     the histogram for the second channel goes into pHisto[1]
                     etc..

                     if you don't want the histogram for a given channel, put a NULL
                     in the appropriate spot in pHisto

   uFlags            bit      meaning
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_GetImageChannelHistograms)(const void *pImage, UINT32 nWidth, UINT32 nHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, UINT32 **pHisto, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetImageChannelHistogramsDouble / _is6_GetImageChannelHistogramsDouble

   Purpose : 
   Find the histograms for each channel in an image made of doubles.

   Notes:

   Mode 1:
   
        1. find the minimum and maximum value used in the image for each pixel channel, 

        2. for each pixel in a channel, perform the following calculation:

                     pixel - min
                 v = -----------  x (uHistoSize - 1)
                      min - max

        3. increase pHisto[channel][v]

        In other words, the range of pixel values in the image is sub-divided 
        into uHistoSize buckets.

        On exit, pfMinVal and pfMaxVal are set to the min and max pixel values, respectively.

   Mode 2:

        1. for each pixel in a channel, apply the following calculation:

                clip the pixel value to the range fMinVal..fMaxVal .

           Where pfMinVal and pfMaxVal are passed into the function by the caller.

        2. perform the following calculation:

                       pixel - pfMinVal
                 v = --------------------  x (uHistoSize - 1)
                      pfMinVal - pfMaxVal


        3. increase pHisto[channel][v]

        In other words, the range of pixel values supplied by the caller 
        is sub-divided into uHistoSize buckets. 

        Note that if you set fMinVal and fMaxVal to the actual min/max pixel value 
        of the input image, this mode is also identical to Mode 1.

   Mode 3:

        1. for each pixel in a channel where the pixel value is 
        in the range fMinVal..fMaxVal, apply the following calculation:

                       pixel - pfMinVal
                 v = --------------------  x (uHistoSize - 1)
                      pfMinVal - pfMaxVal

        Where pfMinVal and pfMaxVal are passed into the function by the caller.

        2. increase pHisto[channel][v]

        In other words, the range of pixel values supplied by the caller 
        is sub-divided into uHistoSize buckets, and any pixel outside that range
        is not counted in the calculation. 
        
        Note that if you set fMinVal and fMaxVal to the actual min/max pixel value 
        of the input image, this mode is also identical to Mode 1.

   Param             Use
   ------------------------------------
   pImage            source image, doubles
   uWidth            width in pixels
   uHeight           height
   
   uChannelsPerPixel number of channels in image. 
   
   uRowStride        number of pixel components in a pixel row

   uHistoSize         number of entries in the histogram

   pHisto            array of uChannelsPerPixel pointers to arrays of uHistoSize UINT32s.
                     the histogram for the first channel goes into pHisto[0]
                     the histogram for the second channel goes into pHisto[1]
                     etc..

                     if you don't want the histogram for a given channel, put a NULL
                     in the appropriate spot in pHisto

   pfMinVal          array of uHistoSize doubles. 
   
                     in mode 1, these receive the mininum pixel value for each channel.
                     in modes 2,3, these specify the low end of the range for each channel

   pfMaxVal          array of uHistoSize doubles. 
   
                     in mode 1, these receive the maximum pixel value for each channel.
                     in modes 2,3, these specify the high end of the range for each channel
  
   uFlags            bits        purpose
                     ---         -------
                     0,1         if both are 0, process in Mode 1 (uFlags = 0)
                                 if only bit 0 is set, process in Mode 2 (uFlags = 1)
                                 if both are set, process in Mode 3 (uFlags = 3)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_GetImageChannelHistogramsDouble)(const double *pImage, UINT32 nWidth, UINT32 nHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, UINT32 uHistoSize, UINT32 **pHisto, double *pfMinVal, double *pfMaxVal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetOneBitHistogram / _is6_GetOneBitHistogram

   Purpose : 
   Find the histogram for a one bit image.

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels
   uHeight           height
   
   uRowStride        number of bytes in a pixel row

   pHisto            array of 2 UINT32
                     the number of 0 bits goes into pHisto[0]
                     the number of 1 bits goes into pHisto[1]

   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn(is6_GetOneBitHistogram)(const BYTE *pImage, UINT32 nWidth, UINT32 nHeight, UINT32 uRowStride, UINT32 *pHisto, UINT32 uFlags);

/*---------------------------------------------------------
   is6_BrightnessHistogramStretchImage / _is6_BrightnessHistogramStretchImage

   Purpose : 
   Perform histogram stretch on an RGB or RGBA image
   The alpha channel is ignored. Also see 
   is6_BrightnessHistogramStretchImageInHSL.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    3 = RGB (or BGR), 4 = RGBA (or BGRA)
   uRowStride        number of bytes in a pixel row
   fLowLimit         percent of pixels to set to 0
   fHiLimit          percent of pixels to set to 255 
   uMidPoint         pixel value to set to middle intensity (1..254)

   uFlags            bit         purpose
                     ---         -------
                      8          if set, use mid-point to divide
                                 output range: 
                                 half the intensity values will 
                                 be scaled to 0...mid-point ,
                                 and the other half will be scaled
                                 mid-point...maxval .

                      9          do stretch in HSL colorspace. this
                                 will preserve colors better than 
                                 doing the stretch in the RGB space.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_BrightnessHistogramStretchImage )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fLowLimit, double fHighLimit, UINT32 uMidPoint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_SaturationHistogramStretchImage / _is6_SaturationHistogramStretchImage

   Purpose : 
   Perform a histogram stretch on the saturation values in
   an RGB or RGBA image (using an HSV conversion).

   The alpha channel is ignored. 

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    3 = RGB (or BGR), 4 = RGBA (or BGRA)
   uRowStride        number of bytes in a pixel row
   fLowLimit         percent of pixels to set to 0
   fHiLimit          percent of pixels to set to 255
   uMidPoint         pixel value to set to middle intensity (1..254)

   uFlags            bit         purpose
                     ---         -------
                      8          if set, use mid-point to divide
                                 output range: 
                                 half the intensity values will 
                                 be scaled to 0...mid-point ,
                                 and the other half will be scaled
                                 mid-point...maxval .

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_SaturationHistogramStretchImage )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, double fLowLimit, double fHighLimit, UINT32 uMidPoint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_HistogramStretchImage / _is6_HistogramStretchImage

   Purpose : 
   Perform histogram stretch on an image.

   Note:
   Color channels are modified independently of each other, 
   so some shifts in color may occur. 
   
   See is6_BrightnessHistogramStretchImage if you want to 
   stretch the brightness of the image.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel max 8
   uRowStride        number of pixel components in a pixel row
   fLowLimit         percent of pixels to set to 0
   fHiLimit          percent of pixels to set to max component value
   uMidPoint         pixel value to set to middle intensity (1..max-1)

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                      8       if set, use mid-point to divide
                              output range: 
                              half the intensity values will 
                              be scaled to 0...mid-point ,
                              and the other half will be scaled
                              mid-point...maxval .

                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_HistogramStretchImage )(void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, double fLowLimit, double fHighLimit, UINT32 uMidPoint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetStretchedHistogramLUT / _is6_GetStretchedHistogramLUT

    Given a histogram, generate the LUT (Look-Up Table) that would
    give a histogram "stretch", if applied to an image. 
    
    The following steps are equivalent to is6_HistogramStretchImage:
        
        1. is6_GetImageChannelHistograms
        2. For each channel in the image:
            
              is6_GetStretchedHistogramLUT
        3. is6_ApplyLUTsToImage

   Param             Use
   ------------------------------------
     pHisto                   histogram of 256 (or 65536, for 16-bit) values
   uTotalElements    number of items counted in the histogram (w*h, if from an image)
   fLowLimit         percent of pixels to set to 0
   fHiLimit          percent of pixels to set to max component value
   uMidPoint         pixel value to set to middle intensity (1..max-1)

    pLUT                     receives LUT of 256 (or 65536, for 16-bit) values,
                     representing the mapping out input values to output values
                     of the stretched histogram.
                             
   uFlags            bit      purpose
                     ---      -------
                      0-7      unused

                      8       if set, use mid-point to divide
                              output range: 
                              half the intensity values will 
                              be scaled to 0...mid-point ,
                              and the other half will be scaled
                              mid-point...maxval .

                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL            _ISCConv  _ISFn(is6_GetStretchedHistogramLUT)(UINT32 *pHisto, UINT32 uTotalElements, double fLowPercent, double fHiPercent, UINT32 midPoint, BYTE *pLUT, UINT32 uFlags);

/*---------------------------------------------------------
   is6_BrightnessHistogramEqualizeImage / _is6_BrightnessHistogramEqualizeImage

   Purpose : 
   Perform histogram equalization on an RGB or RGBA image.
   If RGBA, the A channel is ignored. 

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel. 3 for RGB, 4 for RGBA.
   uRowStride        bytes in an image row
   uLoThresh         low threshold
   uHiThresh         high threshold
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL          _ISCConv _ISFn( is6_BrightnessHistogramEqualizeImage )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uLoThresh, UINT32 uHiThresh, UINT32 uFlags);

/*---------------------------------------------------------
   is6_HistogramEqualizeImage / _is6_HistogramEqualizeImage

   Purpose : 
   Perform histogram equalization on an image.

   Note:
   Color channels are modified independently of each other, 
   so some shifts in color may occur. See 
   is6_BrightnessHistogramEqualizeImage if you want to 
   equalize the brightness of the image.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uChannelsPerPixel max 8.
   uRowStride        pixel components in an image row
   uLoThresh         low threshold
   uHiThresh         high threshold

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)
							  
	Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_HistogramEqualizeImage )(void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, UINT32 uLoThresh, UINT32 uHiThresh, UINT32 uFlags);


/*---------------------------------------------------------
   is6_HistogramSpecification / _is6_HistogramSpecification

   Purpose : 
   Perform a histogram specification (aka "histogram matching")
   on an image.

   Given the histogram from one image, adjust the histogram of 
   the input image to resemble it.

   Note:
   Color channels are modified independently of each other, 
   so some shifts in color may occur. 

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height

   uComponentsPerPixel number of components per pixel in the image

   uRowStride        pixel components in an image row
   pTargetHisto      the target histogram

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_HistogramSpecification)(void *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uRowStride, const UINT32 *pTargetHisto, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ContrastMaskImage / _is6_ContrastMaskImage

   Purpose : 
   Apply a contrast mask to the image to bring out shadow and 
   highlight details. This is much more effective than traditional
   brightness / contrast adjustments for most images.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   pOut              destination image
   uBytesPerPixel    number of bytes per pixel in the image. (3 or 4)
   uInRowStride      bytes per input pixel row

   uBlendMode        0 : softer contrast
                            1 : higher contrast (brighter highlights, darker shadows)

    fSigma          amount of blur in the mask image.
                    all positive values are valid, but the best results
                    are typically between 2.0 and 8.0. too high and
                    there may be ringing around details. too low and 
                    the image will start to blur.

   uFlags            unused

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_ContrastMaskImage)(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, UINT32 uBlendMode, double fSigma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_InpaintImage / _is6_InpaintImage

   Purpose : 
   Repair a damaged section of an image. Requires a user-specified
   mask to denote the damaged area.

   The mask image is an single channel image, and must have the same 
   dimensions as the input and output images. Mask pixels of zero
   specify areas to be copied directly from input to output (no
   modification). All non-zero pixels specify areas to be repaired. 
   
   This uses the NS and Telea fast marching methods from the OpenCV
   library.

   Param             Use
   ------------------------------------
   pIn               source image. grayscale, RGB or RGBA.
   uWidth            width in pixels
   uHeight           height in pixels
   uChannelsPerPixel number of components per pixel in the image (1, 3 or 4)
   uInRowStride      components per input pixel row
   pOut              output image. must be the same dimensions as the input.
   uOutRowStride     components per output pixel row

   pMask             single channel damage selection mask image. must be the same 
                     dimensions as the input.

   uMaskRowStride    components per mask pixel row

   sigma             1 - 100. controls the size of the vectors used to determine 
                     new pixel values. small values (less than 10) will give good 
                     results. processing time increases with the square 
                     of this value.

   uFlags            bit      purpose
                     ---      -------
                      0       use Telea's inpainting algorithm. 
                              gives better results, but takes longer to process.

                      1       do not modify alpha in 4 component images.

                      9-10    00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)
                              11 - images have 32 bits per component (float) (val=0x600)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv  _ISFn(is6_InpaintImage)(const void * pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uInRowStride, void * pOut, UINT32 uOutRowStride, const void *pMask, UINT32 uMaskRowStride, int sigma, UINT32 uFlags);

#define IS_BRIGHTNESS_HISTOGRAM_STRETCH_HSL  (1<<9)

#ifdef __cplusplus
}
#endif
#endif