/*---------------------------------------------------------

   ISMorphoEffects : part of the ImgSource library.

    This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

    No portion of this code may be copied or distributed without 
    permission of Smaller Animals Software, Inc..
    http://www.smalleranimals.com

---------------------------------------------------------*/

#pragma once

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Do not include this file directly. Include ISource.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

    
/*---------------------------------------------------------
   is6_MorphoDilationGray / _is6_MorphoDilationGray

   Purpose : 
   Apply a morphological dilation filter to a grayscale 
   image.

   Note:
   The default behavior for this function is:
   1. pass the mask over the source image.
   2. for each value in the mask, add its value to the pixel it overlays
   3. set the output value to the maximum of these additions.

   The behavior enabled by bit 0 of uFlags is:
   1. pass the mask over the source image.
   2. find the maximum of all pixels below non-zero mask values
   3. set the output value to that maximum.

   In the first mode, the output values are increased by the mask values.
   In the second mode, they are not.

   Param             Use
   ------------------------------------
   pInGray           input grayscale image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pMask             N x N integer mask array

   N                 number of elements in one dimension of pMask.
                     N must be odd and greater than 0

   pOutGray          output grayscale image. allocated by caller
   uOutRowStride     bytes per output pixel row

   uFlags            bit            use
                     ---            ---
                     0              do not add mask elements to output pixel.
                     20             disable multithreaded processing

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_MorphoDilationGray )(const BYTE *pInGray, UINT32 w, UINT32 h, UINT32 uInRowStride, const int *mask, int N, BYTE *pOutGray, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoErosionGray / _is6_MorphoErosionGray

   Purpose : 
   Apply a morphological erosion filter to a grayscale 
   image.

   Note:
   The default behavior for this function is:
   1. pass the mask over the source image.
   2. for each value in the mask, subtract its value from the pixel it overlays
   3. set the output value to the minimum of these subtractions.

   The behavior enabled by bit 0 of uFlags is:
   1. pass the mask over the source image.
   2. find the minimum of all pixels below non-zero mask values
   3. set the output value to that minimum.

   In the first mode, the output values are decreased by the mask values.
   In the second mode, they are not.

   Param             Use
   ------------------------------------
   pInGray           input grayscale image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row

   pMask             N x N integer mask array

   N                 number of elements in one dimension of pMask.
                     N must be odd and greater than 0

   pOutGray          output grayscale image. allocated by caller
   uOutRowStride     bytes per output pixel row


   uFlags            bit            use
                     ---            ---
                     0              do not subtract mask elements from output pixel.
                     20             disable multithreaded processing


   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_MorphoErosionGray )(const BYTE *pInGray, UINT32 w, UINT32 h, UINT32 uInRowStride, const int* mask, int N, BYTE *pOutGray, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CloseGray

   Purpose : 
   Apply morphological "close" to an 8-bit image

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pInGray           source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   piMask            matrix (NxN values)
   N                 matrix size (actual matrix has NxN values)
   pOutGray          output image
   uOutRowStride     bytes per output pixel row


   uFlags            bit            use
                     ---            ---
                     0              use non-summing erosion and dilation.
                                    see notes for is6_MorphoDilationGray, is6_MorphoErosionGray.

                     20             disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_CloseGray )(const BYTE *pInGray, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, const int *piMask, int N, BYTE *pOutGray, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_OpenGray

   Purpose : 
   Apply morphological "open" to an 8-bit image

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pInGray           source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   piMask            matrix (NxN values)
   N                 matrix size (actual matrix has NxN values)
   pOutGray          output image
   uOutRowStride     bytes per output pixel row


   uFlags            bit        use
                     ---        ---
                     0          use non-summing erosion and dilation.
                                see notes for is6_MorphoDilationGray, is6_MorphoErosionGray.

                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_OpenGray )(const BYTE *pInGray, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, const int *piMask, int N, BYTE *pOutGray, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_TophatGray

   Purpose : 
   Apply morphological "tophat" to an 8-bit image.

   By default, this is the 'white' top hat: 

      wth = image - Open(image)

   This will also do the 'black' top hat:

      bth = Close(image) - image

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pInGray           source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   piMask            matrix (NxN values)
   N                 matrix size (actual matrix has NxN values)
   pOutGray          output image
   uOutRowStride     bytes per output pixel row

   uFlags            bit         use
                     ---         ---
                     0           use non-summing erosion and dilation.
                                 see notes for is6_MorphoDilationGray, is6_MorphoErosionGray.

                     1           do 'black' top hat 

                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_TophatGray )(const BYTE *pInGray, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, const int *piMask, int N, BYTE *pOutGray, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoGradientGray

   Purpose : 
   Find the morphological gradient of an 8-bit image.

   This highlights sharp gray-level transitions (edges)
   in an image.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.

   Param             Use
   ------------------------------------
   pInGray           source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pMask             matrix (NxN values)
   N                 matrix size (actual matrix has NxN values)
   pOutGray          output image
   uOutRowStride     bytes per output pixel row

   uFlags            bit        use
                     ---        ---
                     0          use non-summing erosion and dilation.
                                see notes for is6_MorphoDilationGray, is6_MorphoErosionGray.

                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MorphoGradientGray )(const BYTE *pInGray, UINT32 w, UINT32 h, UINT32 uInRowStride, const int *pMask, int N, BYTE *pOutGray, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoDilation1Bit

   Purpose : 
   Apply a morphological dilation to a 1-bit image.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
      *  Output should be set to all 0's before calling this function

   Param             Use
   ------------------------------------
   pIn               input 1-bit image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output 1-bit image
   uOutRowStride     bytes per output pixel row
   pMask             N x J integer mask array

   N                 number of columns in pMask.
                     N must be odd and greater than 0

   J                 number of rows in pMask.
                     J must be odd and greater than 0

   uFlags            bit            use
                     ---            ---
                     20            disable multithreaded processing



   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MorphoDilation1Bit )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, const int *mask, UINT32 uFilterSizeX, UINT32 uFilterSizeY, UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoErosion1Bit

   Purpose : 
   Apply a morphological erosion to a 1-bit image.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
      *  Output should be set to all 0's before calling this function

   Param             Use
   ------------------------------------
   pIn               input 1-bit image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output 1-bit image
   uOutRowStride     bytes per output pixel row
   pMask             N x J integer mask array

   N                 number of columns in pMask.
                     N must be odd and greater than 0

   J                 number of rows in pMask.
                     J must be odd and greater than 0

   uFlags            bit            use
                     ---            ---
                     20             disable multithreaded processing


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MorphoErosion1Bit )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, const int *mask, UINT32 N, UINT32 J, UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoClose1Bit

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
      *  Output should be set to all 0's before calling this function

   Purpose : 
   Apply a morphological close to a 1-bit image.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
      *  Output should be set to all 0's before calling this function

   Param             Use
   ------------------------------------
   pIn               input 1-bit image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output 1-bit image
   uOutRowStride     bytes per output pixel row
   pMask             N x J integer mask array

   N                 number of columns in pMask.
                     N must be odd and greater than 0

   J                 number of rows in pMask.
                     J must be odd and greater than 0

   uFlags            bit            use
                    ---            ---
                    20            disable multithreaded processing


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MorphoClose1Bit )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, const int *mask, UINT32 uFilterSizeX, UINT32 uFilterSizeY ,UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoOpen1Bit

   Purpose : 
   Apply a morphological open to a 1-bit image.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
      *  Output should be set to all 0's before calling this function

   Param             Use
   ------------------------------------
   pIn               input 1-bit image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output 1-bit image
   uOutRowStride     bytes per output pixel row
   pMask             N x J integer mask array

   N                 number of columns in pMask.
                     N must be odd and greater than 0

   J                 number of rows in pMask.
                     J must be odd and greater than 0

   uFlags            bit            use
                     ---            ---
                     20            disable multithreaded processing



   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MorphoOpen1Bit )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, const int *pMask, UINT32 N, UINT32 J, UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoHitOrMiss1Bit

   Purpose : 
   Perform a hit-or-miss transformation on a 1-bit image.

   Hit or Miss is a basic tool for finding geometric shapes
   within an image. The masks used determine the shapes that
   are detected.

   Mask A operates on foreground features and mask B 
   on background features. Together they detect 
   foreground-background transitions in the input image.
   
   The two masks used are normally the same size and _must_ 
   contain mutually-exclusive elements: if a mask position
   in A contains a 1, the corresponding position in B must 
   contain a 0.


   Ex:

   These masks will find vertical edges on the left sides of 
   the foreground image.

   mask A:     mask B:
   0 0 0       0 0 0 
   0 1 1       1 0 0
   0 0 0       0 0 0

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
      *  Output should be set to all 0's before calling this function

   Param             Use
   ------------------------------------
   pIn               input 1-bit image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output 1-bit image
   uOutRowStride     bytes per output pixel row
   pMaskA            Na x Ja integer mask array

   Na                number of columns in pMaskA.
                     Na must be odd and greater than 0 

   Ja                number of rows in pMaskA.
                     Ja must be odd and greater than 0 

   pMaskB            Nb x Jb integer mask array

   Nb                number of columns in pMaskB.
                     Nb must be odd and greater than 0

   Jb                number of rows in pMaskB.
                     Jb must be odd and greater than 0

   uFlags            bit            use
                     ---            ---
                     20            disable multithreaded processing



   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MorphoHitOrMiss1Bit )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, const int *pMaskA, UINT32 Na, UINT32 Ja, const int *maskB, UINT32 Nb, UINT32 Jb, UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoBoundaryExtraction1Bit

   Purpose : 
   Perform a boundary detection/extraction on a 1-bit image.
   This will generate an outline of the foreground features of 
   the input image.

   A typical mask is 3x3 with all elements set to 1.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
      *  Output should be set to all 0's before calling this function

   Param             Use
   ------------------------------------
   pIn               input 1-bit image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output 1-bit image
   uOutRowStride     bytes per output pixel row
   pMask             N x J integer mask array

   N                 number of columns in pMask.
                     N must be odd and greater than 0

   J                 number of rows in pMask.
                     J must be odd and greater than 0

   uFlags            bit            use
                     ---            ---
                     20            disable multithreaded processing


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MorphoBoundaryExtraction1Bit )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, const int *pMask, UINT32 N, UINT32 J, UINT32 uFlags);

/*---------------------------------------------------------
   is6_MorphoThinning1Bit

   Purpose : 
   Apply morphological thinning to a 1-bit image. This
   will reduce the geometric size of an image, leaving only 
   a skeletonized frame behind.

   This function requires two sets of masks. the A masks represents
   foreground matching masks and the B masks should be the compliements
   of the A masks. (swap ones and zeros).

   A typical set of A masks is:

   0 0 0   1 0 0   1 1 0   1 1 1   1 1 1   1 1 1   0 1 1   0 0 1
   1 1 1   1 1 0   1 1 0   1 1 0   1 1 1   0 1 1   0 1 1   0 1 1
   1 1 1   1 1 1   1 1 0   1 0 0   0 0 0   0 0 1   0 1 1   1 1 1

   The B masks are compliments of these (1 and 0 reversed).

   Note that the content of each mask is the preceding mask, rotated 45 deg.
   
   The process works by applying each of these masks in pairs (A[x], B[x])
   for all pairs, until maxIterations has been reached or until there is
   no change in the image. This can be a long process.

   Notes:
      *  pIn != pOut.
      *  Output is allocated by the caller.
      *  Output should be set to all 0's before calling this function

   Param             Use
   ------------------------------------
   pIn               input 1-bit image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output 1-bit image

   pMasksA           an array of foreground masks

   uFilterSizeA      number of elements in one dimension of one 
                     element in pMaskA. must be odd and greater than 0

   pMasksB           an array of background masks. must be the same size
                     as pMasksA.

   uFilterSizeB      number of elements in one dimension of one 
                     element in pMaskB. must be odd and greater than 0.
                     should be the same as uFilterSizeA.

   uMaskCount        number of masks in pMasksA (and pMasksB)

   uMaxIterations    maximum number of thinning iterations to try. may stop
                     early, if the process yields no changes in the image.

   uFlags             bit            use
                     ---            ---
                     20                disable multithreaded processing



   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MorphoThinning1Bit )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, 
                            const int *masksA, UINT32 uElementsInAMask, const int *masksB, UINT32 uElementsInBMask, UINT32 uMaskCount, UINT32 uMaxIterations, UINT32 uFlags);


/*---------------------------------------------------------
   is6_InvertBuffer

   Purpose : 
   Invert all bits in a buffer. 
   
   Among other things, this can be used to implement
   the image compliment of a 1-bit image during 
   morphological processing.

   Param             Use
   ------------------------------------
   pIn               input buffer
   uLen              buffer size in bytes

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_InvertBuffer )(void * pBuf, UINT32 uLen);

/*---------------------------------------------------------
   is6_CopyInvertedBuffer

   Purpose : 
   Copy one buffer to another, with all bits inverted.
   
   Among other things, this can be used to implement
   the image compliment of a 1-bit image during 
   morphological processing.

   Param             Use
   ------------------------------------
   pIn               input buffer
    pOut                    output buffer
   uLen              buffer size in bytes

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec void _ISCConv _ISFn( is6_CopyInvertedBuffer )(const void *pIn, void *pOut, UINT32 uLen);

/*---------------------------------------------------------
   is6_IntersectBuffers

   Purpose : 
   Combine two buffers using a logical AND. 
   
   Among other things, this can be used to implement
   the intersection of two 1-bit images during morphological 
   processing.

   Param             Use
   ------------------------------------
   pA                input buffer 1
   pB                input buffer 2
   pOut              output buffer (may be pA or pB)
   uLen              buffer size in bytes

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_IntersectBuffers )(const void *pA, const void *pB, void *pOut, UINT32 uLen);

/*---------------------------------------------------------
   is6_IntersectRow1Bit

   Purpose : 
   Combine two buffers using a logical AND, without modifing
   any bits in any partial pixels at the end of the row.
   
   Among other things, this can be used to implement
   the intersection of two 1-bit images during morphological 
   processing.

   Param             Use
   ------------------------------------
   pA                input buffer 1
   pB                input buffer 2
   pOut              output buffer (may be pA or pB)
   uLen              buffer size in pixels

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_IntersectRow1Bit)(const void *pA, const void *pB, void *pOut, UINT32 uPix);

/*---------------------------------------------------------
   is6_UnionBuffers

   Purpose : 
   Combine two buffers using a logical OR. 
   
   Among other things, this can be used to implement
   the union of two 1-bit images during morphological 
   processing.

   Param             Use
   ------------------------------------
   pA                input buffer 1
   pB                input buffer 2
   pOut              output buffer (may be pA or pB)
   uLen              buffer size in bytes

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_UnionBuffers )(const void *pA, const void *pB, void *pOut, UINT32 uLen);
 
/*---------------------------------------------------------
   is6_XORBuffers

   Purpose : 
   Combine two buffers using a logical XOR. 

   Param             Use
   ------------------------------------
   pA                input buffer 1
   pB                input buffer 2
   pOut              output buffer (may be pA or pB)
   uLen              buffer size in bytes


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_XORBuffers )(const void *pA, const void *pB, void *pOut, UINT32 uLen);

#ifdef __cplusplus
}
#endif
