/*---------------------------------------------------------

   ISImageNoise : part of the ImgSource library.

    Noise generating functions

    This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

    No portion of this code may be copied or distributed without 
    permission of Smaller Animals Software, Inc..
    http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef CPERLINNOISH
#define CPERLINNOISH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Do not include this file directly. Include ISource.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_AddNoise

   Purpose:
   Add noise to an image.

   Param             Use
   ------------------------------------
   pIn                  input image
   uWidth               width in pixels
   uHeight              height
   uComponentsPerPixel  number of components per pixel in the image. max 8.
   uInRowStride         components per input pixel row

   uMode                0 : Gaussian noise
                        1 : negative exp noise
                        2 : Rayleigh
                        3 : uniform

   fVariance            controls intensity (no upper limit)
   fMean                added to noise output (Gaussian and uniform only)

   bMonochromatic       if TRUE, the same noise is applied equally to 
                        all pixel channels

   uFlags               bit      purpose
                        ---      -------
                        0-7      These bits control which channels to
                                 process. If bit N is set, channel N is 
                                 processed. If bit N is not set, channel
                                 N is skipped (nothing is written to
                                 the output for that channel).

                                 Ex. 
                                 If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                                 (R, G and B) will be processed. 

                                 If uFlags = 1 | 4 , channels 0 and 2 
                                 (R and B) will be processed. Channel 1 (Green)
                                 will be skipped.

                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_AddNoise )(void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, UINT32 uMode, double fVariance, double fMean, BOOL bMonochromatic, UINT32 uFlags);


/*---------------------------------------------------------
   is6_GenerateColoredNoise

   Purpose : 
   Generate a colored noise image. 

   Notes:
      * caller allocates output 

   Param             Use
   ------------------------------------
   pOut              output image
   uWidth            width in pixels
   uHeight           height
   uRowStride        bytes per pixel row
   clr               base image color
   uSize             pattern size (1 to 100)
   uMode             noise generation mode
                     0: Perlin (1984)
                     1: Hugo
                     2: Gaussian
                     3: Negative Exponential
                     4: Rayleigh
                     5: Uniform
                     6: Perlin (2002)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_GenerateColoredNoise )(BYTE *pOut, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, COLORREF clrLow, UINT32 uSize, UINT32 uMode);

/*---------------------------------------------------------
   is6_InitializePerlinNoiseVolume / _is6_InitializePerlinNoiseVolume

   Purpose : 
   Create a new Perlin noise generator object. This is used
   with is6_GetPerlinNoiseSlice. 

   "Perlin" noise is a noise that, at low frequencies, shows a 
    lot of structured clumping - blobs and rolling hills, etc..

   Caller must destroy this object with is6_DestroyPerlinNoiseVolume
   when finished.

   This offers two versions of Perlin's noise. The first is his
   classic (1984) noise, which includes some random variations. The
   second is his "improved" (2002) version, which does not have any
   randomness in it. see http://mrl.nyu.edu/~perlin/noise/ and
   http://mrl.nyu.edu/~perlin/paper445.pdf . 

   You can get different noise patterns in the improved version
   by simply changing the x,y or z offset each time you fetch 
   a new slice.

   Param             Use
   ------------------------------------

   uFlags            bit      purpose
                     ---      -------
                      0       if set, use Perlin's "improved" method.
                              otherwise, use Perlin's classic method.

   Return
   ------
   HISPERLIN - Perlin noise object
---------------------------------------------------------*/
_ISDeclSpec HISPERLIN _ISCConv _ISFn( is6_InitializePerlinNoiseVolume )(UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPerlinNoiseSlice / _is6_GetPerlinNoiseSlice

   Purpose : 
   Create a single 2D plane of Perlin noise. Each call to the 
   function produces a single slice through a "volume" of 
   noise at a given "z" offset. By repeatedly calling this 
   function with different z values, you can generate a full
   volume of noise, or you can use the z value as a time value
   and use this to generate frames for animated noise.

   Param             Use
   ------------------------------------
   hPerlin           Perlin noise object allocated with 
                     is6_InitializePerlinNoiseVolume

   pOut              output image
   uWidth            width in pixels
   uHeight           height
   uRowStride        components per output row

   fScale            controls size of noise features. (0 to 1).
                     a smaller value here yields larger features

   fZOffset          z offset within the volume. 0 is fine.
   fStartX           x offset within the volume. 0 is fine.
   fStartY           y offset within the volume. 0 is fine.

   fRadians          amount to rotate the image. 0 is fine.
   fRotOrigX         center of image rotation, X. 0 is fine.
   fRotOrigY         center of image rotation, Y. 0 is fine.

   uFlags            bit      purpose
                     ---      -------
                      10      image components are 64-bit doubles

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_GetPerlinNoiseSlice)(HISPERLIN hPerlin, void *pOut, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, double fScale, double fZOffset, double xStart, double yStart, double radians, double xRotOrig, double yRotOrig, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DestroyPerlinNoiseVolume / _is6_DestroyPerlinNoiseVolume

   Purpose : 
   Destroy the Perlin noise generator object. This frees
   all memory associated with the given noise object.

   See is6_InitializePerlinNoiseVolume.

   Param             Use
   ------------------------------------
   HISPERLIN        Perlin noise object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_DestroyPerlinNoiseVolume )(HISPERLIN hPerlin);

/*---------------------------------------------------------
   is6_GenerateHugoNoise

   Purpose : 
   Generate an image filled with Hugo Elias's implementation of
   "Perlin" noise. The noise generated by this, while interesting in
   its own right, is quite different from the Perlin noise generated
   by code supplied by Mr Perlin himself. Therefore, We've called this
   "Hugo" noise, to distinguish it from Perlin noise and to honor 
   Mr. Elias for coming up with such a nice noise generator.
     
   http://freespace.virgin.net/hugo.elias/models/m_perlin.htm

   Param             Use
   ------------------------------------
   pOut              output image (8 bit)
   uWidth            width in pixels
   uHeight           height
   uRowStride        bytes per input pixel row
   iOctaves          number of waves contributing to the noise (2-6 is good)
   fPersistence      controls smoothness (0 to 10)
   fScale            controls size of noise features. (0 to 1).
                     a smaller value here yields larger features

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_GenerateHugoNoise )(BYTE *pOut, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, int iOctaves, double fPersistence, double fScale);


#ifdef __cplusplus
}
#endif

#endif
