/*---------------------------------------------------------
   ISImageProcess.h : part of the ImgSource library.
   
   Image processing functions

   This code copyright (c) 2000-2014, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is6_IMAGEPROCH
#define is6_IMAGEPROCH

// shut up, stupid compiler!
#pragma warning(disable:4786)

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_ImageToGrayScale / _is6_ImageToGrayScale

   Purpose : 
   Convert an RGB or RGBA image to 24-bit (or 48-bit) grayscale. 
   Sets all color in a pixel to the luminosity value for that 
   pixel.

   If input is RGBA, alpha channel is not modified.

   Param             Use
   ------------------------------------
   pInput            image
   uWidth            width in pixels
   uHeight           height

   uChannelsPerPixel 3 = RGB
                     4 = RGBA, A channel is skipped

   uRowStride        number of pixel components per row

   uFlags            bit      purpose
                     ---      -------
                     0        if set, treat source pixels as BGR/BGRA, else RGB/RGBA

                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_ImageToGrayScale )(void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ImageToGrayScaleSingle / _is6_ImageToGrayScaleSingle

   Purpose : 
   Convert an RGB or RGBA image to 8-bit (or 16-bit) grayscale. 
   Sets the output pixel to the luminosity value for the input pixel.

   This works for images with 8-bits per component (RGB-24, RGBA-32) and 
   images with 16-bits per component (RGB-48, RGBA-64).

   Alpha channel has no effect on the output.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height

   uBytesPerPixel    3 = input is 24-bit RGB (or RGB-48)
                     4 = input in 32-bit RGBA, (or RGBA-54) A channel is skipped

   uInRowStride      number of components in an input pixel row

   pOutput           output buffer
   
   uOutRowStride     number of components in an output pixel row

   uFlags            bit      purpose
                     ---      -------
                     0        if set, treat source pixels as BGR/BGRA, else RGB/RGBA

                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

                    20        disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_ImageToGrayScaleSingle)(const void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, void *pOutput, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ImageToGrayScaleWeighted / _is6_ImageToGrayScaleWeighted

   Purpose : 
   Convert an RGB or RGBA image to 24-bit (or 48-bit) 
   grayscale using adjustable weightings for each color.   

   If input is RGBA, alpha channel is not modified.

   Ex. 
   To duplicate what is6_ImageToGrayScale does, use weightings of:
      rw = .299
      gw = .587
      bw = .114

   Note:
   Weightings should add up to 1.0, though this function does
   not enforce it. 
   
   Note:
   If the weightings are > 1 and the result of the calculation 
   generates a value > maxPixVal, the output pixel will wrap around to v - maxPixVal.

   Param             Use
   ------------------------------------
   pInput            image
   uWidth            width in pixels
   uHeight           height

   uChannelsPerPixel 3 = RGB
                     4 = RGBA, A channel is skipped

   uRowStride        number of pixel components per row

   rw                weight to use for red channel
   gw                weight to use for green channel
   bw                weight to use for blue channel

   uFlags            bit      purpose
                     ---      -------
                     0        if set, treat source pixels as BGR/BGRA, else RGB/RGBA
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_ImageToGrayScaleWeighted )(void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, double rw, double gw, double bw, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ImageToGrayScaleSingleWeighted / _is6_ImageToGrayScaleSingleWeighted

   Purpose : 
   Convert an RGB or RGBA image to 8 bit (or 16-bit) grayscale
   using adjustable weightings for each color. 
   
   This works for images with 8-bits per component (RGB-24, RGBA-32) and 
   images with 16-bits per component (RGB-48, RGBA-64).

   Alpha channel has no effect on the output.

   Ex. 
   To duplicate what is6_ImageToGrayScale8Bit does, use weightings of:
      rw = .299
      gw = .587
      bw = .114


   Note:
   Weightings should add up to 1.0, though this function does
   not enforce it. 
   
   Note:
   If the weightings are > 1 and the result of the calculation 
   generates a value > maxPixVal, the output pixel will wrap around to v - maxPixVal.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height

   uBytesPerPixel    3 = input is 24-bit or 48-bit RGB
                     4 = input in 32-bit or 64-bit RGBA, A channel is skipped

   uInRowStride      number of components in an input pixel row

   pOutput           output buffer (8-bits or 16-bits per pixel)
   
   uOutRowStride     number of components in an output pixel row

   rw                weight to use for red channel
   gw                weight to use for green channel
   bw                weight to use for blue channel

   uFlags            bit      purpose
                     ---      -------
                     0        if set, treat source pixels as BGR/BGRA, else RGB/RGBA

                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_ImageToGrayScaleSingleWeighted)(const void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, void *pOutput, UINT32 uOutRowStride, double rw, double gw, double bw, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBToBGR / _is6_RGBToBGR

   Purpose : 
   Swap red and blue channels in an RGB image

   Note: An SSE-accelerated function may be used if the
   processor supports the required instructions - for
   BYTE data.

   Param             Use
   ------------------------------------
   pRGB              image
   uWidth            width in pixels
   uHeight           height
   uRowStride        number of pixel components in a pixel row

   uFlags            bit      purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_RGBToBGR )(void *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBAToBGRA / _is6_RGBAToBGRA

   Purpose : 
   Swap red and blue channels in an RGBA image

   Param             Use
   ------------------------------------
   pRGBA             image
   uWidth            width in pixels
   uHeight           height
   uRowStride        number of components in a pixel row

   uFlags            bit      purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_RGBAToBGRA )(void *pRGBA, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBToRGBA / _is6_RGBToRGBA

   Purpose : 
   Copy the RGB channels of an RGB image to the RGB
   channels of an RGBA image.

   Param             Use
   ------------------------------------
   pRGB              image
   uWidth            width in pixels
   uHeight           height 
   uInRowStride      number of bytes in an input row
   pRGBAOut          output RGBA image
   uOutRowStride     number of bytes in an output row
   uFillAlpha        default alpha value (0..255)
   
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_RGBToRGBA)(const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE* pRGBAOut, UINT32 uOutRowStride, UINT32 uFillAlpha);

/*---------------------------------------------------------
   is6_SplitImage / _is6_SplitImage

   Purpose : 
   Copy channels from an input image into separate 
   output buffers.

   Caller must allocate all output buffers.

   See also is6_CopyImageChannel

   Param             Use
   ------------------------------------
   pInput            source image

   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)

   uComponentsPerPixel    number of components per pixel.
                     also, number of elements in pOutPlanes

   uInRowStride      components per input row

   pOutPlanes        array of pointers to output buffers
                     each buffer must be uWidth * uHeight components.
                     if an array element is NULL, that channel
                     will be skipped.

   uOutRowStride     components per output row (for all in pOutPlanes)

   uFlags            bits     purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)
                              10 - images have 64 bits per component (double) (val=0x400)
                              11 - images have 32 bits per component (float) (val=0x600)

  Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_SplitImage)(const void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void **pOutPlanes, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CopyImageChannel / _is6_CopyImageChannel

   Purpose : 
   Copy a single channels from the input image

   Caller must allocate output buffer.

   See also is6_SplitImage

   Param             Use
   ------------------------------------
   pInput            source image

   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uComponentsPerPixel    number of components per pixel
   uInRowStride      components per input row
   pOut              output image. must be uWidth * uHeight components.
   uOutRowStride     components per output row
   uChannelIdx       index of input channel to copy

   uFlags            bits     purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)
                              10 - images have 64 bits per component (double) (val=0x400)
                              11 - images have 32 bits per component (float) (val=0x600)

  Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_CopyImageChannel)(const void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uChannelIdx, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CombineImageChannels / _is6_CombineImageChannels

   Purpose : 
   Combine multiple images to create a
   single multi-channel image.

   The first component from pInputPlanes[0] will be the first 
   component in the combined pixel. The first component from 
   pInputPlanes[1] will be the second component in the combined pixel, etc..

   If any element in pInputPlanes is NULL, that channel
   will be skipped (no value will be written). This
   will allow you to set single color channels in image.
   ie. you can use this to set the R channel in an RGB image,
   or the A channel in an RGBA image.

   Planes are expected to be non-overlapping. The "next" pixel
   in a plane is always expected to be *(pCurPixel + 1).

   See also is6_SetImageChannel

   Param             Use
   ------------------------------------
   pInputPlanes      array of pointers to component buffers.
                     if an array element is NULL, the corresponding
                     channel will be skipped.
                     
                     each buffer must be uHeight * uInRowStride components

   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)

   uComponentsPerPixel    number of components per pixel.
                     also, number of elements in pOutPlanes

   uInRowStride      components per input row

   pOutput           output buffer. this is allocated by the
                     caller and must contain uHeight * uOutRowStride components

   uOutRowStride     components per output pixel row

   uFlags            bits     purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)
                              10 - images have 64 bits per component (double) (val=0x400)
                              11 - images have 32 bits per component (float) (val=0x600)
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_CombineImageChannels)(const void **pInputPlanes, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOutput, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_SetImageChannel / _is6_SetImageChannel

   Purpose : 
   Copy a single channel image into a multi-channel image.
   Can be used to set an alpha channel, for example.

   This is the single-channel version of is6_CombineImageChannels.

   Caller must allocate all output buffers.

   Param             Use
   ------------------------------------
   pSingleChannel       source image (single channel, 8-bit or 16-bit)

   uWidth               width in pixels (of all buffers)
   uHeight              height (of all buffers)

   uInRowStride         components per input row

   pOut                 output image (one or more channels)

   uComponentsPerPixel  number of components per pixel in pOut.

   uOutRowStride        components per output row

   uChannelIdx          channel in pOut which will receive the data 
                        from pSingleChannel (zero-based)

   uFlags               bits     purpose
                        ---      -------
                        9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                                 01 - images have 16 bits per component (ISUINT16) (val=0x200)
                                 10 - images have 64 bits per component (double) (val=0x400)
                                 11 - images have 32 bits per component (float) (val=0x600)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_SetImageChannel)(const void *pSingleChannel, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, void *pOut, UINT32 uComponentsPerPixel, UINT32 uOutRowStride, UINT32 uChannelIdx, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ShuffleImageChannels / _is6_ShuffleImageChannels

   Purpose : 
   Selectively copy the channels of one image to another.

    The pPositions array selects which channels in the input
    go to which channels in the output. There must be 
    uComponentsPerPixel entries in this array.

    Selection is determined by:
    1) The position in the pPositions array
    2) The value at that position.
    3) If the value in a position is 255, that channel is skipped

    For each channel C:
        output_channel[C] = input_channel[pPositions[C]]

    Examples:
    Assume the images each have three channels. 

    Example 1:
    
    If these are the 3 values in the pPositions array:

            2  1  0

    Then...
        Output channel 0    <== Input channel 2
        Output channel 1    <== Input channel 1
        Output channel 2    <== Input channel 0

   Example 2:
    If these are the 3 values in the pPositions array:

            1  1  0

    Then...
        Output channel 0    <== Input channel 1
        Output channel 1    <== Input channel 1
        Output channel 2    <== Input channel 0

   Example 3:
    If these are the 3 values in the pPositions array:

            2  0  255

    Then...
        Output channel 0    <== Input channel 2
        Output channel 1    <== Input channel 0
        Output channel 2    is not modified (255 = skip)

   Param               Use
   ------------------------------------
   pIn                 input image
   uWidth              width of both images
   uHeight             height of both images
   uComponentsPerPixel number of components/channels per pixel
   uInRowStride        components per input pixel row
   pOut                output image
   uOutRowStride       components per output pixel row
   pPositions          array of positions (see Notes)

   uFlags            bits      purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)
                              10 - images have 64 bits per component (double) (val=0x400)
                              11 - images have 32 bits per component (float) (val=0x600)
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_ShuffleImageChannels)(const void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, BYTE *pPositions, UINT32 uFlags);

/*---------------------------------------------------------
   is6_SplitRGBAToRGBPlusA / _is6_SplitRGBAToRGBPlusA

   Purpose : 
   Copy the RGB channels and the Alpha channel in an RGBA
   image to seperate buffers.

   Caller must allocate all output buffers.

   if pRGB is NULL, only the alpha channel is copied.
   if pAlpha is NULL, only the RGB channels are copied.

   Note: An SSE-accelerated function may be used if the
   processor supports the required instructions, for BYTE
   data.

   Param             Use
   ------------------------------------
   pRGBA             source RGBA image
   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uRGBARowStride    components per RGBA pixel row
   pRGB              output RGB buffer
   uRGBRowStride     components per RGB pixel row
   pAlpha            output Alpha buffer
   uARowStride       components per alpha pixel row

   uFlags            bit      purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_SplitRGBAToRGBPlusA)(const void *pRGBA, UINT32 uWidth, UINT32 uHeight, UINT32 uRGBARowStride, void *pRGB, UINT32 uRGBRowStride, void *pAlpha, UINT32 uARowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReorderChannels / _is6_ReorderChannels

   Purpose : 
   Reorder the channels in an image with 8 or 16 bits per component.
    
   If an output image is provided, the function will fill the
   output buffer and leave the input intact.

   If the output buffer is NULL, the function will reorder the 
   channels in the input buffer, in place.

   Examples:

   If uBytesPerPixel is 3 and the pOrderArray is arranged:
      pOrderArray[0] = 2
      pOrderArray[1] = 1
      pOrderArray[2] = 0

   This will copy pInput into pOutput with the blue and red 
   channels reversed.

   If uBytesPerPixel is 4 (input is RGBA) and the pOrderArray 
   is arranged:
      pOrderArray[0] = 3
      pOrderArray[1] = 0
      pOrderArray[2] = 1
      pOrderArray[3] = 2

   This will copy pInput into pOutput as ARGB.

   Param             Use
   ------------------------------------
   pInput            source image
   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uComponentsPerPixel    number of bytes per pixel (of all buffers)
   uInRowStride      components per input pixel row
   pOutput           output buffer (optional)
   uOutRowStride     components per output pixel row
   pOrderArray       array of channel numbers. see examples.
   uOrderArraySize   number of items in pOrderArray

   uFlags            bit      purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_ReorderChannels )(void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOutput, UINT32 uOutRowStride, void *pOrderArray, UINT32 uOrderArraySize, UINT32 uFlags);

/*---------------------------------------------------------
   is6_BlendSolidColor / _is6_BlendSolidColor

   Purpose : 
   Blend an RGB or RGBA image with a single color. 

   Param             Use
   ------------------------------------
   pImage            source image
   uWidth            width in pixels 
   uHeight           height
   uRowStride        bytes per pixel row
   clrBlend          color to use as background
   fAmount           0..1 , color opacity (0=none, 1=solid color)

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR order
                     1        set if image is RGBA/BGRA

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_BlendSolidColor )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, COLORREF clrBlend, double fAmount, UINT32 uFlags);

/*---------------------------------------------------------
   is6_AlphaBlendSolidColor / _is6_AlphaBlendSolidColor

   Purpose : 
   Alpha blend an RGBA image with a solid "background" 
   color. Equivalent to overlaying the RGBA image onto
   a solid color background.

   Param             Use
   ------------------------------------
   pRGBA             source RGBA image
   uWidth            width in pixels
   uHeight           height
   uRowStride        bytes per pixel row
   clrBlend          color to use as background

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGRA order
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_AlphaBlendSolidColor )(BYTE *pRGBA, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, COLORREF clrBlend, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GrayscaleTo1Bit / _is6_GrayscaleTo1Bit

   Purpose : 
   Create a monochrome (one bit per pixel) image from an
   8-bit grayscale image, using one of a variety of different
   thresholding algorithms.

   Note:
   These are the different algorithms used to determine 
   grayscale threshold:

   uMode = 
      0: threshold = uParam
      1: threshold is minimum between two histogram peaks
      2: threshold is the mean pixel value
      3: iterative selection
      4: Otsu's gray level histogram method
      5: balanced averages
      6: set lowest uParam/100.0 percent of pixels to 0
      7: Johannsen's entropy algorithm
      8: moving average. local percent cutoff = uParam/100.0 (try 1500 to start (15%))
      9: One entropy
      10: Ye-Danielsson's iterative selection
      11: Co-occurrence Matrix
      12: Wu
      13: Renyi's entropy


   Pixels with a gray level below the threshold are set to 0, all
   others are set to 1.

   Param             Use
   ------------------------------------
   pImg8             source grayscale image
   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uInRowStride      bytes per pixel row, input image
   pImg1             output 1 bit image
   uOutRowStride     bytes per pixel row, output image

   uMode             threshold locating mode. see Notes.
   uParam            see Notes
   
   uFlags            unused
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_GrayscaleTo1Bit)(const BYTE *pImg8, UINT32 w, UINT32 h, UINT32 uInRowStride, BYTE *pImg1, UINT32 uOutRowStride, UINT32 uMode, UINT32 uParam, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetGrayscaleTo1BitThreshold / _is6_GetGrayscaleTo1BitThreshold

   Purpose : 
   Find the value to use as a threshold value for creating a 
    monochrome image from an 8-bit grayscale image.

   To use this value, set all pixels with a grayscale value
    below the threshold to 0, and all other to 255.

   Note:
    This is the first step in is6_GrayscaleTo1Bit.

   Note:
   These are the different algorithms used to determine 
   grayscale threshold:

   uMode =
		0: threshold = uParam . This merely returns uParam
		1: threshold is minimum between two histogram peaks
		2: threshold is the mean pixel value
		3: iterative selection
		4: Otsu's gray level histogram method
		5: balanced averages
		6: set lowest uParam/100.0 percent of pixels to 0
		7: Johannsen's entropy algorithm
		9: One entropy
		10: Ye-Danielsson's iterative selection
		11: Co-occurrence Matrix
		12: Wu
		13: Renyi's entropy
      
        Mode 8, moving average, is not used here, since it does not
        calculate a global threshold for the whole image.

   Param             Use
   ------------------------------------
   pImg8             source grayscale image
   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uInRowStride      bytes per pixel row, input image

   uMode             threshold locating mode. see Notes.

   uParam            see Notes
   
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL            _ISCConv _ISFn(is6_GetGrayscaleTo1BitThreshold)(const BYTE *pImg8, UINT32 w, UINT32 h, UINT32 uInRowStride, UINT32 uMode, UINT32 uParam1, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Pack8BitBuffer / _is6_Pack8BitBuffer

   Purpose : 
   Pack the pixels from an 8-bit buffer into a 
   buffer with 1-7 bits per pixel. 
   
   This function uses the low uBPP bits from each input pixel 
   for the output pixels.

   Multiple output pixels will share BYTEs and a pixel may 
   span multiple BYTEs. 
   
   This function assumes the output buffer is large enough to
   hold all output pixels. The size of the output buffer must 
   be at least ((uWidth * uHeight * uBPP / 8) + 1) BYTEs. The 
   caller must allocate this buffer. 
   
   Data in the last BYTE of the output buffer will be in the
   HIGH bits.

   Unused bits in the last BYTE (that contains pixel data) of 
   the output buffer will be set to 0.

   Param             Use
   ------------------------------------
   pInput            source 8-bit image
   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uInRowStride      bytes per input pixel row
   pOutput           output image
   uBPP              output bits per pixel (1..7)

   bPadRows          if TRUE, output rows are uOutRowWidth
                     BYTEs wide. if FALSE, no padding is
                     added at the end of a line; the last
                     pixel on a line may share a BYTE with 
                     the first pixel on the next line.

   uOutRowStride     width, in bytes, of output rows.
                     ignored if bPadRows is FALSE

   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_Pack8BitBuffer)(const BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOutput, UINT32 uBPP, BOOL bPadRows, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_UnpackBufferTo8Bit / _is6_UnpackBufferTo8Bit

   Purpose : 
   Unpack packed pixels in a 1-8 bit buffer.
   
   Input rows must be packed pixels - no row end padding
   and no inter-pixel padding.   If you need padding at the
   end of your rows, call this function once for each line, 
   moving the output address to accommodate the desired row 
   padding.

   This function assumes the output buffer is large enough to
   hold all output pixels. 

   The size of the output buffer must be at least (uWidth * 
   uHeight) BYTEs. The caller must allocate this buffer. 

   Data in the last BYTE of the output buffer is expected to 
   be in the HIGH bits.

   Param          Use
   ------------------------------------
   pInput            source packed image
   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uInRowStride      bytes per input row
   uBPP              input bits per pixel (1..7)
   pOutput           output 8-bit buffer 
   uOutRowStride     bytes per output row
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_UnpackBufferTo8Bit)(const BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, UINT32 uInBPP, BYTE *pOut8Bit, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
    is6_ShiftImageChannels / _is6_ShiftImageChannels

   Purpose : 
   Geometrically offset one or more channels of an image.

   Images are clipped at the edge.

   Param             Use
   ------------------------------------
   pImage            source buffer
   uWidth            width in pixels
   uHeight           height (of all buffers)
   uBytesPerPixel    number of bytes per pixel (max 8)
   uInRowStride      bytes per image pixel row, input
   pOutImage         output image
   uOutRowStride     bytes per image pixel row, output
   iXOffset          number of pixels to offset selected channels
   iYOffset          number of pixels to offset selected channels

   uBackGround       value to place into image areas 'uncovered' by
                     the offset.

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn( is6_ShiftImageChannels )(const BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOutImage, UINT32 uOutRowStride, __int32 iXOffset, __int32 iYOffset, UINT32 uBackGround, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CopyBuffer / _is6_CopyBuffer

   Purpose : 
   Copy input buffer to output buffer, adding or removing 
   row padding as specified.

   if uInRowStride = uOutRowStride, the input buffer is
   copied directly to the output buffer.

   if uInRowStride < uOutRowStride, each row in the output 
   buffer will have (uOutRowStride - uInRowStride) BYTEs
   added to the end of it. 
   
   if uInRowStride > uOutRowStride, each row in the output rows
   will be the first uOutRowStride BYTEs of the input row.

   this function can be used to convert an RGB buffer to and from
   a DWORD-aligned buffer.

   the values of any padding BYTEs in the output rows 
   will be whatever data was there when this function was called.
   ie. it is not overwritten.

   This copies min(uInRowStride, uOutRowStride) bytes for each row.

   Param             Use
   ------------------------------------
   pInputBuf         source buffer
   uInRowStride      bytes in an input row 
   uHeight           number of rows to copy
   uOutRowStride     bytes in an output row
   pOutputBuf        output buffer : (uOutRowStride * uHeight) bytes.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_CopyBuffer)(const BYTE *pInputBuf, UINT32 uInWidthBytes, UINT32 uHeight, UINT32 uOutWidthBytes, BYTE *pOutputBuf);

/*---------------------------------------------------------
   is6_8BitToRGB / _is6_8BitToRGB

   Purpose : 
   Make an RGB image from an 8 bit image and a palette

   Param             Use
   ------------------------------------
   p8Bit             source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes in an input image row
   pRGB              destination image
   uOutRowStride     bytes in an output image row
   uColors           number of colors in image
   pPal              palette

   uFlags            bit      purpose
                     ---      -------
                     0        if set, output is BGR. default RGB
         
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_8BitToRGB)(const BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pRGB, UINT32 uOutRowStride, UINT32 uColors, const RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_8BitToRGBA / _is6_8BitToRGBA

   Purpose : 
   Create an RGBA image based on an 8-bit image with a 
   palette that uses the rgbReserved member to store alpha
   channel info.

   Param             Use
   ------------------------------------
   p8Bit             source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes in an input image row
   pRGBA             destination image
   uOutRowStride     bytes in an output image row
   uColors           number of colors in image
   pPal              palette

   uFlags            bit      purpose
                     ---      -------
                     0        if set, output is BGRA. default RGBA
                     1        if set, set all A values to 255
         
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_8BitToRGBA)(const BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pRGBA, UINT32 uOutRowStride, UINT32 uColors, const RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CombineRGBAndAToRGBA / _is6_CombineRGBAndAToRGBA

   Purpose : 
   Copy the RGB channels of an RGB image and an alpha mask
   to an RGBA image. (also works when images are BGR ordered)

   if pAlpha is NULL, the alpha channel is set using uFillAlpha

   Param             Use
   ------------------------------------
   pRGB              input RGB image
   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uRGBRowStride     components per pixel row, RGB image
   pAlpha            input 8-bit alpha image
   uARowStride       components per row, alpha image
   pRGBA             output image (caller allcoated. w * h * 4)
   uRGBARowStride    components per row, output RGBA image
   uFillAlpha        default alpha value (0-maxComponent)
   
   uFlags            bit      purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_CombineRGBAndAToRGBA)(const void *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRGBRowStride, const void *pAlpha, UINT32 uARowStride, void *pRGBA, UINT32 uRGBARowStride, UINT32 uFillAlpha, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CropImage / _is6_CropImage

   Purpose : 
   Crop an image. This will copy a rectangular section 
   of the input image into the output image.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height 
   uInRowStride      number of bytes in an input row

   pOut              cropped image - allocated by the caller.
                     this must be large enough to hold the
                     output image. 

   uOutRowStride     number of bytes in an output row
   pCropRect         rectangle to crop (in source coordinates).
                     the rectangle extends up to, but does not 
                     include, the right and bottom coordinates. 

   uBytesPerPixel    bytes per pixel

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_CropImage )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, const RECT * pCropRect, UINT32 uBytesPerPixel);

/*---------------------------------------------------------
   is6_Crop1Bit / _is6_Crop1Bit

   Purpose : 
   Crop a one-bit image. This will copy a rectangular section 
   of the input image into the output image.

   Image rows must start on even byte boundaries.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height 
   uInRowStride      number of bytes in an input row

   pOut              cropped image - allocated by the caller.
                     this must be large enough to hold the
                     output image. 

   uOutRowStride     number of bytes in an output row.

   pCropRect         rectangle to crop (in source coordinates).
                     the rectangle extends up to, but does not 
                     include, the right and bottom coordinates. 

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_Crop1Bit )(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, const RECT * pCropRect);

/*---------------------------------------------------------
   is6_VerticalFlipImage / _is6_VerticalFlipImage

   Purpose : 
   Vertically flip a buffer. This can be used to vertically
   flip any buffer, not just images.

   Param             Use
   ------------------------------------
   pBuffer           buffer to flip
   uRowStride        bytes in an image row
   uHeight           buffer rows

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_VerticalFlipImage )(BYTE * pBuffer, UINT32 uRowStride, UINT32 uHeight);

/*---------------------------------------------------------
   is6_HorizontalFlipImage / _is6_HorizontalFlipImage

   Purpose : 
   Horizontally flip an image

   Param             Use
   ------------------------------------
   pRGB              source image
   uWidth            width in pixels
   uHeight           height 
   uComponentsPerPixel    channels per pixel
   uRowStride        number of components in an image row

   uFlags            bits     purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)
                              10 - images have 64 bits per component (double) (val=0x400)
                              11 - images have 32 bits per component (float) (val=0x600)
							  
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_HorizontalFlipImage )(BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FlipHorizontal1Bit / _is6_FlipHorizontal1Bit

   Purpose : 
   Horizontally flip a 1 bit image.

   Image rows must start on even byte boundaries.

   Param             Use
   ------------------------------------
   pImg              source image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of bytes in an image row

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_FlipHorizontal1Bit )(BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride);

/*---------------------------------------------------------
   is6_FillSolidRect / _is6_FillSolidRect

   Purpose : 
   Color a rectangle in an image with a solid color.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel (1, 3 or 4)
   uRowStride        number of bytes in an input row

   pFillRect         rectangle to fill. the rectangle extends up to, 
                     but does not include, the right and bottom 
                     coordinates. 
   
   uColorSpec        fill color. if uBytesPerPixel = 1, this is the
                     palette index to use. if uBytesPerPixel = 3 or 4,
                     this is the COLORREF to fill with.

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR (or BGRA) order

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_FillSolidRect )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, const RECT * pFillRect, UINT32 uColorSpec, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FillSolidRect1Bit / _is6_FillSolidRect1Bit

   Purpose : 
   Fill a rectangle in a 1-bit buffer with a single color index.

   Image rows must start on even byte boundaries.

   Param             Use
   ------------------------------------
   p1Bit             image
   uWidth            width in pixels
   uHeight           height
   uRowStride        number of bytes in an input row

   pFillRect         rectangle to fill. the rectangle extends up to, 
                     but does not include, the right and bottom coordinates. 


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_FillSolidRect1Bit )(BYTE *p1Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, const RECT * pFillRect, UINT32 uIdxFill);

/*---------------------------------------------------------
   is6_FillSolidRectSingleChannel / _is6_FillSolidRectSingleChannel

   Purpose : 
   Color a rectangle in a single channel of an image with a solid color.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel
   uRowStride        number of bytes in an input row

   pFillRect         rectangle to fill. the rectangle extends up to, 
                     but does not include, the right and bottom 
                     coordinates. 
   
   uColorSpec        fill value. (0..255)

   uFlags            bit      purpose
                     ---      -------
                     0-7      These bits control which channels to
                              process. If bit N is set, channel N is 
                              processed. If bit N is not set, channel
                              N is skipped (nothing is written to
                              the output for that channel).

                              Ex. 
                              If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                              (R, G and B) will be processed. 

                              If uFlags = 1 | 4 , channels 0 and 2 
                              (R and B) will be processed. Channel 1 (Green)
                              will be skipped.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_FillSolidRectSingleChannel )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, const RECT * pFillRect, BYTE uColorSpec, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FillTransparentRectOnImage / _is6_FillTransparentRectOnImage

   Purpose : 
   Color a rectangle in an image with a transparent color.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    bytes per pixel (3 or 4)
   uRowStride        number of bytes in an input row

   pFillRect         rectangle to fill. the rectangle extends up to, 
                     but does not include, the right and bottom 
                     coordinates. 
   
   clrBlend          fill color

   fOpacity          opacity level of fill color. 0..1. 
                     0 = fully transparent
                     1 = fully opaque

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR (or BGRA) order
                     1        do not modify alpha channel

                     2        if modifying alpha, set alpha to the A part of clrBlend.
                              otherwise, blend exist alpha with the A part of clrBlend, according to fOpacity

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_FillTransparentRectOnImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, const RECT *pFillRect, COLORREF clrBlend, double fOpacity, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FloodFillImage / _is6_FloodFillImage

   Purpose : 
   Perform a flood fill in an RGB or RGBA image. All pixels 
   contiguous to the start pixel are changed to
   clrFill if they match within the tolerance specified. The 
   extent of the fill is controlled by uFillMode.

   If image is RGBA, the A channel is ignored.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height 
   uBytesPerPixel    3 or 4
   uRowStride        number of bytes in a pixel row
   pptStart          point at which to start the fill
   clrFill           color to fill with

   uFillMode         0 - fill area is bounded by 
                     pixels matching clrFill
                     1 - all contiguous pixels matching 
                     the start pixel are set to clrFill.

   uTolerance        0 - only match specified color 
                     (strict color matching)
                     ...up to...
                     3 * (255 * 255) - maximum tolerance
                     (black matches white)

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR (or BGRA) order

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_FloodFillImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, const POINT * pptStart, COLORREF clrFill, UINT32 uFillMode, UINT32 uTolerance, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetColorMatchMaskFromImage / _is6_GetColorMatchMaskFromImage

   Purpose : 
   Get an 8-bit mask image by matching colors in the source image
   against an input color, with variable tolerance.

   This can be used to do things like a "magic wand" selection.

   Also, this mask image represents the area that would 
   have been filled, if you were using is6_FloodFillImage (mode 1).

   The output image is filled with "0" where selection (or filling)
   does not occur and "1" where selection does occur. See uFlags.

   If image is RGBA, A channel is ignored.

   Param                Use
   ------------------------------------
   pImage               source grayscale, RGB or RGBA image
   uWidth               width in pixels
   uHeight              height   
   uBytesPerPixel       1, 3 or 4
   uInRowStride         number of bytes in an input pixel row
   p8BitMask            output image
   uOutRowStride        number of bytes in an output pixel row
   uStartX              point at which to start the matching, X
   uStartY              point at which to start the matching, Y
   clr                  color to match against (index 0..255 if grayscale)

   uFillMode            0 - selection area is bounded by 
                        pixels matching clr

                        1 - all contiguous pixels matching 
                        the start pixel are selected

   uTolerance           
                        for RGB/RGBA:
                           0 - only match specified color 
                           (strict color matching)
                              ...up to...
                           3 * (255 * 255) - maximum tolerance
                           (black matches white)

                        for grayscale:
                           0..255, maximum color difference


   uFlags               bit      purpose
                        ---      -------
                        0        set when image is in BGR (or BGRA) order
                                 (ignored if grayscale) 

                        1        set output pixels to 0,255 instead of 0,1

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_GetColorMatchMaskFromImage )(const BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *p8BitMask, UINT32 uOutRowStride, const POINT * pptCenter, COLORREF clr, UINT32 uFillMode, UINT32 uClrTolerance, UINT32 uFlags);

/*-------------------------------------------------------
   is6_GetHSVColorMatchMaskFromImage

   Purpose : 
   Get an 8-bit mask image by matching colors in the source image
   against an input color, with variable tolerance. Source image 
   must be in the HSV or HSL colorspaces.

   See is6_RGBToHSV, is6_RGBToHSL.

   This can be used to do things like a "magic wand" selection.

   The output image is filled with "0" where selection (or filling)
   does not occur and "1" where selection does occur. See uFlags.

   Note:
   If you are using an HSV image as input, you need to make sure the
   vOrL parameter is between 0 and 255, and if the image is in HSL,
   the vOrL parameter must be between 0 and 1. The H and S parameters
   are in the same ranges for both HSL and HSV.

   This is similar to is6_GetColorMatchMaskFromImage.

   Param                Use
   ------------------------------------
   pImage               source grayscale, HSV or HSL image
   uWidth               width in pixels
   uHeight              height   
   uInRowStride         number of doubles in an input pixel row
   p8BitMask            output image
   uOutRowStride        number of bytes in an output pixel row

   uStartX              point at which to start the matching, X
   uStartY              point at which to start the matching, Y
   
   h                    Hue of target color
   s                    Saturation of target color
   vOrL                 Value or Lightness of target color

   uFillMode            0 - selection area is bounded by 
                        pixels matching h,s,v color. this behaves like
                        a flood fill starting at the start pixel, and
                        where the boundary of the fill is the color 
                        you specify.

                        1 - all contiguous pixels matching 
                        the start pixel are selected. this behaves like
                        a flood fill where the boundary is colors
                        which do not match the starting pixel.

                        2 - all pixels which match the tolerance
                        are set to 1 (or 255).

                        3 - all pixels which match the tolerance
                        are set to the difference between the target
                        Hue and the Hue at the current pixel (scaled to
                        0..255, where 255 = perfect match, 0 = no match)

                        4 - all pixels which match the tolerance
                        are set to the difference between the target
                        Saturation and the Saturation at the current pixel 
                        (scaled to 0..255, where 255 = perfect match, 
                        0 = no match)

                        5 - all pixels which match the tolerance
                        are set to the difference between the target
                        Value (or Lightness) and the Value (or Lightness)
                        at the current pixel (scaled to 0..255, 
                        where 255 = perfect match, 0 = no match)

   fHTolerance          tolerance 0..180 to allow for Hue matching
   fSTolerance          tolerance 0..1 to allow for Saturation matching

   fVOrLTolerance       tolerance for Value or Lightness.
                        0..255 (for HSV) or 0..1 (for HSL) . 

   uFlags               bit      meaning
                        ---      -------
                         0       set if input is HSL (affects output in modes 3,4,5)
                         1       set output pixels to 0,255 instead of 0,1  .
                                 this applies to modes 0 and 1, only.


   Return
   ------
      FALSE on failure
-------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_GetHSVColorMatchMaskFromImage )(double *pHSV, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *p8BitMask, UINT32 uOutRowStride, UINT32 uStartX, UINT32 uStartY,  double h, double s, double vOrL, UINT32 uFillMode, double fHTolerance, double fSTolerance, double fVOrLTolerance, UINT32 uFlags);

/*---------------------------------------------------------
   is6_PolygonFillImage / _is6_PolygonFillImage

   Purpose : 
   Perform a polygon fill onto an image.

   Note:
   If filling with an image, the bottom image and the
   overlay image must have the same number of bytes per
   pixel. 

   Note:
   When filling an 8-bit image with another 8-bit image, the 
   fill image must use the same palette as the bottom image.

   Note:
   If you don't supply a pallete when filling an 8-bit image 
    with an opacity value other than 1.0, this function will 
    assume the image is an 8-bit grayscale image. This will cause 
    unexpected results if the image is not a grayscale image. To 
    avoid confusion, you should always supply a pallete if you are 
    filling 8-bit colormapped images.

   Param                Use
   ------------------------------------
   pImage               image
   uWidth               width in pixels
   uHeight              height 
   uBytesPerPixel       bytes per pixel (1, 3 or 4)
   uInRowStride         number of bytes in a pImage pixel row

   pPoints              array of XYdouble values describing the 
                        polygon to fill   

   uNumPoints           points in the pPoints array

   pOverlayImage        image to be used as the fill pattern. NULL,
                        if filling a solid color.

                        image tiling starts at the minimum X,Y value
                        in pPoints.

                        if uBytesPerPixel = 4, the image is not alpha-blended.
                        it is simply copied onto the output image.

   uOverlayWidth        image width in pixels
   uOverlayHeight       image height 
   uOverRowStride       number of bytes in an overlay image row

   uColorSpec           fill color, if not filling with an image.
                        if uBytesPerPixel = 1, the low byte of 
                        this value is used as the output.

                        if uBytesPerPixel = 3, this value is treated 
                        as a COLORREF value.

                        if uBytesPerPixel = 4, this value is treated 
                        as a COLORREF value with the high BYTE set to 
                        the output alpha value.

   fOpacity             0 to 1. 1 = solid, 0 = transparent. not supported
                        when using an overlay image.

   pPal                 pallete for colormapped images (uBytesPerPixel=1). see Notes.

   uPalColors           entries in pPal

   uFlags               bit      purpose
                        ---      -------
                        0        use opacity value for filling. see Notes for palette info

                        1        set when image is in BGR (or BGRA) order

                        2        smooth the polygon edges
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_PolygonFillImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, XYdouble *pPoints, UINT32 uNumPoints, BYTE *pOverImage, UINT32 uOverW, UINT32 uOverH, UINT32 uOverRowStride, UINT32 uClrSpec, double fOpacity, RGBQUAD *pPal, UINT32 uPalColors, UINT32 uFlags);

/*---------------------------------------------------------
   is6_PolygonFillImage_16 / _is6_PolygonFillImage_16

   Purpose : 
   Perform a polygon fill onto a 16-bit-per-component image.

   Param                Use
   ------------------------------------
   pImage               image
   uWidth               width in pixels
   uHeight              height 
   uComponentsPerPixel  components per pixel (1, 3 or 4)
   uInRowStride         number of components in a pImage pixel row

   pPoints              array of XYdouble values describing the 
                        polygon to fill   

   uNumPoints           points in the pPoints array

   pColorSpec           pointer to an ISRGBA16 structure. this is the
                        fill color.

                        if components = 1, the fill color comes from the
                        'r' member.

   fOpacity             0 to 1. 1 = solid, 0 = transparent. not supported
                        when using an overlay image.

  uFlags                bit      purpose
                        ---      -------
                        0        use opacity value for filling.

                        1        set when image is in BGR (or BGRA) order
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_PolygonFillImage_16 )(ISUINT16 *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, XYdouble *pPoints, UINT32 uNumPoints, ISRGBA16 *pClrSpec, double fOpacity, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPolygonScanlineArray / _is6_GetPolygonScanlineArray

   Purpose : 
   Given a set of points that describe a polygon, returns arrays 
   that represent the endpoints of the horizontal scanlines that 
   make up the filled polygon.

   Param                Use
   ------------------------------------
   pInPoints            array of XYdouble values describing the 
                        polygon to fill   

   uNumInPoints         points in the pPoints array

   puRowArray           receives an array of Y-coordinates, one per scanline
   puDoubleLeftXArray   receives array of left endpoints of the filled lines
   puDoubleRightXArray  receives array of right endpoints of the filled lines

   puSegments           receives number of line segments to fill (this is also
                        the size of the previous three arrays)

   uFlags               unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_GetPolygonScanlineArray )(XYdouble *pInPoints, UINT32 uNumInPoints, HGLOBAL *puRowArray, HGLOBAL *puDoubleLeftXArray, HGLOBAL *puDoubleRightXArray, UINT32 *puNumRows, UINT32 uFlags);

/*---------------------------------------------------------
   is6_PolygonFill1Bit / _is6_PolygonFill1Bit

   Purpose : 
   Perform a polygon fill onto a 1-bit image.

   Image rows must start on even byte boundaries.

   Param                Use
   ------------------------------------
   pImage               image
   uWidth               width in pixels
   uHeight              height       
   uRowStride           number of bytes in a pixel row

   pPoints              array of XYdouble values describing the 
                        polygon to fill
   uNumPoints           points in the pPoints array

   uColor               color to fill with. 1 or 0

   uFlags               unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_PolygonFill1Bit )(BYTE *p1Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, XYdouble *pPoints, UINT32 uNumPoints, UINT32 uColor, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DrawEllipseOnImage / _is6_DrawEllipseOnImage

   Purpose : 
   Draw an ellipse on an image. 
   
   Param             Use
   ------------------------------------
   pImage            image
   uWidth            image width in pixels
   uHeight           image height 
   uBytesPerPixel    number of bytes in an image pixel (1,2,3,4)   
   uRowStride        number of bytes in an image row

   uColorSpec        if uBytesPerPixel = 1, 
                     the low byte of this value is used as
                     the output.

                     if uBytesPerPixel = 2, 
                     the low word of this value is used as
                     the output.

                     if uBytesPerPixel = 3, 
                     this value is treated as a COLORREF value.

                     if uBytesPerPixel = 4, 
                     this value is treated as a COLORREF value.
                     by default, alpha byte is ignored. see uFlags.

   pptCenter         center point
   uHorizRadius      horizontal radius
   uVertRadius       vertical radius

   bFill             1 to fill, 0 for outline

   fOpacity          0 to 1. 1 = solid, 0 = transparent.
                     does not apply when uBytesPerPixel = 2.

   pPal              use when filling. only applies when
                     uBytesPerPixel = 1 and bit 0 of uFlags is set.

   uPalColors        entries in pPal

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR (or BGRA) order

                     1        use opacity value for filling. 
                              if uBytesPerPixel = 1, image is assumed to be
                              grayscale. pallete is ignored. 

                     2        treat the high byte in uColorSpec as an alpha
                              value and write that value to the alpha channel
                              of the output (uBytesPerPixel = 4 only).

                     3        draw with anti-aliasing
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_DrawEllipseOnImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uColorSpec, const POINT * pptCenter, UINT32 uHorizRadius, UINT32 uVertRadius, BOOL bFill, double fOpacity, RGBQUAD *pPal, UINT32 uPalColors, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DrawEllipseOn1Bit / _is6_DrawEllipseOn1Bit

   Purpose : 
   Draw an ellipse on an image. 
   
   Image rows must start on even byte boundaries.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            image width in pixels
   uHeight           image height 
   uRowStride        number of bytes in an image row
   uColor            1 or 0, color to draw with

   pptCenter         center point
   uHorizRadius      horizontal radius
   uVertRadius       vertical radius

   bFill             1 to fill, 0 for outline

   uFlags            currently unused
                     
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_DrawEllipseOn1Bit )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uColor, const POINT * pptCenter, UINT32 uHorizRadius, UINT32 uVertRadius, BOOL bFill, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DrawLineOnImage / _is6_DrawLineOnImage

   Purpose : 
   Draw a solid line on an image

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height 
   uBytesPerPixel    number of bytes in an image pixel (1,2,3,4)
   
   uRowStride        number of bytes in a pixel row

   uColorSpec        if uBytesPerPixel = 1, 
                     the low byte of this value is used as
                     the output.

                     if uBytesPerPixel = 2, 
                     the low word of this value is used as
                     the output.

                     if uBytesPerPixel = 3, 
                     this value is treated as a COLORREF value.

                     if uBytesPerPixel = 4, 
                     this value is copied to the output.

   pptStart          starting point
   pptEnd            end point

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR (or BGRA) order

                     1        smooth line appearance (3 or 4 bytes per pixel only)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_DrawLineOnImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uColorSpec, const POINT * pptStart, const POINT * pptEnd, UINT32 uFlags);


/*---------------------------------------------------------
   is6_DrawThickLineOnImage / _is6_DrawThickLineOnImage

   Purpose : 
   Draw a solid line on an image, with variable thickness and
   selectable end styles.

   The line will be centered on the line from pxyStart to
   pxyEnd, with half the specified thickness on either side.

   All arrowheads will end on the line end points. 

   Note:
   If you don't supply a pallete when drawing on an 8-bit image 
   with an opacity value other than 1.0, this function will 
   assume the image is an 8-bit grayscale image. This will cause 
   unexpected results if the image is not a grayscale image. To 
   avoid confusion, you should always supply a pallete if you are 
   drawing onto 8-bit colormapped images.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height 
   uBytesPerPixel    number of bytes in an image pixel (1,3,4)
   
   uRowStride        number of bytes in a pixel row

   uColorSpec        if uBytesPerPixel = 1, 
                     the low byte of this value is used as
                     the output.

                     if uBytesPerPixel = 3, 
                     this value is treated as a COLORREF value.

                     if uBytesPerPixel = 4, 
                     this value is copied to the output.

   pPal              pallete for colormapped images (uBytesPerPixel=1). see Notes.

   uPalColors        entries in pPal

   fOpacity          0 to 1. 1 = solid, 0 = transparent. 

   pxyStart          starting point
   pxyEnd            end point

   fLineWidth        width of the line, in pixels. if not using edge smoothing
                     (see uFlags), this may not give good results for values
                     less than 1.

   endStyle0         select arrow style at the end of pxyStart.
                         0: square end (no arrow)
                         1: simple arrow
                         2: barbed arrow
                         3: square end
                         4: diamond end
                         5: notched arrow

   endStyle1         select arrow style at the end of pxyEnd. see endStyle0.

   arrowLength       length of arrow heads. size of diamond & square
   arrowWidth        width of arrow heads.

   uFlags            bit      purpose
                     ---      -------
                     0        use opacity value. see Notes for palette info
                     1        set when image is in BGR (or BGRA) order
                     2        smooth the edges
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_DrawThickLineOnImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uColorSpec, RGBQUAD *pal, UINT32 uPalColors, double fOpacity, const XYdouble * pxyStart, const XYdouble * pxyEnd, double fLineWidth, UINT32 endStyle0, UINT32 endStyle1, UINT32 arrowLength, UINT32 arrowWidth, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DrawLineOn1Bit / _is6_DrawLineOn1Bit

   Purpose : 
   Draw a solid line on a 1 bit image

   Image rows must be (int)((uWidth + 7) / 8) bytes wide.
   This ensures that rows start on even byte boundaries.

   Param             Use
   ------------------------------------
   pImage               image
   uWidth               width in pixels
   uHeight              height 
   uRowStride           bytes per pixel row
   uColor               color to draw. 1 or 0.

   pptStart             starting point
   pptEnd               end point

   uFlags               not currently used

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_DrawLineOn1Bit )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uColor, const POINT * pptStart, const POINT * pptEnd, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CountImageColors / _is6_CountImageColors

   Purpose : 
   Count colors in an RGB or RGBA image.
   If RGBA, the A channel is ignored.

   Param             Use
   ------------------------------------
   pRGB              image
   uWidth            width in pixels
   uBytesPerPixel    bytes per image pixel, 3 or 4.
   uRowStride        bytes per pixel row
   uHeight           height 

   uFlags            bit         purpose
                     ---         -------
                      0          if set, uses a faster process 
                                 that requires more memory.
   
   Return
   ------
      number of colors in image
---------------------------------------------------------*/
_ISDeclSpec UINT32      _ISCConv _ISFn(is6_CountImageColors)(const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetFirstNColorsInRGB / _is6_GetFirstNColorsInRGB

   Purpose : 
   Retrieve the first uColors colors from an
   RGB image.

   Stops when uColors unique colors are found,
   or when all pixels have been examined.

   Param             Use
   ------------------------------------
   pRGB              image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per pixel row
   uColors           number of colors to get
   pPal              uColors RGBQUADs

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR (or BGRA) order


   Return
   ------
      number of colors found
---------------------------------------------------------*/
_ISDeclSpec UINT32      _ISCConv _ISFn(is6_GetFirstNColorsInRGB)(const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FindFirstUnusedRGBColor / _is6_FindFirstUnusedRGBColor

   Purpose : 
   Find the first unused color in an RGB image.

   Param             Use
   ------------------------------------
   pRGB              image
   uWidth            width in pixels
   uBytesPerPixel    bytes per image pixel, 3 or 4.
   uRowStride        bytes per pixel row
   uHeight           height 

   uFlags            bit         purpose
                     ---         -------
                      0          if set, uses a faster process 
                                 that requires more memory.

   Return
   ------
      number of colors found
---------------------------------------------------------*/
_ISDeclSpec COLORREF    _ISCConv _ISFn(is6_FindFirstUnusedRGBColor)(const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FindColorsUsedIn8BitImage / _is6_FindColorsUsedIn8BitImage

   Purpose : 
   Find which colors are used in an 8-bit image. 

   This scans the image, and records the number 
   of times each color index is found in the image

   Param             Use
   ------------------------------------
   p8Bit             8-bit input image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per pixel row
   puColors          array of 256 UINT32s, allocated by caller.
                     Records the number of times the color is 
                     found in the image
   
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_FindColorsUsedIn8BitImage)(const BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 *puColors);

/*---------------------------------------------------------
   is6_ColorReplaceImage / _is6_ColorReplaceImage

   Purpose : 
   Replace one color with another in an image.

   If uBytesPerPixel = 4, the default behavior is to ignore
   the 4th when comparing colors (it is assumed to be the alpha 
   channel). Also, by default this will not modify the 4th channel.

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height 
   uBytesPerPixel    1, 3, 4  
   uRowStride        bytes per pixel row

   clrOld            color to replace. if uBytesPerPixel = 1, this is
                     the palette index.

   clrNew            color to replace with. if uBytesPerPixel = 1, this 
                     is the palette index.

   uTolerance        maximum color distance to match 
                     (used when uBytesPerPixel = 3, only)

                     0 - only match specified color 
                        (strict color matching)

                     3 * (255 * 255) + 1 
                       - maximum tolerance
                        (black matches white)

                     see uFlags, bit 3.
   
   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR (or BGRA) order

                     1        set if you want to replace the alpha channel
                              in the output with the high byte of clrNew

                     2        if set, and if uBytesPerPixel = 4, the function 
                              will compare all 4 bytes of the pixel, and will 
                              replace all 4 bytes. tolerance is ignored (assumed 0)

                     3        if set, and if uBytesPerPixel = 4, the function 
                              will compare all 4 bytes of the pixel, and will 
                              replace all 4 bytes. tolerance is in 
                              the range 4 * (255 * 255) + 1. this is not intended
                              for use in RGBA images.
 
                              
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_ColorReplaceImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, COLORREF clrOld, COLORREF clrNew, UINT32 uTolerance, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ColorReplaceImageRangeSpecified / _is6_ColorReplaceImageRangeSpecified

   Purpose : 
   Replace one color with another in an image, with individual
   tolerances for each channel, instead of a combined distance 
   for all channels, as with is6_ColorReplaceImage.

   The color is replaced if all the values match the target color
   within the specified tolerances.

   Ex.
      replace a color in an RGBA image, ignoring the alpha channel
      // search for RGBA(128,255,128,0)
      BYTE target[4] = {128,255,128,0};

      // replace with RGBA(128,128,128,255)
      BYTE newVal[4] = {128,128,128,255};

      // allow a tolerance of +/- 16 for each channel
      BYTE tol[4] = {100,100,100,16};

      // but only look at the first three channels. 
      // a match occurs when these channels all fall within
      // +/- 16 of the target value. channel 4 is ignored.
      // 
      // set bits 0,1,2
      BYTE channelMask = (1 | 2 | 4); 

      // do it
      is6_ColorReplaceImageRangeSpecified( pImg, w, h, 4, w * 4, target, newVal, tol, channelMask, 0);


   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        bytes per pixel row
   uBytesPerPixel    1, 2, 3, 4  

   pTargetArray      array of up to 4 bytes that specify
                     the color to search for.

   pNewValArraycolor array of up to 4 bytes that specify
                     the color to replace with.

   pToleranceArray   array of uBytesPerPixel BYTEs which specify 
                     the tolerance allowed for each channel.
   
   bChannelMask      controls which channels to process. If bit 
                     N is set, channel N is processed. If bit 
                     N is not set, channel N is skipped (the channel
                     is ignored for comparison, and is not modified for
                     output). Only bottom four bits are used.
                     
   uFlags            bit      purpose
                     ---      -------
                     unused         
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_ColorReplaceImageRangeSpecified )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, BYTE *pTargetArray, BYTE *pNewValArray, BYTE *pToleranceArray, BYTE bChannelMask, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetBorder / _is6_GetBorder

   Purpose : 
   Using color matching, this function will attempt to 
   find the coordinates of the image within an image border.

   For instance, this can be used to do an "auto-crop" of a 
   scanned image. Scanned pictures often come from the scanner
   slightly rotated and bordered with a border of scanner 
   backing. This function will attempt to find the picture 
   space vs. the border space. You can then use the is6_Crop* 
   functions to remove the border area.

   ** 8 and 24-bit images only **
   This function can attempt to determine the color of the 
   background on its own, if you don't want to supply a
   border color of your own. To do this, the function 
   examines the four corners. If three of the four corners 
   match, within the tolerance specified, their color is 
   chosen as the border color. If three of the four corners 
   do not match, this function will fail. is6_GetLastError
    will return IS_ERR_CANTFINDBORDER.

   After determining the border color, this function scans 
   along the edges of the image, stopping in each direction 
   when a pixel is detected that does not match the border 
   color, given the specified tolerance.

   ** 1 bit images only **
   For one bit images, this expects that you are providing the
   background color in the uColorSpec parameter. It will not 
   attempt to determine a background color on its own.

   Param             Use
   ------------------------------------
   pImg              image
   uWidth            width in pixels
   uHeight           height 
   uBitsPerPixel     bits per pixel in image (1, 8 or 24)
   uRowStride        bytes per pixel row
   pPal              palette (for 8-bit images)
   
   uTolerance        color matching tolerance (ignored for 1-bit)
                     0 = strict color matching
                     3 * (255 * 255) = maximum tolerance
                        (black matches white)

   uColorSpec        border color (if you want to force your own)
                     COLORREF if 24-bit, palette index if 8-bit.

                     for one bit images, this must be 1 or 0.

   pImageRect        receives coordinates of the non-border image section
   
   uFlags            bit      purpose
                     ---      -------
                      0       use your color as the border color, instead
                              of the "three-corners" method. 8 and 24 bpp only.
                              1-bit always uses your color as border color.

                      1       set when image is in BGR (or BGRA) order
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_GetBorder)(const BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, UINT32 uRowStride, RGBQUAD *pPal, UINT32 uTolerance, UINT32 uColorSpec, RECT *pImageRect, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Create8BitMaskFromColor / _is6_Create8BitMaskFromColor

   Purpose : 
   Given an input RGB image, find an 8-bit mask that 
   corresponds to a single color. Useful for finding
   tranparency masks from RGB images.

   Param             Use
   ------------------------------------
   pRGB              RGB image
   uWidth            width in pixels
   uHeight           height 
   uBytesPerPixel    3 or 4
   uInRowStride      number of bytes in a source image row
   pMask             8 bit mask image (allocated by caller. uOutRowStride * uHeight bytes) 
   uOutRowStride     number of bytes in output image row
   clrMatch          color to match
   uTolerance        maximum color distance to match

                     0 - only match specified color 
                        (strict color matching)

                     3 * (255 * 255) - maximum tolerance
                        (black matches white)

   uOutputLevel      value to set in output upon color match

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR (or BGRA) order
   
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL    _ISCConv _ISFn(is6_Create8BitMaskFromColor)(const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pMask, UINT32 uOutRowStride, COLORREF clrMatch, UINT32 uTolerance, UINT32 uOutputLevel, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Create1BitMaskFromColor / _is6_Create1BitMaskFromColor

   Purpose : 
   Given an 8-bit image, create a 1-bit image that represents all
   pixels from the 8-bit image which are set to a specific value.

   This is the opposite of is6_SetColorFrom1BitMask.

   Param             Use
   ------------------------------------
   pInput8           8-bit image
   uWidth            width in pixels
   uHeight           height 
   uInRowStride      number of bytes in a source image row
   p1Mask            1 bit mask image (allocated by caller. uOutRowStride * uHeight bytes) 
   uOutRowStride     number of bytes in output image row (typically (w + 7) / 8 bytes )
   clrMatch          color to match

   uFlags            bit      purpose
                     ---      -------
                     unused
   
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL     _ISCConv  _ISFn(is6_Create1BitMaskFromColor)(const BYTE *pInput8, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *p1Mask, UINT32 uOutRowStride, BYTE colorMatch, UINT32 uFlags);

/*---------------------------------------------------------
   is6_SetColorFrom1BitMask / _is6_SetColorFrom1BitMask

   Purpose : 
   Given a 1-bit mask image, and an 8-bit output image,
   set all pixels in the 8-bit image to a specific value,
   wherever the corresponding pixel in the mask image is set - all
   other pixels are left unmodified.

   This is the opposite of is6_Create1BitMaskFromColor.

   Param             Use
   ------------------------------------
   p1mask            1-bit mask image
   uWidth            width in pixels
   uHeight           height 
   uInRowStride      number of bytes in a source image row (typically (w + 7) / 8 bytes )
   pOutput8          output 8-bit image (allocated by the caller)
   uOutRowStride     number of bytes in output image row 
   clrMatch          color to set in output

   uFlags            bit      purpose
                     ---      -------
                     unused
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL     _ISCConv  _ISFn(is6_SetColorFrom1BitMask)(const BYTE *p1mask, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOutput8, UINT32 uOutRowStride, BYTE colorSet, UINT32 uFlags);

/*---------------------------------------------------------
   is6_QuantizeRGBTo8Bit / _is6_QuantizeRGBTo8Bit

   Purpose : 
   Create an 8 bit image by performing color quantization
   and dithering on an RGB/BGR image.

   This is equivalent to calling is6_Get8BitPaletteFromRGB 
   followed by is6_RGBTo8BitDithered.

   See is6_Get8BitPaletteFromRGB for notes on the "octree 
   reduction" method.

   Param             Use
   ------------------------------------
   pImage            source RGB image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes in an input pixel row
   p8Bit             output image
   uOutRowStride     bytes in an output pixel row
   uColors           colors in output image
   pPal              receives palette of 8 bit image
   uDitherStyle      0 - two-value dither
                     1 - four-value (Floyd-Steinberg) dither

   uDitherErrorMax   Set the maximum error value that 
                     can be carried between pixels. Higher 
                     values will result in images with a more
                     grainy appearance. Lower values tend to 
                     generate large areas of solid color.
                     Valid values are 0 to 127.
                     20 is a good default.

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR order
                     8        use octree reduction to determine the palette
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_QuantizeRGBTo8Bit)(const BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *p8Bit, UINT32 uOutRowStride, UINT32 uColors, RGBQUAD *pPal, UINT32 uDitherStyle, UINT32 uDitherErrorMax, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBTo8BitBestFit / _is6_RGBTo8BitBestFit

   Purpose : 
   Create an 8 bit image by doing a best-fit (Euclidean
   distance) match of the pixels in the RGB to the values
   in the palette. Errors in color matching are ignored.

   Param             Use
   ------------------------------------
   pImage            source RGB image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   p8Bit             output 8-bit image
   uOutRowStride     bytes in an output pixel row
   uColors           colors in palette
   pPal              target palette

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR order

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_RGBTo8BitBestFit)(const BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *p8Bit, UINT32 uOutRowStride, UINT32 uColors, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBTo8BitDithered / _is6_RGBTo8BitDithered

   Purpose : 
   Create an 8 bit image by doing a best-fit (Euclidean
   distance) match of the pixels in the RGB to the values
   in the palette. Uses the dithering step from 
   is6_QuantizeRGBTo8Bit to diffuse color matching errors.


   Param             Use
   ------------------------------------
   pImage            source RGB image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   p8Bit             output 8-bit image
   uOutRowStride     bytes in an output pixel row
   uColors           colors in palette
   pPal              target palette
   uDitherStyle      0 - two-value dither
                     1 - four-value (Floyd-Steinberg) dither

   uDitherErrorMax   Set the maximum error value that 
                     can be carried between pixels. Higher 
                     values will result in images with a more
                     grainy appearance. Lower values tend to 
                     generate large areas of solid color.

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR order

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_RGBTo8BitDithered)(const BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *p8Bit, UINT32 uOutRowStride, UINT32 uColors, RGBQUAD *pal, UINT32 uDitherStyle, UINT32 uDitherErrorMax, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Get8BitPaletteFromRGB / _is6_Get8BitPaletteFromRGB

   Purpose : 
   Find a palette that corresponds to the input RGB
   image. This palette can contain up to 256 colors.

   The optional "octree reduction" method is significantly 
   slower than the default method. But, it does offer one
   advantage: in certain situations, the default method may
   return a palette with fewer colors than you request,
   even if the image contains more colors than that. This 
   behavior is intentional. The octree method will provide
   a palette with exactly the requested number of colors (if 
   the input image has that many colors, of course).

   We recommend using the default method. In most cases, 
   it's difficult to see a difference between images created
   using the two methods, but the default method is many times
   faster than the octree method.

    If you are using the default method, you can use a 32-bit RGBA
    image as the input. The function will disregard the alpha channel 
    completely. Note that this could lead to undesired results, if 
    the alpha channel is masking large areas of color which aren't 
    in the unmasked area.

   Param             Use
   ------------------------------------
   pImage            source RGB image
   uWidth            width in pixels
   uHeight           height
   uRowStride        bytes per pixel row
   uColors           colors in output image
   pPal              receives palette
   pColorsFound      receives number of colors in the palette

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGR order
                     1        input is RGBA
                     2        use octree reduction to determine the palette
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn(is6_Get8BitPaletteFromRGB)(const BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uColors, RGBQUAD *pPal, UINT32 *puColorsFound, UINT32 uFlags);

/*---------------------------------------------------------
   is6_QuantizeRGBATo8Bit / _is6_QuantizeRGBATo8Bit

   Purpose : 
   Create an 8 bit image by performing color quantization
   and dithering on an RGBA (or BGRA) image. Preserves alpha 
   information in the rgbReserved member of each palette 
   entry.


   Param             Use
   ------------------------------------
   pRGBA             source RGBA image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input image row
   p8Bit             output image
   uOutRowStride     bytes per output image row
   uColors           colors in output image
   pPal              receives palette of 8 bit image
   bFSDither         if TRUE, performs Floyd-Steinberg dithering
                     if FALSE, no dithering is performed
   puColorsUsed      receives the number of colors actually used

   uFlags            bit      purpose
                     ---      -------
                     0        set when image is in BGRA order

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_QuantizeRGBATo8Bit)(const BYTE *pRGBA, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut8, UINT32 uOutRowStride, UINT32 uColors, RGBQUAD *pal, BOOL bFSDither, UINT32 *puColorsUsed, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBTo1Bit / _is6_RGBTo1Bit

   Purpose : 
   Create a 1 bit image from an RGB image. Output is 1 bit
   per pixel, packed.

   Param             Use
   ------------------------------------
   pRGB              source RGB image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input image row
   p1Bit             output image
   uOutRowStride     bytes per output image row
   uK                constant. see uFlags

   uFlags            bit      purpose
                     ---      -------
                      0       set if image is in BGR order

                      1       dithering selector.

                              if clear, image is converted
                              by simple thresholding. all pixels
                              with brightness above uK are set to 1
                              and all others to 0.

                              if set, output image is dithered
                              to distribute color matching errors.

                              uK is the dithering error maximum (see
                              is6_RGBTo8BitDithered)

                      2       if set, input colors are matched to
                              black and white. if not set, colors are
                              matched to colors from a two-color 
                              pallete that is derived from the colors in 
                              the input image.
                              only used when dithering is enabled.
                              
                      3       use 2-value dither. default is 4-value (Floyd-Steinberg)
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_RGBTo1Bit)(const BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *p1Bit, UINT32 uOutRowStride, UINT32 uK, UINT32 uFlags);

/*---------------------------------------------------------
   is6_PolygonWarpImage / _is6_PolygonWarpImage

   Purpose :
   Warp a convex quadrilateral section of a source image to a
   convex section on another image.

   inQuad specifies a quardilateral in the source image.

   outQuad specifies the quadrilateral in the output image.

   The image from the source is warped to fit into the output image.

   Both quadrilaterals must be convex. This will return an error
   code if they aren't. See is6_PolygonIsConvex.

   Note:
   For 8-bit images with a palette, the only interpolation modes
   available are 0 (no interpolation) and 1 (bi-linear).

   Note:
   If no palette is passed for an 8-bit image, this will assume
   the image is 8-bit grayscale. This will allow the use of all
   interpolation methods and will avoid doing the expensive palette
   color matching that a paletted 8-bit image requires.

   Param             Use
   ------------------------------------
   pInput            input image
   uInWidth          width, in pixels, of input image
   uIntHeight        height of input image
   uInRowStride      bytes per input pixel row

   pOutput           output image
   uOutWidth         width, in pixels, of output image
   uOutHeight        height of output image
   uOutRowStride     bytes per output pixel row

   inQuad            source quadrilateral

   outQuad           output quadrilateral

   uBitsPerPixel     bits per pixel in the images
   8, 24 and 32 currently supported.

   pPal              palette for 8-bit images. see Notes.
   uPalColors        number of colors in palette

   uMode             interpolation mode:
                     0 : fast (no interpolation)

                     1 : bi-linear interpolation.
                        this is slower but gives better results than
                        the default method.

                     2 : bi-cubic interpolation.
                        this is slower than bi-linear, but gives
                        sharper results. does not work with 8-bit images.

                     3 : Elliptical Weighted Average. uses
                        an elliptical Gaussian window to sample source
                        pixels. the results are somewhat fuzzier than
                        the other methods, but in areas of high
                        reduction or magnification, it's far superior.
                        it also has much better anti-aliasing.

   uFlags            bit          use
                     ---          -------
                     20           disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv  _ISFn(is6_PolygonWarpImage)(BYTE *pInput, UINT32 uInWidth, UINT32 uInHeight, UINT32 uInRowStride,
                              BYTE *pOutput, UINT32 uOutWidth, UINT32 uOutHeight, UINT32 uOutRowStride,
                              const ISQuadrilateral *inQuad, const ISQuadrilateral *outQuad,
                              UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uPalColors, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is6_PolygonIsConvex / _is6_PolygonIsConvex

   Purpose : 
   Determine if a given polygon is "convex". The
   points MUST be sorted in clockwise order. 

   Param             Use
   ------------------------------------
   pPoints           array of POINT structures
   uPoints           number of points in the array

   Return
   ------
      TRUE if convex
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn( is6_PolygonIsConvex )(const POINT * pPoints, UINT32 uPoints);

/*---------------------------------------------------------
   is6_GetConvexHull / _is6_GetConvexHull

   Purpose : 
   Find the largest convex polygon that can be made from
   a set of input points.


   Param             Use
   ------------------------------------
   pInPoints         array of POINT structures
   uNumInPoints      number of points in pInPoints

   phOutPoints       receives the set of points that make
                     up the convex hull. the points are sorted
                     in clockwise order. caller must free this
                     memory.

   puNumHullPoints   receives the number of points in the hull.   

   Return
   ------
      TRUE if convex
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn( is6_GetConvexHull )(const POINT *pInPoints, UINT32 uNumInPoints, HGLOBAL *phOutPoints, UINT32 *puNumHullPoints);

/*---------------------------------------------------------
   is6_CreateColorMatcher / _is6_CreateColorMatcher

   Purpose : 
   Create an ImgSource color matching object. 
   
   The color matching object can help find the best match 
   for an arbitrary color inside a palette. It is designed 
   internally to be used when matching large numbers of 
   colors against a single palette, though it will work
   adequately when matching a single color, too.

   This object must be destroyed with is6_DestroyColorMatcher

   Param             Use
   ------------------------------------
   pPal              input palette
   uPalColors        number of coor entries in the palette (256 or fewer)

   Return
   ------
      HISCOLORMATCH, or NULL or failure
---------------------------------------------------------*/
_ISDeclSpec HISCOLORMATCH       _ISCConv _ISFn(is6_CreateColorMatcher)(const RGBQUAD *pPal, UINT32 uPalColors);

/*---------------------------------------------------------
   is6_FindClosestColor / _is6_FindClosestColor

   Purpose : 
   Find the closest match for the RGB color specified in the
   palette used when this color matching object was created. See
   is6_CreateColorMatcher.

   The match will be cached, so the next time you look up this 
   color, the match will be quicker.

   Param             Use
   ------------------------------------
   hISCM             color manager created with is6_CreateColorMatcher
   r,g,b             RGB color to match

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec __int32     _ISCConv _ISFn( is6_FindClosestColor )(HISCOLORMATCH hISCM, BYTE r, BYTE g, BYTE b);

/*---------------------------------------------------------
   is6_DestroyColorMatcher / _is6_DestroyColorMatcher

   Purpose : 
   Free the memory associated with an ImgSource color matching object. 

   Param             Use
   ------------------------------------
   hISCM             color manager created with is6_CreateColorMatcher

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_DestroyColorMatcher )(HISCOLORMATCH hISCM);

/*---------------------------------------------------------
   is6_ColorCompare / _is6_ColorCompare

   Purpose : 
   Compare two COLORREFs, see if they match within a 
   given tolerance.

   Param             Use
   ------------------------------------
   x, y              colors to compare
   uTolerance        maximum allowable distance.
                     0 = no tolerance, exact match only
                     ...
                     255 * 255 * 3 = black matches white

   Return
   ------
      TRUE if the color difference is within the given tolerance
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_ColorCompare )(COLORREF x, COLORREF y, UINT32 uTolerance);

/*---------------------------------------------------------
   is6_SwapPaletteEntriesIn8Bit / _is6_SwapPaletteEntriesIn8Bit

   Purpose : 
   Swap two colors in an 8-bit image. All pixels with
   idx1 will be set to idx2 and all idx2 will be set to idx1. 
   And, this will swap the colors in the palette.

   Param             Use
   ------------------------------------
   p8                eight bit image
   uWidth            width in pixels
   uHeight           height in pixels
   pPal              array of RGBQUADs
   idx1              first color
   idx2              second color

   Return
   ------
      FALSE on failure.
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_SwapPaletteEntriesIn8Bit )(BYTE *p8, UINT32 uWidth, UINT32 uHeight, RGBQUAD *pPal, BYTE idx1, BYTE idx2);

/*---------------------------------------------------------
   is6_DoubleArrayOffset / _is6_DoubleArrayOffset

   Purpose : 
   Add and/or multiply the values in an array of doubles
   by an array of offsets/factors.

   Input is uWidth * uHeight * uChannels doubles

   The pAdd, pMul, pMin and pMax arrays must all contain 
   uChannels items.

   Param             Use
   ------------------------------------
   pDouble           double array
   uWidth            width in pixels, of image
   uHeight           height 
   uChannels         doubles per pixel
   uRowStride        number of doubles in an image row

   pAdd              array of doubles. one double per
                     channel. these are added to their 
                     corresponding channel.

   pMul              array of doubles. one double per
                     channel. these are multiplied against their 
                     corresponding channel.

   pMin              array of doubles. one double per
                     channel. these are the minimum values
                     allowed for the corresponding channel.

   pMax              array of doubles. one double per
                     channel. these are the maximum values
                     allowed for the corresponding channel.

  Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_DoubleArrayOffset )(double *pDouble, UINT32 uWidth, UINT32 uHeight, UINT32 uChannels, UINT32 uRowStride, double *pAdd, double *pMul, double *pMin, double *pMax);

/*---------------------------------------------------------
   is6_MakeRectFadeMask / _is6_MakeRectFadeMask

   Purpose : 
   Create an 8 bit buffer to be used as an edge fade mask.
   Can be used in conjunction with is6_CombineImageChannels
   to create an edge-faded alpha mask.

   Param             Use
   ------------------------------------
   p8BitMask         output 8-bit image
   uWidth            width in pixels
   uHeight           height
   fFadeWidthRatio   rate of fade ( < .5 )

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_MakeRectFadeMask )(BYTE *p8BitMask, UINT32 uWidth, UINT32 uHeight, double fFadeWidthRatio);


/*---------------------------------------------------------
   is6_RGB24ToRGB16 / _is6_RGB24ToRGB16

   Purpose : 
   Create a 16-bit RGB image from a 24-bit RGB image.
   
   Low bits are dropped.

   This function assumes the output buffer is large enough to
   hold all output pixels. 

   The size of the output buffer must be at least (uWidth * 
   uHeight * 2) BYTEs. The caller must allocate this buffer. 

   Note:
   is6_WriteBMP and is6_WriteTLA expect x555 buffers when writing 
   16 bit images.

   Param             Use
   ------------------------------------
   pRGB24            source image
   uWidth            width in pixels (of all buffers)
   uHeight           height (of all buffers)
   uInRowStride      number of bytes in an input row
   pRGB16            output 16 bit buffer
   uOutRowStride     number of bytes in an output row

   uFlags            bit      purpose
                     ---      -------
                      0       if set, create  xRRRRRGGGGGBBBBB (x555)
                              else create RRRRRGGGGGGBBBBB (565)

                      1       if set, set the high bit in x555 buffers to 1
                              else, set high bit to 0
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_RGB24ToRGB16 )(const BYTE *pRGB24, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pRGB16, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DespeckleImage / _is6_DespeckleImage

   Purpose : 
   Apply a despeckle filter to an image.

   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, etc.) 

   Best results are acheived by repeated applying this
   filter and inspecting the results.

   Callback is called once per channel

   Param                Use
   ------------------------------------
   pImage               input image
   uWidth               width in pixels
   uHeight              height
   uBytesPerPixel       number of bytes per pixel (max 8)
   uRowStride           rows per pixel

   uFlags               bit      purpose
                        ---      -------
                        0-7      These bits control which channels to
                                 process. If bit N is set, channel N is 
                                 processed. If bit N is not set, channel
                                 N is skipped (nothing is written to
                                 the output for that channel).

                                 Ex. 
                                 If uFlags = 1 | 2 | 4, channels 0, 1 and 2 
                                 (R, G and B) will be processed. 

                                 If uFlags = 1 | 4 , channels 0 and 2 
                                 (R and B) will be processed. Channel 1 (Green)
                                 will be skipped.

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_DespeckleImage )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FindNormalizedDifference / _is6_FindNormalizedDifference

   Purpose : 
   Construct an 8-bit image that is the normalized difference
   of two other 8-bit images. This implements the NDVI algorithm.

   For each pair of pixels in p1 and p2, the following calculation
   is performed:

      n = (p1[i] - p2[i])
      d = (p1[i] + p2[i])
      
      if d < dThreshold, d = -1.0

      t =  n / d
      t is then bound between dMin and dMax
      t is scaled from 0 to 255
      
      pOutput[i] = t


   Param             Use
   ------------------------------------
   p1                first 8-bit image 
   p2                second 8-bit image 
   pOutput           output 8-bit image 
   uWidth            width in pixels, of all images
   uHeight           height of all images
   dMin              minimum limit (0.0 -- 1.0)
   dMax              maximum limit (0.0 -- 1.0) 
   dThreshold        minimum denominator value (see above)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_FindNormalizedDifference )(const BYTE *p1, const BYTE *p2, UINT32 uWidth, UINT32 uHeight, BYTE *pOutput, double dMin, double dMax, double dThreshold);

/*---------------------------------------------------------
   is6_NormalizedDifferenceImage / _is6_NormalizedDifferenceImage

   Purpose : 
   Given two input images of identical size and bit depth, 
   create an output image which contains the difference in 
   corresponding pixel components of the two input,
   normalized to 0..255.

   This should only be used on continuous-tone images (RGBA,
   grayscale, RGB, etc.) 

   Param             Use
   ------------------------------------
   pImage1           first image 
   pImage2           second image 
   uWidth            width in pixels, of all images
   uHeight           height of all images
   uBytesPerPixel    bytes per pixel, all images
   uRowStride
   pOutput           output 8-bit image 
   uFlags            unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_NormalizedDifferenceImage )(const BYTE *pImage1, const BYTE *pImage2, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, BYTE *pImageOut, UINT32 uFlags);

/*---------------------------------------------------------
   is6_NormalizeImage / _is6_NormalizeImage

   Purpose :
   Normalize all pixels to the specified range.

   Note:
   If fLow or fHigh are out of range of the component
   type, the results will be unpredictable

   Param             Use
   ------------------------------------
   pImage            image
   uWidth            width in pixels
   uHeight           height 
   uBytesPerPixel    components per pixel
   uRowStride        components per row
   fLow              low bound
   fHigh             high bound

   uFlags            bit         purpose
                     ---         -------
                     9-10        00 - image has 8 bits per component  (BYTE)      (val=0)
                                 01 - image has 16 bits per component (ISUINT16)  (val=0x200)
                                 10 - image has 64 bits per component (double)    (val=0x400)
                                 11 - image has 32 bits per component (float)     (val=0x600)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_NormalizeImage)(void *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uRowStride, double fLow, double fHigh, UINT32 uFlags);

/*---------------------------------------------------------
   is6_EmbossRGB / _is6_EmbossRGB

   Purpose : 
   Generate an embossed image 

   Note:
   Typically, the input bump map is an 8 bit grayscale 
   version of the image you are embossing. This image 
   controls the peaks and valleys in the output image. 
   
   If the texture image is NULL  you will get the typical
   grayscale embossed output. If it is a pointer to the 
   24-bit source image, the original colors will be applied to 
   the output image, instead of plain gray. This is sometimes
   called "bump mapping" - the original image is given a texture.

   The texture and output images must be at least at large 
   (width and height) as the bump image.


   This function allows for the use of an optional 8-bit mask
   image, when using a texture image. Any place where the
   mask image is 0, the texture image will be duplicated unmodified.
   The mask image must have the same dimensions as the other images.

    Note:
    This process works by using the 3x3 area of pixels surrounding the 
    target pixel, to calculate the 'bump' angles. Obviously, this is
    a problem on edges (because at least 3 of those pixels are off the
    image). The default behavior is to skip the edges when generatnig the
    output image - edges are left with whatever data was in them
    before the function was called - leading to undesirable edges 
    around the image.   But, an option has been added to let the function 
    duplicate edge pixels in place of the off-edge pixels. This could cause 
    border artifacts, depending on the value of the bump mask edge pixels, 
    but they will be less offensive than just leaving the border as-is.

   Param                Use
   ------------------------------------
   pInBump8             input bump map (determines "height")
   uWidth               width in pixels, of all buffers
   uHeight              height of all buffers
   uBumpRowStride       bytes per row, bump map
   azimuth              angle of light source (radians)
   elevation            height of light source (radians)
   uFilterWidth         controls depth of effect 
   pTexture             24-bit texture image (NULL, if you want a gray output)
   uTextureRowStride    bytes per row in the texture image
   pOutRGB              output RGB image
   uOutRowStride        bytes per row in the output image
   pMask8               8-bit mask image (NULL, if you don't have a mask)
   uMaskRowStride       bytes per row of mask image

   uFlags               bit      purpose
                        ---      -------
                         0       use alternative blending option.
                                 this option produces a much nicer
                                 emboss effect, when using a texture
                                 image.

                         1      use edge-duplicating trick for
                                    edge pixels (see Note)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL    _ISCConv _ISFn( is6_EmbossRGB )(const BYTE *pInBump8, UINT32 uWidth, UINT32 uHeight, UINT32 uBumpRowStride, double azimuth, double elevation, UINT32 uWidth45, const BYTE *texture, UINT32 uTextureRowStride, BYTE *pOutRGB, UINT32 uOutRowStride, const BYTE *pMask8, UINT32 uMaskRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ApplyRedEyeFilter / _is6_ApplyRedEyeFilter

   Purpose : 
   Attempt to eliminate "red-eye" from a region of an RGB image.

   You must select an "eye" region from the image, then choose
   a saturation level and color range. These should be done 
   interactively, as the color and saturation of the "red-eye"
   effect varies greatly between images.
   
   Param             Use
   ------------------------------------
   pRGB              input RGB image
   uWidth            width in pixels
   uHeight           height 
   pRect              eye region

   uSatLevel         0..100. saturation threshold. 
                     50 is a good default. roughly
                     the percent that red is allowed 
                     to dominates the other colors 
                     before it is reduced.

   uColorRange       0..100. red band width. 50 is
                     a good default. higher values cause
                     pixels with more cyan and yellow to
                     be reduced.
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_ApplyRedEyeFilter )(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, const RECT * pRect, UINT32 uSatLevel, UINT32 uColorRange);

/*---------------------------------------------------------
   is6_PushImage / _is6_PushImage

   Purpose : 
   Perform a point to point warp of an image.

   This does not resize the image. It just smears the
   pixels around.

   Param             Use
   ------------------------------------
   pInBuf            input image
   pOutBuf           output image
   uWidth            width in pixels
   uHeight           height 

   uBytesPerPixel    1 = colormapped 8-bit image
                     3 = RGB image
                     4 = RGBA image

   uRowStride        number of bytes in an image  row
   pPal              image palette (8-bit only)
   uPalColors        colors in pPal
   ptFrom            push from here
   ptTo              push to here

   uFlags            bit        purpose
                     ---        -------
                      0         use bi-cubic interpolation

                      1         if set, do not modify alpha 
                                channel on RGBA images.

                     20         disable multithreaded processing
   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_PushImage )(const BYTE *pInBuf, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, BYTE *pOutBuf, RGBQUAD *pal, UINT32 uNumColors, const POINT * pptFrom, const POINT * pptTo, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Deinterlace / _is6_Deinterlace

   Purpose : 
   Deinterlace an interlaced image.

   This function assumes the image contains two interlaced
   frames. The resulting image will be the same dimensions 
   as the input image.
   
   Note:
   Image height must be an even number.

    Note:
   
   uFlags examples:
      // duplicate, retain odd
      uFlags = 0x00 | 0x04

      // interpolate, retain even
      uFlags = 0x01 | 0x00

      // merge, using bi-linear
      uFlags = 0x02 | 0x00
     
      // merge, using Lanczos filter
      uFlags = 0x02 | 0x08


   Param             Use
   ------------------------------------
   pImage            input image
   uWidth            width in pixels
   uHeight           height 

   uBytesPerPixel    1 = 8-bit grayscale image
                     3 = RGB image
                     4 = RGBA

   uRowStride        number of bytes in an image row

   uFlags            bit         purpose
                     ---         -------
                     1-0         select de-interlace method.

                                 00 - duplicate
                                 01 - interpolate              (val = 0x01)
                                 10 - merge                    (val = 0x02)
                                 11 - quick interpolate        (val = 0x03)

                     2-4         
                                 For duplicate & interpolate only:
                                 000 - retain even lines
                                 001 - retain odd lines        (val = 0x04)
                                       
                                 For merge only:
                                 000 - bi-linear interpolation
                                 001 - bi-cubic interpolation  (val = 0x04)
                                 010 - Lanczos filter          (val = 0x08)
                                 011 - Hanning filter          (val = 0x0c)

  Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_Deinterlace )(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DeinterlaceCollapse / _is6_DeinterlaceCollapse

   Purpose : 
   Extract a single frame from an interlaced image.

   This function assumes the image contains two interlaced
   frames. The output image will be half the height of the 
   input image.
   
   Note:
   Image height must be an even number.

   Param             Use
   ------------------------------------
   pImage            input image
   uWidth            width in pixels
   uHeight           height 

   uBytesPerPixel    1 = 8-bit grayscale image
                     3 = RGB image
                     4 = RGBA

   uRowStride        number of bytes in an image row

   uFlags            bit         purpose
                     ---         -------
                      2          0 - retain even lines
                                 1 - retain odd lines        (val = 0x04)
  Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL    _ISCConv _ISFn(is6_DeinterlaceCollapse)(const BYTE *pMem, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, BYTE *pOut, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FindDeskewAngleOneBit / _is6_FindDeskewAngleOneBit

   Purpose : 
   Given a 1-bit input image, attempt to find the angle
   at which this image has been rotated. 
   
   By default, this function assumes foreground pixels 
   have a value of 1, background is 0. This can change with uFlags.
      
   This function is intended to be used on images such as 
   FAX documents or scanned printed pages. It can be easily 
   fooled by large non-text sections or by long non-horizonal 
   lines.

   This function does not require the entire image to make its
   determination; you can pass it any image sub-section. For 
   example, if you know where text-only areas are likely to be 
   you can use just that section (ex. pass the top few lines 
   of a FAX image, if you can be sure the FAX timestamp text 
   will be there - do this by passing a height value that is 
   smaller than the whole image).

   Note:
   Large areas of high-density pixels (solid boxes, images, logos,
   etc.) can fool the algorithm. Running a morphological boundary 
   extraction on the image before doing this can remove those areas
   as well as highlight the edges - use a 3x3 kernel of 1s.

   Note:
   This function requires a large amount of memory (proportional to
   the input image size and the chosen resolution). 
   This is another good reason to limit the size of the image you 
   pass to it.

   Note:
   Increasing the uResolutionFactor value will not necessarily 
   increase the accuracy of the results. In fact, higher values
   may decrease accuracy on some images. It is recommended that you
   use a value of 1, for most purposes.

   Changing uPixelLimit allows you to control the number of 
   pixels used to generate the result. By using fewer 
   pixels, you can shorten the time required to get the result,
   though of course the accuracy may be slightly lessened in 
   some cases.


   Param             Use
   ------------------------------------
   p1Bit             input image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of bytes in an image row
   pRho              receives rotated angle (+/- 90 deg)

   uResolutionFactor resolution of results. 
                        1 = minimum 1 degree
                        2 = minimum 1/2 degree
                        3 = minimum 1/3 degree
                        etc..

                     the amount of memory required is directly proportional
                     to the dimensions of the image multiplied by this value.
   
   uLimitAngle       ignore angles beyond +/- this range

   uPixelLimit       maximum number of pixels to sample. (width * height) / 2
                     is a good place to start. 
                     uWidth * uHeight = sample all pixels

   uFlags            bit         purpose
                     ---         -------
                      0          assume background pixels are 1 and forground are 0

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL    _ISCConv _ISFn(is6_FindDeskewAngleOneBit)(const BYTE *p1Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, double *pRho, UINT32 uFactor, UINT32 uLimitAngle, UINT32 uPixelLimit, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Convert8bpcTo16bpc / _is6_Convert8bpcTo16bpc

   Purpose : 
   Create a 16 bit-per-channel version of an 
   8 bit-per-component image.

   Param             Use
   ------------------------------------
   pImageIn          input image
   uWidth            width in pixels
   uHeight           height 
   
   uChannelsPerPixel number of components in each 
                     pixel (1 for grayscale, 3 for RGB, etc)

   uInRowStride      number of components in an input image row
   pImageOut         output image
   uOutRowStride     number of components in an output image row

   uFlags            unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_Convert8bpcTo16bpc)(const BYTE *pImageIn, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uInRowStride, ISUINT16 *pImageOut, UINT32 uOutRowStride, UINT32 uFlags);


/*---------------------------------------------------------
   is6_Convert16bpcTo8bpc / _is6_Convert16bpcTo8bpc

   Purpose : 
   Create a 8 bit-per-component version of an 16 bit-per-component image.


   Param             Use
   ------------------------------------
   pImageIn          input image
   uWidth            width in pixels
   uHeight           height 
   
   uChannelsPerPixel number of components in each 
                     pixel (1 for grayscale, 3 for RGB, etc)

   uInBitsUsed       number of bits used in the input (9..16).
                     this determines the amount of scaling used.

   uInRowStride      number of components in an input image row
   pImageOut         output image
   uOutRowStride     number of components in an output image row

   uFlags            unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL _ISCConv  _ISFn(is6_Convert16bpcTo8bpc)(const ISUINT16 *pImageIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInBitsUsed, UINT32 uInRowStride, BYTE *pImageOut, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_IntegerToFloatingPoint

   Purpose : 
   Convert an array of BYTEs or ISUINT16s to doubles or floats.

   Callback is not used.

   Note:
   BYTE <-> float is SSE-accellerated

   Param             Use
   ------------------------------------
   pIn               input image, BYTEs or ISUINT16s
   uWidth            width in pixels
   uHeight           height 
   uComponentsPerPixel number of components per pixel
   uInRowStride      number of components per pixel row

   pOut              output image. allocated by caller.

   uOutRowStride     number of components in an output row

   fScale            multiply all values by this, after conversion.
    
   uFlags            bit        purpose
                     ---        -------
                     1          input is ISUINT16, otherwise input is BYTE
                     2          output is double, otherwise output is float

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv   _ISFn(is6_IntegerToFloatingPoint)(const void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, double fScale, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FloatingPointToInteger

   Purpose : 
   Convert an array of doubles or floats to BYTEs or ISUINT16s

   Callback is not used.

   Note:
   BYTE <-> float is SSE-accellerated

   Param             Use
   ------------------------------------
   pIn               input image
   uWidth            width in pixels
   uHeight           height 
   uComponentsPerPixel number of components per pixel
   uInRowStride      number of components per pixel row

   pOut              output image. allocated by caller.

   uOutRowStride     number of components in an output row

   fScale            multiply all input values by this, before conversion.

   uFlags            bit        purpose
                     ---        -------
                     1          input is double, otherwise input is float
                     2          output is ISUINT16, otherwise output is BYTE

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv   _ISFn(is6_FloatingPointToInteger)(const void *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, double fScale, UINT32 uFlags);

/*---------------------------------------------------------
   is6_SwapByteOrderIn16bpcImage / _is6_SwapByteOrderIn16bpcImage

   Purpose : 
   Reverse the order of the bytes in the components 
   of a 16-bit per component image.

   Param             Use
   ------------------------------------
   pImageIn          input image
   uWidth            width in pixels
   uHeight           height 
   
   uChannelsPerPixel number of components in each 
                     pixel (1 for grayscale, 3 for RGB, etc)

   uInRowStride      number of components in an input image row

   uFlags            unused

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL    _ISCConv _ISFn( is6_SwapByteOrderIn16bpcImage )(ISUINT16 *pImageIn, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uInRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ShiftImageSubPixel / _is6_ShiftImageSubPixel

   Purpose : 
   Shift the pixels in an image horizontally or vertically
    by less than one pixel.

    Note:
    If shitfing horizontally, the output image must be:
        (uInWidth + 1) x uInHeight pixels

    If shitfing vertically, the output image must be:
        uInWidth x (uInHeight + 1) pixels
        
    Note:
    If bit #1 is set, assumes pImageIn is an offset into the
    middle of the actual source image - at least far enough so that there
    is one pixel column one the left and one on the right for horizontal 
    shifts - or one row each on the top and bottom, for vertical shifts.

    This allows the function to assume it can reference pixels that lie 
    beyond the edges of the source image, and can use those pixels to blend on
    the edges instead of using the blending color.

    Example:
        This demonstrates the technique described in the Note, above:

        // start the shift at 10,10 in the source
        UINT32 xOffset = 10, yOffset = 10;

        // find the location of the pixel at 10,10
        BYTE * pInStart = pSourceImageStart + (xOffset * 3 + yOffset * (sourceWidth * 3));

        // process 100x100 pixels
        UINT32 procWidth = 100, procHeight = 100;

        UINT32 outWidth = procWidth + 1;
        UINT32 outHeight = procHeight;
        BYTE *pOutput = new ISNEWSPEC BYTE[outWidth * 3 * outHeight];

        is6_ShiftImageSubPixel(pInStart, procWidth, procHeight, 3, sourceWidth * 3,
                                                pOutput, outWidth * 3, 
                                                .75, 0, 2);

   For more information on this technique, see the "Row Stride" article 
    in the Help File.

   Param             Use
   ------------------------------------
   pImageIn          input image
   uInWidth          width in pixels or source image
   uInHeight         height 
   
   uChannelsPerPixel number of components in each 
                     (1,2,3 or 4)

   uInRowStride      number of BYTEs in an input image row

   pOutImage         output image (see Notes)
   uOutRowStride     number of BYTEs in an output image row
   fOffset           amount to shift: -1.0 ... 1.0
   uEdgeBlendClr     color to use at the edges of the image, for blending

   uFlags            bit         purpose
                     ---         -------
                      0          if 0, shift horizontally
                                 if 1, shift vertically

                      1          source is a sub-image. see Notes.


   Return
   ------
      FALSE on failure
---------------------------------------------------------*/ 
_ISDeclSpec BOOL _ISCConv  _ISFn( is6_ShiftImageSubPixel )(const BYTE *pInImage, UINT32 uInWidth, UINT32 uInHeight, UINT32 uChannelsPerPixel, UINT32 uInRowStride, BYTE *pOutImage, UINT32 uOutRowStride, double fOffset, UINT32 uEdgeBlendClr, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Get3DSplinePoints

   Purpose : 
   Interpolate the points on a B-spline, given a number of control
   points and a degree value.

   If you need 2D points, set all Z values to 0 for the 
   input points. 

   The degree controls the smoothness of the curve. If degree is 2,
   the curve will pass through all points exactly, and all segments 
   between will be straight lines. The higher the degree, the smoother 
   the curve.

   Code based on :
   http://astronomy.swin.edu.au/~pbourke/curves/spline/

   Param             Use
   ------------------------------------
   uOutPoints        number of points to calculate
   uInPoints         number of control points (at least 3)
   uDegree           degree of curve. usually 3 or 4. (must be less than the number of points)
   inX               X values of control points (array of uInPoints doubles)
   inY               Y values of control points 
   inZ               Z values of control points
   outX              X values of output points (array of uOutPoints doubles)
   outY              Y values of output points 
   outZ              Z values of output points 

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_Get3DSplinePoints )(UINT32 uOutPoints, UINT32 uInPoints, UINT32 uDegree, double *inX, double *inY, double *inZ, double *outX, double *outY, double *outZ);

/*---------------------------------------------------------
   is6_SimpleWhitePointCorrection

   Purpose : 
   Adjust the colors in an image so that the specified color 
   becomes white ( RGB(255,255,255), for 8-bpc images ). 

   This could be used for a "white eyedropper" tool, where
   the user selects a region of the image which should be
   white. The color of the pixel(s) in the selected region 
   is then passed to this function.

   This gives best results when the image suffers from a distinct 
   color cast, as with an image taken by a digital camera using 
   an incorrect white balance setting.

   Callback is not used

   Param             Use
   ------------------------------------
   pInput            input image (RGB24, RGB32, RGB48 or RGB64)
   uWidth            width in pixels
   uHeight           height 
   
   uChannelsPerPixel number of components in each 
                     pixel (3 for RGB, 4 for RGBA)

   uRowStride        number of components in an image row

   wr                the current red value of the color which should be white
   wg                the current green value of the color which should be white
   wb                the current blue value of the color which should be white

   uFlags            bit      purpose
                     ---      -------
                     9-10     00 - images have 8 bits per component  (BYTE) (val=0)
                              01 - images have 16 bits per component (ISUINT16) (val=0x200)

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_SimpleWhitePointCorrection)(void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelsPerPixel, UINT32 uRowStride, int wr, int wg, int wb, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBToXYZ

   Purpose : 
   Convert an RGB image to the XYZ colorspace. 

   Note:
       The pM parameter specifies the RGB working space matrix 
       used when converting to XYZ. It is an array of 9 doubles.
       
       The default colorspace, which is used when you pass NULL 
       for the pM parameter, is sRGB (non-D50). The conversion 
       matrix is:

        const double sRGBM[9] = 
         {
            0.4360747,  0.3850649,  0.1430804,
            0.2225045,  0.7168786,  0.0606169,
            0.0139322,  0.0971045,  0.7141733
         };

       Matricies for other colorspaces can be found 
       at http://www.brucelindbloom.com/ - (Math / RGB To XYZ)

   Callback is not used.
   
   Also see is6_RGBToXYZ

   Param             Use
   ------------------------------------
   pInput            input image (RGB24, RGBA32, RGB48 or RGB6A4)
   uWidth            width in pixels
   uHeight           height 
   
   uInCPP            number of components in each input
                     pixel (3 for RGB, 4 for RGBA)

   uRowStride        number of components in an image row

   pOut              output XYZ image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.
                     output is clamped to the range 0.0-1.0.

   uOutRowStride     number of doubles in an output row

   pM                RGB->XYZ conversion matrix. See notes.

   uCompanding       0-3. determines the RGB inverse companding 
                     method used. 
                              
                     0 - none
                     1 - inverse sRGB 
                     2 - inverse L*
                     3 - inverse gamma

                     be sure to use the same value when calling
                     is6_XYZToRGB.

   fGamma            gamma value to use with inverse gamma companding

   uFlags            bit      purpose
                     ---      -------

                     9        input image has 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_RGBToXYZ)(void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInCPP, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double *pM, UINT32 uCompanding, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_XYZToRGB

   Purpose : 
   Convert an XYZ image to the RGB colorspace. 

   Note:
       The pM parameter specifies the RGB working space matrix 
       used when converting from XYZ. It is an array of 9 doubles,
       and should be the inverse of the matrix used when converting
       RGB to XYZ.

       The default colorspace, which is used when you pass NULL 
       for the pM parameter, is sRGB (non-D50). The conversion 
       matrix is:
       

         const double inv_sRGB[9] = 
         {
            3.1338561, -1.6168667, -0.4906146,
           -0.9787684,  1.9161415,  0.0334540,
            0.0719453, -0.2289914,  1.4052427
         };
   
   Callback is not used.

   Also see is6_XYZToRGB

   Param             Use
   ------------------------------------
   pInput            input XYZ image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output RGB image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 BYTEs.

   uOutCPP           number of components in each 
                     output pixel (3 for RGB, 4 for RGBA).
                     if RGBA, the A channel is not modified.   

   uOutRowStride     number of components in an output row

   pM                XYZ->RGB conversion matrix. See notes.

   uCompanding       0-3. determines the RGB companding 
                     method used. 
                              
                     0 - none
                     1 - sRGB 
                     2 - L*
                     3 - gamma

                     be sure to use the same value when calling
                     is6_RGBToXYZ.

   fGamma            gamma value to use with gamma companding

   uFlags            bit      purpose
                     ---      -------
                     9        output image has 16 bits per component

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_XYZToRGB)(double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, void *pOut, UINT32 uOutCPP, UINT32 uOutRowStride, double *pM, UINT32 uCompanding, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_XYZToLab

   Purpose : 
   Convert an XYZ image to the CIE Lab colorspace. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input XYZ image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output Lab image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row

   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   uFlags            bit        purpose
                     ---        -------
                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_XYZToLab)(double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LabToXYZ

   Purpose : 
   Convert a CIE Lab image to the XYZ colorspace. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input Lab image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output XYZ image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row

   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   uFlags            bit        purpose
                     ---        -------
                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_LabToXYZ)(double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_XYZToLuv

   Purpose : 
   Convert an XYZ image to the CIE Luv colorspace. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input XYZ image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output Luv image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row

   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   uFlags            bit        purpose
                     ---        -------
                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_XYZToLuv)(double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LuvToXYZ

   Purpose : 
   Convert a CIE Luv image to the XYZ colorspace. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input Luv image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output XYZ image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row

   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   uFlags            bit        purpose
                     ---        -------
                     20          disable multithreaded processing


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_LuvToXYZ)(double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBToYCC

   Purpose : 
   Convert an RGB image to the YCbCr colorspace. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of BYTEs in an input image row

   pOut              output YCC image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles
                     or uWidth * uHeight * 3 floats (see uFlags)

   uOutRowStride     number of components (either float or double) in an output row

   uFlags            bit        purpose
                     ---        -------
                     11         if set, output to 32-bit floats, 
                                otherwise output to 64-bit doubles

                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_RGBToYCC)(const BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_YCCToRGB

   Purpose : 
   Convert an image in the YCbCr colorspace to RGB

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input YCbCr image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of components (float or double) in an input image row

   pOut              output RGB image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 BYTEs

   uOutRowStride     number of BYTEs in an output row

   uFlags            bit        purpose
                     ---        -------
                     11         if set, input is 32-bit floats, 
                                otherwise input is 64-bit doubles

                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_YCCToRGB)(const void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LabToLCH

   Purpose : 
   Convert a CIE Lab image to the LCH(ab) colorspace. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input Lab image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output LCH image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row

   uFlags            bit        purpose
                     ---        -------
                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_LabToLCH)(const double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LCHToLab

   Purpose : 
   Convert an image from the LCH(ab) colorspace to CIE Lab. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input LCH image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output Lab image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row

   uFlags            bit        purpose
                     ---        -------
                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_LCHToLab)(const double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LuvToLCH

   Purpose : 
   Convert a CIE Luv image to the LCH(uv) colorspace. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input Luv image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output LCH image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row

   uFlags            bit        purpose
                     ---        -------
                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_LuvToLCH)(const double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LCHToLuv

   Purpose : 
   Convert an image from the LCH(uv) colorspace to CIE Luv. 

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input LCH image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output Luv image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row

   uFlags            bit        purpose
                     ---        -------
                     20          disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL  _ISCConv  _ISFn(is6_LCHToLuv)(const double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBToLab

   Purpose : 
   Convert an image from RGB-24 to the CIE Lab colorspace.

   This is equivalent to RGB->XYZ->Lab, but only requires
   one pass over the input and output images (versus doing
   is6_RGBToXYZ and then is6_XYZToLab).

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of bytes in an input image row

   pOut              output Lab image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row


   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   pM                XYZ->RGB conversion matrix. See is6_RGBToXYZ.

   uCompanding       0-3. determines the RGB companding 
                     method used. See is6_RGBToXYZ.

   fGamma            See is6_RGBToXYZ

   uFlags            bit        purpose
                     ---        -------
                     9          input image has 16 bits per component, else 8
                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_RGBToLab )(const void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, double *pM, UINT32 uCompanding, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LabToRGB

   Purpose : 
   Convert an image from the CIE Lab colorspace to RGB-24 (or RGB-48).

   This is equivalent to Lab->XYZ->RGB, but only requires
   one pass over the input and output images (versus doing
   is6_LabToXYZ then is6_XYZToRGB).

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input Lab image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output RGB image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 components).

   uOutRowStride     number of components in an output row

   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   pM                XYZ->RGB conversion matrix. See is6_RGBToXYZ.

   uCompanding       0-3. determines the RGB companding 
                     method used. See is6_RGBToXYZ.

   fGamma            See is6_RGBToXYZ

   uFlags            bit        purpose
                     ---        -------
                     9          if set, output image has 16 bits per component, else 8
                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_LabToRGB )(const double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, double *pM, UINT32 uCompanding, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LuvToRGB

   Purpose : 
   Convert an image from RGB-24 or RGB-48 to the CIE Luv colorspace.

   This is equivalent to RGB->XYZ->Luv, but only requires
   one pass over the input and output images (versus doing
   is6_RGBToXYZ and then is6_XYZToLuv).

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of components in an input image row

   pOut              output Luv image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row


   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   pM                XYZ->RGB conversion matrix. See is6_RGBToXYZ.

   uCompanding       0-3. determines the RGB companding 
                     method used. See is6_RGBToXYZ.

   fGamma            See is6_RGBToXYZ

   uFlags            bit        purpose
                     ---        -------
                     9          input image has 16 bits per component, else 8
                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_RGBToLuv )(const void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, double *pM, UINT32 uCompanding, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LabToRGB

   Purpose : 
   Convert an image from the CIE Luv colorspace to RGB-24 (or RGB-48).

   This is equivalent to Luv->XYZ->RGB, but only requires
   one pass over the input and output images (versus doing
   is6_LuvToXYZ then is6_XYZToRGB).

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input Luv image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output RGB image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 components).

   uOutRowStride     number of components in an output row

   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   pM                XYZ->RGB conversion matrix. See is6_RGBToXYZ.

   uCompanding       0-3. determines the RGB companding 
                     method used. See is6_RGBToXYZ.

   fGamma            See is6_RGBToXYZ

   uFlags            bit        purpose
                     ---        -------
                     9          if set, output image has 16 bits per component, else 8
                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_LuvToRGB )(const double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, double *pM, UINT32 uCompanding, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RGBToLCH

   Purpose : 
   Convert an image from RGB-24 or RGB-48 to the LCH(ab) or LCH(uv) colorspace.

   This is equivalent to RGB->XYZ->Lab->LCH(ab) (or 
   RGB->XYZ->Luv->LCH(uv)), but only requires one pass over the 
   input and output images (versus doing is6_RGBToXYZ, is6_XYZToLab, is6_LabToLCH).

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input RGB image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of components in an input image row

   pOut              output LCH image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 doubles.

   uOutRowStride     number of doubles in an output row


   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   pM                XYZ->RGB conversion matrix. See is6_RGBToXYZ.

   uCompanding       0-3. determines the RGB companding 
                     method used. See is6_RGBToXYZ.

   fGamma            See is6_RGBToXYZ

   uFlags            bit        purpose
                     ---        -------
                     0          if set, output is LCH(uv), else LCH(ab)
                     9          input image has 16 bits per component, else 8
                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_RGBToLCH )(const void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, double *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, double *pM, UINT32 uCompanding, double fGamma, UINT32 uFlags);

/*---------------------------------------------------------
   is6_LCHToRGB

   Purpose : 
   Convert an image from the LCH(ab) or LCH(uv) colorspace to RGB-24 (or RGB-48).

   This is equivalent to LCH(ab)->Lab->XYZ->RGB (or LCH(uv)->Luv->XYZ->RGB), 
   but only requires one pass over the input and output images (versus doing
   is6_LCHToLab, is6_LabToXYZ then is6_XYZToRGB).

   Callback is not used.

   Param             Use
   ------------------------------------
   pInput            input LCH image
   uWidth            width in pixels
   uHeight           height 
   uRowStride        number of doubles in an input image row

   pOut              output RGB image. allocated by caller.
                     must be (at least) uWidth * uHeight * 3 components).

   uOutRowStride     number of components in an output row

   pXYZ_WhitePoint   a pointer to an XYZ triplet (three doubles: X, Y, Z) 
                     which represent the reference XYZ white point.
                     if NULL, the default d65 whitepoint is used.

   pM                XYZ->RGB conversion matrix. See is6_RGBToXYZ.

   uCompanding       0-3. determines the RGB companding 
                     method used. See is6_RGBToXYZ.

   fGamma            See is6_RGBToXYZ

   uFlags            bit        purpose
                     ---        -------
                     0          if set, input is LCH(uv), else LCH(ab)
                     9          if set, output image has 16 bits per component, else 8
                     20         disable multithreaded processing

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_LCHToRGB )(const double *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, double *pXYZ_WhitePoint, double *pM, UINT32 uCompanding, double fGamma, UINT32 uFlags);


/*---------------------------------------------------------
   is6_RGBToHSL / _is6_RGBToHSL

   Purpose : 
   Convert an RGB image to the HSL or HSV colorspace.

   Note:
   If the RGB values are equal, this function will return a Hue of 0.

   Note:
   For float and double output S, L and V are scaled between 0 and 
   1, H is scaled between 0 and 360.
   
   Note:
   If output is 16bpc, H will be in the range 0 to 36000; S and V,L will 
   be in the range 0 to 10000.
   
   Note:
   If bits 9 & 10 of uFlags are 0,0 the output image is assumed to
   be 64-bit double.

   Param             Use
   ------------------------------------
   pInput            input RGB image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      width of input pixel row, in bytes
   pOut              output HSL/HSV image. allocated by caller
   uOutRowStride     number of components in an output row

   uFlags			   bit      use
					      ---      ---
                      1       output is HSV

                     9-10     01 - output has 16 bits per component (ISUINT16) (val=0x200)
                              10 - output has 64 bits per component (double) (val=0x400)
                              11 - output has 32 bits per component (float) (val=0x600)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_RGBToHSL )(const BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, void *pOut, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_HSLToRGB / _is6_HSLToRGB

   Purpose : 
   Convert an HSL or HSV image to RGB.

   Note:
   If bits 9 & 10 of uFlags are 0,0 the intput image is assumed to
   be 64-bit double.

   Note:
   If input is 16bpc, H is assumed to be in the range 0 to 36000; S and L / V
   is assumed to be in the range 0 to 10000.

   See is6_RGBToHSL

   Param             Use
   ------------------------------------
   pInputD           input HSL image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      number of components in an input row
   pOutput           output RGB image (BYTEs) . allocated by caller
   uOutRowStride     width of output pixel row in bytes

   uFlags			   bit      use
					      ---      ---
                     9-10     01 - input has 16 bits per component (ISUINT16) (val=0x200)
                              10 - input has 64 bits per component (double) (val=0x400)
                              11 - input has 32 bits per component (float) (val=0x600)

   Return
   ------
      FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_HSLToRGB )(const void *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOutput, UINT32 uOutRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CompareColorsDeltaE

   Purpose : 
   Calculate the CIE DeltaE 2000 color distance between two colors.

   Colors may be in any of the following color spaces: RGB-24,
   XYZ, Lab, LCH. But colors must be in the same color space.

   Internally, colors will be converted to the LCH colorspace
   for comparison: RGB -> XYZ -> Lab -> LCH

   For RGB->XYZ, the default colorspace (sRGB) is assumed.
   For XYZ->Lab, the default whitepoint (D65) is used.
   
   Callback is not used.

   Param             Use
   ------------------------------------
   A1, A2, A3        components of the first color
   B1, B2, B3        components of the second color

   uFlags            0 - colors are RGB-24 (components in the range 0..255)
                     1 - colors are XYZ
                     2 - colors are Lab
                     3 - colors are LCH

   Return
   ------
   CIE DeltaE 2000 color distance between the two colors
---------------------------------------------------------*/
_ISDeclSpec double _ISCConv  _ISFn(is6_CompareColorsDeltaE)(double A1, double A2, double A3, double B1, double B2, double B3, UINT32 uFlags);

#ifdef __cplusplus
}
#endif
#endif