/*---------------------------------------------------------
    ISFileUtil.h : Part of the ImgSource library
    
    Image file utilities

    This code copyright (c) 2000-2016, Smaller Animals Software, Inc..

    No portion of this code may be copied or distributed without 
    permission of Smaller Animals Software, Inc..
    http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is6_FILEUTILSJPGH
#define is6_FILEUTILSJPGH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_SetJPGProgressiveReadCallback / _is6_SetJPGProgressiveReadCallback

   Set the progressive JPG callback function.

   Progressive JPGs can be read as a sequence of progressively
   sharper images. These images get sharper as more JPG data is 
   fed into the decoder, and the first images are ready after
   relatively little data has been read. This makes it useful 
   over slow connections.

   In ImgSource, progressive JPG reading is handled by way
   of a callback function that receives a pointer to the
   current image data after each pass.

   Ex.
   
      // the callback function
      BOOL    CALLBACK JPGcb(const HGLOBAL hImageData, 
                             const UINT32 uWidth, 
                             const UINT32 uHeight, 
                             const UINT32 uBPP, 
                             const UINT32 uPass, 
                             const IS_CALLBACK_DATA uUserData)
      {
         is6_DrawRGB(hDC, (BYTE *)hImageData, uWidth, uHeight, uWidth * 3, 0, 0, 0);

         return TRUE;
      }

      ...

      // set the callback
      is6_SetJPGProgressiveReadCallback(JPGcb, 0);

      ...

      // read the file - JPGcb will be called once per pass
      UINT32 w,h;
      HISSRC hSrc = is6_OpenFileSource(filename);

      // note we're setting bit 5 in uFlags, to enable the progressive callback
      HGLOBAL hImg = is6_ReadJPG(hSrc, &w1, &h1, 24, (1<<5));
      is6_CloseSource(hSrc);
      GlobalFree(hImg);

   Note:
   The memory handle returned to the callback function is the same
   that will be returned from is6_ReadJPG.
   
   Note:
   Do not delete, modify or deallocate the image data returned to the
   callback function.

   Note:
   If your callback returns FALSE, file reading will stop.

   Param             Use
   ------------------------------------
   pCB               callback function
   uUserData         user data. this data is thread safe.

---------------------------------------------------------*/
_ISDeclSpec void _ISCConv  _ISFn( is6_SetJPGProgressiveReadCallback )(is6_JPG_PROGREAD_CBPTR pCB, IS_CALLBACK_DATA uUserData);

/*---------------------------------------------------------
   is6_ReadJPGMarkers / _is6_ReadJPGMarkers

   Read JPEG comment and application markers.

   A JPG file may contain any number of text (JPEG_COM)
   and application (JPEG_APPx) markers. These are places
   for applications to store metadata. For example, IPTC and
   EXIF data reside in JPEG_APPx markers.

   This function causes ImgSource to read all comment and
   application markers. They are stored by ImgSource until:
   
      1) the next call to this function
         OR
      2) is6_ClearJPGInputText and is6_ClearJPGInputMarkers are called

   See is6_GetJPGInputTextCount, is6_GetJPGInputText,
   is6_ClearJPGInputText, is6_ClearJPGInputMarkers,
   is6_GetJPGInputMarker, etc.

   Param             Use
   ------------------------------------
   hSource           valid source manager
   uFlags            unused

  Return
     FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_ReadJPGMarkers )(HISSRC hSource, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetJPGInputTextCount                                                                              / _is6_GetJPGInputTextCount                                                                             

   Purpose :
   Get count of text strings read by last JPG file read.

   Return
     number of strings 
---------------------------------------------------------*/
_ISDeclSpec UINT32        _ISCConv _ISFn( is6_GetJPGInputTextCount )();

/*---------------------------------------------------------
   is6_GetJPGInputText / _is6_GetJPGInputText

   Purpose :
   Fetch string read by last JPG read.

   Param            Use
   ------------------------------------
   uIndex           index of string to fetch

   Return
     HGLOBAL pointer to string.
     caller must use GlobalFree to delete this object.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_GetJPGInputText )(UINT32 uIndex);

/*---------------------------------------------------------
   is6_ClearJPGInputText / _is6_ClearJPGInputText

   Purpose :
   Clear all JPG input strings

   Return
     TRUE if ok
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ClearJPGInputText )();

/*---------------------------------------------------------
   is6_GetJPGInputMarkerCount / _is6_GetJPGInputMarkerCount

   Purpose :
   Get count of JPG special markers read by last JPG file 
   (or dimension) read.
   These are application-specific data chunks.

   Return
     number of markers
---------------------------------------------------------*/
_ISDeclSpec UINT32        _ISCConv _ISFn( is6_GetJPGInputMarkerCount )();

/*---------------------------------------------------------
   is6_GetJPGInputMarkerInfo / _is6_GetJPGInputMarkerInfo

   Purpose :
   Get info about a single marker, as read by last JPG read.

   Note:
   The reported size will include the four bytes used for 
   marker length and type.

   Param            Use
   ------------------------------------
   uIndex           index of marker to fetch
   puDataLen        receives data length in bytes

   puDataType       marker type value : 
                     JPEG_APP1 = 0xE1
                     JPEG_APP2 = 0xE2
                     ...up to 
                     JPEG_APP14

   Return
   TRUE on success
   
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn( is6_GetJPGInputMarkerInfo )(UINT32 uIndex, UINT32 *puDataLen, UINT32 *puDataType);

/*---------------------------------------------------------
   is6_GetJPGInputMarker / _is6_GetJPGInputMarker

   Purpose :
   Fetch JPG marker data read by last JPEG read. 


   Note : This data is application-specific. 
   ImgSource does not and cannot interpret the data in 
   any way.

   Note:
   The first four bytes of the data that's returned 
   contain the data length and marker ID. If you want to 
   write this data back out to a JPG, you should skip 
   these first four bytes.


   Param            Use
   ------------------------------------
   uIndex           index of marker to fetch
   phData           receives HGLOBAL to data
   puDataLen        receives data length in bytes

   puDataType       marker type value : 
                     JPEG_APP1 = 0xE1
                     JPEG_APP2 = 0xE2
                     ...up to 
                     JPEG_APP14

   Return
   TRUE on success
   caller must use GlobalFree to delete the phData memory.
   
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetJPGInputMarker )(UINT32 uIndex, HGLOBAL *phData, UINT32 *puDataLen, UINT32 *puDataType);

/*---------------------------------------------------------
   is6_ClearJPGInputMarkers / _is6_ClearJPGInputMarkers

   Purpose :
   Clear all JPG input marker data

   Return
     TRUE if ok
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ClearJPGInputMarkers )();

/*---------------------------------------------------------
   is6_AddJPGOutputText / _is6_AddJPGOutputText

   Purpose :
   Add a string to be written to all subsequent JPG files
   as a JPEG_COM comment.

   Param                   Use
   ------------------------------------
   pText                   zero-terminated string to write

   Return
     FALSE on error
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_AddJPGOutputText )(const char *pText);

/*---------------------------------------------------------
   is6_ClearJPGOutputText / _is6_ClearJPGOutputText

   Purpose :
   Clear all JPG output strings

   Return
     TRUE if ok
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ClearJPGOutputText )();

/*---------------------------------------------------------
   is6_AddJPGOutputMarker / _is6_AddJPGOutputMarker

   Purpose :
   Add JPG app marker to be written on next JPG write. 

   Param            Use
   ------------------------------------
   pData            data
   uDataLen         data length in bytes

   uDataType        marker type value : 
                     JPEG_APP1 = 0xE1
                     JPEG_APP2 = 0xE2
                     ...up to 
                     JPEG_APP14

   Return
     FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_AddJPGOutputMarker )(const BYTE *pData, UINT32 uDataLen, UINT32 uDataType);

/*---------------------------------------------------------
   is6_SetJPGOutputMarker / _is6_SetJPGOutputMarker

   Purpose :
   Change the data in a specific JPG output marker.

   Param            Use
   ------------------------------------
   pData            data
   uDataLen         data length in bytes

   uDataType        marker type value : 
                     JPEG_APP1 = 0xE1
                     JPEG_APP2 = 0xE2
                     ...up to 
                     JPEG_APP14

   Return
     FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv _ISFn(is6_SetJPGOutputMarker)(UINT32 uIdx, const BYTE *pData, UINT32 uDataLen, UINT32 uDataType);

/*---------------------------------------------------------
   is6_GetJPGOutputMarkerCount / _is6_GetJPGOutputMarkerCount

   Purpose :
   Get count of JPEG application markers that have been added
   via is6_AddJPGOutputMarker or 
   is6_CopyJPGInputMetadataToOutputMetadata. 

   Param            Use
   ------------------------------------
   none

   Return
     number of markers
---------------------------------------------------------*/
_ISDeclSpec UINT32       _ISCConv _ISFn( is6_GetJPGOutputMarkerCount )();

/*---------------------------------------------------------
   is6_GetJPGOutputMarker / _is6_GetJPGOutputMarker

   Purpose :
   Get a copy of a JPEG data marker previously added by
   is6_AddJPGOutputMarker, is6_CopyJPGInputMetadataToOutputMetadata,
   or is6_SetJPGOutputMarker.

   Note:
   Unlike is6_GetJPGInputMarker, the first four bytes 
   of the data that's returned DO NOT contain the 
   data length and marker ID. This returns the data 
   without headers.

   Param            Use
   ------------------------------------
   uIndex           index of marker to fetch
   phData           receives HGLOBAL to data
   puDataLen        receives data length in bytes

   puDataType       marker type value : 
                     JPEG_APP1 = 0xE1
                     JPEG_APP2 = 0xE2
                     ...up to 
                     JPEG_APP14
   Return
   TRUE on success
   caller must use GlobalFree to delete the phData memory.
   
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_GetJPGOutputMarker )(UINT32 uIndex, HGLOBAL *phData, UINT32 *puDataLen, UINT32 *puDataType);

/*---------------------------------------------------------
   is6_ReplaceJPGOutputMarker / _is6_ReplaceJPGOutputMarker

   Purpose :
   Replace the first JPEG_APPx output marker of the specified data type 
   with new data. If a marker with the specified data type is not found,
   a new marker is created (as if you had called is6_AddJPGOutputMarker)

   Param             Use
   ------------------------------------
   uDataType         marker type value : 
                        JPEG_APP1 = 0xE1
                        JPEG_APP2 = 0xE2
                        ...up to 
                        JPEG_APP14

   pData             new data (NULL to clear)
   uDataLen          data length in bytes
   uFlags            unused

   Return
     return TRUE if marker was found and replaced, FALSE if marker not found
---------------------------------------------------------*/
_ISDeclSpec BOOL        _ISCConv  _ISFn(is6_ReplaceJPGOutputMarker)(UINT32 uDataType, const BYTE *pData, UINT32 uDataLen, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CopyJPGInputMetadataToOutputMetadata / _is6_CopyJPGInputMetadataToOutputMetadata

   Purpose :
   Copy the JPG markers (JPEG_COM and JPEG_APPx) read by the
   last is6_ReadJPGMarkers call into the output markers.

   This will makes it easy to copy such data from an input
   file to an output file.

   Return
     none
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_CopyJPGInputMetadataToOutputMetadata )();

/*---------------------------------------------------------
   is6_ClearJPGOutputMarkers / _is6_ClearJPGOutputMarkers

   Purpose :
   Clear all JPG output marker data

   Return
     TRUE if ok
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ClearJPGOutputMarkers )();

/*---------------------------------------------------------
   is6_SetJPGDCT / _is6_SetJPGDCT

   Switch the compression method : dct is one of :

   0 - default (usually JDCT_ISLOW)
   1 - JDCT_ISLOW: slow but accurate integer algorithm
   2 - JDCT_IFAST: faster, less accurate integer method
   3 - JDCT_FLOAT: floating-point method

   All JPG files written will be written using this setting.

   Param                   Use
   ------------------------------------
   uDct                    level to set
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_SetJPGDCT )(UINT32 uDCT);

/*---------------------------------------------------------
   is6_SetJPGSmoothingFactor / _is6_SetJPGSmoothingFactor

   Set JPG output smoothing factor.

   Setting a value for this causes the JPG library to apply
   a smoothing effect to output images. This can reduce the
   file size, at the expense of image quality. 

   All JPG files written will be written using this setting.

   Param                   Use
   ------------------------------------
   uSmoothing              1..100, or 0 for no input smoothing
    
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_SetJPGSmoothingFactor )(UINT32 uSmoothing);

/*---------------------------------------------------------
   is6_SetJPGDownsampleFactors / _is6_SetJPGDownsampleFactors

   Set JPG downsample factors.

   The default smoothing factors for RGB images are :

    component 1      H=2,   V=1
    component 2      H=1,   V=1
    component 3      H=1,   V=1
    component 4..10  H=0,   V=0

   This setup will produce images with better color 
   quality :

    component 1      H=1,   V=1
    component 2      H=1,   V=1
    component 3      H=1,   V=1
    component 4..10  H=0,   V=0

   This setup will produce smaller files with reduced 
   image quality :

    component 1      H=2,   V=2
    component 2      H=1,   V=1
    component 3      H=1,   V=1
    component 4..10  H=0,   V=0

   According to JpegLib valid values are 1..4 for all components, 
   though only the three setups above appear to work for RGB images;
   all other combinations produce an error in JpegLib. 

   Grayscale images can use the full range of values.

   ImgSource currently uses the first three values for RGB 
   images and the first value (only) for grayscale images. 
   However, to allow future expansion, this function requires
   that you supply all 10 components. You may set the unused 
   components to 0.

   If either uHFactors or uVFactors is NULL, ImgSource will
   use the default factors.

   All JPG files written will be written using this setting.

   Param               Use
   ------------------------------------
   uHFactors           array of 10 UINTs (1..4)
   uVFactors           array of 10 UINTs (1..4)
    
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_SetJPGDownsampleFactors )(UINT32 *uHFactors, UINT32 *uVFactors);

/*---------------------------------------------------------
   is6_SetJPGDCTScaleFactors / _is6_SetJPGDCTScaleFactors

   Allow the JPG reader to perform simple image scaling during
   reading. This can be useful for quick image previews.

   The JPG reader can scale the image by 1/2, 1/4 or 1/8. If the
   scale factor you specify does not exactly match these fractions,
   the reader will choose the next highest of its supported fractions.

   Param               Use
   ------------------------------------
   uNumerator          numerator of the scale factor fraction,
                     typically "1"

   uDenominator        denominator of the scale factor fraction,
                     typically 1, 2, 4 or 8.
    
---------------------------------------------------------*/
_ISDeclSpec void         _ISCConv _ISFn( is6_SetJPGDCTScaleFactors )(UINT32 uNumerator, UINT32 uDenominator);

/*---------------------------------------------------------
   is6_JPGTransform / _is6_JPGTransform

   Purpose :
   Perform a lossless transformation on the image in a JPG file. This 
   can be used to perform one of a set of simple transformations without 
   decompressing the JPG data. By not decompressing, you don't have to 
   re-compress, thus avoiding adding additional JPG artifacts to the output
   image.

   This works by rearranging the compressed data (DCT coefficients), 
   without ever fully decoding the image.  Therefore, its transformations 
   are lossless: there is no image degradation at all, which is not the case
   if you were to read the image to RGB, transform, and save again as JPG.

   The transpose transformation has no restrictions regarding image 
   dimensions. The other transformations operate rather oddly if the 
   image dimensions are not a multiple of the iMCU size (usually 8 or 16 
   pixels, though sometimes 32 or 48), because they can only transform 
   complete blocks of DCT coefficient data in the desired way.

   The default behavior is to discard these "extra" blocks. This prevents 
   ugly strips of unmodified data along one edge. So, strictly speaking, these
   transformations are not lossless for all images: they are lossless only for 
   images with dimensions that are multiples of 8 or 16. You can override this
   discarding of extra blocks, though it is not recommended.

   Another option is to discard the chrominance channels in standard 
   (YCbCr encoded) JPEGs. This leaves just the luminance which gives you
   a grayscale JPEG.

   All JPEG markers (JPEG_APPx and JPEG_COM) are copied from source to 
   destination.

   Base values for uFlags :
   0       - no transform
   1       - horizontal flip
   2       - vertical flip
   3       - transverse (transpose across UR-to-LL axis)
   4       - transpose (transpose across UL-to-LR axis)
   5       - rotate 90 deg (CW)
   6       - rotate 180 deg
   7       - rotate 270 deg (CW)

   extra flags. Logical "OR" these with the base values:

   bit 8  (0x0100) - grayscale output
   bit 9  (0x0200) - do not trim extra MCU blocks
   bit 10 (0x0400) - progressive output

   Callback is called color components * vertical MCU blocks times.


   Param               Use
   ------------------------------------
   hSource             source JPEG file
   hDest               destination JPEG file
   uFlags              operations to perform:

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_JPGTransform )(HISSRC hSource, HISDEST hDest, UINT32 uFlags);

/*---------------------------------------------------------
   is6_JPGCanDoLosslessTransform / _is6_JPGCanDoLosslessTransform

   Purpose :
   Determine if a lossless transform can be applied to a 
   given JPG file. Some lossless transforms are affected by 
   image dimensions and other encoding options, requiring
   a cropping of image data (on two sides) to align the MCU
   blocks correctly. This function will tell you if the image
   can be transformed without needing to be cropped.


   Param               Use
   ------------------------------------
   hSource             source JPEG file
   uFlags              operation to test

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_JPGCanDoLosslessTransform )(HISSRC hSource, UINT32 uFlag);

/*---------------------------------------------------------
   is6_JPGTransformCrop / _is6_JPGTransformCrop

   Purpose :
   Perform a lossless crop on the image in a JPG file, without 
   decompressing the JPG data. By not decompressing, you don't have to 
   re-compress, thus avoiding adding additional JPG artifacts to the output
   image.

   This works by rearranging the compressed data (DCT coefficients), 
   without ever fully decoding the image.  Therefore, its transformations 
   are lossless: there is no image degradation at all, which is not the case
   if you were to read the image to RGB, transform, and save again as JPG.

   Note:
   JPG files are compressed in rectangular blocks. The
   dimensions of these blocks are always a multiple of 8 
   pixels: 8x8 and 16x16 are common.

   Because the compression blocks on the top and left edges 
   cannot be split during a crop, the size of these blocks 
   affects the crops that this function can perform. This 
   inability to split blocks means that the top-left corner 
   of the crop rect you specify will always be rounded down 
   to the next lowest multiple of the MCU size (see 
   is6_GetJPGMCUSize). The bottom right corner you specify is 
   unaffected.

   Callback is called (color components * vertical MCU blocks) times.

   Param             Use
   ------------------------------------
   hSource           source JPEG file
   hDest             destination JPEG file

   pCropRect         image section to crop. the rectangle extends 
                     up to, but does not include, the right and 
                     bottom coordinates. 

   uFlags            operations to perform

                     bit     value     purpose
                     ----    ------    ----
                     0-7     0-0x0ff   unused
                     8       0x0100    grayscale output
                     9       0x0200    do not trim extra MCU blocks
                     10      0x0400    progressive output

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_JPGTransformCrop )(HISSRC hSource, HISDEST hDest, const RECT * pCropRect, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetJPGMCUSize / _is6_GetJPGMCUSize

   Purpose :
   Get the compression block size for a JPG file. This
   can be used in conjunction with is6_JPGTransformCrop to
   determine the range of crop rectangles that are possible.

   Param             Use
   ------------------------------------
   hSource           source JPEG file
   pMCUX             horizontal MCU size
   pMCUY             vertical MCU size

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_GetJPGMCUSize )(HISSRC hSource, UINT32 *pMCUX, UINT32 *pMCUY);

/*---------------------------------------------------------
    is6_CopyJPGWithNewMarkers / _is6_CopyJPGWithNewMarkers

    Purpose :
    Copy the contents of a JPG file to a new file, discarding 
    existing JPEG_COM or JPEG_APPx (not counting JPEG_APP0) markers 
    and adding any text or markers that have been set via 
    is6_AddJPGOutputMarker and is6_AddJPGOutputText. The image copy 
    is lossless; it does not require a recompression of the image data.
    
    This can be used to remove or replace any JPG comment or
    app markers without requiring a lossy recompression of the
    image. It can also replace ALL JPEG_APPx markers, 
    in-place (see "Replace Mode", below).

    The uFlags parameter can be used to copy JPEG_COM and JPEG_APPx
    blocks from the source to the output. The default is to discard
    these blocks. To copy JPEG_COM, set bit 0 of uFlags (ie. set uFlags to 1);
    to copy JPEG_APP markers, set bit 1 uFlags (ie. set uFlags to 2). These
    can be combined.

    Note:
    For this function, "JPEG_APPx" means JPEG_APP1 through JPEG_APP15. 
    by default, JPEG_APP0 markers are not copied from source. see uFlags.

    Note:
    JPG files are made up of consecutive blocks of data. These
    blocks are usually arranged so that all non-image data comes before
    image data. Specifically, all JPEG_COM and JPEG_APPx blocks usually
    come before JPEG_SOFx (Start Of Frame) blocks. However, some software 
    will place these meta-data blocks after the SOFx blocks:

         Typical:

         ...            <- other JPEG blocks
         JPEG_COM
         JPEG_APPx      <-- all APPx blocks
         JPEG_SOFx      <-- start of frame
         JPEG_SOS       <-- pixel data

      
         Atypical, but still legal per JPEG specs:

         ...            <- other JPEG blocks
         JPEG_COM
         JPEG_APPx      <-- some APPx blocks
         JPEG_SOFx      <-- start of frame
         JPEG_APPx      <-- more APPx blocks
         JPEG_SOS       <-- pixel data

    This function works by scanning the input file, copying or
    discarding blocks, until it finds a SOFx block. When it finds this 
    block, it assumes there is only image infomation left, so it inserts 
    any new markers, adds the SOFx block then copies the rest of the file 
    blindly. This assumption that SOFx signals the start of image data is 
    not always correct.

    Though most JPG writers will put APPx and COM blocks before SOFx blocks,
    some JPG editors (applications that modify IPTC data, for example) will
    put COM blocks between SOFx and SOS. This is still a valid JPG file, but
    it will break applications that assume the typical block arrangement.

    So, this function offers choices:

      1. The default method is what was just described: scan and copy up 
      to SOFx then add the new markers, add SOF and then copy the rest of 
      the file.

      2. The alternate method is to scan and copy up to the SOS marker, 
      insert new markers, add the SOS, then copy the rest of the file. This 
      method will correctly find any meta-data markers that come between
      SOFx and SOS, but it will end up creating the "atypical" block 
      arrangement. Note that this is still a legal JPG file, it is just
      as unusual way of arranging the blocks.

      3. Replace mode. Replace mode replaces all markers, in place, up to 
      the SOS marker. After SOS, it copies the source file without modification.

    Replace Mode:

      Replace mode assumes you have replacements for all of the JPEG_APPx 
      markers (again, not counting JPEG_APP0), in the source file set as 
      output markers. The function runs through the source file and replaces 
      JPEG_APPx[i] from the source with JPEG_APPx[i] from ImgSource's 
      output marker array.

      Example, replacing the first marker:

         // force input marker read
         is6_ReadJPGMarkers(hSrc, 0);

         // rewind
         is6_Seek(hSrc,0,0);
         
         // copy all input markers to output
         is6_CopyJPGInputMetadataToOutputMetadata();

         // change the first output marker in ImgSource's marker array
         is6_SetJPGOutputMarker(0, pNewMarkerData, uNewMarkerLen, uNewMarkerType);

         // run it. the first JPEG_APPx marker will be replaced.
         is6_CopyJPGWithNewMarkers(hSrc, hDest, 1 << 6);
    
      If you don't have enough markers in ImgSource's output marker array to
      cover all the source's markers, this will skip those markers in the
      source and, in effect, replace them with nothing.


    Param                   Use
    ------------------------------------
    hSource                 source JPEG file
    hDest                   output file
    uFlag                   bit     meaning
                            ---     -------
                             0      copy JPEG_COM from source
                             1      copy JPEG_APPx from source (except JPEG_APP0, see bit 7)

                             2      place new comments after SOFn
                                    markers. copy markers up to SOS.
                                    see notes.

                             3      skip JPEG_APP1 markers (EXIF is stored here) 
                                    (if bit 1 is set)

                             4      skip JPEG_APP13 markers (IPTC is stored here) 
                                    (if bit 1 is set)

                             5      skip JPEG_APP2 markers (ICC profile is stored here) 
                                    (if bit 1 is set)

                             6      replace mode (see notes)

                             7      copy JPEG_APP0 from source
    Return
    ------
    FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_CopyJPGWithNewMarkers )(HISSRC hSource, HISDEST hDest, UINT32 uFlag);

#ifdef __cplusplus
}
#endif

#endif