/*---------------------------------------------------------
   ISFilesR.h : part of the ImgSource library.
   
   Whole image reading

   This code copyright (c) 2000-2016, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is6_FILESRH
#define is6_FILESRH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

////////////////////////////////////////////////////////
// Whole image file input

/*---------------------------------------------------------
   is6_ReadImage / _is6_ReadImage

   Purpose:
   Read an image to RGB-24, RGBA-32, DIB, grayscale or 1-bpp.

   This function attempts to determine the file type
   by using is6_GuessFileType, if this fails, it examines
   the HISSRC object to see if the manager is associated
   with a file (from is6_OpenFileSource). If there is a file,
   the file extension is used to determine file type.

   This will not read encrypted TLA files. It will, however,
   read unencrypted TLAs.

   Note:
   By default, this does automatic EXIF orientation correction
   for JPGs. You can disable this with bit 0 of uFlags.

   Note:
   If reading to 1-bpp and source image has more than 1-bpp, 
   the image will be converted to 1-bpp using is6_ImageToGrayScaleSingle
   and is6_GrayscaleTo1Bit with Otsu's gray level histogram method 
   (is6_GrayscaleTo1Bit mode 4). Image rows will be 
   (uWidth + 7) / 8 BYTEs wide, and you should assume a 
   pallete of 0 = black, 1 = white.

   Note:
   RGBA
   If reading to RGBA, and if the image has no alpha 
   information, the A channel is set to 255 for all pixels.
   
   Note:
   PCD images will be read at 768x512 pixels.

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks. This calls back
   for the DIB memory only - it does not use the callbacks
   when reading to RGB or RGBA.

   Note:
   If reading to DIB:
   Once file type has been determined, it determines the 
   bit depth of the input image, then uses the appropriate
   ImgSource file reading function to read the image to 
   its encoded bit depth, or the lowest supported bit-depth 
   for that file format.

   This image is then converted to a DIB and returned to the
   caller. This does not guarantee that the image is read 
   *directly* to a DIB - there may be intermediate steps.

   Param             Use
   ------------------------------------
   hSource           valid source object. if this is
                     not a file source, this function
                     may not be able to determine file 
                     type, if the format is not one 
                     supported by is6_GuessFileType.

   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uMode             0 = 24-bit RGB
                     1 = 32-bit RGBA
                     2 = DIB
                     3 = 1 bit, black and white

   uFlags            bit      purpose
                     ---      -------
                      0       disable EXIF orientation
                              correction for JPGs.

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL      _ISCConv _ISFn( is6_ReadImage )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uMode, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadImageToHBITMAP / _is6_ReadImageToHBITMAP

   Similar to is6_ReadImage, except this takes the extra
   step of converting the image to HBITMAP.

   Param             Use
   ------------------------------------
   hSource           valid source object. if this is
                     not a file source, this function
                     may not be able to determine file 
                     type, if the format is not one 
                     supported by is6_GuessFileType.

   hPal              optional palette
   hDC               device context to use
   uFlags            unused 

   Return
   ------
   HBITMAP image. Caller must use DeleteObject
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HBITMAP     _ISCConv _ISFn( is6_ReadImageToHBITMAP )(HISSRC hSrc, HPALETTE hPal, HDC hDC, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadBMP / _is6_ReadBMP

   Purpose : 
   Read a BMP file.
   
   If the file bit depth is greater than the requested 
   bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks. 

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bits per pixel (1, 8, 24 or 32)
                     if 1, returned image lines will 
                     be ((w + 7) / 8) bytes wide.

   pPal              256 RGBQUADs. receives palette, 
                     if image is colormapped

   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped. don't use this in
                              combination with bit #1.

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadBMP )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadPCX / _is6_ReadPCX

   Purpose : 
   Read a PCX file.

   Note:
   If the file bit depth is greater than the requested 
   bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

   Note:
   For custom source managers writers : this operation
   requires that the input source manager is able to seek 
   from the end of the file. This is because the PCX palette 
   is stored after the pixel data, but must be read first.

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bits per pixel.
                     1 - read to 1-bit
                     8 - read to 8-bit 
                     24 - read to RGB

   pPal              256 RGBQUADs. receives palette, 
                     if image is colormapped

   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped. don't use this in
                              combination with bit #1.

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadPCX )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadTGA / _is6_ReadTGA

   Purpose : 
   Read a TGA file.

   Note:
   If the file bit depth is greater than the requested 
   bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

   Note:
   For custom source managers writers : this operation
   requires that the input source manager is able to seek 
   from the end of the file.

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bits per pixel.
                     8 - read to 8-bit 
                     24 - read to RGB
                     32 - read to RGBA

   pPal              256 RGBQUADs. receives palette, 
                     if image is colormapped.
                     rgbReserved will contain alpha information

   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped. don't use this in
                              combination with bit #1.

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadTGA )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadPNG / _is6_ReadPNG

   Purpose : 
   Read a PNG file. 
   
   Note: 
   Files with less than the requested bit depth 
   are promoted to the requested depth.

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Note:
   You can only read as 48 and 64 bit pixels if the PNG file itself
   contains 48 or 64 bit data. This will not expand lower bit depths
   to 16 bits per channel.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bits per pixel
                     1 - read to 1-bit. Pixel rows start 
                        on whole BYTE boundaries. image lines will 
                        be ((w + 7) / 8) bytes wide.

                     8 - read to 8-bit colormapped

                     16 - read to 8-bit gray plus 8-bit alpha 
                          or 16-bit gray, see uFlags

                     24 - read to 24-bit RGB
                     32 - read to 32-bit RGBA

                     48 - read to RGB pixels with 16 bits per channel
                     64 - read to RGBA pixels with 16 bits per channel

   pPal              256 RGBQUADs. receives palette
                     the rgbReserved member of these
                     RGBQUADs receives alpha information.
                     these values range from 0 (fully 
                     transparent) to 255

   uFlags            bit      purpose
                     ---      -------
                      0       if set, ImgSource will not gamma correct the image 

                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

                      2       if set, reads to 16-bit grayscale when
                              uBitsPerPixel = 16, else reads to 8gray+8alpha
                              when uBitsPerPixel = 16.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped. don't use this in
                              combination with bit #1.

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadPNG )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadJPG / _is6_ReadJPG

   Purpose : 
   Read a JPG file. 

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bit depth:
                     8 - image will be returned as 8-bit grayscale
                     24 - image will be returned as 24-bit RGB
                     32 - CMYK and YCCK images only

   uFlags            bit      purpose
                     ---      -------
                      0       read CMYK and YCCK images without translating 
                              to RGB. uBitsPerPixel must be set to 32!

                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored. CMYK and YCCK are read as RGB,
                              since DIBs do not support CMYK or YCCK.

                      2       read RGB as YUV. this skips the normal 
                              YUV/YCbCr -> RGB conversion step.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped (head-less DIB)

                      4       seek source back to end of JPG data after 
                              reading. this can be useful if there is data 
                              after the JPG that you want to read. 
                              
                              the JPG reading code reads its input in blocks 
                              of 4096 bytes, so if you don't set this flag, 
                              the file pointer could be set up to 4095 bytes 
                              past the actual end of the JPG data.

                              note that this requires the data source to
                              be able to do a negative seek. 

                       5      read progressive images via a JPEG callback
                              pointer. See is6_SetJPGProgressiveReadCallback.

                       6      automatically correct for EXIF orientation.
                              does not apply to progressive reads (bit 5), or 
                              when reading to BGR with padded rows (bit 3).

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadJPG )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadMetafile / _is6_ReadMetafile

   Purpose : 
   Read a metafile (WMF, EMF, AMF) to an RGB buffer.

   Due to the way metafiles are handled by the OS, metafiles
   can only be read from a disk-based file.

   Note:
   Does not use custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   pFileName         path to input file
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   uDefWidth         default width for non-sized files
   uDefHeight        default height for non-sized files
   clrBackground     background color to render the image onto

   uFlags            bit      purpose
                     ---      -------
                      0       read image at uDefWidth x uHeight pixels

                      1       return image in DIB format.

   Return
   ------
   HGLOBAL handle to RGB image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadMetafile )(const char *pFileName, UINT32 *puWidth, UINT32 *puHeight, UINT32 uDefWidth, UINT32 uDefHeight, COLORREF clrBackground, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadMetafileHandle / _is6_ReadMetafileHandle

   Purpose : 
   Read a metafile (WMF, EMF, AMF) to an RGB buffer, given
   a HENHMETAFILE handle to the metafile.

   Note:
   Does not use custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hEmf              Windows metafile handle
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   uDefWidth         default width for non-sized files
   uDefHeight        default height for non-sized files
   clrBackground     background color to render the image onto

   uFlags            bit      purpose
                     ---      -------
                      0       read image at uDefWidth x uHeight pixels

                      1       return image in DIB format.

   Return
   ------
   HGLOBAL handle to RGB image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadMetafileHandle )(HENHMETAFILE hEmf, UINT32 *puWidth, UINT32 *puHeight, UINT32 uDefWidth, UINT32 uDefHeight, COLORREF clrBackground, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadPAX / _is6_ReadPAX

   Purpose : 
   Read a PAX file.

   Note:
   If the file bit depth is greater than the requested 
   bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param            Use
   -----------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bit depth
                     8 - read to 8-bit 
                     24 - read to RGB
   pPal              256 RGBQUADs. receives palette, 
                     if image is colormapped

   pPassword         password for this file. if the correct
                     password is not supplied, this operation
                     will fail.

   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped. don't use this in
                              combination with bit #1.
   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadPAX )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, const char *pPassword, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadPSD / _is6_ReadPSD

   Purpose : 
   Read a PSD (Adobe Photoshop) file.

   Reads the following Photoshop sub-formats:
      Bitmap (compressed only)
      Grayscale
      Indexed
      RGB
      CMYK
      Duotone (without color info)

   Note:
   If the file bit depth is greater than the requested 
   bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

    Note:
   When Photoshop writes a PSD file, it combines all 
   layers, masks, effects, etc. into a single-layered 
   image. This combination is saved into most PSD files. 
   This is what ImgSource reads. The actual layer, mask, 
   etc. data is ignored by the PSD reader. It is possible
   to save PSDs without this composite image, so this
   might not work on all PSDs.

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.
  
   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bit depth
                     8 - read to 8-bit 
                     24 - read to RGB-24
                     32 - read CMYK as CMYK-32. see uFlags
                     48 - read to RGB-48
                     64 - read CMYK as CMYK-64. see uFlags

   pPal              256 RGBQUADs. receives palette, 
                     if image is colormapped

   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

                      2       by default, all CMYK files are automatically
                              converted to RGB-24.
                              if this bit is set, images are returned as
                              CMYK-32 or CMYK-64, unconverted.
                              Note: PSD CMYK is encoded with 0 = max, 65535=min

    Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadPSD )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadTLA / _is6_ReadTLA

   Purpose : 
   Read a TLA file.

   Note:
   Use is6_GuessFileType to determine the encryption
   state (encrypted/unencrypted) of this file before
   attempting to read. Attempting to read an encrypted
   TLA file with the wrong (or no) password will fail.
   
   1 byte per pixel images will usually have a palette, though
   that is not guaranteed.

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   pPassword         password for this file. if you are
                     reading an unencrypted TLA file,
                     this is ignored. otherwise, only the
                     password that was used to create the 
                     file will work for reading the file.

   uBytesPerPixel    format to return the data in.
                     data is returned in one of these formats:

                     1 = 8-bit pixels, with palette, if any.
                     2 = 16 bit RGBA (A-RRRRR-GGGGG-BBBBB)
                     3 = 24-bit RGB
                     4 = 32-bit RGBA
                     
                     * Can not read 2, 3 or 4 bpp to 1 byte per pixel.
                     * Can read all images to higher bit depth (promotion).
                     * Can read 3 and 4 bytes per pixel to 16 bytes per pixel.
                     
   pPal              receives palette, if any. should be 256 RGBQUADs, 
                     though fewer colors may actually be in use.
   
   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBytesPerPixel
                              is ignored.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped. don't use this in
                              combination with bit #1.

   Return
   ------
   HGLOBAL handle to the image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadTLA )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, const char *pPassword, UINT32 uBytesPerPixel, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadTIFF / _is6_ReadTIFF

   Purpose : 
   Read a single page from a TIFF file. 

   For single page TIFFs, use page index 0.

   Use is6_GetTIFFPageCount to determine the number of pages 
    (sub-images) in the TIFF file.

   Note:
    If reading to 24-bit RGB or 32-bit RGBA, this function will 
    automatically convert the input image data to 24-bit (or 32-bit) data.
    
    For other bit depths, no conversion is done, and you must read
    the TIFF to its native bit depth. For example: to get 1-bit data, you 
    must read a 1-bit TIFF; to get 8-bit data, you need an 8-bit TIFF; to 
    get 16-bit data, you need a 16-bit TIFF, etc.. 

   Note:
   Unless reading to 24 or 32-bits, if the file bit depth is not equal 
    to the requested bit depth, this function will fail with 
    IS_ERR_DEPTHMISMATCH.

   Note:
   If reading to 96 bits, the input image must be 96-bits (3x32 bits) -
   this will not promote to 96 bpp. And, the output is 3x32-bit 
   floats (not BYTEs)

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.


   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bit depth
                     1 - monochrome. image lines will be ((w + 7) / 8) bytes wide.
                     8 - colormapped
                     16 - read 16-bit grayscale (single 16-bit value per pixel)
                     24 - RGB 
                     32 - RGBA 
                     48 - RGB with 16 bits per component
                     64 - RGBA with 16 bits per component
                     96 - RGB with 32-bit float components
                     
   pPal              receives image palette

   uFrameIndex       page number to read

   uFlags            bit      purpose
                     ---      -------
                      0       ignore image "orientation" flag.
                              this can greatly speed up image reading, 
                              though the images may be rotated/flipped
                              differently than intended.

                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored. not applicable for 16- or 96-bit images.

                      2       return CMYK images as CMYK, instead of 
                              converting to RGBA

                      3       vertical flip 1-bit input images 

                      4       vertical flip 16- or 96-bit input images 
                              (overrides orientation correction)

                      5       DWORD-align output images.
                              overrides orientation correction in 16-bit images

                      6       swap red and blue in 24, 32, 48 and 64-bit images

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadTIFF )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uFrameIndex, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetTIFFPageCount / _is6_GetTIFFPageCount

   Purpose : 
   Determine the number of frames/pages in the current TIFF file.

   Param             Use
   ------------------------------------
   hSource           valid source object

   Return
   ------
   Number of frames.
---------------------------------------------------------*/
_ISDeclSpec UINT32      _ISCConv _ISFn( is6_GetTIFFPageCount )(HISSRC hSource);

/*---------------------------------------------------------
   is6_ReadWBMP / _is6_ReadWBMP

   Purpose : 
   Read a WBMP file. This is NOT a "Windows Bitmap", this is
   a "Wireless Bitmap", intended for use on cell phones, PDAs,
   etc..
   
   This supports WBMP Level-0, uncompressed, two color images.
   Extension Headers are not supported.

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   pPal              256 RGBQUADs. receives palette
   
   uBitsPerPixel     1: image lines will be ((w + 7) / 8) bytes wide.
                     8: image will be 8-bpp

   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped. don't use this in
                              combination with bit #1.

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadWBMP )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, RGBQUAD *pPal, UINT32 uBitsPerPixel, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadPCD / _is6_ReadPCD

   Purpose : 
   Read a Kodak PhotoCD file to an RGB buffer.
   
   ImgSource supports reading PhotoCD images at the following
   6 image resolutions:

   index    dimensions   label
   ------------------------------
   1.       192 x 128    (Base/16)
   2.       384 x 256    (Base/4)
   3.       768 x 512    (Base)
   4.       1536 x 1024  (Base*4)
   5.       3072 x 2048  (Base*16)
   6.       6144 x 4096  (Base*64)

   These resolutions are stored in the actual image file. When reading,
   you must choose a specific resolution. Images may be rotated a multiple 
   of 90 degrees, in which case, the width and height may be swapped.

   This will not read PCD index image files (ISGuessFileType #15). To
   read those, use is6_ReadPCDIndex.

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   uSubImage         1 - 6, index of sub-image to load (see table above) 

   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadPCD )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uSubImage, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadPCDIndex / _is6_ReadPCDIndex

   Purpose : 
   Read a thumbnail from a Kodak PhotoCD index/overview 
   file to an RGB buffer. These files are generally called 
   "OVERVIEW.PCD".

   These thumbnails are always 192x128 pixels, unless rotated,
   when they'll be 128x192 pixels.

   See is6_GetPDCIndexImageCount to determine the number of
   thumbnails in the index file.

   Note:
   This only works with index/overview images. See 
   is6_GuessFileType to determine if a file is an index or an image.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   uIndex            thumbnail index
   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadPCDIndex )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uIndex, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadGIF / _is6_ReadGIF

   Purpose : 
   Read a GIF file to an 8 bit buffer. Files with 
   less than 8 bits per pixel are promoted to 8 bit. 

   Param                Use
   ------------------------------------
   hSource              valid source object
   puWidth              receives image width (in pixels)
   puHeight             receives image height (in pixels)
   piTransColor         receives index of transparent color. -1 if none.
   pPal                 array of 256 RGBQUAD entries. 
                        receives image palette
   
   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.

                      3       read data with DWORD row-padding. 
                              don't use this in combination with bit #1.

   Return
   ------
   HGLOBAL handle to 8 bit image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadGIF )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, __int32 *piTransColor, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadMultiFrameGIF / _is6_ReadMultiFrameGIF

   Purpose : 
   Read a multi-framed GIF image. Frames are stored and may be
   retreived with is6_GetGIFFrame or is6_GetGIFFrameEx.

   To render the animation:
      
      1. allocate an image of the "screen" size.
      
      2. get a frame with is6_GetGIFFrame or is6_GetGIFFrameEx.

      3. overlay it at the location specified, on top of the image allocated
      in step 1. wait, for the number of ticks specified.

      4. "dispose" of the frame by the method specified

      5. loop to 2, until you've shown the sequence for the specified
      number of times.

   Param                Use
   ------------------------------------
   hSource              valid source object
   puImageCount         receives number of frames
   puWidth              receives GIF screen width
   puHeight             receives GIF screen height
   puLoopCount          receives number of times to loop. 0 = infinity
   puBackGround         receives global background color index
   piTransparency       receives global transparency index. -1 if none.
   pPal                 receives 256 element palette
   uFlags               unused

   Return
   ------
   TRUE on success.
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ReadMultiFrameGIF )(HISSRC hSource, UINT32 *puImageCount, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puLoopCount, UINT32 *puBackGround, __int32 *piTransparency, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ClearGIFFrames / _is6_ClearGIFFrames

   Purpose : 
   Clear GIF frames stored by the last call to 
   is6_ReadMultiFrameGIF.

   Param               Use
   ------------------------------------

   Return
   ------
   TRUE on success.
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ClearGIFFrames )();

/*---------------------------------------------------------
   is6_GetGIFFrameCount / _is6_GetGIFFrameCount

   Purpose : 
   Return the number of GIF frames read by the last call
   to is6_ReadMultiFrameGIF.
   
   Param               Use
   ------------------------------------

   Return
   ------
   UINT32 number of frames
---------------------------------------------------------*/
_ISDeclSpec UINT32        _ISCConv _ISFn( is6_GetGIFFrameCount )();
 
/*---------------------------------------------------------
   is6_GetGIFFrame / _is6_GetGIFFrame

   Purpose : 
   Get information about a single GIF frame read with 
   is6_ReadMultiFrameGIF.

   All frames are returned as top-down 8-bit buffers.
   
   Param                Use
   ------------------------------------
   uIndex               frame index

   phImage              receives HGLOBAL handle to image.
                        do NOT delete this memory. it will be
                        deleted when you call is6_ClearGIFFrames

   pWidth               receives frame width
   pHeight              received frame height
   pXpos                receives frame X offset
   pYpos                receives frame Y offset
   pTicks               receives frame duration, in 1/100 secs

   pDisposalMethod      receives frame disposal method. this
                        determines how you should handle this frame
                        when the next frame is to be displayed

                        0 - perform no action 
                        1 - leave the image on screen, 
                            should be same as '0', basically.
                        2 - replace with the GIF screen background color
                        3 - replace this image with what was there before

   pInputFlag           if 1, the display program is supposed to 
                        wait for user input of some kind before continuing
                        to the next frame. this is rarely used.

   pPalType             0 = global. (uses global palette)
                        1 = local. (uses frame local palette)

   pClrTransIndex       transparent index. -1 if none. applies to local 
                        or global pallete. see pPalType.

   pbInterlaced         returns frame interlace flag (TRUE if interlced)

   pPal                 receives palette. if pPalType is 0, this will be
                        a copy of the global palette. if this is 1, it will
                        be the local frame palette.

   uFlags               unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL       _ISCConv _ISFn( is6_GetGIFFrame )(UINT32 uIndex, HGLOBAL *phImage, UINT32 *pWidth, UINT32 *pHeight, UINT32 *pXpos, UINT32 *pYpos, UINT32 *pTicks, UINT32 *pDisposalMethod, UINT32 *pInputFlag, UINT32 *pPalType, __int32 *pClrTransIndex, BOOL *pbInterlaced, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_AnimatedGIFRenderInit / _is6_AnimatedGIFRenderInit

   Purpose : 
   Initialize an animated GIF rendering object. This object
   will read the GIF image, compose the individual frames and
   return them to you as 32-bit RGBA images.

   Call is6_DestroyAnimatedGIFRender when finished.

   Param                Use
   ------------------------------------
   hSource              valid source object
   uWidth               receives width of animation
   uHeight              receives height of animation
   uFlags               unused

   Return
   ------
   HISGIFANI handle to GIF animator object, NULL on failure
---------------------------------------------------------*/
_ISDeclSpec HISGIFANI   _ISCConv _ISFn( is6_AnimatedGIFRenderInit )(HISSRC hSource, UINT32* uWidth, UINT32 *uHeight, UINT32 uFlags);

/*---------------------------------------------------------
   is6_AnimatedGIFRenderNextFrame / _is6_AnimatedGIFRenderNextFrame

   Purpose : 
   Get the next GIF frame from the GIF animator object.
   The frame is an RGBA image of width * height * 4 bytes
   (you must allocate this memory) - see is6_AnimatedGIFRenderInit
    to get the width and height you'll need to use. The image will have
   alpha values of 0 in places where the GIF is transparent,
   and 255 everywhere else.


   Param                Use
   ------------------------------------
   hGIFAni              valid GIF animator object
   pOutput              receives RGBA image data

   puSleepTicks         hundredths of seconds to display this 
                        image before showing the next one.

   uFlags               bit      purpose
                        ---      -------
                         1       do not loop
   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     _ISCConv _ISFn( is6_AnimatedGIFRenderNextFrame )(HISGIFANI hGIFAni, BYTE *pOutput, UINT32 * puSleepTicks, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DestroyAnimatedGIFRender / _is6_DestroyAnimatedGIFRender

   Purpose : 
   Release all memory associated with the GIF animator object.

   Param                Use
   ------------------------------------
   hGIFAni              valid GIF animator object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL     _ISCConv _ISFn( is6_DestroyAnimatedGIFRender )(HISGIFANI hGIFAni);

/*---------------------------------------------------------
   is6_ReadIconFrames / _is6_ReadIconFrames

   Purpose : 
   Read all frames in a Windows icon (.ICO) file. Frames are 
   stored and may be retreived with is6_GetIconFrame.

   Param                Use
   ------------------------------------
   hSource              valid source object
   puCount              receives number of frames read

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_ReadIconFrames )(HISSRC hSrc, UINT32 *puCount);

/*---------------------------------------------------------
   is6_GetIconFrame / _is6_GetIconFrame

   Purpose : 
   Get information about a single icon frame read with 
   is6_ReadIconFrames.

   Note about icon frames:
   Icons are composed of two parts: 
      1. The "XOR" mask. This is a standard DIB that is combined
      with the destination using an XOR operation. 

      2. The "AND" mask. This is a monochrome image (returned in ImgSource
      as a 1-bpp DIB) that is used to controls transparency. Only draw pixels
      from the XOR image where the corresponding AND image pixel is a 1.

   Both images are the same size.

   Icon files can contain any number of frames.

   Note:
   Do not call GlobalFree on the images in the ISIconStruct structure.
   They will be deleted by ImgSource automatically.

   Param                Use
   ------------------------------------
   uIndex               frame index
   pICS                 receives info about a stored ICO frame

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_GetIconFrame )(UINT32 uIndex, ISIconStruct* pICS);

/*---------------------------------------------------------
   is6_ReadPNM / _is6_ReadPNM

   Purpose : 
   Read a PPM, PGM or PBM image
   
   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Note:
   PBM images can only be read to 1 bpp
   PGM images can only be read to 8 or 16 bpp
   PPM images can only be read to 24 or 48 bpp

   Note:
   uBpp is used to choose between 8 and 16 bpp for PGM images, and
   between 24 and 48 images for PPM images. 

   Note:
   The PGM and PPM formats contain a field that holds the maximum
   value for a pixel in the image. The default behavior for this 
   function is to scale the input pixels, based on that maximum 
   value, to the 0..255 (or 0..65535) range. You can disable this 
   scaling in uFlags.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   uBpp              desired bit depth (1, 8, 16, 24, 48). (see Notes)

   uFlags            bit      purpose
                     ---      -------
                      1       return image in DIB format.
                              image will be read at the lowest 
                              bit depth possible; uBitsPerPixel
                              is ignored.

                      3       do not scale pixels. (see Notes)
                      4       swap red and blue values in RGB images (overridden by bit 1)
                      5       vertical flip images (overridden by bit 1)

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL      _ISCConv _ISFn( is6_ReadPNM )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBpp, UINT32 uFlags);


/*---------------------------------------------------------
   is6_ReadCALS / _is6_ReadCALS

   Purpose : 
   Read a CALS Type I file to buffer of 1-bit pixels.
   
   "The CALS raster format is a standard developed by the 
    Computer Aided Acquisition and Logistics Support (CALS) 
    office of the United States Department of Defense to 
    standardize graphics data interchange for electronic 
    publishing, especially in the areas of technical 
    graphics, CAD/CAM, and image processing applications."
    -- http://www.fileformat.info/format/cals/egff.htm

    Note:
    This function can only handle CALS Type 1 files. 

   Note:
   See notes for is6_SetFileReadAllocCallbacks, if you are
   using custom memory allocation callbacks.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uFlags            unused

   Return
   ------
   HGLOBAL handle to image. Caller must use GlobalFree 
   to release this memory back to the OS.
---------------------------------------------------------*/
_ISDeclSpec HGLOBAL         _ISCConv _ISFn( is6_ReadCALS )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uFlags);

// undocumented. for use with generic reading
_ISDeclSpec void      _ISCConv _ISFn( is6_SetDefaultMetafileInfo )(COLORREF clr, UINT32 w, UINT32 h);


#ifdef __cplusplus
}
#endif

#endif