/*---------------------------------------------------------
   ISFilesRL.h : part of the ImgSource library.
   
   single line file reading

   This code copyright (c) 2000-2016, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/
#ifndef is6_FILESRLH
#define is6_FILESRLH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif        

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_InitReadBMPLines / _is6_InitReadBMPLines

   Purpose : 
   Set up for single line BMP file reading.
   Use with is6_ReadNextBMPLine and is6_DestroyReadBMPLines.

   If pPal==NULL, is6_ReadNextBMPLine returns RGB lines
   else 8 bit lines

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param                Use
   ------------------------------------
   hSource              valid source object
   puWidth              receives image width (in pixels)
   puHeight             receives image height (in pixels)
   pPal                 receives palette

   Return
   ------
   HDBMP - BMP single line read object
---------------------------------------------------------*/
_ISDeclSpec HDBMP         _ISCConv _ISFn( is6_InitReadBMPLines )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, RGBQUAD *pPal);

/*---------------------------------------------------------
   is6_ReadNextBMPLine / _is6_ReadNextBMPLine

   Purpose : 
   Read a single line from a BMP file

   Param                Use
   ------------------------------------
   hSingle              valid HDBMP object

   pRow                 receives image data, see 
                        is6_InitReadBMPLines for format notes

   puRowIndex           receives index of row that was read

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ReadNextBMPLine )(HDBMP hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*---------------------------------------------------------
   is6_DestroyReadBMPLines / _is6_DestroyReadBMPLines

   Purpose : 
   Destroy the HDBMP object and any memory associated 
   with it.

   Param                Use
   ------------------------------------
   hSingle              valid HDBMP object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyReadBMPLines )(HDBMP hSingle);

/*---------------------------------------------------------
   is6_InitReadPCXLines / _is6_InitReadPCXLines

   Purpose : 
   Set up for single line PCX file reading.
   Use with is6_ReadNextPCXLine and is6_DestroyReadPCXLines.

   If pPal==NULL, is6_ReadNextPCXLine returns RGB lines
   else 8 bit lines

   Note for custom source managers writers : this operation
   requires that the input source manager is able to seek 
   from the end of the file. This is because the PCX palette 
   is stored after the pixel data, but must be read first.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   pPal              receives palette

   Return
   ------
   HDPCX - PCX single line read object
---------------------------------------------------------*/
_ISDeclSpec HDPCX         _ISCConv _ISFn( is6_InitReadPCXLines )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, RGBQUAD *pPal);

/*---------------------------------------------------------
   is6_ReadNextPCXLine / _is6_ReadNextPCXLine

   Purpose : 
   Read a single line from a PCX file

   Param             Use
   ------------------------------------
   hSingle           valid HDPCX object

   pRow              receives image data, see 
                     is6_InitReadPCXLines for format notes

   puRowIndex        receives index of row that was read

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ReadNextPCXLine )(HDPCX hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*---------------------------------------------------------
   is6_DestroyReadPCXLines / _is6_DestroyReadPCXLines

   Purpose : 
   Destroy the HDPCX object and any memory associated 
   with it.

   Param            Use
   ------------------------------------
   hSingle            valid HDPCX object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyReadPCXLines )(HDPCX hSingle);

/*---------------------------------------------------------
   is6_InitReadJPGLines / _is6_InitReadJPGLines

   Purpose : 
   Set up for single line JPG file reading.
   Use with is6_ReadNextJPGLine and is6_DestroyReadJPGLines.

   If bGrayscale==0, this returns RGB lines
   else 8 bit lines

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)

   uBitsPerPixel     desired bit depth:
                     8 - image will be returned as 8-bit grayscale
                     24 - image will be returned as 24-bit RGB
                     32 - CMYK and YCCK images only

   uFlags            bit      purpose
                     ---      -------
                      0       read CMYK and YCCK images without translating 
                              to RGB. uBitsPerPixel must be set to 32!

                      2       read RGB as YUV. this skips the normal 
                              YUV/YCbCr -> RGB conversion step.

                      3       read data with DWORD row-padding and with 
                              red and blue swapped (head-less DIB)

                      4       seek source back to end of JPG data after 
                              reading. this can be useful if there is data 
                              after the JPG that you want to read. 
                              
                              the JPG reading code reads its input in blocks 
                              of 4096 bytes, so if you don't set this flag, 
                              the file pointer could be set up to 4095 bytes 
                              past the actual end of the JPG data.

                              note that this requires the data source to
                              be able to do a negative seek. 

   Return
   ------
   HDJPG - JPG single line read object
---------------------------------------------------------*/
_ISDeclSpec HDJPG         _ISCConv _ISFn( is6_InitReadJPGLines )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadNextJPGLine / _is6_ReadNextJPGLine

   Purpose : 
   Read a single line from a JPG file

   Param             Use
   ------------------------------------
   hSingle           valid HDJPG object

   pRow              receives image data, see 
                     is6_InitReadJPGLines for format notes

   puRowIndex        receives index of row that was read

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ReadNextJPGLine )(HDJPG hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*---------------------------------------------------------
   is6_DestroyReadJPGLines / _is6_DestroyReadJPGLines

   Purpose : 
   Destroy the HDJPG object and any memory associated 
   with it.

   Param             Use
   ------------------------------------
   hSingle           valid HDJPG object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyReadJPGLines )(HDJPG hSingle);

/*---------------------------------------------------------
   is6_InitReadPAXLines / _is6_InitReadPAXLines

   Purpose : 
   Set up for single line PAX file reading.
   Use with is6_ReadNextPAXLine and is6_DestroyReadPAXLines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   uBitsPerPixel     8 or 24
   pPal              receives palette (for 8-bit images)

   pPassword         image password. if the correct
                     password is not supplied, this operation
                     will fail.

   Return
   ------
   HDPAX - PAX single line read object
---------------------------------------------------------*/
_ISDeclSpec HDPAX         _ISCConv _ISFn( is6_InitReadPAXLines )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, const char *pPassword);

/*---------------------------------------------------------
   is6_ReadNextPAXLine / _is6_ReadNextPAXLine

   Purpose : 
   Read a single line from a PAX file

   Param             Use
   ------------------------------------
   hSingle           valid HDPAX object

   pRow              receives RGB image data, see 
                     is6_InitReadPAXLines for format notes
   
   puRowIndex        receives index of row that was read

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ReadNextPAXLine )(HDPAX hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*---------------------------------------------------------
   is6_DestroyReadPAXLines / _is6_DestroyReadPAXLines

   Purpose : 
   Destroy the HDPAX object and any memory associated 
   with it.

   Param                Use
   ------------------------------------
   hSingle              valid HDPAX object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyReadPAXLines )(HDPAX hSingle);

/*---------------------------------------------------------
   is6_InitReadTLALines / _is6_InitReadTLALines

   Purpose : 
   Set up for single line TLA file reading.
   Use with is6_ReadNextTLALine
    and is6_DestroyReadTLALines.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param                Use
   ------------------------------------
      ...

   See is6_ReadTLA for parameter notes.

   Return
   ------
   HDTLA - TLA single line read object
---------------------------------------------------------*/
_ISDeclSpec HDTLA         _ISCConv _ISFn( is6_InitReadTLALines )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBytesPerPixel, RGBQUAD *pPal, const char *pPassword);

/*---------------------------------------------------------
   is6_ReadNextTLALine / _is6_ReadNextTLALine

   Purpose : 
   Read a single line from a TLA file

   Param                Use
   ------------------------------------
   hSingle              valid HDTLA object

   pRow                 receives image data, see 
                        is6_ReadTLA for format notes

   puRowIndex           receives index of row that was read

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ReadNextTLALine )(HDTLA hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*---------------------------------------------------------
   is6_DestroyReadTLALines / _is6_DestroyReadTLALines

   Purpose : 
   Destroy the HDTLA object and any memory associated 
   with it.

   Param                Use
   ------------------------------------
   hSingle              valid HDTLA object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyReadTLALines )(HDTLA hSingle);

/*---------------------------------------------------------
   is6_InitReadPNGLines / _is6_InitReadPNGLines

   Purpose : 
   Set up for single line PNG file reading.
   Use with is6_ReadNextPNGLine and is6_DestroyReadPNGLines.

   If pPal==NULL, is6_ReadNextPCXLine returns RGB lines
   else 8 bit lines

   Note: This cannot read interlaced PNGs.

   Note: Single line input/output functions are not 
   available for unregistered users.

   Param             Use
   ------------------------------------
   hSource           valid source object
   puWidth           receives image width (in pixels)
   puHeight          receives image height (in pixels)
   pPal              receives palette

   uFlags            bit      purpose
                     ---      -------
                      0       if set, ImgSource will not gamma correct the image 

   Return
   ------
   HDPNG - PNG single line read object
---------------------------------------------------------*/
_ISDeclSpec HDPNG         _ISCConv _ISFn( is6_InitReadPNGLines )(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, RGBQUAD *pPal, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReadNextPNGLine / _is6_ReadNextPNGLine

   Purpose : 
   Read a single line from a PNG file

   Param             Use
   ------------------------------------
   hSingle           valid HDPNG object

   pRow              receives image data, see 
                     is6_InitReadPNGLines for format notes

   puRowIndex        receives index of row that was read

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_ReadNextPNGLine )(HDPNG hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*---------------------------------------------------------
   is6_DestroyReadPNGLines / _is6_DestroyReadPNGLines

   Purpose : 
   Destroy the HDPNG object and any memory associated 
   with it.

   Param             Use
   ------------------------------------
   hSingle           valid HDPNG object

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL         _ISCConv _ISFn( is6_DestroyReadPNGLines )(HDPNG hSingle);

#ifdef __cplusplus
}
#endif

#endif