/*---------------------------------------------------------

   ISImageEffects.h : part of the ImgSource library.

    Image effects 

    This code copyright (c) 2000-2016, Smaller Animals Software, Inc..

    No portion of this code may be copied or distributed without 
    permission of Smaller Animals Software, Inc..
    http://www.smalleranimals.com

---------------------------------------------------------*/

#pragma once

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Do not include this file directly. Include ISource.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_AdjustImageColorBalance

   Purpose : 
   Adjust the color balance of an image. This is similar to 
   Adobe Photoshop's Adjust / Color Balance feature.

   Note:
   Useful values for all of the color adjustment parameters range
   from -50 to +50. 

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel in the image
                     (3 or 4, alpha channel is not affected)

   uRowStride        bytes per pixel row

   R                 amount to adjust Red midtone
   G                 amount to adjust Green midtone
   B                 amount to adjust Blue midtone
   
   Rshadow           amount to adjust Red shadows
   Gshadow           amount to adjust Green shadows
   Bshadow           amount to adjust Blue shadows

   Rhighlight        amount to adjust Red highlights
   Ghighlight        amount to adjust Green highlights
   Bhighlight        amount to adjust Blue highlights

   Density           varies the intensity of the effect (-50 to +50)

   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR color order


   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv  _ISFn( is6_AdjustImageColorBalance )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, __int32 R, __int32 G, __int32 B,__int32 Rshadow, __int32 Gshadow, __int32 Bshadow,__int32 Rhighlight,__int32 Ghighlight,__int32 Bhighlight,__int32 Density,UINT32 uFlags);

/*---------------------------------------------------------
   is6_DuotoneImage

   Purpose : 
   Generate a simulation of a printer's duotone.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel in the image
                     (3 or 4, alpha channel is not affected)

   uRowStride        bytes per pixel row

   clr               tint color

   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR color order

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_DuotoneImage )(BYTE *pRGB, UINT w, UINT h, UINT32 uBytesPerPixel, UINT32 uRowStride, COLORREF clr, UINT32 uFlags);

/*---------------------------------------------------------
   is6_InvertImage

   Purpose : 
   Invert all channels in the image, aka photo-negative.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel in the image
   uRowStride        bytes per pixel row
   uFlags            if bit 0 is set, this will ignore channel 4 
                     (assumed to be the alpha channel) on images 
                     with 4 bytes per pixel

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_InvertImage )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_PosterizeImage / _is6_PosterizeImage

   Purpose : 
   Generate an image made up of primary and secondary colors from
   an RGB/RGBA image

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height

   uBytesPerPixel        3 or 4
   uRowStride        bytes per pixel row

   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR order

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_PosterizeImage )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_HalftoneImage

   Purpose : 
   Generate a black and white image dithered to large "halftone" dots.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uRowStride        bytes per pixel row
   uBytesPerPixel    number of bytes per pixel (3 only)
   uMaxDotSize       maximum size of the dots
   bSmoothDots       if FALSE, dots are solid black, with sharp edges

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_HalftoneImage )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uMaxDotSize, BOOL bSmoothDots);

/*---------------------------------------------------------
   is6_MosaicImage

   Purpose : 
   Generate a mosaic (large pixel) image

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel (1, 2, 3 or 4)
   uRowStride        bytes per pixel row
   uSquareSize       size of tiles

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_MosaicImage )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uSquareSize);

/*---------------------------------------------------------
   is6_SolarizeImage

   Purpose : 
   Adjust curves to generate the classic "solaraized" image effect.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel (3 or 4)
   uRowStride        bytes per pixel row
   uFlags            if bit 0 is set, this will ignore channel 4 
                     (assumed to be the alpha channel) on images 
                     with 4 bytes per pixel

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_SolarizeImage )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, UINT32 uFlags);

/*---------------------------------------------------------
   is6_AdjustTint

   Purpose : 
   Adjust color balance by simulating a "tint" control on a 
   television.

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel
                     (3 or 4, alpha channel is not affected)
   uRowStride        bytes per pixel row

   iTint             -180 to +180. 
                     positive values enhance green.
                     negative values enhance red.
   
   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR color order

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_AdjustTint )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uRowStride, int iTint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CrackleRGB

   Purpose : 
   Generate an image that looks like a cracked ceramic glaze.
   RGB only.

   Notes:
      * pIn != pOut
      * caller allocates output 

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output image
   uOutRowSTride     bytes per output pixel row
   mode              0 to 4
   uDefinition       controls sharpness of cracks (0 to 100)

   uFlags            bit   purpose
                     ---   -------
                      0    0 = higher contrast
                           1 = normal contrast
   
   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_CrackleRGB)(const BYTE *pIn, UINT32 w, UINT32 h, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, UINT32 mode, UINT32 uDefinition, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GauzyImage

   Purpose : 
   Generate a soft focus, gauzed image.

    Alpha channel is not affected.

   Notes:
      * pIn != pOut
      * caller allocates output 

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
    uBytesPerPixel        3 or 4
   uInRowStride      bytes per input pixel row
   pOut              output image
   uOutRowStride     bytes per output pixel row
   fBlurAmount       0 to 1
   fSharpAmount      0 to 1. amount of original in image
   fClarity          0 to 1. amount to clarify 
   uMode             0: desaturated
                     1: high contrast foreground
                     2: full color
                     3: max (fClarity is not used)
                     4: min (fClarity is not used)
                     
   
   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_GauzyImage)(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, double fBlurAmount, double fSharpAmount, double fClarity, UINT32 uMode);

/*---------------------------------------------------------
   is6_Stipple

   Purpose : 
   Generate a stippled image

   Notes:
      * pIn != pOut
      * caller allocates output 

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
    uBytesPerPixel        must be 3
   uInRowStride      bytes per input pixel row
   pOut              output image
   uOutRowStride     bytes per output pixel row
   uSize             0 to 100. size of spots
   uFilter           controls the kind of blurring filter used
                     0: Arithmetic mean
                     1: Minimum
                     2: Maximum
                     3: Geometric mean
                     4: Yp Mean

   uMode             0: Noise first
                     1: Filter first
   
   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_Stipple )(BYTE *pIn, UINT32 w, UINT32 h, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, UINT32 uSize, UINT uFilter, UINT uMode);

/*---------------------------------------------------------
   is6_Scribble

   Purpose : 
   Generate an image that looks like it was drawn with paint and a
   heavy outline pen.

   Notes:
      * pIn != pOut
      * caller allocates output 

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
    uInRowStride        number of BYTEs per row in the input image
   pOut              output image
    uOutRowStride        number of BYTEs per row in the output image
   uMode             0: blurred image
                     1: filtered image
                     2: posterized image
                     3: despeckled image

   bFilterFirst      Filter before drawing lines

   uDark             0-100. darkness of lines

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_Scribble )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, UINT32 uMode, BOOL bFilterFirst, UINT32 uDark);

/*---------------------------------------------------------
   is6_TintGrayscale

   Purpose : 
   Given an input grayscale image, generate an RGB/RGBA image tinted to 
   a certain color.

   Notes:
      * pIn != pOut
      * caller allocates output 

   Param             Use
   ------------------------------------
   pIn               source image, 8-bit grayscale
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output image
   uOutBpp           3 or 4, output bytes per pixel
   uOutRowStride     bytes per input pixel row
   clrTint           color to tint the output

   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR color order for output

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_TintGrayscale )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutBpp, UINT32 uOutRowStride, COLORREF clrTint, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Fingerprint

   Purpose : 
   Apply an effect that looks something like a bunch of 
   fingerprint smudges. Input image must be 8-bit grayscale.

   Notes:
      * pIn != pOut
      * caller allocates output 

   Param             Use
   ------------------------------------
   pIn               source image, 8-bit grayscale
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOut              output image
   uOutRowStride     bytes per input pixel row

   fDepth            depth of effect must be a positive number. 
                     50 is a good starting point. 

   uFilterSize       3, 5 or 7 are good choices

   uFlags            bit      purpose
                     ---      -------
                      0       if set, use dilation mode

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_Fingerprint)(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride, double fDepth, UINT32 uFilterSize, UINT32 uFlags);

/*---------------------------------------------------------
   is6_OilPaintRGB

   Purpose : 
   Apply an effect that looks something like oil painting. 
   Input image must be 24-bit RGB.

   Notes:
      * pIn != pOut
      * caller allocates output 

   Param             Use
   ------------------------------------
   pInRGB            source image, 24-bit RGB
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOutRGB           output image
   uOutRowStride     bytes per input pixel row

   uBrushWidth       filter width. 
                     low integers are best (3,4,5,etc.)

   uMaxDensity       internal quantization level. (0..255)
                     start with 64 or 128.

   uFlags            bit      purpose
                     ---      -------
                      0       blend edges of colors (value = 1)
                      1       enhance contrast     (value = 2)

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_OilPaintRGB)(const BYTE *pInRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOutRGB, UINT32 uOutRowStride, UINT32 uBrushWidth, UINT32 uMaxDensity, UINT32 uFlags);

/*---------------------------------------------------------
   is6_WatercolorTintRGB

   Purpose : 
   Apply an effect that resembles hand-tinted photographs.

   Notes:
      * pIn != pOut
      * caller allocates output 

   Param             Use
   ------------------------------------
   pInRGB            source image, 24-bit RGB
   uWidth            width in pixels
   uHeight           height
   uInRowStride      bytes per input pixel row
   pOutRGB           output image
   uOutRowStride     bytes per input pixel row

   fBlur             blur amount. 0.0 to 1.0

   uFlags            bit      purpose
                     ---      -------
                      0       increase contrast
                      1       if set, lighten image
                              if not set, darken image

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_WatercolorTintRGB )(BYTE *pInRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOutRGB, UINT32 uOutRowStride, double fBlur, UINT32 uFlags);

/*---------------------------------------------------------
   is6_Painterly

   Purpose :
   Create a copy of the source that is rendered with curving
   brush strokes of variying size and color. Larger brushes 
   are drawn first, followed by subsequently smaller brush sizes.

   The result looks a lot like an oil painting.

   This is based on the process described in Aaron Hertzmann's
   paper "Painterly Rendering with Curved Brush Strokes of Multiple Sizes".

   Notes:
   The IS_Painterly_Opts struct members are:

   maxStrokeLen   - default 12 - Maximum number of segments in a stroke
   minStrokeLen   - default 4 - Minimum number of segments in a stroke
   fCurve         - def 0.5. [0..1] - Higher values reduce stroke curvature
   threshold      - def 50 [0..3x256] - Larger values reduce fidelity
   blurFactor     - def 0.95 - Smaller values allow for more noise
   bSimplfy       - def FALSE - reduce curve complexity
   maxBrushSize   - def 8 - Width of largest brush
   brushes        - def 3 - Total number of brushes used
   opacity        - def 1.0 - Stroke layer opacity
   roundEnds      - def TRUE - Stroke ends are rounded
   blurStrokeEdges - def FALSE - Blur the edges of strokes
   outlineClr     - def black - color to draw edges with
   drawOutlines   - def FALSE - After strokes, overlay edges from source

   For more detailed explanations of most of these parameters, see 
   Hertzmann's paper.

   Note:
   The number of brushes/layers is controlled by the 'brushes'
   member of the IS_Painterly_Opts struct. The size of the brush
   used for each layer is determined like this:

   First layer is drawn with brushSize = maxBrushSize. 
   The second layer is drawn with brushSize = brushSize / 2.
   The third layer is drawn with brushSize = brushSize / 2.
   Etc.

   This continues until 'brushes' layers have been drawn, OR until 
   brushSize == 1.

   Note:
   This is not a quick process.
   
   Note:
   If blurStrokeEdges is set, the default behavior is to blur the
   entire stroke before redering it. This tends to soften the 
   overall image. If you set bit 0 of uFlags, the stroke is not 
   blurred, but edges of the strokes are anti-aliased; the stroke
   will be much sharper. This is also much slower.

   Note:
   * pIn != pOut
   * caller allocates output

   Param                Use
   ------------------------------------
   pInRGB               source image, 24-bit RGB
   uWidth               width in pixels
   uHeight              height
   uComponentsPerPixel  must be 3
   uInRowStride         components per input pixel row
   pOutRGB              output image
   uOutRowStride        components per input pixel row

   pOpts                pointer to fully initialized 
                        IS_Painterly_Opts struct. ALL
                        members must be set. 
                        
                        pass NULL, if you want to use 
                        the default values. 
                        
                        see Notes.

   uFlags               bit      meaning
                        ---      -------
                         0       anti-alias the stroke edges, 
                                 instead of blurring them

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_Painterly)(const BYTE *pInRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uComponentsPerPixel, UINT32 uInRowStride, BYTE *pOutRGB, UINT32 uOutRowStride, IS_Painterly_Opts *pOpts, UINT32 uFlags);

/*-------------------------------------------------------
   is6_ASCIIizeGrayscaleImage

   Purpose : 
   Generate an "ASCII Art" image from an 8-bit grayscale image.

   Note:
   This generates a single ASCII string from your image, with line
   breaks and a trailing NULL character. See the notes for pOutASCII 
   to determine how to allocate the output image.
   
   Param                Use
   ------------------------------------
   pImage               source grayscale
   uWidth               width in pixels
   uHeight              height   
   uInRowStride         number of bytes in an input pixel row

   pOutASCII            receives ASCII image. this must be allocated
                        by the caller. the buffer must be:

                           1 + (uOutCharsX + 1) * uOutCharsY BYTEs

   uOutCharsX           width of output image. this does not count 
                        the carriage returns at the end of each line. see
                        notes for pOutASCII to determine how to allocate 
                        the output image.

   uOutCharsY           height (number of text lines) of output image

   pCharData            characters to use in the ASCII image.
                        Each character here will represent a single
                        grayscale range. The 'lightest' character
                        should come first, followed by the next-lightest, etc..
                        there should be no more than 256 characters.

                        This parameter is optional. If you pass a NULL here,
                        the function will use the following 16-character 
                        string as the character set: "WM$@#%^x*o=+:~. "

   uFlags               unused

   Return
   ------
      FALSE on failure
-------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_ASCIIizeGrayscaleImage)(const BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, char *pOutASCII, UINT32 uOutCharsX, UINT32 uOutCharsY, const char *pCharData, UINT32 uFlags);

/*-------------------------------------------------------
   is6_SegmentedColorizeRGB

   Purpose : 
   Create a color-segmented and filtered 8-bit image from
   an input RGB image.

   The effect generates an output image where individual colors in 
   the output image are clustered into relatively few, large regions.
   This can be used in automated image identification, since
   fine details and subtle colors are removed in favor of large areas
   of solid color. Or, it can be used for purely aesthetic reasons.

   Note:
   The "morphological close" mode uses one of a few preset masks. Since 
   a morphological "close" operation tends to reproduce the shape of
   the selection mask, the color clusters in the output image will take the
   shape of the mask. So, the 5x5 square mask will generate images with 
   lots of horizontal and vertical edges, and the diamond masks will 
   give images with lots of 45 degree (give or take) angles, and the 
   circular masks will give a more rounded appearance. 

   Note:
   The "median filter" mode is much faster, but the results are much
   noisier than the morphological close mode.

   Param                Use
   ------------------------------------
   pImage               source 24-bit RGB image
   uWidth               width in pixels
   uHeight              height   
   uInRowStride         number of bytes in an input pixel row

   pOutput8             receives 8-bit output image. this must be allocated
                        by the caller. 

   uOutRowStride        bytes per pixel row in output image

   pPal                 receives output palette (must be 256 elements)

   maxColors            maximum numbers of colors in the output image.
                        more colors = longer processing time. 
                        try 8 or 4 to start.

   uFilterType          one of:
                        0 - median filter
                        1 - morphological close
                        2 - morphological hit or miss    - this tends to fill with solid color
                        3 - morphological boundary extraction

   uFilterParam         if using the "median filter" mode, this is the filter 
                        radius (if you give an even number here, the function will
                        use the next odd number).

                        if using the "morphological close" mode, this selects
                        one of the following masks:
                           0 - uses 3x3 solid mask
                           1 - 3x3 diamond-shaped mask
                           2 - 5x5 solid mask
                           3 - 5x5 diamond-shaped mask
                           4 - 7x7 solid mask
                           5 - 7x7 circle (roughly) shaped
                           6 - 9x9 circle
                           7 - 9x9 diamond
                           8 - 9x9 skewed diamond
                           9 - 15x15 diamond
                           10 - 15x15 circle
                           11 - 15x15 circle with center hole
   

   uFlags               unused

   Return
   ------
      FALSE on failure
-------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_SegmentedColorizeRGB )(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uInRowStride, BYTE *pOutput8, UINT32 uOutRowStride, RGBQUAD *pPal, UINT32 maxColors, UINT32 uFilterType, UINT32 uFilterParam, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ReflectImage / is6_ReflectImage

   Purpose : 
   Creates an image which has a reflection of the source image at the 
    bottom of the output - a very "Web 2.0" kind of effect. The
    reflected image is blended with a solid color on the 
    reflecting 'surface'.

   Param                        Use
   ------------------------------------
   pIn                        source image
   uWidth                     width in pixels
   uHeight                    height
   uBytesPerPixel             number of bytes per pixel in the image. (3 only)
   uInRowStride               bytes per input pixel row

   pOut                       destination image. must be:
                                 uWidth * (uReflectYPos + uReflectHeight) pixels

   uOutRowStride              bytes per output pixel row

   uReflectYPos               vertical position of the reflection 
                              surface on the source image.

   uReflectHeight             the height of the reflecting surface

   fReflectionRatioStart      controls where the fade between the reflection and the
                              surface color starts.
                              0.0 = top edge, 1.0 = bottom edge. if negative,
                              the reflected image will always have some of the
                              reflected surface color in it. should be less than
                              fReflectionRatioEnd.

    fReflectionRatioEnd       controls where the fade between the reflection and the
                              surface color ends.
                              0.0 = top edge, 1.0 = bottom edge. may be > 1.

    surfaceColor              color of the reflecting surface

    uFadeMode                 0 = linear
                                1 = sine, fast
                                2 = sine, slow

    fTextureIntensity        if using the surface texturizing mode (see uFlags),
                                this controls the intensity of the effect.
                                0.0 to 1.0.

    uTextureScale            controls the size of the texture. try 10 to start.

   uFlags                     bit      meaning
                              ---      -------
                               0       if set, the reflected surface will be texturized
                                       using a Perlin Displacement transform.

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_ReflectImage)(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uInRowStride, BYTE *pOut, UINT32 uOutRowStride,  UINT32 uReflectYPos, UINT32 uReflectHeight, double fReflectionRatioStart, double fReflectionRatioEnd, COLORREF surfaceColor, UINT32 uFadeMode, double fTextureIntensity, UINT32 uTextureScale, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GradientFillImageRadial

   Purpose : 
   Fill an image with a radial color gradient.

      
   Ex.:

      // fade from yellow to green
      RGBQUAD clrs[2];
      double fPercentages[2];

      clrs[0].rgbRed = 255;
      clrs[0].rgbGreen = 255;
      clrs[0].rgbBlue = 0;

      clrs[1].rgbRed = 0;
      clrs[1].rgbGreen = 255;
      clrs[1].rgbBlue = 0;

      // everything below 50% of fGradientSize from the center
      // is solid yellow. at 50%, the transition to green starts.
      fPercentages[0] = 50;  
   
      // every point above 90% of fGradientSize from the center
      // is solid green
      fPercentages[1] = 90;


   Param             Use
   ------------------------------------
   pOut              output image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel in the image (1 to 4)
   fGradientSize     the gradient is spread across this distance.
   iCenterX          center of the gradient                  
   iCenterY          center of the gradient
   
   clrs              array of RGBQUADs. these are the gradient colors.
                     if uBytesPerPixel = 1, fill uses the rgbRed member
                     if uBytesPerPixel = 2, fill uses the rgbRed and rgbGreen members
                     if uBytesPerPixel = 3, fill uses rgbRed, rgbGreen & rgbBlue
                     if uBytesPerPixel = 4, fill uses all members

   fPercentages      array of doubles. these specify how much of the gradient
                     the corresponding color (from clrs) occupies.

   uNumColors        number of entries in clrs and fPercentages. 
                     this must be at least 2

   iMode             unused
      
   fPower            controls blending power. this can be any non-zero positive value.
                     1 is the typical value and gives the same effect as is6_GradientFillImageRadial.

   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR color order

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_GradientFillImageRadial )(BYTE *pOut, UINT32 uWidth, UINT32 uHeight, UINT32 uBpp, UINT32 uRowStride, double fMaxDistance, int iCenterX, int iCenterY, RGBQUAD *clrs, double *fPercentages, UINT32 uNumColors, int iMode, double fPower, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GradientFillImageLinear

   Purpose : 
   Fill an image with a linear color gradient.

   You need to supply two points that define a line. The gradient
   will be constructed about that line. (see iMode for options)

   Color is determined by the distance from the pixel to the start of the
   gradient line.
   
   See is6_GradientFillImageRadial for an example of specifying the
   colors.

   Param             Use
   ------------------------------------
   pOut              output image
   uWidth            width in pixels
   uHeight           height
   uBytesPerPixel    number of bytes per pixel in the image (1 to 4)
   uRowStride        bytes per pixel row
   fLinePoint1X      start of the gradient line
   fLinePoint1Y      start of of the gradient line

   fLinePoint2X      end of the gradient line
   fLinePoint2Y      end of the gradient line

   clrs              array of RGBQUADs. these are the gradient colors.
                     if uBytesPerPixel = 1, fill uses the rgbRed member
                     if uBytesPerPixel = 2, fill uses the rgbRed and rgbGreen members
                     if uBytesPerPixel = 3, fill uses rgbRed, rgbGreen & rgbBlue
                     if uBytesPerPixel = 4, fill uses all members

   fPercentages      array of doubles. these specify where the center of each color lies.
                     ex. with three colors, perc = [25,50,75] means that color 0 
                     is solid until 25% of the gradient length, then starts to blend 
                     with color[1], until 50% of the gradient length, when it starts to blend
                     with color[2]. the final 25% will be solid color[2].

                     also note that the percentage value determines the order of the color,
                     not the position in the clrs array.

   uNumColors        number of entries in clrs and fPercentages. 
                     this must be at least 2

   iMode             Mode 0:
                     The gradient runs parallel to the line. The start point defines 
                     the start of the gradient and the end point determines the direction 
                     and the length of the gradient. The gradient is symmetric about
                     the normal of the line.

                     Mode 1:
                     The gradient extends only on the side of the gradient line
                     that contains the the end point. Pixels on the other side
                     of the line will be colored with the lowest color in the color array.
                     This is similar to the way you draw a gradient in PhotoShop,
                     where you click, drag and release to define the start, direction 
                     and distance of the gradient.

   fPower            controls blending power. this can be any non-zero positive value.
                     1 is the typical value.

   uFlags            bit      purpose
                     ---      -------
                      0       assume BGR color order

   
     

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_GradientFillImageLinear )(BYTE *pOut, UINT32 uWidth, UINT32 uHeight, UINT32 uBpp, UINT32 uRowStride, double fLinePoint1X, double fLinePoint1Y, double fLinePoint2X, double fLinePoint2Y, RGBQUAD *clrs, double *fPercentages, UINT32 uNumColors, int iMode, double fPower, UINT32 uFlags);


/*---------------------------------------------------------
   is6_InnerGlowRGBA

   Purpose : 
   This is similar to is6_OuterGlowRGBA, but the halo is drawn 
   in the inside edge of the borders.

   This is similar, but not identical to, PhotoShop's "Inner Glow"
   layer effect.

    Note:
    For best results, the RGBA image should have alpha values 
    of either zero or 255.

   Param             Use
   ------------------------------------
   pRGBA             source image, 32-bit RGBA
   uWidth            width in pixels
   uHeight           height
   uRowStride        number of bytes in an image row
   uSize             size of effect. larger = larger halo
   fOpacity          controls opacity of the glow effect (0...1)
   glowClr           color of the glow

   pCurveLUT         optional 256 element BYTE array that will control
                     the way the fades away from the border. if you
                     don't supply a LUT, the default is a gamma curve of 1.8.

   uFlags            unused

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_InnerGlowRGBA)(BYTE *pRGBA, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uSize, double fOpacity, COLORREF glowClr, const BYTE *pCurveLUT, UINT32 uFlags);

/*---------------------------------------------------------
   is6_OuterGlowRGBA

   Purpose : 
   Draw a glowing halo around the borders on the content of an RGBA 
   image, using the alpha mask to define the "borders" of the image.
   In effect, it draws a colored halo around areas of high alpha values.

   This will have no visible effect if the image has a solid alpha channel.

   This is similar, but not identical to, PhotoShop's "Outer Glow"
   layer effect.

   LUT Example: 
   // gamma of 2.8 - makes the halo darker
   double gamma = 2.8;
   BYTE lut[256];
    for (int z=0;z<256;z++)
   {
       double ind = (double) z / 256.0;
       lut[z] = (BYTE) (256 * pow (ind, 1. / gamma));
   }

   Param             Use
   ------------------------------------
   pRGBA             source image, 32-bit RGBA
   uWidth            width in pixels
   uHeight           height
   uRowStride        number of bytes in an image row
   uSize             size of effect. larger = larger halo
   fOpacity          controls opacity of the glow effect (0...1)
   glowClr           color of the glow

   pCurveLUT         optional 256 element BYTE array that will control
                     the way the fades away from the border. if you
                     don't supply a LUT, the default is a gamma curve of 1.8.


   uFlags            unused

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn(is6_OuterGlowRGBA)(BYTE *pRGBA, UINT32 uWidth, UINT32 uHeight, UINT32 uRowStride, UINT32 uSize, double fOpacity, COLORREF glowClr, const BYTE *pCurveLUT, UINT32 uFlags);

/*---------------------------------------------------------
   is6_DropShadow

   Purpose : 
   Generate an image with a drop shadow.

   By default, the shadow is based on the image rectangle.
    uFlags can be used to direct the function to use the
    image's alpha mask instead.

   Notes:
      * caller is responsible for calling GlobalFree
      on the output image.

   Param             Use
   ------------------------------------
   pIn               source RGB or RGBA image
   uWidth            width in pixels
   uHeight           height
   uBpp              bytes per pixel in input (3 or 4)
   uInRowStride      components per input pixel row
   phOut             receives HGLOBAL to output image
   pOutW             receives output image width
   pOutH             receives output image height
   iOffsetMax        shadow offset distance
   diffusion         1 to 100, controls shadow diffusion
   clrBackground     background color
   clrShadow         shadow color
   iDeg              shadow position angle

   pOverlayPos       receives offset of original image in final image
   pShadowPos        receives offset of shadow image in final image

   uFlags            bit      purpose
                     ---      -------
                      0       if uBpp == 4, use the alpha channel of the 
                              image as the shadow mask. the shadow will 
                                        ONLY affect the alpha mask.

                      1       if uBpp == 4, use the alpha channel of the 
                              image as the shadow mask. the shadow will 
                                        affect the RGB channels and the alpha mask.

    Note: these modes cannot be combined.

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_DropShadow )(BYTE *pIn, UINT32 inW, UINT32 inH, UINT32 uBpp, UINT32 uInRowStride, HGLOBAL *phOut, UINT32 *pOutW, UINT32 *pOutH, int iOffsetMax, int diffusion, COLORREF clrBackground, COLORREF clrShadow, int iDeg, POINT * pOverlayPos, POINT * pShadowPos, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ShadeBevelEdge

   Purpose : 
   Apply a beveled edge effect to the image.


   Example:
   // Shade from the bottom-right
   is6_ShadeBevelEdge(pRGB, w, h, w * 3, 
                     16, // bevel size
                     3,  // bpp
                     0,  // no flags
                     80, // dim the left edge to 80%
                     60, // dim the top edge to 60%
                     120, // brighten the right edge by 20%
                     150,  // brighten the bottom edge by 50%
                     100); // blur at a factor of 100

   Param             Use
   ------------------------------------
   pIn               source image
   uWidth            width in pixels
   uHeight           height
   uInRowStride      number of bytes in an image row
   uBevelSize        size of the bevel along an edge, in pixels 
   uBpp              bytes per pixel in input

   uFlags            if uBpp = 4 and bit 0 of uFlags is set,
                     this will skip the 4th channel (usefule for RGBA images)

   leftPct           brightness percent of the left facet 
                     0..255. 100 = unchanged, < 100 = darker, > 100 = lighter.

   topPct            brightness percent of the top facet 
   rightPct          brightness percent of the right facet 
   bottomPct         brightness percent of the bottom facet 

   fSmoothFactor     smooth the bevel edge (0 = no effect. reasonable values are from 0 to 500)

   

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL _ISCConv _ISFn( is6_ShadeBevelEdge )(BYTE *pRGB, UINT32 inW, UINT32 inH, UINT32 uInRowStride, UINT32 uBevelSize, UINT32 uBpp, UINT32 uFlags, double leftPct, double topPct, double rightPct, double bottomPct, double fBlurPct);

_ISDeclSpec BOOL _ISCConv _ISFn( is6_SolidBevelEdgeRGB )(BYTE *pRGB, UINT32 inW, UINT32 inH, UINT32 uRowStride, COLORREF highClr, COLORREF lowClr);

#ifdef __cplusplus
}
#endif
