/*---------------------------------------------------------
    ISPSDResource.h : Part of the ImgSource library
    
    Photoshop/Adobe resource management

    This code copyright (c) 2000-2016, Smaller Animals Software, Inc..

    No portion of this code may be copied or distributed without 
    permission of Smaller Animals Software, Inc..
    http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is6_PSDRESOURCESH
#define is6_PSDRESOURCESH

#if !defined is6_SOURCEH && !defined is6_INTERNAL
#error Need to include ISource.h before this file.
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*---------------------------------------------------------
   is6_InitializePSDResourceManager / _is6_InitializePSDResourceManager

   Purpose : 
   Create a PSD Resource Manager object.

   This object will allow you to manage (add, remove, 
   read, write) a set of Photoshop/Adobe 'resources'.

   Note:
   The format specifications for these objects is
   available from Adobe. Older versions of the spec
   were available for free to the public, or with Photoshop. 
   Newer versions require you to sign an agreement with Adobe.

   Note:
   You must use is6_DestroyPSDResourceManager to release 
   this object.

   Param             Use
   ------------------------------------
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec HISPSDRM      _ISCConv _ISFn( is6_InitializePSDResourceManager )(UINT32 uFlags);

/*---------------------------------------------------------
   is6_DestroyPSDResourceManager / _is6_DestroyPSDResourceManager

   Purpose : 
   Destroy a PSD Resource Manager object that was created
   with is6_InitializePSDResourceManager.

   Param             Use
   ------------------------------------
   hPSDMgr           PSD manager object to be destroyed

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_DestroyPSDResourceManager )(HISPSDRM hPSDMgr);

/*---------------------------------------------------------
   is6_FillPSDResourceManagerFromSource / _is6_FillPSDResourceManagerFromSource

   Purpose : 
   Read the Photoshop/Adobe resources from a file.

   This currently supports reading from PSD (Photoshop), 
   JPG and TIFF.

   All resources that were found will be stored in the
   manager object. You can use is6_GetPSDResourceList
   to get a list of the resource IDs, is6_GetPSDResource
   to read a single resource, is6_RemovePSDResource to
   remove a single resource, etc..

   Note:
   This allows you to work with a *copy* of the resources.
   It doesn't operate directly on the file.

   Param             Use
   ------------------------------------
   hPSDMgr           PSD manager object to use
   hSource           opened ImgSource source manager
   uTiffFrameIdx     TIFF frame to read, if any
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_FillPSDResourceManagerFromSource )(HISPSDRM hPSDMgr, HISSRC hSource, UINT32 uTiffFrameIdx, UINT32 uFlags);

/*---------------------------------------------------------
   is6_FillPSDResourceManagerFromByteArray / _is6_FillPSDResourceManagerFromByteArray

   Purpose : 
   Read the Photoshop/Adobe resources from a block of memory.

   Param             Use
   ------------------------------------
   hPSDMgr           PSD manager object to use
   pData             pointer to PSD resource data
   uDataLen          number of bytes in resource data
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn(is6_FillPSDResourceManagerFromByteArray)(HISPSDRM hPSDMgr, const BYTE *pData, UINT32 uDataLen, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPSDResourceList / _is6_GetPSDResourceList

   Purpose : 
   Get a list of resources currently held by the PSD
   resource manager. Returns an array of 16-bit integers
   that holds the resource IDs of the resources. 

   See Adobe's PSD format specs for what each ID represents.

   Note:
   Caller must call GlobalFree on the ID array to return 
   the memory back to Windows.

   Param             Use
   ------------------------------------
   hPSDMgr           PSD manager object to use
   pU16Items         receives a HGLOBAL pointer to an array of 16-bit integers
   puItemCount       receives number of items in the array
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_GetPSDResourceList )(HISPSDRM hPSDMgr, HGLOBAL *pU16Items, UINT32 *puItemCount, UINT32 uFlags);

/*---------------------------------------------------------
   is6_GetPSDResource / _is6_GetPSDResource

   Purpose : 
   Retrieve the data associated with a single PSD resource.

   This returns the resource data only; the '8BIM' header is
   not returned.

   See Adobe's PSD format specs for the format of each data block.

   Note:
   Caller must call GlobalFree on the data and the description 
   string to return the memory back to Windows.

   Param             Use
   ------------------------------------
   hPSDMgr           PSD manager object to use
   phData            receives HGLOBAL pointer to the resource data
   puDataLen         receives size of resource data
   phDesc            receives HGLOBAL pointer to optional 
                     null-terminated description string

   puDescLen         receives length of the description string
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_GetPSDResource )(HISPSDRM hPSDMgr, ISUINT16 resID, HGLOBAL *phData, UINT32 *puDataLen, HGLOBAL *phDesc, UINT32 *puDescLen, UINT32 uFlags);

/*---------------------------------------------------------
   is6_RemovePSDResource / _is6_RemovePSDResource

   Purpose : 
   Remove a resource from a PSD resource manager.

   This removes the first resource matching the ID you
   specify from the PSD resource manager. 

   Param             Use
   ------------------------------------
   hPSDMgr           PSD manager object to use
   resID             resource ID
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_RemovePSDResource )(HISPSDRM hPSDMgr, ISUINT16 resID, UINT32 uFlags);

/*---------------------------------------------------------
   is6_AddPSDResource / _is6_AddPSDResource

   Purpose : 
   Add a resource to a PSD resource manager.

   Param             Use
   ------------------------------------
   hPSDMgr           PSD manager object to use
   resID             resource ID
   pData             pointer to resource data
   uDataLen          length of resource data
   pDescription      optional description string (max 255 chars)
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn(is6_AddPSDResource)(HISPSDRM hPSDMgr, ISUINT16 resID, const BYTE *pData, UINT32 uDataLen, const char *pDescription, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CopyPSDResourceManager / _is6_CopyPSDResourceManager

   Purpose : 
   Copy the data in one resource manager object to 
   another.

   Param             Use
   ------------------------------------
   hPSDMgrSrc        PSD manager object to copy from
   hPSDMgrDest       PSD manager object to copy to
   uFlags            unused

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_CopyPSDResourceManager )(HISPSDRM hPSDMgrSrc, HISPSDRM hPSDMgrDest, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ExportPSDResourceBlock / _is6_ExportPSDResourceBlock

   Purpose : 
   Export all resources to a single properly-formatted 
   resource block. You can use this in 
   is6_SetPSDOutputResourceBlock, is6_SetTIFFBinaryTag or  
   is6_AddJPGOutputMarker to add Photoshop data to those 
   files.

   Note:
   Use GlobalFree to release the memory returned back to Windows.

   Examples:

      // for JPG:
      is6_ExportPSDResourceBlock(hPSDMgr, &hData, &dataLen, 2);
      if (hData)
      {
         is6_AddJPGOutputMarker((BYTE *)hData, dataLen, 0xed);
         HISDEST hDest = is6_OpenFileDest("jpeg.jpg", 0);
         is6_WriteJPG(...);
         is6_CloseDest(hDest);
         GlobalFree(hData);
      }

      // for TIFF:
      is6_ExportPSDResourceBlock(hPSDMgr, &hData, &dataLen, 0);
      if (hData)
      {
         is6_SetTIFFBinaryTag(34377, (BYTE *)hData, dataLen, 0);
         HISDEST hDest = is6_OpenFileDest(DPIX"tiff.tif", 0);
         is6_WriteTIFF(...);
         is6_CloseDest(hDest);
         GlobalFree(hData);
      }

      // for Photoshop:
      is6_ExportPSDResourceBlock(hPSDMgr, &hData, &dataLen, 0);
      if (hData)
      {
         is6_SetPSDOutputResourceBlock((BYTE *)hData, dataLen);
         HISDEST hDest = is6_OpenFileDest(DPIX"photoshop.psd", 0);
         is6_WritePSD(...);
         is6_CloseDest(hDest);
         GlobalFree(hData);
      }

   Param             Use
   ------------------------------------
   hPSDMgr           PSD manager object to export
   phData            receives HGLOBAL to the data block
   puDataLen         receives length of data

   uFlags            bit      purpose
                     ---      -------
                      0       add 4-byte length header

                      1       add "Photoshop 3.0\0" header.
                              this is required for JPG

   Return
   ------
   FALSE on failure
---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_ExportPSDResourceBlock )(HISPSDRM hPSDMgr, HGLOBAL *phData, UINT32 *puDataLen, UINT32 uFlags);

/*---------------------------------------------------------
   is6_ExaminePhotoshopResources / _is6_ExaminePhotoshopResources

   Purpose : 
   Examine a block of memory to see if it contains
   Photoshop resources.

   This can be used to determine if the memory block is 
   suitable for use as input to is6_FillPSDResourceManagerFromByteArray.
   If the data block contains no Photoshop data, 
   is6_FillPSDResourceManagerFromByteArray will not be able to operate
   on the input.

   Param             Use
   ------------------------------------
   pPhotoshopData    block of memory which holds Photoshop 
                     resources. 

   uPSLength         size of that block

   uFlags            one of:
                        
                        0 - return TRUE if block contains 
                        Photoshop resources.

                        1 - return TRUE if the block contains 
                        a valid IPTC resource block.

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn(is6_ExaminePhotoshopResources)(const BYTE *pPhotoshopData, const UINT32 uPSLength, UINT32 uFlags);

/*---------------------------------------------------------
   is6_CopyPSDWithNewResources / _is6_CopyPSDWithNewResources

   Purpose : 
   Copy the data in a PSD (Photoshop) file to a new file
   and replace the image resource data with the data set
   in the last call to is6_SetPSDOutputResourceBlock.

   Among other things, this can be used as part of an 
   'in-place' edit of IPTC data.

   Param             Use
   ------------------------------------
   hSource           opened source PSD file
   hDest             opened output file
   uFlags            unused

   Return
   ------
   FALSE on failure

---------------------------------------------------------*/
_ISDeclSpec BOOL      _ISCConv _ISFn( is6_CopyPSDWithNewResources )(HISSRC hSource, HISDEST hDest, UINT32 uFlags);

#ifdef __cplusplus
}
#endif

#endif