/*---------------------------------------------------------
   ISTypes.h : part of the ImgSource library.
   
   This code copyright (c) 2000-2016, Smaller Animals Software, Inc..

   No portion of this code may be copied or distributed without 
   permission of Smaller Animals Software, Inc..
   http://www.smalleranimals.com

---------------------------------------------------------*/

#ifndef is6_TYPESH
#define is6_TYPESH

////////////////////////////////////////////////////////
// Types and structures used by ImgSource

// UINT32 is a 32-bit unsigned integer
#if     _MSC_VER >= 1200
// we'll use what MS uses
#else
// no UINT32 in VC5
typedef unsigned __int32 UINT32;
#endif

typedef unsigned __int16 ISUINT16;

////////////////////////////////////////////////////////

// source manager object
/*
   If you get a compilation error here, try adding #include <windows.h> before
   you #include the ImgSource headers.
*/
typedef HANDLE HISSRC;

////////////////////////////////////////////////////////
// destination manager object
typedef HANDLE HISDEST;

typedef void * HISPERLIN;

#pragma pack(push, 1)

////////////////////////////////////////////////////////
// double-precision point structure, used in some ImgSource functions
#ifndef IS_XYDOUBLE_STRUCT_DEF
#define IS_XYDOUBLE_STRUCT_DEF
typedef struct XY_tag 
{
   double x; 
   double y;
} XYdouble;
#endif

////////////////////////////////////////////////////////
// structure which can describe a quadrilateral 
#ifndef IS_QUAD_STRUCT_DEF
#define IS_QUAD_STRUCT_DEF
typedef struct ISQuadrilateral_tag 
{
   POINT topLeft;
   POINT bottomLeft;
   POINT topRight;
   POINT bottomRight;
} ISQuadrilateral;
#endif

////////////////////////////////////////////////////////
// structure which can describe a line 
#ifndef IS_LINE_STRUCT_DEF
#define IS_LINE_STRUCT_DEF
typedef struct ISLine_tag 
{
   POINT p0;
   POINT p1;
} ISLine;
#endif

////////////////////////////////////////////////////////

// used when reading Photoshop thumbnails
#ifndef IS_PSDTHUMBNAIL_STRUCT_DEF
#define IS_PSDTHUMBNAIL_STRUCT_DEF
typedef struct ISPSDThumbnailInfo_t
{
   __int32 colorOrder;
   __int32 format;
   __int32 width;
   __int32 height;
   __int32 widthbytes;
   __int32 size;
   __int32 csize;
   __int16 bitspixel;
   __int16 planes;
} ISPSDThumbnailInfo;
#endif

////////////////////////////////////////////////////////
// structure used to set DPI to files
#ifndef IS_DPI_STRUCT_DEF
#define IS_DPI_STRUCT_DEF
typedef struct IS4DPIStruct_t
{
   double fDPIX;
   double fDPIY;
   UINT32 uDPIUnits;
} IS4DPIStruct;
#endif

#ifndef IS3_DPI_STRUCT_DEF
#define IS3_DPI_STRUCT_DEF
typedef struct ISDPIStruct_t
{
   UINT32 uDPIX;
   UINT32 uDPIY;
   UINT32 uDPIUnits;
} ISDPIStruct;
#endif

////////////////////////////////////////////////////////
// multi-frame GIF frame definition structs
#ifndef IS_HAVE_MFI_GLOBALSTRUCTS
#define IS_HAVE_MFI_GLOBALSTRUCTS

typedef struct ISGGStruct_t
{
   UINT32   imageCount;
   UINT32   width;
   UINT32   height;
   UINT32   loopCount;
   UINT32   backGround;
   //__int32  trans;
   RGBQUAD   palette[256];
} ISGGStruct;

typedef struct ISGFStruct_t
{
   UINT32   index;
   UINT32   imageType;
   HGLOBAL  hImage;
   UINT32   width;
   UINT32   height;
   UINT32   xpos;
   UINT32   ypos;
   UINT32   ticks;
   UINT32   disposalMethod;
   UINT32   inputFlag;

   UINT32   palType;
   UINT32   clrTransIndex;
   RGBQUAD  palette[256];
} ISGFStruct;

//////////////////////////////////////////////////////////

typedef struct ISGFStructEx_t
{
   UINT32   index;
   UINT32   imageType;
   HGLOBAL  hImage;

   UINT32   width;
   UINT32   height;
   UINT32   xpos;
   UINT32   ypos;

   UINT32   ticks;
   UINT32   disposalMethod;
   UINT32   inputFlag;

   UINT32   palType;
   __int32   clrTransIndex;
   BOOL     bIsInterlaced;

   RGBQUAD  palette[256];
} ISGFStructEx;


//////////////////////////////////////////////////////////

typedef struct ISIconStruct_t
{
   UINT32   index;
   HGLOBAL  hDIBXOR;
   HGLOBAL  hDIBAND;

   UINT32   width;
   UINT32   height;
   UINT32   bitDepth;
} ISIconStruct;

//////////////////////////////////////////////////////////

typedef struct ISEXIFTagInfo_t
{
    ISUINT16    uTagID;
    BYTE    uIFDID;
    BYTE    uTagType;
    UINT32    uTagCount;
} ISEXIFTagInfo;
#endif

//////////////////////////////////////////////////////////

typedef struct IS_Painterly_Opts_t
{
   int maxStrokeLen;    // 12
   int minStrokeLen;    // 4
   double fCurve;       // .5
   double threshold;    // 50
   double blurFactor;   // .95
   BOOL bSimplfy;       // FALSE
   int maxBrushSize;    // 8
   int brushes;         // 3
   double opacity;      // FALSE
   BOOL roundEnds;      // TRUE
   BOOL blurStrokeEdges;   // FALSE
   COLORREF outlineClr;    // RGB(0,0,0)
   BOOL drawOutlines;      // FALSE
} IS_Painterly_Opts;

typedef struct ISRGBA16_t
{
    ISUINT16 r;    // red, or gray for grayscale images
    ISUINT16 g;    // green
    ISUINT16 b;    // blue
    ISUINT16 a;    // alpha, for RGBA-64 images
} ISRGBA16;
#pragma pack(pop)

// IS_CALLBACK_DATA will be the size of a pointer (32 or 64 bits, depending on OS)
typedef INT_PTR IS_CALLBACK_DATA;

////////////////////////////////////////////////////////
// callback function definition
typedef BOOL      (CALLBACK* ISCALLBACKPTR)(const UINT32 uCurRow, const UINT32 uHeight, const IS_CALLBACK_DATA uUserData);

////////////////////////////////////////////////////////
// memory alloc callback function definitions
typedef void *    (CALLBACK* IS3IMGMEM_ALLOC_CBPTR)(const UINT32 uSize, const IS_CALLBACK_DATA uUserData);
typedef void *    (CALLBACK* IS3IMGMEM_REALLOC_CBPTR)(void * pData, const UINT32 uSize, const IS_CALLBACK_DATA uUserData);
typedef BOOL      (CALLBACK* IS3IMGMEM_FREE_CBPTR)(void * pData, const IS_CALLBACK_DATA uUserData);

////////////////////////////////////////////////////////
// User source / dest function types
typedef BOOL      (CALLBACK* is6_SourceOpenFn)(const IS_CALLBACK_DATA uUserInfo);
typedef UINT32    (CALLBACK* is6_SourceReadFn)(void *pData, const UINT32 uDataLen, const IS_CALLBACK_DATA uUserInfo);
typedef BOOL      (CALLBACK* is6_SourceWriteFn)(const void *pData, const UINT32 uDataLen, const IS_CALLBACK_DATA uUserInfo);
typedef BOOL      (CALLBACK* is6_SourceSeekFn)(const __int32 uOffset, const UINT32 uOrigin, const IS_CALLBACK_DATA uUserInfo);
typedef BOOL      (CALLBACK* is6_SourceGetPosFn)(const IS_CALLBACK_DATA uUserInfo, UINT32 *puPos);
typedef BOOL      (CALLBACK* is6_SourceGetSizeFn)(const IS_CALLBACK_DATA uUserInfo, UINT32 *puSize);
typedef BOOL      (CALLBACK* is6_SourceEOFFn)(const IS_CALLBACK_DATA uUserInfo);
typedef BOOL      (CALLBACK* is6_SourceCloseFn)(const IS_CALLBACK_DATA uUserInfo);

////////////////////////////////////////////////////////
// multi-page TIFF writing callback
typedef BOOL      (CALLBACK *is6_TIFFMULTIPAGE_CBPTR)(BYTE **pImage, UINT32 *nWidth, UINT32 *nHeight, RGBQUAD **pPal, UINT32 *uBitsPerPixel, UINT32 *uCompression, IS4DPIStruct *pDPI, const IS_CALLBACK_DATA uData);

////////////////////////////////////////////////////////
// progressive JPEG reading callback
typedef BOOL      (CALLBACK* is6_JPG_PROGREAD_CBPTR)(const HGLOBAL hImageData, const UINT32 uWidth, const UINT32 uHeight, const UINT32 uBPP, const UINT32 uPass, IS_CALLBACK_DATA uUserData);

////////////////////////////////////////////////////////
// user transform callback function
typedef BOOL (CALLBACK* ISXFORMCALLBACK)(const double fCurX, const double fCurY, double *pFromX, double *pFormY, const IS_CALLBACK_DATA uUserData);

////////////////////////////////////////////////////////
// error reporting callback function
typedef BOOL (CALLBACK* ISERRORCALLBACK)(UINT32 uErrorCode, const char *pErrorText, const IS_CALLBACK_DATA uUserData);

////////////////////////////////////////////////////////
// Line-at-a-time file input objects
typedef HANDLE HDBMP;
typedef HANDLE HDPCX;
typedef HANDLE HDPNG;
typedef HANDLE HDJPG;
typedef HANDLE HDPAX;
typedef HANDLE HDTLA;

// single line compression object types
typedef HANDLE HCBMP;
typedef HANDLE HCPCX;
typedef HANDLE HCPNG;
typedef HANDLE HCJPG;
typedef HANDLE HCTGA;
typedef HANDLE HCPAX;
typedef HANDLE HCTLA;

// single TIFF tile/strip reading object
typedef HANDLE  HISTIFFTILEREADER;

// TIFF writer and TIFF line-at-a-time writer
typedef HANDLE HISTIFFWRITER;

// TIFF reader
typedef HANDLE HISTIFFREADER;

// multi-frame GIF object
typedef HANDLE HCGIF;

////////////////////////////////////////////////////////
// EXIF data reader
typedef HANDLE HISEXIFREADER;

////////////////////////////////////////////////////////
// EXIF data writer
typedef HANDLE HISEXIFWRITER;

// IPTC data parser
typedef HANDLE HISIPTC;

// IPTC data writer
typedef HANDLE HISIPTCW;

////////////////////////////////////////////////////////
// color matcher object
typedef HGLOBAL HISCOLORMATCH;

////////////////////////////////////////////////////////
// add-in library I/O hook object
typedef HANDLE HISEXTHK;

typedef HANDLE HISGIFANI ;

////////////////////////////////////////////////////////
// photoshop resource manager object
typedef HANDLE HISPSDRM;

////////////////////////////////////////////////////////
// GeoTIFF tag reader object
typedef HANDLE HISGEOTIFFREADER;

////////////////////////////////////////////////////////
// all the image read functions use this flag to read to DIB
#define READ_FLAG_DIB                       (1<<1) // = 0x02

////////////////////////////////////////////////////////
// image write functions use this flag to signal that the 
// input image is vertically flipped
#define IS_FILE_WRITE_INPUT_VFLIPPED       (1<<2)  // = 0x04

////////////////////////////////////////////////////////
// image write functions use this flag to signal that the 
// input image uses BGR order for 24 and 32bpp images
#define IS_FILE_WRITE_INPUT_BGR            (1<<3)  // = 0x08

////////////////////////////////////////////////////////
// disable multithreaded processing for this call
#define IS_NO_MULTITHREADED_PROCESS        (1<<20)

////////////////////////////////////////////////////////
// flag for image component sizes
#define IS_IMGPROC_8_UINT     (0)
#define IS_IMGPROC_16_UINT    (0x200)     // bit #9
#define IS_IMGPROC_64_DOUBLE  (0x400)     // bit #10
#define IS_IMGPROC_32_FLOAT   (0x600)     // bit #9 | #10

// currently these only apply to is6_SobelFilter
#define IS_FILTER_NO_LEFT_BOUNDSCHECK    (1<<20)
#define IS_FILTER_NO_RIGHT_BOUNDSCHECK   (1<<21)
#define IS_FILTER_NO_TOP_BOUNDSCHECK     (1<<22)
#define IS_FILTER_NO_BOTTOM_BOUNDSCHECK  (1<<23)


////////////////////////////////////////////////////////
// easy way to create 32-bit color values
#define RGBA(_r,_g,_b,_a)     (RGB((_r),(_g),(_b)) | ((_a) << 24))

////////////////////////////////////////////////////////
// resize method constants
enum
{
   is6_RESIZE_MODE_BOX              = 0, 
   is6_RESIZE_MODE_TRIANGLE         = 1, 
   is6_RESIZE_MODE_HAMMING          = 2, 
   is6_RESIZE_MODE_GAUSSIAN         = 3, 
   is6_RESIZE_MODE_BELL             = 4, 
   is6_RESIZE_MODE_BSPLINE          = 5, 
   is6_RESIZE_MODE_CUBIC1           = 6, 
   is6_RESIZE_MODE_CUBIC2           = 7, 
   is6_RESIZE_MODE_LANCZOS3         = 8, 
   is6_RESIZE_MODE_MITCHELL         = 9, 
   is6_RESIZE_MODE_SINC             = 10,
   is6_RESIZE_MODE_HERMITE          = 11,
   is6_RESIZE_MODE_HANNING          = 12,
   is6_RESIZE_MODE_CATROM           = 13,
   is6_RESIZE_MODE_AREAAVG_FAST     = 14,
   is6_RESIZE_MODE_AREAAVG          = 15,
   is6_RESIZE_MODE_BILINEAR         = 16,
   is6_RESIZE_MODE_BICUBIC          = 17,
   is6_RESIZE_MODE_NEAREST_NEIGHBOR = 18
};

////////////////////////////////////////////////////////
// 1-bit overlay method constants
enum
{
   is6_OVERLAY_1BIT_SET   = 0,
   is6_OVERLAY_1BIT_AND   = 1,
   is6_OVERLAY_1BIT_OR    = 2,
   is6_OVERLAY_1BIT_XOR   = 3,
   is6_OVERLAY_1BIT_NOR   = 4,
   is6_OVERLAY_1BIT_NAND  = 5,
   is6_OVERLAY_1BIT_XNOR  = 6
};
   
////////////////////////////////////////////////////////
// EXIF data types
enum {
    is6_EXIFTYPE_BYTE    = 1,       /*  8-bit unsigned integer */
    is6_EXIFTYPE_ASCII    = 2,       /*  8-bit bytes w/ last byte null */
    is6_EXIFTYPE_SHORT    = 3,       /* 16-bit unsigned integer */
    is6_EXIFTYPE_LONG    = 4,       /* 32-bit unsigned integer */
    is6_EXIFTYPE_RATIONAL    = 5,    /* 64-bit unsigned fraction */
    is6_EXIFTYPE_SBYTE    = 6,       /*  8-bit signed integer */
    is6_EXIFTYPE_UNDEFINED    = 7,    /*  8-bit untyped data */
    is6_EXIFTYPE_SSHORT    = 8,       /* 16-bit signed integer */
    is6_EXIFTYPE_SLONG    = 9,       /* 32-bit signed integer */
    is6_EXIFTYPE_SRATIONAL    = 10,    /* 64-bit signed fraction */
    is6_EXIFTYPE_FLOAT    = 11,       /* 32-bit IEEE floating point */
    is6_EXIFTYPE_DOUBLE    = 12,       /* 64-bit IEEE floating point */
};

// GeoTIFF data types (note, these are not the same as the EXIF types)
enum 
{
   is6_GeoTIFFTYPE_BYTE=1,         /*  8-bit unsigned integer */       
   is6_GeoTIFFTYPE_SHORT=2,        /* 16-bit unsigned integer */      
   is6_GeoTIFFTYPE_LONG=3,         /* 32-bit unsigned integer */      
   is6_GeoTIFFTYPE_RATIONAL=4,     /* 64-bit unsigned fraction */     
   is6_GeoTIFFTYPE_ASCII=5,        /*  8-bit bytes w/ last byte null */
   is6_GeoTIFFTYPE_FLOAT=6,        /* 32-bit IEEE floating point */  
   is6_GeoTIFFTYPE_DOUBLE=7,       /* 64-bit IEEE floating point */
   is6_GeoTIFFTYPE_SBYTE=8,        /*  8-bit signed integer */       
   is6_GeoTIFFTYPE_SSHORT=9,       /* 16-bit signed integer */       
   is6_GeoTIFFTYPE_SLONG=10,       /* 32-bit signed fraction */      
   is6_GeoTIFFTYPE_UNKNOWN=11      /*  8-bit untyped data */
};                          

// PNG settings mask values
enum
{
    is6_PNG_SetFilter                     = 0x01,
    is6_PNG_SetCompressionLevel           = 0x02,
    is6_PNG_SetCompressionMemory          = 0x04,
    is6_PNG_SetCompressionStrategy        = 0x08,
    is6_PNG_SetCompressionWindowBits      = 0x10
};


enum
{
   is6_XYZRGB_Companding_None    = 0,
   is6_XYZRGB_Companding_sRGB    = 1,
   is6_XYZRGB_Companding_LStar   = 2,
   is6_XYZRGB_Companding_Gamma   = 3,
};

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//    Do not change anything in the following section unless you are building the
//    library (static lib or DLL) from the ImgSource source code. Changing anything
//    here outside of an ImgSource build situation will likely cause 
//    compilation and/or linking errors.
//
//    begin: DO NOT CHANGE

//////
// _ISCConv can be defined as __stdcall, __fastcall, __cdecl, etc., and the
// DLL will work fine from C/C++ and C#. But, __stdcall is preferred because 
// that's what VB6 requires. C/C++ and C# are flexible about calling convention.
#define _ISCConv __stdcall    

#ifdef __cplusplus
#define EXTERNC extern "C"
#else
#define EXTERNC extern
#endif

#ifdef IS60_USING_DLL

    // all DLL functions are prefixed with "_" .  ex. _is6_ReadJPG
    #  define _ISFn(fn) _##fn

    // functions look like this:
    // _ISDeclSpec returnType _ISCConv _ISFn(fnName)(...)

    #   if !defined(_ISDeclSpec)
            // **ONLY** define IS60_BUILD_DLL if you are building the DLL from source
    #       if defined(IS60_BUILD_DLL)
    #           define     _ISDeclSpec EXTERNC __declspec(dllexport)
    #       else
    #           define     _ISDeclSpec EXTERNC __declspec(dllimport)
    #       endif
    #   endif
    #else              
        // for static lib building and usage, these macros do nothing
    #   define _ISFn(fn) fn
    #   define     _ISDeclSpec EXTERNC
#endif

//    end: DO NOT CHANGE
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

#endif