/**
 * Copyright (c) 2008 by The Shoe Shine Boy.
 * 
 * This file is part of Java Dependency Finder.
 * 
 * Java Dependency Finder is free software: you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or (at your 
 * option) any later version.
 *
 * Java Dependency Finder is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with 
 * Java Dependency Finder. If not, see <http://www.gnu.org/licenses/>.
 */
package com.gt;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * The Java Dependency Finder Main file. It simply does the following tasks:
 * 1. Loads all the specified jars into a hashtable
 * 2. Provides a console based input where user can enter a class name
 * 3. Looks up that class name in the hashtable to find out the corresponding jar file
 * 4. Program ends if user presses CTRL+C
 * 
 * @author The Shoe Shine Boy
 *
 */
public class JavaDependencyFinder implements Runnable {
	// The hash of jars
	private Hashtable m_jarsHash = new Hashtable();
	
	/**
	 * The constructor. This takes only one argument for now, i.e. the folders containing all the jars/zips
	 * 
	 * @param args	String[] (for now only 1 argument is needed, i.e. 'deps'
	 */
	public JavaDependencyFinder(String[] args) {
		if (args.length == 1 && args[0].startsWith("deps=")) {
			StringTokenizer stTok = new StringTokenizer(args[0].substring(5), ",");
			while (stTok.hasMoreTokens()) {
				File f = new File(stTok.nextToken());
				if (f.exists() && f.isDirectory()) {
					addAllJars(f);
				} else {
					System.err.println(f.getName() + " :: Cannot find this folder.");
				}
			}
		} else {
			System.err.println("Usage: java -jar jdf.jar deps=<comma separated list of folders having all the jars/zips you'd like to include in your search>");
			System.exit(1);
		}
	}
	
	/**
	 * Recursively looks into the File folder and adds all the jars/zips into the hash.
	 * 
	 * @param f		The File (either a folder or a jar/zip)
	 */
	private void addAllJars(File f) {
		if (f.isDirectory()) {
			File[] files = f.listFiles();
			for (int i=0; i<files.length; i++) {
				addAllJars(files[i]);
			}
		} else {
			if (f.getName().toLowerCase().endsWith(".jar") || f.getName().toLowerCase().endsWith(".zip")) {
				try {
					JarFile jf = new JarFile(f);
					Enumeration entries = jf.entries();
					while (entries.hasMoreElements()) {
						String entryName = ((JarEntry)entries.nextElement()).getName();
						if (entryName.endsWith(".class")) {
							entryName = entryName.substring(0, entryName.indexOf("."));
							StringTokenizer stTok = new StringTokenizer(entryName, "/");
							String str = "";
							while (stTok.hasMoreTokens()) {
								str += stTok.nextToken() + ".";
							}
							str = str.substring(0, str.length()-1);
							m_jarsHash.put(str, f.getCanonicalPath());
						} else {
							// skip it
						}
					}
					jf.close();
				} catch (IOException e) {
					System.err.println(f.getName() + " :: " + e.getMessage());
				}
			} else {
				// skip this file
			}
		}
	}
	
	/**
	 * The main method.
	 * 
	 * @param args	The runtime arguments
	 */
	public static void main(String[] args) {
		System.out.println("Initializing Dependy finder...");
		JavaDependencyFinder df = new JavaDependencyFinder(args);
		System.out.println("Ready. [Press CTRL+C to break]");
		new Thread(df).start();
	}

	/**
	 * The thread's run() method.
	 */
	public void run() {
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		System.out.print("\nEnter class to find: ");
		while (true) {
			try {
				String c = br.readLine();
				if (c!= null) {
					String jar = (String)m_jarsHash.get(c);
					if (jar != null) {
						System.out.println("Found in jar: " + jar);
					} else {
						System.out.println("Class not found!");
					}
					System.out.print("\nEnter class to find: ");
				} else {
					// continue
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
}
