package neutrino.dialogs;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.ListDataListener;
import static javax.swing.JOptionPane.*;
import neutrino.text.BackupManager;

/**
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class DocumentRestorer {

	private class AbstractRestoreDialog extends JDialog {
		
		protected JList lBackups = new JList();
		protected JButton bClear = new JButton("Clear");
		protected JButton bClearAll = new JButton("Clear All");
		protected JButton bCancel = new JButton("Cancel");
		protected int option = JOptionPane.CANCEL_OPTION;
		protected List<File> backups = null;
		protected ListModel listModel = new ListModel() {

			public void addListDataListener(ListDataListener l) { }

			public Object getElementAt(int index) {
				return getFilePath(backups.get(index));
			}

			public int getSize() {
				return backups.size();
			}

			public void removeListDataListener(ListDataListener l) { }
			
		};
		
		public AbstractRestoreDialog(JFrame owner) {
			super(owner);
		}
		

		protected String getFilePath(File backup) {
			ObjectInputStream stream = null;
			String filePath = null;
			try {
				stream = new ObjectInputStream(new FileInputStream(backup));
				try {
					filePath = (String) stream.readObject();
				} catch (EOFException e) {
					e.printStackTrace();
				}
			} catch (FileNotFoundException e) {
				filePath = null;
			} catch (IOException e) {
				filePath = null;
			} catch (ClassNotFoundException e) {
				filePath = null;
			} finally {
				if (stream != null) {
					try {
						stream.close();
					} catch (IOException e) {
						filePath = null;
					}
				}
			}
			if (filePath == null) {
				backups = getBackups(getOwner());
				if (!backups.isEmpty()) {
					lBackups.setSelectedIndex(0);
				} else {
					this.option = JOptionPane.CANCEL_OPTION;
					setVisible(false);
				}
				lBackups.revalidate();
				lBackups.repaint();
			}
			return filePath;
		}
		
		public int getOption() {
			return this.option;
		}
		
	}
	
	private class RestoreDialog extends AbstractRestoreDialog implements ActionListener {
		
		private JButton bOk = new JButton("Ok");
		protected File selectedBackup = null;
		
		public RestoreDialog(JFrame owner, List<File> backups) {
			super(owner);
			this.backups = backups;
			setTitle("Restore");
			setModal(true);
			setResizable(false);
			getRootPane().setDefaultButton(bOk);
			setDefaultCloseOperation(DISPOSE_ON_CLOSE);
			
			lBackups.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			lBackups.setModel(listModel);
			Dimension size = new Dimension(400, 300);
			lBackups.setMinimumSize(size);
			lBackups.setPreferredSize(size);
			lBackups.setMaximumSize(size);
			lBackups.setSelectedIndex(0);
			
			JPanel pBackups = new JPanel();
			pBackups.setBorder(new CompoundBorder(new TitledBorder("Backup"), new EmptyBorder(0, 5, 5, 5)));
			pBackups.setLayout(new BorderLayout());
			pBackups.add(new JScrollPane(lBackups), BorderLayout.CENTER);
			
			
			bOk.setMnemonic(KeyEvent.VK_O);
			bClear.setMnemonic(KeyEvent.VK_C);
			bClearAll.setMnemonic(KeyEvent.VK_A);
			bCancel.setMnemonic(KeyEvent.VK_N);
			
			bOk.addActionListener(this);
			bClear.addActionListener(this);
			bClearAll.addActionListener(this);
			bCancel.addActionListener(this);
			
			
			GridBagLayout layout = new GridBagLayout();
			getContentPane().setLayout(layout);
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 5;
			c.insets = new Insets(5, 5, 5, 0);
			getContentPane().add(pBackups, c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(10, 5, 0, 5);
			getContentPane().add(bOk, c);
			c.gridy = 1;
			c.insets = new Insets(5, 5, 0, 5);
			getContentPane().add(bClear, c);
			c.gridy = 2;
			getContentPane().add(bClearAll, c);
			c.gridy = 3;
			getContentPane().add(bCancel, c);		
			pack();
		}
		
		public File getSelectedBackup() {
			return selectedBackup;
		}

		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == bOk) {
				int index = lBackups.getSelectedIndex();
				if (index == -1) return;
				selectedBackup = backups.get(index);
				this.option = JOptionPane.OK_OPTION;
				setVisible(false);
			} else if (e.getSource() == bClear) {
				int index = lBackups.getSelectedIndex();
				if (index == -1) return;
				backups.get(index).delete();
				backups.remove(index);
				if (backups.isEmpty()) {
					this.option = JOptionPane.CANCEL_OPTION;
					setVisible(false);
				}					
				lBackups.repaint();
			} else if (e.getSource() == bClearAll) {
				Iterator<File> iterator = backups.iterator();
				while (iterator.hasNext()) {
					iterator.next().delete();
				}
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			} else if (e.getSource() == bCancel) {
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			}
		}
		
	}
	
	private class MultiRestoreDialog extends AbstractRestoreDialog implements ActionListener {
		
		private JButton bRestore = new JButton("Restore");
		private JButton bRestoreAll = new JButton("Restore All");
		protected File[] selectedBackups = null;
		
		public MultiRestoreDialog(JFrame owner, List<File> backups) {
			super(owner);
			this.backups = backups;
			setTitle("Restore from backup");
			setModal(true);
			setResizable(false);
			getRootPane().setDefaultButton(bRestore);
			setDefaultCloseOperation(DISPOSE_ON_CLOSE);
			
			lBackups.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
			lBackups.setModel(listModel);
			Dimension size = new Dimension(400, 300);
			lBackups.setMinimumSize(size);
			lBackups.setPreferredSize(size);
			lBackups.setMaximumSize(size);
			lBackups.setSelectedIndex(0);
			
			JPanel pBackups = new JPanel();
			pBackups.setBorder(new CompoundBorder(new TitledBorder("Backup"), new EmptyBorder(0, 5, 5, 5)));
			pBackups.setLayout(new BorderLayout());
			pBackups.add(new JScrollPane(lBackups), BorderLayout.CENTER);
			
			
			bRestore.setMnemonic(KeyEvent.VK_R);
			bRestoreAll.setMnemonic(KeyEvent.VK_A);
			bClear.setMnemonic(KeyEvent.VK_C);
			bClearAll.setMnemonic(KeyEvent.VK_L);
			bCancel.setMnemonic(KeyEvent.VK_N);
			
			bRestore.addActionListener(this);
			bRestoreAll.addActionListener(this);
			bClear.addActionListener(this);
			bClearAll.addActionListener(this);
			bCancel.addActionListener(this);
			
			
			GridBagLayout layout = new GridBagLayout();
			getContentPane().setLayout(layout);
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 6;
			c.insets = new Insets(5, 5, 5, 0);
			getContentPane().add(pBackups, c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(10, 5, 0, 5);
			getContentPane().add(bRestore, c);
			c.gridy = 1;
			c.insets = new Insets(5, 5, 0, 5);
			getContentPane().add(bRestoreAll, c);
			c.gridy = 2;
			c.insets = new Insets(5, 5, 0, 5);
			getContentPane().add(bClear, c);
			c.gridy = 3;
			getContentPane().add(bClearAll, c);
			c.gridy = 4;
			getContentPane().add(bCancel, c);		
			pack();
		}
		
		public File[] getSelectedBackups() {
			return selectedBackups;
		}

		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == bRestore) {
				int[] indices = lBackups.getSelectedIndices();
				if (indices == null || indices.length == 0) return;
				selectedBackups = new File[indices.length];
				for (int i = 0; i < indices.length; i++) {
					selectedBackups[i] = backups.get(indices[i]);
				}
				this.option = JOptionPane.OK_OPTION;
				setVisible(false);
			} else if (e.getSource() == bRestoreAll) {
				selectedBackups = new File[backups.size()];
				for (int i = 0; i < selectedBackups.length; i++) {
					selectedBackups[i] = backups.get(i);
				}
				this.option = JOptionPane.OK_OPTION;
				setVisible(false);
			} else if (e.getSource() == bClear) {
				int[] indices = lBackups.getSelectedIndices();
				if (indices == null || indices.length == 0) return;
				ArrayList<File> selectedFiles = new ArrayList<File>(); 
				for (int i = 0; i < indices.length; i++) {
					backups.get(indices[i]).delete();
					selectedFiles.add(backups.get(indices[i]));
				}
				backups.removeAll(selectedFiles);
				if (backups.isEmpty()) {
					this.option = JOptionPane.CANCEL_OPTION;
					setVisible(false);
				}					
				lBackups.repaint();
			} else if (e.getSource() == bClearAll) {
				Iterator<File> iterator = backups.iterator();
				while (iterator.hasNext()) {
					iterator.next().delete();
				}
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			} else if (e.getSource() == bCancel) {
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			}
		}
		
	}
	
	private static void establishBounds(JDialog dialog, Window owner) {
	      Dimension d1 = dialog.getSize();
	      Dimension d2 = owner.getSize();
	      Dimension ds = dialog.getToolkit().getScreenSize();
	      int x = Math.max((d2.width-d1.width)/2, 0);
	      int y = Math.max((d2.height-d1.height)/2, 0);
	      int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
	      int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
	      dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
	}
	
	public static ArrayList<File> getBackups(Window owner) {
		File backupsDirectory = new File(BackupManager.BACKUPS_DIRECTORY);
		if (!backupsDirectory.isDirectory()) {
			showMessageDialog(owner, "Can not locate backups directory", "Error", ERROR_MESSAGE);
			return null;
		}
		ArrayList<File> list = new ArrayList<File>();
		File[] files = backupsDirectory.listFiles();
		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			if (file.isFile() && file.getAbsolutePath().endsWith(BackupManager.BACKUP_FILE_EXTENSION)) {
				list.add(file);
			}
		}
		return list;
	}
	
	/**
	 * Creates and shows restore dialog. Returns selected backup file or null othrewise.
	 * @return File
	 */
	public static File showRestoreDialog(JFrame owner) {
		ArrayList<File> list = getBackups(owner);
		if (list.isEmpty()) {
			showMessageDialog(owner, "There is no backups", "Message", INFORMATION_MESSAGE);
			return null;
		}
		File backup = null;
		DocumentRestorer plainDocumentRestorer = new DocumentRestorer();
		RestoreDialog dialog = plainDocumentRestorer.new RestoreDialog(owner, list);
		establishBounds(dialog, owner);
		dialog.setVisible(true);
		if (dialog.getOption() == JOptionPane.OK_OPTION) {
			backup = dialog.getSelectedBackup();
		}
		dialog.dispose();
		return backup;
	}
	
	/**
	 * Creates and shows restore dialog. Returns selected backup files or null othrewise.
	 * @return File
	 */
	public static File[] showMultiRestoreDialog(JFrame owner) {
		ArrayList<File> list = getBackups(owner);
		if (list.isEmpty()) {
			showMessageDialog(owner, "There is no backups", "Message", INFORMATION_MESSAGE);
			return null;
		}
		File[] backups = null;
		DocumentRestorer plainDocumentRestorer = new DocumentRestorer();
		MultiRestoreDialog dialog = plainDocumentRestorer.new MultiRestoreDialog(owner, list);
		establishBounds(dialog, owner);
		dialog.setVisible(true);
		if (dialog.getOption() == JOptionPane.OK_OPTION) {
			backups = dialog.getSelectedBackups();
		}
		dialog.dispose();
		return backups;
	}
	
	public static boolean hasBackup() {
		File backupsDirectory = new File(BackupManager.BACKUPS_DIRECTORY);
		if (!backupsDirectory.isDirectory()) {
			return false;
		}
		File[] files = backupsDirectory.listFiles();
		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			if (file.isFile() && file.getAbsolutePath().endsWith(BackupManager.BACKUP_FILE_EXTENSION)) {
				return true;
			}
		}
		return false;
	}
	
	public static String getTextComponentClassName(File backup) {
		ObjectInputStream stream = null;
		String className = null;
		try {
			stream = new ObjectInputStream(new FileInputStream(backup));
			try {
				stream.readObject(); // read file name
				stream.readObject(); // read document
				className = (String) stream.readObject();
			} catch (EOFException e) {
				e.printStackTrace();
			}
		} catch (FileNotFoundException e) {
			className = null;
		} catch (IOException e) {
			className = null;
		} catch (ClassNotFoundException e) {
			className = null;
		} finally {
			if (stream != null) {
				try {
					stream.close();
				} catch (IOException e) {
					className = null;
				}
			}
		}
		return className;
	}
	
}
