package neutrino.dialogs;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.nio.charset.Charset;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.border.TitledBorder;

/**
 * Dialog for choosing character encoding.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class EncodingChooser {
	
	private class EncodingDialog extends JDialog implements ActionListener {
		private JButton bOk = new JButton("Ok");
		private JButton bCancel = new JButton("Cancel");
		private JButton bDefault = new JButton("Default");
		private JList lEncoding = new JList(); 
		private JScrollPane spEncoding = new JScrollPane();
		private JPanel pEncoding = new JPanel();
		private int option = JOptionPane.CANCEL_OPTION;
		private Charset encoding = null;
		private Charset defaultEncoding = null;
			
		private JPanel getMainPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 4;
			c.insets = new Insets(5, 5, 5, 5);
			c.fill = GridBagConstraints.BOTH;
			panel.add(pEncoding, c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.insets = new Insets(10, 0, 0, 5);
			c.fill = GridBagConstraints.HORIZONTAL;
			panel.add(bOk, c);
			c.gridx = 1;
			c.gridy = 1;
			c.gridheight = 1;
			c.insets = new Insets(5, 0, 0, 5);
			c.fill = GridBagConstraints.HORIZONTAL;
			panel.add(bCancel, c);
			c.gridx = 1;
			c.gridy = 2;
			c.gridheight = 1;
			c.insets = new Insets(5, 0, 0, 5);
			c.fill = GridBagConstraints.HORIZONTAL;
			panel.add(bDefault, c);
			return panel;
		}

		private void initEncodingList() {
			Iterator<String> iterator = Charset.availableCharsets().keySet().iterator();
			Vector<String> listData = new Vector<String>();
			while (iterator.hasNext()) {
				listData.add(iterator.next());
			}
			lEncoding.setListData(listData);
		}
		
		public EncodingDialog(JFrame owner, Charset currentEncoding, Charset defaultEncoding) {
			super(owner);
			setTitle("Encoding");
			setModal(true);
			setResizable(false);
			setDefaultCloseOperation(DISPOSE_ON_CLOSE);
			
			getRootPane().setDefaultButton(bOk);
			bOk.setMnemonic(KeyEvent.VK_O);
			bCancel.setMnemonic(KeyEvent.VK_C);
			bDefault.setMnemonic(KeyEvent.VK_D);
			bOk.addActionListener(this);
			bCancel.addActionListener(this);
			bDefault.addActionListener(this);
			spEncoding.setMinimumSize(new Dimension(200, 200));
			spEncoding.setPreferredSize(new Dimension(200, 200));
			spEncoding.getViewport().setView(lEncoding);
			pEncoding.setBorder(new TitledBorder("Encoding"));
			pEncoding.add(spEncoding);
			getContentPane().add(getMainPanel());
			initEncodingList();
			if (defaultEncoding != null) {
				this.defaultEncoding = defaultEncoding;
			} else {
				this.defaultEncoding = Charset.defaultCharset();
			}
			lEncoding.setSelectedValue(currentEncoding.name(), true);
			
			pack();
		}
		
		/**
		 * Return OK_OPTION or CANCEL_OPTION respectively to pressed button. 
		 * @return
		 */
		public int getOption() {
			return this.option;
		}
		
		/**
		 * Return selected encoding or null is encoding is not selected. 
		 * @return
		 */
		public Charset getSelectedEncoding() {
			return this.encoding;
		}
		
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == bOk) {
				this.option = JOptionPane.OK_OPTION;
				String charsetName = (String) lEncoding.getSelectedValue();
				if (charsetName != null && Charset.isSupported(charsetName)) {
					this.encoding = Charset.forName(charsetName);
				}
				setVisible(false);
			} else if (e.getSource() == bCancel) {
				this.option = JOptionPane.CANCEL_OPTION;
				setVisible(false);
			} else if (e.getSource() == bDefault) {
				lEncoding.setSelectedValue(defaultEncoding.name(), true);
			}
		}
	}
	
	private static void establishBounds(JDialog dialog, Window owner) {
	      Dimension d1 = dialog.getSize();
	      Dimension d2 = owner.getSize();
	      Dimension ds = dialog.getToolkit().getScreenSize();
	      int x = Math.max((d2.width-d1.width)/2, 0);
	      int y = Math.max((d2.height-d1.height)/2, 0);
	      int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
	      int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
	      dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
	}
		
	/**
	 * Create and show encoding dialog. Return selected encoding ot null.
	 * @param owner - parent frame
	 * @param currentEncoding - current selected encoding 
	 * @param defaultEncoding - default encoding 
	 * @return selected encoding or null
	 */
	public static Charset showEncodingDialog(JFrame owner, Charset currentEncoding, Charset defaultEncoding) {
		Charset encoding = null;
		EncodingChooser encodingChooser = new EncodingChooser();
		EncodingDialog dialog = encodingChooser.new EncodingDialog(owner, currentEncoding, defaultEncoding);
		establishBounds(dialog, owner);
		dialog.setVisible(true);
		if (dialog.getOption() == JOptionPane.OK_OPTION) {
			encoding = dialog.getSelectedEncoding();
		}
		dialog.dispose();
		return encoding;
	}

	/**
	 * Create and show encoding dialog. Return selected encoding ot null.
	 * @param owner - parent frame
	 * @param currentEncoding - current selected encoding 
	 * @return selected encoding or null
	 */
	public static Charset showEncodingDialog(JFrame owner, Charset currentEncoding) {
		Charset encoding = null;
		EncodingChooser encodingChooser = new EncodingChooser();
		EncodingDialog dialog = encodingChooser.new EncodingDialog(owner, currentEncoding, null);
		establishBounds(dialog, owner);
		dialog.setVisible(true);
		if (dialog.getOption() == JOptionPane.OK_OPTION) {
			encoding = dialog.getSelectedEncoding();
		}
		dialog.dispose();
		return encoding;
	}

}
