package neutrino.dialogs;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.*;
import javax.swing.border.EtchedBorder;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import static javax.swing.JOptionPane.*;
import neutrino.OpenList;

/**
 * Font choosing dialog.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class FontChooser {

	private class FontDialog extends JDialog implements ActionListener {
		
		  protected int m_option = JOptionPane.CLOSED_OPTION;
		  protected OpenList m_olFontName;
		  protected OpenList m_olFontSize;
		  protected JCheckBox m_cbBold = new JCheckBox("Bold");;
		  protected JCheckBox m_cbItalic = new JCheckBox("Italic");
		  protected JLabel m_lPreview;
		  private JButton bOk = new JButton("OK");
		  private JButton bCancel = new JButton("Cancel");
		  
		  private JPanel getFontPanel() {
				GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
				String[] fontNames = ge.getAvailableFontFamilyNames();		
				String[] fontSizes = new String[] {"8", "9", "10", "11", "12", "14",
							"16", "18", "20", "22", "24", "26", "28", "36", "48", "72"};
				ListSelectionListener lsel = new ListSelectionListener() {
					public void valueChanged(ListSelectionEvent e) {
						updatePreview();
					}
				};
				KeyListener keyListener = new KeyListener() {
					@Override
					public void keyTyped(KeyEvent e) { }
					@Override
					public void keyReleased(KeyEvent e) {
						updatePreview();
					}
					@Override
					public void keyPressed(KeyEvent e) { }
				};

				m_olFontName = new OpenList("Name:");
				m_olFontName.setDisplayedMnemonic(KeyEvent.VK_N);
				m_olFontName.setFixedSelectedValue(true);
				m_olFontName.setListData(fontNames, true);
				m_olFontName.addListSelectionListener(lsel);
				m_olFontName.addKeyListener(keyListener);
				
				m_olFontSize = new OpenList("Size:");
				m_olFontSize.setDisplayedMnemonic(KeyEvent.VK_S);
				m_olFontSize.setFixedSelectedValue(false);
				m_olFontSize.setListData(fontSizes, false);
				m_olFontSize.addListSelectionListener(lsel);
				m_olFontSize.addKeyListener(keyListener);

				JPanel p = new JPanel(new GridLayout(1, 2, 10, 2));
				p.setBorder(new TitledBorder(new EtchedBorder(), "Font"));
				p.add(m_olFontName);
				p.add(m_olFontSize);
				return p;
		  }
		  
		  private JPanel getEffectsPanel() {
				m_cbBold.setMnemonic(KeyEvent.VK_B);
				m_cbItalic.setMnemonic(KeyEvent.VK_I);
				ActionListener lst = new ActionListener() { 
					public void actionPerformed(ActionEvent e) {
						updatePreview();
					}
				};
				m_cbBold.addActionListener(lst);
				m_cbItalic.addActionListener(lst);
				JPanel p = new JPanel(new GridLayout(1, 2, 10, 5));
				p.setBorder(new TitledBorder(new EtchedBorder(), "Effects"));
				p.add(m_cbBold);
				p.add(m_cbItalic);
				return p;
		  }
		  
		  private JPanel getPreviewPanel() {
				JPanel p = new JPanel(new BorderLayout());
				p.setBorder(new TitledBorder(new EtchedBorder(), "Preview"));
				m_lPreview = new JLabel("Preview Font", JLabel.CENTER);
				m_lPreview.setBackground(Color.white);
				m_lPreview.setForeground(Color.black);
				m_lPreview.setOpaque(true);
				m_lPreview.setBorder(new LineBorder(Color.black));
				m_lPreview.setPreferredSize(new Dimension(120, 50));
				p.add(m_lPreview, BorderLayout.CENTER);
				return p;
		  }
		  
		  private JPanel getButtonsPanel() {
				bOk.addActionListener(this);
				bOk.setToolTipText("Set font");
				bOk.setMnemonic(KeyEvent.VK_O);
				getRootPane().setDefaultButton(bOk);
				bCancel.setMnemonic(KeyEvent.VK_C);
				bCancel.setToolTipText("Abort font choosing dialog");
				bCancel.addActionListener(this);
				JPanel p = new JPanel(new FlowLayout());
				JPanel p1 = new JPanel(new GridLayout(1, 2, 10, 2));
				p1.add(bOk);
				p1.add(bCancel);
				p.add(p1);
				return p;
		  }
		  
		  private void initialize(Font currentFont) {
			    m_olFontName.setSelectedValue(currentFont.getFamily());
			    m_olFontSize.setSelectedValue(Integer.toString(currentFont.getSize()));
			    m_cbBold.setSelected(currentFont.isBold());
			    m_cbItalic.setSelected(currentFont.isItalic());
		  }

		  public FontDialog(JFrame parent, Font currentFont) {
			  super(parent, "Font", true);
			  getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.Y_AXIS));
			  getContentPane().add(getFontPanel());
			  getContentPane().add(getEffectsPanel());
			  getContentPane().add(Box.createVerticalStrut(5));
			  getContentPane().add(getPreviewPanel());
			  getContentPane().add(getButtonsPanel());
			  pack();
			  setResizable(false);
			  initialize(currentFont);
			  updatePreview();
		  }

		  public Font getSelectedFont() {
			  String name = m_olFontName.getSelectedValue();
			  int size;
			  try {
				  size = Integer.parseInt(m_olFontSize.getSelectedValue());
				  if (size <= 0) throw new NumberFormatException("Size mast be positive");
			  } catch (NumberFormatException e) {
				  size = 16;
			  }

			  int style = Font.PLAIN;
			  if (m_cbBold.isSelected())
				  style |= Font.BOLD;
			  if (m_cbItalic.isSelected())
				  style |= Font.ITALIC;

			  Font fn = new Font(name, style, size);
			  return fn;
		  }

		  public int getOption() { 
			  return m_option; 
		  }

		  protected void updatePreview() {
			  m_lPreview.setFont(getSelectedFont());
			  m_lPreview.repaint();
		  }

		  private boolean validateInput() {
			  try {
				  int fontSize = Integer.parseInt(m_olFontSize.getSelectedValue().toString());
				  if (fontSize <= 0) {
					  showMessageDialog(getParent(), "Font size mast be positive integer", "Error message", ERROR_MESSAGE);
					  return false;
				  }
			  } catch (NumberFormatException e) {
				  showMessageDialog(getParent(), "Font size mast be positive integer", "Error message", ERROR_MESSAGE);
				  return false;
			  }
			  return true;
		  }
		  
		@Override
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == bOk) {
				if (validateInput()) {
				    m_option = JOptionPane.OK_OPTION;
				    setVisible(false);
				}
			} else if (e.getSource() == bCancel) {
			    m_option = JOptionPane.CANCEL_OPTION;
			    setVisible(false);
			}
		}
	}	

	private static void establishBounds(JDialog dialog, Window owner) {
      Dimension d1 = dialog.getSize();
      Dimension d2 = owner.getSize();
      Dimension ds = dialog.getToolkit().getScreenSize();
      int x = Math.max((d2.width-d1.width)/2, 0);
      int y = Math.max((d2.height-d1.height)/2, 0);
      int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
      int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
      dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
	}
	
	/**
	 * Show font dialog. Return selected font.
	 * @param parent - parent frame
	 * @param currentFont - current font
	 * @return selected font or null
	 */
	public static Font showFontDialog(JFrame parent, Font currentFont) {
		Font font = null;
		FontChooser fontChooser = new FontChooser();
	    FontDialog fontDialog = fontChooser.new FontDialog(parent, currentFont);
	    establishBounds(fontDialog, parent);
        fontDialog.setVisible(true);
		if (fontDialog.getOption()==JOptionPane.OK_OPTION) {
			font = fontDialog.getSelectedFont();
		}
		fontDialog.dispose();
		return font;
	}

}
