package neutrino.dialogs;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.text.*;
import static javax.swing.JOptionPane.*;

/**
 * Contains GoToLine dialog for changing current line in TextComponent.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class LineChooser {
  
  private class GoToLineDialog extends JDialog implements ActionListener {
    private JLabel lLine = new JLabel();
    private JTextField tfLine = new JTextField(20);
    private JButton bOk = new JButton("Ok");
    private JButton bCancel = new JButton("Cancel");
    private JTextComponent textComponent;
    
    private class NumberDocument extends PlainDocument {      
      private boolean isNumber(String str) {
        char[] content = str.toCharArray();
        for (char c : content) {
          if (c < '0' || c > '9') {
            return false;
          }
        }
        return true;
      }
      
        public void replace(int offset, int length, String text,
                    AttributeSet attrs) throws BadLocationException {
            if (length == 0 && (text == null || text.length() == 0)) {
                return;
            }
            if (text != null && !isNumber(text)) {
              return;
            }
          super.replace(offset, length, text, attrs);
        }
        
        public void insertString(int offs, String str, AttributeSet a) throws BadLocationException {
            if ((str == null) || (str.length() == 0) || (!isNumber(str))) {
              return;
          }
          super.insertString(offs, str, a);
        }
    }
    
    public GoToLineDialog(JFrame owner, JTextComponent textComponent) {
      super(owner);
      this.textComponent = textComponent;
      setTitle("Go to line");
      setModal(true);
      setResizable(false);
      setDefaultCloseOperation(DISPOSE_ON_CLOSE);
      getContentPane().add(getMainPanel());
      pack();
    }
    
    private JPanel getMainPanel() {
      int lineNumber = textComponent.getDocument().
        getDefaultRootElement().getElementIndex(textComponent.getCaretPosition()) + 1;
      int lineCount = textComponent.getDocument().getDefaultRootElement().getElementCount();
      String text = "Line number (1 - " + lineCount + ") :";
      lLine.setText(text);
      lLine.setLabelFor(tfLine);
      lLine.setDisplayedMnemonic(KeyEvent.VK_L);
      tfLine.setDocument(new NumberDocument());
      tfLine.setText("" + lineNumber);
      tfLine.selectAll();
      tfLine.grabFocus();
      JPanel panel = new JPanel();
      panel.setBorder(new EmptyBorder(5, 5, 5, 5));
      GridBagLayout layout = new GridBagLayout();
      panel.setLayout(layout);
      GridBagConstraints c = new GridBagConstraints();
      c.gridx = 0;
      c.gridy = 0;
      c.anchor = GridBagConstraints.WEST;
      c.insets = new Insets(0, 0, 5, 0);
      panel.add(lLine, c);
      c.gridx = 0;
      c.gridy = 1;
      c.anchor = GridBagConstraints.WEST;
      c.insets = new Insets(0, 0, 5, 0);
      panel.add(tfLine, c);
      c.gridx = 0;
      c.gridy = 3;
      c.anchor = GridBagConstraints.EAST;
      c.insets = new Insets(0, 0, 0, 0);
      panel.add(getButtonsPanel(), c);
      return panel;
    }
    
    private JPanel getButtonsPanel() {
      bOk.setMnemonic(KeyEvent.VK_O);
      bCancel.setMnemonic(KeyEvent.VK_C);
      this.getRootPane().setDefaultButton(bOk);
      bOk.addActionListener(this);
      bCancel.addActionListener(this);
      JPanel panel = new JPanel();
      BoxLayout layout = new BoxLayout(panel, BoxLayout.X_AXIS);
      panel.setLayout(layout);
      panel.add(Box.createHorizontalGlue());
      panel.add(bOk);
      panel.add(Box.createHorizontalStrut(5));
      panel.add(bCancel);
      return panel;
    }

    public void actionPerformed(ActionEvent e) {
      if (e.getSource() == bOk || e.getSource() == bCancel) {
        if (e.getSource() == bOk) {
          int lineNumber = 0;
          int lineCount = textComponent.getDocument().
            getDefaultRootElement().getElementCount();
          try {
            lineNumber = Integer.parseInt(tfLine.getText());
          } catch (NumberFormatException e1) {
            showMessageDialog(this.getOwner(), "Line number is not valid", "Error", ERROR_MESSAGE);
            return;
          }
          if (lineNumber < 1) {
            showMessageDialog(this.getOwner(), "Line number is less than one", "Error", ERROR_MESSAGE);
            return;
          }
          if (lineNumber > lineCount) {
            showMessageDialog(this.getOwner(), "Line number is greater than number of  lines in document", "Error", ERROR_MESSAGE);
            return;
          }
          int position = textComponent.getDocument().
            getDefaultRootElement().getElement(lineNumber - 1).getStartOffset();
          textComponent.setCaretPosition(position);
        }
        setVisible(false);
        dispose();
      }
    }
    
  }

	private static void establishBounds(JDialog dialog, Window owner) {
      Dimension d1 = dialog.getSize();
      Dimension d2 = owner.getSize();
      Dimension ds = dialog.getToolkit().getScreenSize();
      int x = Math.max((d2.width-d1.width)/2, 0);
      int y = Math.max((d2.height-d1.height)/2, 0);
      int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
      int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
      dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
	}
  
  /**
   * Shows "Go to line" dialog for JTextComponent
   * <p>Precondition : text component should be not empty</p>
   * @param owner - the frame owner of dialog
   * @param textComponent - the text component
   */
  public static void showGoToLineDialog(JFrame owner, JTextComponent textComponent) {
	  boolean isTextEmpty = textComponent.getDocument().getLength() == 0;
	  if (isTextEmpty) {
		  showMessageDialog(owner, "Text is empty", "Warning", WARNING_MESSAGE);
		  return;
	  }
    LineChooser lineChooser = new LineChooser();
    GoToLineDialog dialog = lineChooser.new GoToLineDialog(owner, textComponent);    
    establishBounds(dialog, owner);
    dialog.setVisible(true);
  }

}
