package neutrino.dialogs;

import java.awt.*;
import java.awt.event.*;
import java.util.Random;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.text.*;
import javax.swing.text.rtf.RTFEditorKit;

import static javax.swing.JOptionPane.*;
import neutrino.OpenList;

/*
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class StyleChooser {
	
	private class SmallToggleButton extends JToggleButton implements ItemListener
	{
	  protected Border m_raised;
	  protected Border m_lowered;

	  public SmallToggleButton(boolean selected, ImageIcon imgUnselected, 
	   ImageIcon imgSelected, String tip) 
	  {
	    super(imgUnselected, selected);
	    setHorizontalAlignment(CENTER);
	    setBorderPainted(true);
	    m_raised = new BevelBorder(BevelBorder.RAISED);
	    m_lowered = new BevelBorder(BevelBorder.LOWERED);
	    setBorder(selected ? m_lowered : m_raised);
	    setMargin(new Insets(1,1,1,1));
	    setToolTipText(tip);
	    setRequestFocusEnabled(false);
	    setSelectedIcon(imgSelected);
	    addItemListener(this);
	  }

	  public float getAlignmentY() { return 0.5f; }

	  public void itemStateChanged(ItemEvent e) {
	    setBorder(isSelected() ? m_lowered : m_raised);
	  }
	}

	private class ParagraphDialog extends JDialog implements ActionListener {
		
		protected int m_option = JOptionPane.CLOSED_OPTION;
		protected MutableAttributeSet m_attributes;
		protected JLabel lLineSpacing = new JLabel("Line spacing:");
		protected JLabel lSpaceAbove = new JLabel("Space above:");
		protected JLabel lSpaceBelow = new JLabel("Space below:");
		protected JLabel lFirstIndent = new JLabel("First indent:");
		protected JLabel lLeftIndent = new JLabel("Left indent:");
		protected JLabel lRightIndent = new JLabel("Right indent:");
		protected JTextField tfLineSpacing = new JTextField();
		protected JTextField tfSpaceAbove = new JTextField();
		protected JTextField tfSpaceBelow = new JTextField();
		protected JTextField tfFirstIndent = new JTextField();
		protected JTextField tfLeftIndent = new JTextField();
		protected JTextField tfRightIndent = new JTextField();
		private ImageIcon iconLeft = new ImageIcon(ParagraphDialog.class.getClassLoader().getResource("neutrino/dialogs/images/al_left.gif"));
		private ImageIcon iconCenter = new ImageIcon(ParagraphDialog.class.getClassLoader().getResource("neutrino/dialogs/images/al_center.gif"));
		private ImageIcon iconRight = new ImageIcon(ParagraphDialog.class.getClassLoader().getResource("neutrino/dialogs/images/al_right.gif"));
		private ImageIcon iconJustify = new ImageIcon(ParagraphDialog.class.getClassLoader().getResource("neutrino/dialogs/images/al_justify.gif"));
		protected SmallToggleButton m_btLeft = new SmallToggleButton(false, iconLeft, iconLeft, "Left");
		protected SmallToggleButton m_btCenter = new SmallToggleButton(false, iconCenter, iconCenter, "Center");
		protected SmallToggleButton m_btRight = new SmallToggleButton(false, iconRight, iconRight, "Right");
		protected SmallToggleButton m_btJustified = new SmallToggleButton(false, iconJustify, iconJustify, "Justify");
	    private JButton btOK = new JButton("OK");
	    private JButton btCancel = new JButton("Cancel");
		protected ParagraphPreview m_preview;
	  
		private JPanel getSpacePanel() {
			lLineSpacing.setDisplayedMnemonic(KeyEvent.VK_S);
			lLineSpacing.setLabelFor(tfLineSpacing);
			lSpaceAbove.setDisplayedMnemonic(KeyEvent.VK_A);
			lSpaceAbove.setLabelFor(tfSpaceAbove);
			lSpaceBelow.setDisplayedMnemonic(KeyEvent.VK_B);
			lSpaceBelow.setLabelFor(tfSpaceBelow);
		    tfLineSpacing.addFocusListener(flst);
		    tfSpaceAbove.addFocusListener(flst);
		    tfSpaceBelow.addFocusListener(flst);
			JPanel ps = new JPanel(new GridLayout(3, 2, 10, 2));
			ps.setBorder(new TitledBorder(new EtchedBorder(), "Space"));
			ps.add(lLineSpacing);
			ps.add(tfLineSpacing);
			ps.add(lSpaceAbove);
			ps.add(tfSpaceAbove);
			ps.add(lSpaceBelow);
			ps.add(tfSpaceBelow);
			return ps;
		}
		
		private JPanel getIndentPanel() {
			lFirstIndent.setLabelFor(tfFirstIndent);
			lFirstIndent.setDisplayedMnemonic(KeyEvent.VK_F);
			lLeftIndent.setLabelFor(tfLeftIndent);
			lLeftIndent.setDisplayedMnemonic(KeyEvent.VK_L);
			lRightIndent.setLabelFor(tfRightIndent);
			lRightIndent.setDisplayedMnemonic(KeyEvent.VK_R);
		    tfFirstIndent.addFocusListener(flst);
		    tfLeftIndent.addFocusListener(flst);
		    tfRightIndent.addFocusListener(flst);
			
		    JPanel pi = new JPanel(new GridLayout(3, 2, 10, 2));
		    pi.setBorder(new TitledBorder(new EtchedBorder(), "Indent"));
		    pi.add(lFirstIndent);
		    pi.add(tfFirstIndent);
		    pi.add(lLeftIndent);
		    pi.add(tfLeftIndent);
		    pi.add(lRightIndent);
		    pi.add(tfRightIndent);
		    return pi;
		}
		
		private JPanel getAlignmentPanel() {
		    m_btLeft.addActionListener(this);
		    m_btCenter.addActionListener(this);
		    m_btRight.addActionListener(this);
		    m_btJustified.addActionListener(this);
		    
			JPanel p = new JPanel();
		    p.setLayout(new BoxLayout(p, BoxLayout.X_AXIS));
		    p.add(Box.createHorizontalStrut(10));
		    p.add(new JLabel("Alignment:"));
		    p.add(Box.createHorizontalStrut(20));

		    ButtonGroup bg = new ButtonGroup();
		    bg.add(m_btLeft);
		    p.add(m_btLeft);
		    bg.add(m_btCenter);
		    p.add(m_btCenter);
		    bg.add(m_btRight);
		    p.add(m_btRight);
		    bg.add(m_btJustified);
		    p.add(m_btJustified);
			return p;
		}
		
		private JPanel getPreviewPanel() {
		    JPanel p = new JPanel(new BorderLayout());
		    p.setBorder(new TitledBorder(new EtchedBorder(), "Preview"));
		    m_preview = new ParagraphPreview();
		    p.add(m_preview, BorderLayout.CENTER);
		    return p;
		}
		
		private JPanel getButtonsPanel() {
			btOK.setMnemonic(KeyEvent.VK_O);
			btCancel.setMnemonic(KeyEvent.VK_C);
			getRootPane().setDefaultButton(btOK);
			btOK.setToolTipText("Set paragraph style");
			btCancel.setToolTipText("Abort paragraph chooser dialog");
		    btOK.addActionListener(this);
		    btCancel.addActionListener(this);
		    JPanel p = new JPanel(new FlowLayout());
		    JPanel p1 = new JPanel(new GridLayout(1, 2, 10, 2));
		    p1.add(btOK);
		    p1.add(btCancel);
		    p.add(p1);
		    return p;
		}
	  
	  public ParagraphDialog(JFrame parent) {
	    super(parent, "Paragraph", true);
	    getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.Y_AXIS));
	    JPanel p = new JPanel(new GridLayout(1, 2, 5, 2));
	    p.add(getSpacePanel());
	    p.add(getIndentPanel());
	    getContentPane().add(p);
	    getContentPane().add(Box.createVerticalStrut(5));
	    getContentPane().add(getAlignmentPanel());
	    getContentPane().add(getPreviewPanel());
	    getContentPane().add(getButtonsPanel());
	    pack();
	    setResizable(false);
	  }
	  
	  private FocusListener flst = new FocusListener() { 
		  public void focusGained(FocusEvent e) {}
		  public void focusLost(FocusEvent e) { 
			  updatePreview(); 
		  }
	  };

	  public void setAttributes(AttributeSet a) {
	    m_attributes = new SimpleAttributeSet(a);
	    tfLineSpacing.setText(Float.toString(
	      StyleConstants.getLineSpacing(a)));
	    tfSpaceAbove.setText(Float.toString(
	      StyleConstants.getSpaceAbove(a)));
	    tfSpaceBelow.setText(Float.toString(
	      StyleConstants.getSpaceBelow(a)));
	    tfFirstIndent.setText(Float.toString(
	      StyleConstants.getFirstLineIndent(a)));
	    tfLeftIndent.setText(Float.toString(
	      StyleConstants.getLeftIndent(a)));
	    tfRightIndent.setText(Float.toString(
	      StyleConstants.getRightIndent(a)));

	    int alignment = StyleConstants.getAlignment(a);
	    if (alignment == StyleConstants.ALIGN_LEFT)
	      m_btLeft.setSelected(true);
	    else if (alignment == StyleConstants.ALIGN_CENTER)
	      m_btCenter.setSelected(true);
	    else if (alignment == StyleConstants.ALIGN_RIGHT)
	      m_btRight.setSelected(true);
	    else if (alignment == StyleConstants.ALIGN_JUSTIFIED)
	      m_btJustified.setSelected(true);

	    updatePreview();
	  }

	  public AttributeSet getAttributes() {
	    if (m_attributes == null)
	      return null;
	    float value;
	    try { 
	      value = Float.parseFloat(tfLineSpacing.getText());
	      StyleConstants.setLineSpacing(m_attributes, value); 
	    } catch (NumberFormatException ex) {}
	    try { 
	      value = Float.parseFloat(tfSpaceAbove.getText());
	      StyleConstants.setSpaceAbove(m_attributes, value); 
	    } catch (NumberFormatException ex) {}
	    try { 
	      value = Float.parseFloat(tfSpaceBelow.getText());
	      StyleConstants.setSpaceBelow(m_attributes, value); 
	    } catch (NumberFormatException ex) {}
	    try { 
	      value = Float.parseFloat(tfFirstIndent.getText());
	      StyleConstants.setFirstLineIndent(m_attributes, value); 
	    } catch (NumberFormatException ex) {}
	    try { 
	      value = Float.parseFloat(tfLeftIndent.getText());
	      StyleConstants.setLeftIndent(m_attributes, value); 
	    } catch (NumberFormatException ex) {}
	    try { 
	      value = Float.parseFloat(tfRightIndent.getText());
	      StyleConstants.setRightIndent(m_attributes, value); 
	    } catch (NumberFormatException ex) {}

	    StyleConstants.setAlignment(m_attributes, getAlignment());

	    return m_attributes;
	  }

	  public int getOption() {
	    return m_option;
	  }

	  protected void updatePreview() {
	    m_preview.repaint();
	  }

	  protected int getAlignment() {
	    if (m_btLeft.isSelected())
	      return StyleConstants.ALIGN_LEFT;
	    if (m_btCenter.isSelected())
	      return StyleConstants.ALIGN_CENTER;
	    else if (m_btRight.isSelected())
	      return StyleConstants.ALIGN_RIGHT;
	    else
	      return StyleConstants.ALIGN_JUSTIFIED;
	  }

	  class ParagraphPreview extends JPanel
	  {
	    protected Font m_fn = new Font("Monospace", Font.PLAIN, 6);
	    protected String m_dummy = "abcdefghjklm";
	    protected float  m_scaleX = 0.25f;
	    protected float  m_scaleY = 0.25f;
	    protected Random m_random = new Random();

	    public ParagraphPreview() {
	      setBackground(Color.white);
	      setForeground(Color.black);
	      setOpaque(true);
	      setBorder(new LineBorder(Color.black));
	      setPreferredSize(new Dimension(120, 56));
	    }

	    public void paintComponent(Graphics g) {
	      super.paintComponent(g);
	      float lineSpacing = 0;
	      float spaceAbove  = 0;
	      float spaceBelow  = 0;
	      float firstIndent = 0;
	      float leftIndent  = 0;
	      float rightIndent = 0;

	      try { 
	        lineSpacing = Float.parseFloat(tfLineSpacing.getText());
	      } catch (NumberFormatException ex) {}
	      try { 
	        spaceAbove = Float.parseFloat(tfSpaceAbove.getText()); 
	      } catch (NumberFormatException ex) {}
	      try { 
	        spaceBelow = Float.parseFloat(tfSpaceBelow.getText()); 
	      } catch (NumberFormatException ex) {}
	      try { 
	        firstIndent = Float.parseFloat(tfFirstIndent.getText()); 
	      } catch (NumberFormatException ex) {}
	      try { 
	        leftIndent = Float.parseFloat(tfLeftIndent.getText()); 
	      } catch (NumberFormatException ex) {}
	      try { 
	        rightIndent = Float.parseFloat(tfRightIndent.getText()); 
	      } catch (NumberFormatException ex) {}

	      m_random.setSeed(1959);    // Use same seed every time

	      g.setFont(m_fn);
	      FontMetrics fm = g.getFontMetrics();
	      int h = fm.getAscent();
	      int s  = Math.max((int)(lineSpacing*m_scaleY), 1);
	      int s1 = Math.max((int)(spaceAbove*m_scaleY), 0) + s;
	      int s2 = Math.max((int)(spaceBelow*m_scaleY), 0) + s;
	      int y = 5+h;

	      int xMarg = 20;
	      int x0 = Math.max((int)(firstIndent*m_scaleX)+xMarg, 3);
	      int x1 = Math.max((int)(leftIndent*m_scaleX)+xMarg, 3);
	      int x2 = Math.max((int)(rightIndent*m_scaleX)+xMarg, 3);
	      int xm0 = getWidth()-xMarg;
	      int xm1 = getWidth()-x2;
	            
	      int n = (int)((getHeight()-(2*h+s1+s2-s+10))/(h+s));
	      n = Math.max(n, 1);

	      g.setColor(Color.lightGray);
	      int x = xMarg;
	      drawLine(g, x, y, xm0, xm0, fm, StyleConstants.ALIGN_LEFT);
	      y += h+s1;

	      g.setColor(Color.gray);
	      int alignment = getAlignment();
	      for (int k=0; k<n; k++) {
	        x = (k==0 ? x0 : x1);
	        int xLen = (k==n-1 ? xm1/2 : xm1);
	        if (k==n-1 && alignment==StyleConstants.ALIGN_JUSTIFIED)
	          alignment = StyleConstants.ALIGN_LEFT;
	        drawLine(g, x, y, xm1, xLen, fm, alignment);
	        y += h+s;
	      }

	      y += s2-s;
	      x = xMarg;
	      g.setColor(Color.lightGray);
	      drawLine(g, x, y, xm0, xm0, fm, StyleConstants.ALIGN_LEFT);
	    }

	    protected void drawLine(Graphics g, int x, int y, int xMax, 
	     int xLen, FontMetrics fm, int alignment)
	    {
	      if (y > getHeight()-3)
	        return;
	      StringBuffer s = new StringBuffer();
	      String str1;
	      int xx = x;
	      while (true) {
	        int m = m_random.nextInt(10)+1;
	        str1 = m_dummy.substring(0, m)+" ";
	        int len = fm.stringWidth(str1);
	        if (xx+len >= xLen)
	          break;
	        xx += len;
	        s.append(str1);
	      }
	      String str = s.toString();

	      switch (alignment) {
	        case StyleConstants.ALIGN_LEFT:
	          g.drawString(str, x, y);
	          break;
	        case StyleConstants.ALIGN_CENTER:
	          xx = (xMax+x-fm.stringWidth(str))/2;
	          g.drawString(str, xx, y);
	          break;
	        case StyleConstants.ALIGN_RIGHT:
	          xx = xMax-fm.stringWidth(str);
	          g.drawString(str, xx, y);
	          break;
	        case StyleConstants.ALIGN_JUSTIFIED:
	          while (x+fm.stringWidth(str) < xMax)
	            str += "a";
	          g.drawString(str, x, y);
	          break;
	      }
	    }
	  }

	  	private boolean validateInput() {
	  		try {
	  			Float.parseFloat(tfLineSpacing.getText());
	  		} catch (NumberFormatException e) {
	  			showMessageDialog(getParent(), "Line spacing is not a number", "Error message", ERROR_MESSAGE);
	  			return false;
	  		}
	  		try {
	  			Float.parseFloat(tfSpaceAbove.getText());
	  		} catch (NumberFormatException e) {
	  			showMessageDialog(getParent(), "Space above is not a number", "Error message", ERROR_MESSAGE);
	  			return false;
	  		}
	  		try {
	  			Float.parseFloat(tfSpaceBelow.getText());
	  		} catch (NumberFormatException e) {
	  			showMessageDialog(getParent(), "Space below is not a number", "Error message", ERROR_MESSAGE);
	  			return false;
	  		}
	  		try {
	  			Float.parseFloat(tfFirstIndent.getText());
	  		} catch (NumberFormatException e) {
	  			showMessageDialog(getParent(), "First indent is not a number", "Error message", ERROR_MESSAGE);
	  			return false;
	  		}
	  		try {
	  			Float.parseFloat(tfLeftIndent.getText());
	  		} catch (NumberFormatException e) {
	  			showMessageDialog(getParent(), "Left indent is not a number", "Error message", ERROR_MESSAGE);
	  			return false;
	  		}
	  		try {
	  			Float.parseFloat(tfRightIndent.getText());
	  		} catch (NumberFormatException e) {
	  			showMessageDialog(getParent(), "Right indent is not a number", "Error message", ERROR_MESSAGE);
	  			return false;
	  		}
	  		return true;
	  	}
	  
		@Override
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == m_btLeft || e.getSource() == m_btCenter
					|| e.getSource() == m_btRight || e.getSource() == m_btJustified) {
				updatePreview();
			} else if (e.getSource() == btOK) {
				if (validateInput()) {
			        m_option = JOptionPane.OK_OPTION;
			        setVisible(false);
				}
			} else if (e.getSource() == btCancel) {
		        m_option = JOptionPane.CANCEL_OPTION;
		        setVisible(false);
			}
		}
	
	}

	private class StyleDialog extends JDialog implements ActionListener {
		protected int m_option = JOptionPane.CLOSED_OPTION;
		protected OpenList m_lstFontName;
		protected OpenList m_lstFontSize;
		protected JCheckBox m_chkBold = new JCheckBox("Bold");
		protected JCheckBox m_chkItalic = new JCheckBox("Italic");
		protected JCheckBox m_chkUnderline = new JCheckBox("Underline");
		protected JCheckBox m_chkStrikethrough = new JCheckBox("Strikeout");
		protected JCheckBox m_chkSubscript = new JCheckBox("Subscript");
		protected JCheckBox m_chkSuperscript = new JCheckBox("Superscript");
	    private JButton btOK = new JButton("OK");
	    private JButton btCancel = new JButton("Cancel");
	    private final int PREVIEW_HEIGHT = 70;
		protected JTextPane m_preview;
		
		private JLabel lForeground = new JLabel("Foreground :");
		private JPanel pForeground = new JPanel();
		private JButton bForeground = new JButton("...");
		private Color cForeground = Color.black;
		private JLabel lBackground = new JLabel("Background :");
		private JPanel pBackground = new JPanel();
		private JButton bBackground = new JButton("...");
		private Color cBackground = Color.white;
		
		private JPanel getFontPanel() {
		    JPanel p = new JPanel(new GridLayout(1, 2, 10, 2));
		    p.setBorder(new TitledBorder(new EtchedBorder(), "Font"));
		    GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
		    String[] m_fontNames = ge.getAvailableFontFamilyNames();
		    m_lstFontName = new OpenList("Name:");
		    m_lstFontName.setDisplayedMnemonic(KeyEvent.VK_N);
		    m_lstFontName.setFixedSelectedValue(true);
		    m_lstFontName.setAutoSelectingMode(true);
		    m_lstFontName.setListData(m_fontNames, true);
		    p.add(m_lstFontName);

		    String[] m_fontSizes = new String[] {"8", "9", "10", "11", "12", "14",
		  	      "16", "18", "20", "22", "24", "26", "28", "36", "48", "72"};
		    m_lstFontSize = new OpenList("Size:");
		    m_lstFontSize.setDisplayedMnemonic(KeyEvent.VK_S);
		    m_lstFontSize.setAutoSelectingMode(true);
		    m_lstFontSize.setFixedSelectedValue(false);
		    m_lstFontSize.setListData(m_fontSizes, false);
		    p.add(m_lstFontSize);
		    ListSelectionListener lsel = new ListSelectionListener() {
		    	public void valueChanged(ListSelectionEvent e) {
		    		updatePreview();
		    	}
		    };
		    KeyListener keyListener = new KeyListener() {
				@Override
				public void keyTyped(KeyEvent e) { }
				@Override
				public void keyReleased(KeyEvent e) { 
					updatePreview();
				}
				@Override
				public void keyPressed(KeyEvent e) { }
			};
		    m_lstFontName.addListSelectionListener(lsel);
		    m_lstFontName.addKeyListener(keyListener);
		    m_lstFontSize.addListSelectionListener(lsel);
		    m_lstFontSize.addKeyListener(keyListener);
		    return p;
		}
		
		private JPanel getEffectsPanel() {
			m_chkBold.setMnemonic(KeyEvent.VK_O);
			m_chkItalic.setMnemonic(KeyEvent.VK_I);
			m_chkUnderline.setMnemonic(KeyEvent.VK_U);
			m_chkStrikethrough.setMnemonic(KeyEvent.VK_T);
			m_chkSubscript.setMnemonic(KeyEvent.VK_B);
			m_chkSuperscript.setMnemonic(KeyEvent.VK_P);
			
		    m_chkBold.addActionListener(this);
		    m_chkItalic.addActionListener(this);
		    m_chkUnderline.addActionListener(this);
		    m_chkStrikethrough.addActionListener(this);
		    m_chkSubscript.addActionListener(this);
		    m_chkSuperscript.addActionListener(this);
		    
		    JPanel p = new JPanel(new GridLayout(2, 3, 10, 5));
		    p.setBorder(new TitledBorder(new EtchedBorder(), "Effects"));
		    p.add(m_chkBold);
		    p.add(m_chkItalic);
		    p.add(m_chkUnderline);
		    p.add(m_chkStrikethrough);
		    p.add(m_chkSubscript);
		    p.add(m_chkSuperscript);
		    return p;
		}

		private JPanel getColorPanel() {
			lForeground.setLabelFor(bForeground);
			lForeground.setDisplayedMnemonic(KeyEvent.VK_F);
			bForeground.setMnemonic(KeyEvent.VK_F);
			lBackground.setLabelFor(bBackground);
			lBackground.setDisplayedMnemonic(KeyEvent.VK_B);
			bBackground.setMnemonic(KeyEvent.VK_B);
			bForeground.setToolTipText("Select foreground color");
			bBackground.setToolTipText("Select background color");
			
			final int HEIGHT = lForeground.getPreferredSize().height + 5;
			pForeground.setPreferredSize(new Dimension(HEIGHT, HEIGHT));
			pBackground.setPreferredSize(new Dimension(HEIGHT, HEIGHT));
			pForeground.setMaximumSize(new Dimension(HEIGHT, HEIGHT));
			pBackground.setMaximumSize(new Dimension(HEIGHT, HEIGHT));
			pForeground.setBackground(Color.blue);
			pBackground.setBackground(Color.blue);
			pForeground.setBorder(new LineBorder(Color.black));
			pBackground.setBorder(new LineBorder(Color.black));
			
			bForeground.setPreferredSize(new Dimension(HEIGHT + 5, HEIGHT));
			bBackground.setPreferredSize(new Dimension(HEIGHT + 5, HEIGHT));
			bForeground.setMaximumSize(new Dimension(HEIGHT + 5, HEIGHT));
			bBackground.setMaximumSize(new Dimension(HEIGHT + 5, HEIGHT));
			bForeground.addActionListener(new ActionListener() {
				@Override
				public void actionPerformed(ActionEvent e) {
					Color color = JColorChooser.showDialog(getParent(), "Foreground color", cForeground);
					if (color != null) {
						cForeground = color;
						updatePreview();
					}
				}
			});
			bBackground.addActionListener(new ActionListener() {
				@Override
				public void actionPerformed(ActionEvent e) {
					Color color = JColorChooser.showDialog(getParent(), "Background color", cBackground);
					if (color != null) {
						cBackground = color;
						updatePreview();
					}
				}
			});
			
			JPanel fp = new JPanel();
			fp.setBorder(new EmptyBorder(5, 5, 5, 5));
			fp.setLayout(new BoxLayout(fp, BoxLayout.X_AXIS));
			fp.add(lForeground);
			fp.add(Box.createHorizontalStrut(5));
			fp.add(pForeground);
			fp.add(Box.createHorizontalStrut(5));
			fp.add(bForeground);
			JPanel bp = new JPanel();
			bp.setBorder(new EmptyBorder(5, 5, 5, 5));
			bp.setLayout(new BoxLayout(bp, BoxLayout.X_AXIS));
			bp.add(lBackground);
			bp.add(Box.createHorizontalStrut(5));
			bp.add(pBackground);
			bp.add(Box.createHorizontalStrut(5));
			bp.add(bBackground);
			
			JPanel p = new JPanel(new GridLayout(1, 2, 10, 5));
			p.setBorder(new TitledBorder(new EtchedBorder(), "Colors"));
			p.add(fp);
			p.add(bp);
			return p;
		}
		
		private JPanel getButtonsPanel() {
			getRootPane().setDefaultButton(btOK);
			btOK.setMnemonic(KeyEvent.VK_O);
			btCancel.setMnemonic(KeyEvent.VK_C);
		    btOK.addActionListener(this);
		    btCancel.addActionListener(this);
		    btOK.setToolTipText("Set style");
		    btCancel.setToolTipText("Abort style choosing dialog");
		    JPanel p = new JPanel(new FlowLayout());
		    JPanel p1 = new JPanel(new GridLayout(1, 2, 10, 2));
		    p1.add(btOK);
		    p1.add(btCancel);
		    p.add(p1);		
		    return p;
		}
		
		private JPanel getPreviewPanel() {
		    JPanel p = new JPanel(new BorderLayout());
		    p.setBorder(new TitledBorder(new EtchedBorder(), "Preview"));
		    m_preview = new JTextPane();
		    m_preview.setEditorKit(new RTFEditorKit());
		    try {
		    	String text = "Preview Style";
				m_preview.getStyledDocument().insertString(0, text, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		    m_preview.setBackground(Color.white);
		    m_preview.setForeground(Color.black);
		    m_preview.setOpaque(true);
		    m_preview.setBorder(new LineBorder(Color.black));
		    m_preview.setPreferredSize(new Dimension(120, PREVIEW_HEIGHT));
		    p.add(m_preview, BorderLayout.CENTER);
		    return p;
		}

	  public StyleDialog(JFrame parent) {
	    super(parent, "Style", true);
	    getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.Y_AXIS));
	    getContentPane().add(getFontPanel());
	    getContentPane().add(getEffectsPanel());
	    getContentPane().add(getColorPanel());
	    getContentPane().add(Box.createVerticalStrut(5));
	    getContentPane().add(getPreviewPanel());
	    getContentPane().add(getButtonsPanel());
	    pack();
	    setResizable(false);
	  }

	  public void setAttributes(AttributeSet a) {
		  String name = StyleConstants.getFontFamily(a);
		  m_lstFontName.setSelectedValue(name);
		  int size = StyleConstants.getFontSize(a);
		  m_lstFontSize.setSelectedValue(Integer.toString(size));
		  m_chkBold.setSelected(StyleConstants.isBold(a));
		  m_chkItalic.setSelected(StyleConstants.isItalic(a));
		  m_chkUnderline.setSelected(StyleConstants.isUnderline(a));
		  m_chkStrikethrough.setSelected(StyleConstants.isStrikeThrough(a));
		  m_chkSubscript.setSelected(StyleConstants.isSubscript(a));
		  m_chkSuperscript.setSelected(StyleConstants.isSuperscript(a));
		  cForeground = StyleConstants.getForeground(a);
		  cBackground = StyleConstants.getBackground(a);
		  updatePreview();
	  }

	  public AttributeSet getAttributes() {
		  MutableAttributeSet m_attributes = new SimpleAttributeSet();
		  StyleConstants.setFontFamily(m_attributes, m_lstFontName.getSelectedValue());
		  StyleConstants.setFontSize(m_attributes, Integer.parseInt(m_lstFontSize.getSelectedValue()));
		  StyleConstants.setBold(m_attributes, m_chkBold.isSelected());
		  StyleConstants.setItalic(m_attributes, m_chkItalic.isSelected());
		  StyleConstants.setUnderline(m_attributes, m_chkUnderline.isSelected());
		  StyleConstants.setStrikeThrough(m_attributes, m_chkStrikethrough.isSelected());
		  StyleConstants.setSubscript(m_attributes, m_chkSubscript.isSelected());
		  StyleConstants.setSuperscript(m_attributes, m_chkSuperscript.isSelected());
		  StyleConstants.setForeground(m_attributes, cForeground);
		  StyleConstants.setBackground(m_attributes, cBackground);
		  return m_attributes;
	  }

	  public int getOption() { 
		  return m_option; 
	  }

	  protected void updatePreview() {
		  int size;
		  try {
			  size = Integer.parseInt(m_lstFontSize.getSelectedValue());
			  if (size <= 0) throw new NumberFormatException("Size mast be positive");
		  } catch (NumberFormatException e) {
			  size = 16;
		  }
		  pForeground.setBackground(cForeground);
		  pBackground.setBackground(cBackground);
		  m_preview.setForeground(cForeground);
		  m_preview.setBackground(cBackground);
		  MutableAttributeSet paragraphAttributes = new SimpleAttributeSet();
		  StyleConstants.setAlignment(paragraphAttributes, StyleConstants.ALIGN_CENTER);
		  StyleConstants.setSpaceAbove(paragraphAttributes, (PREVIEW_HEIGHT - size) / 2 );
		  m_preview.getStyledDocument().setParagraphAttributes(0, m_preview.getStyledDocument().getLength(), paragraphAttributes, false);
		  m_preview.getStyledDocument().setCharacterAttributes(0, m_preview.getStyledDocument().getLength(), getAttributes(), false);
		  m_preview.repaint();
	  }
	  
	  private boolean validateInput() {
		  try {
			  int fontSize = Integer.parseInt(m_lstFontSize.getSelectedValue().toString());
			  if (fontSize <= 0) {
				  showMessageDialog(getParent(), "Font size mast be positive integer", "Error message", ERROR_MESSAGE);
				  return false;
			  }
		  } catch (NumberFormatException e) {
			  showMessageDialog(getParent(), "Font size mast be positive integer", "Error message", ERROR_MESSAGE);
			  return false;
		  }
		  return true;
	  }

		@Override
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == m_chkBold || e.getSource() == m_chkItalic
					|| e.getSource() == m_chkUnderline || e.getSource() == m_chkStrikethrough
					|| e.getSource() == m_chkSubscript || e.getSource() == m_chkSuperscript) {
				updatePreview();
			} else if (e.getSource() == btOK) {
				if (validateInput()) {
			        m_option = JOptionPane.OK_OPTION;
			        setVisible(false);
				}
			} else if (e.getSource() == btCancel) {
		        m_option = JOptionPane.CANCEL_OPTION;
		        setVisible(false);				
			}
		}
	}


	private static void establishBounds(JDialog dialog, Window owner) {
      Dimension d1 = dialog.getSize();
      Dimension d2 = owner.getSize();
      Dimension ds = dialog.getToolkit().getScreenSize();
      int x = Math.max((d2.width-d1.width)/2, 0);
      int y = Math.max((d2.height-d1.height)/2, 0);
      int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
      int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
      dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
	}
	
	/**
	 * Shows paragraph dialog. Return selected style.
	 * @param parent - parent frame
	 * @param attributeSet - AttributeSet
	 * @return selected AttributeSet or null
	 */
	public static AttributeSet showParagraphDialog(JFrame parent, AttributeSet attributeSet) {
		StyleChooser styleChooser = new StyleChooser();
	    ParagraphDialog paragraphDialog = styleChooser.new ParagraphDialog(parent);
	    paragraphDialog.setAttributes(attributeSet);
	    establishBounds(paragraphDialog, parent);
	    paragraphDialog.setVisible(true);
		if (paragraphDialog.getOption()==JOptionPane.OK_OPTION) {
			return paragraphDialog.getAttributes();
		}
		return null;
	}

	public static AttributeSet showStyleDialog(JFrame parent, AttributeSet attributeSet) {
		StyleChooser styleChooser = new StyleChooser();
		StyleDialog styleDialog = styleChooser.new StyleDialog(parent);
        styleDialog.setAttributes(attributeSet);
      	establishBounds(styleDialog, parent);
      	styleDialog.setVisible(true);
        if (styleDialog.getOption()==JOptionPane.OK_OPTION) {
        	return styleDialog.getAttributes();
        }
        return null;
	}


}
