package neutrino.dialogs;

import java.awt.*;
import java.awt.event.*;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.text.*;
import neutrino.multitext.EditorChangeEvent;
import neutrino.multitext.EditorChangeListener;
import neutrino.multitext.MultiTextComponent;
import neutrino.text.PlainTextArea;
import neutrino.text.TextComponent;
import neutrino.text.TextEditor;

/**
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class TextFinder {
	
	private static final String FROM_CURSOR = "From cursor"; 
	private static final String SELECTED_TEXT = "Selected text"; 
	private static final String ENTIRE_SCOPE = "Entire scope"; 
	private static final String ALL_TEXTS = "All texts"; 
	private static HashSet<String> m_keys = new HashSet<String>();
	private static Hashtable<String, HashSet<String>> m_replacements = new Hashtable<String, HashSet<String>>();
	private static ArrayList<String> m_origins = new ArrayList<String>();
	static {
		m_origins.add(FROM_CURSOR);
		m_origins.add(SELECTED_TEXT);
		m_origins.add(ENTIRE_SCOPE);
		m_origins.add(ALL_TEXTS);
	}
	
	private static ComboBoxModel m_modelFind = new ComboBoxModel() {
		private Object selectedItem = null;
		@Override
		public void removeListDataListener(ListDataListener l) { }
		@Override
		public int getSize() {
			return m_keys.size();
		}
		@Override
		public Object getElementAt(int index) {
			return m_keys.toArray()[index];
		}
		@Override
		public void addListDataListener(ListDataListener l) { }
		@Override
		public void setSelectedItem(Object anItem) {
			selectedItem = anItem;
		}
		@Override
		public Object getSelectedItem() {
			return selectedItem;
		}
	}; 
	private static ComboBoxModel m_modelReplace = new ComboBoxModel() {
		private Object selectedItem = null;
		@Override
		public Object getSelectedItem() {
			return selectedItem;
		}
		@Override
		public void setSelectedItem(Object anItem) {
			selectedItem = anItem;
		}
		@Override
		public void addListDataListener(ListDataListener l) { }
		@Override
		public Object getElementAt(int index) {
			HashSet<String> replacements = null;
			try {
				replacements = m_replacements.get(m_modelFind.getSelectedItem());
			} catch (NullPointerException e) { }
			if (replacements == null) return null;
			else {
				try {
					return replacements.toArray()[index];
				} catch (IndexOutOfBoundsException e) {
					return null;
				}
			}
		}
		@Override
		public int getSize() {
			HashSet<String> replacements = null;
			try {
				replacements = m_replacements.get(m_modelFind.getSelectedItem());
			} catch (NullPointerException e) { }
			if (replacements == null) return 0;
			else return replacements.size();
		}
		@Override
		public void removeListDataListener(ListDataListener l) { }
	};
	private static ComboBoxModel m_modelOrigin = new ComboBoxModel() {
		private Object selectedItem = FROM_CURSOR;
		@Override
		public void removeListDataListener(ListDataListener l) { }
		@Override
		public int getSize() {
			return m_origins.size();
		}
		@Override
		public Object getElementAt(int index) {
			return m_origins.get(index);
		}
		@Override
		public void addListDataListener(ListDataListener l) { }
		@Override
		public void setSelectedItem(Object anItem) {
			selectedItem = anItem;
		}
		@Override
		public Object getSelectedItem() {
			return selectedItem;
		}
	};
	private static ButtonModel m_modelWord = null;
	private static ButtonModel m_modelCase = null;
	private static ButtonModel m_modelUp = null;
	private static ButtonModel m_modelDown = null;
	private static ButtonModel m_modelExpression = null;
	private static ButtonModel m_modelIncremental = null;
	
	private static boolean isTextSelected(JTextComponent textComponent) {
		return textComponent.getSelectionStart() != textComponent.getSelectionEnd();
	}
	
	private static void replaceSelection(JTextComponent textComponent, String replacement) {
		if (textComponent instanceof PlainTextArea) {
	    	((PlainTextArea) textComponent).beginEdit();
	    	textComponent.replaceSelection(replacement);
	    	((PlainTextArea) textComponent).endEdit();
		} else if (textComponent instanceof TextComponent) {
	    	((TextComponent) textComponent).beginEdit();
	    	textComponent.replaceSelection(replacement);
	    	((TextComponent) textComponent).endEdit();
		} else {
	    	textComponent.replaceSelection(replacement);
		}
	}
	
	private static int getSelectionLength(JTextComponent textComponent) {
		return textComponent.getSelectionEnd() - textComponent.getSelectionStart();
	}
	
	private static boolean isTextEmpty(JTextComponent textComponent) {
		return textComponent.getDocument().getLength() == 0;
	}
	
	private interface ISearchEngine {
		public void find();
		public void replace();
		public void replaceAll();
		public ItemListener getOptionsListener();
	}
	
	private abstract class AbstractSearchEngine implements ISearchEngine {
		
		protected AbstractSearchDialog dialog = null;
		
		public AbstractSearchEngine(AbstractSearchDialog dialog) {
			this.dialog = dialog;
		}
		
		protected class Selection {
			public int start = 0;
			public int end = 0;
			public Selection(int start, int end) {
				this.start = start;
				this.end = end;
			}
		}

		protected void warning(String message) {
			JOptionPane.showMessageDialog(dialog.getOwner(), message, "Warning", JOptionPane.INFORMATION_MESSAGE);
		}
		
		protected String getKey() {
		    String key = (String) m_modelFind.getSelectedItem(); 
		    if (key == null || key.length()==0) {
		    	warning("Please enter the target to search");
	            return null;
		    } else
		    	return key;
		}
		
		protected String getReplacement() {
		    String replacement = (String) m_modelReplace.getSelectedItem();
		    if (replacement == null || replacement.length() == 0) {
		    	warning("Please enter the replacement");
		    	return null;
		    } else 
		    	return replacement;
		}
		
		protected boolean checkInputForFinding() {
			String key = getKey();
		    if (key == null) return false;
		    m_keys.add(key);
		    dialog.updateFindUI();
			return true;
		}
		
		protected boolean checkInputForReplacement() {
			String key = getKey();
		    if (key == null) return false;
		    String replacement = getReplacement();
		    if (replacement == null) return false;
		    m_keys.add(key);
			if (m_replacements.get(key) == null) {
				m_replacements.put(key, new HashSet<String>());
			}
		    m_replacements.get(key).add(replacement);
		    dialog.updateFindUI();
		    dialog.updateReplaceUI();
		    return true;
		}
		
		protected Pattern getPattern(String key) {
		    String regex;
		    if (m_modelWord.isSelected()) {
		    	StringTokenizer tokenizer = new StringTokenizer(key, " \t\n\f\r", false);
		    	regex = "\\b";
		    	boolean isFirst = true;
		    	while (tokenizer.hasMoreTokens()) {
		    		if (!isFirst) regex += "\\s";
		    		if (isFirst) isFirst = false;
		    		regex += (m_modelExpression.isSelected()) ? tokenizer.nextToken() : Pattern.quote(tokenizer.nextToken());
		    	}
		    	regex += "\\b";
		    } else
		    	regex = (m_modelExpression.isSelected()) ? key  : Pattern.quote(key);
		    Pattern pattern;
		    if (m_modelCase.isSelected())
		    	pattern = Pattern.compile(regex);
		    else
		    	pattern = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
		    return pattern;
		}
		
	}
	
	private final class SearchEngine extends AbstractSearchEngine {

		protected JTextComponent textComponent = null;
		private String currentKey = null;
		private int startPosition = -1;
		
		public SearchEngine(AbstractSearchDialog dialog, JTextComponent textComponent) {
			super(dialog);
			this.textComponent = textComponent;
		}

		private Selection findIncremental(boolean showWarning) {
			if (startPosition < 0)	{ 
		    	if (showWarning) warning("Text not found");
	            return null;
			}
			boolean m_searchUp = m_modelUp.isSelected(); 
			String  m_searchData;
		    int pos = textComponent.getCaretPosition();
		    String key = getKey(); 
		    if (key == null) return null;
			
	    	Document doc = textComponent.getDocument();
		    if (!m_searchUp && pos >= startPosition) pos = 0; 
		    if (m_searchUp) {
		    	if (pos <= startPosition) 
			    	pos = doc.getLength();
			    else if (isTextSelected(textComponent)) {
	    			String selection = textComponent.getSelectedText();
	    			Matcher matcher = getPattern(key).matcher(selection);
	    			if (matcher.matches()) {
	    				pos -= selection.length(); 
	    			}
	    		}
		    }
		    
		    try {
		    	if (m_searchUp) {
		    		m_searchData = doc.getText(startPosition, pos - startPosition);
		    	} else
		    		m_searchData = doc.getText(pos, startPosition - pos);
		    } catch (BadLocationException ex) {
		    	warning(ex.toString());
	            return null;
		    }
		    
		    Matcher matcher = getPattern(key).matcher(m_searchData);
	    	if (m_searchUp) {
	    		int start = 0, end = 0, counter = 0;
    			while (matcher.find()) {
    				start = matcher.start();
    				end = matcher.end();
    				counter++;
    			};
    			if (counter > 0) {
    				Selection selection = new Selection(startPosition + start, startPosition + end);
    				if (counter == 1) startPosition = -1;
	    			return selection;
    			}
	    	} else {
	    		if (matcher.find()) {
	    			Selection selection = new Selection(pos + matcher.start(), pos + matcher.end());
    				if (!matcher.find()) startPosition = -1;
	    			return selection;
	    		}
	    	}
	    	startPosition = -1;
	    	if (showWarning) warning("Text not found");
            return null;	    				
		}
		
		private boolean canFindIncremental() {
			return m_modelIncremental.isSelected() && (startPosition != -1) && (currentKey != null);
		}
		
		private Selection find(boolean showWarning) {
	    	if (m_modelIncremental.isSelected() && (startPosition < 0)) {
		    	if (showWarning) warning("Text not found");
	            return null;
	    	}
			boolean m_searchUp = m_modelUp.isSelected(); // a search direction flag
			String  m_searchData; // string to search for
		    int pos = textComponent.getCaretPosition();
		    String key = getKey(); 
		    if (key == null) return null;
			
		    if (canFindIncremental()
		    		&& ((!m_modelUp.isSelected() && pos < startPosition) || (m_modelUp.isSelected() && pos > startPosition))) {
		    	return findIncremental(showWarning);
		    }
		    
		    try {
		    	Document doc = textComponent.getDocument();
		    	if (m_searchUp) {
		    		if (isTextSelected(textComponent)) {
		    			String selection = textComponent.getSelectedText();
		    			Matcher matcher = getPattern(key).matcher(selection);
		    			if (matcher.matches()) {
		    				pos -= selection.length(); 
		    			}
		    		}
		    		m_searchData = doc.getText(0, pos);
		    	} else
		    		m_searchData = doc.getText(pos, doc.getLength()-pos);
		    } catch (BadLocationException ex) {
		    	warning(ex.toString());
	            return null;
		    }
		    
		    Matcher matcher = getPattern(key).matcher(m_searchData);
		    if (matcher.find()) {
		    	if (m_searchUp) {
		    		int start = matcher.start();
		    		int end = matcher.end();
	    			while (matcher.find()) {
	    				start = matcher.start();
	    				end = matcher.end();
	    			};
	    			return new Selection(start, end);
		    	} else 
		    		return new Selection(pos + matcher.start(), pos + matcher.end());
		    } else {
		    	if (canFindIncremental()) {
		    		return findIncremental(showWarning);
		    	} else {
			    	if (showWarning) warning("Text not found");
		            return null;
		    	}
		    }			
		}
		
		private boolean replace(boolean showWarning) {
			String key = getKey();
		    if (key == null) return false;
		    String replacement = getReplacement();
		    if (replacement == null) return false;
		    
    		if (isTextSelected(textComponent)) {
    			String selectedText = textComponent.getSelectedText();
    			Matcher matcher = getPattern(key).matcher(selectedText);
    			if (matcher.matches()) {
    				int selectionStart = textComponent.getSelectionStart();
    				replaceSelection(textComponent, replacement);
    				textComponent.select(selectionStart, selectionStart + replacement.length());
    				return true;
    			}
    		}		    
		    
			Selection selection = find(showWarning);
			if (selection == null) return false;
			textComponent.select(selection.start, selection.end);
			replaceSelection(textComponent, replacement);
			textComponent.select(selection.start, selection.start + replacement.length());
			return true;
		}
		
		private void updateStartPostition() {
			String key = getKey();
			if (key == null) return;
			if (!key.equals(currentKey)) {
				currentKey = key;
				startPosition = textComponent.getCaretPosition();
			}
		}		
		
		private ItemListener optionsListener = new ItemListener() {
			@Override
			public void itemStateChanged(ItemEvent e) {
				currentKey = null;
			}
		};
		
		public void setTextComponent(JTextComponent textComponent) {
			this.textComponent = textComponent;
		}
		
		@Override
		public ItemListener getOptionsListener() {
			return optionsListener;
		}

		@Override
		public void find() {
			if (!checkInputForFinding()) return;
			updateStartPostition();
			Selection selection = find(true);
			if (selection != null) textComponent.select(selection.start, selection.end);
		}

		@Override
		public void replace() {
			if (!checkInputForReplacement()) return;
			updateStartPostition();
			replace(true);
		}

		@Override
		public void replaceAll() {
			if (!checkInputForReplacement()) return;
			updateStartPostition();
	        int counter = 0;
	        while (replace(false)) counter++;
	        if (counter > 0) {
		        JOptionPane.showMessageDialog(dialog.getOwner(), 
		          counter+" replacement(s) have been done", "Info",
		          JOptionPane.INFORMATION_MESSAGE);
	        } else {
	        	warning("Text not found");
	        }
		}
		
	}
	
	private final class SelectedTextSearchEngine extends AbstractSearchEngine {

		protected JTextComponent textComponent = null;
		
		public SelectedTextSearchEngine(AbstractSearchDialog dialog, JTextComponent textComponent) {
			super(dialog);
			this.textComponent = textComponent;
		}

		private Selection find(boolean showWarning) {
			if (!isTextSelected(textComponent)) {
				warning("Text is not selected");
				return null;
			}
			boolean m_searchUp = m_modelUp.isSelected(); // a search direction flag
			String  m_searchData; // string to search for
		    int selectionStart = textComponent.getSelectionStart();
		    String key = getKey(); 
		    if (key == null) return null;
		    Matcher matcher;
		    
		    matcher = getPattern(key).matcher(textComponent.getSelectedText());
			if (matcher.matches()) {
		    	if (showWarning) warning("Text not found");
	            return null;
			}		
			
		    try {
		    	Document doc = textComponent.getDocument();
	    		m_searchData = doc.getText(selectionStart, getSelectionLength(textComponent));
		    } catch (BadLocationException ex) {
		    	warning(ex.toString());
	            return null;
		    }
		    
		    matcher = getPattern(key).matcher(m_searchData);
		    if (matcher.find()) {
		    	if (m_searchUp) {
		    		int start = matcher.start();
		    		int end = matcher.end();
	    			while (matcher.find()) {
	    				start = matcher.start();
	    				end = matcher.end();
	    			};
	    			return new Selection(selectionStart + start, selectionStart + end);
		    	} else 
		    		return new Selection(selectionStart + matcher.start(), selectionStart + matcher.end());
		    } else {
		    	if (showWarning) warning("Text not found");
	            return null;
		    }			
		}
		
		private boolean replace(boolean showWarning) {
			if (!isTextSelected(textComponent)) {
				warning("Text is not selected");
				return false;
			}
			String key = getKey();
		    if (key == null) return false;
		    String replacement = getReplacement();
		    if (replacement == null) return false;
		    
    		if (isTextSelected(textComponent)) {
    			String selectedText = textComponent.getSelectedText();
    			Matcher matcher = getPattern(key).matcher(selectedText);
    			if (matcher.matches()) {
    				int selectionStart = textComponent.getSelectionStart();
    		    	replaceSelection(textComponent, replacement);
    				textComponent.select(selectionStart, selectionStart + replacement.length());
    				return true;
    			}
    		}		    
		    
			Selection selection = find(showWarning);
			if (selection == null) return false;
			int selectionStart = textComponent.getSelectionStart();
			int selectionLength = getSelectionLength(textComponent);
			textComponent.select(selection.start, selection.end);
	    	replaceSelection(textComponent, replacement);
	    	int difference = replacement.length() - (selection.end - selection.start);
			textComponent.select(selectionStart, selectionStart + selectionLength + difference);
			return true;
		}
		
		public void setTextComponent(JTextComponent textComponent) {
			this.textComponent = textComponent;
		}
		
		@Override
		public ItemListener getOptionsListener() {
			return null;
		}

		@Override
		public void find() {
			if (!checkInputForFinding()) return;
			Selection selection = find(true);
			if (selection != null) textComponent.select(selection.start, selection.end);
		}

		@Override
		public void replace() {
			if (!checkInputForReplacement()) return;
			replace(true);
		}

		@Override
		public void replaceAll() {
			if (!checkInputForReplacement()) return;
			if (!isTextSelected(textComponent)) {
				warning("Text is not selected");
				return;
			}
	        int counter = 0;
	        while (replace(false)) counter++;
	        if (counter > 0) {
		        JOptionPane.showMessageDialog(dialog.getOwner(), 
		          counter+" replacement(s) have been done", "Info",
		          JOptionPane.INFORMATION_MESSAGE);
	        } else {
	        	warning("Text not found");
	        }
		}
		
	}
	
	private abstract class AbstractEntireScopeSearchEngine extends AbstractSearchEngine {

		protected String currentKey = null;
		protected boolean isSearchStarted = false;
		
		public AbstractEntireScopeSearchEngine(AbstractSearchDialog dialog) {
			super(dialog);
		}
		
		protected Selection find(JTextComponent textComponent, boolean showWarning) {
			boolean m_searchUp = m_modelUp.isSelected(); // a search direction flag
			String  m_searchData; // string to search for
		    int pos = textComponent.getCaretPosition();
		    String key = getKey(); 
		    if (key == null) return null;
			
	    	Document doc = textComponent.getDocument();
		    try {
		    	if (!isSearchStarted) {
		    		m_searchData = doc.getText(0, doc.getLength());
		    		pos = 0;
		    	} else {
			    	if (m_searchUp) {
			    		if (isTextSelected(textComponent)) {
			    			String selection = textComponent.getSelectedText();
			    			Matcher matcher = getPattern(key).matcher(selection);
			    			if (matcher.matches()) {
			    				pos -= selection.length(); 
			    			}
			    		}
			    		m_searchData = doc.getText(0, pos);
			    	} else
			    		m_searchData = doc.getText(pos, doc.getLength()-pos);
		    	}
		    } catch (BadLocationException ex) {
		    	warning(ex.toString());
	            return null;
		    }
		    
		    Matcher matcher = getPattern(key).matcher(m_searchData);
		    if (matcher.find()) {
		    	if (m_searchUp) {
		    		int start = matcher.start();
		    		int end = matcher.end();
	    			while (matcher.find()) {
	    				start = matcher.start();
	    				end = matcher.end();
	    			};
	    			return new Selection(start, end);
		    	} else 
		    		return new Selection(pos + matcher.start(), pos + matcher.end());
		    } else {
		    	if (showWarning) warning("Text not found");
	            return null;
		    }			
		}
		
		protected boolean replace(JTextComponent textComponent, boolean showWarning) {
			String key = getKey();
		    if (key == null) return false;
		    String replacement = getReplacement();
		    if (replacement == null) return false;
		    
    		if (isSearchStarted && isTextSelected(textComponent)) {
    			String selectedText = textComponent.getSelectedText();
    			Matcher matcher = getPattern(key).matcher(selectedText);
    			if (matcher.matches()) {
    				int selectionStart = textComponent.getSelectionStart();
    		    	replaceSelection(textComponent, replacement);
    				textComponent.select(selectionStart, selectionStart + replacement.length());
    				return true;
    			}
    		}		    
		    
			Selection selection = find(textComponent, showWarning);
			if (selection == null) return false;
			textComponent.select(selection.start, selection.end);
	    	replaceSelection(textComponent, replacement);
			textComponent.select(selection.start, selection.start + replacement.length());
			return true;
		}
		
	}
	
	private final class EntireScopeSearchEngine extends AbstractEntireScopeSearchEngine {

		protected JTextComponent m_textComponent = null;
		
		public EntireScopeSearchEngine(AbstractSearchDialog dialog, JTextComponent textComponent) {
			super(dialog);
			m_textComponent = textComponent;
		}
		
		private void updateSearch() {
			String key = getKey();
			if (key != null && !key.equals(currentKey)) {
				currentKey = key;
				isSearchStarted = false;
			} else 
				isSearchStarted = true;
		}
		
		private ItemListener optionsListener = new ItemListener() {
			@Override
			public void itemStateChanged(ItemEvent e) {
				currentKey = null;
			}
		};
		
		public void setTextComponent(JTextComponent textComponent) {
			m_textComponent = textComponent;
		}
		
		@Override
		public ItemListener getOptionsListener() {
			return optionsListener;
		}

		@Override
		public void find() {
			if (!checkInputForFinding()) return;
			updateSearch();
			Selection selection = find(m_textComponent, true);
			if (selection != null) m_textComponent.select(selection.start, selection.end);
		}

		@Override
		public void replace() {
			if (!checkInputForReplacement()) return;
			updateSearch();
			replace(m_textComponent, true);
		}

		@Override
		public void replaceAll() {
			if (!checkInputForReplacement()) return;
			updateSearch();
	        int counter = 0;
	        while (replace(m_textComponent, false)) counter++;
	        if (counter > 0) {
		        JOptionPane.showMessageDialog(dialog.getOwner(), 
		          counter+" replacement(s) have been done", "Info",
		          JOptionPane.INFORMATION_MESSAGE);
	        } else {
	        	warning("Text not found");
	        }
		}
		
	}	
	
	
	private final class AllTextsSearchEngine extends AbstractEntireScopeSearchEngine {

		protected MultiTextComponent m_multiTextComponent = null;
		protected TextEditor currentTextEditor = null;
		
		public AllTextsSearchEngine(AbstractSearchDialog dialog, MultiTextComponent multiTextComponent) {
			super(dialog);
			m_multiTextComponent = multiTextComponent;
		}
		
		private void updateSearch() {
			String key = getKey();
			if (key != null && !key.equals(currentKey)) {
				currentKey = key;
				currentTextEditor = m_multiTextComponent.getCurrentTextEditor();
				isSearchStarted = false;
			} else 
				isSearchStarted = true;
		}
		
		private ItemListener optionsListener = new ItemListener() {
			@Override
			public void itemStateChanged(ItemEvent e) {
				currentKey = null;
				currentTextEditor = null;
			}
		};
		
		@Override
		public ItemListener getOptionsListener() {
			return optionsListener;
		}

		@Override
		public void find() {
			if (!checkInputForFinding()) return;
			updateSearch();
			Selection selection = null;
			TextEditor[] textEditors = m_multiTextComponent.getTextEditors();
			if (!m_modelUp.isSelected()) {
				int index = 0;
				if (currentTextEditor != null) {   
					for (; index < textEditors.length; index++) {
						if (currentTextEditor == textEditors[index]) break;
					}
				}
				for (; index < textEditors.length; index++) {
					if (isTextEmpty(textEditors[index].getTextComponent())) continue;
					currentTextEditor = textEditors[index];
					selection = find(textEditors[index].getTextComponent(), false);
					if (selection != null) {
						m_multiTextComponent.activateTab(textEditors[index]);
						textEditors[index].getTextComponent().select(selection.start, selection.end);
						dialog.setVisible(true);
						return;
					}
					isSearchStarted = false;
				}
			} else {
				int index = textEditors.length - 1;
				if (currentTextEditor != null) {   
					for (; index >= 0; index--) {
						if (currentTextEditor == textEditors[index]) break;
					}
				}
				for (; index >= 0; index--) {
					if (isTextEmpty(textEditors[index].getTextComponent())) continue;
					currentTextEditor = textEditors[index];
					selection = find(textEditors[index].getTextComponent(), false);
					if (selection != null) {
						m_multiTextComponent.activateTab(textEditors[index]);
						textEditors[index].getTextComponent().select(selection.start, selection.end);
						dialog.setVisible(true);
						return;
					}
					isSearchStarted = false;
				}
			}
	    	warning("Text not found");
		}

		private boolean performReplacement() {
			TextEditor[] textEditors = m_multiTextComponent.getTextEditors();
			if (!m_modelUp.isSelected()) {
				int index = 0;
				if (currentTextEditor != null) {   
					for (; index < textEditors.length; index++) {
						if (currentTextEditor == textEditors[index]) break;
					}
				}
				for (; index < textEditors.length; index++) {
					if (isTextEmpty(textEditors[index].getTextComponent())) continue;
					currentTextEditor = textEditors[index];
					if (replace(textEditors[index].getTextComponent(), false)) {
						m_multiTextComponent.activateTab(textEditors[index]);
						dialog.setVisible(true);
						return true;
					}
					isSearchStarted = false;
				}
			} else {
				int index = textEditors.length - 1;
				if (currentTextEditor != null) {   
					for (; index >= 0; index--) {
						if (currentTextEditor == textEditors[index]) break;
					}
				}
				for (; index >= 0; index--) {
					if (isTextEmpty(textEditors[index].getTextComponent())) continue;
					currentTextEditor = textEditors[index];
					if (replace(textEditors[index].getTextComponent(), false)) {
						m_multiTextComponent.activateTab(textEditors[index]);
						dialog.setVisible(true);
						return true;
					}
					isSearchStarted = false;
				}
			}
			return false;
		}
		
		@Override
		public void replace() {
			if (!checkInputForReplacement()) return;
			updateSearch();
			if (!performReplacement()) warning("Text not found");
		}

		@Override
		public void replaceAll() {
			if (!checkInputForReplacement()) return;
			updateSearch();
	        int counter = 0;
	        while (performReplacement()) counter++;
	        if (counter > 0) {
		        JOptionPane.showMessageDialog(dialog.getOwner(), 
		          counter+" replacement(s) have been done", "Info",
		          JOptionPane.INFORMATION_MESSAGE);
	        } else {
	        	warning("Text not found");
	        }
		}
		
	}	
	
	private abstract class AbstractSearchDialog extends JDialog {

		protected JRadioButton rbSearchUp = new JRadioButton("Search up");
		protected JRadioButton rbSearchDown = new JRadioButton("Search down");
		protected ButtonGroup bgSearch = new ButtonGroup();
		protected JCheckBox cbWord = new JCheckBox("Whole words only");
		protected JCheckBox cbCase = new JCheckBox("Case sensitive");
		protected JCheckBox cbRegularExpression = new JCheckBox("Regular expression");
		protected JCheckBox cbIncremental = new JCheckBox("Incremental");
		protected JLabel lFind = new JLabel ("Search for :");  // Find What
		protected JComboBox cbFind = new JComboBox();
		protected JLabel lReplace = new JLabel("Replace with :");
		protected JComboBox cbReplace = new JComboBox();
		protected JLabel lOrigin = new JLabel("Origin :");
		protected JComboBox cbOrigin = new JComboBox();
		protected JButton bRegularExpressions = new JButton(">");
		protected JPopupMenu pmRegularExpressions = new JPopupMenu();
		protected JButton bFindNext = new JButton("Find Next");
		protected JButton bReplace = new JButton("Replace");
		protected JButton bReplaceAll = new JButton("Replace All");	
		protected JButton bClose = new JButton("Close");
		
		private void setDataModel() {
			if (m_modelWord == null) {
				m_modelWord = cbWord.getModel();
				m_modelWord.setSelected(false);
			} else 
				cbWord.setModel(m_modelWord);
			if (m_modelCase == null) {
				m_modelCase = cbCase.getModel();
				m_modelCase.setSelected(false);
			} else 
				cbCase.setModel(m_modelCase);
			if (m_modelUp == null) {
				m_modelUp = rbSearchUp.getModel();
				m_modelUp.setSelected(false);
			} else 
				rbSearchUp.setModel(m_modelUp);
			if (m_modelDown == null) {
				m_modelDown = rbSearchDown.getModel();
				m_modelDown.setSelected(true);
			} else 
				rbSearchDown.setModel(m_modelDown);
			if (m_modelExpression == null) {
				m_modelExpression = cbRegularExpression.getModel();
				m_modelExpression.setSelected(false);
			} else 
				cbRegularExpression.setModel(m_modelExpression);
			if (m_modelIncremental == null) {
				m_modelIncremental = cbIncremental.getModel();
				m_modelIncremental.setSelected(false);
			} else 
				cbIncremental.setModel(m_modelIncremental);
			bRegularExpressions.setEnabled(m_modelExpression.isSelected());
			cbFind.setModel(m_modelFind);
			cbReplace.setModel(m_modelReplace);
			cbOrigin.setModel(m_modelOrigin);
		}

		private void insertRegularExpression(String expression) {
			cbFind.getEditor().setItem((String) cbFind.getEditor().getItem() + expression);
		}
		
		private JMenuItem miAnySingleCharacter = new JMenuItem("Any Single Character");
		private JMenuItem miADigit = new JMenuItem("A Digit");
		private JMenuItem miANondigit = new JMenuItem("A Nondigit");
		private JMenuItem miAWhiteSpaceCharacter = new JMenuItem("A Whitespace Character");
		private JMenuItem miANonwhiteSpaceCharacter = new JMenuItem("A Nonwhitespace Character");
		private JMenuItem miAWordCharacter = new JMenuItem("A Word Character");
		private JMenuItem miANonwordCharacter = new JMenuItem("A Nonword Character");
		private JMenuItem miBeginningOfLine = new JMenuItem("Beginning Of Line");
	    private JMenuItem miEndOfLine = new JMenuItem("End Of Line");
		private JMenuItem miAWordBoundary = new JMenuItem("A Word Boundary");
		private JMenuItem miANonwordBoundary = new JMenuItem("A Nonword Boundary");
		private JMenuItem miZeroOrOneMatches = new JMenuItem("Zero Or One Match");
		private JMenuItem miZeroOrMoreMatches = new JMenuItem("Zero Or More Matches");
		private JMenuItem miOneOrMoreMatches = new JMenuItem("One Or More Matches");
		
		private ActionListener regularExpressionsMenuListener = new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (e.getSource() == miAnySingleCharacter) {
					insertRegularExpression(".");
				} else if (e.getSource() == miADigit) {
					insertRegularExpression("\\d");
				} else if (e.getSource() == miANondigit) {
					insertRegularExpression("\\D");
				} else if (e.getSource() == miAWhiteSpaceCharacter) {
					insertRegularExpression("\\s");
				} else if (e.getSource() == miANonwhiteSpaceCharacter) {
					insertRegularExpression("\\S");
				} else if (e.getSource() == miAWordCharacter) {
					insertRegularExpression("\\w");
				} else if (e.getSource() == miANonwordCharacter) {
					insertRegularExpression("\\W");
				} else if (e.getSource() == miBeginningOfLine) {
					insertRegularExpression("^");
				} else if (e.getSource() == miEndOfLine) {
					insertRegularExpression("$");
				} else if (e.getSource() == miAWordBoundary) {
					insertRegularExpression("\\b");
				} else if (e.getSource() == miANonwordBoundary) {
					insertRegularExpression("\\B");
				} else if (e.getSource() == miZeroOrOneMatches) {
					insertRegularExpression("?");
				} else if (e.getSource() == miZeroOrMoreMatches) {
					insertRegularExpression("*");
				} else if (e.getSource() == miOneOrMoreMatches) {
					insertRegularExpression("+");
				}
				cbFind.grabFocus();
			}
		};
		
		private void buildRegularExpressionsMenu() {
			pmRegularExpressions.add(miAnySingleCharacter);
			pmRegularExpressions.add(miADigit);
			pmRegularExpressions.add(miANondigit);
			pmRegularExpressions.add(miAWhiteSpaceCharacter);
			pmRegularExpressions.add(miANonwhiteSpaceCharacter);
			pmRegularExpressions.add(miAWordCharacter);
			pmRegularExpressions.add(miANonwordCharacter);
			pmRegularExpressions.add(miBeginningOfLine);
			pmRegularExpressions.add(miEndOfLine);
			pmRegularExpressions.add(miAWordBoundary);
			pmRegularExpressions.add(miANonwordBoundary);
			pmRegularExpressions.add(miZeroOrOneMatches);
			pmRegularExpressions.add(miZeroOrMoreMatches);
			pmRegularExpressions.add(miOneOrMoreMatches);
			
			miAnySingleCharacter.setMnemonic(KeyEvent.VK_A);
			miADigit.setMnemonic(KeyEvent.VK_D);
			miANondigit.setMnemonic(KeyEvent.VK_N);
			miAWhiteSpaceCharacter.setMnemonic(KeyEvent.VK_W);
			miANonwhiteSpaceCharacter.setMnemonic(KeyEvent.VK_O);
			miAWordCharacter.setMnemonic(KeyEvent.VK_R);
			miANonwordCharacter.setMnemonic(KeyEvent.VK_D);
			miBeginningOfLine.setMnemonic(KeyEvent.VK_B);
			miEndOfLine.setMnemonic(KeyEvent.VK_L);
			miAWordBoundary.setMnemonic(KeyEvent.VK_U);
			miANonwordBoundary.setMnemonic(KeyEvent.VK_Y);
			miZeroOrOneMatches.setMnemonic(KeyEvent.VK_Z);
			miZeroOrMoreMatches.setMnemonic(KeyEvent.VK_H);
			miOneOrMoreMatches.setMnemonic(KeyEvent.VK_M);
			
			miAnySingleCharacter.addActionListener(regularExpressionsMenuListener);
			miADigit.addActionListener(regularExpressionsMenuListener);
			miANondigit.addActionListener(regularExpressionsMenuListener);
			miAWhiteSpaceCharacter.addActionListener(regularExpressionsMenuListener);
			miANonwhiteSpaceCharacter.addActionListener(regularExpressionsMenuListener);
			miAWordCharacter.addActionListener(regularExpressionsMenuListener);
			miANonwordCharacter.addActionListener(regularExpressionsMenuListener);
			miBeginningOfLine.addActionListener(regularExpressionsMenuListener);
			miEndOfLine.addActionListener(regularExpressionsMenuListener);
			miAWordBoundary.addActionListener(regularExpressionsMenuListener);
			miANonwordBoundary.addActionListener(regularExpressionsMenuListener);	
			miZeroOrOneMatches.addActionListener(regularExpressionsMenuListener);
			miZeroOrMoreMatches.addActionListener(regularExpressionsMenuListener);
			miOneOrMoreMatches.addActionListener(regularExpressionsMenuListener);
		}
		
		private String getCurrentWord(JTextComponent textComponent) {
			String currentWord = "", currentLine = "";
			int position = 0, start = 0, end = 0;
			Element root = textComponent.getDocument().getDefaultRootElement();
			if (root.getElementCount() == 0) return "";
			int index = root.getElementIndex(textComponent.getCaretPosition());
			Element element = root.getElement(index);
			try {
				currentLine = textComponent.getDocument().getText(element.getStartOffset(), element.getEndOffset() - element.getStartOffset());
				position = textComponent.getCaretPosition() - element.getStartOffset();
			} catch (BadLocationException e) {
				return "";
			}
			for (start = (position > 0) ? (position - 1) : position; start >= 0; start--) {
				if (!Character.isLetter(currentLine.charAt(start))) break;
			}
			for (end = position; end < currentLine.length(); end++) {
				if (!Character.isLetter(currentLine.charAt(end))) break;
			}
			if (end <= start) return "";
			try {
				currentWord = currentLine.substring(start + 1, end);
			} catch (IndexOutOfBoundsException  e) {
				e.printStackTrace();
			}
			return currentWord;
		}
		
		protected void fillTextForSearch(JTextComponent textComponent) {
			String currentWord;
			if (isTextSelected(textComponent)) {
				String selection = textComponent.getSelectedText();
				if (selection.indexOf('\n') >= 0) {
					selection = selection.substring(0, selection.indexOf('\n'));
				}
				cbFind.getEditor().setItem(selection);
			} else if ((currentWord = getCurrentWord(textComponent)).length() > 0) {
				cbFind.getEditor().setItem(currentWord);
			} else if (m_keys.size() > 0) {
				cbFind.getEditor().setItem(m_keys.toArray()[0]);
			} else {
				cbFind.getEditor().setItem("");
			}
		    cbFind.getEditor().selectAll();
		    cbFind.grabFocus();
		}
		
		public AbstractSearchDialog(Window owner) {
		    super(owner);
		    setModal(false);
		    setResizable(false);
		    setDefaultCloseOperation(DISPOSE_ON_CLOSE);
		    addWindowListener(windowListener);
			getRootPane().setDefaultButton(bFindNext);
			lFind.setDisplayedMnemonic(KeyEvent.VK_S);
			lFind.setLabelFor(cbFind);
			lReplace.setDisplayedMnemonic(KeyEvent.VK_P);
			lReplace.setLabelFor(cbReplace);
			lOrigin.setDisplayedMnemonic(KeyEvent.VK_O);
			lOrigin.setLabelFor(cbOrigin);
			bFindNext.setMnemonic(KeyEvent.VK_F);
			bReplace.setMnemonic(KeyEvent.VK_R);
			bReplaceAll.setMnemonic(KeyEvent.VK_A);
		    bClose.setMnemonic(KeyEvent.VK_C);
		    bClose.addActionListener(alClose);
		    setDataModel();
			buildRegularExpressionsMenu();
			bRegularExpressions.setMargin(new Insets(0, 3, 0, 3));
			bRegularExpressions.addActionListener(new ActionListener() {
				@Override
				public void actionPerformed(ActionEvent e) {
					pmRegularExpressions.show(bRegularExpressions, 
							bRegularExpressions.getWidth() / 2, bRegularExpressions.getHeight() / 2);
				}
			});
			cbFind.setPreferredSize(new Dimension(200, cbFind.getPreferredSize().height));
			cbFind.setEditable(true);
			cbFind.setAutoscrolls(true);
			cbReplace.setPreferredSize(new Dimension(200, cbReplace.getPreferredSize().height));
			cbReplace.setEditable(true);
			cbReplace.setAutoscrolls(true);
			cbOrigin.setPreferredSize(new Dimension(200, cbReplace.getPreferredSize().height));
			cbOrigin.setEditable(false);
			cbOrigin.setAutoscrolls(false);
		}
		
		public void updateFindUI() {
		    cbFind.revalidate();
		    cbFind.repaint();
		    cbFind.updateUI();
		}
		
		public void updateReplaceUI() {
		    cbReplace.revalidate();
		    cbReplace.repaint();
		    cbReplace.updateUI();
		}
		
		protected void addOptionsItemListener(ItemListener listener) {
			cbCase.addItemListener(listener);
			cbIncremental.addItemListener(listener);
			cbRegularExpression.addItemListener(listener);
			cbWord.addItemListener(listener);
			rbSearchUp.addItemListener(listener);
			rbSearchDown.addItemListener(listener);
			cbOrigin.addItemListener(listener);
		}
		
		private ActionListener alClose = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setVisible(false);
				dispose();
				instance = null;
				System.gc();
			}
		};
		
		protected JPanel createOptionsPanel() {
			JPanel panel = new JPanel();
			bgSearch.add(rbSearchUp);
			bgSearch.add(rbSearchDown);
			rbSearchUp.setMnemonic(KeyEvent.VK_U);
			rbSearchDown.setMnemonic(KeyEvent.VK_D);
			cbWord.setMnemonic(KeyEvent.VK_W);
			cbCase.setMnemonic(KeyEvent.VK_E);
			cbRegularExpression.setMnemonic(KeyEvent.VK_X);
			cbRegularExpression.addChangeListener(new ChangeListener() {
				@Override
				public void stateChanged(ChangeEvent e) {
					bRegularExpressions.setEnabled(m_modelExpression.isSelected());
				}
			});
			cbIncremental.setMnemonic(KeyEvent.VK_I);
		    panel.setLayout(new GridLayout(3, 2, 8, 2));
		    panel.setBorder(new TitledBorder(new EtchedBorder(), "Options"));
		    panel.add(cbCase);
		    panel.add(rbSearchUp);
		    panel.add(cbWord);
		    panel.add(rbSearchDown);
		    panel.add(cbRegularExpression);
		    panel.add(cbIncremental);
			return panel;
		}
		
		private WindowListener windowListener = new WindowAdapter() { 
	        public void windowActivated(WindowEvent e) {
	        	cbFind.grabFocus();
	        }
	        public void windowDeactivated(WindowEvent e) { }
	    };

	}	
	
	private abstract class SingleTextSearchDialog extends AbstractSearchDialog {

		protected SearchEngine searchEngine = null;
		protected JTextComponent textComponent = null;
		
		public SingleTextSearchDialog(Window owner, JTextComponent textComponent) {
			super(owner);
		    this.textComponent = textComponent;
			bFindNext.addActionListener(alFindNext);
			bReplace.addActionListener(alReplace);
			bReplaceAll.addActionListener(alReplaceAll);
			fillTextForSearch(textComponent);
			searchEngine = new SearchEngine(this, textComponent);
			addOptionsItemListener(searchEngine.getOptionsListener());
		}
		
		private ActionListener alFindNext = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				searchEngine.find();
			}
		};
		
		private ActionListener alReplace = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				searchEngine.replace();
			}
		};
		
		private ActionListener alReplaceAll = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				searchEngine.replaceAll();
			}
		};
		
	}
	
	private final class FindDialog extends SingleTextSearchDialog {

		public FindDialog(Window owner, JTextComponent textComponent) {
			super(owner, textComponent);
			setTitle("Find");
			getContentPane().add(createContentPanel());
			pack();
		}
		
		private JPanel createFindPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.insets = new Insets(0, 0, 0, 5);
			panel.add(lFind, c);
			c.gridx = 1;
			panel.add(cbFind, c);
			c.gridx = 2;
			c.insets = new Insets(0, 0, 0, 0);
			panel.add(bRegularExpressions, c);
			return panel;
		}
		
		private JPanel createMainPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(0, 2, 5, 2);
			panel.add(createFindPanel(), c);
			c.gridx = 0;
			c.gridy = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(0, 0, 0, 0);
			panel.add(createOptionsPanel(), c);
			return panel;
		}
		
		private JPanel createContentPanel() {
			JPanel panel = new JPanel();
			panel.setBorder(new EmptyBorder(5, 5, 5, 5));
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 2;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.anchor = GridBagConstraints.NORTH;
			panel.add(createMainPanel(), c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bFindNext, c);
			c.gridx = 1;
			c.gridy = 1;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 0, 0);
			panel.add(bClose, c);
			return panel;
		}

	}
	
	private final class ReplaceDialog extends SingleTextSearchDialog {
		
		public ReplaceDialog(Window owner, JTextComponent textComponent) {
			super(owner, textComponent);
			setTitle("Replace");
			getContentPane().add(createContentPanel());
			pack();
		}

		private JPanel createFindPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.anchor = GridBagConstraints.WEST;
			c.insets = new Insets(0, 0, 0, 5);
			panel.add(lFind, c);
			c.gridx = 1;
			c.gridy = 0;
			c.anchor = GridBagConstraints.CENTER;
			c.insets = new Insets(0, 0, 5, 0);
			panel.add(cbFind, c);
			c.gridx = 2;
			c.gridy = 0;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bRegularExpressions, c);
			c.gridx = 0;
			c.gridy = 1;
			c.anchor = GridBagConstraints.WEST;
			c.insets = new Insets(0, 0, 0, 5);
			panel.add(lReplace, c);
			c.gridx = 1;
			c.gridy = 1;
			c.anchor = GridBagConstraints.CENTER;
			c.insets = new Insets(0, 0, 5, 0);
			panel.add(cbReplace, c);
			return panel;
		}	
		
		private JPanel createMainPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(0, 2, 5, 2);
			panel.add(createFindPanel(), c);
			c.gridx = 0;
			c.gridy = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(0, 0, 0, 0);
			panel.add(createOptionsPanel(), c);
			return panel;
		}
		
		private JPanel createContentPanel() {
			JPanel panel = new JPanel();
			panel.setBorder(new EmptyBorder(5, 5, 5, 5));
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 4;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.anchor = GridBagConstraints.NORTH;
			panel.add(createMainPanel(), c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bFindNext, c);
			c.gridx = 1;
			c.gridy = 1;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bReplace, c);
			c.gridx = 1;
			c.gridy = 2;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bReplaceAll, c);
			c.gridx = 1;
			c.gridy = 3;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 0, 0);
			panel.add(bClose, c);
			return panel;
		}
		
	}
	
	private abstract class MultiTextSearchDialog extends AbstractSearchDialog {

		protected SearchEngine searchEngine = null;
		protected SelectedTextSearchEngine selectedTextSearchEngine = null;
		protected EntireScopeSearchEngine entireScopeSearchEngine = null;
		protected AllTextsSearchEngine allTextsSearchEngine = null;
		
		private void updateStateOfOptions() {
		    if (m_modelOrigin.getSelectedItem().equals(FROM_CURSOR))
		    	cbIncremental.setEnabled(true);
		    else 
		    	cbIncremental.setEnabled(false);
		}
		
		public MultiTextSearchDialog(Window owner, MultiTextComponent multiTextComponent) {
			super(owner);
			JTextComponent textComponent = multiTextComponent.getCurrentTextEditor().getTextComponent();
		    updateStateOfOptions();
		    cbOrigin.addItemListener(new ItemListener() {
				@Override
				public void itemStateChanged(ItemEvent e) {
					updateStateOfOptions();
				}
			});
			bFindNext.addActionListener(alFindNext);
			bReplace.addActionListener(alReplace);
			bReplaceAll.addActionListener(alReplaceAll);
			fillTextForSearch(textComponent);
			searchEngine = new SearchEngine(this, textComponent);
			addOptionsItemListener(searchEngine.getOptionsListener());
			selectedTextSearchEngine = new SelectedTextSearchEngine(this, textComponent);
			addOptionsItemListener(selectedTextSearchEngine.getOptionsListener());
			entireScopeSearchEngine = new EntireScopeSearchEngine(this, textComponent);
			addOptionsItemListener(entireScopeSearchEngine.getOptionsListener());
			allTextsSearchEngine = new AllTextsSearchEngine(this, multiTextComponent);
			addOptionsItemListener(allTextsSearchEngine.getOptionsListener());
			multiTextComponent.addEditorChangeListener(editorChangeListener);
		}
		
		private EditorChangeListener editorChangeListener = new EditorChangeListener() {
			@Override
			public void editorClosed(EditorChangeEvent e) { }
			@Override
			public void editorChanged(EditorChangeEvent e) {
				searchEngine.setTextComponent(e.getTextEditor().getTextComponent());
				selectedTextSearchEngine.setTextComponent(e.getTextEditor().getTextComponent());
				entireScopeSearchEngine.setTextComponent(e.getTextEditor().getTextComponent());
			}
			@Override
			public void editorAdded(EditorChangeEvent e) { }
		};
		
		private ISearchEngine getSearchEngine() {
		    if (m_modelOrigin.getSelectedItem().equals(FROM_CURSOR))
		    	return searchEngine;
		    else if (m_modelOrigin.getSelectedItem().equals(SELECTED_TEXT))
		    	return selectedTextSearchEngine;
		    else if (m_modelOrigin.getSelectedItem().equals(ENTIRE_SCOPE))
		    	return entireScopeSearchEngine;
		    else if (m_modelOrigin.getSelectedItem().equals(ALL_TEXTS))
		    	return allTextsSearchEngine;
		    else 
		    	return searchEngine;
		}
		
		private ActionListener alFindNext = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				getSearchEngine().find();
			}
		};
		
		private ActionListener alReplace = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				getSearchEngine().replace();
			}
		};
		
		private ActionListener alReplaceAll = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				getSearchEngine().replaceAll();
			}
		};
		
	}	
	
	private final class MultiTextFindDialog extends MultiTextSearchDialog {

		public MultiTextFindDialog(Window owner, MultiTextComponent multiTextComponent) {
			super(owner, multiTextComponent);
			setTitle("Find");
			getContentPane().add(createContentPanel());
			pack();
		}
		
		private JPanel createFindPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.anchor = GridBagConstraints.WEST;
			c.insets = new Insets(0, 0, 0, 5);
			panel.add(lFind, c);
			c.gridx = 1;
			c.gridy = 0;
			c.anchor = GridBagConstraints.CENTER;
			c.insets = new Insets(0, 0, 5, 0);
			panel.add(cbFind, c);
			c.gridx = 2;
			c.gridy = 0;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bRegularExpressions, c);
			c.gridx = 0;
			c.gridy = 1;
			c.anchor = GridBagConstraints.WEST;
			c.insets = new Insets(0, 0, 0, 5);
			panel.add(lOrigin, c);
			c.gridx = 1;
			c.gridy = 1;
			c.anchor = GridBagConstraints.CENTER;
			c.insets = new Insets(0, 0, 5, 0);
			panel.add(cbOrigin, c);
			return panel;
		}
				
		private JPanel createMainPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(0, 2, 5, 2);
			panel.add(createFindPanel(), c);
			c.gridx = 0;
			c.gridy = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(0, 0, 0, 0);
			panel.add(createOptionsPanel(), c);
			return panel;
		}
		
		private JPanel createContentPanel() {
			JPanel panel = new JPanel();
			panel.setBorder(new EmptyBorder(5, 5, 5, 5));
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 2;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.anchor = GridBagConstraints.NORTH;
			panel.add(createMainPanel(), c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bFindNext, c);
			c.gridx = 1;
			c.gridy = 1;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 0, 0);
			panel.add(bClose, c);
			return panel;
		}

	}
	
	private final class MultiTextReplaceDialog extends MultiTextSearchDialog {
		
		public MultiTextReplaceDialog(Window owner, MultiTextComponent multiTextComponent) {
			super(owner, multiTextComponent);
			setTitle("Replace");
			getContentPane().add(createContentPanel());
			pack();
		}

		private JPanel createFindPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.anchor = GridBagConstraints.WEST;
			c.insets = new Insets(0, 0, 0, 5);
			panel.add(lFind, c);
			c.gridx = 1;
			c.gridy = 0;
			c.anchor = GridBagConstraints.CENTER;
			c.insets = new Insets(0, 0, 5, 0);
			panel.add(cbFind, c);
			c.gridx = 2;
			c.gridy = 0;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bRegularExpressions, c);
			c.gridx = 0;
			c.gridy = 1;
			c.anchor = GridBagConstraints.WEST;
			c.insets = new Insets(0, 0, 0, 5);
			panel.add(lReplace, c);
			c.gridx = 1;
			c.gridy = 1;
			c.anchor = GridBagConstraints.CENTER;
			c.insets = new Insets(0, 0, 5, 0);
			panel.add(cbReplace, c);
			c.gridx = 0;
			c.gridy = 2;
			c.anchor = GridBagConstraints.WEST;
			c.insets = new Insets(0, 0, 0, 5);
			panel.add(lOrigin, c);
			c.gridx = 1;
			c.gridy = 2;
			c.anchor = GridBagConstraints.CENTER;
			c.insets = new Insets(0, 0, 5, 0);
			panel.add(cbOrigin, c);
			return panel;
		}	
		
		private JPanel createMainPanel() {
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(0, 2, 5, 2);
			panel.add(createFindPanel(), c);
			c.gridx = 0;
			c.gridy = 1;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.insets = new Insets(0, 0, 0, 0);
			panel.add(createOptionsPanel(), c);
			return panel;
		}
		
		private JPanel createContentPanel() {
			JPanel panel = new JPanel();
			panel.setBorder(new EmptyBorder(5, 5, 5, 5));
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.gridheight = 4;
			c.fill = GridBagConstraints.HORIZONTAL;
			c.anchor = GridBagConstraints.NORTH;
			panel.add(createMainPanel(), c);
			c.gridx = 1;
			c.gridy = 0;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bFindNext, c);
			c.gridx = 1;
			c.gridy = 1;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bReplace, c);
			c.gridx = 1;
			c.gridy = 2;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 5, 0);
			panel.add(bReplaceAll, c);
			c.gridx = 1;
			c.gridy = 3;
			c.gridheight = 1;
			c.insets = new Insets(0, 5, 0, 0);
			panel.add(bClose, c);
			return panel;
		}
		
	}
		
	private static JDialog instance = null;
	
	private static void establishBounds(JDialog dialog, Window owner) {
        Dimension d1 = dialog.getSize();
        Dimension d2 = owner.getSize();
        Dimension ds = dialog.getToolkit().getScreenSize();
        int x = Math.max((d2.width-d1.width)/2, 0);
        int y = Math.max((d2.height-d1.height)/2, 0);
        int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
        int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
        dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
	}
	
	public static void showFindDialog(Window owner, JTextComponent textComponent) {
		if (instance != null) {
			instance.setVisible(true);
			return; 
		}
		FindDialog dialog = new TextFinder().new FindDialog(owner, textComponent);
		establishBounds(dialog, owner);
		instance = dialog;
		dialog.setVisible(true);
	}
	
	public static void showReplaceDialog(Window owner, JTextComponent textComponent) {
		if (instance != null) {
			instance.setVisible(true);
			return; 
		}
		ReplaceDialog dialog = new TextFinder().new ReplaceDialog(owner, textComponent);
		establishBounds(dialog, owner);
		instance = dialog;
		dialog.setVisible(true);
	}

	public static void showFindDialog(Window owner, MultiTextComponent multiTextComponent) {
		if (instance != null) {
			instance.setVisible(true);
			return; 
		}
		MultiTextFindDialog dialog = new TextFinder().new MultiTextFindDialog(owner, multiTextComponent);
		establishBounds(dialog, owner);
		instance = dialog;
		dialog.setVisible(true);
	}
	
	public static void showReplaceDialog(Window owner, MultiTextComponent multiTextComponent) {
		if (instance != null) {
			instance.setVisible(true);
			return; 
		}
		MultiTextReplaceDialog dialog = new TextFinder().new MultiTextReplaceDialog(owner, multiTextComponent);
		establishBounds(dialog, owner);
		instance = dialog;
		dialog.setVisible(true);
	}
		
}
