package neutrino.multitext;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.nio.charset.Charset;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.JTextComponent;

import neutrino.dialogs.DocumentRestorer;
import neutrino.dialogs.TextSaver;
import neutrino.text.*;
import neutrino.text.TextComponent;

/**
 * Component for interaction with several text components. 
 * When contains greater then one text component changes view to tabbed pane otherwise
 * shows only text component
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class MultiTextComponent extends JComponent {

	protected ArrayList<TextEditorAdapter> m_components = null;
	private JTabbedPane m_tabbedPane = null;
	private ArrayList<EditorChangeListener> editorChangeListeners = null;
	private JPopupMenu popupMenu = null;
	private JPopupMenu popupMenuForTextComponents = null;
	private boolean autoIndentMode = true;
	private int tabSize = 4;
	private boolean lineWrap = false;
	private boolean wrapStyleWord = false;
	private File currentDirectory = null;
	private boolean isPersistBackups = false;
	private Charset encoding = Charset.defaultCharset();
	private int margin = 0;
	
	public MultiTextComponent() {
		super();
		m_components = new ArrayList<TextEditorAdapter>();
		TextEditorAdapter textEditorAdapter = new PlainTextAreaEditorAdapter(); 
		m_components.add(textEditorAdapter);
		establishSettings(textEditorAdapter.getTextEditor());
		m_tabbedPane = new JTabbedPane();
		setLayout(new BorderLayout());
		add(textEditorAdapter, BorderLayout.CENTER);
		
		editorChangeListeners = new ArrayList<EditorChangeListener>();
		m_tabbedPane.addChangeListener(new ChangeListener() {
			@Override
			public void stateChanged(ChangeEvent e) {
				fireEditorChanged();
			}
		});
		m_tabbedPane.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				if (popupMenu != null && e.getButton() == e.BUTTON3) {
					popupMenu.show(m_tabbedPane, e.getX(), e.getY());
				}
			}
		});
		createDefaultPopupMenu();
	}
	
	protected void createDefaultPopupMenu() {
		// create meu items
		final JMenu pmNewEditor = new JMenu("New Editor");
		final JMenuItem pmiPlainText = new JMenuItem("Plain Text");
		final JMenuItem pmiRTF = new JMenuItem("RTF");
		final JMenuItem pmiHTML = new JMenuItem("HTML");
		final JMenuItem pmiClose = new JMenuItem("Close");
		final JMenuItem pmiCloseOthers = new JMenuItem("Close Others");
		final JMenuItem pmiCloseAll = new JMenuItem("Close All");
		// create popup menu
		popupMenu = new JPopupMenu();
		popupMenu.addPopupMenuListener(new PopupMenuListener() {
			@Override
			public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
				pmiClose.setEnabled(canClose());
				pmiCloseOthers.setEnabled(canCloseOthers());
				pmiCloseAll.setEnabled(canCloseAll());
			}
			@Override
			public void popupMenuWillBecomeInvisible(PopupMenuEvent e) { }
			@Override
			public void popupMenuCanceled(PopupMenuEvent e) { }
		});
		popupMenu.add(pmNewEditor);
		pmNewEditor.add(pmiPlainText);
		pmNewEditor.add(pmiRTF);
		pmNewEditor.add(pmiHTML);
		popupMenu.addSeparator();
		popupMenu.add(pmiClose);
		popupMenu.add(pmiCloseOthers);
		popupMenu.add(pmiCloseAll);
		// build mnemonics
		pmNewEditor.setMnemonic(KeyEvent.VK_N);
		pmiPlainText.setMnemonic(KeyEvent.VK_P);
		pmiRTF.setMnemonic(KeyEvent.VK_R);
		pmiHTML.setMnemonic(KeyEvent.VK_H);
		pmiClose.setMnemonic(KeyEvent.VK_C);
		pmiCloseOthers.setMnemonic(KeyEvent.VK_O);
		pmiCloseAll.setMnemonic(KeyEvent.VK_A);		
		// build actions
		ActionListener listener = new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (e.getSource() == pmiPlainText) {
					newEditor();
				} else if (e.getSource() == pmiRTF) {
					newEditor(new RTFEditorAdapter());
				} else if (e.getSource() == pmiHTML) {
					newEditor(new HTMLEditorAdapter());
				} else if (e.getSource() == pmiClose) {
					close();
				} else if (e.getSource() == pmiCloseOthers) {
					closeOthers();
				} else if (e.getSource() == pmiCloseAll) {
					closeAll();
				}
			}
		};
		pmiPlainText.addActionListener(listener);
		pmiRTF.addActionListener(listener);
		pmiHTML.addActionListener(listener);
		pmiClose.addActionListener(listener);
		pmiCloseOthers.addActionListener(listener);
		pmiCloseAll.addActionListener(listener);		
	}
	
	public void setPopupMenu(JPopupMenu popupMenu) {
		this.popupMenu = popupMenu;
	}
	
	public void setPopupMenuForTextComponents(JPopupMenu popupMenu) {
		if (popupMenu == null) return;
		popupMenuForTextComponents = popupMenu;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.getTextComponent() instanceof PlainTextArea) {
				PlainTextArea textComponent = (PlainTextArea) textEditor.getTextComponent();
				textComponent.setPopupMenu(popupMenuForTextComponents);
			} else if (textEditor.getTextComponent() instanceof PlainTextComponent) {
				PlainTextComponent textComponent = (PlainTextComponent) textEditor.getTextComponent();
				textComponent.setPopupMenu(popupMenuForTextComponents);
			} else if (textEditor.getTextComponent() instanceof StyledTextComponent) {
				StyledTextComponent textComponent = (StyledTextComponent) textEditor.getTextComponent();
				textComponent.setPopupMenu(popupMenuForTextComponents);
			}
		}
	}
	
	private boolean isTextEmpty(JTextComponent textComponent) {
		return textComponent.getDocument().getLength() == 0;
	}
	
	/**
	 * Returns true when current tab can be closed
	 * @return boolean
	 */
	public boolean canClose() {
		TextEditor textEditor = getCurrentTextEditor();
		return !isSingleTextComponent() || textEditor.isFileLoaded() || !isTextEmpty(textEditor.getTextComponent());
	}
	
	/**
	 * Returns true when others tabs may be closed
	 * @return boolean
	 */
	public boolean canCloseOthers() {
		return !isSingleTextComponent();
	}
	
	/**
	 * Returns true when all tab may be closed
	 * @return boolean
	 */
	public boolean canCloseAll() {
		TextEditor textEditor = getCurrentTextEditor();
		return !isSingleTextComponent() || textEditor.isFileLoaded() || !isTextEmpty(textEditor.getTextComponent());
	}
	
	/**
	 * Returns the current text editor
	 * @return TextEditor
	 */
	public TextEditor getCurrentTextEditor() {
		return getCurrentEditorAdapter().getTextEditor();
	}
	
	/**
	 * Returns the current text editor adapter
	 * @return TextEditorAdapter
	 */
	protected TextEditorAdapter getCurrentEditorAdapter() {
		if (isSingleTextComponent()) {
			return m_components.get(0);
		} else {
			int index = m_tabbedPane.getSelectedIndex();
			if (index == -1) return m_components.get(0);
			return (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
		}
	}

	protected void establishSettings(TextEditor textEditor) {
		textEditor.getTextComponent().setMargin(new Insets(margin, margin, margin, margin));
		textEditor.getBackupManager().setEnabled(isPersistBackups);
		textEditor.setEncoding(encoding);
		textEditor.setCurrentDirectory(currentDirectory);
		textEditor.removeFileChangeListener(fileListener);
		textEditor.addFileChangeListener(fileListener);
		if (textEditor.getTextComponent() instanceof JTextArea) {
			JTextArea textComponent = (JTextArea) textEditor.getTextComponent();
			textComponent.setFont(getFont());
			textComponent.setForeground(getForeground());
			textComponent.setCaretColor(getForeground());
			textComponent.setSelectionColor(getForeground());
			textComponent.setBackground(getBackground());
			textComponent.setSelectedTextColor(getBackground());
			textComponent.setTabSize(getTabSize());
			textComponent.setLineWrap(getLineWrap());
			textComponent.setWrapStyleWord(getWrapStyleWord());
		}
		if (textEditor.getTextComponent() instanceof PlainTextArea) {
			PlainTextArea textComponent = (PlainTextArea) textEditor.getTextComponent();
			if (popupMenuForTextComponents != null) {
				textComponent.setPopupMenu(popupMenuForTextComponents);
			}
			textComponent.setAutoIndentMode(isAutoIndentMode());
		}
		if (textEditor.getTextComponent() instanceof TextComponent) {
			TextComponent textComponent = (TextComponent) textEditor.getTextComponent();
			if (popupMenuForTextComponents != null) {
				textComponent.setPopupMenu(popupMenuForTextComponents);
			}
		}
		if (textEditor.getTextComponent() instanceof PlainTextComponent) {
			PlainTextComponent textComponent = (PlainTextComponent) textEditor.getTextComponent();
			textComponent.setAutoIndentMode(isAutoIndentMode());
			textComponent.setFont(getFont());
			textComponent.setForeground(getForeground());
			textComponent.setCaretColor(getForeground());
			textComponent.setSelectionColor(getForeground());
			textComponent.setBackground(getBackground());
			textComponent.setSelectedTextColor(getBackground());
		}
		if (textEditor.getTextComponent() instanceof StyledTextComponent) {
			StyledTextComponent textComponent = (StyledTextComponent) textEditor.getTextComponent();
			if (popupMenuForTextComponents != null) {
				textComponent.setPopupMenu(popupMenuForTextComponents);
			}
		}
	}

	/**
	 * Returns true when can changed current editor
	 * @return boolean
	 */
	protected boolean canChangeEditor() {
		TextEditor currentTextEditor = getCurrentTextEditor();
		return !currentTextEditor.canClear();
	}
	
	/**
	 * Replaces current text editor component with new one
	 * @param textEditorAdapter - TextEditorAdapter 
	 */
	protected void changeEditor(TextEditorAdapter textEditorAdapter) {
		TextEditor currentTextEditor = getCurrentTextEditor();
		currentTextEditor.getBackupManager().turnOff();
		establishSettings(textEditorAdapter.getTextEditor());
		if (isSingleTextComponent()) {
			TextEditorAdapter firstComponentAdapter = m_components.get(0);
			remove(firstComponentAdapter);
			m_components.clear();
			m_components.add(textEditorAdapter);
			add(textEditorAdapter, BorderLayout.CENTER);
			textEditorAdapter.revalidate();
			textEditorAdapter.repaint();
		} else {
			TextEditorAdapter currentTextEditorAdapter = getCurrentEditorAdapter(); 
			int index = m_components.indexOf(currentTextEditorAdapter);
			m_components.set(index, textEditorAdapter);
			int selectedIndex = m_tabbedPane.getSelectedIndex();
			m_tabbedPane.removeTabAt(selectedIndex);
			m_tabbedPane.insertTab(createTitleForTextComponent(textEditorAdapter.getTextEditor()), null, textEditorAdapter, null, selectedIndex);
			m_tabbedPane.setSelectedIndex(selectedIndex);
			m_tabbedPane.revalidate();
			m_tabbedPane.repaint();
		}
	}
	
	/**
	 * Creates new tab with empty text component
	 */
	public void newEditor() {
		newEditor(new PlainTextAreaEditorAdapter());
	}
	
	public void newEditor(TextEditorAdapter textEditorAdapter) {
		if (canChangeEditor()) {
			changeEditor(textEditorAdapter);
		} else {
			TextEditor textEditor = textEditorAdapter.getTextEditor();
			establishSettings(textEditor);
			if (isSingleTextComponent()) {
				TextEditorAdapter firstComponentAdapter = m_components.get(0);
				m_tabbedPane.addTab(createTitleForTextComponent(m_components.get(0).getTextEditor()), firstComponentAdapter);
				m_tabbedPane.addTab(createTitleForTextComponent(textEditor), textEditorAdapter);		
				remove(firstComponentAdapter);
				add(m_tabbedPane, BorderLayout.CENTER);
				m_tabbedPane.setSelectedIndex(1);
			} else {
				m_tabbedPane.addTab(createTitleForTextComponent(textEditor), textEditorAdapter);		
				m_tabbedPane.setSelectedIndex(m_tabbedPane.getTabCount() - 1);
			}
			m_components.add(textEditorAdapter);
			m_tabbedPane.revalidate();
			m_tabbedPane.repaint();
		}
		fireEditorAdded();
	}
	
	public static String createTitleForTextComponent(TextEditor textEditor) {
		if (!textEditor.isFileLoaded()) {
			return "noname";
		} else {
			String fname = textEditor.getFile().getName();
			final int MAX_LENGTH = 40;
			return  (fname.length() >= MAX_LENGTH) ? (fname.substring(0, MAX_LENGTH - 1) + "...") : fname;
		}
	}
	
	/**
	 * Returns true when only one text component is created
	 * @return boolean
	 */
	public boolean isSingleTextComponent() {
		return m_components.size() == 1;
	}
	
	protected void replaceTabbedPaneOnScrollPane() {
		if (m_tabbedPane.getTabCount() == 1) {
			TextEditorAdapter firstComponentAdapter = m_components.get(0);
			m_tabbedPane.removeTabAt(0);
			remove(m_tabbedPane);
			add(firstComponentAdapter, BorderLayout.CENTER);
			firstComponentAdapter.revalidate();
			firstComponentAdapter.repaint();
		} else {
			m_tabbedPane.revalidate();
			m_tabbedPane.repaint();
		}
	}
	
	/**
	 * Closes current tab. Clears text component when it's single
	 */
	public void close() {
		TextEditorAdapter textEditorAdapter = getCurrentEditorAdapter();
		TextEditor textEditor = textEditorAdapter.getTextEditor(); 
		if (!textEditor.isTextSaved()) return;
		if (isSingleTextComponent()) {
			textEditor.clear();
		} else {
			textEditor.getBackupManager().turnOff();
			m_components.remove(textEditorAdapter);
			int index = m_tabbedPane.getSelectedIndex();
			if (index != -1) m_tabbedPane.removeTabAt(index);
			replaceTabbedPaneOnScrollPane();
			getCurrentTextEditor().getTextComponent().grabFocus();
		}
		fireEditorClosed();
		System.gc();
	}
	
	/**
	 * Closes all tabs without current. Returns true when success
	 * @return boolean
	 */
	public boolean closeOthers() {
		if (isSingleTextComponent()) return true;
		TextEditor currentTextComponent = getCurrentTextEditor();
		for (int index = 0; index < m_tabbedPane.getTabCount(); index++) {
			TextEditorAdapter textEditorAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
			TextEditor textEditor = textEditorAdapter.getTextEditor();
			if (textEditor == currentTextComponent) continue;
			if (textEditor.conditionOfSaving()) {
				activateTab(textEditor);
				if (!textEditor.isTextSaved()) return false;
			}
			textEditor.getBackupManager().turnOff();
			m_components.remove(textEditorAdapter);
			m_tabbedPane.removeTabAt(index);
			index--;
			fireEditorClosed();
		}
		replaceTabbedPaneOnScrollPane();
		getCurrentTextEditor().getTextComponent().grabFocus();
		System.gc();
		return true;
	}
	
	/**
	 * Closes all tabs. Clears current text component
	 */
	public void closeAll() {
		if (!closeOthers()) return;
		TextEditor textEditor = getCurrentTextEditor();
		if (!textEditor.isTextSaved()) return;
		textEditor.clear();
		getCurrentTextEditor().getTextComponent().grabFocus();
		fireEditorClosed();
	}
	
	/**
	 * Returns the array of text editors in order of their appearance in the tabbed pane
	 * @return array of text component
	 */
	public TextEditor[] getTextEditors() {
		TextEditor[] array = new TextEditor[m_components.size()];
		if (isSingleTextComponent()) {
			array[0] = m_components.get(0).getTextEditor();
		} else {
			for (int i = 0; i < m_tabbedPane.getTabCount(); i++) {
				TextEditorAdapter componentAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(i);
				array[i] = componentAdapter.getTextEditor();
			}
		}
		return array;
	}
	
	/**
	 * Activates tab with given text editor
	 * @param a text component
	 */
	public void activateTab(TextEditor textEditor) {
		if (textEditor == null) return;
		for (int index = 0; index < m_tabbedPane.getTabCount(); index++) {
			TextEditorAdapter textEditorAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
			if (textEditorAdapter.getTextEditor() == textEditor) {
				m_tabbedPane.setSelectedIndex(index);
				textEditor.getTextComponent().grabFocus();
				fireEditorChanged();
				return;
			}
		}
	}
	
	/**
	 * Activates tab with given file
	 * @param a File
	 */
	public void activateTab(File file) {
		if (file == null && isSingleTextComponent()) return;
		for (int index = 0; index < m_tabbedPane.getTabCount(); index++) {
			TextEditorAdapter textEditorAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
			if (textEditorAdapter.getTextEditor().isFileLoaded() 
					&&  textEditorAdapter.getTextEditor().getFile().equals(file)) {
				m_tabbedPane.setSelectedIndex(index);
				textEditorAdapter.getTextEditor().getTextComponent().grabFocus();
				fireEditorChanged();
				return;
			}
		}
	}
	

	
	/**
	 * Returns true when all text components is empty
	 * @return boolean
	 */
	public boolean isTextsEmpty() {
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (!isTextEmpty(textEditor.getTextComponent())) return false;
		}
		return true;
	}
	
	public ArrayList<TextInfo> getCurrentFiles() {
		ArrayList<TextInfo> list = new ArrayList<TextInfo>();
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditorAdapter textEditorAdapter = iterator.next();
			TextEditor textEditor = textEditorAdapter.getTextEditor();
			if (textEditor.isFileLoaded()) {
				TextInfo textInfo = new TextInfo();
				textInfo.setFile(textEditor.getFile());
				textInfo.setCaretPosition(textEditor.getTextComponent().getCaretPosition());
				textInfo.setAdapterClassName(textEditorAdapter.getClass().getCanonicalName());
				list.add(textInfo);
			}
		}
		return list;
	}
	
	public void openFiles(String[] fileNames) {
		ArrayList<TextInfo> files = new ArrayList<TextInfo>();
		for (int i = 0; i < fileNames.length; i++) {
			TextInfo text = new TextInfo();
			text.setFile(new File(fileNames[i]));
		}
		openFiles(files, false);
	}
	
	public void openFiles(ArrayList<TextInfo> files, boolean isKeepCaretPosition) {
		if (files == null || files.size() == 0) return;
		TextEditor currentTextEditor = getCurrentTextEditor();
		Iterator<TextInfo> iterator = files.iterator();
		while (iterator.hasNext()) {
			TextInfo text = iterator.next();
			if (text.getFile().exists() && text.getFile().isFile()) {
				try {
					TextEditorAdapter textEditorAdapter = (TextEditorAdapter) Class.forName(text.getAdapterClassName()).newInstance();
					newEditor(textEditorAdapter);
					currentTextEditor = getCurrentTextEditor();
					currentTextEditor.open(text.getFile());
					if (isKeepCaretPosition) {
						try {
							currentTextEditor.getTextComponent().setCaretPosition(text.getCaretPosition());
						} catch (IllegalArgumentException e) { 
							// don't set caret position 
						}
					}
				} catch (Exception e) {
					// skip file opening
				}
			}
		}
	}
	
	/**
	 * Creates appropriate TextEditorAdapter for given TextComponent class name
	 * @param className - String
	 * @return TextEditorAdapter
	 */
	protected TextEditorAdapter createTextEditorAdapterForTextComponent(String className) {
		if (className == null) {
			return new PlainTextAreaEditorAdapter();
		} if (className.equals(RTFComponent.class.getCanonicalName())) {
			return new RTFEditorAdapter();
		} else if (className.equals(HTMLComponent.class.getCanonicalName())) {
			return new HTMLEditorAdapter();
		} else {
			return new PlainTextAreaEditorAdapter();
		}
	}
	
	public void restoreBackups(File[] backups) {
		if (backups == null || backups.length == 0) return;
		for (int i = 0; i < backups.length; i++) {
			File file = backups[i];
			if (file.exists() && file.isFile()) {
				String className = DocumentRestorer.getTextComponentClassName(file);
				newEditor(createTextEditorAdapterForTextComponent(className));
				getCurrentTextEditor().restore(file);
			}
		}		
	}
	
	/**
	 * Returns true when can save at least one text
	 * @return boolean
	 */
	public boolean canSaveAll() {
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.canSave()) return true;
		}
		return false;
	}
	
	/**
	 * Saves all texts. Returns true when success
	 * @return boolean
	 */
	public boolean saveAll() {
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.canSave()) { 
				if (!textEditor.save()) return false;
			}
		}
		return true;
	}
	
	/**
	 * Returns list of text component that need to save
	 * @return a list of text components
	 */
	public ArrayList<TextEditor> getTextsForSaving() {
		ArrayList<TextEditor> list = new ArrayList<TextEditor>();
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.conditionOfSaving()) list.add(textEditor);
		}
		return list;
	}

	/**
	 * Returns true when all texts is saved. Forces saving of text in the file
	 * @return boolean
	 */
	public boolean isTextsSaved() {
		ArrayList<TextEditor> list = getTextsForSaving();
		if (list.size() == 0) { 
			return true;
		} else if (list.size() == 1) {
			TextEditor textEditor = list.get(0);
			activateTab(textEditor); 
			return textEditor.isTextSaved();
		} else {
			return TextSaver.showSavingConfirmationDialog(this, list);	
		}
	}
	
	/**
	 * Stops creating backups for all text component
	 */
	public void shutdown() {
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			textEditor.getBackupManager().turnOff();
		}
	}
	
	public void setFont(Font font) {
		super.setFont(font);
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			JTextComponent textComponent = textEditor.getTextComponent(); 
			if (textComponent instanceof JTextArea || textComponent instanceof PlainTextComponent) {
				textComponent.setFont(font);
			}
		}
	}
	
	public void setForeground(Color color) {
		super.setForeground(color);
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			JTextComponent textComponent = textEditor.getTextComponent(); 
			if (textComponent instanceof JTextArea || textComponent instanceof PlainTextComponent) {
				textComponent.setForeground(color);
				textComponent.setCaretColor(color);
				textComponent.setSelectionColor(color);
			}
		}
	}
	
	public void setBackground(Color color) {
		super.setBackground(color);
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			JTextComponent textComponent = textEditor.getTextComponent(); 
			if (textComponent instanceof JTextArea || textComponent instanceof PlainTextComponent) {
				textComponent.setBackground(color);
				textComponent.setSelectedTextColor(color);
			}
		}
	}
	
	public void setTabSize(int tabSize) {
		this.tabSize = tabSize;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.getTextComponent() instanceof JTextArea) {
				((JTextArea) textEditor.getTextComponent()).setTabSize(tabSize);
			}
		}
	}
	
	public int getTabSize() {
		return this.tabSize;
	}
	
	public void setMargin(int margin) {
		if (margin < 0) return;
		this.margin = margin;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			textEditor.getTextComponent().setMargin(new Insets(margin, margin, margin, margin));
		}
	}
	
	public int getMargin() {
		return this.margin;
	}
	
	public void setLineWrap(boolean lineWrap) {
		this.lineWrap = lineWrap;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.getTextComponent() instanceof JTextArea) {
				((JTextArea) textEditor.getTextComponent()).setLineWrap(lineWrap);
			}
		}
	}
	
	public boolean getLineWrap() {
		return this.lineWrap;
	}
	
	public void setWrapStyleWord(boolean wrapStyleWord) {
		this.wrapStyleWord = wrapStyleWord;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.getTextComponent() instanceof JTextArea) {
				((JTextArea) textEditor.getTextComponent()).setWrapStyleWord(wrapStyleWord);
			}
		}
	}
	
	public boolean getWrapStyleWord() {
		return this.wrapStyleWord;
	}
	
	public void setAutoIndentMode(boolean autoIndentMode) {
		this.autoIndentMode = autoIndentMode;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.getTextComponent() instanceof PlainTextArea) {
				((PlainTextArea) textEditor.getTextComponent()).setAutoIndentMode(autoIndentMode);
			} else if (textEditor.getTextComponent() instanceof PlainTextComponent) {
				((PlainTextComponent) textEditor.getTextComponent()).setAutoIndentMode(autoIndentMode);
			}
		}
	}
	
	public boolean isAutoIndentMode() {
		return this.autoIndentMode;
	}
	
	public boolean isPersistBackups() {
		return this.isPersistBackups;
	}
	
	public void setPersistBackups(boolean persistBackups) {
		this.isPersistBackups = persistBackups;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			textEditor.getBackupManager().setEnabled(persistBackups);
		}
	}
	
	public void setEncoding(Charset encoding) {
		if (encoding == null) return;
		this.encoding = encoding;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			textEditor.setEncoding(encoding);
		}
	}
	
	public Charset getEncoding() {
		return this.encoding;
	}
	
	/**
	 * Set current directory for save and open operations.
	 * @param dir - current directory
	 */
	public void setCurrentDirectory(File dir) {
		this.currentDirectory = dir;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			textEditor.setCurrentDirectory(dir);
		}
	}
	
	public File getCurrentDirectory() {
		return this.currentDirectory;
	}
	
	/**
	 * Adds EditorChangeListener for notification
	 * @param listener
	 */
	public void addEditorChangeListener(EditorChangeListener listener) {
		editorChangeListeners.add(listener);
	}
	
	/**
	 * Removes EditorChangeListener from list
	 * @param listener
	 */
	public void removeEditorChangeListener(EditorChangeListener listener) {
		editorChangeListeners.remove(listener);
	}
	
	/**
	 * Notifies listeners about changing of current tab
	 */
	protected void fireEditorChanged() {
		EditorChangeEvent event = new EditorChangeEvent(this, getCurrentTextEditor());
		Iterator<EditorChangeListener> iterator = editorChangeListeners.iterator();
		while (iterator.hasNext()) {
			EditorChangeListener listener = iterator.next();
			listener.editorChanged(event);
		}
	}

	/**
	 * Notifies listeners about adding of new tab with editor
	 */
	protected void fireEditorAdded() {
		EditorChangeEvent event = new EditorChangeEvent(this, getCurrentTextEditor());
		Iterator<EditorChangeListener> iterator = editorChangeListeners.iterator();
		while (iterator.hasNext()) {
			EditorChangeListener listener = iterator.next();
			listener.editorAdded(event);
		}
	}
	
	/**
	 * Notifies listeners about changing of closing of current tab
	 */
	protected void fireEditorClosed() {
		EditorChangeEvent event = new EditorChangeEvent(this, null);
		Iterator<EditorChangeListener> iterator = editorChangeListeners.iterator();
		while (iterator.hasNext()) {
			EditorChangeListener listener = iterator.next();
			listener.editorClosed(event);
		}
	}
	

	private FileChangeListener fileListener = new FileChangeListener() {

		public void fileChanged(FileChangeEvent event) {
			if (isSingleTextComponent()) return;
			TextEditor textEditor;
			for (int index = 0; index < m_tabbedPane.getTabCount(); index++) {
				TextEditorAdapter componentAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
				textEditor = componentAdapter.getTextEditor();
				if (textEditor == event.getSource()) {
					m_tabbedPane.setTitleAt(index, createTitleForTextComponent(textEditor));
					break;
				}
			}
		}
		
	};

}
